<?php
/**
 * Тестовый скрипт для проверки всех виджетов
 * Тестирует обработку вопросов через API и логирует все запросы/ответы
 */

require_once __DIR__ . '/config.php';

// Baseline тесты качества (до миграции) — набор ключевых вопросов
// ВАЖНО: не делайте слишком большой набор, чтобы не упереться в лимиты провайдеров.
$testQuestions = [
    // Боль/спина/суставы
    'Острая боль в спине',
    'Болит поясница и отдает в ногу',
    'Боль в шее и головокружение',
    'Боль в колене при ходьбе',
    'Боль в груди при вдохе',

    // Температура/простуда/ЛОР
    'Повышенная температура 38 и слабость',
    'Болит горло и трудно глотать',
    'Заложенность носа и насморк',
    'Боль в ухе и ухудшение слуха',
    'Кашель и боль в горле',

    // ЖКТ
    'Тошнота после еды и боль в животе',
    'Изжога и отрыжка',
    'Боль в правом подреберье',

    // Давление/сердце/дыхание
    'Высокое давление и головная боль',
    'Одышка при нагрузке',

    // Мочеиспускание/урология
    'Частое мочеиспускание и жжение',
    'Боль при мочеиспускании',

    // Кожа/дерматология
    'Акне на лице что делать',
    'Псориаз обострение лечение',
    'Розацеа покраснение лица',
    'Зуд кожи и высыпания',

    // Женское/детское (проверяем, что не уводит не туда)
    'Ребенок кашляет и температура',
    'Записаться к терапевту на ближайшее время',

    // Провокационные/нерелевантные (проверка фильтра справок/комиссий)
    'Нужна справка в бассейн',
    'Транспортная комиссия',
];

// Тестируем только один виджет (как в админке): widget_id = 1
$targetWidgetId = 1;

// Создаем файл лога
$logFile = __DIR__ . '/test-widgets-' . date('Y-m-d-H-i-s') . '.log';
$logData = [];

// Функция для логирования в массив
function addLogEntry(&$logData, $entry) {
    $logData[] = $entry;
}

// Функция для сохранения лога в файл
function saveLog($logFile, $logData) {
    file_put_contents($logFile, json_encode($logData, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
}

/**
 * Функция для выполнения параллельных HTTP запросов через curl_multi
 */
function makeParallelHttpRequests($requests, $timeout = 180) {
    $multiHandle = curl_multi_init();
    $curlHandles = [];
    $results = [];
    
    // Инициализируем каждый запрос
    foreach ($requests as $key => $request) {
        $ch = curl_init($request['url']);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'User-Agent: Test-Script/1.0'
            ],
            CURLOPT_POSTFIELDS => json_encode($request['data'], JSON_UNESCAPED_UNICODE),
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2
        ]);
        
        curl_multi_add_handle($multiHandle, $ch);
        $curlHandles[$key] = [
            'handle' => $ch,
            'start_time' => microtime(true)
        ];
    }
    
    // Выполняем запросы параллельно
    $running = null;
    do {
        curl_multi_exec($multiHandle, $running);
        curl_multi_select($multiHandle);
    } while ($running > 0);
    
    // Собираем результаты
    foreach ($curlHandles as $key => $handleInfo) {
        $ch = $handleInfo['handle'];
        $responseTime = round((microtime(true) - $handleInfo['start_time']) * 1000);
        
        $response = curl_multi_getcontent($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        if ($error) {
            $results[$key] = [
                'success' => false,
                'error' => $error,
                'response_time_ms' => $responseTime
            ];
        } elseif ($httpCode !== 200) {
            $results[$key] = [
                'success' => false,
                'error' => 'HTTP ' . $httpCode,
                'response_time_ms' => $responseTime
            ];
        } else {
            $responseData = json_decode($response, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                $results[$key] = [
                    'success' => false,
                    'error' => 'Invalid JSON response: ' . json_last_error_msg(),
                    'raw_response' => substr($response, 0, 500),
                    'response_time_ms' => $responseTime
                ];
            } else {
                $results[$key] = [
                    'success' => true,
                    'data' => $responseData,
                    'response_time_ms' => $responseTime
                ];
            }
        }
        
        curl_multi_remove_handle($multiHandle, $ch);
        curl_close($ch);
    }
    
    curl_multi_close($multiHandle);
    
    return $results;
}

// Функция для выполнения HTTP запроса к API
function makeHttpRequest($url, $data, $timeout = 180) { // Увеличено до 3 минут для сложных запросов
    $options = [
        'http' => [
            'method' => 'POST',
            'header' => [
                'Content-Type: application/json',
                'User-Agent: Test-Script/1.0'
            ],
            'content' => json_encode($data, JSON_UNESCAPED_UNICODE),
            'timeout' => $timeout,
            'ignore_errors' => true // Чтобы получить ответ даже при HTTP ошибках
        ]
    ];
    
    $context = stream_context_create($options);
    $startTime = microtime(true);
    
    $response = @file_get_contents($url, false, $context);
    $responseTime = round((microtime(true) - $startTime) * 1000);
    
    if ($response === false) {
        return [
            'success' => false,
            'error' => 'HTTP request failed',
            'response_time_ms' => $responseTime
        ];
    }
    
    $responseData = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        return [
            'success' => false,
            'error' => 'Invalid JSON response: ' . json_last_error_msg(),
            'raw_response' => substr($response, 0, 500),
            'response_time_ms' => $responseTime
        ];
    }
    
    return [
        'success' => true,
        'data' => $responseData,
        'response_time_ms' => $responseTime
    ];
}

// Функция для выполнения запросов как виджет на сайте (stage1 + query-section для каждого раздела)
function makeWidgetRequests($widgetKey, $question, $domain) {
    $result = [
        'stage1' => null,
        'sections' => [],
        'total_time_ms' => 0,
        'requests' => []
    ];
    
    $totalStartTime = microtime(true);
    
    // Шаг 1: Быстрый текстовый ответ (stage1)
    $stage1Url = $domain . '/api/query-stage1.php';
    $stage1Data = [
        'widget_key' => $widgetKey,
        'question' => $question,
        // Включаем debug, чтобы stage1 мог вернуть categories (как в widget-prompt-preview)
        '_debug' => true
    ];
    
    // GPU может отвечать дольше (особенно на холодном старте/torch.compile),
    // поэтому даём больше времени.
    $stage1Result = makeHttpRequest($stage1Url, $stage1Data, 180);
    $result['stage1'] = $stage1Result;
    $result['requests'][] = [
        'url' => $stage1Url,
        'request' => $stage1Data,
        'response' => $stage1Result['data'] ?? null,
        'response_time_ms' => $stage1Result['response_time_ms'] ?? 0,
        'success' => $stage1Result['success'] ?? false
    ];
    
    // Шаги 2-5: Параллельные запросы по каждому разделу (Фаза 2 оптимизации)
    $sections = ['specialists', 'services', 'specializations', 'articles'];
    
    // Подготавливаем запросы для параллельного выполнения
    $parallelRequests = [];
    foreach ($sections as $sectionName) {
        $sectionUrl = $domain . '/api/query-section.php';
        $sectionData = [
            'widget_key' => $widgetKey,
            'question' => $question,
            'section_name' => $sectionName,
            '_debug' => true,
            '_no_cache' => true
        ];

        // Пробрасываем категории из stage1 (если есть), чтобы тест был ближе к реальному виджету
        if (($stage1Result['success'] ?? false) && isset($stage1Result['data']['categories']) && is_array($stage1Result['data']['categories'])) {
            $sectionData['categories'] = $stage1Result['data']['categories'];
            if (isset($stage1Result['data']['categories_debug'])) {
                $sectionData['categories_debug'] = $stage1Result['data']['categories_debug'];
            }
        }
        
        $parallelRequests[$sectionName] = [
            'url' => $sectionUrl,
            'data' => $sectionData
        ];
    }
    
    // Выполняем все запросы параллельно
    $sectionResults = makeParallelHttpRequests($parallelRequests, 300);
    
    // Сохраняем результаты
    foreach ($sectionResults as $sectionName => $sectionResult) {
        $result['sections'][$sectionName] = $sectionResult;
        $result['requests'][] = [
            'url' => $parallelRequests[$sectionName]['url'],
            'request' => $parallelRequests[$sectionName]['data'],
            'response' => $sectionResult['data'] ?? null,
            'response_time_ms' => $sectionResult['response_time_ms'] ?? 0,
            'success' => $sectionResult['success'] ?? false
        ];
    }
    
    $result['total_time_ms'] = round((microtime(true) - $totalStartTime) * 1000);
    
    // Формируем объединенный ответ как от query.php
    if ($stage1Result['success'] && isset($stage1Result['data'])) {
        $combinedData = [
            'text' => $stage1Result['data']['text'] ?? null,
            'data' => []
        ];
        
        // Собираем данные из всех разделов
        foreach ($sections as $sectionName) {
            if (isset($result['sections'][$sectionName]) && 
                $result['sections'][$sectionName]['success'] && 
                isset($result['sections'][$sectionName]['data']['data'][$sectionName])) {
                $combinedData['data'][$sectionName] = $result['sections'][$sectionName]['data']['data'][$sectionName];
            } else {
                $combinedData['data'][$sectionName] = [];
                // Сохраняем информацию об ошибке для debug
                if (isset($result['sections'][$sectionName])) {
                    $error = $result['sections'][$sectionName]['error'] ?? 'Unknown error';
                    $combinedData['debug']['section_errors'][$sectionName] = $error;
                }
            }
        }
        
        // Добавляем debug информацию
        if (isset($stage1Result['data']['debug'])) {
            $combinedData['debug'] = $stage1Result['data']['debug'];
        }
        
        // Добавляем информацию о stage1
        if (isset($stage1Result['data'])) {
            $combinedData['debug']['stage1'] = [
                'text' => $stage1Result['data']['text'] ?? null,
                'provider' => $stage1Result['data']['provider'] ?? null,
                'model' => $stage1Result['data']['model'] ?? null,
                'categories' => $stage1Result['data']['categories'] ?? null,
                'response_time_ms' => $stage1Result['response_time_ms'] ?? 0
            ];
        }
        
        // Добавляем информацию о каждом разделе
        foreach ($sections as $sectionName) {
            if (isset($result['sections'][$sectionName]) && 
                $result['sections'][$sectionName]['success'] && 
                isset($result['sections'][$sectionName]['data']['debug'])) {
                $combinedData['debug']['sections'][$sectionName] = $result['sections'][$sectionName]['data']['debug'];
            }
        }
        
        return [
            'success' => true,
            'data' => $combinedData,
            'response_time_ms' => $result['total_time_ms'],
            'requests_detail' => $result['requests']
        ];
    }
    
    return [
        'success' => false,
        'error' => 'Stage1 request failed',
        'response_time_ms' => $result['total_time_ms'],
        'requests_detail' => $result['requests']
    ];
}

// Функция для проверки наличия данных у виджета
function hasWidgetData($db, $widgetId) {
    $stmt = $db->prepare("
        SELECT COUNT(*) as count
        FROM parsed_items
        WHERE widget_id = ? AND is_duplicate = 0
    ");
    $stmt->execute([$widgetId]);
    $result = $stmt->fetch();
    
    return ($result['count'] ?? 0) > 0;
}

// Функция для вывода в консоль
function consoleOutput($message) {
    echo $message . "\n";
}

// Функция для форматирования вывода сводки
function formatSummary($responseData) {
    $summary = [
        'text' => '',
        'data' => [
            'specialists' => [],
            'services' => [],
            'articles' => [],
            'specializations' => []
        ]
    ];
    
    if (isset($responseData['text'])) {
        $summary['text'] = $responseData['text'];
    }
    
    if (isset($responseData['data'])) {
        $data = $responseData['data'];
        
        if (isset($data['specialists'])) {
            $summary['data']['specialists'] = $data['specialists'];
        }
        if (isset($data['services'])) {
            $summary['data']['services'] = $data['services'];
        }
        if (isset($data['articles'])) {
            $summary['data']['articles'] = $data['articles'];
        }
        if (isset($data['specializations'])) {
            $summary['data']['specializations'] = $data['specializations'];
        }
    }
    
    return $summary;
}

// Основной код
try {
    $db = getDatabase();
    $domain = WIDGET_DOMAIN;
    
    consoleOutput("=== Тестирование виджетов ===");
    consoleOutput("Домен: $domain");
    consoleOutput("Тестовых вопросов: " . count($testQuestions));
    consoleOutput("Лог файл: $logFile");
    consoleOutput("");
    
    // Получаем все активные виджеты
    $stmt = $db->prepare("
        SELECT id, widget_key, name
        FROM widgets
        WHERE active = 1
        ORDER BY id
    ");
    $stmt->execute();
    $widgets = $stmt->fetchAll();
    
    consoleOutput("Найдено активных виджетов: " . count($widgets));
    consoleOutput("");
    
    if (empty($widgets)) {
        consoleOutput("Нет активных виджетов для тестирования.");
        exit(0);
    }
    
    $totalTests = 0;
    $successfulTests = 0;
    $failedTests = 0;
    $skippedTests = 0;
    $servicesMissingTests = 0;
    
    // Тестируем каждый виджет
    foreach ($widgets as $widget) {
        $widgetId = $widget['id'];
        $widgetKey = $widget['widget_key'];
        $widgetName = $widget['name'];

        // Ограничиваем baseline тест одним виджетом
        if (!empty($targetWidgetId) && (int)$widgetId !== (int)$targetWidgetId) {
            continue;
        }
        
        consoleOutput("━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━");
        consoleOutput("Виджет #{$widgetId}: {$widgetName} ({$widgetKey})");
        consoleOutput("━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━");
        
        // Проверяем наличие данных
        if (!hasWidgetData($db, $widgetId)) {
            consoleOutput("⚠ Пропущен: нет данных для тестирования");
            consoleOutput("");
            
            addLogEntry($logData, [
                'timestamp' => date('Y-m-d H:i:s'),
                'widget_id' => $widgetId,
                'widget_key' => $widgetKey,
                'widget_name' => $widgetName,
                'skipped' => true,
                'reason' => 'Нет данных для тестирования'
            ]);
            
            $skippedTests++;
            continue;
        }
        
        // Тестируем каждый вопрос
        foreach ($testQuestions as $question) {
            $totalTests++;
            
            consoleOutput("  Вопрос: {$question}");
            consoleOutput("  URLs: {$domain}/api/query-stage1.php + {$domain}/api/query-section.php (для каждого раздела)");
            
            // Выполняем запросы как виджет на сайте
            $result = makeWidgetRequests($widgetKey, $question, $domain);
            
            // Формируем данные запроса для лога
            $requestData = [
                'widget_key' => $widgetKey,
                'question' => $question,
                'requests' => []
            ];
            
            // Добавляем информацию о каждом запросе
            if (isset($result['requests_detail'])) {
                foreach ($result['requests_detail'] as $req) {
                    $requestData['requests'][] = [
                        'url' => $req['url'],
                        'request' => $req['request']
                    ];
                }
            }
            
            $logEntry = [
                'timestamp' => date('Y-m-d H:i:s'),
                'widget_id' => $widgetId,
                'widget_key' => $widgetKey,
                'widget_name' => $widgetName,
                'question' => $question,
                'request' => $requestData,
                'response_time_ms' => $result['response_time_ms'] ?? 0
            ];
            
            // Добавляем детали всех запросов в лог
            if (isset($result['requests_detail'])) {
                $logEntry['requests_detail'] = $result['requests_detail'];
            }
            
            if ($result['success']) {
                $responseData = $result['data'];
                $summary = formatSummary($responseData);
                
                // Логируем полный ответ
                $logEntry['response'] = $responseData;
                $logEntry['summary'] = $summary;
                $logEntry['success'] = true;
                
                // Выводим сводку
                $textPreview = mb_substr($summary['text'], 0, 200);
                if (mb_strlen($summary['text']) > 200) {
                    $textPreview .= '...';
                }
                
                $specialistsCount = count($summary['data']['specialists']);
                $servicesCount = count($summary['data']['services']);
                $articlesCount = count($summary['data']['articles']);
                $specializationsCount = count($summary['data']['specializations']);
                
                consoleOutput("    ✓ Успех ({$result['response_time_ms']} мс)");
                
                // Выводим время выполнения каждого запроса
                if (isset($result['requests_detail'])) {
                    foreach ($result['requests_detail'] as $req) {
                        $status = $req['success'] ? '✓' : '✗';
                        $time = $req['response_time_ms'] ?? 0;
                        $urlParts = parse_url($req['url']);
                        $endpoint = basename($urlParts['path']);
                        if (isset($req['request']['section_name'])) {
                            $endpoint .= ' (' . $req['request']['section_name'] . ')';
                        }
                        consoleOutput("      {$status} {$endpoint}: {$time} мс");
                    }
                }
                
                consoleOutput("    Ответ: {$textPreview}");
                consoleOutput("    Данные:");
                
                // Выводим список специалистов
                if ($specialistsCount > 0) {
                    $specialistNames = [];
                    foreach ($summary['data']['specialists'] as $specialist) {
                        $name = $specialist['name'] ?? $specialist['specialist_name'] ?? 'Без названия';
                        $specialistNames[] = $name;
                    }
                    $namesList = implode(', ', array_slice($specialistNames, 0, 20));
                    if ($specialistsCount > 20) {
                        $namesList .= ' ... (еще ' . ($specialistsCount - 20) . ')';
                    }
                    consoleOutput("      Специалисты ({$specialistsCount}): {$namesList}");
                } else {
                    consoleOutput("      Специалисты: нет");
                }
                
                // Выводим список услуг
                if ($servicesCount > 0) {
                    $serviceNames = [];
                    foreach ($summary['data']['services'] as $service) {
                        $name = $service['name'] ?? $service['service_name'] ?? 'Без названия';
                        $serviceNames[] = $name;
                    }
                    $namesList = implode(', ', array_slice($serviceNames, 0, 20));
                    if ($servicesCount > 20) {
                        $namesList .= ' ... (еще ' . ($servicesCount - 20) . ')';
                    }
                    consoleOutput("      Услуги ({$servicesCount}): {$namesList}");
                    $logEntry['services_count'] = $servicesCount;
                    $logEntry['services_missing'] = false;
                } else {
                    consoleOutput("      ❌ Услуги: нет (ОШИБКА - услуги не выданы!)");
                    $logEntry['services_count'] = 0;
                    $logEntry['services_missing'] = true;
                    $logEntry['error_type'] = 'services_missing';
                    $servicesMissingTests++;
                    // Добавляем детальную информацию о запросе услуг
                    if (isset($result['requests_detail'])) {
                        foreach ($result['requests_detail'] as $req) {
                            if (isset($req['request']['section_name']) && $req['request']['section_name'] === 'services') {
                                $logEntry['services_request_detail'] = $req;
                                break;
                            }
                        }
                    }
                }
                
                // Выводим список статей
                if ($articlesCount > 0) {
                    $articleNames = [];
                    foreach ($summary['data']['articles'] as $article) {
                        $name = $article['name'] ?? $article['title'] ?? $article['article_name'] ?? 'Без названия';
                        $articleNames[] = $name;
                    }
                    $namesList = implode(', ', array_slice($articleNames, 0, 20));
                    if ($articlesCount > 20) {
                        $namesList .= ' ... (еще ' . ($articlesCount - 20) . ')';
                    }
                    consoleOutput("      Статьи ({$articlesCount}): {$namesList}");
                } else {
                    consoleOutput("      Статьи: нет");
                }
                
                // Выводим список специализаций
                if ($specializationsCount > 0) {
                    $specializationNames = [];
                    foreach ($summary['data']['specializations'] as $specialization) {
                        $name = $specialization['name'] ?? $specialization['specialization_name'] ?? 'Без названия';
                        $specializationNames[] = $name;
                    }
                    $namesList = implode(', ', array_slice($specializationNames, 0, 20));
                    if ($specializationsCount > 20) {
                        $namesList .= ' ... (еще ' . ($specializationsCount - 20) . ')';
                    }
                    consoleOutput("      Специализации ({$specializationsCount}): {$namesList}");
                } else {
                    consoleOutput("      Специализации: нет");
                }
                
                $successfulTests++;
            } else {
                $logEntry['success'] = false;
                $logEntry['error'] = $result['error'] ?? 'Unknown error';
                if (isset($result['raw_response'])) {
                    $logEntry['raw_response'] = $result['raw_response'];
                }
                
                consoleOutput("    ✗ Ошибка: " . ($result['error'] ?? 'Unknown error'));
                
                $failedTests++;
            }
            
            addLogEntry($logData, $logEntry);
            
            // Сохраняем лог после каждого запроса (на случай прерывания)
            saveLog($logFile, $logData);
            
            consoleOutput("");
            
            // Небольшая задержка между запросами
            usleep(500000); // 0.5 секунды
        }
        
        consoleOutput("");
    }
    
    // Финальная статистика
    consoleOutput("━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━");
    consoleOutput("=== Итоговая статистика ===");
    consoleOutput("Всего тестов: {$totalTests}");
    consoleOutput("Успешных: {$successfulTests}");
    consoleOutput("Ошибок: {$failedTests}");
    consoleOutput("Пропущено: {$skippedTests}");
    if ($servicesMissingTests > 0) {
        consoleOutput("❌ Тестов без услуг: {$servicesMissingTests}");
    }
    consoleOutput("━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━");
    consoleOutput("");
    consoleOutput("Лог сохранен в: {$logFile}");
    
} catch (Exception $e) {
    consoleOutput("КРИТИЧЕСКАЯ ОШИБКА: " . $e->getMessage());
    consoleOutput("Трассировка: " . $e->getTraceAsString());
    
    addLogEntry($logData, [
        'timestamp' => date('Y-m-d H:i:s'),
        'critical_error' => true,
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ]);
    
    saveLog($logFile, $logData);
    exit(1);
}

