<?php
/**
 * Быстрый тест новых модулей валидации
 */

require_once '/root/config.php';
require_once '/root/api/text-validation.php';
require_once '/root/api/medical-dictionary.php';

echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║  ТЕСТ МОДУЛЕЙ ВАЛИДАЦИИ                                                  ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

// Тестовые случаи из анализа
$testCases = [
    [
        'name' => 'Незавершенное предложение (обрыв)',
        'question' => 'Ветрянка у детей',
        'text' => 'Ветрянка – это инфекционное заболевание, передающееся воздушно-капельным путем. Оно вызывается вирусом и чаще всего встречается у детей. Симптомы включают красные.',
        'expected_score' => 0
    ],
    [
        'name' => 'Английские слова (hands)',
        'question' => 'Остеохондроз шейного отдела',
        'text' => 'Остеохондроз шейного отдела позвоночника может вызывать боль в шее, плечах и hands. Рекомендуется обратиться к неврологу.',
        'expected_score' => 3
    ],
    [
        'name' => 'Неправильная аббревиатура (ЧОБЛ)',
        'question' => 'Хроническая обструктивная болезнь легких',
        'text' => 'ЧОБЛ — это хроническое заболевание легких, характеризующееся ограничением воздушного потока. Рекомендуется обратиться к пульмонологу.',
        'expected_score' => 5
    ],
    [
        'name' => 'Неверная аббревиатура ФГДС',
        'question' => 'Гастроскопия ФГДС',
        'text' => 'Гастроскопия ФГДС (фистуло-дуодено-сфинктерная гистероскопия) — это метод диагностики.',
        'expected_score' => 2
    ],
    [
        'name' => 'Неправильный специалист (офтальмолог для носа)',
        'question' => 'Кровотечение из носа',
        'text' => 'При кровотечении из носа рекомендуется обратиться к офтальмологу для диагностики.',
        'expected_score' => 6
    ],
    [
        'name' => 'Хороший текст без ошибок',
        'question' => 'Болит поясница и отдает в ногу',
        'text' => 'Боль в пояснице, отдающая в ногу, может быть связана с защемлением седалищного нерва, грыжей межпозвонкового диска или остеохондрозом. Для диагностики необходимы МРТ поясничного отдела позвоночника и консультация невролога или травматолога-ортопеда. Рекомендуется не откладывать визит к врачу.',
        'expected_score' => 10
    ]
];

$passed = 0;
$failed = 0;

foreach ($testCases as $i => $case) {
    $num = $i + 1;
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "ТЕСТ {$num}: {$case['name']}\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    echo "Вопрос: {$case['question']}\n";
    echo "Текст: " . mb_substr($case['text'], 0, 100) . "...\n\n";
    
    // Валидация
    $validation = validateTextQuality($case['text'], $case['question']);
    
    // Проверка медицинских ошибок
    $medicalErrors = checkMedicalErrors($case['text'], $case['question']);
    
    // Проверка рекомендаций специалистов
    $specialistValidation = validateSpecialistRecommendation($case['question'], $case['text']);
    
    echo "📊 РЕЗУЛЬТАТЫ ВАЛИДАЦИИ:\n";
    echo "  Оценка: {$validation['score']}/10\n";
    echo "  Вердикт: " . getVerdict($validation['score']) . "\n";
    echo "  Валиден: " . ($validation['is_valid'] ? "ДА" : "НЕТ") . "\n";
    echo "  Завершенность: " . ($validation['completeness'] ? "ДА" : "НЕТ") . "\n";
    echo "  Адекватность: " . ($validation['adequacy'] ? "ДА" : "НЕТ") . "\n";
    
    if (!empty($validation['issues'])) {
        echo "\n  ⚠️ Проблемы:\n";
        foreach ($validation['issues'] as $issue) {
            echo "    - {$issue}\n";
        }
    }
    
    if (!empty($validation['artifacts'])) {
        echo "\n  🔍 Артефакты:\n";
        foreach ($validation['artifacts'] as $artifact) {
            echo "    - {$artifact}\n";
        }
    }
    
    if (!empty($medicalErrors)) {
        echo "\n  🏥 Медицинские ошибки:\n";
        foreach ($medicalErrors as $error) {
            echo "    - {$error['type']}: {$error['found']} → {$error['correct']}\n";
        }
    }
    
    if (!$specialistValidation['valid']) {
        echo "\n  ❌ Ошибка рекомендации специалиста:\n";
        echo "    {$specialistValidation['error']}\n";
    }
    
    // Проверка ожидаемого результата
    $scoreDiff = abs($validation['score'] - $case['expected_score']);
    if ($scoreDiff <= 2) {
        echo "\n  ✅ ТЕСТ ПРОЙДЕН (оценка в пределах ±2 от ожидаемой)\n";
        $passed++;
    } else {
        echo "\n  ❌ ТЕСТ НЕ ПРОЙДЕН (ожидалось ~{$case['expected_score']}/10, получено {$validation['score']}/10)\n";
        $failed++;
    }
    
    echo "\n";
}

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "ИТОГО:\n";
echo "  ✅ Пройдено: {$passed}\n";
echo "  ❌ Не пройдено: {$failed}\n";
echo "  📊 Успешность: " . round($passed / count($testCases) * 100, 1) . "%\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
