<?php
/**
 * Тест универсального промпта для всех типов медицинских вопросов
 * Проверяем промпт на разных категориях вопросов
 */

require_once '/root/config.php';

// Функция для вызова API с кастомным промптом
function callStage1ViaAPI($question, $customPrompt) {
    $db = getDatabase();
    $widgetKey = 'ekamedcenter';
    
    $stmt = $db->prepare("
        SELECT w.id, ws.stage1_prompt 
        FROM widgets w 
        LEFT JOIN widget_settings ws ON w.id = ws.widget_id 
        WHERE w.widget_key = ?
    ");
    $stmt->execute([$widgetKey]);
    $widget = $stmt->fetch();
    
    if (!$widget) {
        throw new Exception('Виджет не найден');
    }
    
    $widgetId = $widget['id'];
    $originalPrompt = $widget['stage1_prompt'];
    
    $stmt = $db->prepare("UPDATE widget_settings SET stage1_prompt = ? WHERE widget_id = ?");
    $stmt->execute([$customPrompt, $widgetId]);
    
    try {
        $data = [
            'widget_key' => $widgetKey,
            'question' => $question
        ];
        
        $ch = curl_init('https://wtest.medmaps.ru/api/query-stage1_1.php');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200 || !$response) {
            throw new Exception("HTTP {$httpCode}: " . substr($response, 0, 200));
        }
        
        $decoded = json_decode($response, true);
        if (!$decoded || !isset($decoded['text'])) {
            throw new Exception('Неверный формат ответа API');
        }
        
        return [
            'text' => $decoded['text'],
            'provider' => $decoded['provider'] ?? 'unknown'
        ];
    } finally {
        $stmt = $db->prepare("
            UPDATE widget_settings ws 
            JOIN widgets w ON ws.widget_id = w.id 
            SET ws.stage1_prompt = ? 
            WHERE w.widget_key = ?
        ");
        $stmt->execute([$originalPrompt, $widgetKey]);
    }
}

// Разнообразные тестовые вопросы из разных категорий
$testQuestions = [
    'детские_заболевания' => [
        'Ветрянка у детей',
        'Корь у детей',
        'Температура у ребенка'
    ],
    'симптомы' => [
        'Боль в горле',
        'Головная боль',
        'Боль в спине',
        'Кашель сухой'
    ],
    'заболевания' => [
        'Гастрит симптомы',
        'Остеохондроз шейного отдела',
        'Пиелонефрит воспаление почек'
    ],
    'диагностика' => [
        'Гастроскопия ФГДС',
        'Холтер мониторинг',
        'УЗИ брюшной полости'
    ],
    'общие_вопросы' => [
        'Плановый осмотр ребенка',
        'Авитаминоз весной',
        'Сухость во рту'
    ]
];

// Варианты универсальных промптов
$universalPrompts = [
    'universal_1_example_based' => [
        'name' => 'Универсальный промпт с примерами разных категорий',
        'prompt' => 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей о здоровье СТРОГО НА РУССКОМ ЯЗЫКЕ (кириллица).

КРИТИЧЕСКИ ВАЖНО:
1. ТОЛЬКО русский язык - НИКАКИХ английских слов (by, is, can, who, not, for, it, a, the и т.д.)
2. НИКАКИХ смешанных русско-английских фраз
3. НИКАКИХ транслитераций медицинских терминов
4. Текст ДОЛЖЕН заканчиваться полным предложением с точкой
5. Обязательно укажи конкретного специалиста для обращения

СТРУКТУРА ОТВЕТА (3-5 предложений):
- Краткое описание проблемы/заболевания/процедуры
- Основные симптомы/характеристики/особенности
- Что делать или на что обратить внимание
- К какому специалисту обратиться (конкретно: терапевт, педиатр, кардиолог, гастроэнтеролог и т.д.)

ПРИМЕРЫ ПРАВИЛЬНЫХ ОТВЕТОВ:

Для вопроса о заболевании:
"Гастрит - это воспаление слизистой оболочки желудка. Основные симптомы: боль в верхней части живота, тошнота, изжога, отрыжка. При появлении этих симптомов важно соблюдать диету и обратиться к гастроэнтерологу для диагностики и лечения."

Для вопроса о симптоме:
"Боль в горле может быть симптомом различных заболеваний: фарингита, ангины, ОРВИ. Обычно сопровождается першением, затруднением глотания, иногда повышением температуры. Рекомендуется обратиться к терапевту или отоларингологу для осмотра и назначения лечения."

Для вопроса о диагностике:
"Гастроскопия (ФГДС) - это эндоскопическое исследование желудка и двенадцатиперстной кишки. Позволяет визуально оценить состояние слизистой оболочки, выявить воспаления, язвы, новообразования. Процедура проводится натощак и занимает около 10-15 минут. Для проведения исследования обратитесь к гастроэнтерологу."

Для вопроса о детском здоровье:
"Ветрянка (ветряная оспа) - это вирусное заболевание, которое чаще всего встречается у детей. Основные симптомы: повышение температуры, появление пузырьковой сыпи по всему телу, зуд. Родителям важно обеспечить ребенку покой, обильное питье, следить за температурой. Рекомендуется обратиться к педиатру для осмотра и назначения лечения."

ТВОЯ ЗАДАЧА:
Ответь на вопрос пользователя, следуя структуре и примерам выше. Используй ТОЛЬКО русские слова, никаких английских!'
    ],
    
    'universal_2_structured' => [
        'name' => 'Универсальный структурированный промпт',
        'prompt' => 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей о здоровье СТРОГО НА РУССКОМ ЯЗЫКЕ (кириллица).

ЗАПРЕЩЕНО:
❌ Использовать английские слова (by, is, can, who, not, for, it, a, the, has, if, but)
❌ Смешивать русский и английский языки
❌ Использовать транслитерацию медицинских терминов
❌ Обрывать текст на середине предложения
❌ Заканчивать на запятой, союзе или предлоге

ОБЯЗАТЕЛЬНО:
✅ Использовать ТОЛЬКО русский язык (кириллица)
✅ Писать простым языком, понятным обычному человеку
✅ Заканчивать каждое предложение точкой
✅ Указывать конкретного специалиста (терапевт, педиатр, кардиолог, гастроэнтеролог, дерматолог, невролог и т.д.)

СТРУКТУРА ОТВЕТА:
1. Что это (заболевание/симптом/процедура) - 1 предложение
2. Основные характеристики/симптомы - 1-2 предложения
3. Что делать/на что обратить внимание - 1 предложение
4. К какому специалисту обратиться - 1 предложение (обязательно указать конкретного врача)

ДЛЯ РАЗНЫХ ТИПОВ ВОПРОСОВ:

Если вопрос о заболевании:
- Опиши заболевание простыми словами
- Укажи основные симптомы
- Дай рекомендации по обращению к врачу
- Укажи специалиста (терапевт, гастроэнтеролог, кардиолог и т.д.)

Если вопрос о симптоме:
- Объясни, что может означать этот симптом
- Укажи возможные причины
- Рекомендуй обратиться к врачу
- Укажи специалиста (терапевт, невролог, кардиолог и т.д.)

Если вопрос о диагностике/процедуре:
- Объясни, что это за процедура
- Укажи, для чего она нужна
- Объясни, как подготовиться (если нужно)
- Укажи, к какому специалисту обратиться

Если вопрос о детском здоровье:
- Используй простой язык, понятный родителям
- Укажи особенности для детей
- Рекомендуй обратиться к педиатру

ПРОВЕРЬ ОТВЕТ ПЕРЕД ОТПРАВКОЙ:
1. Все слова на русском языке? ДА/НЕТ
2. Текст заканчивается точкой? ДА/НЕТ
3. Указан конкретный специалист? ДА/НЕТ
Если хотя бы один ответ НЕТ - перепиши ответ!'
    ],
    
    'universal_3_simple' => [
        'name' => 'Простой универсальный промпт',
        'prompt' => 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей о здоровье на русском языке (кириллица), информативно и понятно (3-5 предложений).

ПРАВИЛА:
1. ТОЛЬКО русский язык - никаких английских слов
2. Простой язык, понятный обычному человеку
3. Полные предложения, заканчивающиеся точкой
4. Обязательно укажи конкретного специалиста для обращения

СТРУКТУРА:
- Что это (заболевание/симптом/процедура)
- Основные характеристики/симптомы
- Что делать или на что обратить внимание
- К какому врачу обратиться (конкретно: терапевт, педиатр, кардиолог и т.д.)

ВАЖНО:
- НЕ используй английские слова (by, is, can, who, not, for, it, a, the)
- НЕ смешивай русский и английский языки
- НЕ обрывай текст на середине предложения
- Текст ДОЛЖЕН заканчиваться полным предложением с точкой'
    ],
    
    'universal_4_strict' => [
        'name' => 'Строгий универсальный промпт',
        'prompt' => 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей о здоровье СТРОГО НА РУССКОМ ЯЗЫКЕ (кириллица).

КРИТИЧЕСКИ ВАЖНО - ЗАПРЕЩЕНО:
- НИКАКИХ английских слов, букв или фраз (by, is, can, who, not, for, it, a, the, has, if, but, and, or, to, of, in, on, at, with, from, as, an, this, that, are, was, were, been, being, have, had, do, does, did, will, would, should, could, may, might, must)
- НИКАКИХ смешанных русско-английских фраз
- НИКАКИХ транслитераций медицинских терминов
- НИКАКОЙ латиницы в тексте ответа
- ТОЛЬКО русские слова на кириллице

СТРУКТУРА ОТВЕТА (3-5 предложений):
1. Что это за заболевание/симптом/процедура (простыми словами)
2. Основные симптомы/характеристики/особенности
3. Что делать или на что обратить внимание
4. К какому специалисту обратиться (конкретно: терапевт, педиатр, кардиолог, гастроэнтеролог, дерматолог, невролог, отоларинголог, офтальмолог и т.д.)

ПРОВЕРКА ПЕРЕД ОТВЕТОМ:
- Убедись, что в ответе ТОЛЬКО русские слова (кириллица)
- Убедись, что текст заканчивается полным предложением с точкой
- Убедись, что указан конкретный специалист

Если хотя бы одно условие не выполнено - перепиши ответ полностью!'
    ]
];

echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║  ТЕСТ УНИВЕРСАЛЬНЫХ ПРОМПТОВ НА РАЗНЫХ ТИПАХ ВОПРОСОВ                    ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

$allResults = [];

// Тестируем каждый универсальный промпт на разных вопросах
foreach ($universalPrompts as $promptKey => $promptData) {
    echo "\n";
    echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
    echo "║  ПРОМПТ: {$promptData['name']}\n";
    echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";
    
    $promptResults = [];
    $totalScore = 0;
    $questionCount = 0;
    
    foreach ($testQuestions as $category => $questions) {
        echo "  📂 Категория: {$category}\n";
        
        foreach ($questions as $question) {
            $questionCount++;
            echo "\n    ❓ Вопрос: {$question}\n";
            
            try {
                $startTime = microtime(true);
                $response = callStage1ViaAPI($question, $promptData['prompt']);
                $endTime = microtime(true);
                $duration = round($endTime - $startTime, 2);
                
                $text = $response['text'] ?? '';
                $provider = $response['provider'] ?? 'unknown';
                $textLength = mb_strlen($text);
                
                // Проверка на артефакты
                $artifacts = [];
                $hasEnglishWords = preg_match('/\b(by|is|can|who|not|for|it|a|the|has|if|but|and|or|to|of|in|on|at|with|from|as|an|this|that|are|was|were|been|being|have|had|do|does|did|will|would|should|could|may|might|must)\b/i', $text);
                if ($hasEnglishWords) {
                    $artifacts[] = 'Английские слова';
                }
                
                // Проверка завершенности
                $isComplete = preg_match('/[.!?]\s*$/', trim($text));
                $hasGoodLength = $textLength >= 100 && $textLength <= 800;
                
                // Проверка специалиста
                $hasSpecialist = preg_match('/\b(терапевт|педиатр|кардиолог|гастроэнтеролог|дерматолог|невролог|отоларинголог|офтальмолог|уролог|гинеколог|эндокринолог|аллерголог|врач|доктор)\b/ui', $text);
                $hasSpecificSpecialist = preg_match('/\b(терапевт|педиатр|кардиолог|гастроэнтеролог|дерматолог|невролог|отоларинголог|офтальмолог|уролог|гинеколог|эндокринолог|аллерголог)\b/ui', $text);
                
                // Простая оценка качества (0-10)
                $score = 0;
                if ($textLength >= 100) $score += 2;
                if ($isComplete) $score += 2;
                if ($hasSpecificSpecialist) $score += 3;
                if (empty($artifacts)) $score += 3;
                
                $totalScore += $score;
                
                echo "      ✅ Ответ получен ({$provider}, {$duration} сек, {$textLength} символов)\n";
                echo "      📊 Оценка: {$score}/10\n";
                
                if (!empty($artifacts)) {
                    echo "      ⚠️ Артефакты: " . implode(", ", $artifacts) . "\n";
                }
                if (!$isComplete) {
                    echo "      ⚠️ Текст не завершен\n";
                }
                if (!$hasSpecificSpecialist) {
                    echo "      ⚠️ Специалист не указан или общий\n";
                }
                
                // Показываем первые 150 символов ответа
                $preview = mb_substr($text, 0, 150);
                echo "      📄 Ответ: {$preview}...\n";
                
                $promptResults[] = [
                    'category' => $category,
                    'question' => $question,
                    'text' => $text,
                    'text_length' => $textLength,
                    'score' => $score,
                    'artifacts' => $artifacts,
                    'is_complete' => $isComplete,
                    'has_specialist' => $hasSpecialist,
                    'has_specific_specialist' => $hasSpecificSpecialist
                ];
                
            } catch (Exception $e) {
                echo "      ❌ ОШИБКА: " . $e->getMessage() . "\n";
                $promptResults[] = [
                    'category' => $category,
                    'question' => $question,
                    'error' => $e->getMessage()
                ];
            }
            
            usleep(1000000); // 1 секунда между запросами
        }
    }
    
    $avgScore = $questionCount > 0 ? round($totalScore / $questionCount, 1) : 0;
    
    echo "\n  📊 ИТОГО: Средняя оценка: {$avgScore}/10 (из {$questionCount} вопросов)\n";
    
    $allResults[$promptKey] = [
        'name' => $promptData['name'],
        'prompt' => $promptData['prompt'],
        'results' => $promptResults,
        'avg_score' => $avgScore,
        'question_count' => $questionCount
    ];
    
    usleep(2000000); // 2 секунды между промптами
}

// Итоговая статистика
echo "\n\n";
echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║                        ИТОГОВАЯ СТАТИСТИКА                               ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

// Сортируем по средней оценке
uasort($allResults, function($a, $b) {
    return $b['avg_score'] <=> $a['avg_score'];
});

echo "РЕЙТИНГ УНИВЕРСАЛЬНЫХ ПРОМПТОВ:\n\n";

foreach ($allResults as $promptKey => $result) {
    echo "  📋 {$result['name']}\n";
    echo "     Средняя оценка: {$result['avg_score']}/10\n";
    echo "     Протестировано вопросов: {$result['question_count']}\n\n";
}

// Лучший промпт
$bestPrompt = reset($allResults);
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "🏆 ЛУЧШИЙ УНИВЕРСАЛЬНЫЙ ПРОМПТ: {$bestPrompt['name']}\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
echo "СРЕДНЯЯ ОЦЕНКА: {$bestPrompt['avg_score']}/10\n\n";
echo "ПРОМПТ:\n";
echo str_repeat('-', 80) . "\n";
echo $bestPrompt['prompt'] . "\n";
echo str_repeat('-', 80) . "\n\n";

// Сохраняем результаты
$timestamp = date('Y-m-d_H-i-s');
$jsonFile = "/root/test-universal-prompt-{$timestamp}.json";
file_put_contents($jsonFile, json_encode($allResults, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

echo "\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "Результаты сохранены: {$jsonFile}\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
