<?php
/**
 * Тест текстовых ответов виджета
 * Проверяет наличие и качество поля 'text' в ответах
 */

// Разнообразные запросы для тестирования текстов
$queries = [
    'Судороги в ногах ночью',
    'Пневмония симптомы',
    'Скарлатина признаки',
    'ХОБЛ что это',
    'МРТ сустава',
    'Острая боль в животе',
    'Инфаркт профилактика',
    'Транспортная комиссия',
    'Справка на оружие',
    'Выписка из медицинской карты'
];

echo "╔═══════════════════════════════════════════════════════════════════════╗\n";
echo "║  ТЕСТ ТЕКСТОВЫХ ОТВЕТОВ ВИДЖЕТА (10 запросов)                         ║\n";
echo "╚═══════════════════════════════════════════════════════════════════════╝\n";
echo "\n";

$results = [];
$total = 0;
$withText = 0;
$emptyText = 0;
$warningText = 0;
$informativeText = 0;

foreach ($queries as $index => $query) {
    $num = $index + 1;
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "ВОПРОС {$num}/10: {$query}\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    
    // Запрос к виджету (используем api-wrapper.php для specialists)
    $cmd = sprintf(
        'php /root/api-wrapper.php %s %s %s 2>&1',
        escapeshellarg('ekamedcenter'),
        escapeshellarg($query),
        escapeshellarg('specialists')
    );
    
    $output = shell_exec($cmd);
    
    // Парсим JSON
    $lines = explode("\n", $output);
    $jsonLine = '';
    foreach ($lines as $line) {
        if (strpos($line, '{') === 0) {
            $jsonLine = $line;
            break;
        }
    }
    
    $decoded = json_decode($jsonLine, true);
    
    if (!$decoded) {
        echo "  ❌ ОШИБКА: Не удалось распарсить JSON\n\n";
        continue;
    }
    
    $text = $decoded['text'] ?? '';
    $textLength = mb_strlen($text);
    $warning = isset($decoded['warning']) && $decoded['warning'] === true;
    $specialistsCount = count($decoded['data']['specialists'] ?? []);
    
    $total++;
    
    // Определяем тип текста
    $textType = 'none';
    if ($textLength > 0) {
        $withText++;
        $textType = 'informative';
        
        if (mb_strpos($text, '⚠️') !== false || mb_strpos($text, 'ВНИМАНИЕ') !== false || mb_strpos($text, 'ВАЖНО') !== false) {
            $warningText++;
            $textType = 'warning';
        } elseif (mb_strpos($text, 'симптом') !== false || mb_strpos($text, 'признак') !== false || mb_strpos($text, 'диагностик') !== false) {
            $informativeText++;
            $textType = 'informative';
        } elseif (mb_strpos($text, 'К сожалению') !== false || mb_strpos($text, 'Для получения') !== false) {
            $textType = 'administrative';
        }
    } else {
        $emptyText++;
    }
    
    // Выводим результат
    echo "  Текст: ";
    if ($textLength > 0) {
        $preview = mb_substr($text, 0, 100);
        if ($textLength > 100) {
            $preview .= '...';
        }
        echo "✅ ({$textLength} символов)\n";
        echo "  Предпросмотр: " . str_replace("\n", " ", $preview) . "\n";
    } else {
        echo "❌ ПУСТОЙ\n";
    }
    
    echo "  Тип: ";
    switch ($textType) {
        case 'warning':
            echo "⚠️ WARNING (экстренная/срочная ситуация)\n";
            break;
        case 'informative':
            echo "ℹ️ ИНФОРМАТИВНЫЙ (описание симптомов/заболевания)\n";
            break;
        case 'administrative':
            echo "📋 АДМИНИСТРАТИВНЫЙ (справки, документы)\n";
            break;
        default:
            echo "❌ ОТСУТСТВУЕТ\n";
    }
    
    echo "  Специалисты: {$specialistsCount}\n";
    
    // Оценка качества
    $quality = '❌ ПЛОХО';
    if ($textLength > 0) {
        if ($textLength > 200) {
            $quality = '✅ ОТЛИЧНО';
        } elseif ($textLength > 50) {
            $quality = '⚠️ ХОРОШО';
        } else {
            $quality = '⚠️ УДОВЛЕТВОРИТЕЛЬНО';
        }
    }
    
    echo "  Качество: {$quality}\n";
    echo "\n";
    
    $results[] = [
        'query' => $query,
        'text' => $text,
        'textLength' => $textLength,
        'textType' => $textType,
        'specialists' => $specialistsCount,
        'quality' => $quality
    ];
}

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "ИТОГОВАЯ СТАТИСТИКА\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "Всего запросов: {$total}\n";
echo "С текстом: {$withText} (" . round($withText / $total * 100, 1) . "%)\n";
echo "  ⚠️ WARNING: {$warningText}\n";
echo "  ℹ️ Информативные: {$informativeText}\n";
echo "  📋 Административные: " . ($withText - $warningText - $informativeText) . "\n";
echo "Без текста: {$emptyText} (" . round($emptyText / $total * 100, 1) . "%)\n";
echo "\n";

// Средняя длина текста
$avgLength = 0;
if ($withText > 0) {
    $totalLength = 0;
    foreach ($results as $r) {
        $totalLength += $r['textLength'];
    }
    $avgLength = round($totalLength / $withText, 1);
}
echo "Средняя длина текста: {$avgLength} символов\n";
echo "\n";

// Качество текстов
$excellent = 0;
$good = 0;
$satisfactory = 0;
$bad = 0;

foreach ($results as $r) {
    if ($r['textLength'] > 200) {
        $excellent++;
    } elseif ($r['textLength'] > 50) {
        $good++;
    } elseif ($r['textLength'] > 0) {
        $satisfactory++;
    } else {
        $bad++;
    }
}

echo "Качество текстов:\n";
echo "  ✅ ОТЛИЧНО (>200 символов): {$excellent}\n";
echo "  ⚠️ ХОРОШО (50-200 символов): {$good}\n";
echo "  ⚠️ УДОВЛЕТВОРИТЕЛЬНО (<50 символов): {$satisfactory}\n";
echo "  ❌ ПЛОХО (нет текста): {$bad}\n";
echo "\n";

// Детальная таблица
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "ДЕТАЛЬНАЯ ТАБЛИЦА\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

printf("%-35s | %-8s | %-6s | %-15s | %s\n", "Вопрос", "Длина", "Тип", "Специалисты", "Качество");
echo str_repeat("-", 100) . "\n";

foreach ($results as $r) {
    $typeIcon = '';
    switch ($r['textType']) {
        case 'warning': $typeIcon = '⚠️'; break;
        case 'informative': $typeIcon = 'ℹ️'; break;
        case 'administrative': $typeIcon = '📋'; break;
        default: $typeIcon = '❌';
    }
    
    printf("%-35s | %8d | %-6s | %15d | %s\n", 
        mb_substr($r['query'], 0, 33),
        $r['textLength'],
        $typeIcon,
        $r['specialists'],
        $r['quality']
    );
}

echo "\n";

// Рекомендации
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "РЕКОМЕНДАЦИИ\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

if ($emptyText > 0) {
    echo "❌ ПРОБЛЕМА: {$emptyText} запросов без текста:\n";
    foreach ($results as $r) {
        if ($r['textLength'] === 0) {
            echo "   - {$r['query']}\n";
        }
    }
    echo "\n";
    echo "   РЕШЕНИЕ: Добавить информативные тексты в словарь симптомов\n";
    echo "   (строки 3368-3402 в query-section.php)\n\n";
}

if ($satisfactory > 0) {
    echo "⚠️ ВНИМАНИЕ: {$satisfactory} запросов с коротким текстом (<50 символов)\n";
    echo "   Рекомендуется расширить тексты для лучшей информативности\n\n";
}

if ($excellent + $good >= $total * 0.7) {
    echo "✅ ОТЛИЧНО! Большинство запросов имеют информативные тексты\n";
} elseif ($withText >= $total * 0.5) {
    echo "⚠️ НЕПЛОХО, но есть что улучшить\n";
} else {
    echo "❌ ПЛОХО! Многие запросы не имеют текстовых ответов\n";
}

echo "\n";

