<?php
/**
 * Тест текстовых ответов виджета на 310 вопросах
 * Использует список вопросов из test-300-with-expert.php
 */

// Загружаем вопросы из файла
require_once '/root/questions-list.php';

$widgetKey = 'ekamedcenter';

// Ограничиваем тест первыми 50 запросами
$testLimit = 50;
$questions = array_slice($questions, 0, $testLimit);

$total = count($questions);
$results = [];

echo "╔═══════════════════════════════════════════════════════════════════════╗\n";
echo "║  ТЕСТ ТЕКСТОВЫХ ОТВЕТОВ ВИДЖЕТА ({$total} запросов из первых 50)         ║\n";
echo "╚═══════════════════════════════════════════════════════════════════════╝\n";
echo "\n";
echo "Запуск теста...\n";
echo "Это может занять несколько минут...\n\n";

$startTime = microtime(true);
$processed = 0;
$withText = 0;
$emptyText = 0;
$warningText = 0;
$informativeText = 0;
$administrativeText = 0;

// Статистика по длине
$excellent = 0;  // >200 символов
$good = 0;        // 50-200 символов
$satisfactory = 0; // <50 символов
$bad = 0;         // нет текста

$totalLength = 0;

foreach ($questions as $index => $query) {
    $num = $index + 1;
    
    // Прогресс каждые 50 вопросов
    if ($num % 50 === 0) {
        $elapsed = round(microtime(true) - $startTime, 1);
        echo "  Обработано: {$num}/{$total} ({$elapsed} сек)\n";
    }
    
    // Запрос к виджету через query-stage1.php (это то, что использует виджет для текста!)
    // Виджет вызывает query-stage1.php для получения текста, а затем query-section.php для данных
    $postData = json_encode([
        'widget_key' => $widgetKey,
        'question' => $query
    ]);
    
    $ch = curl_init('https://wtest.medmaps.ru/api/query-stage1.php');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200 || !$response) {
        $results[] = [
            'query' => $query,
            'text' => '',
            'textLength' => 0,
            'textType' => 'error',
            'specialists' => 0,
            'error' => "HTTP {$httpCode}"
        ];
        $bad++;
        $emptyText++;
        continue;
    }
    
    $decoded = json_decode($response, true);
    
    if (!$decoded) {
        $results[] = [
            'query' => $query,
            'text' => '',
            'textLength' => 0,
            'textType' => 'error',
            'specialists' => 0,
            'error' => 'JSON parse error'
        ];
        $bad++;
        $emptyText++;
        continue;
    }
    
    // Проверяем text из ответа (это то, что показывается в ai-widget-answer-text)
    $text = $decoded['text'] ?? null;
    
    // Если text null или пустая строка, считаем что текста нет
    if ($text === null || $text === '') {
        $text = '';
    }
    
    $textLength = mb_strlen($text);
    $warning = isset($decoded['warning']) && $decoded['warning'] === true;
    // query-stage1.php не возвращает data, только text
    $specialistsCount = 0; // Для теста текстов не нужны данные о специалистах
    
    $processed++;
    
    // Определяем тип текста
    $textType = 'none';
    if ($textLength > 0) {
        $withText++;
        $totalLength += $textLength;
        $textType = 'informative';
        
        if (mb_strpos($text, '⚠️') !== false || mb_strpos($text, 'ВНИМАНИЕ') !== false || mb_strpos($text, 'ВАЖНО') !== false) {
            $warningText++;
            $textType = 'warning';
        } elseif (mb_strpos($text, 'симптом') !== false || mb_strpos($text, 'признак') !== false || mb_strpos($text, 'диагностик') !== false || mb_strpos($text, 'заболевание') !== false) {
            $informativeText++;
            $textType = 'informative';
        } elseif (mb_strpos($text, 'К сожалению') !== false || mb_strpos($text, 'Для получения') !== false || mb_strpos($text, 'справк') !== false) {
            $administrativeText++;
            $textType = 'administrative';
        } else {
            $informativeText++; // По умолчанию считаем информативным
        }
    } else {
        $emptyText++;
    }
    
    // Качество текста
    if ($textLength > 200) {
        $excellent++;
    } elseif ($textLength > 50) {
        $good++;
    } elseif ($textLength > 0) {
        $satisfactory++;
    } else {
        $bad++;
    }
    
    $results[] = [
        'query' => $query,
        'text' => $text,
        'textLength' => $textLength,
        'textType' => $textType,
        'specialists' => $specialistsCount,
        'provider' => $provider
    ];
}

$endTime = microtime(true);
$totalTime = round($endTime - $startTime, 1);
$avgLength = $withText > 0 ? round($totalLength / $withText, 1) : 0;

// Выводим результаты
echo "\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "ИТОГОВАЯ СТАТИСТИКА\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "Всего запросов: {$total}\n";
echo "Обработано: {$processed}\n";
echo "Время выполнения: {$totalTime} сек\n";
echo "Скорость: " . round($processed / $totalTime, 1) . " запросов/сек\n";
echo "\n";

echo "ТЕКСТОВЫЕ ОТВЕТЫ:\n";
echo "  С текстом: {$withText} (" . round($withText / $total * 100, 1) . "%)\n";
echo "    ⚠️ WARNING: {$warningText}\n";
echo "    ℹ️ Информативные: {$informativeText}\n";
echo "    📋 Административные: {$administrativeText}\n";
echo "  Без текста: {$emptyText} (" . round($emptyText / $total * 100, 1) . "%)\n";
echo "\n";

echo "КАЧЕСТВО ТЕКСТОВ:\n";
echo "  ✅ ОТЛИЧНО (>200 символов): {$excellent} (" . round($excellent / $total * 100, 1) . "%)\n";
echo "  ⚠️ ХОРОШО (50-200 символов): {$good} (" . round($good / $total * 100, 1) . "%)\n";
echo "  ⚠️ УДОВЛЕТВОРИТЕЛЬНО (<50 символов): {$satisfactory} (" . round($satisfactory / $total * 100, 1) . "%)\n";
echo "  ❌ ПЛОХО (нет текста): {$bad} (" . round($bad / $total * 100, 1) . "%)\n";
echo "\n";

echo "Средняя длина текста: {$avgLength} символов\n";
echo "\n";

// Топ-10 запросов без текста
$emptyQueries = array_filter($results, function($r) {
    return $r['textLength'] === 0;
});

if (count($emptyQueries) > 0) {
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "ТОП-10 ЗАПРОСОВ БЕЗ ТЕКСТА:\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    $emptyQueries = array_slice($emptyQueries, 0, 10);
    foreach ($emptyQueries as $r) {
        echo "  - {$r['query']} (специалистов: {$r['specialists']})\n";
    }
    echo "\n";
}

// Топ-10 лучших текстов
$bestTexts = array_filter($results, function($r) {
    return $r['textLength'] > 200;
});
usort($bestTexts, function($a, $b) {
    return $b['textLength'] - $a['textLength'];
});

if (count($bestTexts) > 0) {
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "ТОП-10 ЛУЧШИХ ТЕКСТОВ (по длине):\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    $bestTexts = array_slice($bestTexts, 0, 10);
    foreach ($bestTexts as $r) {
        $preview = mb_substr($r['text'], 0, 80);
        echo "  {$r['textLength']} символов: {$r['query']}\n";
        echo "    " . str_replace("\n", " ", $preview) . "...\n\n";
    }
}

// Сохраняем детальный отчет
$timestamp = date('Y-m-d_H-i-s');
$reportFile = "/root/text-test-300-report-{$timestamp}.txt";

$report = "ОТЧЕТ: ТЕСТ ТЕКСТОВЫХ ОТВЕТОВ ВИДЖЕТА\n";
$report .= "Дата: " . date('Y-m-d H:i:s') . "\n";
$report .= "Всего запросов: {$total}\n";
$report .= "Время выполнения: {$totalTime} сек\n\n";
$report .= "СТАТИСТИКА:\n";
$report .= "  С текстом: {$withText} (" . round($withText / $total * 100, 1) . "%)\n";
$report .= "  Без текста: {$emptyText} (" . round($emptyText / $total * 100, 1) . "%)\n";
$report .= "  Средняя длина: {$avgLength} символов\n\n";
$report .= "КАЧЕСТВО:\n";
$report .= "  ОТЛИЧНО (>200): {$excellent} (" . round($excellent / $total * 100, 1) . "%)\n";
$report .= "  ХОРОШО (50-200): {$good} (" . round($good / $total * 100, 1) . "%)\n";
$report .= "  УДОВЛЕТВОРИТЕЛЬНО (<50): {$satisfactory} (" . round($satisfactory / $total * 100, 1) . "%)\n";
$report .= "  ПЛОХО (нет текста): {$bad} (" . round($bad / $total * 100, 1) . "%)\n\n";
$report .= "ЗАПРОСЫ БЕЗ ТЕКСТА:\n";
foreach ($emptyQueries as $r) {
    $report .= "  - {$r['query']}\n";
}

file_put_contents($reportFile, $report);

// Сохраняем детальные результаты в JSON
$jsonFile = "/root/text-test-300-results-{$timestamp}.json";
file_put_contents($jsonFile, json_encode($results, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

echo "\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "Детальный отчет сохранен: {$reportFile}\n";
echo "JSON с текстами: {$jsonFile}\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

