<?php
/**
 * Тест виджета - проверка адекватности вывода specializations и articles
 * - Получение specializations и articles через query-section.php
 * - Оценка адекватности через AI эксперта (OpenRouter)
 * - Тестовый сервер: wtest.medmaps.ru
 */

// Используем список плохих запросов из предыдущего теста для проверки исправлений
require '/root/bad-questions-list.php';  // Плохие запросы (score <= 5 или ПЛОХО/УЖАСНО)

require_once '/root/config.php';

$widgetKey = 'ekamedcenter';

// Используем все плохие запросы для проверки
$total = count($questions);

echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║  ТЕСТ SPECIALIZATIONS & ARTICLES ({$total} ВОПРОСОВ) - ПРОВЕРКА АДЕКВАТНОСТИ ║\n";
echo "║  Сервер: wtest.medmaps.ru | API: query-section.php | Эксперт: Mimo    ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

// Получаем widget_id
$db = getDatabase();
$stmt = $db->prepare("SELECT id FROM widgets WHERE widget_key = ?");
$stmt->execute([$widgetKey]);
$widget = $stmt->fetch();
if (!$widget) {
    die("Ошибка: виджет '{$widgetKey}' не найден в базе данных\n");
}
$widgetId = $widget['id'];

// Получаем полный список всех specializations из БД
echo "→ Загрузка списка всех specializations из базы данных...\n";
function getAllSpecializationsFromDB($widgetId) {
    $db = getDatabase();
    $stmt = $db->prepare("
        SELECT DISTINCT pi.id, pf.field_name, pf.field_value
        FROM parsed_items pi
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.widget_id = ? 
        AND pi.section_name = 'specializations'
        AND pi.is_duplicate = 0
        ORDER BY pi.id, pf.field_name
    ");
    $stmt->execute([$widgetId]);
    
    $specializations = [];
    while ($row = $stmt->fetch()) {
        $itemId = (int)$row['id'];
        if (!isset($specializations[$itemId])) {
            $specializations[$itemId] = ['id' => $itemId];
        }
        $specializations[$itemId][$row['field_name']] = $row['field_value'];
    }
    
    return array_values($specializations);
}

$allSpecializations = getAllSpecializationsFromDB($widgetId);
$totalSpecializationsInDB = count($allSpecializations);
echo "  ✓ Загружено {$totalSpecializationsInDB} specializations из базы данных\n";

// Получаем полный список всех articles из БД
echo "→ Загрузка списка всех articles из базы данных...\n";
function getAllArticlesFromDB($widgetId) {
    $db = getDatabase();
    $stmt = $db->prepare("
        SELECT DISTINCT pi.id, pf.field_name, pf.field_value
        FROM parsed_items pi
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.widget_id = ? 
        AND pi.section_name = 'articles'
        AND pi.is_duplicate = 0
        ORDER BY pi.id, pf.field_name
    ");
    $stmt->execute([$widgetId]);
    
    $articles = [];
    while ($row = $stmt->fetch()) {
        $itemId = (int)$row['id'];
        if (!isset($articles[$itemId])) {
            $articles[$itemId] = ['id' => $itemId];
        }
        $articles[$itemId][$row['field_name']] = $row['field_value'];
    }
    
    return array_values($articles);
}

$allArticles = getAllArticlesFromDB($widgetId);
$totalArticlesInDB = count($allArticles);
echo "  ✓ Загружено {$totalArticlesInDB} articles из базы данных\n\n";

// Формируем краткое текстовое представление всех specializations для промпта (только названия)
function formatSpecializationsForPrompt($specializations) {
    $formatted = [];
    foreach ($specializations as $index => $spec) {
        $name = $spec['name'] ?? 'Неизвестно';
        $formatted[] = ($index + 1) . ". " . $name;
    }
    return implode("\n", $formatted);
}

// Формируем краткое текстовое представление всех articles для промпта (только названия, ограничиваем до 100)
function formatArticlesForPrompt($articles) {
    $formatted = [];
    // Ограничиваем до 100 статей для промпта, чтобы не перегружать
    $limitedArticles = array_slice($articles, 0, 100);
    foreach ($limitedArticles as $index => $article) {
        $title = $article['title'] ?? 'Неизвестно';
        $formatted[] = ($index + 1) . ". " . $title;
    }
    if (count($articles) > 100) {
        $formatted[] = "... и еще " . (count($articles) - 100) . " статей";
    }
    return implode("\n", $formatted);
}

// Для справочного списка используем только названия
function formatSpecializationsListForReference($specializations) {
    $names = [];
    foreach ($specializations as $spec) {
        $name = $spec['name'] ?? 'Неизвестно';
        $names[] = $name;
    }
    return implode(", ", $names);
}

function formatArticlesListForReference($articles, $totalCount) {
    $titles = [];
    // Ограничиваем до 50 статей для справочного списка
    $limitedArticles = array_slice($articles, 0, 50);
    foreach ($limitedArticles as $article) {
        $title = $article['title'] ?? 'Неизвестно';
        $titles[] = $title;
    }
    $result = implode(", ", $titles);
    if (count($articles) > 50) {
        $result .= " ... (всего {$totalCount} статей)";
    }
    return $result;
}

$allSpecializationsText = formatSpecializationsListForReference($allSpecializations);
$allArticlesText = formatArticlesListForReference($allArticles, $totalArticlesInDB);

$results = [];
$stats = [
    'total' => $total,
    'excellent' => 0,  // 9-10
    'good' => 0,       // 7-8
    'poor' => 0,       // 4-6
    'bad' => 0,        // 0-3
    'total_score' => 0,
    'with_specializations' => 0,
    'without_specializations' => 0,
    'with_articles' => 0,
    'without_articles' => 0
];

$startTime = microtime(true);

// Определяем эмодзи для вердикта (используется в итоговой статистике)
$verdictEmoji = [
    'ОТЛИЧНО' => '⭐',
    'ХОРОШО' => '✅',
    'ПЛОХО' => '⚠️',
    'УЖАСНО' => '❌'
];

// Разбиваем вопросы на батчи по 30 для параллельной обработки запросов к API
$batchSize = 30;
$questionBatches = array_chunk($questions, $batchSize, true); // Сохраняем ключи
$globalQuestionIndex = 0; // Глобальный счетчик вопросов

foreach ($questionBatches as $batchIndex => $batch) {
    $batchNum = $batchIndex + 1;
    $totalBatches = count($questionBatches);
    echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "БАТЧ {$batchNum}/{$totalBatches} (обработка " . count($batch) . " вопросов параллельно)\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    // ===== 1. ПОДГОТОВКА ПАРАЛЛЕЛЬНЫХ ЗАПРОСОВ ДЛЯ SPECIALIZATIONS (query-section.php) =====
    echo "  → Отправка запросов к query-section.php для получения specializations (параллельно)...\n";
    
    $multiHandle = curl_multi_init();
    $handles = [];
    
    foreach ($batch as $index => $query) {
        $sectionData = [
            'widget_key' => $widgetKey,
            'question' => $query,
            'section_name' => 'specializations'
        ];
        
        $ch = curl_init('https://wtest.medmaps.ru/api/query-section.php');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($sectionData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
        
        $handleId = (int)$ch;
        $handles[$handleId] = [
            'handle' => $ch,
            'type' => 'specializations',
            'index' => $index,
            'query' => $query
        ];
        
        curl_multi_add_handle($multiHandle, $ch);
    }
    
    // Выполняем параллельные запросы для specializations
    $running = null;
    do {
        curl_multi_exec($multiHandle, $running);
        curl_multi_select($multiHandle, 0.1);
    } while ($running > 0);
    
    // Собираем результаты specializations
    $specializationsResults = [];
    foreach ($handles as $handleId => $handleData) {
        if ($handleData['type'] === 'specializations') {
            $response = curl_multi_getcontent($handleData['handle']);
            $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
            
            $specializations = [];
            $text = '';
            
            if ($httpCode === 200 && $response) {
                $response = trim($response);
                $response = preg_replace('/^\xEF\xBB\xBF/', '', $response);
                $decoded = json_decode($response, true);
                
                if ($decoded && isset($decoded['data']['specializations'])) {
                    $specializations = $decoded['data']['specializations'];
                }
                if ($decoded && isset($decoded['text'])) {
                    $text = $decoded['text'];
                }
            }
            
            $specializationsResults[$handleData['index']] = [
                'specializations' => $specializations,
                'text' => $text,
                'query' => $handleData['query']
            ];
            
            curl_multi_remove_handle($multiHandle, $handleData['handle']);
            curl_close($handleData['handle']);
        }
    }
    
    curl_multi_close($multiHandle);
    
    // ===== 2. ПОДГОТОВКА ПАРАЛЛЕЛЬНЫХ ЗАПРОСОВ ДЛЯ ARTICLES (query-section.php) =====
    echo "  → Отправка запросов к query-section.php для получения articles (параллельно)...\n";
    
    $multiHandle = curl_multi_init();
    $handles = [];
    
    foreach ($batch as $index => $query) {
        $sectionData = [
            'widget_key' => $widgetKey,
            'question' => $query,
            'section_name' => 'articles'
        ];
        
        $ch = curl_init('https://wtest.medmaps.ru/api/query-section.php');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($sectionData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
        
        $handleId = (int)$ch;
        $handles[$handleId] = [
            'handle' => $ch,
            'type' => 'articles',
            'index' => $index,
            'query' => $query
        ];
        
        curl_multi_add_handle($multiHandle, $ch);
    }
    
    // Выполняем параллельные запросы для articles
    $running = null;
    do {
        curl_multi_exec($multiHandle, $running);
        curl_multi_select($multiHandle, 0.1);
    } while ($running > 0);
    
    // Собираем результаты articles
    $articlesResults = [];
    foreach ($handles as $handleId => $handleData) {
        if ($handleData['type'] === 'articles') {
            $response = curl_multi_getcontent($handleData['handle']);
            $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
            
            $articles = [];
            $text = '';
            
            if ($httpCode === 200 && $response) {
                $response = trim($response);
                $response = preg_replace('/^\xEF\xBB\xBF/', '', $response);
                $decoded = json_decode($response, true);
                
                if ($decoded && isset($decoded['data']['articles'])) {
                    $articles = $decoded['data']['articles'];
                }
                if ($decoded && isset($decoded['text'])) {
                    $text = $decoded['text'];
                }
            }
            
            $articlesResults[$handleData['index']] = [
                'articles' => $articles,
                'text' => $text,
                'query' => $handleData['query']
            ];
            
            curl_multi_remove_handle($multiHandle, $handleData['handle']);
            curl_close($handleData['handle']);
        }
    }
    
    curl_multi_close($multiHandle);
    
    // ===== 3. ПОДГОТОВКА ДАННЫХ ДЛЯ AI-ЭКСПЕРТА (проверка адекватности) =====
    $expertData = [];
    foreach ($batch as $index => $query) {
        $specializationsResult = $specializationsResults[$index] ?? ['specializations' => [], 'text' => '', 'query' => $query];
        $articlesResult = $articlesResults[$index] ?? ['articles' => [], 'text' => '', 'query' => $query];
        
        $returnedSpecializations = $specializationsResult['specializations'];
        $returnedArticles = $articlesResult['articles'];
        
        $expertData[$index] = [
            'query' => $query,
            'returned_specializations' => $returnedSpecializations,
            'returned_articles' => $returnedArticles,
            'returned_specializations_count' => count($returnedSpecializations),
            'returned_articles_count' => count($returnedArticles)
        ];
    }
    
    // ===== 4. ПАРАЛЛЕЛЬНЫЕ ЗАПРОСЫ К AI-ЭКСПЕРТУ (батчами по 10, все параллельно) =====
    // Разбиваем данные для эксперта на батчи по 10
    $expertBatchSize = 10;
    $expertBatches = array_chunk($expertData, $expertBatchSize, true);
    $expertResults = []; // Инициализируем массив результатов эксперта
    
    echo "  → Подготовка запросов к AI-эксперту (" . count($expertBatches) . " батчей по 10 вопросов)...\n";
    
    // Создаем промпты для всех батчей
    $expertPrompts = [];
    foreach ($expertBatches as $expertBatchIndex => $expertBatch) {
        $questionsData = [];
        $questionNumInBatch = 0; // Порядковый номер в батче (1-10)
        foreach ($expertBatch as $index => $data) {
            $questionNumInBatch++; // Увеличиваем порядковый номер
            $returnedSpecializations = $data['returned_specializations'];
            $returnedArticles = $data['returned_articles'];
            
            // Форматируем возвращенные specializations для промпта
            $returnedSpecializationsText = '';
            if (!empty($returnedSpecializations)) {
                $returnedSpecializationsText = formatSpecializationsForPrompt($returnedSpecializations);
            } else {
                $returnedSpecializationsText = "(специализации не возвращены)";
            }
            
            // Форматируем возвращенные articles для промпта
            $returnedArticlesText = '';
            if (!empty($returnedArticles)) {
                $returnedArticlesText = formatArticlesForPrompt($returnedArticles);
            } else {
                $returnedArticlesText = "(статьи не возвращены)";
            }
            
            $questionsData[] = [
                'question_number' => $questionNumInBatch, // Порядковый номер в батче (1-10)
                'original_index' => $index, // Сохраняем оригинальный индекс для сопоставления
                'query' => $data['query'],
                'returned_specializations_count' => count($returnedSpecializations),
                'returned_specializations' => $returnedSpecializationsText,
                'returned_articles_count' => count($returnedArticles),
                'returned_articles' => $returnedArticlesText
            ];
        }
        
        // Формируем один промпт для всех 10 вопросов
        $questionsText = "";
        foreach ($questionsData as $qData) {
            $questionsText .= "\n\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
            $questionsText .= "ВОПРОС #{$qData['question_number']}:\n";
            $questionsText .= "\"{$qData['query']}\"\n\n";
            $questionsText .= "СПЕЦИАЛИЗАЦИИ, ВОЗВРАЩЕННЫЕ ВИДЖЕТОМ ({$qData['returned_specializations_count']} специализаций):\n";
            $questionsText .= "{$qData['returned_specializations']}\n\n";
            $questionsText .= "СТАТЬИ, ВОЗВРАЩЕННЫЕ ВИДЖЕТОМ ({$qData['returned_articles_count']} статей):\n";
            $questionsText .= "{$qData['returned_articles']}\n";
        }
        
        $expertPrompt = "Ты - строгий эксперт по медицинским виджетам. Проверь АДЕКВАТНОСТЬ вывода СПЕЦИАЛИЗАЦИЙ И СТАТЕЙ виджетом медицинской клиники для 10 вопросов.

СПРАВОЧНЫЙ СПИСОК СПЕЦИАЛИЗАЦИЙ В БАЗЕ ДАННЫХ КЛИНИКИ ({$totalSpecializationsInDB} специализаций):
{$allSpecializationsText}

СПРАВОЧНЫЙ СПИСОК СТАТЕЙ В БАЗЕ ДАННЫХ КЛИНИКИ (показаны первые 50 из {$totalArticlesInDB} статей):
{$allArticlesText}
{$questionsText}

ТВОЯ ЗАДАЧА - проверить адекватность вывода specializations и articles для КАЖДОГО из 10 вопросов на три критерия:

1. РЕЛЕВАНТНОСТЬ:
   Проверь, что все возвращенные специализации и статьи релевантны вопросу пользователя.
   Например, для вопроса о боли в спине должны возвращаться специализации по неврологии, ортопедии, но НЕ офтальмология или гинекология.
   Статьи должны быть по теме вопроса, а не случайные.
   Если вернулись нерелевантные данные - это проблема.

2. ПОЛНОТА:
   Проверь, что вернулись все необходимые специализации и статьи для данного вопроса.
   Используй полные списки из базы данных как справочник.
   Если для вопроса о гастрите должны быть специализация гастроэнтерология и статьи о гастрите, но вернулась только специализация - это проблема.
   Если данные вообще не вернулись, но должны были - это критическая проблема.

3. ОТСУТСТВИЕ ЛИШНИХ:
   Проверь, что не вернулись специализации или статьи, которые не нужны для данного вопроса.
   Например, для вопроса о простуде не должны возвращаться специализации по хирургии или статьи о хирургических операциях.

ОЦЕНКА (0-10 баллов) для каждого вопроса:
- 10: все данные релевантны, полный список необходимых специализаций и статей, нет лишних
- 8-9: данные релевантны, но может не хватать 1-2 важных или есть 1 лишний
- 6-7: данные в целом релевантны, но есть заметные проблемы (не хватает важных или есть несколько лишних)
- 4-5: есть нерелевантные данные или критически не хватает важных
- 0-3: данные не релевантны вопросу или данные не возвращены, хотя должны были

ФОРМАТ ОТВЕТА (строго JSON массив из 10 элементов):
[
  {
    \"question_number\": 1,
    \"score\": <число 0-10>,
    \"verdict\": \"ОТЛИЧНО\" | \"ХОРОШО\" | \"ПЛОХО\" | \"УЖАСНО\",
    \"is_relevant\": <true/false>,
    \"is_complete\": <true/false>,
    \"missing_specializations\": [\"<список недостающих специализаций, если есть>\", ...],
    \"extra_specializations\": [\"<список лишних специализаций, если есть>\", ...],
    \"missing_articles\": [\"<список недостающих статей, если есть>\", ...],
    \"extra_articles\": [\"<список лишних статей, если есть>\", ...],
    \"comment\": \"<краткий комментарий о найденных проблемах>\",
    \"issues\": [\"<список проблем, если есть>\"]
  },
  {
    \"question_number\": 2,
    ...
  },
  ...
  {
    \"question_number\": 10,
    ...
  }
]

ВЕРДИКТЫ:
- ОТЛИЧНО: 9-10 баллов (все данные релевантны и полны)
- ХОРОШО: 7-8 баллов (незначительные проблемы)
- ПЛОХО: 4-6 баллов (заметные проблемы)
- УЖАСНО: 0-3 балла (критичные проблемы или данные не возвращены)

Отвечай ТОЛЬКО валидным JSON массивом, без дополнительного текста!";
        
        $expertPrompts[$expertBatchIndex] = [
            'prompt' => $expertPrompt,
            'batch_index' => $expertBatchIndex,
            'indices' => array_keys($expertBatch) // Сохраняем оригинальные индексы
        ];
    }
    
    // Отправляем все запросы параллельно
    echo "  → Отправка " . count($expertPrompts) . " запросов к AI-эксперту параллельно (по 10 вопросов в каждом)...\n";
    
    $multiHandle = curl_multi_init();
    $expertHandles = [];
    
    foreach ($expertPrompts as $batchIndex => $promptData) {
        $expertApiData = [
            'model' => 'xiaomi/mimo-v2-flash:free',
            'messages' => [
                ['role' => 'user', 'content' => $promptData['prompt']]
            ]
        ];
        
        $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($expertApiData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . OPENROUTER_API_KEY,
            'HTTP-Referer: https://wtest.medmaps.ru',
            'X-Title: Medical Widget Tester'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 120); // Увеличиваем таймаут для больших промптов
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        
        $handleId = (int)$ch;
        $expertHandles[$handleId] = [
            'handle' => $ch,
            'batch_index' => $batchIndex,
            'indices' => $promptData['indices']
        ];
        
        curl_multi_add_handle($multiHandle, $ch);
    }
    
    // Выполняем параллельные запросы к эксперту
    $running = null;
    do {
        curl_multi_exec($multiHandle, $running);
        curl_multi_select($multiHandle, 0.1);
    } while ($running > 0);
    
    // Собираем результаты эксперта
    $failedBatches = [];
    
    foreach ($expertHandles as $handleId => $handleData) {
        $response = curl_multi_getcontent($handleData['handle']);
        $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
        
        $expertResultsArray = null;
        if ($httpCode === 200 && $response) {
            $expertDecoded = json_decode($response, true);
            $expertContent = $expertDecoded['choices'][0]['message']['content'] ?? '';
            
            // Отладочная информация
            if (empty($expertContent)) {
                echo "  ⚠️ DEBUG: Empty content for batch {$handleData['batch_index']}\n";
                echo "  ⚠️ DEBUG: Response preview: " . substr($response, 0, 500) . "\n";
            }
            
            // Очищаем контент от markdown код блоков
            $cleanedContent = $expertContent;
            // Убираем markdown код блоки (```json ... ``` или ``` ... ```)
            $cleanedContent = preg_replace('/```json\s*/i', '', $cleanedContent);
            $cleanedContent = preg_replace('/```\s*/', '', $cleanedContent);
            $cleanedContent = trim($cleanedContent);
            
            // Отладочная информация о размере контента
            if (strlen($cleanedContent) > 0 && strlen($cleanedContent) < 200) {
                echo "  ⚠️ DEBUG batch {$handleData['batch_index']}: Content too short (" . strlen($cleanedContent) . " chars): " . substr($cleanedContent, 0, 200) . "\n";
            }
            
            // Пробуем разные способы парсинга JSON
            $expertResultsArray = null;
            
            // 1. Пробуем распарсить очищенный контент как JSON
            $decoded = json_decode($cleanedContent, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                if (isset($decoded[0]) && is_array($decoded[0])) {
                    // Это массив объектов - отлично
                    $expertResultsArray = $decoded;
                } elseif (isset($decoded['results']) && is_array($decoded['results'])) {
                    // Это объект с полем results
                    $expertResultsArray = $decoded['results'];
                }
            }
            
            // 2. Если не получилось, ищем JSON массив в тексте (может быть обрезан или с markdown)
            if (!$expertResultsArray) {
                // Ищем начало массива [
                $startPos = strpos($cleanedContent, '[');
                if ($startPos !== false) {
                    // Пробуем найти полный массив, начиная с найденной позиции
                    // Используем рекурсивный подсчет скобок
                    $jsonStr = '';
                    $bracketCount = 0;
                    $inString = false;
                    $escapeNext = false;
                    $foundStart = false;
                    
                    for ($i = $startPos; $i < strlen($cleanedContent); $i++) {
                        $char = $cleanedContent[$i];
                        
                        if ($escapeNext) {
                            $jsonStr .= $char;
                            $escapeNext = false;
                            continue;
                        }
                        
                        if ($char === '\\') {
                            $jsonStr .= $char;
                            $escapeNext = true;
                            continue;
                        }
                        
                        if ($char === '"') {
                            $inString = !$inString;
                            $jsonStr .= $char;
                            continue;
                        }
                        
                        if (!$inString) {
                            if ($char === '[') {
                                $bracketCount++;
                                $foundStart = true;
                                $jsonStr .= $char;
                            } elseif ($char === ']') {
                                $bracketCount--;
                                $jsonStr .= $char;
                                if ($bracketCount === 0 && $foundStart) {
                                    // Нашли закрывающую скобку массива
                                    break;
                                }
                            } else {
                                $jsonStr .= $char;
                            }
                        } else {
                            $jsonStr .= $char;
                        }
                    }
                    
                    if ($bracketCount === 0 && !empty($jsonStr)) {
                        $decoded = json_decode($jsonStr, true);
                        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded) && isset($decoded[0])) {
                            $expertResultsArray = $decoded;
                        }
                    }
                }
            }
            
            // 3. Если все еще не получилось, пробуем найти объект с results
            if (!$expertResultsArray) {
                $resultsPos = strpos($cleanedContent, '"results"');
                if ($resultsPos !== false) {
                    $arrayStartPos = strpos($cleanedContent, '[', $resultsPos);
                    if ($arrayStartPos !== false) {
                        // Ищем закрывающую скобку массива
                        $jsonStr = '';
                        $bracketCount = 0;
                        $inString = false;
                        $escapeNext = false;
                        
                        for ($i = $arrayStartPos; $i < strlen($cleanedContent); $i++) {
                            $char = $cleanedContent[$i];
                            
                            if ($escapeNext) {
                                $jsonStr .= $char;
                                $escapeNext = false;
                                continue;
                            }
                            
                            if ($char === '\\') {
                                $jsonStr .= $char;
                                $escapeNext = true;
                                continue;
                            }
                            
                            if ($char === '"') {
                                $inString = !$inString;
                                $jsonStr .= $char;
                                continue;
                            }
                            
                            if (!$inString) {
                                if ($char === '[') {
                                    $bracketCount++;
                                    $jsonStr .= $char;
                                } elseif ($char === ']') {
                                    $bracketCount--;
                                    $jsonStr .= $char;
                                    if ($bracketCount === 0) {
                                        break;
                                    }
                                } else {
                                    $jsonStr .= $char;
                                }
                            } else {
                                $jsonStr .= $char;
                            }
                        }
                        
                        if ($bracketCount === 0 && !empty($jsonStr)) {
                            $decoded = json_decode($jsonStr, true);
                            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                                $expertResultsArray = $decoded;
                            }
                        }
                    }
                }
            }
            
            // Отладочная информация если не удалось распарсить
            if (!$expertResultsArray) {
                echo "  ⚠️ DEBUG batch {$handleData['batch_index']}: Failed to parse JSON\n";
                echo "  ⚠️ DEBUG: JSON error: " . json_last_error_msg() . "\n";
                echo "  ⚠️ DEBUG: Content length: " . strlen($cleanedContent) . " chars\n";
                $preview = substr($cleanedContent, 0, 500);
                echo "  ⚠️ DEBUG: Content preview: " . $preview . "\n";
                // Пробуем найти хотя бы начало массива
                if (strpos($cleanedContent, '[') !== false) {
                    $arrayStart = strpos($cleanedContent, '[');
                    $arrayPreview = substr($cleanedContent, $arrayStart, min(1000, strlen($cleanedContent) - $arrayStart));
                    echo "  ⚠️ DEBUG: Array preview from position {$arrayStart}: " . $arrayPreview . "\n";
                }
            } else {
                echo "  ✓ DEBUG batch {$handleData['batch_index']}: Successfully parsed " . count($expertResultsArray) . " results\n";
            }
        } else {
            error_log("DEBUG expert: HTTP error {$httpCode} for batch {$handleData['batch_index']}, response: " . substr($response ?? '', 0, 500));
        }
        
        if ($expertResultsArray && is_array($expertResultsArray) && count($expertResultsArray) > 0) {
            echo "  ✓ DEBUG batch {$handleData['batch_index']}: Parsed " . count($expertResultsArray) . " results\n";
            // Распределяем результаты по индексам
            $batchIndices = $handleData['indices'];
            $batchIndicesArray = array_values($batchIndices); // Преобразуем в массив с последовательными индексами
            $matchedCount = 0;
            
            foreach ($expertResultsArray as $resultIndex => $result) {
                if (isset($result['question_number'])) {
                    // Используем question_number для сопоставления
                    $questionNum = (int)$result['question_number'];
                    if ($questionNum >= 1 && $questionNum <= count($batchIndicesArray)) {
                        $originalIndex = $batchIndicesArray[$questionNum - 1];
                        if (isset($result['score'])) {
                            $expertResults[$originalIndex] = $result;
                            $matchedCount++;
                            echo "    ✓ Matched question_number {$questionNum} to original_index {$originalIndex} (score: {$result['score']})\n";
                        } else {
                            echo "    ⚠️ Question {$questionNum} has no score field\n";
                        }
                    } else {
                        echo "    ⚠️ Question number {$questionNum} out of range (1-" . count($batchIndicesArray) . ")\n";
                    }
                } elseif (isset($batchIndicesArray[$resultIndex])) {
                    // Если нет question_number, используем порядковый номер в массиве результатов
                    $originalIndex = $batchIndicesArray[$resultIndex];
                    if (isset($result['score'])) {
                        $expertResults[$originalIndex] = $result;
                        $matchedCount++;
                        echo "    ✓ Matched result index {$resultIndex} to original_index {$originalIndex} (score: {$result['score']})\n";
                    }
                } else {
                    echo "    ⚠️ Result index {$resultIndex} out of range for batch indices\n";
                }
            }
            
            echo "  ✓ DEBUG batch {$handleData['batch_index']}: Matched {$matchedCount} out of " . count($expertResultsArray) . " results\n";
        } else {
            // Ошибка - сохраняем для повторной попытки
            echo "  ⚠️ DEBUG batch {$handleData['batch_index']}: No valid results array, adding to retry list\n";
            $failedBatches[] = $handleData['batch_index'];
        }
        
        curl_multi_remove_handle($multiHandle, $handleData['handle']);
        curl_close($handleData['handle']);
    }
    
    // Повторные попытки для неудачных батчей (до 3 раз)
    if (!empty($failedBatches)) {
        $maxRetries = 3;
        $retryCount = 0;
        
        while (!empty($failedBatches) && $retryCount < $maxRetries) {
            $retryCount++;
            echo "  → Повторная попытка оценки AI (попытка {$retryCount}/{$maxRetries}) для " . count($failedBatches) . " батчей...\n";
            
            $retryMultiHandle = curl_multi_init();
            $retryHandles = [];
            
            foreach ($failedBatches as $batchIndex) {
                $promptData = $expertPrompts[$batchIndex];
                $expertApiData = [
                    'model' => 'xiaomi/mimo-v2-flash:free',
                    'messages' => [
                        ['role' => 'user', 'content' => $promptData['prompt']]
                    ]
                ];
                
                $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($expertApiData));
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Content-Type: application/json',
                    'Authorization: Bearer ' . OPENROUTER_API_KEY,
                    'HTTP-Referer: https://wtest.medmaps.ru',
                    'X-Title: Medical Widget Tester'
                ]);
                curl_setopt($ch, CURLOPT_TIMEOUT, 120);
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
                
                $retryHandles[(int)$ch] = [
                    'handle' => $ch,
                    'batch_index' => $batchIndex,
                    'indices' => $promptData['indices']
                ];
                
                curl_multi_add_handle($retryMultiHandle, $ch);
            }
            
            // Выполняем повторные запросы
            $running = null;
            do {
                curl_multi_exec($retryMultiHandle, $running);
                curl_multi_select($retryMultiHandle, 0.1);
            } while ($running > 0);
            
            // Обрабатываем результаты повторных попыток
            $stillFailed = [];
            foreach ($retryHandles as $handleId => $handleData) {
                $response = curl_multi_getcontent($handleData['handle']);
                $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
                
                $expertResultsArray = null;
                if ($httpCode === 200 && $response) {
                    $expertDecoded = json_decode($response, true);
                    $expertContent = $expertDecoded['choices'][0]['message']['content'] ?? '';
                    
                    if (preg_match('/\[[^\]]*(?:\{[^{}]*\}[^\]]*)*\]/s', $expertContent, $matches)) {
                        $expertResultsArray = json_decode($matches[0], true);
                    } elseif (preg_match('/\{[^{}]*(?:\{[^{}]*\}[^{}]*)*\}/s', $expertContent, $matches)) {
                        $decoded = json_decode($matches[0], true);
                        if (isset($decoded['results']) && is_array($decoded['results'])) {
                            $expertResultsArray = $decoded['results'];
                        }
                    }
                }
                
                if ($expertResultsArray && is_array($expertResultsArray) && count($expertResultsArray) > 0) {
                    // Распределяем результаты по индексам
                    $batchIndices = $handleData['indices'];
                    foreach ($expertResultsArray as $resultIndex => $result) {
                        if (isset($result['question_number'])) {
                            $questionNum = $result['question_number'];
                            if ($questionNum >= 1 && $questionNum <= count($batchIndices)) {
                                $originalIndex = array_values($batchIndices)[$questionNum - 1];
                                if (isset($result['score'])) {
                                    $expertResults[$originalIndex] = $result;
                                }
                            }
                        } elseif (isset($batchIndices[$resultIndex])) {
                            $originalIndex = array_values($batchIndices)[$resultIndex];
                            if (isset($result['score'])) {
                                $expertResults[$originalIndex] = $result;
                            }
                        }
                    }
                } else {
                    $stillFailed[] = $handleData['batch_index'];
                }
                
                curl_multi_remove_handle($retryMultiHandle, $handleData['handle']);
                curl_close($handleData['handle']);
            }
            
            curl_multi_close($retryMultiHandle);
            $failedBatches = $stillFailed;
            
            // Небольшая задержка перед следующей попыткой
            if (!empty($failedBatches) && $retryCount < $maxRetries) {
                usleep(500000); // 0.5 секунды
            }
        }
    }
    
    curl_multi_close($multiHandle);
    
    // Заполняем пустые результаты значениями по умолчанию
    foreach ($batch as $index => $query) {
        if (!isset($expertResults[$index])) {
            $expertResults[$index] = [
                'score' => 0,
                'verdict' => 'УЖАСНО',
                'is_relevant' => false,
                'is_complete' => false,
                'missing_specializations' => [],
                'extra_specializations' => [],
                'missing_articles' => [],
                'extra_articles' => [],
                'comment' => 'Ошибка оценки AI',
                'issues' => ['AI эксперт не ответил']
            ];
        }
    }
    
    // ===== 5. ВЫВОД РЕЗУЛЬТАТОВ БАТЧА =====
    foreach ($batch as $index => $query) {
        $globalQuestionIndex++;
        $num = $globalQuestionIndex;
        
        echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
        echo "ВОПРОС {$num}/{$total}: {$query}\n";
        echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
        
        $data = $expertData[$index];
        $returnedSpecializations = $data['returned_specializations'];
        $returnedArticles = $data['returned_articles'];
        $returnedSpecializationsCount = $data['returned_specializations_count'];
        $returnedArticlesCount = $data['returned_articles_count'];
        
        $hasSpecializations = !empty($returnedSpecializations);
        $hasArticles = !empty($returnedArticles);
        
        if ($hasSpecializations) {
            $stats['with_specializations']++;
        } else {
            $stats['without_specializations']++;
        }
        
        if ($hasArticles) {
            $stats['with_articles']++;
        } else {
            $stats['without_articles']++;
        }
        
        echo "  → Специализации получены: " . ($hasSpecializations ? "да ({$returnedSpecializationsCount} специализаций)" : "НЕТ") . "\n";
        if ($hasSpecializations) {
            $specNames = [];
            foreach ($returnedSpecializations as $spec) {
                $name = $spec['name'] ?? 'Неизвестно';
                $specNames[] = $name;
            }
            echo "  → Возвращенные специализации: " . implode(", ", array_slice($specNames, 0, 5));
            if (count($specNames) > 5) {
                echo " и еще " . (count($specNames) - 5);
            }
            echo "\n";
        }
        
        echo "  → Статьи получены: " . ($hasArticles ? "да ({$returnedArticlesCount} статей)" : "НЕТ") . "\n";
        if ($hasArticles) {
            $articleTitles = [];
            foreach ($returnedArticles as $article) {
                $title = $article['title'] ?? 'Неизвестно';
                $articleTitles[] = $title;
            }
            echo "  → Возвращенные статьи: " . implode(", ", array_slice($articleTitles, 0, 5));
            if (count($articleTitles) > 5) {
                echo " и еще " . (count($articleTitles) - 5);
            }
            echo "\n";
        }
        echo "\n";
        
        // Результат эксперта
        $expertResult = $expertResults[$index] ?? [
            'score' => 0,
            'verdict' => 'УЖАСНО',
            'comment' => 'Ошибка получения оценки',
            'issues' => [],
            'missing_specializations' => [],
            'extra_specializations' => [],
            'missing_articles' => [],
            'extra_articles' => []
        ];
        
        $score = $expertResult['score'];
        $verdict = $expertResult['verdict'];
        
        // Обновляем статистику
        $stats['total_score'] += $score;
        if ($score >= 9) {
            $stats['excellent']++;
        } elseif ($score >= 7) {
            $stats['good']++;
        } elseif ($score >= 4) {
            $stats['poor']++;
        } else {
            $stats['bad']++;
        }
        
        // Определяем эмодзи для вердикта
        $emoji = $verdictEmoji[$verdict] ?? '❓';
        
        echo "  → Оценка AI-экспертом получена\n\n";
        echo "  {$emoji} ОЦЕНКА: {$score}/10 | ВЕРДИКТ: {$verdict}\n";
        echo "  💬 Комментарий: {$expertResult['comment']}\n";
        
        // Выводим детали оценки
        if (isset($expertResult['is_relevant'])) {
            echo "  🔍 Релевантность: " . ($expertResult['is_relevant'] ? "ДА" : "НЕТ") . "\n";
        }
        if (isset($expertResult['is_complete'])) {
            echo "  ✅ Полнота: " . ($expertResult['is_complete'] ? "ДА" : "НЕТ") . "\n";
        }
        if (!empty($expertResult['missing_specializations']) && is_array($expertResult['missing_specializations'])) {
            echo "  ⚠️ Недостающие специализации: " . implode(", ", $expertResult['missing_specializations']) . "\n";
        }
        if (!empty($expertResult['extra_specializations']) && is_array($expertResult['extra_specializations'])) {
            echo "  ⚠️ Лишние специализации: " . implode(", ", $expertResult['extra_specializations']) . "\n";
        }
        if (!empty($expertResult['missing_articles']) && is_array($expertResult['missing_articles'])) {
            echo "  ⚠️ Недостающие статьи: " . implode(", ", $expertResult['missing_articles']) . "\n";
        }
        if (!empty($expertResult['extra_articles']) && is_array($expertResult['extra_articles'])) {
            echo "  ⚠️ Лишние статьи: " . implode(", ", $expertResult['extra_articles']) . "\n";
        }
        if (!empty($expertResult['issues'])) {
            echo "  ⚠️ Проблемы: " . implode(", ", $expertResult['issues']) . "\n";
        }
        echo "\n";
        
        // Прогресс
        $elapsed = round(microtime(true) - $startTime, 1);
        $avgTime = $elapsed / $num;
        $remaining = round(($total - $num) * $avgTime / 60, 1);
        echo "  📊 Прогресс: {$num}/{$total} ({$elapsed} сек) | ~{$remaining} мин. осталось\n\n";
        
        // Сохраняем результат
        $results[] = [
            'num' => $num,
            'query' => $query,
            'returned_specializations' => $returnedSpecializations,
            'returned_articles' => $returnedArticles,
            'returned_specializations_count' => $returnedSpecializationsCount,
            'returned_articles_count' => $returnedArticlesCount,
            'score' => $score,
            'verdict' => $verdict,
            'expert_details' => $expertResult
        ];
        
        // Небольшая пауза между запросами
        usleep(500000); // 0.5 секунды
    }
}

$endTime = microtime(true);
$totalTime = round($endTime - $startTime, 1);
$avgScore = round($stats['total_score'] / $total, 1);

// ===== ИТОГОВАЯ СТАТИСТИКА =====
echo "\n";
echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║                        ИТОГОВАЯ СТАТИСТИКА                               ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

echo "Всего вопросов: {$total}\n";
echo "Время выполнения: {$totalTime} сек (" . round($totalTime / 60, 1) . " мин)\n";
echo "Средняя скорость: " . round($total / ($totalTime / 60), 1) . " вопросов/мин\n\n";

echo "ОЦЕНКИ AI-ЭКСПЕРТА (адекватность specializations и articles):\n";
echo "  ⭐ ОТЛИЧНО (9-10): {$stats['excellent']} (" . round($stats['excellent'] / $total * 100, 1) . "%)\n";
echo "  ✅ ХОРОШО (7-8): {$stats['good']} (" . round($stats['good'] / $total * 100, 1) . "%)\n";
echo "  ⚠️ ПЛОХО (4-6): {$stats['poor']} (" . round($stats['poor'] / $total * 100, 1) . "%)\n";
echo "  ❌ УЖАСНО (0-3): {$stats['bad']} (" . round($stats['bad'] / $total * 100, 1) . "%)\n";
echo "  📊 Средний балл: {$avgScore}/10\n\n";

echo "ДАННЫЕ:\n";
echo "  Со specializations: {$stats['with_specializations']} (" . round($stats['with_specializations'] / $total * 100, 1) . "%)\n";
echo "  Без specializations: {$stats['without_specializations']} (" . round($stats['without_specializations'] / $total * 100, 1) . "%)\n";
echo "  Со articles: {$stats['with_articles']} (" . round($stats['with_articles'] / $total * 100, 1) . "%)\n";
echo "  Без articles: {$stats['without_articles']} (" . round($stats['without_articles'] / $total * 100, 1) . "%)\n";
echo "  Всего specializations в БД: {$totalSpecializationsInDB}\n";
echo "  Всего articles в БД: {$totalArticlesInDB}\n\n";

// Топ-10 лучших
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "ТОП-10 ЛУЧШИХ РЕЗУЛЬТАТОВ:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

usort($results, function($a, $b) { return $b['score'] - $a['score']; });
$top10 = array_slice($results, 0, 10);

foreach ($top10 as $result) {
    $emoji = $verdictEmoji[$result['verdict']] ?? '❓';
    echo "  {$emoji} {$result['score']}/10 - {$result['query']}\n";
    echo "     Возвращено специализаций: {$result['returned_specializations_count']}, статей: {$result['returned_articles_count']}\n";
    echo "     {$result['expert_details']['comment']}\n\n";
}

// Топ-10 худших
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "ТОП-10 ПРОБЛЕМНЫХ ВОПРОСОВ:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$bottom10 = array_slice(array_reverse($results), 0, 10);

foreach ($bottom10 as $result) {
    $emoji = $verdictEmoji[$result['verdict']] ?? '❓';
    echo "  {$emoji} {$result['score']}/10 - {$result['query']}\n";
    echo "     Возвращено специализаций: {$result['returned_specializations_count']}, статей: {$result['returned_articles_count']}\n";
    if (!empty($result['expert_details']['missing_specializations'])) {
        echo "     Недостающие специализации: " . implode(', ', $result['expert_details']['missing_specializations']) . "\n";
    }
    if (!empty($result['expert_details']['extra_specializations'])) {
        echo "     Лишние специализации: " . implode(', ', $result['expert_details']['extra_specializations']) . "\n";
    }
    if (!empty($result['expert_details']['missing_articles'])) {
        echo "     Недостающие статьи: " . implode(', ', $result['expert_details']['missing_articles']) . "\n";
    }
    if (!empty($result['expert_details']['extra_articles'])) {
        echo "     Лишние статьи: " . implode(', ', $result['expert_details']['extra_articles']) . "\n";
    }
    if (!empty($result['expert_details']['issues'])) {
        echo "     Проблемы: " . implode(', ', $result['expert_details']['issues']) . "\n";
    }
    echo "\n";
}

// Сохраняем результаты
$timestamp = date('Y-m-d_H-i-s');
$jsonFile = "/root/test-specializations-articles-results-{$timestamp}.json";
$reportFile = "/root/test-specializations-articles-report-{$timestamp}.txt";

file_put_contents($jsonFile, json_encode($results, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

$report = "ТЕСТ АДЕКВАТНОСТИ ВЫВОДА SPECIALIZATIONS И ARTICLES ВИДЖЕТА\n";
$report .= "Дата: " . date('Y-m-d H:i:s') . "\n";
$report .= "Сервер: wtest.medmaps.ru\n";
$report .= "API: query-section.php\n";
$report .= "Эксперт: Mimo v2 Flash\n\n";
$report .= "Всего вопросов: {$total}\n";
$report .= "Время: {$totalTime} сек\n";
$report .= "Средний балл: {$avgScore}/10\n\n";
$report .= "ОТЛИЧНО: {$stats['excellent']}\n";
$report .= "ХОРОШО: {$stats['good']}\n";
$report .= "ПЛОХО: {$stats['poor']}\n";
$report .= "УЖАСНО: {$stats['bad']}\n\n";
$report .= "Со specializations: {$stats['with_specializations']}\n";
$report .= "Без specializations: {$stats['without_specializations']}\n";
$report .= "Со articles: {$stats['with_articles']}\n";
$report .= "Без articles: {$stats['without_articles']}\n";
$report .= "Всего specializations в БД: {$totalSpecializationsInDB}\n";
$report .= "Всего articles в БД: {$totalArticlesInDB}\n";

file_put_contents($reportFile, $report);

echo "\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "Результаты сохранены:\n";
echo "  JSON: {$jsonFile}\n";
echo "  Отчет: {$reportFile}\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
