<?php
/**
 * Тест виджета - проверка адекватности вывода specialists
 * - Получение specialists через query-section.php
 * - Оценка адекватности через AI эксперта (OpenRouter)
 * - Тестовый сервер: wtest.medmaps.ru
 */

// Используем полный список вопросов для тестирования (310 вопросов, ~3.8 часа)
require '/root/questions-list.php';  // Полный список - загружает $questions

// Для быстрого тестирования раскомментируйте строку ниже (34 вопроса, ~1.5 мин):
// require '/root/questions-list-problematic.php';  // Проблемные вопросы - загружает $questions

require_once '/root/config.php';

$widgetKey = 'ekamedcenter';

// Ограничиваем тест случайными 30 фразами
// $totalQuestions = count($questions);
// $randomKeys = array_rand($questions, min(10, $totalQuestions));
// if (!is_array($randomKeys)) {
//     $randomKeys = [$randomKeys];
// }
// $randomQuestions = [];
// foreach ($randomKeys as $key) {
//     $randomQuestions[$key] = $questions[$key];
// }
// $questions = $randomQuestions;

$total = count($questions);

echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║  ТЕСТ SPECIALISTS ({$total} ВОПРОСОВ) - ПРОВЕРКА АДЕКВАТНОСТИ         ║\n";
echo "║  Сервер: wtest.medmaps.ru | API: query-section.php | Эксперт: Mimo    ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

// Получаем widget_id
$db = getDatabase();
$stmt = $db->prepare("SELECT id FROM widgets WHERE widget_key = ?");
$stmt->execute([$widgetKey]);
$widget = $stmt->fetch();
if (!$widget) {
    die("Ошибка: виджет '{$widgetKey}' не найден в базе данных\n");
}
$widgetId = $widget['id'];

// Получаем полный список всех specialists из БД
echo "→ Загрузка списка всех specialists из базы данных...\n";
function getAllSpecialistsFromDB($widgetId) {
    $db = getDatabase();
    $stmt = $db->prepare("
        SELECT DISTINCT pi.id, pf.field_name, pf.field_value
        FROM parsed_items pi
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.widget_id = ? 
        AND pi.section_name = 'specialists'
        AND pi.is_duplicate = 0
        ORDER BY pi.id, pf.field_name
    ");
    $stmt->execute([$widgetId]);
    
    $specialists = [];
    while ($row = $stmt->fetch()) {
        $itemId = (int)$row['id'];
        if (!isset($specialists[$itemId])) {
            $specialists[$itemId] = ['id' => $itemId];
        }
        $specialists[$itemId][$row['field_name']] = $row['field_value'];
    }
    
    return array_values($specialists);
}

$allSpecialists = getAllSpecialistsFromDB($widgetId);
$totalSpecialistsInDB = count($allSpecialists);
echo "  ✓ Загружено {$totalSpecialistsInDB} specialists из базы данных\n\n";

// Формируем текстовое представление всех specialists для промпта
function formatSpecialistsForPrompt($specialists) {
    $formatted = [];
    foreach ($specialists as $index => $spec) {
        $parts = [];
        if (isset($spec['name'])) {
            $parts[] = "Имя: " . $spec['name'];
        }
        if (isset($spec['specialization'])) {
            $parts[] = "Специализация: " . $spec['specialization'];
        }
        if (isset($spec['description'])) {
            $parts[] = "Описание: " . mb_substr($spec['description'], 0, 100);
        }
        $formatted[] = ($index + 1) . ". " . implode(" | ", $parts);
    }
    return implode("\n", $formatted);
}

$allSpecialistsText = formatSpecialistsForPrompt($allSpecialists);

$results = [];
$stats = [
    'total' => $total,
    'excellent' => 0,  // 9-10
    'good' => 0,       // 7-8
    'poor' => 0,       // 4-6
    'bad' => 0,        // 0-3
    'total_score' => 0,
    'with_specialists' => 0,
    'without_specialists' => 0
];

$startTime = microtime(true);

// Определяем эмодзи для вердикта (используется в итоговой статистике)
$verdictEmoji = [
    'ОТЛИЧНО' => '⭐',
    'ХОРОШО' => '✅',
    'ПЛОХО' => '⚠️',
    'УЖАСНО' => '❌'
];

// Разбиваем вопросы на батчи по 30 для параллельной обработки
$batchSize = 30;
$questionBatches = array_chunk($questions, $batchSize, true); // Сохраняем ключи
$globalQuestionIndex = 0; // Глобальный счетчик вопросов

foreach ($questionBatches as $batchIndex => $batch) {
    $batchNum = $batchIndex + 1;
    $totalBatches = count($questionBatches);
    echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "БАТЧ {$batchNum}/{$totalBatches} (обработка " . count($batch) . " вопросов параллельно)\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    // ===== 1. ПОДГОТОВКА ПАРАЛЛЕЛЬНЫХ ЗАПРОСОВ ДЛЯ SPECIALISTS (query-section.php) =====
    echo "  → Отправка запросов к query-section.php для получения specialists (параллельно)...\n";
    
    $multiHandle = curl_multi_init();
    $handles = [];
    
    foreach ($batch as $index => $query) {
        $sectionData = [
            'widget_key' => $widgetKey,
            'question' => $query,
            'section_name' => 'specialists'
        ];
        
        $ch = curl_init('https://wtest.medmaps.ru/api/query-section.php');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($sectionData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
        
        $handleId = (int)$ch;
        $handles[$handleId] = [
            'handle' => $ch,
            'type' => 'section',
            'index' => $index,
            'query' => $query
        ];
        
        curl_multi_add_handle($multiHandle, $ch);
    }
    
    // Выполняем параллельные запросы
    $running = null;
    do {
        curl_multi_exec($multiHandle, $running);
        curl_multi_select($multiHandle, 0.1);
    } while ($running > 0);
    
    // Собираем результаты
    $sectionResults = [];
    foreach ($handles as $handleId => $handleData) {
        if ($handleData['type'] === 'section') {
            $response = curl_multi_getcontent($handleData['handle']);
            $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
            
            $specialists = [];
            $text = '';
            
            if ($httpCode === 200 && $response) {
                $response = trim($response);
                $response = preg_replace('/^\xEF\xBB\xBF/', '', $response);
                $decoded = json_decode($response, true);
                
                if ($decoded && isset($decoded['data']['specialists'])) {
                    $specialists = $decoded['data']['specialists'];
                }
                if ($decoded && isset($decoded['text'])) {
                    $text = $decoded['text'];
                }
            }
            
            $sectionResults[$handleData['index']] = [
                'specialists' => $specialists,
                'text' => $text,
                'query' => $handleData['query']
            ];
            
            curl_multi_remove_handle($multiHandle, $handleData['handle']);
            curl_close($handleData['handle']);
        }
    }
    
    curl_multi_close($multiHandle);
    
    // ===== 2. ПОДГОТОВКА ДАННЫХ ДЛЯ AI-ЭКСПЕРТА (проверка адекватности specialists) =====
    $expertData = [];
    foreach ($batch as $index => $query) {
        $sectionResult = $sectionResults[$index] ?? ['specialists' => [], 'text' => '', 'query' => $query];
        $returnedSpecialists = $sectionResult['specialists'];
        
        $expertData[$index] = [
            'query' => $query,
            'returned_specialists' => $returnedSpecialists,
            'returned_count' => count($returnedSpecialists)
        ];
    }
    
    // ===== 3. ПАРАЛЛЕЛЬНЫЕ ЗАПРОСЫ К AI-ЭКСПЕРТУ (батчами по 10, все параллельно) =====
    // Разбиваем данные для эксперта на батчи по 10
    $expertBatchSize = 10;
    $expertBatches = array_chunk($expertData, $expertBatchSize, true);
    $expertResults = []; // Инициализируем массив результатов эксперта
    
    echo "  → Подготовка запросов к AI-эксперту (" . count($expertBatches) . " батчей по 10 вопросов)...\n";
    
    // Создаем промпты для всех батчей
    $expertPrompts = [];
    foreach ($expertBatches as $expertBatchIndex => $expertBatch) {
        $questionsData = [];
        $questionNumInBatch = 0;
        foreach ($expertBatch as $index => $data) {
            $questionNumInBatch++;
            $returnedSpecialists = $data['returned_specialists'];
            
            // Форматируем возвращенных specialists для промпта
            $returnedSpecialistsText = '';
            if (!empty($returnedSpecialists)) {
                $returnedSpecialistsText = formatSpecialistsForPrompt($returnedSpecialists);
            } else {
                $returnedSpecialistsText = "(специалисты не возвращены)";
            }
            
            $questionsData[] = [
                'question_number' => $questionNumInBatch,
                'original_index' => $index,
                'query' => $data['query'],
                'returned_specialists_count' => count($returnedSpecialists),
                'returned_specialists' => $returnedSpecialistsText
            ];
        }
        
        // Формируем один промпт для всех 10 вопросов
        $questionsText = "";
        foreach ($questionsData as $qData) {
            $questionsText .= "\n\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
            $questionsText .= "ВОПРОС #{$qData['question_number']}:\n";
            $questionsText .= "\"{$qData['query']}\"\n\n";
            $questionsText .= "СПЕЦИАЛИСТЫ, ВОЗВРАЩЕННЫЕ ВИДЖЕТОМ ({$qData['returned_specialists_count']} специалистов):\n";
            $questionsText .= "{$qData['returned_specialists']}\n";
        }
        
        $expertPrompt = "Ты - строгий эксперт по медицинским виджетам. Проверь АДЕКВАТНОСТЬ вывода СПЕЦИАЛИСТОВ виджетом медицинской клиники для 10 вопросов.

ПОЛНЫЙ СПИСОК ВСЕХ СПЕЦИАЛИСТОВ В БАЗЕ ДАННЫХ КЛИНИКИ ({$totalSpecialistsInDB} специалистов):
{$allSpecialistsText}
{$questionsText}

ТВОЯ ЗАДАЧА - проверить адекватность вывода specialists для КАЖДОГО из 10 вопросов на три критерия:

1. РЕЛЕВАНТНОСТЬ:
   Проверь, что все возвращенные специалисты релевантны вопросу пользователя.
   Например, для вопроса о боли в спине должны возвращаться неврологи, травматологи-ортопеды, но НЕ офтальмологи или гинекологи.
   Если вернулись нерелевантные специалисты - это проблема.

2. ПОЛНОТА:
   Проверь, что вернулись все необходимые специалисты для данного вопроса.
   Используй полный список специалистов из базы данных как справочник.
   Если для вопроса о гастрите должны быть гастроэнтеролог и терапевт, но вернулся только терапевт - это проблема.
   Если специалисты вообще не вернулись, но должны были - это критическая проблема.

3. ОТСУТСТВИЕ ЛИШНИХ:
   Проверь, что не вернулись специалисты, которые не нужны для данного вопроса.
   Например, для вопроса о простуде не должны возвращаться хирурги или гинекологи.

ОЦЕНКА (0-10 баллов) для каждого вопроса:
- 10: все специалисты релевантны, полный список необходимых специалистов, нет лишних
- 8-9: специалисты релевантны, но может не хватать 1-2 важных или есть 1 лишний
- 6-7: специалисты в целом релевантны, но есть заметные проблемы (не хватает важных или есть несколько лишних)
- 4-5: есть нерелевантные специалисты или критически не хватает важных
- 0-3: специалисты не релевантны вопросу или специалисты не возвращены, хотя должны были

ФОРМАТ ОТВЕТА (строго JSON массив из 10 элементов):
[
  {
    \"question_number\": 1,
    \"score\": <число 0-10>,
    \"verdict\": \"ОТЛИЧНО\" | \"ХОРОШО\" | \"ПЛОХО\" | \"УЖАСНО\",
    \"is_relevant\": <true/false>,
    \"is_complete\": <true/false>,
    \"missing_specialists\": [\"<список недостающих специалистов, если есть>\", \"<специализация или имя>\", ...],
    \"extra_specialists\": [\"<список лишних специалистов, если есть>\", \"<специализация или имя>\", ...],
    \"comment\": \"<краткий комментарий о найденных проблемах>\",
    \"issues\": [\"<список проблем, если есть>\"]
  },
  {
    \"question_number\": 2,
    ...
  },
  ...
  {
    \"question_number\": 10,
    ...
  }
]

ВЕРДИКТЫ:
- ОТЛИЧНО: 9-10 баллов (все специалисты релевантны и полны)
- ХОРОШО: 7-8 баллов (незначительные проблемы)
- ПЛОХО: 4-6 баллов (заметные проблемы)
- УЖАСНО: 0-3 балла (критичные проблемы или специалисты не возвращены)

Отвечай ТОЛЬКО валидным JSON массивом, без дополнительного текста!";
        
        $expertPrompts[$expertBatchIndex] = [
            'prompt' => $expertPrompt,
            'batch_index' => $expertBatchIndex,
            'indices' => array_keys($expertBatch) // Сохраняем оригинальные индексы
        ];
    }
    
    // Отправляем все запросы параллельно
    echo "  → Отправка " . count($expertPrompts) . " запросов к AI-эксперту параллельно (по 10 вопросов в каждом)...\n";
    
    $multiHandle = curl_multi_init();
    $expertHandles = [];
    
    foreach ($expertPrompts as $batchIndex => $promptData) {
        $expertApiData = [
            'model' => 'xiaomi/mimo-v2-flash:free',
            'messages' => [
                ['role' => 'user', 'content' => $promptData['prompt']]
            ]
        ];
        
        $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($expertApiData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . OPENROUTER_API_KEY,
            'HTTP-Referer: https://wtest.medmaps.ru',
            'X-Title: Medical Widget Tester'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 120); // Увеличиваем таймаут для больших промптов
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        
        $handleId = (int)$ch;
        $expertHandles[$handleId] = [
            'handle' => $ch,
            'batch_index' => $batchIndex,
            'indices' => $promptData['indices']
        ];
        
        curl_multi_add_handle($multiHandle, $ch);
    }
    
    // Выполняем параллельные запросы к эксперту
    $running = null;
    do {
        curl_multi_exec($multiHandle, $running);
        curl_multi_select($multiHandle, 0.1);
    } while ($running > 0);
    
    // Собираем результаты эксперта
    $failedBatches = [];
    
    foreach ($expertHandles as $handleId => $handleData) {
        $response = curl_multi_getcontent($handleData['handle']);
        $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
        
        $expertResultsArray = null;
        if ($httpCode === 200 && $response) {
            $expertDecoded = json_decode($response, true);
            $expertContent = $expertDecoded['choices'][0]['message']['content'] ?? '';
            
            // Пытаемся найти JSON массив
            if (preg_match('/\[[^\]]*(?:\{[^{}]*\}[^\]]*)*\]/s', $expertContent, $matches)) {
                $expertResultsArray = json_decode($matches[0], true);
            } elseif (preg_match('/\{[^{}]*(?:\{[^{}]*\}[^{}]*)*\}/s', $expertContent, $matches)) {
                // Если вернулся объект вместо массива, пробуем распарсить
                $decoded = json_decode($matches[0], true);
                if (isset($decoded['results']) && is_array($decoded['results'])) {
                    $expertResultsArray = $decoded['results'];
                }
            }
        }
        
        if ($expertResultsArray && is_array($expertResultsArray) && count($expertResultsArray) > 0) {
            // Распределяем результаты по индексам
            $batchIndices = $handleData['indices'];
            foreach ($expertResultsArray as $resultIndex => $result) {
                if (isset($result['question_number'])) {
                    // Используем question_number для сопоставления
                    $questionNum = $result['question_number'];
                    if ($questionNum >= 1 && $questionNum <= count($batchIndices)) {
                        $originalIndex = array_values($batchIndices)[$questionNum - 1];
                        if (isset($result['score'])) {
                            $expertResults[$originalIndex] = $result;
                        }
                    }
                } elseif (isset($batchIndices[$resultIndex])) {
                    // Если нет question_number, используем порядковый номер
                    $originalIndex = array_values($batchIndices)[$resultIndex];
                    if (isset($result['score'])) {
                        $expertResults[$originalIndex] = $result;
                    }
                }
            }
        } else {
            // Ошибка - сохраняем для повторной попытки
            $failedBatches[] = $handleData['batch_index'];
        }
        
        curl_multi_remove_handle($multiHandle, $handleData['handle']);
        curl_close($handleData['handle']);
    }
    
    // Повторные попытки для неудачных батчей (до 3 раз)
    if (!empty($failedBatches)) {
        $maxRetries = 3;
        $retryCount = 0;
        
        while (!empty($failedBatches) && $retryCount < $maxRetries) {
            $retryCount++;
            echo "  → Повторная попытка оценки AI (попытка {$retryCount}/{$maxRetries}) для " . count($failedBatches) . " батчей...\n";
            
            $retryMultiHandle = curl_multi_init();
            $retryHandles = [];
            
            foreach ($failedBatches as $batchIndex) {
                $promptData = $expertPrompts[$batchIndex];
                $expertApiData = [
                    'model' => 'xiaomi/mimo-v2-flash:free',
                    'messages' => [
                        ['role' => 'user', 'content' => $promptData['prompt']]
                    ]
                ];
                
                $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($expertApiData));
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Content-Type: application/json',
                    'Authorization: Bearer ' . OPENROUTER_API_KEY,
                    'HTTP-Referer: https://wtest.medmaps.ru',
                    'X-Title: Medical Widget Tester'
                ]);
                curl_setopt($ch, CURLOPT_TIMEOUT, 120);
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
                
                $retryHandles[(int)$ch] = [
                    'handle' => $ch,
                    'batch_index' => $batchIndex,
                    'indices' => $promptData['indices']
                ];
                
                curl_multi_add_handle($retryMultiHandle, $ch);
            }
            
            // Выполняем повторные запросы
            $running = null;
            do {
                curl_multi_exec($retryMultiHandle, $running);
                curl_multi_select($retryMultiHandle, 0.1);
            } while ($running > 0);
            
            // Обрабатываем результаты повторных попыток
            $stillFailed = [];
            foreach ($retryHandles as $handleId => $handleData) {
                $response = curl_multi_getcontent($handleData['handle']);
                $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
                
                $expertResultsArray = null;
                if ($httpCode === 200 && $response) {
                    $expertDecoded = json_decode($response, true);
                    $expertContent = $expertDecoded['choices'][0]['message']['content'] ?? '';
                    
                    if (preg_match('/\[[^\]]*(?:\{[^{}]*\}[^\]]*)*\]/s', $expertContent, $matches)) {
                        $expertResultsArray = json_decode($matches[0], true);
                    } elseif (preg_match('/\{[^{}]*(?:\{[^{}]*\}[^{}]*)*\}/s', $expertContent, $matches)) {
                        $decoded = json_decode($matches[0], true);
                        if (isset($decoded['results']) && is_array($decoded['results'])) {
                            $expertResultsArray = $decoded['results'];
                        }
                    }
                }
                
                if ($expertResultsArray && is_array($expertResultsArray) && count($expertResultsArray) > 0) {
                    // Распределяем результаты по индексам
                    $batchIndices = $handleData['indices'];
                    foreach ($expertResultsArray as $resultIndex => $result) {
                        if (isset($result['question_number'])) {
                            $questionNum = $result['question_number'];
                            if ($questionNum >= 1 && $questionNum <= count($batchIndices)) {
                                $originalIndex = array_values($batchIndices)[$questionNum - 1];
                                if (isset($result['score'])) {
                                    $expertResults[$originalIndex] = $result;
                                }
                            }
                        } elseif (isset($batchIndices[$resultIndex])) {
                            $originalIndex = array_values($batchIndices)[$resultIndex];
                            if (isset($result['score'])) {
                                $expertResults[$originalIndex] = $result;
                            }
                        }
                    }
                } else {
                    $stillFailed[] = $handleData['batch_index'];
                }
                
                curl_multi_remove_handle($retryMultiHandle, $handleData['handle']);
                curl_close($handleData['handle']);
            }
            
            curl_multi_close($retryMultiHandle);
            $failedBatches = $stillFailed;
            
            // Небольшая задержка перед следующей попыткой
            if (!empty($failedBatches) && $retryCount < $maxRetries) {
                usleep(500000); // 0.5 секунды
            }
        }
    }
    
    curl_multi_close($multiHandle);
    
    // Заполняем пустые результаты значениями по умолчанию
    foreach ($batch as $index => $query) {
        if (!isset($expertResults[$index])) {
            $expertResults[$index] = [
                'score' => 0,
                'verdict' => 'УЖАСНО',
                'is_relevant' => false,
                'is_complete' => false,
                'missing_specialists' => [],
                'extra_specialists' => [],
                'comment' => 'Ошибка оценки AI',
                'issues' => ['AI эксперт не ответил']
            ];
        }
    }
    
    // ===== 4. ВЫВОД РЕЗУЛЬТАТОВ БАТЧА =====
    foreach ($batch as $index => $query) {
        $globalQuestionIndex++;
        $num = $globalQuestionIndex;
        
        echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
        echo "ВОПРОС {$num}/{$total}: {$query}\n";
        echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
        
        $data = $expertData[$index];
        $returnedSpecialists = $data['returned_specialists'];
        $returnedCount = $data['returned_count'];
        
        $hasSpecialists = !empty($returnedSpecialists);
        if ($hasSpecialists) {
            $stats['with_specialists']++;
        } else {
            $stats['without_specialists']++;
        }
        
        echo "  → Специалисты получены: " . ($hasSpecialists ? "да ({$returnedCount} специалистов)" : "НЕТ") . "\n";
        if ($hasSpecialists) {
            $specNames = [];
            foreach ($returnedSpecialists as $spec) {
                $name = $spec['name'] ?? $spec['specialization'] ?? 'Неизвестно';
                $specNames[] = $name;
            }
            echo "  → Возвращенные специалисты: " . implode(", ", array_slice($specNames, 0, 5));
            if (count($specNames) > 5) {
                echo " и еще " . (count($specNames) - 5);
            }
            echo "\n";
        }
        echo "\n";
        
        // Результат эксперта
        $expertResult = $expertResults[$index] ?? [
            'score' => 0,
            'verdict' => 'УЖАСНО',
            'comment' => 'Ошибка получения оценки',
            'issues' => [],
            'missing_specialists' => [],
            'extra_specialists' => []
        ];
        
        $score = $expertResult['score'];
        $verdict = $expertResult['verdict'];
        
        // Обновляем статистику
        $stats['total_score'] += $score;
        if ($score >= 9) {
            $stats['excellent']++;
        } elseif ($score >= 7) {
            $stats['good']++;
        } elseif ($score >= 4) {
            $stats['poor']++;
        } else {
            $stats['bad']++;
        }
        
        // Определяем эмодзи для вердикта
        $emoji = $verdictEmoji[$verdict] ?? '❓';
        
        echo "  → Оценка AI-экспертом получена\n\n";
        echo "  {$emoji} ОЦЕНКА: {$score}/10 | ВЕРДИКТ: {$verdict}\n";
        echo "  💬 Комментарий: {$expertResult['comment']}\n";
        
        // Выводим детали оценки
        if (isset($expertResult['is_relevant'])) {
            echo "  🔍 Релевантность: " . ($expertResult['is_relevant'] ? "ДА" : "НЕТ") . "\n";
        }
        if (isset($expertResult['is_complete'])) {
            echo "  ✅ Полнота: " . ($expertResult['is_complete'] ? "ДА" : "НЕТ") . "\n";
        }
        if (!empty($expertResult['missing_specialists']) && is_array($expertResult['missing_specialists'])) {
            echo "  ⚠️ Недостающие специалисты: " . implode(", ", $expertResult['missing_specialists']) . "\n";
        }
        if (!empty($expertResult['extra_specialists']) && is_array($expertResult['extra_specialists'])) {
            echo "  ⚠️ Лишние специалисты: " . implode(", ", $expertResult['extra_specialists']) . "\n";
        }
        if (!empty($expertResult['issues'])) {
            echo "  ⚠️ Проблемы: " . implode(", ", $expertResult['issues']) . "\n";
        }
        echo "\n";
        
        // Прогресс
        $elapsed = round(microtime(true) - $startTime, 1);
        $avgTime = $elapsed / $num;
        $remaining = round(($total - $num) * $avgTime / 60, 1);
        echo "  📊 Прогресс: {$num}/{$total} ({$elapsed} сек) | ~{$remaining} мин. осталось\n\n";
        
        // Сохраняем результат
        $results[] = [
            'num' => $num,
            'query' => $query,
            'returned_specialists' => $returnedSpecialists,
            'returned_count' => $returnedCount,
            'score' => $score,
            'verdict' => $verdict,
            'expert_details' => $expertResult
        ];
        
        // Небольшая пауза между запросами
        usleep(500000); // 0.5 секунды
    }
}

$endTime = microtime(true);
$totalTime = round($endTime - $startTime, 1);
$avgScore = round($stats['total_score'] / $total, 1);

// ===== ИТОГОВАЯ СТАТИСТИКА =====
echo "\n";
echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║                        ИТОГОВАЯ СТАТИСТИКА                               ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

echo "Всего вопросов: {$total}\n";
echo "Время выполнения: {$totalTime} сек (" . round($totalTime / 60, 1) . " мин)\n";
echo "Средняя скорость: " . round($total / ($totalTime / 60), 1) . " вопросов/мин\n\n";

echo "ОЦЕНКИ AI-ЭКСПЕРТА (адекватность specialists):\n";
echo "  ⭐ ОТЛИЧНО (9-10): {$stats['excellent']} (" . round($stats['excellent'] / $total * 100, 1) . "%)\n";
echo "  ✅ ХОРОШО (7-8): {$stats['good']} (" . round($stats['good'] / $total * 100, 1) . "%)\n";
echo "  ⚠️ ПЛОХО (4-6): {$stats['poor']} (" . round($stats['poor'] / $total * 100, 1) . "%)\n";
echo "  ❌ УЖАСНО (0-3): {$stats['bad']} (" . round($stats['bad'] / $total * 100, 1) . "%)\n";
echo "  📊 Средний балл: {$avgScore}/10\n\n";

echo "ДАННЫЕ:\n";
echo "  Со specialists: {$stats['with_specialists']} (" . round($stats['with_specialists'] / $total * 100, 1) . "%)\n";
echo "  Без specialists: {$stats['without_specialists']} (" . round($stats['without_specialists'] / $total * 100, 1) . "%)\n";
echo "  Всего specialists в БД: {$totalSpecialistsInDB}\n\n";

// Топ-10 лучших
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "ТОП-10 ЛУЧШИХ РЕЗУЛЬТАТОВ:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

usort($results, function($a, $b) { return $b['score'] - $a['score']; });
$top10 = array_slice($results, 0, 10);

foreach ($top10 as $result) {
    $emoji = $verdictEmoji[$result['verdict']] ?? '❓';
    echo "  {$emoji} {$result['score']}/10 - {$result['query']}\n";
    echo "     Возвращено специалистов: {$result['returned_count']}\n";
    echo "     {$result['expert_details']['comment']}\n\n";
}

// Топ-10 худших
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "ТОП-10 ПРОБЛЕМНЫХ ВОПРОСОВ:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$bottom10 = array_slice(array_reverse($results), 0, 10);

foreach ($bottom10 as $result) {
    $emoji = $verdictEmoji[$result['verdict']] ?? '❓';
    echo "  {$emoji} {$result['score']}/10 - {$result['query']}\n";
    echo "     Возвращено специалистов: {$result['returned_count']}\n";
    if (!empty($result['expert_details']['missing_specialists'])) {
        echo "     Недостающие: " . implode(', ', $result['expert_details']['missing_specialists']) . "\n";
    }
    if (!empty($result['expert_details']['extra_specialists'])) {
        echo "     Лишние: " . implode(', ', $result['expert_details']['extra_specialists']) . "\n";
    }
    if (!empty($result['expert_details']['issues'])) {
        echo "     Проблемы: " . implode(', ', $result['expert_details']['issues']) . "\n";
    }
    echo "\n";
}

// Сохраняем результаты
$timestamp = date('Y-m-d_H-i-s');
$jsonFile = "/root/test-specialists-results-{$timestamp}.json";
$reportFile = "/root/test-specialists-report-{$timestamp}.txt";

file_put_contents($jsonFile, json_encode($results, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

$report = "ТЕСТ АДЕКВАТНОСТИ ВЫВОДА SPECIALISTS ВИДЖЕТА\n";
$report .= "Дата: " . date('Y-m-d H:i:s') . "\n";
$report .= "Сервер: wtest.medmaps.ru\n";
$report .= "API: query-section.php\n";
$report .= "Эксперт: Mimo v2 Flash\n\n";
$report .= "Всего вопросов: {$total}\n";
$report .= "Время: {$totalTime} сек\n";
$report .= "Средний балл: {$avgScore}/10\n\n";
$report .= "ОТЛИЧНО: {$stats['excellent']}\n";
$report .= "ХОРОШО: {$stats['good']}\n";
$report .= "ПЛОХО: {$stats['poor']}\n";
$report .= "УЖАСНО: {$stats['bad']}\n\n";
$report .= "Со specialists: {$stats['with_specialists']}\n";
$report .= "Без specialists: {$stats['without_specialists']}\n";
$report .= "Всего specialists в БД: {$totalSpecialistsInDB}\n";

file_put_contents($reportFile, $report);

echo "\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "Результаты сохранены:\n";
echo "  JSON: {$jsonFile}\n";
echo "  Отчет: {$reportFile}\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
