<?php
/**
 * Тест улучшений - проверяем только самые проблемные вопросы
 */

require_once '/root/config.php';

$widgetKey = 'ekamedcenter';

// Самые проблемные вопросы из предыдущего теста (score < 5)
$problematicQuestions = [
    'Холтер мониторинг',
    'Отеки ног к вечеру',
    'Тонзиллит хронический',
    'Скарлатина симптомы',
    'Боль в сердце колющая',
    'Гастроскопия ФГДС',
    'Аллергия на пыльцу поллиноз',
    'Корь у детей',
    'Ветрянка у детей',
    'Остеохондроз шейного отдела',
    'Тремор рук причины',
    'Анемия низкий гемоглобин',
    'Боль в груди при физической нагрузке',
    'Пиелонефрит воспаление почек',
    'Астигматизм коррекция',
    'Плановый осмотр ребенка',
    'Боль в правом подреберье',
    'Эндометриоз симптомы',
    'Авитаминоз весной',
    'Сухость во рту',
];

echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║     ТЕСТ УЛУЧШЕНИЙ - " . count($problematicQuestions) . " ПРОБЛЕМНЫХ ВОПРОСОВ                           ║\n";
echo "║     Сервер: wtest.medmaps.ru | GPU: Qwen2.5-7B-Instruct                ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

$results = [];
$stats = [
    'total' => count($problematicQuestions),
    'with_text' => 0,
    'without_text' => 0,
    'no_truncation' => 0,
    'truncation_detected' => 0,
    'good_specialists' => 0,
    'bad_specialists' => 0
];

foreach ($problematicQuestions as $idx => $query) {
    $num = $idx + 1;
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "ВОПРОС {$num}/{$stats['total']}: {$query}\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    // Запрос к API
    $data = [
        'widget_key' => $widgetKey,
        'question' => $query
    ];
    
    $ch = curl_init('https://wtest.medmaps.ru/api/query-stage1_1.php');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200 || !$response) {
        echo "  ❌ ОШИБКА: HTTP {$httpCode}\n\n";
        continue;
    }
    
    $decoded = json_decode($response, true);
    $text = $decoded['text'] ?? '';
    $provider = $decoded['provider'] ?? 'unknown';
    
    if (empty($text)) {
        echo "  ❌ Текст отсутствует\n\n";
        $stats['without_text']++;
        continue;
    }
    
    $stats['with_text']++;
    
    echo "  📄 ТЕКСТ ({$provider}, " . mb_strlen($text) . " символов):\n";
    echo "  " . str_replace("\n", "\n  ", $text) . "\n\n";
    
    // Анализ проблем
    $issues = [];
    $hasProblems = false;
    
    // Проверка на обрыв текста
    $truncationPatterns = [
        '/таких\s+как\s+(Рекомендуется|Для|При)/ui' => 'Обрыв после "таких как"',
        '/(воспалени[ием]+)\s+(Рекомендуется|Для)/ui' => 'Обрыв после "воспаление"',
        '/метод\s+непрерывного\s+(Рекомендуется|Для)/ui' => 'Обрыв определения метода',
        '/характеризуется\s+(Рекомендуется|Для)/ui' => 'Обрыв после "характеризуется"',
        '/,\s+(Рекомендуется|Для диагностики)/ui' => 'Обрыв перечисления',
        '/\s+и\s+(Рекомендуется|Для|При)/ui' => 'Обрыв после союза "и"',
        '/таких\s+или\s+даже/ui' => 'Грамматическая ошибка "таких или даже"',
        '/\b(hands|feet|arms|legs|head|body)\b/i' => 'Английские слова',
        '/\.\s*([A-Z][a-z]+\s+(?:is|are|can|should))/u' => 'Английские предложения',
        '/[\x{4e00}-\x{9fff}]/u' => 'Китайские символы',
    ];
    
    foreach ($truncationPatterns as $pattern => $description) {
        if (preg_match($pattern, $text)) {
            $issues[] = $description;
            $hasProblems = true;
            $stats['truncation_detected']++;
            break;
        }
    }
    
    if (!$hasProblems) {
        echo "  ✅ Обрывов не обнаружено\n";
        $stats['no_truncation']++;
    } else {
        echo "  ⚠️ ПРОБЛЕМЫ: " . implode(', ', $issues) . "\n";
    }
    
    // Проверка специалистов
    $badSpecialists = ['врач', 'врача', 'врачу', 'легкодышечный врач', 'дыхательный врач', 'доктор'];
    $goodSpecialists = ['терапевт', 'кардиолог', 'невролог', 'гастроэнтеролог', 'пульмонолог', 
                        'эндокринолог', 'уролог', 'гинеколог', 'офтальмолог', 'дерматолог', 
                        'ортопед', 'отоларинголог', 'педиатр', 'аллерголог'];
    
    $hasBadSpecialist = false;
    foreach ($badSpecialists as $bad) {
        if (mb_stripos($text, $bad) !== false) {
            echo "  ⚠️ Общий термин: '{$bad}' (нужен конкретный специалист)\n";
            $hasBadSpecialist = true;
            $stats['bad_specialists']++;
            break;
        }
    }
    
    $hasGoodSpecialist = false;
    foreach ($goodSpecialists as $good) {
        if (mb_stripos($text, $good) !== false) {
            echo "  ✅ Специалист указан: {$good}\n";
            $hasGoodSpecialist = true;
            $stats['good_specialists']++;
            break;
        }
    }
    
    if (!$hasGoodSpecialist && !$hasBadSpecialist) {
        echo "  ⚠️ Специалист не указан вообще\n";
    }
    
    echo "\n";
    
    // Сохраняем результат
    $results[] = [
        'num' => $num,
        'query' => $query,
        'text' => $text,
        'length' => mb_strlen($text),
        'provider' => $provider,
        'has_truncation' => $hasProblems,
        'issues' => $issues,
        'has_bad_specialist' => $hasBadSpecialist,
        'has_good_specialist' => $hasGoodSpecialist
    ];
    
    // Пауза между запросами
    usleep(500000); // 0.5 секунды
}

// ===== ИТОГОВАЯ СТАТИСТИКА =====
echo "\n";
echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║                        ИТОГОВАЯ СТАТИСТИКА                               ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

echo "Всего вопросов: {$stats['total']}\n";
echo "С текстом: {$stats['with_text']} (" . round($stats['with_text'] / $stats['total'] * 100, 1) . "%)\n";
echo "Без текста: {$stats['without_text']} (" . round($stats['without_text'] / $stats['total'] * 100, 1) . "%)\n\n";

echo "КАЧЕСТВО ТЕКСТА:\n";
echo "  ✅ Без обрывов: {$stats['no_truncation']} (" . round($stats['no_truncation'] / max($stats['with_text'], 1) * 100, 1) . "%)\n";
echo "  ⚠️ С обрывами: {$stats['truncation_detected']} (" . round($stats['truncation_detected'] / max($stats['with_text'], 1) * 100, 1) . "%)\n\n";

echo "УКАЗАНИЕ СПЕЦИАЛИСТОВ:\n";
echo "  ✅ Конкретные специалисты: {$stats['good_specialists']} (" . round($stats['good_specialists'] / max($stats['with_text'], 1) * 100, 1) . "%)\n";
echo "  ⚠️ Общие термины (врач/доктор): {$stats['bad_specialists']} (" . round($stats['bad_specialists'] / max($stats['with_text'], 1) * 100, 1) . "%)\n\n";

// Процент улучшения (было 30 плохих вопросов из старого теста)
$oldBadCount = 30;
$newBadCount = $stats['truncation_detected'] + $stats['bad_specialists'];
$improvement = round((($oldBadCount - $newBadCount) / $oldBadCount) * 100, 1);

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "ОЦЕНКА УЛУЧШЕНИЙ:\n";
echo "  Было проблемных: ~{$oldBadCount} из 310 (9.7%)\n";
echo "  Стало проблемных: {$newBadCount} из {$stats['with_text']} (" . round($newBadCount / max($stats['with_text'], 1) * 100, 1) . "%)\n";
if ($improvement > 0) {
    echo "  📈 УЛУЧШЕНИЕ: {$improvement}%\n";
} else {
    echo "  📉 Ухудшение: " . abs($improvement) . "%\n";
}
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

// Сохраняем результаты
$timestamp = date('Y-m-d_H-i-s');
$jsonFile = "/root/test-improvements-results-{$timestamp}.json";
file_put_contents($jsonFile, json_encode($results, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

echo "Результаты сохранены: {$jsonFile}\n";
