<?php
/**
 * Полный тест после критических исправлений
 */

require_once __DIR__ . '/config.php';

$testQueries = [
    'Острая боль в спине',
    'Головокружение при ходьбе',
    'Тошнота после еды',
    'Одышка при нагрузке',
    'Повышенная температура',
    'Боль в груди',
    'Потеря аппетита',
    'Бессонница',
    'Частое мочеиспускание',
    'Кровотечение из носа'
];

$timestamp = date('Y-m-d_H-i-s');
$outputFile = "/root/test_after_fixes_{$timestamp}.txt";
$summaryFile = "/root/test_summary_after_fixes_{$timestamp}.txt";

$output = "";
$summary = "";

$output .= "╔═══════════════════════════════════════════════════════════════════════════════╗\n";
$output .= "║          ТЕСТ ПОСЛЕ КРИТИЧЕСКИХ ИСПРАВЛЕНИЙ - ВСЕ ЗАПРОСЫ                  ║\n";
$output .= "╚═══════════════════════════════════════════════════════════════════════════════╝\n\n";
$output .= "Дата: " . date('d.m.Y H:i:s') . "\n";
$output .= "Количество запросов: " . count($testQueries) . "\n\n";

$summary .= "╔═══════════════════════════════════════════════════════════════════════════════╗\n";
$summary .= "║                    СВОДКА ПОСЛЕ ИСПРАВЛЕНИЙ                                  ║\n";
$summary .= "╚═══════════════════════════════════════════════════════════════════════════════╝\n\n";

// Получаем виджет
$db = getDatabase();
$stmt = $db->prepare("SELECT * FROM widgets WHERE widget_key = ?");
$stmt->execute(['ekamedcenter']);
$widget = $stmt->fetch();

if (!$widget) {
    echo "❌ ERROR: Widget not found\n";
    exit(1);
}

$results = [];

foreach ($testQueries as $index => $question) {
    $num = $index + 1;
    $output .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    $output .= "ЗАПРОС {$num}: {$question}\n";
    $output .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    // Определяем специальности
    $relevantSpecialties = [];
    
    if (preg_match('/(бессонн|невролог|голов|головокруж|мигрен|головн.*бол|нервн|тремор|онемен|защемл|спин.*бол|боль.*спин)/ui', $question)) {
        $relevantSpecialties[] = 'невролог';
    }
    if (preg_match('/(температур|просту|грипп|орви|орз|кашель|насморк|слабость|терапевт|одышк|аппетит)/ui', $question)) {
        $relevantSpecialties[] = 'терапевт';
    }
    if (preg_match('/(травм|ортопед|перелом|вывих|растяжен|ушиб|спин.*бол|боль.*спин|сустав|колен|плеч)/ui', $question)) {
        $relevantSpecialties[] = 'травматолог';
    }
    if (preg_match('/(сердц|давлен|гипертон|аритми|тахикарди|кардиолог|одышк|груд.*бол|боль.*груд)/ui', $question)) {
        $relevantSpecialties[] = 'кардиолог';
    }
    if (preg_match('/(живот|желуд|гастрит|язв|тошнот|рвот|изжог|пищеварен|гастроэнтеролог|аппетит)/ui', $question)) {
        $relevantSpecialties[] = 'гастроэнтеролог';
    }
    if (preg_match('/(кож|сыпь|зуд|прыщ|акне|псориаз|розацеа|дерматолог|дерматит)/ui', $question)) {
        $relevantSpecialties[] = 'дерматолог';
    }
    if (preg_match('/(мочеиспускан|урина|уролог|моч[еа]|почк|мочевой пузырь|цистит|пиелонефрит|простат)/ui', $question)) {
        $relevantSpecialties[] = 'уролог';
    }
    if (preg_match('/(нос|горл|ухо|лор|оториноларинголог|ангин|фарингит|тонзиллит|ринит|синусит|гайморит|отит|кровотечение.*нос|носов.*кровотечен)/ui', $question)) {
        $relevantSpecialties[] = 'лор';
    }
    if (preg_match('/(аллерг|аллерголог|аллергическ|крапивниц)/ui', $question)) {
        $relevantSpecialties[] = 'аллерголог';
    }
    if (preg_match('/(глаз|зрен|офтальмолог|конъюнктивит|катаракт|глауком)/ui', $question)) {
        $relevantSpecialties[] = 'офтальмолог';
    }
    
    $output .= "Определенные специальности: " . (empty($relevantSpecialties) ? '❌ НЕТ' : implode(', ', $relevantSpecialties)) . "\n\n";
    
    // Маппинг
    $specialtyMapping = [
        'лор' => 'оториноларинголог',
        'невролог' => 'невролог',
        'терапевт' => 'терапевт',
        'кардиолог' => 'кардиолог',
        'гастроэнтеролог' => 'гастроэнтеролог',
        'дерматолог' => 'дерматолог',
        'уролог' => 'уролог',
        'аллерголог' => 'аллерголог',
        'офтальмолог' => 'офтальмолог',
        'травматолог' => 'травматолог'
    ];
    
    // Ищем консультации
    $consultationIds = [];
    $consultationNames = [];
    foreach ($relevantSpecialties as $specialty) {
        $searchSpecialty = $specialtyMapping[$specialty] ?? $specialty;
        
        $stmtConsult = $db->prepare("
            SELECT DISTINCT pi.id, pf.field_value as name,
                CASE 
                    WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 1
                    WHEN LOWER(pf.field_value) LIKE '%повторная%' THEN 2
                    ELSE 3
                END as sort_order
            FROM parsed_items pi
            JOIN parsed_fields pf ON pi.id = pf.item_id
            WHERE pi.widget_id = ? 
            AND pi.section_name = 'services'
            AND pf.field_name = 'name'
            AND LOWER(pf.field_value) LIKE ?
            AND LOWER(pf.field_value) LIKE '%консультация%'
            AND pi.is_duplicate = 0
            ORDER BY sort_order
            LIMIT 2
        ");
        $stmtConsult->execute([$widget['id'], '%' . $searchSpecialty . '%']);
        
        while ($row = $stmtConsult->fetch()) {
            $consultationIds[] = (int)$row['id'];
            $consultationNames[] = $row['name'];
        }
    }
    
    $output .= "Найденные консультации:\n";
    if (empty($consultationNames)) {
        $output .= "  ❌ НЕТ\n";
    } else {
        foreach ($consultationNames as $i => $name) {
            $output .= "  " . ($i + 1) . ". {$name}\n";
        }
    }
    $output .= "\n";
    
    // Результаты
    $hasConsultations = !empty($consultationIds);
    $hasPrimary = false;
    foreach ($consultationNames as $name) {
        if (mb_stripos($name, 'первичная') !== false) {
            $hasPrimary = true;
            break;
        }
    }
    
    $output .= "📊 СТАТУС:\n";
    $output .= "  Консультации: " . ($hasConsultations ? '✅ Найдены (' . count($consultationIds) . ')' : '❌ НЕТ') . "\n";
    $output .= "  Первичная: " . ($hasPrimary ? '✅ Есть' : '❌ НЕТ') . "\n";
    $output .= "\n";
    
    $results[] = [
        'question' => $question,
        'specialties' => $relevantSpecialties,
        'consultations' => count($consultationIds),
        'hasPrimary' => $hasPrimary,
        'hasConsultations' => $hasConsultations
    ];
}

// Формируем сводку
$totalQueries = count($results);
$withConsultations = 0;
$withPrimary = 0;
$withSpecialties = 0;

foreach ($results as $result) {
    if ($result['hasConsultations']) $withConsultations++;
    if ($result['hasPrimary']) $withPrimary++;
    if (!empty($result['specialties'])) $withSpecialties++;
}

$summary .= "Всего запросов: {$totalQueries}\n\n";
$summary .= "✅ Определены специальности: {$withSpecialties}/{$totalQueries} (" . round($withSpecialties/$totalQueries*100) . "%)\n";
$summary .= "✅ Найдены консультации: {$withConsultations}/{$totalQueries} (" . round($withConsultations/$totalQueries*100) . "%)\n";
$summary .= "✅ Есть первичная консультация: {$withPrimary}/{$totalQueries} (" . round($withPrimary/$totalQueries*100) . "%)\n\n";

$summary .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
$summary .= "РЕЗУЛЬТАТЫ ПО ЗАПРОСАМ:\n";
$summary .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

foreach ($results as $i => $result) {
    $num = $i + 1;
    $icon = ($result['hasConsultations'] && $result['hasPrimary']) ? '✅' : '⚠️';
    $summary .= "{$num}. {$icon} {$result['question']}\n";
    $summary .= "   Специальности: " . (empty($result['specialties']) ? 'НЕТ' : implode(', ', $result['specialties'])) . "\n";
    $summary .= "   Консультаций: {$result['consultations']}" . ($result['hasPrimary'] ? ' (есть первичная)' : '') . "\n\n";
}

$output .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
$output .= "ТЕСТИРОВАНИЕ ЗАВЕРШЕНО\n";
$output .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

// Сохраняем результаты
file_put_contents($outputFile, $output);
file_put_contents($summaryFile, $summary);

echo $summary;
echo "\n📁 Детальные результаты сохранены в: {$outputFile}\n";
echo "📄 Сводка сохранена в: {$summaryFile}\n";

