<?php
/**
 * Полный тест виджета на 310 вопросах с улучшенным промптом
 * - Текстовые ответы через query-stage1_1.php (GPU)
 * - Оценка качества текста через AI эксперта (OpenRouter)
 * - Тестовый сервер: wtest.medmaps.ru
 */

require '/root/questions-list.php';  // Полный список - загружает $questions
require_once '/root/config.php';

$widgetKey = 'ekamedcenter';
$total = count($questions);

echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║  ПОЛНЫЙ ТЕСТ С УЛУЧШЕННЫМ ПРОМПТОМ ({$total} ВОПРОСОВ)                    ║\n";
echo "║  Сервер: wtest.medmaps.ru | GPU: Qwen2.5-7B | Эксперт: Gemini 2.5       ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

$results = [];
$stats = [
    'total' => $total,
    'excellent' => 0,  // 9-10
    'good' => 0,       // 7-8
    'poor' => 0,       // 4-6
    'bad' => 0,        // 0-3
    'total_score' => 0,
    'with_text' => 0,
    'without_text' => 0,
    'with_artifacts' => 0,
    'incomplete' => 0
];

$startTime = microtime(true);

// Определяем эмодзи для вердикта
$verdictEmoji = [
    'ОТЛИЧНО' => '⭐',
    'ХОРОШО' => '✅',
    'ПЛОХО' => '⚠️',
    'УЖАСНО' => '❌'
];

// Разбиваем вопросы на батчи по 30 для параллельной обработки
$batchSize = 30;
$questionBatches = array_chunk($questions, $batchSize, true);
$globalQuestionIndex = 0;

foreach ($questionBatches as $batchIndex => $batch) {
    $batchNum = $batchIndex + 1;
    $totalBatches = count($questionBatches);
    echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "БАТЧ {$batchNum}/{$totalBatches} (обработка " . count($batch) . " вопросов параллельно)\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    // Подготавливаем все запросы для параллельного выполнения
    $multiHandle = curl_multi_init();
    $handles = [];
    
    // ===== 1. ПОДГОТОВКА ПАРАЛЛЕЛЬНЫХ ЗАПРОСОВ ДЛЯ ТЕКСТА (Stage1) =====
    foreach ($batch as $index => $query) {
        $textData = [
            'widget_key' => $widgetKey,
            'question' => $query
        ];
        
        $ch = curl_init('https://wtest.medmaps.ru/api/query-stage1_1.php');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($textData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
        
        $handleId = (int)$ch;
        $handles[$handleId] = [
            'handle' => $ch,
            'type' => 'stage1',
            'index' => $index,
            'query' => $query
        ];
        
        curl_multi_add_handle($multiHandle, $ch);
    }
    
    // Выполняем параллельные запросы
    $running = null;
    do {
        curl_multi_exec($multiHandle, $running);
        curl_multi_select($multiHandle, 0.1);
    } while ($running > 0);
    
    // Собираем результаты
    $textResults = [];
    foreach ($handles as $handleId => $handleData) {
        if ($handleData['type'] === 'stage1') {
            $response = curl_multi_getcontent($handleData['handle']);
            $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
            
            $text = '';
            $textProvider = 'unknown';
            
            if ($httpCode === 200 && $response) {
                $response = trim($response);
                $response = preg_replace('/^\xEF\xBB\xBF/', '', $response);
                $decoded = json_decode($response, true);
                
                if ($decoded && isset($decoded['text'])) {
                    $text = $decoded['text'];
                    $textProvider = $decoded['provider'] ?? 'unknown';
                }
            }
            
            $textResults[$handleData['index']] = [
                'text' => $text,
                'provider' => $textProvider,
                'query' => $handleData['query']
            ];
            
            curl_multi_remove_handle($multiHandle, $handleData['handle']);
            curl_close($handleData['handle']);
        }
    }
    
    // Очищаем multi handle
    curl_multi_close($multiHandle);
    
    // ===== 2. ПОДГОТОВКА ДАННЫХ ДЛЯ AI-ЭКСПЕРТА =====
    $expertData = [];
    foreach ($batch as $index => $query) {
        $textResult = $textResults[$index] ?? ['text' => '', 'provider' => 'unknown', 'query' => $query];
        $text = $textResult['text'];
        
        $hasText = !empty($text);
        
        // Проверка артефактов
        $textArtifacts = [];
        if ($hasText) {
            if (preg_match('/\.(runtuit|jpg|jpeg|png|gif|webp|svg)[_\w\d]*/i', $text)) {
                $textArtifacts[] = 'Ссылки на изображения в тексте';
            }
            if (preg_match('/\b(other\s+symptoms?|other\s+conditions?|Apply\s+ice|appropriate\s+лечение|appropriate\s+treatment)\b/ui', $text)) {
                $textArtifacts[] = 'Смешанные русско-английские фразы';
            }
            if (preg_match('/\b(endocrinolog|gastroenterolog|pulmonologist|respirologist|orthopedic|ophthalmologist|dermatologist|allergologist|immunologist|cardiologist|neurologist|rheumatologist|urologist|gynecologist|surgeon|therapist|pediatrician)[уеаио]\b/ui', $text)) {
                $textArtifacts[] = 'Транслитерация медицинских терминов';
            }
            if (preg_match('/\.\s*([A-Z][a-z]+(?:\s+[a-z]+)*\s+(?:can|should|is|are|may|might|will|would|could)\s+[^.]*\.)/u', $text)) {
                $textArtifacts[] = 'Английские предложения в русском тексте';
            }
            if (preg_match('/[\x{4e00}-\x{9fff}]/u', $text)) {
                $textArtifacts[] = 'Китайские символы в тексте';
            }
            if (preg_match('/[^.!?]\s*$/', trim($text)) && mb_strlen($text) < 50) {
                $textArtifacts[] = 'Незавершенный текст';
            }
            if (preg_match('/<[^>]+>/', $text)) {
                $textArtifacts[] = 'HTML теги в тексте';
            }
            if (preg_match('/(http:\/\/|https:\/\/|www\.)/i', $text)) {
                $textArtifacts[] = 'URL в тексте';
            }
            // Проверка на повторения
            $sentences = preg_split('/[.!?]+/', $text);
            $uniqueSentences = array_unique(array_map('trim', array_filter($sentences)));
            if (count($sentences) - count($uniqueSentences) > 1) {
                $textArtifacts[] = 'Повторение фраз';
            }
        }
        
        if (!empty($textArtifacts)) {
            $stats['with_artifacts']++;
        }
        
        $artifactsInfo = !empty($textArtifacts) ? "\n\n⚠️ ОБНАРУЖЕНЫ АРТЕФАКТЫ В ТЕКСТЕ:\n" . implode("\n", array_map(function($a) { return "- " . $a; }, $textArtifacts)) : "";
        
        $expertPrompt = "Ты - строгий эксперт по медицинским виджетам. Проверь ТЕКСТОВЫЙ ответ виджета медицинской клиники на наличие проблем.

ВОПРОС ПОЛЬЗОВАТЕЛЯ:
\"{$query}\"

ТЕКСТОВЫЙ ОТВЕТ ВИДЖЕТА:
" . ($hasText ? $text : "(текст отсутствует)") . $artifactsInfo . "

ТВОЯ ЗАДАЧА - проверить текст на три критерия:

1. АРТЕФАКТЫ:
   Проверь наличие в тексте:
   - Ссылок на изображения (.jpg, .png, .runtuit и т.п.)
   - Смешанных русско-английских фраз (например: \"appropriate лечение\", \"Apply ice\")
   - Транслитерации медицинских терминов (например: \"endocrinologistу\", \"cardiologistе\")
   - Английских предложений в русском тексте
   - Китайских символов
   - HTML тегов
   - URL адресов
   - Повторений фраз или предложений

2. ЗАВЕРШЕННОСТЬ ПРЕДЛОЖЕНИЙ:
   Проверь, что текст не обрывается на середине предложения.
   Текст должен заканчиваться точкой, восклицательным или вопросительным знаком.
   Если текст обрывается неожиданно - это проблема.

3. АДЕКВАТНОСТЬ:
   Проверь, что текст:
   - Написан на русском языке (без английских предложений)
   - Имеет смысл и логичен
   - Не содержит бессмысленных фраз или символов
   - Соответствует медицинской тематике

ОЦЕНКА (0-10 баллов):
- 10: текст идеален, нет артефактов, предложения завершены, текст адекватен
- 8-9: текст хороший, есть незначительные проблемы
- 6-7: текст приемлемый, но есть заметные проблемы
- 4-5: текст плохой, много проблем
- 0-3: текст очень плохой или отсутствует

ФОРМАТ ОТВЕТА (строго JSON):
{
  \"score\": <число 0-10>,
  \"verdict\": \"ОТЛИЧНО\" | \"ХОРОШО\" | \"ПЛОХО\" | \"УЖАСНО\",
  \"has_artifacts\": <true/false>,
  \"is_complete\": <true/false>,
  \"is_adequate\": <true/false>,
  \"comment\": \"<краткий комментарий о найденных проблемах>\",
  \"issues\": [\"<список проблем, если есть>\"],
  \"text_artifacts\": [\"<список найденных артефактов в тексте, если есть>\"],
  \"text_adequacy\": \"<адекватен ли текст: ДА/НЕТ/ЧАСТИЧНО, с пояснением>\",
  \"completeness\": \"<завершены ли предложения: ДА/НЕТ/ЧАСТИЧНО, с пояснением>\"
}

ВЕРДИКТЫ:
- ОТЛИЧНО: 9-10 баллов (нет проблем)
- ХОРОШО: 7-8 баллов (незначительные проблемы)
- ПЛОХО: 4-6 баллов (заметные проблемы)
- УЖАСНО: 0-3 балла (критичные проблемы или текст отсутствует)

Отвечай ТОЛЬКО валидным JSON, без дополнительного текста!";
        
        $expertData[$index] = [
            'query' => $query,
            'text' => $text,
            'textProvider' => $textResult['provider'],
            'textArtifacts' => $textArtifacts,
            'prompt' => $expertPrompt
        ];
    }
    
    // ===== 3. ПАРАЛЛЕЛЬНЫЕ ЗАПРОСЫ К AI-ЭКСПЕРТУ =====
    echo "  → Отправка запросов к AI-эксперту (параллельно)...\n";
    
    $multiHandle = curl_multi_init();
    $expertHandles = [];
    
    foreach ($expertData as $index => $data) {
        $expertApiData = [
            'model' => 'xiaomi/mimo-v2-flash:free',
            'messages' => [
                ['role' => 'user', 'content' => $data['prompt']]
            ]
        ];
        
        $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($expertApiData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . OPENROUTER_API_KEY,
            'HTTP-Referer: https://wtest.medmaps.ru',
            'X-Title: Medical Widget Tester'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        
        $handleId = (int)$ch;
        $expertHandles[$handleId] = [
            'handle' => $ch,
            'index' => $index
        ];
        
        curl_multi_add_handle($multiHandle, $ch);
    }
    
    // Выполняем параллельные запросы к эксперту
    $running = null;
    do {
        curl_multi_exec($multiHandle, $running);
        curl_multi_select($multiHandle, 0.1);
    } while ($running > 0);
    
    // Собираем результаты эксперта
    $expertResults = [];
    $failedIndices = [];
    
    foreach ($expertHandles as $handleId => $handleData) {
        $response = curl_multi_getcontent($handleData['handle']);
        $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
        
        $expertResult = null;
        if ($httpCode === 200 && $response) {
            $expertDecoded = json_decode($response, true);
            $expertContent = $expertDecoded['choices'][0]['message']['content'] ?? '';
            
            if (preg_match('/\{[^{}]*(?:\{[^{}]*\}[^{}]*)*\}/s', $expertContent, $matches)) {
                $expertResult = json_decode($matches[0], true);
            }
        }
        
        if (!$expertResult || !isset($expertResult['score'])) {
            $failedIndices[] = $handleData['index'];
            $expertResult = [
                'score' => 0,
                'verdict' => 'УЖАСНО',
                'has_artifacts' => false,
                'is_complete' => false,
                'is_adequate' => false,
                'comment' => 'Ошибка оценки AI',
                'issues' => ['AI эксперт не ответил']
            ];
        }
        
        $expertResults[$handleData['index']] = $expertResult;
        
        curl_multi_remove_handle($multiHandle, $handleData['handle']);
        curl_close($handleData['handle']);
    }
    
    // Повторные попытки для неудачных запросов (до 3 раз)
    if (!empty($failedIndices)) {
        $maxRetries = 3;
        $retryCount = 0;
        
        while (!empty($failedIndices) && $retryCount < $maxRetries) {
            $retryCount++;
            echo "  → Повторная попытка оценки AI (попытка {$retryCount}/{$maxRetries}) для " . count($failedIndices) . " вопросов...\n";
            
            $retryMultiHandle = curl_multi_init();
            $retryHandles = [];
            
            foreach ($failedIndices as $index) {
                $data = $expertData[$index];
                $expertApiData = [
                    'model' => 'xiaomi/mimo-v2-flash:free',
                    'messages' => [
                        ['role' => 'user', 'content' => $data['prompt']]
                    ]
                ];
                
                $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($expertApiData));
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Content-Type: application/json',
                    'Authorization: Bearer ' . OPENROUTER_API_KEY,
                    'HTTP-Referer: https://wtest.medmaps.ru',
                    'X-Title: Medical Widget Tester'
                ]);
                curl_setopt($ch, CURLOPT_TIMEOUT, 60);
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
                
                $retryHandles[(int)$ch] = [
                    'handle' => $ch,
                    'index' => $index
                ];
                
                curl_multi_add_handle($retryMultiHandle, $ch);
            }
            
            // Выполняем повторные запросы
            $running = null;
            do {
                curl_multi_exec($retryMultiHandle, $running);
                curl_multi_select($retryMultiHandle, 0.1);
            } while ($running > 0);
            
            // Обрабатываем результаты повторных попыток
            $stillFailed = [];
            foreach ($retryHandles as $handleId => $handleData) {
                $response = curl_multi_getcontent($handleData['handle']);
                $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
                
                $expertResult = null;
                if ($httpCode === 200 && $response) {
                    $expertDecoded = json_decode($response, true);
                    $expertContent = $expertDecoded['choices'][0]['message']['content'] ?? '';
                    
                    if (preg_match('/\{[^{}]*(?:\{[^{}]*\}[^{}]*)*\}/s', $expertContent, $matches)) {
                        $expertResult = json_decode($matches[0], true);
                    }
                }
                
                if ($expertResult && isset($expertResult['score'])) {
                    $expertResults[$handleData['index']] = $expertResult;
                } else {
                    $stillFailed[] = $handleData['index'];
                }
                
                curl_multi_remove_handle($retryMultiHandle, $handleData['handle']);
                curl_close($handleData['handle']);
            }
            
            curl_multi_close($retryMultiHandle);
            $failedIndices = $stillFailed;
            
            if (!empty($failedIndices) && $retryCount < $maxRetries) {
                usleep(500000);
            }
        }
    }
    
    curl_multi_close($multiHandle);
    
    // ===== 4. ВЫВОД РЕЗУЛЬТАТОВ БАТЧА =====
    foreach ($batch as $index => $query) {
        $globalQuestionIndex++;
        $num = $globalQuestionIndex;
        
        echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
        echo "ВОПРОС {$num}/{$total}: {$query}\n";
        echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
        
        $data = $expertData[$index];
        $text = $data['text'];
        $textProvider = $data['textProvider'];
        $textArtifacts = $data['textArtifacts'];
        
        $hasText = !empty($text);
        if ($hasText) {
            $stats['with_text']++;
        } else {
            $stats['without_text']++;
        }
        
        echo "  → Текст получен: " . ($hasText ? "да ({$textProvider}, " . mb_strlen($text) . " символов)" : "НЕТ") . "\n\n";
        
        if (!empty($textArtifacts)) {
            echo "  ⚠️ Обнаружены артефакты в тексте: " . implode(", ", $textArtifacts) . "\n\n";
        }
        
        // Результат эксперта
        $expertResult = $expertResults[$index] ?? [
            'score' => 0,
            'verdict' => 'УЖАСНО',
            'comment' => 'Ошибка получения оценки',
            'issues' => []
        ];
        
        $score = $expertResult['score'];
        $verdict = $expertResult['verdict'];
        
        // Обновляем статистику
        $stats['total_score'] += $score;
        if ($score >= 9) {
            $stats['excellent']++;
        } elseif ($score >= 7) {
            $stats['good']++;
        } elseif ($score >= 4) {
            $stats['poor']++;
        } else {
            $stats['bad']++;
        }
        
        if (!($expertResult['is_complete'] ?? true)) {
            $stats['incomplete']++;
        }
        
        // Определяем эмодзи для вердикта
        $emoji = $verdictEmoji[$verdict] ?? '❓';
        
        echo "  → Оценка AI-экспертом получена\n\n";
        echo "  {$emoji} ОЦЕНКА: {$score}/10 | ВЕРДИКТ: {$verdict}\n";
        echo "  💬 Комментарий: {$expertResult['comment']}\n";
        
        if (isset($expertResult['has_artifacts'])) {
            echo "  🔍 Артефакты: " . ($expertResult['has_artifacts'] ? "ДА" : "НЕТ") . "\n";
        }
        if (isset($expertResult['is_complete'])) {
            echo "  ✅ Завершенность: " . ($expertResult['is_complete'] ? "ДА" : "НЕТ") . "\n";
        }
        if (isset($expertResult['is_adequate'])) {
            echo "  📝 Адекватность: " . ($expertResult['is_adequate'] ? "ДА" : "НЕТ") . "\n";
        }
        if (!empty($expertResult['issues'])) {
            echo "  ⚠️ Проблемы: " . implode(", ", array_slice($expertResult['issues'], 0, 3)) . "\n";
        }
        echo "\n";
        
        // Прогресс
        $elapsed = round(microtime(true) - $startTime, 1);
        $avgTime = $elapsed / $num;
        $remaining = round(($total - $num) * $avgTime / 60, 1);
        echo "  📊 Прогресс: {$num}/{$total} ({$elapsed} сек) | ~{$remaining} мин. осталось\n\n";
        
        // Сохраняем результат
        $results[] = [
            'num' => $num,
            'query' => $query,
            'text' => $text,
            'text_length' => mb_strlen($text),
            'text_provider' => $textProvider,
            'text_artifacts_detected' => $textArtifacts,
            'score' => $score,
            'verdict' => $verdict,
            'expert_details' => $expertResult
        ];
        
        usleep(500000); // 0.5 секунды
    }
}

$endTime = microtime(true);
$totalTime = round($endTime - $startTime, 1);
$avgScore = round($stats['total_score'] / $total, 1);

// ===== ИТОГОВАЯ СТАТИСТИКА =====
echo "\n";
echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║                        ИТОГОВАЯ СТАТИСТИКА                               ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

echo "Всего вопросов: {$total}\n";
echo "Время выполнения: {$totalTime} сек (" . round($totalTime / 60, 1) . " мин)\n";
echo "Средняя скорость: " . round($total / ($totalTime / 60), 1) . " вопросов/мин\n\n";

echo "ОЦЕНКИ AI-ЭКСПЕРТА (качество текста):\n";
echo "  ⭐ ОТЛИЧНО (9-10): {$stats['excellent']} (" . round($stats['excellent'] / $total * 100, 1) . "%)\n";
echo "  ✅ ХОРОШО (7-8): {$stats['good']} (" . round($stats['good'] / $total * 100, 1) . "%)\n";
echo "  ⚠️ ПЛОХО (4-6): {$stats['poor']} (" . round($stats['poor'] / $total * 100, 1) . "%)\n";
echo "  ❌ УЖАСНО (0-3): {$stats['bad']} (" . round($stats['bad'] / $total * 100, 1) . "%)\n";
echo "  📊 Средний балл: {$avgScore}/10\n\n";

echo "ДАННЫЕ:\n";
echo "  С текстом: {$stats['with_text']} (" . round($stats['with_text'] / $total * 100, 1) . "%)\n";
echo "  Без текста: {$stats['without_text']} (" . round($stats['without_text'] / $total * 100, 1) . "%)\n";
echo "  С артефактами: {$stats['with_artifacts']} (" . round($stats['with_artifacts'] / $total * 100, 1) . "%)\n";
echo "  Незавершенные: {$stats['incomplete']} (" . round($stats['incomplete'] / $total * 100, 1) . "%)\n\n";

// Топ-10 лучших
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "ТОП-10 ЛУЧШИХ РЕЗУЛЬТАТОВ:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

usort($results, function($a, $b) { return $b['score'] - $a['score']; });
$top10 = array_slice($results, 0, 10);

foreach ($top10 as $result) {
    $emoji = $verdictEmoji[$result['verdict']] ?? '❓';
    echo "  {$emoji} {$result['score']}/10 - {$result['query']}\n";
    echo "     {$result['expert_details']['comment']}\n\n";
}

// Топ-10 худших
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "ТОП-10 ПРОБЛЕМНЫХ ВОПРОСОВ:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$bottom10 = array_slice(array_reverse($results), 0, 10);

foreach ($bottom10 as $result) {
    $emoji = $verdictEmoji[$result['verdict']] ?? '❓';
    echo "  {$emoji} {$result['score']}/10 - {$result['query']}\n";
    echo "     Проблемы: " . implode(', ', array_slice($result['expert_details']['issues'] ?? [], 0, 3)) . "\n";
    if (!empty($result['text_artifacts_detected'])) {
        echo "     Артефакты: " . implode(', ', $result['text_artifacts_detected']) . "\n";
    }
    echo "\n";
}

// Сохраняем результаты
$timestamp = date('Y-m-d_H-i-s');
$jsonFile = "/root/test-text-only-improved-results-{$timestamp}.json";
$reportFile = "/root/test-text-only-improved-report-{$timestamp}.txt";

file_put_contents($jsonFile, json_encode($results, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

$report = "ТЕСТ ТЕКСТОВЫХ ОТВЕТОВ ВИДЖЕТА (С УЛУЧШЕННЫМ ПРОМПТОМ)\n";
$report .= "Дата: " . date('Y-m-d H:i:s') . "\n";
$report .= "Сервер: wtest.medmaps.ru\n";
$report .= "GPU: Qwen2.5-7B-Instruct\n";
$report .= "Эксперт: Gemini 2.5 Flash\n\n";
$report .= "Всего вопросов: {$total}\n";
$report .= "Время: {$totalTime} сек\n";
$report .= "Средний балл: {$avgScore}/10\n\n";
$report .= "ОТЛИЧНО: {$stats['excellent']}\n";
$report .= "ХОРОШО: {$stats['good']}\n";
$report .= "ПЛОХО: {$stats['poor']}\n";
$report .= "УЖАСНО: {$stats['bad']}\n\n";
$report .= "С текстом: {$stats['with_text']}\n";
$report .= "С артефактами: {$stats['with_artifacts']}\n";
$report .= "Незавершенные: {$stats['incomplete']}\n";

file_put_contents($reportFile, $report);

echo "\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "Результаты сохранены:\n";
echo "  JSON: {$jsonFile}\n";
echo "  Отчет: {$reportFile}\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
