<?php
/**
 * Быстрый тест исправлений для проблемных вопросов
 */

// Проблемные вопросы из последнего теста с оценкой < 7
$problematicQuestions = [
    'Простуда и насморк',                    // 0/10 - артефакт ``` ```
    'Остеохондроз шейного отдела',           // 3/10 - английское слово "hands"
    'Гастроскопия ФГДС',                     // 3/10 - неверная терминология
    'Корь у детей',                          // 3/10 - грамматические ошибки
    'Боль в сердце колющая',                 // 2/10 - обрыв предложения
    'Колики у новорожденного',               // 2/10 - обрыв предложения
    'Кровотечение из носа',                  // 5/10 - неверный совет
    'Печеночные пробы анализ',               // 5/10 - медицинская неточность
    'Гипотиреоз симптомы',                   // 4/10 - тахикардия вместо брадикардии
    'Авитаминоз весной',                     // 4/10 - незавершенный текст
    'Тошнота по утрам',                      // 4/10 - неполный текст
    'Кашель сухой что делать',               // 6/10 - грамматическая ошибка
    'Холтер мониторинг',                     // 6/10 - орфографическая ошибка
    'Аденома простаты',                      // 6/10 - грамматическая ошибка
    'Остеопороз профилактика',               // 6/10 - неконкретный ответ
];

$widgetKey = 'ekamedcenter';
$apiUrl = 'https://wtest.medmaps.ru/api/query-stage1_1.php';

echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║                    ТЕСТ ИСПРАВЛЕНИЙ (15 ВОПРОСОВ)                       ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

$results = [];
$startTime = microtime(true);

foreach ($problematicQuestions as $index => $question) {
    $num = $index + 1;
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "ВОПРОС {$num}/15: {$question}\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    // Делаем запрос к API
    $data = [
        'widget_key' => $widgetKey,
        'question' => $question
    ];
    
    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    $text = '';
    $provider = 'error';
    
    if ($httpCode === 200 && $response) {
        $decoded = json_decode($response, true);
        if ($decoded && isset($decoded['text'])) {
            $text = $decoded['text'];
            $provider = $decoded['provider'] ?? 'unknown';
        }
    }
    
    // Проверка артефактов
    $artifacts = [];
    if (!empty($text)) {
        // Артефакт: тройные кавычки
        if (preg_match('/```/u', $text)) {
            $artifacts[] = 'Тройные кавычки ```';
        }
        // Артефакт: английские слова
        if (preg_match('/\b(hands|can|should|is|are|may|might|will|would|could|have|has|had)\b/ui', $text)) {
            $artifacts[] = 'Английские слова';
        }
        // Артефакт: HTML теги
        if (preg_match('/<[^>]+>/', $text) && !preg_match('/<[bi]>/', $text)) {
            $artifacts[] = 'HTML теги';
        }
        // Артефакт: URLs
        if (preg_match('/(https?:\/\/|www\.)/i', $text)) {
            $artifacts[] = 'URL адреса';
        }
        // Артефакт: ссылки на изображения
        if (preg_match('/\.(jpg|jpeg|png|gif|webp|svg|runtuit)/ui', $text)) {
            $artifacts[] = 'Ссылки на изображения';
        }
        // Незавершенность: текст не заканчивается знаком препинания
        if (!preg_match('/[.!?…]\s*$/u', $text)) {
            $artifacts[] = 'Текст не завершен';
        }
        // Незавершенность: обрыв на союзе или предлоге
        if (preg_match('/\s+(или|и|а также|таких как|включая|к|с|для|в|на)\s*[.!?]?\s*$/ui', $text)) {
            $artifacts[] = 'Обрыв на союзе/предлоге';
        }
    }
    
    $hasIssues = !empty($artifacts);
    $status = $hasIssues ? '❌ ЕСТЬ ПРОБЛЕМЫ' : '✅ ОК';
    
    echo "  Текст: " . mb_substr($text, 0, 200) . (mb_strlen($text) > 200 ? '...' : '') . "\n\n";
    echo "  Провайдер: {$provider}\n";
    echo "  Длина: " . mb_strlen($text) . " символов\n";
    echo "  Статус: {$status}\n";
    
    if ($hasIssues) {
        echo "  ⚠️ Проблемы: " . implode(", ", $artifacts) . "\n";
    }
    
    echo "\n";
    
    $results[] = [
        'question' => $question,
        'text' => $text,
        'provider' => $provider,
        'length' => mb_strlen($text),
        'artifacts' => $artifacts,
        'has_issues' => $hasIssues
    ];
    
    // Небольшая пауза между запросами
    usleep(500000); // 0.5 секунды
}

$endTime = microtime(true);
$totalTime = round($endTime - $startTime, 1);

// Статистика
$totalQuestions = count($results);
$withIssues = count(array_filter($results, fn($r) => $r['has_issues']));
$withoutIssues = $totalQuestions - $withIssues;

echo "\n╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║                           ИТОГОВАЯ СТАТИСТИКА                            ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

echo "Всего вопросов: {$totalQuestions}\n";
echo "Время выполнения: {$totalTime} сек\n\n";

echo "✅ БЕЗ ПРОБЛЕМ: {$withoutIssues} (" . round($withoutIssues / $totalQuestions * 100, 1) . "%)\n";
echo "❌ С ПРОБЛЕМАМИ: {$withIssues} (" . round($withIssues / $totalQuestions * 100, 1) . "%)\n\n";

if ($withIssues > 0) {
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "ВОПРОСЫ С ПРОБЛЕМАМИ:\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    foreach ($results as $result) {
        if ($result['has_issues']) {
            echo "  ❌ {$result['question']}\n";
            echo "     Проблемы: " . implode(", ", $result['artifacts']) . "\n";
            echo "     Текст: " . mb_substr($result['text'], 0, 150) . "...\n\n";
        }
    }
}

// Сохраняем результаты
$timestamp = date('Y-m-d_H-i-s');
$jsonFile = "/root/test-fixes-results-{$timestamp}.json";

file_put_contents($jsonFile, json_encode($results, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "Результаты сохранены: {$jsonFile}\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
