<?php
/**
 * Тест промптов для фразы "Ветрянка у детей"
 * Тестируем разные варианты промптов и настроек для получения лучшего ответа
 */

require_once '/root/config.php';

// Функция для вызова API напрямую через executeStage1
// Используем прямое подключение функций без выполнения верхнего уровня query-stage1_1.php
function callStage1Directly($question, $model, $systemPrompt) {
    // Загружаем только необходимые функции
    require_once __DIR__ . '/api/embedding-functions.php';
    
    // Копируем логику executeStage1, но без HTTP-контекста
    // Определяем, какая модель используется
    $isYandexGPT = (strpos($model, 'gpt://') === 0 || strpos($model, 'yandexgpt') !== false);
    $isClaudeModel = (strpos($model, 'claude') !== false || strpos($model, 'anthropic') !== false);
    $useOpenRouter = false;
    
    if (!$isYandexGPT && !$isClaudeModel && defined('USE_OPENROUTER_FIRST') && USE_OPENROUTER_FIRST) {
        $useOpenRouter = true;
    }
    
    // Добавляем инструкции по экстренной помощи
    $questionLower = mb_strtolower($question);
    $emergencyInstructions = '';
    
    if (preg_match('/(ребенок.*температур|температур.*ребенок|детск.*температур)/ui', $questionLower)) {
        $emergencyInstructions = "\n\nВАЖНО: При температуре у ребенка укажи, что при температуре выше 38.5°C у грудных детей или при наличии других тревожных симптомов (сыпь, рвота, вялость, отказ от питья) необходимо срочно обратиться к педиатру или вызвать скорую.";
    }
    
    if (!empty($emergencyInstructions)) {
        $systemPrompt .= $emergencyInstructions;
    }
    
    $userPrompt = "Вопрос пользователя: " . $question;
    
    // Используем GPU если включен
    $useGPU = defined('GPU_LLM_ENABLED') && GPU_LLM_ENABLED;
    
    if ($useGPU) {
        // Загружаем функцию callGPUAPIStage1 из query-stage1_1.php
        // Но делаем это через eval, чтобы не выполнять верхний уровень файла
        $fileContent = file_get_contents(__DIR__ . '/api/query-stage1_1.php');
        // Извлекаем только функции, пропуская верхний уровень
        // Это сложно, поэтому лучше вызвать через API
        
        // Вместо этого вызываем API через curl
        return callStage1ViaAPI($question, $systemPrompt);
    }
    
    throw new Exception('GPU не включен или модель не поддерживается');
}

// Функция для вызова API через curl с кастомным промптом
function callStage1ViaAPI($question, $customPrompt) {
    // Временно изменяем промпт в БД
    $db = getDatabase();
    $widgetKey = 'ekamedcenter';
    
    // Получаем текущий промпт из widget_settings
    $stmt = $db->prepare("
        SELECT w.id, ws.stage1_prompt 
        FROM widgets w 
        LEFT JOIN widget_settings ws ON w.id = ws.widget_id 
        WHERE w.widget_key = ?
    ");
    $stmt->execute([$widgetKey]);
    $widget = $stmt->fetch();
    
    if (!$widget) {
        throw new Exception('Виджет не найден');
    }
    
    $widgetId = $widget['id'];
    $originalPrompt = $widget['stage1_prompt'];
    
    // Временно устанавливаем новый промпт в widget_settings
    $stmt = $db->prepare("UPDATE widget_settings SET stage1_prompt = ? WHERE widget_id = ?");
    $stmt->execute([$customPrompt, $widgetId]);
    
    try {
        // Вызываем API
        $data = [
            'widget_key' => $widgetKey,
            'question' => $question
        ];
        
        $ch = curl_init('https://wtest.medmaps.ru/api/query-stage1_1.php');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200 || !$response) {
            throw new Exception("HTTP {$httpCode}: " . substr($response, 0, 200));
        }
        
        $decoded = json_decode($response, true);
        if (!$decoded || !isset($decoded['text'])) {
            throw new Exception('Неверный формат ответа API');
        }
        
        return [
            'text' => $decoded['text'],
            'provider' => $decoded['provider'] ?? 'unknown'
        ];
    } finally {
        // Восстанавливаем оригинальный промпт
        $stmt = $db->prepare("
            UPDATE widget_settings ws 
            JOIN widgets w ON ws.widget_id = w.id 
            SET ws.stage1_prompt = ? 
            WHERE w.widget_key = ?
        ");
        $stmt->execute([$originalPrompt, $widgetKey]);
    }
}

$widgetKey = 'ekamedcenter';
$question = 'Ветрянка у детей';

echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║  ТЕСТ ПРОМПТОВ ДЛЯ ФРАЗЫ: \"{$question}\"                                 ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

// Получаем виджет из БД (для получения модели)
$db = getDatabase();
$stmt = $db->prepare("
    SELECT w.*, ws.*
    FROM widgets w
    LEFT JOIN widget_settings ws ON w.id = ws.widget_id
    WHERE w.widget_key = ? AND w.active = 1
");
$stmt->execute([$widgetKey]);
$widget = $stmt->fetch();

if (!$widget) {
    die("Виджет не найден!\n");
}

// Варианты промптов для тестирования
$promptVariants = [
    'variant_1_basic' => [
        'name' => 'Базовый промпт (текущий)',
        'prompt' => $widget['stage1_prompt'] ?? 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей кратко и по делу (2-3 предложения).

ПРАВИЛА:
1. Используй только русский язык (кириллица), без латиницы, цифр и технических символов.
2. Пиши полными русскими словами, без аббревиатур и транслитерации.
3. Структура ответа: опиши проблему → укажи 2-3 причины → заверши рекомендацией обратиться к врачу.'
    ],
    
    'variant_2_detailed' => [
        'name' => 'Детальный промпт с акцентом на завершенность',
        'prompt' => 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей СТРОГО НА РУССКОМ ЯЗЫКЕ, информативно и полезно (3-5 предложений). ВАЖНО: используй только русский язык, никакого китайского или других языков! Давай практические советы, когда это уместно, но всегда подчеркивай необходимость консультации с врачом.

КРИТИЧЕСКИ ВАЖНО - ЗАВЕРШЕННОСТЬ ТЕКСТА:
- Текст ДОЛЖЕН заканчиваться полным предложением с точкой
- НЕ обрывай текст на середине предложения, запятой, союзе или предлоге
- НЕ заканчивай на фразах типа "может быть вызвана.", "могут быть связаны с", "проблемы с."
- НЕ используй дефисы или тире вместо завершения
- Если начинаешь перечислять причины, обязательно заверши перечисление и закончи предложение
- ПРОВЕРЬ: последнее предложение должно быть завершено точкой и содержать полную мысль!

СТРУКТУРА ОТВЕТА:
1. Краткое описание проблемы/заболевания
2. Основные симптомы или характеристики
3. Рекомендации по обращению к специалисту (конкретный специалист, например: педиатр, дерматолог)'
    ],
    
    'variant_3_pediatric_focused' => [
        'name' => 'Промпт с акцентом на детские заболевания',
        'prompt' => 'Ты - помощник медицинской клиники, специализирующейся на детском здоровье. Отвечай на вопросы о детских заболеваниях СТРОГО НА РУССКОМ ЯЗЫКЕ, информативно и понятно для родителей (4-6 предложений).

ОСОБЕННОСТИ ОТВЕТОВ О ДЕТСКИХ ЗАБОЛЕВАНИЯХ:
- Опиши заболевание простым языком, понятным родителям
- Укажи основные симптомы, характерные для детей
- Объясни, когда необходимо обратиться к педиатру
- Дай краткие рекомендации по первой помощи (если уместно)
- Всегда подчеркни важность консультации с детским врачом

КРИТИЧЕСКИ ВАЖНО:
- Используй ТОЛЬКО русский язык (кириллица)
- НЕ используй транслитерацию медицинских терминов
- Текст ДОЛЖЕН быть завершенным - заканчивай полным предложением с точкой
- Указывай конкретных специалистов (педиатр, дерматолог), а не просто "врач"'
    ],
    
    'variant_4_structured' => [
        'name' => 'Структурированный промпт с четкими блоками',
        'prompt' => 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей СТРОГО НА РУССКОМ ЯЗЫКЕ.

ФОРМАТ ОТВЕТА (обязательно соблюдай):
1. ОПИСАНИЕ: Что это за заболевание/проблема (1-2 предложения)
2. СИМПТОМЫ/ХАРАКТЕРИСТИКИ: Основные признаки (1-2 предложения)
3. РЕКОМЕНДАЦИИ: К какому специалисту обратиться и почему (1-2 предложения)

ПРАВИЛА:
- Используй ТОЛЬКО русский язык (кириллица)
- НЕ используй транслитерацию (например, "педиатр", а не "pediatrician")
- НЕ используй английские слова или фразы
- Каждый блок должен быть завершен полным предложением с точкой
- Последнее предложение должно содержать рекомендацию обратиться к конкретному специалисту'
    ],
    
    'variant_5_conversational' => [
        'name' => 'Разговорный стиль с акцентом на понятность',
        'prompt' => 'Ты - дружелюбный помощник медицинской клиники. Отвечай на вопросы родителей о здоровье детей простым, понятным языком, как будто объясняешь другу (4-6 предложений).

СТИЛЬ ОБЩЕНИЯ:
- Используй простые слова, избегай сложных медицинских терминов без объяснения
- Будь дружелюбным, но профессиональным
- Объясняй так, чтобы родитель понял, что происходит и что делать

СТРОГО НА РУССКОМ ЯЗЫКЕ:
- ТОЛЬКО кириллица, никакой латиницы
- НЕ используй транслитерацию медицинских терминов
- НЕ смешивай русский и английский языки

ЗАВЕРШЕННОСТЬ:
- Текст ДОЛЖЕН заканчиваться полным предложением с точкой
- Последнее предложение должно содержать четкую рекомендацию (например: "Рекомендуется обратиться к педиатру для осмотра и назначения лечения")'
    ],
    
    'variant_6_minimal' => [
        'name' => 'Минималистичный промпт',
        'prompt' => 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей на русском языке, кратко и по делу. Указывай конкретного специалиста для обращения.'
    ],
    
    'variant_7_expert' => [
        'name' => 'Экспертный промпт с акцентом на точность',
        'prompt' => 'Ты - медицинский консультант клиники. Отвечай на вопросы пользователей профессионально, но понятно (3-5 предложений).

ТРЕБОВАНИЯ:
1. ЯЗЫК: ТОЛЬКО русский язык (кириллица). НЕ используй транслитерацию, английские слова или смешанные фразы.
2. СОДЕРЖАНИЕ: 
   - Дай краткое определение/описание заболевания
   - Укажи ключевые симптомы или характеристики
   - Рекомендуй конкретного специалиста (педиатр, дерматолог и т.д.)
3. ЗАВЕРШЕННОСТЬ: Текст ДОЛЖЕН заканчиваться полным предложением с точкой. НЕ обрывай на середине мысли.
4. ТОЧНОСТЬ: Используй правильные медицинские термины на русском языке.

ПРИМЕР ХОРОШЕГО ОТВЕТА:
"Ветряная оспа (ветрянка) - это вирусное заболевание, которое чаще всего встречается у детей. Основные симптомы: повышение температуры, появление пузырьковой сыпи по всему телу, зуд. Рекомендуется обратиться к педиатру для осмотра и назначения лечения."'
    ]
];

$results = [];

// Тестируем каждый вариант промпта
foreach ($promptVariants as $variantKey => $variant) {
    echo "\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "ТЕСТ: {$variant['name']}\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    // Получаем модель из настроек виджета
    $model = $widget['stage1_model'] ?? GPU_LLM_MODEL;
    
    // Вызываем API с кастомным промптом
    try {
        $startTime = microtime(true);
        $response = callStage1ViaAPI($question, $variant['prompt']);
        $endTime = microtime(true);
        $duration = round($endTime - $startTime, 2);
        
        $text = $response['text'] ?? '';
        $provider = $response['provider'] ?? 'unknown';
        $textLength = mb_strlen($text);
        
        echo "  → Ответ получен ({$provider}, {$duration} сек, {$textLength} символов)\n\n";
        echo "  📄 ТЕКСТ ОТВЕТА:\n";
        echo "  " . str_replace("\n", "\n  ", $text) . "\n\n";
        
        // Проверка на артефакты
        $artifacts = [];
        if (preg_match('/\.(runtuit|jpg|jpeg|png|gif|webp|svg)[_\w\d]*/i', $text)) {
            $artifacts[] = 'Ссылки на изображения';
        }
        if (preg_match('/\b(other\s+symptoms?|other\s+conditions?|Apply\s+ice|appropriate\s+лечение|appropriate\s+treatment)\b/ui', $text)) {
            $artifacts[] = 'Смешанные русско-английские фразы';
        }
        if (preg_match('/\b(endocrinolog|gastroenterolog|pulmonologist|respirologist|orthopedic|ophthalmologist|dermatologist|allergologist|immunologist|cardiologist|neurologist|rheumatologist|urologist|gynecologist|surgeon|therapist|pediatrician)[уеаио]\b/ui', $text)) {
            $artifacts[] = 'Транслитерация медицинских терминов';
        }
        if (preg_match('/\.\s*([A-Z][a-z]+(?:\s+[a-z]+)*\s+(?:can|should|is|are|may|might|will|would|could)\s+[^.]*\.)/u', $text)) {
            $artifacts[] = 'Английские предложения';
        }
        if (preg_match('/[\x{4e00}-\x{9fff}]/u', $text)) {
            $artifacts[] = 'Китайские символы';
        }
        if (preg_match('/[^.!?]\s*$/', trim($text)) && mb_strlen($text) < 50) {
            $artifacts[] = 'Незавершенный текст';
        }
        if (preg_match('/<[^>]+>/', $text)) {
            $artifacts[] = 'HTML теги';
        }
        if (preg_match('/(http:\/\/|https:\/\/|www\.)/i', $text)) {
            $artifacts[] = 'URL в тексте';
        }
        
        // Проверка завершенности
        $isComplete = preg_match('/[.!?]\s*$/', trim($text));
        $hasGoodLength = $textLength >= 100 && $textLength <= 1000;
        
        // Проверка специалиста
        $hasSpecialist = preg_match('/\b(педиатр|дерматолог|терапевт|врач|доктор)\b/ui', $text);
        $hasSpecificSpecialist = preg_match('/\b(педиатр|дерматолог)\b/ui', $text);
        
        echo "  📊 АНАЛИЗ:\n";
        echo "    Длина: {$textLength} символов " . ($hasGoodLength ? "✅" : "⚠️") . "\n";
        echo "    Завершенность: " . ($isComplete ? "✅ Завершен" : "❌ Не завершен") . "\n";
        echo "    Специалист: " . ($hasSpecificSpecialist ? "✅ Конкретный" : ($hasSpecialist ? "⚠️ Общий" : "❌ Не указан")) . "\n";
        
        if (!empty($artifacts)) {
            echo "    ⚠️ Артефакты: " . implode(", ", $artifacts) . "\n";
        } else {
            echo "    ✅ Артефакты не обнаружены\n";
        }
        
        // Оценка через AI-эксперта
        echo "\n  🤖 Оценка AI-экспертом...\n";
        
        $artifactsInfo = !empty($artifacts) ? "\n\n⚠️ ОБНАРУЖЕНЫ АРТЕФАКТЫ В ТЕКСТЕ:\n" . implode("\n", array_map(function($a) { return "- " . $a; }, $artifacts)) : "";
        
        $expertPrompt = "Ты - строгий эксперт по медицинским виджетам. Проверь ТЕКСТОВЫЙ ответ виджета медицинской клиники на наличие проблем.

ВОПРОС ПОЛЬЗОВАТЕЛЯ:
\"{$question}\"

ТЕКСТОВЫЙ ОТВЕТ ВИДЖЕТА:
" . ($text ? $text : "(текст отсутствует)") . $artifactsInfo . "

ТВОЯ ЗАДАЧА - проверить текст на три критерия:

1. АРТЕФАКТЫ:
   Проверь наличие в тексте:
   - Ссылок на изображения (.jpg, .png, .runtuit и т.п.)
   - Смешанных русско-английских фраз (например: \"appropriate лечение\", \"Apply ice\")
   - Транслитерации медицинских терминов (например: \"endocrinologistу\", \"cardiologistе\")
   - Английских предложений в русском тексте
   - Китайских символов
   - HTML тегов
   - URL адресов

2. ЗАВЕРШЕННОСТЬ ПРЕДЛОЖЕНИЙ:
   Проверь, что текст не обрывается на середине предложения.
   Текст должен заканчиваться точкой, восклицательным или вопросительным знаком.
   Если текст обрывается неожиданно - это проблема.

3. АДЕКВАТНОСТЬ И РЕЛЕВАНТНОСТЬ:
   Проверь, что текст:
   - Написан на русском языке (без английских предложений)
   - Имеет смысл и логичен
   - Релевантен вопросу о ветрянке у детей
   - Содержит полезную информацию для родителей
   - Указывает на необходимость обращения к специалисту

ОЦЕНКА (0-10 баллов):
- 10: текст идеален, нет артефактов, предложения завершены, текст адекватен и релевантен
- 8-9: текст хороший, есть незначительные проблемы
- 6-7: текст приемлемый, но есть заметные проблемы
- 4-5: текст плохой, много проблем
- 0-3: текст очень плохой или отсутствует

ФОРМАТ ОТВЕТА (строго JSON):
{
  \"score\": <число 0-10>,
  \"verdict\": \"ОТЛИЧНО\" | \"ХОРОШО\" | \"ПЛОХО\" | \"УЖАСНО\",
  \"has_artifacts\": <true/false>,
  \"is_complete\": <true/false>,
  \"is_adequate\": <true/false>,
  \"comment\": \"<краткий комментарий о найденных проблемах>\",
  \"issues\": [\"<список проблем, если есть>\"],
  \"text_artifacts\": [\"<список найденных артефактов в тексте, если есть>\"],
  \"text_adequacy\": \"<адекватен ли текст: ДА/НЕТ/ЧАСТИЧНО, с пояснением>\",
  \"completeness\": \"<завершены ли предложения: ДА/НЕТ/ЧАСТИЧНО, с пояснением>\"
}

ВЕРДИКТЫ:
- ОТЛИЧНО: 9-10 баллов (нет проблем)
- ХОРОШО: 7-8 баллов (незначительные проблемы)
- ПЛОХО: 4-6 баллов (заметные проблемы)
- УЖАСНО: 0-3 балла (критичные проблемы или текст отсутствует)

Отвечай ТОЛЬКО валидным JSON, без дополнительного текста!";
        
        // Вызываем AI-эксперта
        $expertApiData = [
            'model' => 'xiaomi/mimo-v2-flash:free',
            'messages' => [
                ['role' => 'user', 'content' => $expertPrompt]
            ]
        ];
        
        $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($expertApiData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . OPENROUTER_API_KEY,
            'HTTP-Referer: https://wtest.medmaps.ru',
            'X-Title: Medical Widget Tester'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        
        $expertResponse = curl_exec($ch);
        $expertHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        $expertResult = null;
        if ($expertHttpCode === 200 && $expertResponse) {
            $expertDecoded = json_decode($expertResponse, true);
            $expertContent = $expertDecoded['choices'][0]['message']['content'] ?? '';
            
            if (preg_match('/\{[^{}]*(?:\{[^{}]*\}[^{}]*)*\}/s', $expertContent, $matches)) {
                $expertResult = json_decode($matches[0], true);
            }
        }
        
        if ($expertResult && isset($expertResult['score'])) {
            $score = $expertResult['score'];
            $verdict = $expertResult['verdict'];
            $emoji = [
                'ОТЛИЧНО' => '⭐',
                'ХОРОШО' => '✅',
                'ПЛОХО' => '⚠️',
                'УЖАСНО' => '❌'
            ][$verdict] ?? '❓';
            
            echo "  {$emoji} ОЦЕНКА: {$score}/10 | ВЕРДИКТ: {$verdict}\n";
            echo "  💬 Комментарий: {$expertResult['comment']}\n";
            
            if (!empty($expertResult['issues'])) {
                echo "  ⚠️ Проблемы: " . implode(", ", $expertResult['issues']) . "\n";
            }
        } else {
            echo "  ⚠️ Не удалось получить оценку от AI-эксперта\n";
            $score = 0;
            $verdict = 'ОШИБКА';
        }
        
        // Сохраняем результат
        $results[$variantKey] = [
            'name' => $variant['name'],
            'prompt' => $variant['prompt'],
            'text' => $text,
            'text_length' => $textLength,
            'provider' => $provider,
            'duration' => $duration,
            'artifacts' => $artifacts,
            'is_complete' => $isComplete,
            'has_good_length' => $hasGoodLength,
            'has_specialist' => $hasSpecialist,
            'has_specific_specialist' => $hasSpecificSpecialist,
            'expert_score' => $score,
            'expert_verdict' => $verdict,
            'expert_comment' => $expertResult['comment'] ?? '',
            'expert_issues' => $expertResult['issues'] ?? []
        ];
        
    } catch (Exception $e) {
        echo "  ❌ ОШИБКА: " . $e->getMessage() . "\n";
        $results[$variantKey] = [
            'name' => $variant['name'],
            'error' => $e->getMessage()
        ];
    }
    
    // Пауза между запросами
    usleep(1000000); // 1 секунда
}

// ===== ИТОГОВАЯ СТАТИСТИКА =====
echo "\n\n";
echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║                        ИТОГОВАЯ СТАТИСТИКА                               ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

// Сортируем результаты по оценке эксперта
uasort($results, function($a, $b) {
    $scoreA = $a['expert_score'] ?? 0;
    $scoreB = $b['expert_score'] ?? 0;
    return $scoreB <=> $scoreA;
});

echo "РЕЙТИНГ ПРОМПТОВ (по оценке AI-эксперта):\n\n";

$emojiMap = [
    'ОТЛИЧНО' => '⭐',
    'ХОРОШО' => '✅',
    'ПЛОХО' => '⚠️',
    'УЖАСНО' => '❌'
];

foreach ($results as $variantKey => $result) {
    if (isset($result['error'])) {
        echo "  ❌ {$result['name']}: ОШИБКА - {$result['error']}\n\n";
        continue;
    }
    
    $score = $result['expert_score'] ?? 0;
    $verdict = $result['expert_verdict'] ?? 'НЕТ ОЦЕНКИ';
    $emoji = $emojiMap[$verdict] ?? '❓';
    
    echo "  {$emoji} {$result['name']}\n";
    echo "     Оценка: {$score}/10 ({$verdict})\n";
    echo "     Длина: {$result['text_length']} символов\n";
    echo "     Время: {$result['duration']} сек\n";
    echo "     Завершенность: " . ($result['is_complete'] ? "✅" : "❌") . "\n";
    echo "     Специалист: " . ($result['has_specific_specialist'] ? "✅ Конкретный" : ($result['has_specialist'] ? "⚠️ Общий" : "❌ Нет")) . "\n";
    if (!empty($result['artifacts'])) {
        echo "     Артефакты: " . implode(", ", $result['artifacts']) . "\n";
    }
    echo "     Комментарий: {$result['expert_comment']}\n";
    echo "\n";
}

// Лучший промпт
$bestVariant = null;
$bestScore = 0;
foreach ($results as $variantKey => $result) {
    if (!isset($result['error']) && isset($result['expert_score'])) {
        if ($result['expert_score'] > $bestScore) {
            $bestScore = $result['expert_score'];
            $bestVariant = $variantKey;
        }
    }
}

if ($bestVariant) {
    $best = $results[$bestVariant];
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "🏆 ЛУЧШИЙ ПРОМПТ: {$best['name']}\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    echo "ОЦЕНКА: {$best['expert_score']}/10 ({$best['expert_verdict']})\n\n";
    echo "ПРОМПТ:\n";
    echo str_repeat('-', 80) . "\n";
    echo $best['prompt'] . "\n";
    echo str_repeat('-', 80) . "\n\n";
    echo "ОТВЕТ:\n";
    echo str_repeat('-', 80) . "\n";
    echo $best['text'] . "\n";
    echo str_repeat('-', 80) . "\n\n";
}

// Сохраняем результаты
$timestamp = date('Y-m-d_H-i-s');
$jsonFile = "/root/test-chickenpox-prompts-{$timestamp}.json";
file_put_contents($jsonFile, json_encode($results, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

echo "\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "Результаты сохранены: {$jsonFile}\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
