<?php
/**
 * Улучшенные промпты для фразы "Ветрянка у детей"
 * Тестируем варианты с более строгими инструкциями против артефактов
 */

require_once '/root/config.php';

// Используем ту же функцию callStage1ViaAPI из предыдущего скрипта
function callStage1ViaAPI($question, $customPrompt) {
    $db = getDatabase();
    $widgetKey = 'ekamedcenter';
    
    $stmt = $db->prepare("
        SELECT w.id, ws.stage1_prompt 
        FROM widgets w 
        LEFT JOIN widget_settings ws ON w.id = ws.widget_id 
        WHERE w.widget_key = ?
    ");
    $stmt->execute([$widgetKey]);
    $widget = $stmt->fetch();
    
    if (!$widget) {
        throw new Exception('Виджет не найден');
    }
    
    $widgetId = $widget['id'];
    $originalPrompt = $widget['stage1_prompt'];
    
    $stmt = $db->prepare("UPDATE widget_settings SET stage1_prompt = ? WHERE widget_id = ?");
    $stmt->execute([$customPrompt, $widgetId]);
    
    try {
        $data = [
            'widget_key' => $widgetKey,
            'question' => $question
        ];
        
        $ch = curl_init('https://wtest.medmaps.ru/api/query-stage1_1.php');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200 || !$response) {
            throw new Exception("HTTP {$httpCode}: " . substr($response, 0, 200));
        }
        
        $decoded = json_decode($response, true);
        if (!$decoded || !isset($decoded['text'])) {
            throw new Exception('Неверный формат ответа API');
        }
        
        return [
            'text' => $decoded['text'],
            'provider' => $decoded['provider'] ?? 'unknown'
        ];
    } finally {
        $stmt = $db->prepare("
            UPDATE widget_settings ws 
            JOIN widgets w ON ws.widget_id = w.id 
            SET ws.stage1_prompt = ? 
            WHERE w.widget_key = ?
        ");
        $stmt->execute([$originalPrompt, $widgetKey]);
    }
}

$widgetKey = 'ekamedcenter';
$question = 'Ветрянка у детей';

echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║  ТЕСТ УЛУЧШЕННЫХ ПРОМПТОВ: \"{$question}\"                                ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

// Улучшенные варианты промптов с акцентом на борьбу с артефактами
$improvedPrompts = [
    'improved_1_strict' => [
        'name' => 'Строгий промпт против артефактов',
        'prompt' => 'Ты - помощник медицинской клиники. Отвечай на вопросы родителей о здоровье детей СТРОГО НА РУССКОМ ЯЗЫКЕ (кириллица).

КРИТИЧЕСКИ ВАЖНО - ЗАПРЕЩЕНО:
- НИКАКИХ английских слов, букв или фраз (by, is, can, who, not, for, it, a, the и т.д.)
- НИКАКИХ смешанных русско-английских фраз
- НИКАКИХ транслитераций медицинских терминов
- НИКАКОЙ латиницы в тексте ответа
- ТОЛЬКО русские слова на кириллице

СТРУКТУРА ОТВЕТА (4-5 предложений):
1. Что это за заболевание (простыми словами для родителей)
2. Основные симптомы у детей
3. Что делать родителям
4. К какому специалисту обратиться (конкретно: педиатр)

ПРОВЕРКА ПЕРЕД ОТВЕТОМ:
- Убедись, что в ответе ТОЛЬКО русские слова (кириллица)
- Убедись, что текст заканчивается полным предложением с точкой
- Убедись, что указан конкретный специалист (педиатр)

ПРИМЕР ПРАВИЛЬНОГО ОТВЕТА:
"Ветрянка (ветряная оспа) - это вирусное заболевание, которое чаще всего встречается у детей. Основные симптомы: повышение температуры, появление пузырьковой сыпи по всему телу, зуд. Родителям важно обеспечить ребенку покой, обильное питье и следить за температурой. Рекомендуется обратиться к педиатру для осмотра и назначения лечения."'
    ],
    
    'improved_2_step_by_step' => [
        'name' => 'Пошаговый промпт с проверками',
        'prompt' => 'Ты - помощник медицинской клиники. Отвечай на вопросы о детских заболеваниях.

ШАГ 1 - ОПИСАНИЕ (1 предложение):
Опиши заболевание простыми словами для родителей.

ШАГ 2 - СИМПТОМЫ (1-2 предложения):
Перечисли основные симптомы у детей.

ШАГ 3 - РЕКОМЕНДАЦИИ (1-2 предложения):
Что делать родителям (первая помощь, уход).

ШАГ 4 - СПЕЦИАЛИСТ (1 предложение):
К какому врачу обратиться (обязательно укажи: педиатр).

СТРОГО ЗАПРЕЩЕНО:
- Использовать английские слова (by, is, can, who, not, for, it, a, the)
- Смешивать русский и английский языки
- Использовать транслитерацию
- Обрывать текст на середине предложения

ПРОВЕРЬ ОТВЕТ:
1. Все слова на русском языке (кириллица)? ДА/НЕТ
2. Текст заканчивается точкой? ДА/НЕТ
3. Указан специалист (педиатр)? ДА/НЕТ

Если хотя бы один ответ НЕТ - перепиши ответ!'
    ],
    
    'improved_3_example_based' => [
        'name' => 'Промпт с примером правильного ответа',
        'prompt' => 'Ты - помощник медицинской клиники. Отвечай на вопросы родителей о здоровье детей.

ПРАВИЛА:
1. ТОЛЬКО русский язык (кириллица) - никаких английских слов
2. Простой язык, понятный родителям
3. Полные предложения, заканчивающиеся точкой
4. Обязательно укажи специалиста (педиатр)

ПРИМЕР ПРАВИЛЬНОГО ОТВЕТА НА ВОПРОС "Ветрянка у детей":
"Ветрянка (ветряная оспа) - это вирусное заболевание, которое чаще всего встречается у детей дошкольного и школьного возраста. Основные симптомы включают повышение температуры тела, появление красных пятен на коже, которые затем превращаются в пузырьки с прозрачной жидкостью, вызывающие сильный зуд. Заболевание обычно длится 7-10 дней. Родителям важно обеспечить ребенку покой, обильное питье, следить за температурой и не позволять расчесывать высыпания. Рекомендуется обратиться к педиатру для осмотра и назначения лечения."

ТВОЯ ЗАДАЧА:
Ответь на вопрос пользователя, следуя примеру выше. Используй ТОЛЬКО русские слова, никаких английских!'
    ],
    
    'improved_4_negative_instructions' => [
        'name' => 'Промпт с негативными инструкциями',
        'prompt' => 'Ты - помощник медицинской клиники. Отвечай на вопросы родителей о здоровье детей.

ЧТО НЕЛЬЗЯ ДЕЛАТЬ (ЗАПРЕЩЕНО):
❌ НЕ используй английские слова: by, is, can, who, not, for, it, a, the, has, if, but
❌ НЕ смешивай русский и английский языки
❌ НЕ используй транслитерацию медицинских терминов
❌ НЕ обрывай текст на середине предложения
❌ НЕ заканчивай на запятой, союзе или предлоге

ЧТО НУЖНО ДЕЛАТЬ (ОБЯЗАТЕЛЬНО):
✅ Используй ТОЛЬКО русский язык (кириллица)
✅ Пиши простыми словами для родителей
✅ Заканчивай каждое предложение точкой
✅ Указывай конкретного специалиста (педиатр)

СТРУКТУРА ОТВЕТА:
1. Что это за заболевание (1 предложение)
2. Симптомы у детей (1-2 предложения)
3. Что делать родителям (1 предложение)
4. К какому врачу обратиться (1 предложение, обязательно: педиатр)'
    ]
];

$results = [];

foreach ($improvedPrompts as $variantKey => $variant) {
    echo "\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "ТЕСТ: {$variant['name']}\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    try {
        $startTime = microtime(true);
        $response = callStage1ViaAPI($question, $variant['prompt']);
        $endTime = microtime(true);
        $duration = round($endTime - $startTime, 2);
        
        $text = $response['text'] ?? '';
        $provider = $response['provider'] ?? 'unknown';
        $textLength = mb_strlen($text);
        
        echo "  → Ответ получен ({$provider}, {$duration} сек, {$textLength} символов)\n\n";
        echo "  📄 ТЕКСТ ОТВЕТА:\n";
        echo "  " . str_replace("\n", "\n  ", $text) . "\n\n";
        
        // Проверка на артефакты
        $artifacts = [];
        $hasEnglishWords = preg_match('/\b(by|is|can|who|not|for|it|a|the|has|if|but|and|or|to|of|in|on|at|with|from|as|an|this|that|these|those|are|was|were|been|being|have|had|has|do|does|did|will|would|should|could|may|might|must)\b/i', $text);
        if ($hasEnglishWords) {
            $artifacts[] = 'Английские слова в тексте';
        }
        if (preg_match('/\.(runtuit|jpg|jpeg|png|gif|webp|svg)[_\w\d]*/i', $text)) {
            $artifacts[] = 'Ссылки на изображения';
        }
        if (preg_match('/\b(other\s+symptoms?|other\s+conditions?|Apply\s+ice|appropriate\s+лечение|appropriate\s+treatment)\b/ui', $text)) {
            $artifacts[] = 'Смешанные русско-английские фразы';
        }
        if (preg_match('/\b(endocrinolog|gastroenterolog|pulmonologist|respirologist|orthopedic|ophthalmologist|dermatologist|allergologist|immunologist|cardiologist|neurologist|rheumatologist|urologist|gynecologist|surgeon|therapist|pediatrician)[уеаио]\b/ui', $text)) {
            $artifacts[] = 'Транслитерация медицинских терминов';
        }
        if (preg_match('/\.\s*([A-Z][a-z]+(?:\s+[a-z]+)*\s+(?:can|should|is|are|may|might|will|would|could)\s+[^.]*\.)/u', $text)) {
            $artifacts[] = 'Английские предложения';
        }
        if (preg_match('/[\x{4e00}-\x{9fff}]/u', $text)) {
            $artifacts[] = 'Китайские символы';
        }
        if (preg_match('/[^.!?]\s*$/', trim($text)) && mb_strlen($text) < 50) {
            $artifacts[] = 'Незавершенный текст';
        }
        
        // Проверка завершенности
        $isComplete = preg_match('/[.!?]\s*$/', trim($text));
        $hasGoodLength = $textLength >= 150 && $textLength <= 800;
        
        // Проверка специалиста
        $hasSpecialist = preg_match('/\b(педиатр|дерматолог|терапевт|врач|доктор)\b/ui', $text);
        $hasSpecificSpecialist = preg_match('/\b(педиатр)\b/ui', $text);
        
        echo "  📊 АНАЛИЗ:\n";
        echo "    Длина: {$textLength} символов " . ($hasGoodLength ? "✅" : "⚠️") . "\n";
        echo "    Завершенность: " . ($isComplete ? "✅ Завершен" : "❌ Не завершен") . "\n";
        echo "    Специалист: " . ($hasSpecificSpecialist ? "✅ Конкретный (педиатр)" : ($hasSpecialist ? "⚠️ Общий" : "❌ Не указан")) . "\n";
        
        if (!empty($artifacts)) {
            echo "    ⚠️ Артефакты: " . implode(", ", $artifacts) . "\n";
        } else {
            echo "    ✅ Артефакты не обнаружены\n";
        }
        
        // Оценка через AI-эксперта
        echo "\n  🤖 Оценка AI-экспертом...\n";
        
        $artifactsInfo = !empty($artifacts) ? "\n\n⚠️ ОБНАРУЖЕНЫ АРТЕФАКТЫ В ТЕКСТЕ:\n" . implode("\n", array_map(function($a) { return "- " . $a; }, $artifacts)) : "";
        
        $expertPrompt = "Ты - строгий эксперт по медицинским виджетам. Проверь ТЕКСТОВЫЙ ответ виджета медицинской клиники.

ВОПРОС ПОЛЬЗОВАТЕЛЯ:
\"{$question}\"

ТЕКСТОВЫЙ ОТВЕТ ВИДЖЕТА:
" . ($text ? $text : "(текст отсутствует)") . $artifactsInfo . "

ОЦЕНКА (0-10 баллов):
- 10: идеальный текст на русском, нет артефактов, завершен, релевантен, указан специалист
- 8-9: хороший текст, незначительные проблемы
- 6-7: приемлемый текст, заметные проблемы
- 4-5: плохой текст, много проблем
- 0-3: очень плохой текст или отсутствует

ФОРМАТ ОТВЕТА (строго JSON):
{
  \"score\": <число 0-10>,
  \"verdict\": \"ОТЛИЧНО\" | \"ХОРОШО\" | \"ПЛОХО\" | \"УЖАСНО\",
  \"has_artifacts\": <true/false>,
  \"is_complete\": <true/false>,
  \"is_adequate\": <true/false>,
  \"has_specialist\": <true/false>,
  \"comment\": \"<краткий комментарий>\",
  \"issues\": [\"<список проблем, если есть>\"]
}

Отвечай ТОЛЬКО валидным JSON!";
        
        $expertApiData = [
            'model' => 'xiaomi/mimo-v2-flash:free',
            'messages' => [
                ['role' => 'user', 'content' => $expertPrompt]
            ]
        ];
        
        $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($expertApiData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . OPENROUTER_API_KEY,
            'HTTP-Referer: https://wtest.medmaps.ru',
            'X-Title: Medical Widget Tester'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        
        $expertResponse = curl_exec($ch);
        $expertHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        $expertResult = null;
        if ($expertHttpCode === 200 && $expertResponse) {
            $expertDecoded = json_decode($expertResponse, true);
            $expertContent = $expertDecoded['choices'][0]['message']['content'] ?? '';
            
            if (preg_match('/\{[^{}]*(?:\{[^{}]*\}[^{}]*)*\}/s', $expertContent, $matches)) {
                $expertResult = json_decode($matches[0], true);
            }
        }
        
        if ($expertResult && isset($expertResult['score'])) {
            $score = $expertResult['score'];
            $verdict = $expertResult['verdict'];
            $emoji = [
                'ОТЛИЧНО' => '⭐',
                'ХОРОШО' => '✅',
                'ПЛОХО' => '⚠️',
                'УЖАСНО' => '❌'
            ][$verdict] ?? '❓';
            
            echo "  {$emoji} ОЦЕНКА: {$score}/10 | ВЕРДИКТ: {$verdict}\n";
            echo "  💬 Комментарий: {$expertResult['comment']}\n";
            
            if (!empty($expertResult['issues'])) {
                echo "  ⚠️ Проблемы: " . implode(", ", $expertResult['issues']) . "\n";
            }
        } else {
            echo "  ⚠️ Не удалось получить оценку от AI-эксперта\n";
            $score = 0;
            $verdict = 'ОШИБКА';
        }
        
        $results[$variantKey] = [
            'name' => $variant['name'],
            'prompt' => $variant['prompt'],
            'text' => $text,
            'text_length' => $textLength,
            'provider' => $provider,
            'duration' => $duration,
            'artifacts' => $artifacts,
            'is_complete' => $isComplete,
            'has_good_length' => $hasGoodLength,
            'has_specialist' => $hasSpecialist,
            'has_specific_specialist' => $hasSpecificSpecialist,
            'expert_score' => $score,
            'expert_verdict' => $verdict,
            'expert_comment' => $expertResult['comment'] ?? '',
            'expert_issues' => $expertResult['issues'] ?? []
        ];
        
    } catch (Exception $e) {
        echo "  ❌ ОШИБКА: " . $e->getMessage() . "\n";
        $results[$variantKey] = [
            'name' => $variant['name'],
            'error' => $e->getMessage()
        ];
    }
    
    usleep(2000000); // 2 секунды между запросами
}

// Итоговая статистика
echo "\n\n";
echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║                        ИТОГОВАЯ СТАТИСТИКА                               ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

uasort($results, function($a, $b) {
    $scoreA = $a['expert_score'] ?? 0;
    $scoreB = $b['expert_score'] ?? 0;
    return $scoreB <=> $scoreA;
});

echo "РЕЙТИНГ УЛУЧШЕННЫХ ПРОМПТОВ:\n\n";

$emojiMap = [
    'ОТЛИЧНО' => '⭐',
    'ХОРОШО' => '✅',
    'ПЛОХО' => '⚠️',
    'УЖАСНО' => '❌'
];

foreach ($results as $variantKey => $result) {
    if (isset($result['error'])) {
        echo "  ❌ {$result['name']}: ОШИБКА - {$result['error']}\n\n";
        continue;
    }
    
    $score = $result['expert_score'] ?? 0;
    $verdict = $result['expert_verdict'] ?? 'НЕТ ОЦЕНКИ';
    $emoji = $emojiMap[$verdict] ?? '❓';
    
    echo "  {$emoji} {$result['name']}\n";
    echo "     Оценка: {$score}/10 ({$verdict})\n";
    echo "     Длина: {$result['text_length']} символов\n";
    echo "     Время: {$result['duration']} сек\n";
    echo "     Завершенность: " . ($result['is_complete'] ? "✅" : "❌") . "\n";
    echo "     Специалист: " . ($result['has_specific_specialist'] ? "✅ Педиатр" : ($result['has_specialist'] ? "⚠️ Общий" : "❌ Нет")) . "\n";
    if (!empty($result['artifacts'])) {
        echo "     Артефакты: " . implode(", ", $result['artifacts']) . "\n";
    } else {
        echo "     Артефакты: ✅ Нет\n";
    }
    echo "     Комментарий: {$result['expert_comment']}\n";
    echo "\n";
}

// Лучший промпт
$bestVariant = null;
$bestScore = 0;
foreach ($results as $variantKey => $result) {
    if (!isset($result['error']) && isset($result['expert_score'])) {
        if ($result['expert_score'] > $bestScore) {
            $bestScore = $result['expert_score'];
            $bestVariant = $variantKey;
        }
    }
}

if ($bestVariant) {
    $best = $results[$bestVariant];
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "🏆 ЛУЧШИЙ УЛУЧШЕННЫЙ ПРОМПТ: {$best['name']}\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    echo "ОЦЕНКА: {$best['expert_score']}/10 ({$best['expert_verdict']})\n\n";
    echo "ПРОМПТ:\n";
    echo str_repeat('-', 80) . "\n";
    echo $best['prompt'] . "\n";
    echo str_repeat('-', 80) . "\n\n";
    echo "ОТВЕТ:\n";
    echo str_repeat('-', 80) . "\n";
    echo $best['text'] . "\n";
    echo str_repeat('-', 80) . "\n\n";
}

// Сохраняем результаты
$timestamp = date('Y-m-d_H-i-s');
$jsonFile = "/root/test-chickenpox-improved-{$timestamp}.json";
file_put_contents($jsonFile, json_encode($results, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

echo "\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "Результаты сохранены: {$jsonFile}\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
