<?php
/**
 * Тест проблемных вопросов с AI экспертом (Gemini 2.5 Flash)
 */

// Список проблемных вопросов (получили "ПЛОХО" в предыдущем тесте)
$badQuestions = [
    "Увеличение лимфоузлов подмышками",
    "Приступ удушья",
    "Синюшность пальцев",
    "Хроническая обструктивная болезнь легких",
    "Остеопороз профилактика",
    "Подагра приступ",
    "Сухость во рту",
    "Скарлатина симптомы",
    "Утренняя скованность суставов",
    "Люмбаго острая боль",
    "Пневмония симптомы",
    "Потеря обоняния",
    "Отек Квинке",
    "МРТ сустава",
    "Диспансеризация взрослых"
];

// Функция для вызова виджета через api-wrapper
function queryWidget($question, $section) {
    $cmd = sprintf(
        'php /root/api-wrapper.php %s %s %s 2>&1',
        escapeshellarg('ekamedcenter'),
        escapeshellarg($question),
        escapeshellarg($section)
    );
    
    $output = shell_exec($cmd);
    
    // Парсим JSON ответ - берем последнюю строку с JSON
    $lines = explode("\n", trim($output));
    foreach (array_reverse($lines) as $line) {
        if (empty(trim($line))) continue;
        $data = json_decode($line, true);
        if (json_last_error() === JSON_ERROR_NONE && isset($data['data'])) {
            // Формат: {"text":"...", "data":{"specialists":[...], ...}}
            $result = $data['data'];
            $result['text'] = isset($data['text']) ? $data['text'] : '';
            return $result;
        }
    }
    
    return null;
}

// Функция для оценки ответа через AI эксперта
function evaluateWithAI($question, $specialists, $services, $specializations, $text) {
    $openrouterApiKey = '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';
    
    $openrouterModel = 'google/gemini-2.5-flash';
    
    // Формируем промпт для эксперта с учетом экстренных и административных ситуаций
    $prompt = <<<PROMPT
Ты — медицинский эксперт. Оцени качество ответа виджета на медицинский вопрос пациента.

ВОПРОС ПАЦИЕНТА: "$question"

ОТВЕТ ВИДЖЕТА:
- Специалисты: {$specialists}
- Услуги: {$services}
- Специализации: {$specializations}
- Текст: {$text}

КРИТЕРИИ ОЦЕНКИ:

1. **ЭКСТРЕННЫЕ СИТУАЦИИ:**
   - Если вопрос касается экстренной ситуации (сердечный приступ, инсульт, удушье, синюшность, сильное кровотечение):
     * Если текст содержит предупреждение о вызове скорой помощи → оценка "ОТЛИЧНО" (40/40)
     * Если есть только специалисты без предупреждения → оценка снижается

2. **АДМИНИСТРАТИВНЫЕ ЗАПРОСЫ:**
   - Если вопрос касается справок, комиссий, медосмотров (НЕ диагностика симптомов):
     * Если есть релевантные услуги/специалисты или информативный текст → оценка "ХОРОШО" или "ОТЛИЧНО"
     * Если текст объясняет отсутствие услуги → оценка "ХОРОШО" (25-35/40)

3. **ДИАГНОСТИЧЕСКИЕ ЗАПРОСЫ (МРТ, КТ, УЗИ, анализы):**
   - Должны быть услуги диагностики и терапевт/профильный специалист
   - Оценка "ХОРОШО" (25-35/40) при наличии диагностики

4. **МЕДИЦИНСКИЕ СИМПТОМЫ И ЗАБОЛЕВАНИЯ:**
   - Специалисты должны быть релевантны симптомам
   - Услуги должны включать как консультации, так и диагностику (анализы, УЗИ)
   - Специализации должны соответствовать проблеме
   - Если все 3 раздела полные и релевантные → "ОТЛИЧНО" (35-40/40)
   - Если есть недочеты, но в целом правильно → "ХОРОШО" (25-35/40)
   - Если частично правильно → "УДОВЛЕТВОРИТЕЛЬНО" (15-25/40)
   - Если совсем неправильно или пусто → "ПЛОХО" (0-15/40)

ВАЖНО:
- Для редких специальностей (пульмонолог, инфекционист) допустима замена на терапевта
- Обязательно должна быть диагностика (анализы, УЗИ) для симптомов
- Для детских болезней (скарлатина, корь) обязателен педиатр

Ответь строго в формате JSON:
{
  "rating": "ОТЛИЧНО|ХОРОШО|УДОВЛЕТВОРИТЕЛЬНО|ПЛОХО",
  "score": <число от 0 до 40>,
  "reasoning": "<краткое объяснение>"
}
PROMPT;

    $data = [
        'model' => $openrouterModel,
        'messages' => [
            [
                'role' => 'user',
                'content' => $prompt
            ]
        ],
        'temperature' => 0.1,
        'max_tokens' => 500
    ];
    
    $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $openrouterApiKey,
        'Content-Type: application/json',
        'HTTP-Referer: http://localhost',
        'X-Title: Medical Widget Test'
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        return ['rating' => 'ERROR', 'reasoning' => 'CURL error: ' . curl_error($ch)];
    }
    
    curl_close($ch);
    
    $result = json_decode($response, true);
    
    if (!isset($result['choices'][0]['message']['content'])) {
        return ['rating' => 'ERROR', 'reasoning' => 'Invalid API response'];
    }
    
    $content = $result['choices'][0]['message']['content'];
    
    // Извлекаем JSON из ответа
    if (preg_match('/\{[\s\S]*\}/', $content, $matches)) {
        $evaluation = json_decode($matches[0], true);
        if ($evaluation && isset($evaluation['rating'])) {
            return $evaluation;
        }
    }
    
    return ['rating' => 'ERROR', 'reasoning' => 'Failed to parse AI response'];
}

echo "═══════════════════════════════════════════════════════════════════\n";
echo "ТЕСТИРОВАНИЕ ПРОБЛЕМНЫХ ВОПРОСОВ С AI ЭКСПЕРТОМ (Gemini 2.5 Flash)\n";
echo "═══════════════════════════════════════════════════════════════════\n\n";

$totalQuestions = count($badQuestions);
$results = [];

foreach ($badQuestions as $index => $question) {
    $num = $index + 1;
    
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "ВОПРОС {$num}/{$totalQuestions}: {$question}\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    // Запрашиваем все секции
    $specialistsResponse = queryWidget($question, 'specialists');
    $servicesResponse = queryWidget($question, 'services');
    $specializationsResponse = queryWidget($question, 'specializations');
    
    if (!$specialistsResponse && !$servicesResponse && !$specializationsResponse) {
        echo "❌ ОШИБКА: Не удалось получить ответ от виджета\n\n";
        $results[] = [
            'question' => $question,
            'rating' => 'ERROR',
            'score' => 0
        ];
        continue;
    }
    
    $specialists = isset($specialistsResponse['specialists']) ? array_column($specialistsResponse['specialists'], 'name') : [];
    $services = isset($servicesResponse['services']) ? array_column($servicesResponse['services'], 'name') : [];
    $specializations = isset($specializationsResponse['specializations']) ? array_column($specializationsResponse['specializations'], 'name') : [];
    
    // Берем текст из любого ответа, где он есть
    $text = '';
    if (!empty($specialistsResponse['text']) && $specialistsResponse['text'] !== '[]') {
        $text = $specialistsResponse['text'];
    } elseif (!empty($servicesResponse['text']) && $servicesResponse['text'] !== '[]') {
        $text = $servicesResponse['text'];
    } elseif (!empty($specializationsResponse['text']) && $specializationsResponse['text'] !== '[]') {
        $text = $specializationsResponse['text'];
    }
    
    $specialistsStr = !empty($specialists) ? implode(', ', $specialists) : '(нет)';
    $servicesStr = !empty($services) ? implode(', ', array_slice($services, 0, 5)) : '(нет)';
    if (count($services) > 5) $servicesStr .= '... +' . (count($services) - 5);
    $specializationsStr = !empty($specializations) ? implode(', ', $specializations) : '(нет)';
    
    echo "📊 ОТВЕТ ВИДЖЕТА:\n";
    echo "  • Специалисты (" . count($specialists) . "): {$specialistsStr}\n";
    echo "  • Услуги (" . count($services) . "): {$servicesStr}\n";
    echo "  • Специализации (" . count($specializations) . "): {$specializationsStr}\n";
    if ($text) {
        $shortText = mb_substr($text, 0, 150);
        if (mb_strlen($text) > 150) $shortText .= '...';
        echo "  • Текст: {$shortText}\n";
    }
    echo "\n";
    
    // Оценка AI эксперта
    echo "🤖 Запрос оценки от AI эксперта...\n";
    $evaluation = evaluateWithAI(
        $question,
        $specialistsStr,
        $servicesStr,
        $specializationsStr,
        $text
    );
    
    if ($evaluation['rating'] === 'ERROR') {
        echo "❌ ОШИБКА ОЦЕНКИ: {$evaluation['reasoning']}\n\n";
        $results[] = [
            'question' => $question,
            'rating' => 'ERROR',
            'score' => 0
        ];
        continue;
    }
    
    $rating = $evaluation['rating'];
    $score = isset($evaluation['score']) ? $evaluation['score'] : 0;
    $reasoning = isset($evaluation['reasoning']) ? $evaluation['reasoning'] : '';
    
    // Определяем эмодзи
    $emoji = match($rating) {
        'ОТЛИЧНО' => '🟢',
        'ХОРОШО' => '🟡',
        'УДОВЛЕТВОРИТЕЛЬНО' => '🟠',
        'ПЛОХО' => '🔴',
        default => '⚪'
    };
    
    echo "📝 ОЦЕНКА ЭКСПЕРТА: {$emoji} {$rating} ({$score}/40)\n";
    echo "💭 Обоснование: {$reasoning}\n\n";
    
    $results[] = [
        'question' => $question,
        'rating' => $rating,
        'score' => $score,
        'specialists_count' => count($specialists),
        'services_count' => count($services),
        'reasoning' => $reasoning
    ];
    
    // Небольшая задержка между запросами
    usleep(500000); // 0.5 секунды
}

// Итоговая статистика
echo "\n═══════════════════════════════════════════════════════════════════\n";
echo "ИТОГОВАЯ СТАТИСТИКА\n";
echo "═══════════════════════════════════════════════════════════════════\n\n";

$ratingCounts = [
    'ОТЛИЧНО' => 0,
    'ХОРОШО' => 0,
    'УДОВЛЕТВОРИТЕЛЬНО' => 0,
    'ПЛОХО' => 0,
    'ERROR' => 0
];

$totalScore = 0;
$validScores = 0;

foreach ($results as $result) {
    $ratingCounts[$result['rating']]++;
    if ($result['rating'] !== 'ERROR') {
        $totalScore += $result['score'];
        $validScores++;
    }
}

foreach ($ratingCounts as $rating => $count) {
    $percentage = $totalQuestions > 0 ? round(($count / $totalQuestions) * 100, 1) : 0;
    $emoji = match($rating) {
        'ОТЛИЧНО' => '🟢',
        'ХОРОШО' => '🟡',
        'УДОВЛЕТВОРИТЕЛЬНО' => '🟠',
        'ПЛОХО' => '🔴',
        'ERROR' => '⚫'
    };
    echo "{$emoji} {$rating}: {$count} ({$percentage}%)\n";
}

$avgScore = $validScores > 0 ? round($totalScore / $validScores, 1) : 0;
echo "\n📊 Средний балл: {$avgScore}/40\n";

// Список вопросов, которые остались "ПЛОХО"
$stillBad = array_filter($results, fn($r) => $r['rating'] === 'ПЛОХО');
if (!empty($stillBad)) {
    echo "\n🔴 Вопросы, которые остались с оценкой ПЛОХО (" . count($stillBad) . "):\n";
    foreach ($stillBad as $bad) {
        echo "  • {$bad['question']} (спец: {$bad['specialists_count']}, услуг: {$bad['services_count']})\n";
        echo "    → {$bad['reasoning']}\n";
    }
}

echo "\n✅ Тест завершен!\n";

