#!/usr/bin/env php
<?php

// Читаем 79 проблемных вопросов
$questionsFile = '/root/problem-questions-79.txt';
$questions = file($questionsFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);

// Настройки
$widgetKey = 'ekamedcenter';
$openrouterKey = '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';
$openrouterModel = 'google/gemini-2.5-flash';

$timestamp = date('Y-m-d_H-i-s');
$reportFile = "/root/test_79_expert_review_{$timestamp}.txt";
$summaryFile = "/root/test_79_expert_summary_{$timestamp}.txt";

// Функция для вызова виджета
function callWidgetAPI($question, $widgetKey) {
    $sections = ['specialists', 'services', 'specializations'];
    $result = [];
    
    foreach ($sections as $section) {
        $cmd = sprintf(
            'php /root/api-wrapper.php %s %s %s 2>&1',
            escapeshellarg($widgetKey),
            escapeshellarg($question),
            escapeshellarg($section)
        );
        
        $output = shell_exec($cmd);
        $lines = explode("\n", $output);
        $jsonLine = '';
        foreach ($lines as $line) {
            if (strpos($line, '{') === 0) {
                $jsonLine = $line;
                break;
            }
        }
        
        $decoded = json_decode($jsonLine, true);
        
        if ($decoded && isset($decoded['data'][$section])) {
            $result[$section] = $decoded['data'][$section];
        } else {
            $result[$section] = [];
        }
        
        usleep(50000);
    }
    
    return $result;
}

// Функция для запроса к Gemini
function askExpert($question, $widgetResponse, $openrouterKey, $model) {
    $specialists = $widgetResponse['specialists'] ?? [];
    $services = $widgetResponse['services'] ?? [];
    $specializations = $widgetResponse['specializations'] ?? [];
    
    $widgetData = "ЗАПРОС ПАЦИЕНТА: {$question}\n\n";
    $widgetData .= "ОТВЕТ ВИДЖЕТА:\n";
    $widgetData .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    $widgetData .= "СПЕЦИАЛИСТЫ (" . count($specialists) . "):\n";
    if (empty($specialists)) {
        $widgetData .= "  (не найдено)\n";
    } else {
        foreach (array_slice($specialists, 0, 5) as $i => $spec) {
            $widgetData .= "  " . ($i+1) . ". " . ($spec['name'] ?? 'Без имени') . 
                          " - " . ($spec['description'] ?? 'Без специальности') . "\n";
        }
        if (count($specialists) > 5) {
            $widgetData .= "  ... и ещё " . (count($specialists) - 5) . "\n";
        }
    }
    
    $widgetData .= "\nУСЛУГИ (" . count($services) . "):\n";
    if (empty($services)) {
        $widgetData .= "  (не найдено)\n";
    } else {
        foreach (array_slice($services, 0, 5) as $i => $serv) {
            $widgetData .= "  " . ($i+1) . ". " . ($serv['name'] ?? 'Без названия') . "\n";
        }
        if (count($services) > 5) {
            $widgetData .= "  ... и ещё " . (count($services) - 5) . "\n";
        }
    }
    
    $widgetData .= "\nСПЕЦИАЛИЗАЦИИ (" . count($specializations) . "):\n";
    if (empty($specializations)) {
        $widgetData .= "  (не найдено)\n";
    } else {
        foreach ($specializations as $i => $spec) {
            $widgetData .= "  " . ($i+1) . ". " . ($spec['name'] ?? 'Без названия') . "\n";
        }
    }
    
    $prompt = "Ты - медицинский эксперт. Оцени качество ответа медицинского виджета на запрос пациента.

{$widgetData}

Оцени ответ по шкале от 0 до 40 баллов:
- Релевантность специалистов (0-10): насколько правильно подобраны врачи
- Релевантность услуг (0-10): насколько услуги соответствуют запросу
- Корректность специализаций (0-10): правильно ли определены разделы медицины
- Полнота ответа (0-10): достаточно ли информации для пациента

В конце дай общую оценку и вердикт:
- ОТЛИЧНО (32-40 баллов)
- ХОРОШО (24-31 балл)
- УДОВЛЕТВОРИТЕЛЬНО (16-23 балла)
- ПЛОХО (0-15 баллов)

Формат ответа:
ОЦЕНКА: X/40
РЕЛЕВАНТНОСТЬ СПЕЦИАЛИСТОВ: X/10 - (комментарий)
РЕЛЕВАНТНОСТЬ УСЛУГ: X/10 - (комментарий)
КОРРЕКТНОСТЬ СПЕЦИАЛИЗАЦИЙ: X/10 - (комментарий)
ПОЛНОТА ОТВЕТА: X/10 - (комментарий)

ВЕРДИКТ: (ОТЛИЧНО/ХОРОШО/УДОВЛЕТВОРИТЕЛЬНО/ПЛОХО)

РЕКОМЕНДАЦИИ: (если есть проблемы)";

    $data = [
        'model' => $model,
        'messages' => [
            ['role' => 'user', 'content' => $prompt]
        ],
        'temperature' => 0.3,
        'max_tokens' => 1000
    ];
    
    $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $openrouterKey,
        'Content-Type: application/json',
        'HTTP-Referer: https://wtest.medmaps.ru',
        'X-Title: AI Medical Widget Test'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        return "ОШИБКА: HTTP {$httpCode}\n{$response}";
    }
    
    $result = json_decode($response, true);
    return $result['choices'][0]['message']['content'] ?? 'Ошибка парсинга ответа';
}

// Основной процесс
echo "\n╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║   ТЕСТ С GEMINI НА 79 ПРОБЛЕМНЫХ ВОПРОСАХ (ПОСЛЕ УЛУЧШЕНИЙ)            ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

$results = [];
$totalScore = 0;
$maxScore = 0;
$verdicts = ['ОТЛИЧНО' => 0, 'ХОРОШО' => 0, 'УДОВЛЕТВОРИТЕЛЬНО' => 0, 'ПЛОХО' => 0];

file_put_contents($reportFile, "");
$startTime = microtime(true);

foreach ($questions as $index => $question) {
    $num = $index + 1;
    
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "ВОПРОС {$num}/79: {$question}\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    echo "  → Запрос к виджету...";
    $widgetResponse = callWidgetAPI($question, $widgetKey);
    
    $specCount = count($widgetResponse['specialists'] ?? []);
    $servCount = count($widgetResponse['services'] ?? []);
    $spezCount = count($widgetResponse['specializations'] ?? []);
    
    echo " OK ({$specCount} спец., {$servCount} усл., {$spezCount} спец.)\n";
    
    echo "  → Запрос к Gemini...";
    $expertReview = askExpert($question, $widgetResponse, $openrouterKey, $openrouterModel);
    echo " OK\n\n";
    
    preg_match('/ОЦЕНКА:\s*(\d+)\/40/i', $expertReview, $matches);
    $score = isset($matches[1]) ? (int)$matches[1] : 0;
    
    preg_match('/ВЕРДИКТ:\s*(.+)/i', $expertReview, $verdictMatches);
    $verdict = isset($verdictMatches[1]) ? trim($verdictMatches[1]) : 'N/A';
    
    if (isset($verdicts[$verdict])) {
        $verdicts[$verdict]++;
    }
    
    echo "  ⭐ ОЦЕНКА: {$score}/40 | ВЕРДИКТ: {$verdict}\n";
    
    $elapsed = round(microtime(true) - $startTime, 1);
    $avgTime = $elapsed / $num;
    $remaining = round((79 - $num) * $avgTime / 60, 1);
    echo "  📊 Прогресс: {$num}/79 ({$elapsed}с, ~{$remaining} мин. осталось)\n\n";
    
    $totalScore += $score;
    $maxScore += 40;
    
    // Сохраняем в файл
    $report = "\n" . str_repeat("=", 80) . "\n";
    $report .= "ВОПРОС {$num}/79: {$question}\n";
    $report .= str_repeat("=", 80) . "\n\n";
    $report .= "ОТВЕТ ВИДЖЕТА:\n";
    $report .= "Специалисты: {$specCount}, Услуги: {$servCount}, Специализации: {$spezCount}\n\n";
    $report .= "ЭКСПЕРТНАЯ ОЦЕНКА:\n";
    $report .= $expertReview . "\n\n";
    
    file_put_contents($reportFile, $report, FILE_APPEND);
}

$endTime = microtime(true);
$totalTime = round(($endTime - $startTime) / 60, 1);
$avgScore = $maxScore > 0 ? round($totalScore / 79, 1) : 0;

// Итоговая статистика
$summary = "\n\nИТОГОВАЯ СТАТИСТИКА (79 ПРОБЛЕМНЫХ ВОПРОСОВ - ПОСЛЕ УЛУЧШЕНИЙ)\n";
$summary .= str_repeat("━", 80) . "\n\n";
$summary .= "Средняя оценка: {$avgScore}/40\n";
$summary .= "Общий балл: {$totalScore}/{$maxScore}\n";
$summary .= "Время: {$totalTime} минут\n\n";
$summary .= "Распределение вердиктов:\n";
foreach ($verdicts as $v => $count) {
    $percent = round(($count / 79) * 100, 1);
    $summary .= "  • {$v}: {$count} ({$percent}%)\n";
}

echo $summary;
file_put_contents($reportFile, $summary, FILE_APPEND);

// Сохраняем сводку
file_put_contents($summaryFile, $summary);

echo "\n📁 Детальный отчет: {$reportFile}\n";
echo "📁 Сводка: {$summaryFile}\n\n";




