<?php
/**
 * Комплексный тест виджета на 50 запросах
 * - Текстовые ответы через query-stage1.php (GPU)
 * - Все категории через query-section.php (GPU)
 * - Оценка качества через AI эксперта (OpenRouter)
 * - Тестовый сервер: wtest.medmaps.ru
 */

//require_once '/root/questions-list.php';  // Закомментировано для теста fallback
//require_once '/root/problematic-questions.php';  // Закомментировано для теста fallback
require_once '/root/questions-list.php';  // Полный список вопросов (310 вопросов)
//require_once '/root/test-questions.php';  // Закомментировано для полного теста
//require_once '/root/test-fixes-questions.php';  // Тестовый список для проверки всех исправлений
require_once '/root/config.php';

// Используем полный список вопросов (переменная $questions уже определена в questions-list.php)
// $questions уже определена в questions-list.php, ничего не нужно присваивать

// Выбираем 30 случайных вопросов для быстрого теста
$totalQuestions = count($questions);
$randomKeys = array_rand($questions, min(100, $totalQuestions));
$randomQuestions = [];
foreach ($randomKeys as $key) {
    $randomQuestions[$key] = $questions[$key];
}
$questions = $randomQuestions;

$widgetKey = 'ekamedcenter';
$total = count($questions);

echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║  ПОЛНЫЙ ТЕСТ ВИДЖЕТА ({$total} ВОПРОСОВ)                                  ║\n";
echo "║  Сервер: wtest.medmaps.ru | GPU: Qwen2.5-7B | Эксперт: Gemini 2.5       ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

$results = [];
$stats = [
    'total' => $total,
    'excellent' => 0,  // 38-40
    'good' => 0,       // 30-37
    'poor' => 0,       // 20-29
    'bad' => 0,        // 0-19
    'total_score' => 0,
    'with_text' => 0,
    'without_text' => 0,
    'with_data' => 0,
    'without_data' => 0
];

$startTime = microtime(true);

// Определяем эмодзи для вердикта (используется в итоговой статистике)
$verdictEmoji = [
    'ОТЛИЧНО' => '⭐',
    'ХОРОШО' => '✅',
    'ПЛОХО' => '⚠️',
    'УЖАСНО' => '❌'
];

// Разбиваем вопросы на батчи по 10 для параллельной обработки
$batchSize = 30;
$questionBatches = array_chunk($questions, $batchSize, true); // Сохраняем ключи
$globalQuestionIndex = 0; // Глобальный счетчик вопросов

foreach ($questionBatches as $batchIndex => $batch) {
    $batchNum = $batchIndex + 1;
    $totalBatches = count($questionBatches);
    echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "БАТЧ {$batchNum}/{$totalBatches} (обработка " . count($batch) . " вопросов параллельно)\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    // Подготавливаем все запросы для параллельного выполнения
    $multiHandle = curl_multi_init();
    $handles = [];
    $questionMap = []; // Маппинг handle_id => question_index
    
    // ===== 1. ПОДГОТОВКА ПАРАЛЛЕЛЬНЫХ ЗАПРОСОВ ДЛЯ ТЕКСТА (Stage1) =====
    foreach ($batch as $index => $query) {
        $textData = [
            'widget_key' => $widgetKey,
            'question' => $query
        ];
        
        $ch = curl_init('https://wtest.medmaps.ru/api/query-stage1.php');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($textData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
        
        $handleId = (int)$ch;
        $handles[$handleId] = [
            'handle' => $ch,
            'type' => 'stage1',
            'index' => $index,
            'query' => $query
        ];
        $questionMap[$handleId] = $index;
        
        curl_multi_add_handle($multiHandle, $ch);
    }
    
    // Выполняем параллельные запросы
    $running = null;
    do {
        curl_multi_exec($multiHandle, $running);
        curl_multi_select($multiHandle, 0.1);
    } while ($running > 0);
    
    // Собираем результаты
    $textResults = [];
    foreach ($handles as $handleId => $handleData) {
        if ($handleData['type'] === 'stage1') {
            $response = curl_multi_getcontent($handleData['handle']);
            $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
            
            $text = '';
            $textProvider = 'unknown';
            
            if ($httpCode === 200 && $response) {
                $response = trim($response);
                $response = preg_replace('/^\xEF\xBB\xBF/', '', $response);
                $decoded = json_decode($response, true);
                
                if ($decoded && isset($decoded['text'])) {
                    $text = $decoded['text'];
                    $textProvider = $decoded['provider'] ?? 'unknown';
                }
            }
            
            $textResults[$handleData['index']] = [
                'text' => $text,
                'provider' => $textProvider,
                'query' => $handleData['query']
            ];
            
            curl_multi_remove_handle($multiHandle, $handleData['handle']);
            curl_close($handleData['handle']);
        }
    }
    
    // Очищаем multi handle для следующего этапа
    curl_multi_close($multiHandle);
    
    // ===== 2. ПОДГОТОВКА ПАРАЛЛЕЛЬНЫХ ЗАПРОСОВ ДЛЯ СЕКЦИЙ =====
    $sections = ['specialists', 'services', 'specializations', 'articles'];
    $multiHandle = curl_multi_init();
    $handles = [];
    
    foreach ($batch as $index => $query) {
        foreach ($sections as $section) {
            $postData = [
                'widget_key' => $widgetKey,
                'question' => $query,
                'section_name' => $section
            ];
            
            $ch = curl_init('https://wtest.medmaps.ru/api/query-section.php');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($postData));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'HTTP-Referer: https://wtest.medmaps.ru'
            ]);
            curl_setopt($ch, CURLOPT_TIMEOUT, 60);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            
            $handleId = (int)$ch;
            $handles[$handleId] = [
                'handle' => $ch,
                'type' => 'section',
                'index' => $index,
                'section' => $section,
                'query' => $query
            ];
            
            curl_multi_add_handle($multiHandle, $ch);
        }
    }
    
    // Выполняем параллельные запросы для секций
    $running = null;
    do {
        curl_multi_exec($multiHandle, $running);
        curl_multi_select($multiHandle, 0.1);
    } while ($running > 0);
    
    // Собираем результаты секций
    $sectionResults = [];
    foreach ($handles as $handleId => $handleData) {
        if ($handleData['type'] === 'section') {
            $response = curl_multi_getcontent($handleData['handle']);
            $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
            
            $sectionData = [];
            if ($httpCode === 200 && $response) {
                $decoded = json_decode($response, true);
                $dataField = $decoded['data'] ?? [];
                
                if (isset($dataField[$handleData['section']])) {
                    $sectionData = $dataField[$handleData['section']];
                } elseif (is_array($dataField) && !isset($dataField[0])) {
                    $sectionData = $dataField;
                } else {
                    $sectionData = $dataField;
                }
            }
            
            if (!isset($sectionResults[$handleData['index']])) {
                $sectionResults[$handleData['index']] = [];
            }
            $sectionResults[$handleData['index']][$handleData['section']] = $sectionData;
            
            curl_multi_remove_handle($multiHandle, $handleData['handle']);
            curl_close($handleData['handle']);
        }
    }
    
    curl_multi_close($multiHandle);
    
    // ===== 3. ПОДГОТОВКА ДАННЫХ ДЛЯ AI-ЭКСПЕРТА (проверка артефактов для всех вопросов) =====
    $expertData = [];
    foreach ($batch as $index => $query) {
        $textResult = $textResults[$index] ?? ['text' => '', 'provider' => 'unknown', 'query' => $query];
        $text = $textResult['text'];
        $sectionData = $sectionResults[$index] ?? [];
        
        $specialistsCount = count($sectionData['specialists'] ?? []);
        $servicesCount = count($sectionData['services'] ?? []);
        $specializationsCount = count($sectionData['specializations'] ?? []);
        $articlesCount = count($sectionData['articles'] ?? []);
        
        $hasText = !empty($text);
        
        // Проверка артефактов
        $textArtifacts = [];
        if ($hasText) {
            if (preg_match('/\.(runtuit|jpg|jpeg|png|gif|webp|svg)[_\w\d]*/i', $text)) {
                $textArtifacts[] = 'Ссылки на изображения в тексте';
            }
            if (preg_match('/\b(other\s+symptoms?|other\s+conditions?|Apply\s+ice|appropriate\s+лечение|appropriate\s+treatment)\b/ui', $text)) {
                $textArtifacts[] = 'Смешанные русско-английские фразы';
            }
            if (preg_match('/\b(endocrinolog|gastroenterolog|pulmonologist|respirologist|orthopedic|ophthalmologist|dermatologist|allergologist|immunologist|cardiologist|neurologist|rheumatologist|urologist|gynecologist|surgeon|therapist|pediatrician)[уеаио]\b/ui', $text)) {
                $textArtifacts[] = 'Транслитерация медицинских терминов';
            }
            if (preg_match('/\.\s*([A-Z][a-z]+(?:\s+[a-z]+)*\s+(?:can|should|is|are|may|might|will|would|could)\s+[^.]*\.)/u', $text)) {
                $textArtifacts[] = 'Английские предложения в русском тексте';
            }
            if (preg_match('/[\x{4e00}-\x{9fff}]/u', $text)) {
                $textArtifacts[] = 'Китайские символы в тексте';
            }
            if (preg_match('/[^.!?]\s*$/', trim($text)) && mb_strlen($text) < 50) {
                $textArtifacts[] = 'Незавершенный текст';
            }
            if (preg_match('/<[^>]+>/', $text)) {
                $textArtifacts[] = 'HTML теги в тексте';
            }
            if (preg_match('/(http:\/\/|https:\/\/|www\.)/i', $text)) {
                $textArtifacts[] = 'URL в тексте';
            }
        }
        
        $artifactsInfo = !empty($textArtifacts) ? "\n\n⚠️ ОБНАРУЖЕНЫ АРТЕФАКТЫ В ТЕКСТЕ:\n" . implode("\n", array_map(function($a) { return "- " . $a; }, $textArtifacts)) : "";
        
        $expertPrompt = "Ты - строгий эксперт по медицинским виджетам. Оцени качество ответа виджета медицинской клиники.

ВОПРОС ПОЛЬЗОВАТЕЛЯ:
\"{$query}\"

ОТВЕТ ВИДЖЕТА:

ТЕКСТ:
" . ($hasText ? $text : "(текст отсутствует)") . $artifactsInfo . "

СПЕЦИАЛИСТЫ ({$specialistsCount}):
" . ($specialistsCount > 0 ? json_encode($sectionData['specialists'], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) : "(нет данных)") . "

УСЛУГИ ({$servicesCount}):
" . ($servicesCount > 0 ? json_encode($sectionData['services'], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) : "(нет данных)") . "

СПЕЦИАЛИЗАЦИИ ({$specializationsCount}):
" . ($specializationsCount > 0 ? json_encode($sectionData['specializations'], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) : "(нет данных)") . "

СТАТЬИ ({$articlesCount}):
" . ($articlesCount > 0 ? json_encode($sectionData['articles'], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) : "(нет данных)") . "

КРИТЕРИИ ОЦЕНКИ (всего 40 баллов):

⚠️ ВАЖНО: Каждый компонент оценивается НЕЗАВИСИМО от других!

1. ТЕКСТ (10 баллов) - ОЦЕНИВАЙ ТОЛЬКО КАЧЕСТВО ТЕКСТА:
   text_score должен быть таким же, как если бы ты оценивал ТОЛЬКО текст, игнорируя специалистов и услуги!
   
   Критерии оценки текста (ТОЛЬКО эти факторы влияют на text_score):
   - 10: отличный информативный текст на русском, релевантный вопросу, БЕЗ АРТЕФАКТОВ, предложения завершены
   - 7-9: хороший текст, небольшие недочеты или незначительные артефакты
   - 4-6: текст есть, но слабый, частично нерелевантный или содержит артефакты
   - 0-3: текст отсутствует, полностью нерелевантен или содержит критичные артефакты
   
   ОБЯЗАТЕЛЬНО ПРОВЕРЬ ТЕКСТ НА:
   - Артефакты (ссылки на изображения .jpg/.png/.runtuit, смешанные русско-английские фразы, транслитерация, HTML теги, URL)
   - Адекватность (текст должен быть на русском языке, без английских предложений, логичен)
   - Завершенность (текст не должен обрываться на середине предложения, должен заканчиваться точкой/восклицательным/вопросительным знаком)
   - Релевантность (текст должен отвечать на вопрос пользователя)
   
   ❌ НЕ снижай text_score, если:
   - Специалисты нерелевантны или отсутствуют
   - Услуги нерелевантны или отсутствуют
   - Другие компоненты плохие
   
   ✅ Снижай text_score ТОЛЬКО если:
   - В тексте есть артефакты
   - Текст не завершен (обрывается)
   - Текст неадекватен (английский, бессмысленный)
   - Текст нерелевантен вопросу
   
   ЕСЛИ ОБНАРУЖЕНЫ АРТЕФАКТЫ - обязательно укажи их в text_artifacts и issues, и снизь text_score!

2. СПЕЦИАЛИСТЫ (10 баллов):
   - 10: идеально релевантные специалисты
   - 7-9: релевантные, возможно не все нужные
   - 4-6: частично релевантные
   - 0-3: нерелевантные или отсутствуют

3. УСЛУГИ (10 баллов):
   - 10: отличный набор услуг (консультации + диагностика)
   - 7-9: хорошие услуги, возможны небольшие пробелы
   - 4-6: услуги есть, но недостаточно релевантные
   - 0-3: нерелевантные или отсутствуют

4. ПОЛНОТА И КАЧЕСТВО (10 баллов):
   - 10: комплексный ответ, помогает пользователю
   - 7-9: хороший ответ, небольшие недочеты
   - 4-6: базовая информация есть
   - 0-3: ответ неполный или бесполезен

ОСОБЫЕ СЛУЧАИ:
- Если это ЭКСТРЕННАЯ СИТУАЦИЯ (инфаркт, инсульт, сильное кровотечение и т.п.) и виджет правильно указал на срочность - ставь 38-40 баллов ДАЖЕ ЕСЛИ нет специалистов/услуг
- Если вопрос административный (справки, расписание) и нет конкретных специалистов - это нормально, оценивай текст

ФОРМАТ ОТВЕТА (строго JSON):
{
  \"score\": <число 0-40>,
  \"verdict\": \"ОТЛИЧНО\" | \"ХОРОШО\" | \"ПЛОХО\" | \"УЖАСНО\",
  \"text_score\": <0-10>,
  \"specialists_score\": <0-10>,
  \"services_score\": <0-10>,
  \"completeness_score\": <0-10>,
  \"comment\": \"<краткий комментарий>\",
  \"issues\": [\"<список проблем, если есть>\", \"<обязательно укажи артефакты, если обнаружены>\"],
  \"text_artifacts\": [\"<список найденных артефактов в тексте, если есть>\"],
  \"text_adequacy\": \"<адекватен ли текст: ДА/НЕТ/ЧАСТИЧНО, с пояснением>\"
}

ВЕРДИКТЫ:
- ОТЛИЧНО: 38-40 баллов
- ХОРОШО: 30-37 баллов  
- ПЛОХО: 20-29 баллов
- УЖАСНО: 0-19 баллов

Отвечай ТОЛЬКО валидным JSON, без дополнительного текста!";
        
        $expertData[$index] = [
            'query' => $query,
            'text' => $text,
            'textProvider' => $textResult['provider'],
            'textArtifacts' => $textArtifacts,
            'sectionData' => $sectionData,
            'specialistsCount' => $specialistsCount,
            'servicesCount' => $servicesCount,
            'specializationsCount' => $specializationsCount,
            'articlesCount' => $articlesCount,
            'prompt' => $expertPrompt
        ];
    }
    
    // ===== 4. ПАРАЛЛЕЛЬНЫЕ ЗАПРОСЫ К AI-ЭКСПЕРТУ =====
    echo "  → Отправка запросов к AI-эксперту (параллельно)...\n";
    
    $multiHandle = curl_multi_init();
    $expertHandles = [];
    
    foreach ($expertData as $index => $data) {
        $expertApiData = [
            'model' => 'xiaomi/mimo-v2-flash:free',
            'messages' => [
                ['role' => 'user', 'content' => $data['prompt']]
            ]
        ];
        
        $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($expertApiData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . OPENROUTER_API_KEY,
            'HTTP-Referer: https://wtest.medmaps.ru',
            'X-Title: Medical Widget Tester'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        
        $handleId = (int)$ch;
        $expertHandles[$handleId] = [
            'handle' => $ch,
            'index' => $index
        ];
        
        curl_multi_add_handle($multiHandle, $ch);
    }
    
    // Выполняем параллельные запросы к эксперту
    $running = null;
    do {
        curl_multi_exec($multiHandle, $running);
        curl_multi_select($multiHandle, 0.1);
    } while ($running > 0);
    
    // Собираем результаты эксперта
    $expertResults = [];
    $failedIndices = [];
    
    foreach ($expertHandles as $handleId => $handleData) {
        $response = curl_multi_getcontent($handleData['handle']);
        $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
        
        $expertResult = null;
        if ($httpCode === 200 && $response) {
            $expertDecoded = json_decode($response, true);
            $expertContent = $expertDecoded['choices'][0]['message']['content'] ?? '';
            
            if (preg_match('/\{[^{}]*(?:\{[^{}]*\}[^{}]*)*\}/s', $expertContent, $matches)) {
                $expertResult = json_decode($matches[0], true);
            }
        }
        
        if (!$expertResult || !isset($expertResult['score'])) {
            // Сохраняем индекс для повторной попытки
            $failedIndices[] = $handleData['index'];
            $expertResult = [
                'score' => 0,
                'verdict' => 'УЖАСНО',
                'text_score' => 0,
                'specialists_score' => 0,
                'services_score' => 0,
                'completeness_score' => 0,
                'comment' => 'Ошибка оценки AI',
                'issues' => ['AI эксперт не ответил']
            ];
        }
        
        $expertResults[$handleData['index']] = $expertResult;
        
        curl_multi_remove_handle($multiHandle, $handleData['handle']);
        curl_close($handleData['handle']);
    }
    
    // Повторные попытки для неудачных запросов (до 3 раз)
    if (!empty($failedIndices)) {
        $maxRetries = 3;
        $retryCount = 0;
        
        while (!empty($failedIndices) && $retryCount < $maxRetries) {
            $retryCount++;
            echo "  → Повторная попытка оценки AI (попытка {$retryCount}/{$maxRetries}) для " . count($failedIndices) . " вопросов...\n";
            
            $retryMultiHandle = curl_multi_init();
            $retryHandles = [];
            
            foreach ($failedIndices as $index) {
                $data = $expertData[$index];
                $expertApiData = [
                    'model' => 'xiaomi/mimo-v2-flash:free',
                    'messages' => [
                        ['role' => 'user', 'content' => $data['prompt']]
                    ]
                ];
                
                $ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($expertApiData));
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Content-Type: application/json',
                    'Authorization: Bearer ' . OPENROUTER_API_KEY,
                    'HTTP-Referer: https://wtest.medmaps.ru',
                    'X-Title: Medical Widget Tester'
                ]);
                curl_setopt($ch, CURLOPT_TIMEOUT, 60);
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
                
                $retryHandles[(int)$ch] = [
                    'handle' => $ch,
                    'index' => $index
                ];
                
                curl_multi_add_handle($retryMultiHandle, $ch);
            }
            
            // Выполняем повторные запросы
            $running = null;
            do {
                curl_multi_exec($retryMultiHandle, $running);
                curl_multi_select($retryMultiHandle, 0.1);
            } while ($running > 0);
            
            // Обрабатываем результаты повторных попыток
            $stillFailed = [];
            foreach ($retryHandles as $handleId => $handleData) {
                $response = curl_multi_getcontent($handleData['handle']);
                $httpCode = curl_getinfo($handleData['handle'], CURLINFO_HTTP_CODE);
                
                $expertResult = null;
                if ($httpCode === 200 && $response) {
                    $expertDecoded = json_decode($response, true);
                    $expertContent = $expertDecoded['choices'][0]['message']['content'] ?? '';
                    
                    if (preg_match('/\{[^{}]*(?:\{[^{}]*\}[^{}]*)*\}/s', $expertContent, $matches)) {
                        $expertResult = json_decode($matches[0], true);
                    }
                }
                
                if ($expertResult && isset($expertResult['score'])) {
                    // Успешно получили результат
                    $expertResults[$handleData['index']] = $expertResult;
                } else {
                    // Все еще ошибка
                    $stillFailed[] = $handleData['index'];
                }
                
                curl_multi_remove_handle($retryMultiHandle, $handleData['handle']);
                curl_close($handleData['handle']);
            }
            
            curl_multi_close($retryMultiHandle);
            $failedIndices = $stillFailed;
            
            // Небольшая задержка перед следующей попыткой
            if (!empty($failedIndices) && $retryCount < $maxRetries) {
                usleep(500000); // 0.5 секунды
            }
        }
    }
    
    curl_multi_close($multiHandle);
    
    // ===== 5. ВЫВОД РЕЗУЛЬТАТОВ БАТЧА =====
    foreach ($batch as $index => $query) {
        $globalQuestionIndex++;
        $num = $globalQuestionIndex;
        
        echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
        echo "ВОПРОС {$num}/{$total}: {$query}\n";
        echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
        
        $data = $expertData[$index];
        $text = $data['text'];
        $textProvider = $data['textProvider'];
        $textArtifacts = $data['textArtifacts'];
        $sectionData = $data['sectionData'];
        $specialistsCount = $data['specialistsCount'];
        $servicesCount = $data['servicesCount'];
        $specializationsCount = $data['specializationsCount'];
        $articlesCount = $data['articlesCount'];
        
        $hasText = !empty($text);
        if ($hasText) {
            $stats['with_text']++;
        } else {
            $stats['without_text']++;
        }
        
        echo "  → Текст получен: " . ($hasText ? "да ({$textProvider}, " . mb_strlen($text) . " символов)" : "НЕТ") . "\n\n";
        
        foreach ($sections as $section) {
            $count = count($sectionData[$section] ?? []);
            echo "  ✓ {$section}: {$count} элементов\n";
        }
        
        $hasData = ($specialistsCount > 0 || $servicesCount > 0);
        if ($hasData) {
            $stats['with_data']++;
        } else {
            $stats['without_data']++;
        }
        
        echo "\n  📊 Итого: {$specialistsCount} спец., {$servicesCount} усл., {$specializationsCount} специализаций, {$articlesCount} статей\n\n";
        
        // Выводим информацию об артефактах, если они обнаружены
        if (!empty($textArtifacts)) {
            echo "  ⚠️ Обнаружены артефакты в тексте: " . implode(", ", $textArtifacts) . "\n";
        }
        
        // Результат эксперта
        $expertResult = $expertResults[$index] ?? [
            'score' => 0,
            'verdict' => 'УЖАСНО',
            'comment' => 'Ошибка получения оценки',
            'issues' => []
        ];
        
        $score = $expertResult['score'];
        $verdict = $expertResult['verdict'];
        
        // Обновляем статистику
        $stats['total_score'] += $score;
        if ($score >= 38) {
            $stats['excellent']++;
        } elseif ($score >= 30) {
            $stats['good']++;
        } elseif ($score >= 20) {
            $stats['poor']++;
        } else {
            $stats['bad']++;
        }
        
        // Определяем эмодзи для вердикта
        $emoji = $verdictEmoji[$verdict] ?? '❓';
        
        echo "  → Оценка AI-экспертом получена\n\n";
        echo "  {$emoji} ОЦЕНКА: {$score}/40 | ВЕРДИКТ: {$verdict}\n";
        echo "  💬 Комментарий: {$expertResult['comment']}\n";
        
        // Выводим информацию об артефактах, если они обнаружены
        if (!empty($textArtifacts)) {
            echo "  ⚠️ Артефакты (авто): " . implode(", ", $textArtifacts) . "\n";
        }
        
        // Выводим информацию об артефактах из оценки эксперта
        if (isset($expertResult['text_artifacts']) && !empty($expertResult['text_artifacts']) && is_array($expertResult['text_artifacts'])) {
            echo "  🔍 Артефакты (эксперт): " . implode(", ", $expertResult['text_artifacts']) . "\n";
        }
        if (isset($expertResult['text_adequacy']) && !empty($expertResult['text_adequacy'])) {
            echo "  ✅ Адекватность текста: {$expertResult['text_adequacy']}\n";
        }
        echo "\n";
        
        // Прогресс
        $elapsed = round(microtime(true) - $startTime, 1);
        $avgTime = $elapsed / $num;
        $remaining = round(($total - $num) * $avgTime / 60, 1);
        echo "  📊 Прогресс: {$num}/{$total} ({$elapsed} сек) | ~{$remaining} мин. осталось\n\n";
        
        // Сохраняем результат
        $results[] = [
            'num' => $num,
            'query' => $query,
            'text' => $text,
            'text_length' => mb_strlen($text),
            'text_provider' => $textProvider,
            'text_artifacts_detected' => $textArtifacts,
            'specialists_count' => $specialistsCount,
            'services_count' => $servicesCount,
            'specializations_count' => $specializationsCount,
            'articles_count' => $articlesCount,
            'specialists' => $sectionData['specialists'] ?? [],
            'services' => $sectionData['services'] ?? [],
            'specializations' => $sectionData['specializations'] ?? [],
            'articles' => $sectionData['articles'] ?? [],
            'score' => $score,
            'verdict' => $verdict,
            'expert_details' => $expertResult
        ];
        
        // Небольшая пауза между запросами
        usleep(500000); // 0.5 секунды
    }
}

$endTime = microtime(true);
$totalTime = round($endTime - $startTime, 1);
$avgScore = round($stats['total_score'] / $total, 1);

// ===== ИТОГОВАЯ СТАТИСТИКА =====
echo "\n";
echo "╔══════════════════════════════════════════════════════════════════════════╗\n";
echo "║                        ИТОГОВАЯ СТАТИСТИКА                               ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════╝\n\n";

echo "Всего вопросов: {$total}\n";
echo "Время выполнения: {$totalTime} сек (" . round($totalTime / 60, 1) . " мин)\n";
echo "Средняя скорость: " . round($total / ($totalTime / 60), 1) . " вопросов/мин\n\n";

echo "ОЦЕНКИ AI-ЭКСПЕРТА:\n";
echo "  ⭐ ОТЛИЧНО (38-40): {$stats['excellent']} (" . round($stats['excellent'] / $total * 100, 1) . "%)\n";
echo "  ✅ ХОРОШО (30-37): {$stats['good']} (" . round($stats['good'] / $total * 100, 1) . "%)\n";
echo "  ⚠️ ПЛОХО (20-29): {$stats['poor']} (" . round($stats['poor'] / $total * 100, 1) . "%)\n";
echo "  ❌ УЖАСНО (0-19): {$stats['bad']} (" . round($stats['bad'] / $total * 100, 1) . "%)\n";
echo "  📊 Средний балл: {$avgScore}/40\n\n";

echo "ДАННЫЕ:\n";
echo "  С текстом: {$stats['with_text']} (" . round($stats['with_text'] / $total * 100, 1) . "%)\n";
echo "  Без текста: {$stats['without_text']} (" . round($stats['without_text'] / $total * 100, 1) . "%)\n";
echo "  С данными (спец/усл): {$stats['with_data']} (" . round($stats['with_data'] / $total * 100, 1) . "%)\n";
echo "  Без данных: {$stats['without_data']} (" . round($stats['without_data'] / $total * 100, 1) . "%)\n\n";

// Топ-10 лучших
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "ТОП-10 ЛУЧШИХ РЕЗУЛЬТАТОВ:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

usort($results, function($a, $b) { return $b['score'] - $a['score']; });
$top10 = array_slice($results, 0, 10);

foreach ($top10 as $result) {
    $emoji = $verdictEmoji[$result['verdict']] ?? '❓';
    echo "  {$emoji} {$result['score']}/40 - {$result['query']}\n";
    echo "     {$result['expert_details']['comment']}\n\n";
}

// Топ-10 худших
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "ТОП-10 ПРОБЛЕМНЫХ ВОПРОСОВ:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$bottom10 = array_slice(array_reverse($results), 0, 10);

foreach ($bottom10 as $result) {
    $emoji = $verdictEmoji[$result['verdict']] ?? '❓';
    echo "  {$emoji} {$result['score']}/40 - {$result['query']}\n";
    echo "     Проблемы: " . implode(', ', $result['expert_details']['issues'] ?? []) . "\n\n";
}

// Сохраняем результаты
$timestamp = date('Y-m-d_H-i-s');
$jsonFile = "/root/test-50-full-results-{$timestamp}.json";
$reportFile = "/root/test-50-full-report-{$timestamp}.txt";

file_put_contents($jsonFile, json_encode($results, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

$report = "ПОЛНЫЙ ТЕСТ ВИДЖЕТА С AI-ЭКСПЕРТОМ\n";
$report .= "Дата: " . date('Y-m-d H:i:s') . "\n";
$report .= "Сервер: wtest.medmaps.ru\n";
$report .= "GPU: Qwen2.5-7B-Instruct\n";
$report .= "Эксперт: Gemini 2.5 Flash\n\n";
$report .= "Всего вопросов: {$total}\n";
$report .= "Время: {$totalTime} сек\n";
$report .= "Средний балл: {$avgScore}/40\n\n";
$report .= "ОТЛИЧНО: {$stats['excellent']}\n";
$report .= "ХОРОШО: {$stats['good']}\n";
$report .= "ПЛОХО: {$stats['poor']}\n";
$report .= "УЖАСНО: {$stats['bad']}\n\n";
$report .= "С текстом: {$stats['with_text']}\n";
$report .= "С данными: {$stats['with_data']}\n";

file_put_contents($reportFile, $report);

echo "\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "Результаты сохранены:\n";
echo "  JSON: {$jsonFile}\n";
echo "  Отчет: {$reportFile}\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";



// Генерация HTML отчета
echo "\n📊 Генерация HTML отчета...\n";
$htmlFile = '/root/test-report.html';

// Вызываем генератор с правильными параметрами
$generatorCode = file_get_contents('/root/generate-html-report.php');
// Заменяем пути к файлам на переменные
$generatorCode = preg_replace(
    "/\\\$jsonFile = '[^']+';/",
    "\$jsonFile = '$jsonFile';",
    $generatorCode
);
$generatorCode = preg_replace(
    "/\\\$htmlFile = '[^']+';/",
    "\$htmlFile = '$htmlFile';",
    $generatorCode
);

// Выполняем код генератора
eval('?>' . $generatorCode);

if (file_exists($htmlFile)) {
    echo "  ✅ HTML отчет создан: $htmlFile\n";
    echo "  🌐 Доступен по ссылке: https://wtest.medmaps.ru/test-report.html\n";
} else {
    echo "  ⚠️ Не удалось создать HTML отчет\n";
}
