<?php
/**
 * Модуль валидации текстовых ответов виджета
 * Проверяет качество текста и логирует проблемные случаи
 */

require_once __DIR__ . '/../config.php';

/**
 * Комплексная валидация текста с детальным отчётом
 * Возвращает массив с результатами проверки
 */
function validateTextQuality($text, $question = '') {
    $result = [
        'is_valid' => true,
        'score' => 10, // Оценка качества от 0 до 10
        'issues' => [],
        'artifacts' => [],
        'completeness' => true,
        'adequacy' => true,
        'length_ok' => true
    ];
    
    $trimmedText = trim($text);
    $textLength = mb_strlen($trimmedText);
    
    // 1. Проверка длины текста
    if ($textLength < 50) {
        $result['is_valid'] = false;
        $result['score'] -= 5;
        $result['length_ok'] = false;
        $result['issues'][] = "Текст слишком короткий ({$textLength} символов)";
    } elseif ($textLength < 100) {
        $result['score'] -= 2;
        $result['length_ok'] = false;
        $result['issues'][] = "Текст недостаточно информативен ({$textLength} символов)";
    }
    
    // 2. Проверка завершенности предложений
    if (!preg_match('/[.!?]\s*$/u', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 4;
        $result['completeness'] = false;
        $result['issues'][] = "Текст не завершен (нет точки/восклицательного/вопросительного знака в конце)";
        $result['artifacts'][] = "Незавершенное предложение";
    }
    
    // Проверка на обрыв в середине слова или на незавершенном прилагательном/существительном
    $lastWords = mb_substr($trimmedText, -50); // Берем последние 50 символов
    if (preg_match('/\s+(красные?|белые?|желтые?|черные?|синие?|зеленые?|большие?|маленькие?|новые?|старые?|прочие?)\s*[.,]\s*$/ui', $lastWords)) {
        $result['is_valid'] = false;
        $result['score'] -= 6;
        $result['completeness'] = false;
        $result['issues'][] = "Текст обрывается на незавершенном прилагательном (например: 'красные.')";
        $result['artifacts'][] = "Обрыв на незавершенной фразе";
    }
    
    // Проверка на обрыв в середине перечисления
    if (preg_match('/,\s*и\s*[а-яё]{1,15}\s*[.!?]\s*$/ui', $lastWords) || preg_match('/,\s*[а-яё]{1,10}\s*[.!?]\s*$/ui', $lastWords)) {
        $lastWord = preg_replace('/.*\s+([а-яё]+)\s*[.!?]\s*$/ui', '$1', $lastWords);
        // Проверяем, не является ли последнее слово завершающим (существительное, специалист)
        if (!preg_match('/(врач[уаеоюи]?|специалист[уаеоюи]?|терапевт[уаеоюи]?|невролог[уаеоюи]?|диагностик[иуое]?|лечени[яюе]?|обследовани[яюе]?)/ui', $lastWord)) {
            $result['score'] -= 2;
            $result['completeness'] = false;
            $result['issues'][] = "Возможный обрыв в перечислении";
        }
    }
    
    // Проверка на обрыв в середине предложения
    if (preg_match('/\s+(или|и|а также|таких как|например|с|к|для|в|на)\s*[.!?]?\s*$/ui', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 4;
        $result['completeness'] = false;
        $result['issues'][] = "Текст обрывается на союзе или предлоге";
        $result['artifacts'][] = "Обрыв на середине предложения";
    }
    
    // ===== КРИТИЧНЫЕ ПРОВЕРКИ НА ОБРЫВЫ (самые частые проблемы!) =====
    
    // Обрыв типа "таких как Рекомендуется" или "таких как Для"
    if (preg_match('/таких\s+как\s+(Рекомендуется|Для|При)/ui', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 8;
        $result['completeness'] = false;
        $result['issues'][] = "КРИТИЧНЫЙ ОБРЫВ: текст обрывается после 'таких как' без указания причин";
        $result['artifacts'][] = "Обрыв перечисления";
    }
    
    // Обрыв типа "воспаление Рекомендуется" или "воспалением Рекомендуется"
    if (preg_match('/(воспалени[ием]+)\s+(Рекомендуется|Для|При)/ui', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 8;
        $result['completeness'] = false;
        $result['issues'][] = "КРИТИЧНЫЙ ОБРЫВ: текст обрывается после 'воспаление' без указания чего";
        $result['artifacts'][] = "Незавершенное определение";
    }
    
    // Обрыв типа "метод непрерывного Рекомендуется"
    if (preg_match('/метод\s+непрерывного\s+(Рекомендуется|Для)/ui', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 8;
        $result['completeness'] = false;
        $result['issues'][] = "КРИТИЧНЫЙ ОБРЫВ: текст обрывается в середине определения метода";
        $result['artifacts'][] = "Неполное определение";
    }
    
    // Обрыв типа "характеризуется Рекомендуется"
    if (preg_match('/характеризуется\s+(Рекомендуется|Для)/ui', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 8;
        $result['completeness'] = false;
        $result['issues'][] = "КРИТИЧНЫЙ ОБРЫВ: текст обрывается после 'характеризуется' без описания";
        $result['artifacts'][] = "Незавершенное описание";
    }
    
    // Обрыв типа "боли внизу живота, Рекомендуется" (запятая перед Рекомендуется)
    if (preg_match('/,\s+(Рекомендуется|Для диагностики)/ui', $trimmedText) && 
        !preg_match('/врач[уаеоюи]?|специалист[уаеоюи]?|терапевт[уаеоюи]?|кардиолог[уаеоюи]?|невролог[уаеоюи]?|гастроэнтеролог[уаеоюи]?/ui', mb_substr($trimmedText, -80))) {
        $result['is_valid'] = false;
        $result['score'] -= 7;
        $result['completeness'] = false;
        $result['issues'][] = "КРИТИЧНЫЙ ОБРЫВ: незавершенное перечисление перед рекомендацией";
        $result['artifacts'][] = "Обрыв перечисления симптомов";
    }
    
    // Обрыв типа "чихание и Рекомендуется" или "боли и Для диагностики"
    if (preg_match('/\s+и\s+(Рекомендуется|Для диагностики|При подозрении)/ui', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 8;
        $result['completeness'] = false;
        $result['issues'][] = "КРИТИЧНЫЙ ОБРЫВ: текст обрывается после союза 'и' без завершения перечисления";
        $result['artifacts'][] = "Обрыв в середине перечисления";
    }
    
    // Грамматическая ошибка: "таких или даже" вместо "таких как ... или даже"
    if (preg_match('/таких\s+или\s+даже/ui', $trimmedText)) {
        $result['score'] -= 6;
        $result['adequacy'] = false;
        $result['issues'][] = "Грамматическая ошибка: 'таких или даже' (должно быть 'таких как ... или даже')";
        $result['artifacts'][] = "Грамматическая ошибка";
    }
    
    // Обрыв типа "обратиться к или" (пропущено слово)
    if (preg_match('/обратиться\s+к\s+(или|и)\s+/ui', $trimmedText)) {
        $result['score'] -= 7;
        $result['completeness'] = false;
        $result['issues'][] = "Пропущено слово после 'обратиться к'";
        $result['artifacts'][] = "Неполная рекомендация";
    }
    
    // Проверка на "легкодышечный врач" и другие странные термины
    if (preg_match('/(легкодышечный\s+врач|врач\s+по\s+легким|дыхательный\s+врач)/ui', $trimmedText)) {
        $result['score'] -= 5;
        $result['adequacy'] = false;
        $result['issues'][] = "Некорректный термин: должно быть 'пульмонолог'";
        $result['artifacts'][] = "Неправильная терминология";
    }
    
    // Проверка на общие термины "к врачу" / "к доктору" вместо конкретных специалистов
    if (preg_match('/обратиться\s+к\s+(врачу|доктору)\b/ui', $trimmedText)) {
        // Проверяем, есть ли хоть где-то конкретный специалист в тексте
        $hasSpecificSpecialist = preg_match('/(терапевт|кардиолог|невролог|гастроэнтеролог|пульмонолог|эндокринолог|уролог|гинеколог|офтальмолог|дерматолог|ортопед|отоларинголог|педиатр|аллерголог|иммунолог|хирург|онколог|ревматолог|психотерапевт)/ui', $trimmedText);
        
        if (!$hasSpecificSpecialist) {
            $result['score'] -= 3;
            $result['adequacy'] = false;
            $result['issues'][] = "Общий термин 'врач' вместо конкретного специалиста";
            $result['artifacts'][] = "Неконкретная рекомендация";
        }
    }
    
    // Проверка на незавершенные фразы
    if (preg_match('/\s+(может быть вызван[а-яё]*|связан[а-яё]* с|обратиться к)\s*[.!?]?\s*$/ui', $trimmedText)) {
        $result['score'] -= 3;
        $result['completeness'] = false;
        $result['issues'][] = "Незавершенная фраза в конце текста";
        $result['artifacts'][] = "Незавершенная рекомендация";
    }
    
    // 3. Проверка на английские вставки
    $englishMatches = [];
    
    // Английские слова
    if (preg_match_all('/\b(hands|pediatrician|cardiologist|dermatologist|endocrinologist|neurologist|surgeon|therapist|procedure|procedures|treatment|therapy|diagnosis|symptoms?|conditions?|appropriate|apply|ice|runny|nose|cough|fever|measles|other)\b/i', $trimmedText, $englishMatches)) {
        $result['is_valid'] = false;
        $result['score'] -= 5;
        $result['adequacy'] = false;
        $result['issues'][] = "Английские слова в русском тексте: " . implode(', ', array_unique($englishMatches[0]));
        $result['artifacts'][] = "Английские слова: " . implode(', ', array_unique($englishMatches[0]));
    }
    
    // Английские предложения
    if (preg_match('/\.\s*([A-Z][a-z]+(?:\s+[a-z]+)*\s+(?:can|should|is|are|may|might|will|would|could)\s+[^.]*\.)/u', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 6;
        $result['adequacy'] = false;
        $result['issues'][] = "Английские предложения в русском тексте";
        $result['artifacts'][] = "Английские предложения";
    }
    
    // Смешанные русско-английские фразы
    if (preg_match('/\b(appropriate\s+лечение|Apply\s+ice|other\s+symptoms?|such\s+as)\b/ui', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 5;
        $result['adequacy'] = false;
        $result['issues'][] = "Смешанные русско-английские фразы";
        $result['artifacts'][] = "Русско-английские фразы";
    }
    
    // 4. Проверка на транслитерацию медицинских терминов
    if (preg_match('/\b(endocrinolog|gastroenterolog|pulmonologist|respirologist|orthopedic|ophthalmologist|dermatologist|allergologist|immunologist|cardiologist|neurologist|rheumatologist|urologist|gynecologist|pediatrician)[уеаио]\b/ui', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 6;
        $result['adequacy'] = false;
        $result['issues'][] = "Транслитерация медицинских терминов вместо русских названий";
        $result['artifacts'][] = "Транслитерация терминов";
    }
    
    // 5. Проверка на артефакты аббревиатур
    if (preg_match('/\b(ЧОБЛ|COPD|AIDS|HIV)\b/ui', $trimmedText)) {
        $result['score'] -= 3;
        $result['issues'][] = "Некорректные медицинские аббревиатуры (ЧОБЛ вместо ХОБЛ)";
        $result['artifacts'][] = "Некорректные аббревиатуры";
    }
    
    // 6. Проверка на китайские символы
    if (preg_match('/[\x{4e00}-\x{9fff}]/u', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 7;
        $result['adequacy'] = false;
        $result['issues'][] = "Китайские символы в тексте";
        $result['artifacts'][] = "Китайские символы";
    }
    
    // 7. Проверка на ссылки на изображения
    if (preg_match('/\.(runtuit|jpg|jpeg|png|gif|webp|svg)[_\w\d]*/i', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 4;
        $result['issues'][] = "Ссылки на изображения в тексте";
        $result['artifacts'][] = "Ссылки на изображения";
    }
    
    // 8. Проверка на HTML теги и URL
    if (preg_match('/<[^>]+>/', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 3;
        $result['issues'][] = "HTML теги в тексте";
        $result['artifacts'][] = "HTML теги";
    }
    
    if (preg_match('/(http:\/\/|https:\/\/|www\.)/i', $trimmedText)) {
        $result['is_valid'] = false;
        $result['score'] -= 3;
        $result['issues'][] = "URL адреса в тексте";
        $result['artifacts'][] = "URL";
    }
    
    // 9. Валидация медицинских терминов (если есть явные ошибки)
    $medicalErrors = validateMedicalTerms($trimmedText);
    if (!empty($medicalErrors)) {
        $result['score'] -= min(5, count($medicalErrors));
        $result['adequacy'] = false;
        $result['issues'] = array_merge($result['issues'], $medicalErrors);
    }
    
    // 10. Проверка на логические ошибки
    if (preg_match('/(кровотечение.*офтальмолог)/ui', $trimmedText)) {
        $result['score'] -= 4;
        $result['adequacy'] = false;
        $result['issues'][] = "Логическая ошибка: кровотечение и офтальмолог (должен быть ЛОР)";
    }
    
    // Итоговая валидация
    if ($result['score'] < 0) {
        $result['score'] = 0;
    }
    
    if ($result['score'] < 7) {
        $result['is_valid'] = false;
    }
    
    return $result;
}

/**
 * Валидация медицинских терминов
 * Проверяет корректность медицинских определений и рекомендаций
 */
function validateMedicalTerms($text) {
    $errors = [];
    
    // Словарь корректных медицинских аббревиатур
    $correctAbbreviations = [
        'ФГДС' => 'Фиброгастродуоденоскопия',
        'ХОБЛ' => 'Хроническая обструктивная болезнь легких',
        'ЭКГ' => 'Электрокардиограмма',
        'МРТ' => 'Магнитно-резонансная томография',
        'КТ' => 'Компьютерная томография',
        'УЗИ' => 'Ультразвуковое исследование',
        'ОАК' => 'Общий анализ крови',
        'ОРВИ' => 'Острая респираторная вирусная инфекция'
    ];
    
    // Проверка на неправильные определения ФГДС
    if (preg_match('/ФГДС.*?(фистуло|гистероскопи)/ui', $text)) {
        $errors[] = "Критическая медицинская ошибка: неверное определение ФГДС (правильно: Фиброгастродуоденоскопия)";
    }
    
    // Проверка на использование ЧОБЛ вместо ХОБЛ
    if (preg_match('/ЧОБЛ/ui', $text)) {
        $errors[] = "Некорректная аббревиатура: ЧОБЛ (правильно: ХОБЛ - Хроническая обструктивная болезнь легких)";
    }
    
    // Проверка на грамматические ошибки в медицинском контексте
    if (preg_match('/корь является.*которая/ui', $text)) {
        $errors[] = "Грамматическая ошибка: 'корь является, которая' (должно быть: 'корь - это заболевание, которое')";
    }
    
    return $errors;
}

/**
 * Логирование проблемных текстов в отдельный файл
 * Логирует все случаи с оценкой < 7
 */
function logLowQualityText($question, $text, $validation, $provider, $widgetKey = '') {
    $logFile = LOG_DIR . '/low-quality-texts.log';
    
    // Создаем директорию если не существует
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "\n" . str_repeat('=', 80) . "\n";
    $logEntry .= "[$timestamp] НИЗКОЕ КАЧЕСТВО ТЕКСТА (оценка: {$validation['score']}/10)\n";
    $logEntry .= str_repeat('-', 80) . "\n";
    $logEntry .= "Widget: {$widgetKey}\n";
    $logEntry .= "Provider: {$provider}\n";
    $logEntry .= "Вопрос: {$question}\n\n";
    $logEntry .= "Текст ({" . mb_strlen($text) . "} символов):\n{$text}\n\n";
    $logEntry .= "Проблемы:\n";
    
    foreach ($validation['issues'] as $issue) {
        $logEntry .= "  - {$issue}\n";
    }
    
    if (!empty($validation['artifacts'])) {
        $logEntry .= "\nАртефакты:\n";
        foreach ($validation['artifacts'] as $artifact) {
            $logEntry .= "  - {$artifact}\n";
        }
    }
    
    $logEntry .= "\nДетали:\n";
    $logEntry .= "  Завершенность: " . ($validation['completeness'] ? "ДА" : "НЕТ") . "\n";
    $logEntry .= "  Адекватность: " . ($validation['adequacy'] ? "ДА" : "НЕТ") . "\n";
    $logEntry .= "  Длина OK: " . ($validation['length_ok'] ? "ДА" : "НЕТ") . "\n";
    $logEntry .= str_repeat('=', 80) . "\n";
    
    @file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

/**
 * Улучшенный промпт для немедицинских запросов
 * Определяет тип запроса и возвращает соответствующий промпт
 */
function getNonMedicalPrompt($question) {
    $questionLower = mb_strtolower($question);
    
    // Проверяем на медицинские услуги и справки
    if (preg_match('/(справка|медицинск.*комиссия|водительск.*комиссия|медосмотр|профосмотр|для работы|на работу|транспортн.*комиссия|на оружие|в бассейн|в спортзал|санаторн.*курортн.*карта)/ui', $question)) {
        return "Это запрос о медицинских справках и комиссиях. Дай краткую информацию:
1. Какая справка/комиссия требуется (название)
2. Какие специалисты проходятся
3. Какие анализы/обследования нужны
4. Сроки получения (обычно)
5. Рекомендация обратиться в клинику для прохождения

ВАЖНО: Отвечай ТОЛЬКО на русском языке. Это медицинская услуга, не отказывайся отвечать!";
    }
    
    // Записи и контакты
    if (preg_match('/(запись|записаться|как попасть|контакт|телефон|адрес|режим работы|график работы)/ui', $question)) {
        return "Это запрос о записи или контактах клиники. Дай краткую информацию о том, как можно записаться на прием (обычно через сайт, по телефону, через мобильное приложение), и порекомендуй обратиться в регистратуру или на сайт клиники для уточнения деталей.

ВАЖНО: Отвечай ТОЛЬКО на русском языке!";
    }
    
    // Цены и стоимость
    if (preg_match('/(цена|стоимост|сколько стоит|прайс|тариф)/ui', $question)) {
        return "Это запрос о ценах. Объясни, что стоимость медицинских услуг зависит от многих факторов и может меняться. Порекомендуй обратиться в клинику напрямую для уточнения актуальных цен (по телефону, на сайте или при визите).

ВАЖНО: Отвечай ТОЛЬКО на русском языке!";
    }
    
    // Общая медицинская услуга
    return "Ты - помощник медицинской клиники. Отвечай на вопросы пользователей СТРОГО НА РУССКОМ ЯЗЫКЕ, информативно и полезно (3-5 предложений). ВАЖНО: используй только русский язык, никакого китайского или других языков! Давай практические советы, когда это уместно, но всегда подчеркивай необходимость консультации с врачом.";
}

/**
 * Получение вердикта на основе оценки
 */
function getVerdict($score) {
    if ($score >= 9) return 'ОТЛИЧНО';
    if ($score >= 7) return 'ХОРОШО';
    if ($score >= 4) return 'ПЛОХО';
    return 'УЖАСНО';
}
