<?php
/**
 * API endpoint для отправки запроса к Claude
 * POST /api/query.php
 * Body: { "widget_key": "clinic-001", "question": "У меня болит спина" }
 */

// Отключаем вывод ошибок на экран
ini_set('display_errors', 0);
error_reporting(E_ALL);

// Устанавливаем заголовок JSON сразу (только для прямых API запросов)
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'CLI_TEST' && ($_SERVER['REQUEST_METHOD'] ?? '') !== 'GET') {
    header('Content-Type: application/json');
}

// Обработка фатальных ошибок для гарантии JSON ответа
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        http_response_code(500);
        echo json_encode([
            'error' => 'Internal server error: ' . $error['message'],
            'ready' => false,
            'debug' => [
                'file' => $error['file'],
                'line' => $error['line'],
                'type' => $error['type']
            ]
        ], JSON_UNESCAPED_UNICODE);
    }
});

// Временное логирование для отладки
error_log("API query.php called from: " . ($_SERVER['REQUEST_URI'] ?? 'unknown'));

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/embedding-functions.php';

// Обработка preflight запроса (OPTIONS)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
    
    if ($origin) {
        // Проверяем origin в базе данных
        $db = getDatabase();
        $stmt = $db->prepare("SELECT allowed_domains FROM widget_settings WHERE allowed_domains LIKE ?");
        $stmt->execute(['%' . $origin . '%']);
        $result = $stmt->fetch();
        
        if ($result) {
            header("Access-Control-Allow-Origin: $origin");
            header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
            header("Access-Control-Allow-Headers: Content-Type");
            header("Access-Control-Max-Age: 86400"); // 24 часа
            http_response_code(204);
            exit;
        }
    }
    
    // Если домен не найден или origin пустой
    http_response_code(403);
    exit;
}

// Проверяем, что это прямой запрос к API, а не подключение из другого файла
// Если REQUEST_METHOD не POST или это CLI_TEST (из админки), пропускаем основной код
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'CLI_TEST') {
    // Это подключение из админки, не выполняем основной код API
    // Функции будут доступны, но основной код не выполнится
} elseif (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
    // Это не POST запрос, не выполняем основной код
} else {
    // CORS headers
    $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
    
    // Получаем данные запроса
    $input = json_decode(file_get_contents('php://input'), true);
    $widget_key = $input['widget_key'] ?? '';
    $question = trim($input['question'] ?? '');
    $debugMode = $input['_debug'] ?? false; // Режим отладки для preview
    $noCache = $input['_no_cache'] ?? false; // Отключение кеша для тестирования
    
    if (!$widget_key || !$question) {
        http_response_code(400);
        echo json_encode(['error' => 'widget_key and question are required']);
        exit;
    }

    try {
    $db = getDatabase();
    
    // Получаем виджет и настройки
    $stmt = $db->prepare("
        SELECT w.*, ws.*
        FROM widgets w
        LEFT JOIN widget_settings ws ON w.id = ws.widget_id
        WHERE w.widget_key = ? AND w.active = 1
    ");
    $stmt->execute([$widget_key]);
    $widget = $stmt->fetch();
    
    if (!$widget) {
        http_response_code(404);
        echo json_encode(['error' => 'Widget not found']);
        exit;
    }
    
    // Проверяем CORS
    $allowed_domains = json_decode($widget['allowed_domains'] ?? '[]', true);
    
    // Определяем домен запроса (из origin или из Referer)
    $request_host = '';
    if ($origin) {
        $request_host = parse_url($origin, PHP_URL_HOST);
    } elseif (isset($_SERVER['HTTP_REFERER'])) {
        $request_host = parse_url($_SERVER['HTTP_REFERER'], PHP_URL_HOST);
    } elseif (isset($_SERVER['HTTP_HOST'])) {
        $request_host = $_SERVER['HTTP_HOST'];
    }
    
    // Проверяем, является ли запрос с того же домена (включая админку)
    $own_domain = parse_url(WIDGET_DOMAIN, PHP_URL_HOST);
    $is_own_domain = false;
    if ($request_host) {
        // Проверяем совпадение домена или поддомена
        $is_own_domain = (
            $request_host === $own_domain ||
            strpos($request_host, $own_domain) !== false ||
            strpos($own_domain, $request_host) !== false ||
            // Проверяем общий корневой домен (medmaps.ru)
            (preg_match('/\.medmaps\.ru$/', $request_host) && preg_match('/\.medmaps\.ru$/', $own_domain))
        );
    }
    
    // Разрешаем доступ если:
    // 1. Origin пустой (запрос с того же домена или прямой доступ)
    // 2. Origin = наш домен (w2.medmaps.ru или w.medmaps.ru)
    // 3. Origin в списке разрешенных доменов
    // 4. Debug режим включен
    // 5. Запрос с того же домена (включая админку)
    $is_allowed = $debugMode || empty($origin) || $is_own_domain || in_array($origin, $allowed_domains);
    
    if (!empty($allowed_domains) && !$is_allowed && !$debugMode && !$is_own_domain) {
        http_response_code(403);
        echo json_encode(['error' => 'Domain not allowed']);
        exit;
    }
    
    // Устанавливаем CORS headers
    if ($origin) {
        if ($is_own_domain || in_array($origin, $allowed_domains)) {
            header("Access-Control-Allow-Origin: $origin");
            header("Access-Control-Allow-Methods: GET, POST");
            header("Access-Control-Allow-Headers: Content-Type");
        }
    }
    
    // Проверяем кеш ответов СНАЧАЛА (локальное кеширование на 1 час)
    // Кешируем только ответ Claude (текст + ID записей), данные всегда берем свежие из БД
    $cacheEnabled = (bool)$widget['cache_enabled'];
    $cacheHash = md5($widget_key . '|' . $question);
    $answerCache = null;
    
    // Пропускаем проверку кеша если передан параметр _no_cache (для тестирования)
    if ($cacheEnabled && !$noCache) {
        $cacheTTL = (int)($widget['cache_ttl_minutes'] ?? 60); // Дефолт 1 час
        $stmt = $db->prepare("
            SELECT answer, claude_data_ids FROM widget_logs 
            WHERE widget_id = ? AND question = ? AND error_message IS NULL
            AND created_at > DATE_SUB(NOW(), INTERVAL ? MINUTE)
            ORDER BY created_at DESC LIMIT 1
        ");
        $stmt->execute([$widget['id'], $question, $cacheTTL]);
        $answerCache = $stmt->fetch();
    }
    
    $startTime = microtime(true);
    
    if ($answerCache && !empty($answerCache['claude_data_ids']) && !$noCache) {
        // Возвращаем из кеша (не тратим токены Claude, не считаем в rate limit)
        // Но данные берем СВЕЖИЕ из БД
        $dataIds = json_decode($answerCache['claude_data_ids'], true);
        
        // ВАЖНО: Автоматически добавляем услуги консультаций для специалистов
        // даже при загрузке из кеша (на случай, если в старом кеше их нет)
        if (!empty($dataIds['specialists']) && count($dataIds['specialists']) > 0) {
            $specialistServices = findConsultationServicesForSpecialists($db, $widget['id'], $dataIds['specialists']);
            
            if (!empty($specialistServices)) {
                // Объединяем с уже выбранными услугами, убирая дубликаты
                $existingServices = $dataIds['services'] ?? [];
                $dataIds['services'] = array_values(array_unique(array_merge($existingServices, $specialistServices)));
            }
        }
        
        $fullData = enrichDataWithFullRecords($db, $widget['id'], $dataIds);
        
        $response = [
            'text' => $answerCache['answer'],
            'data' => $fullData,
            'cached' => true
        ];
        
        // Добавляем debug информацию если включен режим отладки (даже для кешированного ответа)
        if ($debugMode) {
            $response['debug'] = [
                'provider' => 'cached',
                'provider_model' => 'N/A (cached)',
                'cached' => true
            ];
        }
        
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    // Если нет в кеше - проверяем Rate limiting
    $ip = getClientIP();
    $cacheKey = "rate_limit_$ip";
    $cacheFile = WIDGET_ROOT . "/cache/$cacheKey";
    
    if (file_exists($cacheFile)) {
        $requests = json_decode(file_get_contents($cacheFile), true);
        $requests = array_filter($requests, fn($time) => $time > time() - RATE_LIMIT_PERIOD);
        
        if (count($requests) >= RATE_LIMIT_REQUESTS) {
            http_response_code(429);
            echo json_encode([
                'error' => 'Слишком много запросов (' . RATE_LIMIT_REQUESTS . ' запросов в минуту). Пожалуйста, подождите немного.',
                'type' => 'rate_limit'
            ]);
            exit;
        }
        
        $requests[] = time();
        file_put_contents($cacheFile, json_encode($requests));
    } else {
        if (!is_dir(dirname($cacheFile))) {
            mkdir(dirname($cacheFile), 0755, true);
        }
        file_put_contents($cacheFile, json_encode([time()]));
    }
    
    // Трёхэтапный режим: всегда выполняем этапы 2-3
    $threeStageDebug = []; // Для debug информации
    
    // Инициализируем глобальные переменные для сбора стоимостей OpenRouter
    $GLOBALS['openrouter_costs'] = [];
    $GLOBALS['embedding_costs'] = [];
    
    try {
        // Этап 2: Извлечение медицинских терминов
        logParser("Three-stage search: Extracting medical terms from question: " . substr($question, 0, 100));
        $stage2Start = microtime(true);
        $stage2Model = normalizeModelName($widget['stage2_model'] ?? 'qwen/qwen2.5-14b-instruct');
        $stage2Prompt = $widget['stage2_prompt'] ?? null;
        $stage2DebugInfo = null;
        $extractionResult = extractMedicalTerms($question, $widget['id'], $stage2Model, $stage2Prompt, $debugMode, $stage2DebugInfo, $widget['custom_api_url'] ?? null, $widget['custom_api_key'] ?? null);
        $stage2Time = round((microtime(true) - $stage2Start) * 1000);
            
            // extractMedicalTerms возвращает массив с medical_terms и cached флагом
            $medicalTerms = $extractionResult['medical_terms'] ?? $extractionResult; // Обратная совместимость
            $isCached = $extractionResult['cached'] ?? false;
            $cacheTimestamp = $extractionResult['cache_timestamp'] ?? null;
            $keywords = $medicalTerms['keywords'] ?? [];
            
            // Объединяем все термины для отображения (symptoms + specializations + service_types + keywords)
            $allExtractedTerms = [];
            if (isset($medicalTerms['symptoms']) && is_array($medicalTerms['symptoms'])) {
                $allExtractedTerms = array_merge($allExtractedTerms, $medicalTerms['symptoms']);
            }
            if (isset($medicalTerms['specializations']) && is_array($medicalTerms['specializations'])) {
                $allExtractedTerms = array_merge($allExtractedTerms, $medicalTerms['specializations']);
            }
            if (isset($medicalTerms['service_types']) && is_array($medicalTerms['service_types'])) {
                $allExtractedTerms = array_merge($allExtractedTerms, $medicalTerms['service_types']);
            }
            if (!empty($keywords)) {
                $allExtractedTerms = array_merge($allExtractedTerms, $keywords);
            }
            $allExtractedTerms = array_unique($allExtractedTerms);
            
        // Извлекаем стоимость Stage 2 из глобальной переменной (если использовался OpenRouter)
        $stage2Cost = null;
        if (isset($GLOBALS['openrouter_costs'])) {
            foreach ($GLOBALS['openrouter_costs'] as $costItem) {
                if ($costItem['stage'] === 'stage2') {
                    $stage2Cost = $costItem['cost'];
                    break;
                }
            }
        }
        
        // Формируем промпт для stage2 (если не был передан кастомный)
        $stage2PromptText = $stage2Prompt;
        if (!$stage2PromptText) {
            $stage2PromptText = "Ты - помощник для извлечения медицинских терминов из вопроса пользователя.\n\nВАЖНО: Верни ТОЛЬКО валидный JSON, без дополнительного текста до или после JSON.\n\nТвоя задача - максимально полно извлечь все релевантные медицинские термины из вопроса пользователя, включая:\n- Прямо упомянутые симптомы, заболевания, состояния\n- Связанные симптомы и состояния, которые могут быть связаны с упомянутыми\n- Все возможные специализации и специалистов, которые могут помочь\n- Типы услуг, диагностики и лечения, которые могут быть релевантны\n- Ключевые слова для поиска, включая синонимы, медицинские термины и разговорные названия\n\nДумай широко: если пользователь упоминает симптом, подумай о возможных причинах, связанных состояниях и специалистах, которые могут помочь.\n\nИз вопроса извлеки и верни в формате JSON:\n{\n  \"symptoms\": [массив всех симптомов, включая связанные и возможные],\n  \"specializations\": [массив всех релевантных специализаций и специалистов],\n  \"service_types\": [массив всех типов услуг, диагностики и лечения],\n  \"keywords\": [объединенный массив всех релевантных ключевых слов, включая синонимы и связанные термины]\n}\n\nВопрос пользователя: " . $question;
        }
        
        $threeStageDebug['stage2'] = [
            'enabled' => true,
            'question' => $question,
            'model' => $stage2Model,
            'prompt' => $stage2PromptText,
            'response' => json_encode($medicalTerms, JSON_UNESCAPED_UNICODE),
            'medical_terms' => $medicalTerms,
            'keywords' => $keywords, // Только keywords для поиска
            'all_extracted_terms' => array_values($allExtractedTerms), // Все извлеченные термины для отображения
            'time_ms' => $stage2Time,
            'cached' => $isCached, // Флаг использования кеша
            'cache_timestamp' => $cacheTimestamp, // Время создания кеша
            'yandex_debug' => $stage2DebugInfo, // Debug информация о запросах/ответах Yandex GPT
            'openrouter_cost' => $stage2Cost // Стоимость из OpenRouter (если использовался)
        ];
        
        logParser("Three-stage search: Extracted keywords: " . implode(', ', array_slice($keywords, 0, 10)));
        
        // Определение релевантных категорий для каждого раздела
        $relevantCategoriesBySection = [];
        if (!empty($keywords)) {
            // Получаем список разделов для виджета
            $stmt = $db->prepare("SELECT DISTINCT section_name FROM widget_sections WHERE widget_id = ? AND is_active = 1");
            $stmt->execute([$widget['id']]);
            $sections = $stmt->fetchAll();
            
            // Сначала пытаемся найти категории в БД для всех разделов
            $sectionsNeedingAI = [];
            foreach ($sections as $sectionRow) {
                $sectionName = $sectionRow['section_name'];
                $categories = getRelevantCategories($db, $widget['id'], $keywords, $question, $sectionName, false); // false = не использовать AI fallback
                if (!empty($categories)) {
                    $relevantCategoriesBySection[$sectionName] = $categories;
                    logParser("Three-stage search: Found " . count($categories) . " relevant categories for section: $sectionName");
                } else {
                    // Если не найдено в БД, добавляем в список для AI fallback
                    $sectionsNeedingAI[] = $sectionName;
                }
            }
            
            // Если есть разделы без категорий в БД, делаем один AI запрос для всех сразу
            if (!empty($sectionsNeedingAI) && !empty($question)) {
                $aiCategories = getRelevantCategoriesForAllSections($db, $widget['id'], $keywords, $question, $sectionsNeedingAI);
                foreach ($aiCategories as $sectionName => $categories) {
                    if (!empty($categories)) {
                        $relevantCategoriesBySection[$sectionName] = $categories;
                        logParser("Three-stage search: Found " . count($categories) . " relevant categories for section: $sectionName (via AI)");
                    }
                }
            }
        }
        
        // Поиск релевантных записей по ключевым словам (с фильтрацией по категориям)
        $relevantIds = [];
        $totalFound = 0;
        $sectionCounts = [];
        $searchStart = microtime(true);
        
        if (!empty($keywords)) {
            // Защита от случая, когда $medicalTerms не определен (например, при ошибке этапа 2)
            $medicalTerms = $medicalTerms ?? [];
            $specializations = isset($medicalTerms['specializations']) && is_array($medicalTerms['specializations']) ? $medicalTerms['specializations'] : [];
            $serviceTypes = isset($medicalTerms['service_types']) && is_array($medicalTerms['service_types']) ? $medicalTerms['service_types'] : [];
            
            // Используем embedding поиск, если включен, иначе обычный поиск по ключевым словам
            $embeddingDebugInfo = null;
            if (defined('USE_EMBEDDING_SEARCH') && USE_EMBEDDING_SEARCH) {
                $relevantIds = searchByEmbeddingAndCategories($db, $widget, $question, $relevantCategoriesBySection, $keywords, $specializations, $serviceTypes, $embeddingDebugInfo);
            } else {
                // Передаем категории для каждого раздела, а также специализации и типы услуг для расширенного поиска
                $relevantIds = searchByKeywords($db, $widget, $keywords, $relevantCategoriesBySection, $specializations, $serviceTypes);
            }
            
            foreach ($relevantIds as $section => $sectionIds) {
                $count = count($sectionIds);
                $totalFound += $count;
                $sectionCounts[$section] = $count;
                logParser("Three-stage search: Found $count items for section: $section");
            }
            
            logParser("Three-stage search: Found $totalFound relevant items across " . count($relevantIds) . " sections");
            
            // Фильтруем нерелевантные услуги на основе вопроса пользователя
            if (isset($relevantIds['services']) && !empty($relevantIds['services'])) {
                $filteredServiceIds = filterIrrelevantServices($db, $widget['id'], $relevantIds['services'], $question);
                $removedCount = count($relevantIds['services']) - count($filteredServiceIds);
                if ($removedCount > 0) {
                    logParser("Filtered out $removedCount irrelevant services based on question: " . substr($question, 0, 100));
                    $relevantIds['services'] = $filteredServiceIds;
                    $sectionCounts['services'] = count($filteredServiceIds);
                    $totalFound -= $removedCount;
                }
            }
        }
        
        // Этап 3: Формирование промпта с фильтрацией и основной запрос
        // Используем stage3_prompt если указан, иначе основной claude_prompt
        $stage3SystemPrompt = !empty($widget['stage3_prompt']) ? $widget['stage3_prompt'] : $widget['claude_prompt'];
        
        // Добавляем динамические инструкции на основе количества найденных записей
        if (!empty($sectionCounts)) {
            $dynamicInstructions = "\n\nКРИТИЧЕСКИ ВАЖНО - КОЛИЧЕСТВО ЗАПИСЕЙ В КАТАЛОГЕ:\n";
            foreach ($sectionCounts as $section => $count) {
                $sectionNameRu = [
                    'specialists' => 'специалистов',
                    'services' => 'услуг',
                    'articles' => 'статей',
                    'specializations' => 'специализаций'
                ][$section] ?? $section;
                
                $dynamicInstructions .= "- В каталоге найдено $count $sectionNameRu для твоего запроса. ";
                
                if ($section === 'services') {
                    if ($count <= 10) {
                        $dynamicInstructions .= "ВЫБЕРИ ВСЕ $count услуг! Они все релевантны запросу пользователя!\n";
                    } elseif ($count <= 15) {
                        $dynamicInstructions .= "Выбери МИНИМУМ 10-12 услуг из $count найденных! Не фильтруй слишком строго!\n";
                    } else {
                        $dynamicInstructions .= "Выбери МИНИМУМ 10-15 самых релевантных услуг из $count найденных!\n";
                    }
                } elseif ($section === 'specialists') {
                    if ($count <= 8) {
                        $dynamicInstructions .= "ВЫБЕРИ ВСЕ $count специалистов! Они все релевантны запросу!\n";
                    } else {
                        $dynamicInstructions .= "Выбери МИНИМУМ 5-7 самых релевантных специалистов!\n";
                    }
                } elseif ($section === 'articles') {
                    if ($count <= 6) {
                        $dynamicInstructions .= "ВЫБЕРИ ВСЕ $count статей! Они все релевантны запросу!\n";
                    } else {
                        $dynamicInstructions .= "Выбери МИНИМУМ 4-5 самых релевантных статей!\n";
                    }
                } else {
                    $dynamicInstructions .= "Выбери МИНИМУМ " . min($count, 5) . " самых релевантных записей!\n";
                }
            }
            $dynamicInstructions .= "\nПОМНИ: Если в каталоге мало записей (менее 10) - выбирай ВСЕ релевантные! Не фильтруй слишком строго!\n";
            $dynamicInstructions .= "\nКРИТИЧЕСКИ ВАЖНО - ИСПОЛЬЗОВАНИЕ ID И РАЗДЕЛОВ:\n";
            $dynamicInstructions .= "- Используй ТОЛЬКО те ID записей, которые указаны в каталоге данных ниже!\n";
            $dynamicInstructions .= "- НЕ выдумывай и НЕ генерируй новые ID - используй ТОЛЬКО существующие ID из каталога!\n";
            $dynamicInstructions .= "- КРИТИЧЕСКИ ВАЖНО: Каждый раздел в каталоге имеет свои ID! НЕ ПУТАЙ разделы!\n";
            $dynamicInstructions .= "- ID из раздела \"specialists\" должны идти ТОЛЬКО в массив \"specialists\"!\n";
            $dynamicInstructions .= "- ID из раздела \"services\" должны идти ТОЛЬКО в массив \"services\"!\n";
            $dynamicInstructions .= "- ID из раздела \"articles\" должны идти ТОЛЬКО в массив \"articles\"!\n";
            $dynamicInstructions .= "- ID из раздела \"specializations\" должны идти ТОЛЬКО в массив \"specializations\"!\n";
            $dynamicInstructions .= "- НЕ ПЕРЕНОСИ ID между разделами! Каждый ID принадлежит только своему разделу!\n";
            $dynamicInstructions .= "- Если в каталоге есть раздел \"services\" с ID [X, Y, Z], то в ответе в массив \"services\" должны попасть ТОЛЬКО ID из этого раздела!\n";
            $dynamicInstructions .= "- Если ты вернешь ID из другого раздела в неправильный массив - твой ответ будет неверным!\n";
            $stage3SystemPrompt .= $dynamicInstructions;
            logParser("Added dynamic instructions to prompt. Section counts: " . json_encode($sectionCounts, JSON_UNESCAPED_UNICODE));
        } else {
            logParser("WARNING: sectionCounts is empty, dynamic instructions NOT added!");
        }
        
        $promptDataBefore = buildPrompt($db, $widget, $question); // Для сравнения
        $promptSizeBefore = mb_strlen($promptDataBefore);
        $promptTokensBefore = estimateTokens($promptDataBefore);
        
        if (!empty($relevantIds)) {
            // Логируем количество записей по разделам перед построением промпта
            foreach ($relevantIds as $section => $ids) {
                $firstIds = array_slice($ids, 0, 5);
                logParser("Building prompt with " . count($ids) . " items for section: $section (first 5 IDs: " . implode(', ', $firstIds) . ")");
            }
            $promptData = buildPrompt($db, $widget, $question, $relevantIds);
            
            // Добавляем явное указание ID для каждого раздела перед каталогом
            $sectionIdsInfo = "\n\nВАЖНО - СПИСОК ID ПО РАЗДЕЛАМ (используй ТОЛЬКО эти ID!):\n";
            foreach ($relevantIds as $section => $ids) {
                $sectionNameRu = [
                    'specialists' => 'специалистов',
                    'services' => 'услуг',
                    'articles' => 'статей',
                    'specializations' => 'специализаций'
                ][$section] ?? $section;
                
                $idsList = implode(', ', array_slice($ids, 0, 20));
                if (count($ids) > 20) {
                    $idsList .= ' ... (всего ' . count($ids) . ' ID)';
                }
                $sectionIdsInfo .= "- Раздел \"$section\" ($sectionNameRu): ID = [$idsList]\n";
                $sectionIdsInfo .= "  → В твоем ответе в массив \"$section\" должны попасть ТОЛЬКО ID из этого списка!\n";
            }
            $sectionIdsInfo .= "\nПОМНИ: Каждый ID принадлежит только своему разделу! Не путай ID между разделами!\n";
            
            // Добавляем вопрос пользователя в начало промпта
            $userQuestionHeader = "\n\n=== ВОПРОС ПОЛЬЗОВАТЕЛЯ ===\n" . $question . "\n\n=== КАТАЛОГ ДАННЫХ ===\n";
            $promptData = $sectionIdsInfo . $userQuestionHeader . $promptData;
        } else {
            // Если поиск не дал результатов - используем все данные
            logParser("Keyword search returned no results - building prompt with ALL data (no filtering)");
            $promptData = buildPrompt($db, $widget, $question, []); // Пустой массив = все данные
            
            // Получаем все ID по разделам для инструкций
            $stmt = $db->prepare("
                SELECT section_name, GROUP_CONCAT(id ORDER BY id SEPARATOR ', ') as ids, COUNT(*) as count
                FROM parsed_items 
                WHERE widget_id = ? AND is_duplicate = 0 
                GROUP BY section_name
            ");
            $stmt->execute([$widget['id']]);
            $allIdsBySection = [];
            while ($row = $stmt->fetch()) {
                $allIdsBySection[$row['section_name']] = explode(', ', $row['ids']);
                $sectionCounts[$row['section_name']] = (int)$row['count'];
            }
            
            // Добавляем явное указание ID для каждого раздела перед каталогом
            $sectionIdsInfo = "\n\nВАЖНО - СПИСОК ID ПО РАЗДЕЛАМ (используй ТОЛЬКО эти ID!):\n";
            foreach ($allIdsBySection as $section => $ids) {
                $sectionNameRu = [
                    'specialists' => 'специалистов',
                    'services' => 'услуг',
                    'articles' => 'статей',
                    'specializations' => 'специализаций'
                ][$section] ?? $section;
                
                $idsList = implode(', ', array_slice($ids, 0, 20));
                if (count($ids) > 20) {
                    $idsList .= ' ... (всего ' . count($ids) . ' ID)';
                }
                $sectionIdsInfo .= "- Раздел \"$section\" ($sectionNameRu): ID = [$idsList]\n";
                $sectionIdsInfo .= "  → В твоем ответе в массив \"$section\" должны попасть ТОЛЬКО ID из этого списка!\n";
            }
            $sectionIdsInfo .= "\nПОМНИ: Каждый ID принадлежит только своему разделу! Не путай ID между разделами!\n";
            
            // Добавляем вопрос пользователя в начало промпта
            $userQuestionHeader = "\n\n=== ВОПРОС ПОЛЬЗОВАТЕЛЯ ===\n" . $question . "\n\n=== КАТАЛОГ ДАННЫХ ===\n";
            $promptData = $sectionIdsInfo . $userQuestionHeader . $promptData;
            
            // Обновляем динамические инструкции на основе реального количества записей
            if (!empty($sectionCounts)) {
                $dynamicInstructions = "\n\nКРИТИЧЕСКИ ВАЖНО - КОЛИЧЕСТВО ЗАПИСЕЙ В КАТАЛОГЕ:\n";
                foreach ($sectionCounts as $section => $count) {
                    $sectionNameRu = [
                        'specialists' => 'специалистов',
                        'services' => 'услуг',
                        'articles' => 'статей',
                        'specializations' => 'специализаций'
                    ][$section] ?? $section;
                    
                    $dynamicInstructions .= "- В каталоге найдено $count $sectionNameRu. ";
                    
                    if ($section === 'services') {
                        $dynamicInstructions .= "Выбери МИНИМУМ 10-15 самых релевантных услуг!\n";
                    } elseif ($section === 'specialists') {
                        $dynamicInstructions .= "Выбери МИНИМУМ 3-5 самых релевантных специалистов!\n";
                    } elseif ($section === 'articles') {
                        $dynamicInstructions .= "Выбери МИНИМУМ 3-5 самых релевантных статей!\n";
                    } else {
                        $dynamicInstructions .= "Выбери МИНИМУМ " . min($count, 5) . " самых релевантных записей!\n";
                    }
                }
                $stage3SystemPrompt .= $dynamicInstructions;
            }
        }
        
        $promptSizeAfter = mb_strlen($promptData);
        $promptTokensAfter = estimateTokens($promptData);
        
        $threeStageDebug['stage3'] = [
            'enabled' => true,
            'model' => normalizeModelName($widget['stage3_model'] ?? 'qwen/qwen3-235b-a22b-2507'),
            'prompt_size_before' => $promptSizeBefore,
            'prompt_tokens_before' => $promptTokensBefore,
            'prompt_size_after' => $promptSizeAfter,
            'prompt_tokens_after' => $promptTokensAfter,
            'reduction_percent' => $promptSizeBefore > 0 ? round((1 - $promptSizeAfter / $promptSizeBefore) * 100, 1) : 0,
            'prompt_before' => $stage3SystemPrompt . "\n\n" . $promptDataBefore, // Полный промпт до оптимизации
            'prompt_after' => $stage3SystemPrompt . "\n\n" . $promptData, // Полный промпт после оптимизации
            'relevant_categories' => $relevantCategoriesBySection, // Категории по разделам
            'items_found' => $sectionCounts, // Количество найденных записей по разделам
            'total_items_found' => $totalFound, // Общее количество найденных записей
            'embedding_search' => $embeddingDebugInfo, // Информация об embedding поиске
            'yandex_debug' => null // Будет заполнено после выполнения этапа 3
        ];
        
        // Этап 3: Финальный запрос (без кеширования, т.к. промпт динамический или _no_cache)
        $stage3Model = normalizeModelName($widget['stage3_model'] ?? 'qwen/qwen3-235b-a22b-2507');
        $stage3DebugInfo = null;
        $useCacheForRequest = !$noCache; // Отключаем кеш если передан _no_cache
        $claudeResponse = sendClaudeRequest($stage3SystemPrompt, $promptData, $question, $useCacheForRequest, $stage3Model, $debugMode, $stage3DebugInfo, $widget['custom_api_url'] ?? null, $widget['custom_api_key'] ?? null);
        
        // Обновляем debug информацию для этапа 3 (может быть обновлена внутри sendClaudeRequest)
        if ($debugMode && $stage3DebugInfo !== null) {
            $threeStageDebug['stage3']['yandex_debug'] = $stage3DebugInfo;
        }
        
    } catch (Exception $e) {
        // Fallback на старый метод при ошибке
        $threeStageDebug['error'] = $e->getMessage();
        $threeStageDebug['fallback'] = true;
        logParser("Three-stage search failed, falling back to old method: " . $e->getMessage());
        $promptData = buildPrompt($db, $widget, $question);
        $stage3Model = normalizeModelName($widget['stage3_model'] ?? 'qwen/qwen3-235b-a22b-2507');
        // Используем stage3_prompt если указан, иначе основной claude_prompt
        $stage3SystemPrompt = !empty($widget['stage3_prompt']) ? $widget['stage3_prompt'] : $widget['claude_prompt'];
        $stage3DebugInfo = null;
        $useCacheForRequest = !$noCache; // Отключаем кеш если передан _no_cache
        $claudeResponse = sendClaudeRequest($stage3SystemPrompt, $promptData, $question, $useCacheForRequest, $stage3Model, $debugMode, $stage3DebugInfo, $widget['custom_api_url'] ?? null, $widget['custom_api_key'] ?? null);
        
        // Обновляем debug информацию для этапа 3 (может быть обновлена внутри sendClaudeRequest)
        if ($debugMode && $stage3DebugInfo !== null && isset($threeStageDebug['stage3'])) {
            $threeStageDebug['stage3']['yandex_debug'] = $stage3DebugInfo;
        }
    }
    
    $responseTime = round((microtime(true) - $startTime) * 1000);
    
    // Проверяем, что $claudeResponse определен и успешен
    if (!isset($claudeResponse) || !$claudeResponse['success']) {
        $errorMsg = isset($claudeResponse['error']) ? $claudeResponse['error'] : 'Неизвестная ошибка при обработке запроса';
        throw new Exception($errorMsg);
    }
    
    // Логируем что вернул AI
    $aiDataIds = $claudeResponse['data_ids'] ?? [];
    
    // Логируем сырые данные от AI перед валидацией
    if (isset($aiDataIds['specialists'])) {
        logParser("AI returned specialists IDs (before validation): " . implode(', ', $aiDataIds['specialists']));
    }
    
    // Валидация: проверяем, что все ID существуют в БД и принадлежат правильному разделу
    $validatedDataIds = [];
    foreach ($aiDataIds as $section => $ids) {
        if (empty($ids) || !is_array($ids)) {
            $validatedDataIds[$section] = [];
            continue;
        }
        
        // Преобразуем строковые ID в числовые
        $ids = array_map(function($id) {
            return is_string($id) ? (int)$id : $id;
        }, $ids);
        
        // Проверяем существование ID в БД и правильность раздела
        $placeholders = str_repeat('?,', count($ids) - 1) . '?';
        $stmt = $db->prepare("
            SELECT DISTINCT id, section_name
            FROM parsed_items
            WHERE widget_id = ? 
            AND id IN ($placeholders)
            AND is_duplicate = 0
        ");
        $params = array_merge([$widget['id']], $ids);
        $stmt->execute($params);
        
        $validIds = [];
        $wrongSectionIds = [];
        while ($row = $stmt->fetch()) {
            $itemId = (int)$row['id'];
            $itemSection = $row['section_name'];
            
            // Проверяем, что ID принадлежит правильному разделу
            if ($itemSection === $section) {
                $validIds[] = $itemId;
            } else {
                $wrongSectionIds[] = [
                    'id' => $itemId,
                    'expected_section' => $section,
                    'actual_section' => $itemSection
                ];
            }
        }
        
        // Логируем несуществующие ID
        $foundIds = array_map(function($item) {
            return $item['id'];
        }, $wrongSectionIds);
        $foundIds = array_merge($foundIds, $validIds);
        $invalidIds = array_diff($ids, $foundIds);
        
        if (!empty($invalidIds)) {
            logParser("WARNING: AI returned non-existent IDs for section '$section': " . implode(', ', array_slice($invalidIds, 0, 10)) . (count($invalidIds) > 10 ? '...' : ''));
        }
        
        // Логируем ID из неправильных разделов
        if (!empty($wrongSectionIds)) {
            $wrongSectionDetails = array_map(function($item) {
                return "ID {$item['id']} (ожидался раздел '{$item['expected_section']}', но найден в разделе '{$item['actual_section']}')";
            }, array_slice($wrongSectionIds, 0, 5));
            logParser("WARNING: AI returned IDs from wrong sections for '$section': " . implode(', ', $wrongSectionDetails) . (count($wrongSectionIds) > 5 ? '...' : ''));
            logParser("  - Requested: " . count($ids) . " IDs, Valid (correct section): " . count($validIds) . " IDs, Wrong section: " . count($wrongSectionIds) . " IDs, Non-existent: " . count($invalidIds) . " IDs");
        } else if (!empty($invalidIds)) {
            logParser("  - Requested: " . count($ids) . " IDs, Valid: " . count($validIds) . " IDs, Invalid: " . count($invalidIds) . " IDs");
        }
        
        $validatedDataIds[$section] = $validIds;
        
        // Fallback: если AI вернул неправильные ID (все отфильтрованы), но в промпте были правильные ID для этого раздела
        // Используем первые N ID из промпта как fallback
        if (empty($validIds) && !empty($relevantIds) && isset($relevantIds[$section]) && !empty($relevantIds[$section])) {
            $fallbackCount = min(count($relevantIds[$section]), 5); // Берем максимум 5 ID как fallback
            $fallbackIds = array_slice($relevantIds[$section], 0, $fallbackCount);
            logParser("WARNING: AI returned no valid IDs for section '$section', using fallback: " . implode(', ', $fallbackIds) . " (from prompt)");
            $validatedDataIds[$section] = $fallbackIds;
        }
    }
    
    
    // Заменяем данные на валидированные (временно, для логирования)
    $claudeResponse['data_ids'] = $validatedDataIds;
    
    $aiServicesCount = isset($validatedDataIds['services']) ? count($validatedDataIds['services']) : 0;
    $aiSpecialistsCount = isset($validatedDataIds['specialists']) ? count($validatedDataIds['specialists']) : 0;
    $aiArticlesCount = isset($validatedDataIds['articles']) ? count($validatedDataIds['articles']) : 0;
    $aiSpecializationsCount = isset($validatedDataIds['specializations']) ? count($validatedDataIds['specializations']) : 0;
    
    logParser("AI Response Analysis:");
    logParser("  - AI selected specialists: $aiSpecialistsCount (was in prompt: " . ($sectionCounts['specialists'] ?? 0) . ")");
    logParser("  - AI selected services: $aiServicesCount (was in prompt: " . ($sectionCounts['services'] ?? 0) . ")");
    logParser("  - AI selected articles: $aiArticlesCount (was in prompt: " . ($sectionCounts['articles'] ?? 0) . ")");
    logParser("  - AI selected specializations: $aiSpecializationsCount (was in prompt: " . ($sectionCounts['specializations'] ?? 0) . ")");
    
    if ($aiServicesCount > 0 && isset($validatedDataIds['services'])) {
        $firstServices = array_slice($validatedDataIds['services'], 0, 5);
        logParser("  - First 5 service IDs selected by AI (after validation): " . implode(', ', $firstServices));
    }
    
    // Автоматически добавляем услуги консультаций для выбранных специалистов
    if (!empty($validatedDataIds['specialists']) && count($validatedDataIds['specialists']) > 0) {
        $specialistServices = findConsultationServicesForSpecialists($db, $widget['id'], $validatedDataIds['specialists']);
        
        if (!empty($specialistServices)) {
            // Объединяем с уже выбранными услугами, убирая дубликаты
            $existingServices = $validatedDataIds['services'] ?? [];
            $validatedDataIds['services'] = array_values(array_unique(array_merge($existingServices, $specialistServices)));
            
            logParser("Auto-added " . count($specialistServices) . " consultation services for " . count($validatedDataIds['specialists']) . " specialists");
            if (count($specialistServices) > 0) {
                logParser("  - Added service IDs: " . implode(', ', array_slice($specialistServices, 0, 10)) . (count($specialistServices) > 10 ? '...' : ''));
            }
        }
    }
    
    // Финальная проверка релевантности услуг через AI модель
    if (!empty($validatedDataIds['services']) && count($validatedDataIds['services']) > 0) {
        $stage3Model = normalizeModelName($widget['stage3_model'] ?? 'qwen/qwen3-235b-a22b-2507');
        $filteredServices = filterServicesByAI($db, $widget['id'], $validatedDataIds['services'], $question, $stage3Model, $widget['custom_api_url'] ?? null, $widget['custom_api_key'] ?? null);
        
        $removedCount = count($validatedDataIds['services']) - count($filteredServices);
        if ($removedCount > 0) {
            logParser("AI final filter removed $removedCount irrelevant services (from " . count($validatedDataIds['services']) . " to " . count($filteredServices) . ")");
        }
        
        $validatedDataIds['services'] = $filteredServices;
    }
    
    // ВАЖНО: Обновляем claudeResponse['data_ids'] ПОСЛЕ добавления автоматических услуг и финальной фильтрации
    // чтобы в кеш сохранялись полные данные с автоматически добавленными услугами
    $claudeResponse['data_ids'] = $validatedDataIds;
    
    // Обогащаем данные полными записями из БД
    $fullData = enrichDataWithFullRecords($db, $widget['id'], $validatedDataIds);
    
    // Парсим ответ Claude
    $answer = $claudeResponse['answer'];
    $tokensUsed = $claudeResponse['tokens_used'];
    
    // Собираем стоимость OpenRouter запросов для сохранения в логах
    $totalOpenRouterCost = 0;
    
    // Стоимость Stage 2 (если использовался OpenRouter)
    if (isset($threeStageDebug['stage2']['openrouter_cost'])) {
        $totalOpenRouterCost += $threeStageDebug['stage2']['openrouter_cost'];
    }
    
    // Стоимость Stage 3 (если использовался OpenRouter)
    if (isset($claudeResponse['cost'])) {
        $totalOpenRouterCost += $claudeResponse['cost'];
    }
    
    // Стоимость Embedding запросов
    if (isset($GLOBALS['embedding_costs']) && is_array($GLOBALS['embedding_costs'])) {
        $totalEmbeddingCost = array_sum($GLOBALS['embedding_costs']);
        $totalOpenRouterCost += $totalEmbeddingCost;
    }
    
    // Логируем тестирование промпта в БД (если debugMode включен)
    if ($debugMode) {
        logPromptTest($db, $widget['id'], $question, $threeStageDebug, $stage3SystemPrompt, $promptData, $claudeResponse, $responseTime);
    }
    
    // Логируем запрос
    $stmt = $db->prepare("
        INSERT INTO widget_logs (widget_id, question, answer, response_json, claude_data_ids, tokens_used, response_time_ms, ip_address, user_agent, openrouter_cost)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    $stmt->execute([
        $widget['id'],
        $question,
        $answer,
        json_encode($fullData, JSON_UNESCAPED_UNICODE),
        json_encode($claudeResponse['data_ids'], JSON_UNESCAPED_UNICODE),
        $tokensUsed,
        $responseTime,
        $ip,
        getUserAgent(),
        $totalOpenRouterCost > 0 ? $totalOpenRouterCost : null
    ]);
    
    // Формируем ответ
    $response = [
        'text' => $answer,
        'data' => $fullData,
        'cached' => false
    ];
    
    // Добавляем отладочную информацию если включен режим отладки
    if ($debugMode) {
        // Определяем какой провайдер был использован
        $providerUsed = 'claude';
        if (isset($claudeResponse['provider'])) {
            if ($claudeResponse['provider'] === 'openrouter') {
                $providerUsed = 'openrouter';
            } elseif ($claudeResponse['provider'] === 'yandexgpt') {
                $providerUsed = 'yandexgpt';
            }
        }
        
        // Определяем модель провайдера
        $providerModel = CLAUDE_MODEL;
        if ($providerUsed === 'openrouter') {
            $providerModel = OPENROUTER_MODEL;
        } elseif ($providerUsed === 'yandexgpt') {
            // Берем модель из ответа или из настроек виджета
            $providerModel = $claudeResponse['provider_model'] ?? ($widget['stage3_model'] ?? 'gpt://' . YANDEXGPT_FOLDER_ID . '/yandexgpt-lite/latest');
        }
        
        // Расчет стоимости для Claude Haiku 4.5
        $inputTokens = estimateTokens($promptData) + estimateTokens($widget['claude_prompt']);
        $outputTokens = $tokensUsed;
        $useCacheInRequest = false; // Кеш выключен при трёхэтапном поиске (промпт динамический)
        
        $cost = [
            'input_tokens' => $inputTokens,
            'output_tokens' => $outputTokens,
            'use_cache' => $useCacheInRequest,
            'cache_ttl' => CLAUDE_CACHE_TTL
        ];
        
        // Цены для Claude Haiku 4.5
        $prices = [
            'input_base' => 1.0,
            'input_cache_5m' => 1.25,
            'input_cache_1h' => 2.0,
            'output' => 5.0
        ];
        
        if ($useCacheInRequest) {
            if (CLAUDE_CACHE_TTL === '5m') {
                $cost['input_cost'] = ($inputTokens / 1000000) * $prices['input_cache_5m'];
            } else {
                $cost['input_cost'] = ($inputTokens / 1000000) * $prices['input_cache_1h'];
            }
        } else {
            $cost['input_cost'] = ($inputTokens / 1000000) * $prices['input_base'];
        }
        
        $cost['output_cost'] = ($outputTokens / 1000000) * $prices['output'];
        $cost['total_cost'] = $cost['input_cost'] + $cost['output_cost'];
        
        // Добавляем стоимость в этап 3 трёхэтапного поиска
        if (isset($threeStageDebug['stage3'])) {
            $threeStageDebug['stage3']['cost'] = $cost;
        }
        
        // Собираем все стоимости из OpenRouter запросов
        $totalOpenRouterCost = 0;
        $openRouterCosts = [];
        
        // Стоимость Stage 2 (если использовался OpenRouter)
        if (isset($threeStageDebug['stage2']['openrouter_cost'])) {
            $openRouterCosts['stage2'] = $threeStageDebug['stage2']['openrouter_cost'];
            $totalOpenRouterCost += $threeStageDebug['stage2']['openrouter_cost'];
        }
        
        // Стоимость Stage 3 (если использовался OpenRouter)
        if (isset($claudeResponse['cost'])) {
            $openRouterCosts['stage3'] = $claudeResponse['cost'];
            $totalOpenRouterCost += $claudeResponse['cost'];
        }
        
        // Стоимость Embedding запросов
        $totalEmbeddingCost = 0;
        if (isset($GLOBALS['embedding_costs']) && is_array($GLOBALS['embedding_costs'])) {
            $totalEmbeddingCost = array_sum($GLOBALS['embedding_costs']);
            $openRouterCosts['embedding'] = $totalEmbeddingCost;
            $totalOpenRouterCost += $totalEmbeddingCost;
        }
        
        // Добавляем стоимость для этапа 2 (если есть)
        if (isset($threeStageDebug['stage2'])) {
            if (isset($threeStageDebug['stage2']['openrouter_cost'])) {
                // Используем реальную стоимость из OpenRouter
                $threeStageDebug['stage2']['cost'] = [
                    'total_cost' => $threeStageDebug['stage2']['openrouter_cost'],
                    'source' => 'openrouter'
                ];
            } else {
                // Используем оценку для Claude/Yandex GPT
                $stage2InputTokens = 500; // Примерная оценка
                $stage2OutputTokens = 200;
                $threeStageDebug['stage2']['cost'] = [
                    'input_tokens' => $stage2InputTokens,
                    'output_tokens' => $stage2OutputTokens,
                    'input_cost' => ($stage2InputTokens / 1000000) * $prices['input_base'],
                    'output_cost' => ($stage2OutputTokens / 1000000) * $prices['output'],
                    'total_cost' => (($stage2InputTokens / 1000000) * $prices['input_base']) + (($stage2OutputTokens / 1000000) * $prices['output']),
                    'source' => 'estimated'
                ];
            }
        }
        
        // Обновляем стоимость Stage 3, если использовался OpenRouter
        if (isset($threeStageDebug['stage3']) && isset($openRouterCosts['stage3'])) {
            if (isset($threeStageDebug['stage3']['cost'])) {
                $threeStageDebug['stage3']['cost']['openrouter_cost'] = $openRouterCosts['stage3'];
                $threeStageDebug['stage3']['cost']['source'] = 'openrouter';
            } else {
                $threeStageDebug['stage3']['cost'] = [
                    'total_cost' => $openRouterCosts['stage3'],
                    'source' => 'openrouter'
                ];
            }
        }
        
        // Итоговая стоимость всех OpenRouter запросов
        if ($totalOpenRouterCost > 0) {
            $cost['openrouter_total'] = $totalOpenRouterCost;
            $cost['openrouter_breakdown'] = $openRouterCosts;
            logParser("Total OpenRouter cost: $" . number_format($totalOpenRouterCost, 6) . " (breakdown: " . json_encode($openRouterCosts, JSON_UNESCAPED_UNICODE) . ")");
        }
        
        $response['debug'] = [
            'provider' => $providerUsed, // 'yandexgpt', 'openrouter' или 'claude'
            'provider_model' => $providerModel,
            'claude_raw_response_text' => $claudeResponse['raw_response'], // Сырой текст от AI (не распарсенный)
            'claude_raw_response' => json_decode($claudeResponse['raw_response'], true), // Распарсенный
            'data_ids' => $claudeResponse['data_ids'],
            'db_data' => $fullData,
            'cache_stats' => $claudeResponse['cache_stats'] ?? null,
            'prompt_size_chars' => strlen($promptData),
            'prompt_size_tokens' => estimateTokens($promptData),
            'cost' => $cost,
            'three_stage_search' => $threeStageDebug,
            'openrouter_debug' => $claudeResponse['openrouter_debug'] ?? null // Информация об ошибках OpenRouter
        ];
    }
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    logError("Query API error: " . $e->getMessage());
    
    // Логируем ошибку
    if (isset($widget)) {
        try {
            // Очищаем сообщение от невалидных UTF-8 символов
            $errorMessage = mb_convert_encoding($e->getMessage(), 'UTF-8', 'UTF-8');
            $errorMessage = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x80-\x9F]/u', '', $errorMessage);
            
            // Пытаемся собрать стоимость OpenRouter, если она была собрана до ошибки
            $totalOpenRouterCostOnError = null;
            if (isset($threeStageDebug['stage2']['openrouter_cost'])) {
                $totalOpenRouterCostOnError = ($totalOpenRouterCostOnError ?? 0) + $threeStageDebug['stage2']['openrouter_cost'];
            }
            if (isset($claudeResponse['cost'])) {
                $totalOpenRouterCostOnError = ($totalOpenRouterCostOnError ?? 0) + $claudeResponse['cost'];
            }
            if (isset($GLOBALS['embedding_costs']) && is_array($GLOBALS['embedding_costs'])) {
                $totalEmbeddingCost = array_sum($GLOBALS['embedding_costs']);
                $totalOpenRouterCostOnError = ($totalOpenRouterCostOnError ?? 0) + $totalEmbeddingCost;
            }
            
            $stmt = $db->prepare("
                INSERT INTO widget_logs (widget_id, question, error_message, ip_address, user_agent, openrouter_cost)
                VALUES (?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $widget['id'],
                $question ?? '',
                $errorMessage,
                getClientIP(),
                getUserAgent(),
                $totalOpenRouterCostOnError > 0 ? $totalOpenRouterCostOnError : null
            ]);
        } catch (Exception $logError) {
            // Если не удалось залогировать - просто пропускаем
            logError("Failed to log error to database: " . $logError->getMessage());
        }
    }
    
    // Определяем тип ошибки и возвращаем соответствующий код
    $errorMsg = $e->getMessage();
    
    if (strpos($errorMsg, '429') !== false || strpos($errorMsg, 'rate_limit') !== false) {
        // Rate limit ошибка
        http_response_code(429);
        echo json_encode([
            'error' => 'Слишком много запросов. Пожалуйста, подождите несколько минут и попробуйте снова.',
            'type' => 'rate_limit',
            'debug' => $debugMode ? [
                'message' => $errorMsg,
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ] : null
        ], JSON_UNESCAPED_UNICODE);
    } else {
        // Другие ошибки
        http_response_code(500);
        echo json_encode([
            'error' => 'Сервис временно недоступен',
            'debug' => $debugMode ? [
                'message' => $errorMsg,
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => substr($e->getTraceAsString(), 0, 2000) // Ограничиваем размер trace
            ] : null
        ], JSON_UNESCAPED_UNICODE);
    }
    } // Конец блока else для POST запроса
}

/**
 * Валидация релевантности извлеченных медицинских терминов к исходному вопросу
 */
function validateMedicalTermsRelevance($medicalTerms, $question) {
    if (empty($medicalTerms) || !is_array($medicalTerms)) {
        return false;
    }
    
    // Стоп-слова, которые не являются медицинскими терминами
    $stopWords = ['после', 'перед', 'во', 'в', 'на', 'с', 'для', 'от', 'до', 'по', 'из', 'к', 'у', 'о', 'об', 'при', 'про', 'за', 'под', 'над', 'между', 'среди', 'через', 'без', 'со', 'обо'];
    
    $questionWords = preg_split('/[\s,\-\.]+/u', mb_strtolower($question));
    // Фильтруем: оставляем только значимые слова (длина >= 3 и не стоп-слово)
    $questionWords = array_filter($questionWords, function($word) use ($stopWords) {
        return mb_strlen($word) >= 3 && !in_array($word, $stopWords);
    });
    
    if (empty($questionWords)) {
        return true; // Если вопрос слишком короткий или состоит только из стоп-слов, пропускаем валидацию
    }
    
    // Собираем все извлеченные термины
    $allExtractedTerms = [];
    if (isset($medicalTerms['symptoms']) && is_array($medicalTerms['symptoms'])) {
        $allExtractedTerms = array_merge($allExtractedTerms, $medicalTerms['symptoms']);
    }
    if (isset($medicalTerms['keywords']) && is_array($medicalTerms['keywords'])) {
        $allExtractedTerms = array_merge($allExtractedTerms, $medicalTerms['keywords']);
    }
    if (isset($medicalTerms['specializations']) && is_array($medicalTerms['specializations'])) {
        $allExtractedTerms = array_merge($allExtractedTerms, $medicalTerms['specializations']);
    }
    if (isset($medicalTerms['service_types']) && is_array($medicalTerms['service_types'])) {
        $allExtractedTerms = array_merge($allExtractedTerms, $medicalTerms['service_types']);
    }
    
    if (empty($allExtractedTerms)) {
        return false; // Нет извлеченных терминов - данные нерелевантны
    }
    
    $allExtractedTermsLower = array_map('mb_strtolower', $allExtractedTerms);
    $allExtractedText = implode(' ', $allExtractedTermsLower);
    
    // Проверяем, что хотя бы одно значимое слово из вопроса присутствует в извлеченных терминах
    foreach ($questionWords as $word) {
        if (mb_strlen($word) >= 3) {
            // Проверяем точное совпадение слова или его части в извлеченных терминах
            foreach ($allExtractedTermsLower as $term) {
                if (mb_strpos($term, $word) !== false || mb_strpos($word, $term) !== false) {
                    return true; // Найдено совпадение
                }
            }
        }
    }
    
    return false; // Нет совпадений - данные нерелевантны
}

/**
 * Извлечение медицинских терминов из вопроса пользователя (Этап 1)
 * С постоянным кешированием для одинаковых вопросов
 */
function extractMedicalTerms($question, $widgetId, $model = null, $customPrompt = null, $debugMode = false, &$debugInfo = null, $custom_api_url = null, $custom_api_key = null) {
    // Нормализуем вопрос для кеша (убираем лишние пробелы, приводим к нижнему регистру)
    $normalizedQuestion = mb_strtolower(trim($question));
    $cacheKey = md5($widgetId . '_' . $normalizedQuestion);
    $cacheDir = LOG_DIR . '/term_extraction_cache';
    $cacheFile = $cacheDir . '/' . $cacheKey . '.json';
    
    // Если указан кастомный API, используем его
    if (!empty($custom_api_url)) {
        try {
            $requestData = [
                'model' => $model ?? 'custom-model',
                'max_tokens' => 1000,
                'messages' => [
                    ['role' => 'user', 'content' => $customPrompt ?: "Ты - помощник для извлечения медицинских терминов из вопроса пользователя.\n\nВАЖНО: Верни ТОЛЬКО валидный JSON, без дополнительного текста до или после JSON.\n\nТвоя задача - максимально полно извлечь все релевантные медицинские термины из вопроса пользователя, включая:\n- Прямо упомянутые симптомы, заболевания, состояния\n- Связанные симптомы и состояния, которые могут быть связаны с упомянутыми\n- Все возможные специализации и специалистов, которые могут помочь\n- Типы услуг, диагностики и лечения, которые могут быть релевантны\n- Ключевые слова для поиска, включая синонимы, медицинские термины и разговорные названия\n\nДумай широко: если пользователь упоминает симптом, подумай о возможных причинах, связанных состояниях и специалистах, которые могут помочь.\n\nИз вопроса извлеки и верни в формате JSON:\n{\n  \"symptoms\": [массив всех симптомов, включая связанные и возможные],\n  \"specializations\": [массив всех релевантных специализаций и специалистов],\n  \"service_types\": [массив всех типов услуг, диагностики и лечения],\n  \"keywords\": [объединенный массив всех релевантных ключевых слов, включая синонимы и связанные термины]\n}\n\nВопрос пользователя: " . $question . "\n\nВерни ТОЛЬКО JSON объект, без объяснений и дополнительного текста."]
                ],
                'temperature' => 0.3
            ];
            
            $response = callCustomAPI($custom_api_url, $custom_api_key, $requestData);
            
            // Извлекаем JSON из ответа
            $answer = '';
            if (isset($response['choices'][0]['message']['content'])) {
                $answer = trim($response['choices'][0]['message']['content']);
            } elseif (isset($response['content'][0]['text'])) {
                $answer = trim($response['content'][0]['text']);
            }
            
            // Извлекаем JSON из markdown code block, если есть
            if (preg_match('/```json\s*(.*?)\s*```/s', $answer, $matches)) {
                $answer = trim($matches[1]);
            } elseif (preg_match('/```\s*(.*?)\s*```/s', $answer, $matches)) {
                $answer = trim($matches[1]);
            } else {
                // Пытаемся найти JSON объект в тексте (на случай, если модель добавила текст до/после JSON)
                // Ищем JSON объект, который начинается с { и содержит "symptoms"
                if (preg_match('/\{[\s\S]*?"symptoms"[\s\S]*?\}/s', $answer, $matches)) {
                    // Многострочный JSON с "symptoms"
                    $answer = trim($matches[0]);
                } elseif (preg_match('/\{[^{}]*"symptoms"[^{}]*\}/s', $answer, $matches)) {
                    // Однострочный JSON с "symptoms"
                    $answer = trim($matches[0]);
                } else {
                    // Если не нашли по "symptoms", ищем первый { и последний } в ответе
                    $firstBrace = strpos($answer, '{');
                    $lastBrace = strrpos($answer, '}');
                    if ($firstBrace !== false && $lastBrace !== false && $lastBrace > $firstBrace) {
                        $answer = substr($answer, $firstBrace, $lastBrace - $firstBrace + 1);
                    }
                }
            }
            
            $medicalTerms = json_decode($answer, true);
            if (!$medicalTerms || json_last_error() !== JSON_ERROR_NONE) {
                // Если не удалось распарсить, пытаемся найти JSON объект более агрессивно
                // Ищем первый { и последний } в ответе
                $firstBrace = strpos($answer, '{');
                $lastBrace = strrpos($answer, '}');
                if ($firstBrace !== false && $lastBrace !== false && $lastBrace > $firstBrace) {
                    $answer = substr($answer, $firstBrace, $lastBrace - $firstBrace + 1);
                    $medicalTerms = json_decode($answer, true);
                }
                
                if (!$medicalTerms || json_last_error() !== JSON_ERROR_NONE) {
                    logParser("WARNING: Failed to parse JSON from stage2 response. Raw answer: " . substr($answer, 0, 500));
                    throw new Exception('Invalid JSON response from custom API: ' . json_last_error_msg() . '. Response: ' . substr($answer, 0, 200));
                }
            }
            
            // Валидация перед сохранением в кеш
            if (!validateMedicalTermsRelevance($medicalTerms, $question)) {
                logParser("WARNING: Extracted medical terms are not relevant to question '" . substr($question, 0, 50) . "'. Not saving to cache.");
            } else {
                // Сохраняем в кеш
                if (!is_dir($cacheDir)) {
                    mkdir($cacheDir, 0755, true);
                }
                file_put_contents($cacheFile, json_encode([
                    'medical_terms' => $medicalTerms,
                    'question' => $question,
                    'widget_id' => $widgetId,
                    'timestamp' => time()
                ], JSON_UNESCAPED_UNICODE));
            }
            
            return [
                'medical_terms' => $medicalTerms,
                'cached' => false,
                'cache_timestamp' => time(),
                'provider' => 'custom_api'
            ];
        } catch (Exception $e) {
            logParser("Custom API extractMedicalTerms error: " . $e->getMessage());
            throw $e;
        }
    }
    
    // Проверяем файловый кеш
    if (file_exists($cacheFile)) {
        $cachedData = json_decode(file_get_contents($cacheFile), true);
        if ($cachedData && isset($cachedData['medical_terms']) && isset($cachedData['timestamp'])) {
            // ВАЛИДАЦИЯ: Проверяем релевантность данных из кеша
            // Используем сохраненный вопрос из кеша (если есть) или текущий вопрос для валидации
            $cachedQuestion = $cachedData['question'] ?? $question;
            
            // Проверяем, что сохраненный вопрос совпадает с текущим (нормализованный)
            if (isset($cachedData['question'])) {
                $cachedQuestionNormalized = mb_strtolower(trim($cachedData['question']));
                if ($cachedQuestionNormalized !== $normalizedQuestion) {
                    logParser("WARNING: Cached question '" . substr($cachedQuestion, 0, 50) . "' doesn't match current question '" . substr($question, 0, 50) . "'. Ignoring cache.");
                    @unlink($cacheFile);
                } else {
                    // Дополнительная валидация релевантности терминов
                    if (!validateMedicalTermsRelevance($cachedData['medical_terms'], $question)) {
                        logParser("WARNING: Cached medical terms are not relevant to question '" . substr($question, 0, 50) . "'. Ignoring cache.");
                        @unlink($cacheFile);
                    } else {
                        logParser("Three-stage search: Using cached medical terms for question: " . substr($question, 0, 50));
                        return [
                            'medical_terms' => $cachedData['medical_terms'],
                            'cached' => true,
                            'cache_timestamp' => $cachedData['timestamp']
                        ];
                    }
                }
            } else {
                // Старый формат кеша без сохраненного вопроса - используем валидацию релевантности
                logParser("DEBUG: Old cache format detected (no 'question' field). Validating relevance for question: " . substr($question, 0, 50));
                if (!validateMedicalTermsRelevance($cachedData['medical_terms'], $question)) {
                    logParser("WARNING: Cached medical terms are not relevant to question '" . substr($question, 0, 50) . "'. Ignoring cache.");
                    @unlink($cacheFile);
                } else {
                    logParser("Three-stage search: Using cached medical terms for question: " . substr($question, 0, 50));
                    return [
                        'medical_terms' => $cachedData['medical_terms'],
                        'cached' => true,
                        'cache_timestamp' => $cachedData['timestamp']
                    ];
                }
            }
        }
    }
    
    // Используем кастомный промпт если указан, иначе дефолтный
    if (!empty($customPrompt)) {
        $extractionPrompt = str_replace('{QUESTION}', $question, $customPrompt);
    } else {
        $extractionPrompt = "Ты - помощник для извлечения медицинских терминов из вопроса пользователя.

ВАЖНО: Верни ТОЛЬКО валидный JSON, без дополнительного текста до или после JSON.

Твоя задача - максимально полно извлечь все релевантные медицинские термины из вопроса пользователя, включая:
- Прямо упомянутые симптомы, заболевания, состояния
- Связанные симптомы и состояния, которые могут быть связаны с упомянутыми
- Все возможные специализации и специалистов, которые могут помочь
- Типы услуг, диагностики и лечения, которые могут быть релевантны
- Ключевые слова для поиска, включая синонимы, медицинские термины и разговорные названия

Думай широко: если пользователь упоминает симптом, подумай о возможных причинах, связанных состояниях и специалистах, которые могут помочь.

Из вопроса извлеки и верни в формате JSON:
{
  \"symptoms\": [массив всех симптомов, включая связанные и возможные],
  \"specializations\": [массив всех релевантных специализаций и специалистов],
  \"service_types\": [массив всех типов услуг, диагностики и лечения],
  \"keywords\": [объединенный массив всех релевантных ключевых слов, включая синонимы и связанные термины]
}

Вопрос пользователя: " . $question . "

Верни ТОЛЬКО JSON объект, без объяснений и дополнительного текста.";
    }

    try {
        $useOpenRouterFirst = defined('USE_OPENROUTER_FIRST') && USE_OPENROUTER_FIRST;
        $useYandexGPTFirst = defined('USE_YANDEXGPT_FIRST') && USE_YANDEXGPT_FIRST;
        $answer = null;
        $stopReason = null;
        
        // Определяем модель для использования
        $useModel = $model ?? OPENROUTER_MODEL;
        $isClaudeModel = (strpos($useModel, 'claude') !== false || strpos($useModel, 'anthropic') !== false);
        $isYandexGPT = (strpos($useModel, 'gpt://') === 0 || strpos($useModel, 'yandexgpt') !== false);
        
        // Если USE_YANDEXGPT_FIRST = true, принудительно используем Yandex GPT модель
        if ($useYandexGPTFirst && !$isYandexGPT) {
            $useModel = 'gpt://' . YANDEXGPT_FOLDER_ID . '/yandexgpt-lite/latest';
            $isYandexGPT = true;
            logParser("USE_YANDEXGPT_FIRST enabled, forcing Yandex GPT model: $useModel");
        }
        
        // Приоритет: Yandex GPT (если включен) → OpenRouter → Claude
        if ($isYandexGPT && $useYandexGPTFirst) {
            for ($attempt = 0; $attempt < YANDEXGPT_MAX_RETRIES; $attempt++) {
                try {
                    logParser("Extracting medical terms via Yandex GPT with model: $useModel");
                    // Для этапа 2 extractionPrompt уже содержит весь промпт с вопросом, передаем его как promptData
                    // question передаем пустым, т.к. он уже включен в extractionPrompt
                    $yandexDebugInfo = null;
                    if ($debugMode) {
                        $result = callYandexGPTAPI("", $extractionPrompt, "", $useModel, 2000, $yandexDebugInfo);
                    } else {
                        $result = callYandexGPTAPI("", $extractionPrompt, "", $useModel, 2000);
                    }
                    if (isset($result['medical_terms'])) {
                        // Валидация перед сохранением в кеш
                        if (!validateMedicalTermsRelevance($result['medical_terms'], $question)) {
                            logParser("WARNING: Extracted medical terms are not relevant to question '" . substr($question, 0, 50) . "'. Not saving to cache.");
                        } else {
                            // Сохраняем в кеш
                            if (!is_dir($cacheDir)) {
                                mkdir($cacheDir, 0755, true);
                            }
                            file_put_contents($cacheFile, json_encode([
                                'medical_terms' => $result['medical_terms'],
                                'question' => $question,
                                'widget_id' => $widgetId,
                                'timestamp' => time()
                            ], JSON_UNESCAPED_UNICODE));
                        }
                        
                        $returnData = [
                            'medical_terms' => $result['medical_terms'],
                            'cached' => false,
                            'cache_timestamp' => time(),
                            'provider' => 'yandexgpt'
                        ];
                        
                        // Добавляем debug информацию если включен режим отладки
                        if ($debugMode && $yandexDebugInfo !== null) {
                            $returnData['yandex_debug'] = $yandexDebugInfo;
                            if ($debugInfo !== null) {
                                $debugInfo = $yandexDebugInfo;
                            }
                        }
                        
                        return $returnData;
                    }
                } catch (Exception $e) {
                    $errorMsg = $e->getMessage();
                    logParser("Yandex GPT Stage2 error (attempt " . ($attempt + 1) . "): " . $errorMsg);
                    
                    // Если это ошибка rate limit или server error, retry
                    if (strpos($errorMsg, '429') !== false || 
                        strpos($errorMsg, '500') !== false || 
                        strpos($errorMsg, '502') !== false ||
                        strpos($errorMsg, '503') !== false) {
                        
                        if ($attempt < YANDEXGPT_MAX_RETRIES - 1) {
                            $delay = YANDEXGPT_RETRY_DELAYS[$attempt] ?? YANDEXGPT_RETRY_DELAYS[0];
                            sleep($delay);
                            continue;
                        }
                    }
                    
                    // Fallback на OpenRouter или Claude
                    logParser("Yandex GPT Stage2 failed, switching to fallback");
                    break;
                }
            }
        }
        
        // Пытаемся использовать OpenRouter, если модель не Claude и не Yandex GPT и включен
        if (!$isYandexGPT && !$isClaudeModel && $useOpenRouterFirst) {
            try {
                logParser("Extracting medical terms via OpenRouter with model: $useModel");
                // Используем указанную модель
                $data = [
                    'model' => $useModel,
                    'max_tokens' => 4000, // Увеличено для полного JSON ответа
                    'messages' => [
                        [
                            'role' => 'user',
                            'content' => $extractionPrompt
                        ]
                    ],
                    'temperature' => 0.3, // Низкая температура для более точного извлечения
                    'response_format' => ['type' => 'json_object'] // Принудительный JSON формат (если поддерживается моделью)
                ];
                
                $ch = curl_init(OPENROUTER_API_URL);
                configureOpenRouterProxy($ch); // Настройка прокси для OpenRouter
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
                curl_setopt($ch, CURLOPT_TIMEOUT, 30); // Увеличено до 30 секунд для больших моделей
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Content-Type: application/json',
                    'Authorization: Bearer ' . OPENROUTER_API_KEY,
                    'HTTP-Referer: ' . WIDGET_DOMAIN,
                    'X-Title: AI Widget',
                    'X-Provider: deepinfra/fp8' // Принудительно использовать только deepinfra/fp8
                ]);
                
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                
                if (curl_errno($ch)) {
                    curl_close($ch);
                    throw new Exception('OpenRouter API error: ' . curl_error($ch));
                }
                
                curl_close($ch);
                
                if ($httpCode !== 200) {
                    throw new Exception("OpenRouter API returned HTTP $httpCode: $response");
                }
                
                $result = json_decode($response, true);
                
                if (!isset($result['choices'][0]['message']['content'])) {
                    throw new Exception('Invalid OpenRouter API response');
                }
                
                $answer = trim($result['choices'][0]['message']['content']);
                $finishReason = $result['choices'][0]['finish_reason'] ?? null;
                
                // Логируем исходный ответ для отладки (первые 200 символов)
                $answerPreview = mb_substr($answer, 0, 200);
                if (mb_strpos($answer, '{') !== 0) {
                    logParser("Stage 2 (OpenRouter): Model added text before JSON. First 200 chars: " . $answerPreview);
                }
                
                // Извлекаем JSON из markdown code block или текста, если модель добавила лишний текст
                if (preg_match('/```json\s*(.*?)\s*```/s', $answer, $matches)) {
                    $answer = trim($matches[1]);
                    logParser("Stage 2 (OpenRouter): Extracted JSON from markdown code block");
                } elseif (preg_match('/```\s*(.*?)\s*```/s', $answer, $matches)) {
                    $answer = trim($matches[1]);
                    logParser("Stage 2 (OpenRouter): Extracted JSON from code block");
                } elseif (preg_match('/\{[\s\S]*?"symptoms"[\s\S]*?\}/s', $answer, $matches)) {
                    $answer = trim($matches[0]);
                    logParser("Stage 2 (OpenRouter): Extracted JSON by symptoms pattern");
                } else {
                    // Ищем первый { и последний } в ответе
                    $firstBrace = strpos($answer, '{');
                    $lastBrace = strrpos($answer, '}');
                    if ($firstBrace !== false && $lastBrace !== false && $lastBrace > $firstBrace) {
                        $answer = substr($answer, $firstBrace, $lastBrace - $firstBrace + 1);
                        logParser("Stage 2 (OpenRouter): Extracted JSON by first/last brace");
                    }
                }
                
                // Извлекаем стоимость из ответа OpenRouter для Stage 2
                $stage2Cost = null;
                if (isset($result['usage']['cost'])) {
                    $stage2Cost = floatval($result['usage']['cost']);
                } elseif (isset($result['cost'])) {
                    $stage2Cost = floatval($result['cost']);
                } elseif (isset($result['usage']['total_cost'])) {
                    $stage2Cost = floatval($result['usage']['total_cost']);
                }
                
                if ($stage2Cost !== null) {
                    // Сохраняем стоимость для последующего суммирования
                    if (!isset($GLOBALS['openrouter_costs'])) {
                        $GLOBALS['openrouter_costs'] = [];
                    }
                    $GLOBALS['openrouter_costs'][] = ['stage' => 'stage2', 'cost' => $stage2Cost];
                    logParser("Stage 2 (OpenRouter) cost: $" . number_format($stage2Cost, 6));
                }
                
                // Проверяем, не обрезан ли ответ
                if ($finishReason === 'length') {
                    logParser("Warning: OpenRouter response was truncated due to max_tokens limit. Consider increasing max_tokens.");
                    // Не бросаем исключение, пытаемся восстановить JSON ниже
                }
                
                // Парсим JSON из ответа
                $medicalTerms = json_decode($answer, true);
                if (!$medicalTerms || json_last_error() !== JSON_ERROR_NONE) {
                    // Если не удалось распарсить, пытаемся найти JSON объект более агрессивно
                    $firstBrace = strpos($answer, '{');
                    $lastBrace = strrpos($answer, '}');
                    if ($firstBrace !== false && $lastBrace !== false && $lastBrace > $firstBrace) {
                        $answer = substr($answer, $firstBrace, $lastBrace - $firstBrace + 1);
                        $medicalTerms = json_decode($answer, true);
                    }
                    if (!$medicalTerms || json_last_error() !== JSON_ERROR_NONE) {
                        throw new Exception('Invalid JSON response from OpenRouter: ' . json_last_error_msg() . '. Response: ' . substr($answer, 0, 200));
                    }
                }
                
                $answer = json_encode($medicalTerms); // Нормализуем для дальнейшей обработки
                logParser("Medical terms extracted successfully via OpenRouter");
            } catch (Exception $e) {
                logParser("OpenRouter failed for medical terms extraction: " . $e->getMessage() . ". Falling back to Claude.");
                // Продолжаем к Claude fallback
            }
        }
        
        // Fallback на Claude API, если OpenRouter не использовался или произошла ошибка
        if ($answer === null) {
            // Если модель была Yandex GPT, используем модель Claude по умолчанию
            $claudeModel = ($isYandexGPT || $isClaudeModel) ? ($isClaudeModel ? $useModel : CLAUDE_MODEL) : CLAUDE_MODEL;
            logParser("Extracting medical terms via Claude API with model: $claudeModel");
            $data = [
                'model' => $claudeModel,
                'max_tokens' => 4000, // Увеличено для полного JSON ответа
                'messages' => [
                    [
                        'role' => 'user',
                        'content' => $extractionPrompt
                    ]
                ]
            ];
            
            $ch = curl_init(CLAUDE_API_URL);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_TIMEOUT, 30); // Увеличено до 30 секунд для больших моделей
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'x-api-key: ' . CLAUDE_API_KEY,
                'anthropic-version: 2023-06-01'
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            
            if (curl_errno($ch)) {
                curl_close($ch);
                throw new Exception('Claude API error: ' . curl_error($ch));
            }
            
            curl_close($ch);
            
            if ($httpCode !== 200) {
                throw new Exception("Claude API returned HTTP $httpCode: $response");
            }
            
            $result = json_decode($response, true);
            
            if (!isset($result['content'][0]['text'])) {
                throw new Exception('Invalid Claude API response');
            }
            
            $answer = $result['content'][0]['text'];
            $stopReason = $result['stop_reason'] ?? null;
            logParser("Medical terms extracted successfully via Claude");
        }
        
        // Проверяем, не обрезан ли ответ (только для Claude, т.к. у OpenRouter другой формат)
        if ($stopReason && $stopReason !== 'end_turn') {
            logParser("Warning: Claude response was truncated. Stop reason: " . $stopReason);
            if ($stopReason === 'max_tokens') {
                throw new Exception('Claude response was truncated due to max_tokens limit. Consider increasing max_tokens.');
            }
        }
        
        // Извлекаем JSON из markdown code block, если есть
        if (preg_match('/```json\s*(.*?)\s*```/s', $answer, $matches)) {
            $answer = trim($matches[1]);
        } elseif (preg_match('/```\s*(.*?)\s*```/s', $answer, $matches)) {
            $answer = trim($matches[1]);
        } else {
            // Пытаемся найти JSON объект в тексте (на случай, если модель добавила текст до/после JSON)
            if (preg_match('/\{[\s\S]*?"symptoms"[\s\S]*?\}/s', $answer, $matches)) {
                $answer = trim($matches[0]);
            } else {
                // Ищем первый { и последний } в ответе
                $firstBrace = strpos($answer, '{');
                $lastBrace = strrpos($answer, '}');
                if ($firstBrace !== false && $lastBrace !== false && $lastBrace > $firstBrace) {
                    $answer = substr($answer, $firstBrace, $lastBrace - $firstBrace + 1);
                }
            }
        }
        
        // Удаляем управляющие символы
        $answer = preg_replace('/[\x00-\x1F\x7F]/u', '', $answer);
        $answer = trim($answer);
        
        // Проверяем, не обрезан ли JSON (если ответ не заканчивается на })
        if (!empty($answer) && !preg_match('/\}\s*$/', $answer)) {
            logParser("Warning: Response appears to be truncated. Last 100 chars: " . substr($answer, -100));
            // Пытаемся восстановить JSON, добавив закрывающую скобку
            if (preg_match('/\{.*/s', $answer, $matches)) {
                $answer = $matches[0];
                // Подсчитываем открывающие и закрывающие скобки
                $openBraces = substr_count($answer, '{');
                $closeBraces = substr_count($answer, '}');
                if ($openBraces > $closeBraces) {
                    $answer .= str_repeat('}', $openBraces - $closeBraces);
                }
            }
        }
        
        $medicalTerms = json_decode($answer, true, 512, JSON_INVALID_UTF8_IGNORE);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            $errorMsg = 'JSON decode error: ' . json_last_error_msg();
            $errorMsg .= '. Response length: ' . strlen($answer);
            $errorMsg .= '. Response preview: ' . substr($answer, 0, 300);
            $errorMsg .= '. Stop reason: ' . ($stopReason ?? 'unknown');
            throw new Exception($errorMsg);
        }
        
        // Убеждаемся что есть массив keywords
        if (!isset($medicalTerms['keywords']) || !is_array($medicalTerms['keywords'])) {
            $medicalTerms['keywords'] = [];
        }
        
        // Объединяем все массивы в keywords если их нет
        if (empty($medicalTerms['keywords'])) {
            $allKeywords = [];
            if (isset($medicalTerms['symptoms']) && is_array($medicalTerms['symptoms'])) {
                $allKeywords = array_merge($allKeywords, $medicalTerms['symptoms']);
            }
            if (isset($medicalTerms['specializations']) && is_array($medicalTerms['specializations'])) {
                $allKeywords = array_merge($allKeywords, $medicalTerms['specializations']);
            }
            if (isset($medicalTerms['service_types']) && is_array($medicalTerms['service_types'])) {
                $allKeywords = array_merge($allKeywords, $medicalTerms['service_types']);
            }
            $medicalTerms['keywords'] = array_unique($allKeywords);
        }
        
        // Валидация перед сохранением в кеш
        if (!validateMedicalTermsRelevance($medicalTerms, $question)) {
            logParser("WARNING: Extracted medical terms are not relevant to question '" . substr($question, 0, 50) . "'. Not saving to cache.");
        } else {
            // Сохраняем в файловый кеш для постоянного хранения
            if (!is_dir($cacheDir)) {
                mkdir($cacheDir, 0755, true);
            }
            $timestamp = time();
            file_put_contents($cacheFile, json_encode([
                'medical_terms' => $medicalTerms,
                'question' => $question,
                'widget_id' => $widgetId,
                'timestamp' => $timestamp
            ], JSON_UNESCAPED_UNICODE));
        }
        
        return [
            'medical_terms' => $medicalTerms,
            'cached' => false,
            'cache_timestamp' => $timestamp
        ];
        
    } catch (Exception $e) {
        logParser("Failed to extract medical terms: " . $e->getMessage());
        throw $e;
    }
}

/**
 * Определение релевантных категорий для всех разделов одним запросом
 * Использует AI fallback для разделов, где категории не найдены в БД
 */
function getRelevantCategoriesForAllSections($db, $widgetId, $keywords, $question, $sectionNames) {
    if (empty($sectionNames) || empty($question)) {
        return [];
    }
    
    $result = [];
    
    try {
        // Получаем все категории для всех разделов
        $allCategoriesBySection = [];
        foreach ($sectionNames as $sectionName) {
            $stmt = $db->prepare("
                SELECT category_url, category_name, section_name
                FROM category_keywords
                WHERE widget_id = ? AND section_name = ?
                ORDER BY category_name
                LIMIT 50
            ");
            $stmt->execute([$widgetId, $sectionName]);
            $categories = $stmt->fetchAll();
            
            if (!empty($categories)) {
                $allCategoriesBySection[$sectionName] = $categories;
            }
        }
        
        if (empty($allCategoriesBySection)) {
            return [];
        }
        
        // Формируем промпт для всех разделов сразу
        $prompt = "Определи, к каким категориям медицинских услуг относится вопрос пользователя для каждого раздела.\n\n";
        $prompt .= "Вопрос: {$question}\n";
        $prompt .= "Ключевые слова: " . implode(", ", $keywords) . "\n\n";
        
        foreach ($allCategoriesBySection as $sectionName => $categories) {
            $prompt .= "Раздел: {$sectionName}\n";
            $prompt .= "Доступные категории:\n";
            foreach ($categories as $cat) {
                $prompt .= "- {$cat['category_name']} ({$cat['category_url']})\n";
            }
            $prompt .= "\n";
        }
        
        $prompt .= "Верни JSON в формате:\n";
        $prompt .= "{\n";
        foreach ($sectionNames as $sectionName) {
            $prompt .= "  \"{$sectionName}\": [\"category_url1\", \"category_url2\", ...],\n";
        }
        $prompt .= "}\n\n";
        $prompt .= "Если для раздела нет релевантных категорий, верни пустой массив для этого раздела.";
        
        // Отправляем один запрос в Gemini через OpenRouter
        $data = [
            'model' => GEMINI_MODEL,
            'max_tokens' => 2000, // Увеличено для нескольких разделов
            'messages' => [
                [
                    'role' => 'user',
                    'content' => $prompt
                ]
            ],
            'temperature' => 0.3
        ];
        
        $ch = curl_init(GEMINI_API_URL);
        configureOpenRouterProxy($ch); // Настройка прокси для OpenRouter
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_TIMEOUT, 20);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . OPENROUTER_API_KEY,
            'HTTP-Referer: ' . WIDGET_DOMAIN,
            'X-Title: AI Widget'
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if (curl_errno($ch)) {
            curl_close($ch);
            logParser("AI fallback failed for category detection (all sections): " . curl_error($ch));
            return [];
        }
        
        curl_close($ch);
        
        if ($httpCode === 200) {
            $apiResult = json_decode($response, true);
            
            if (isset($apiResult['choices'][0]['message']['content'])) {
                $answer = trim($apiResult['choices'][0]['message']['content']);
                
                // Извлекаем JSON из ответа
                if (preg_match('/```json\s*(.*?)\s*```/s', $answer, $matches)) {
                    $answer = $matches[1];
                } elseif (preg_match('/```\s*(.*?)\s*```/s', $answer, $matches)) {
                    $answer = $matches[1];
                }
                
                $categoriesData = json_decode($answer, true);
                
                if ($categoriesData && is_array($categoriesData)) {
                    // Возвращаем категории для каждого раздела
                    foreach ($sectionNames as $sectionName) {
                        if (isset($categoriesData[$sectionName]) && is_array($categoriesData[$sectionName])) {
                            $result[$sectionName] = $categoriesData[$sectionName];
                        }
                    }
                    return $result;
                }
            }
        }
        
        logParser("AI fallback returned invalid response for category detection (all sections)");
    } catch (Exception $e) {
        logParser("AI fallback error for category detection (all sections): " . $e->getMessage());
    }
    
    return [];
}

/**
 * Определение релевантных категорий по ключевым словам
 * Использует маппинг из БД, если не найдено - AI fallback (если $useAIFallback = true)
 */
function getRelevantCategories($db, $widgetId, $keywords, $question = '', $sectionName = 'services', $useAIFallback = true) {
    if (empty($keywords) || !is_array($keywords)) {
        return [];
    }
    
    $categories = [];
    
    // Быстрый путь: поиск в таблице category_keywords
    // Проверяем, есть ли маппинг для этого виджета и раздела
    $stmt = $db->prepare("SELECT COUNT(*) as count FROM category_keywords WHERE widget_id = ? AND section_name = ?");
    $stmt->execute([$widgetId, $sectionName]);
    $hasMapping = $stmt->fetch()['count'] > 0;
    
    if ($hasMapping) {
        // Ищем категории по ключевым словам
        // Используем JSON_SEARCH для поиска в массиве keywords
        $foundCategories = [];
        foreach ($keywords as $keyword) {
            $keywordLower = mb_strtolower($keyword);
            
            // Ищем категории, где keywords содержит это ключевое слово
            $stmt = $db->prepare("
                SELECT DISTINCT category_url, category_name
                FROM category_keywords
                WHERE widget_id = ? AND section_name = ?
                  AND JSON_SEARCH(keywords, 'one', ?, NULL, '$[*]') IS NOT NULL
            ");
            $stmt->execute([$widgetId, $sectionName, $keywordLower]);
            
            while ($row = $stmt->fetch()) {
                if (!in_array($row['category_url'], $foundCategories)) {
                    $foundCategories[$row['category_url']] = $row['category_name'];
                }
            }
        }
        
        if (!empty($foundCategories)) {
            return array_keys($foundCategories);
        }
    }
    
    // Fallback: используем AI для определения категорий в реальном времени (только если $useAIFallback = true)
    if (!empty($question) && $useAIFallback) {
        try {
            // Получаем все категории для этого виджета и раздела из category_keywords
            $stmt = $db->prepare("
                SELECT category_url, category_name
                FROM category_keywords
                WHERE widget_id = ? AND section_name = ?
                ORDER BY category_name
                LIMIT 50
            ");
            $stmt->execute([$widgetId, $sectionName]);
            $allCategories = $stmt->fetchAll();
            
            if (empty($allCategories)) {
                return [];
            }
            
            // Формируем список категорий для промпта
            $categoriesList = array_map(function($cat) {
                return '- ' . $cat['category_name'] . ' (' . $cat['category_url'] . ')';
            }, array_slice($allCategories, 0, 50));
            
            // Формируем промпт для AI
            $prompt = "Определи, к каким категориям медицинских услуг относится вопрос пользователя.

Вопрос: {$question}
Ключевые слова: " . implode(", ", $keywords) . "

Доступные категории:
" . implode("\n", $categoriesList) . "

Верни JSON с массивом URL категорий (только parent_url из списка выше):
{\"categories\": [\"category_url1\", \"category_url2\", ...]}

Если вопрос не относится ни к одной категории, верни пустой массив.";
            
            // Отправляем запрос в Gemini через OpenRouter
            $data = [
                'model' => GEMINI_MODEL,
                'max_tokens' => 1000,
                'messages' => [
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'temperature' => 0.3
            ];
            
            $ch = curl_init(GEMINI_API_URL);
            configureOpenRouterProxy($ch); // Настройка прокси для OpenRouter
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_TIMEOUT, 15);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Authorization: Bearer ' . OPENROUTER_API_KEY,
                'HTTP-Referer: ' . WIDGET_DOMAIN,
                'X-Title: AI Widget'
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            
            if (curl_errno($ch)) {
                curl_close($ch);
                logParser("AI fallback failed for category detection: " . curl_error($ch));
                return [];
            }
            
            curl_close($ch);
            
            if ($httpCode === 200) {
                $result = json_decode($response, true);
                
                if (isset($result['choices'][0]['message']['content'])) {
                    $answer = trim($result['choices'][0]['message']['content']);
                    
                    // Извлекаем JSON из ответа
                    if (preg_match('/```json\s*(.*?)\s*```/s', $answer, $matches)) {
                        $answer = $matches[1];
                    } elseif (preg_match('/```\s*(.*?)\s*```/s', $answer, $matches)) {
                        $answer = $matches[1];
                    }
                    
                    $categoriesData = json_decode($answer, true);
                    
                    if ($categoriesData && isset($categoriesData['categories']) && is_array($categoriesData['categories'])) {
                        return $categoriesData['categories'];
                    }
                }
            }
            
            logParser("AI fallback returned invalid response for category detection");
        } catch (Exception $e) {
            logParser("AI fallback error for category detection: " . $e->getMessage());
        }
    }
    
    return [];
}

/**
 * Поиск релевантных записей по ключевым словам (Этап 2)
 * $categoriesBySection - массив ['section_name' => ['category_url1', 'category_url2', ...]]
 * $specializations - массив специализаций для расширенного поиска
 * $serviceTypes - массив типов услуг для расширенного поиска
 */
function searchByKeywords($db, $widget, $keywords, $categoriesBySection = [], $specializations = [], $serviceTypes = []) {
    if (empty($keywords) || !is_array($keywords)) {
        return [];
    }
    
    // Извлекаем отдельные слова из фраз для более гибкого поиска
    $searchWords = [];
    foreach ($keywords as $keyword) {
        $keyword = trim($keyword);
        if (empty($keyword)) continue;
        
        // Разбиваем фразу на слова (убираем стоп-слова)
        $words = preg_split('/[\s,\-\.]+/u', mb_strtolower($keyword));
        $stopWords = ['в', 'на', 'с', 'для', 'от', 'до', 'по', 'из', 'к', 'у', 'о', 'об', 'при', 'про', 'за', 'под', 'над', 'между', 'среди', 'через', 'без', 'после', 'перед', 'во', 'со', 'обо', 'об', 'обо', 'обо', 'обо'];
        
        foreach ($words as $word) {
            $word = trim($word);
            // Оставляем только значимые слова (длина >= 3 символов и не стоп-слово)
            if (mb_strlen($word) >= 3 && !in_array($word, $stopWords)) {
                $searchWords[] = $word;
            }
        }
        
        // Также добавляем оригинальную фразу для точных совпадений
        if (mb_strlen($keyword) >= 3) {
            $searchWords[] = $keyword;
        }
    }
    
    // Добавляем специализации и типы услуг для расширенного поиска
    // Это позволяет находить услуги типа "Консультация терапевта" даже если в запросе нет слова "терапевт"
    foreach ($specializations as $spec) {
        $spec = trim(mb_strtolower($spec));
        if (mb_strlen($spec) >= 3) {
            $searchWords[] = $spec;
            // Также разбиваем на слова (например, "детский травматолог" -> "детский", "травматолог")
            $specWords = preg_split('/[\s,\-\.]+/u', $spec);
            foreach ($specWords as $word) {
                $word = trim($word);
                if (mb_strlen($word) >= 3 && !in_array($word, $stopWords)) {
                    $searchWords[] = $word;
                }
            }
        }
    }
    
    foreach ($serviceTypes as $serviceType) {
        $serviceType = trim(mb_strtolower($serviceType));
        if (mb_strlen($serviceType) >= 3) {
            $searchWords[] = $serviceType;
            // Также разбиваем на слова (например, "консультация невролога" -> "консультация", "невролога")
            $serviceWords = preg_split('/[\s,\-\.]+/u', $serviceType);
            foreach ($serviceWords as $word) {
                $word = trim($word);
                if (mb_strlen($word) >= 3 && !in_array($word, $stopWords)) {
                    $searchWords[] = $word;
                }
            }
        }
    }
    
    // Убираем дубликаты
    $searchWords = array_unique($searchWords);
    
    if (empty($searchWords)) {
        return [];
    }
    
    // Получаем разделы и поля для поиска
    $stmt = $db->prepare("
        SELECT DISTINCT ws.section_name, sf.field_name
        FROM widget_sections ws
        JOIN section_fields sf ON ws.id = sf.section_id
        WHERE ws.widget_id = ? AND ws.is_active = 1 AND sf.use_in_prompt = 1
        
        UNION
        
        SELECT DISTINCT ws.section_name, scf.child_field_name as field_name
        FROM widget_sections ws
        JOIN section_fields sf ON ws.id = sf.section_id
        JOIN section_child_fields scf ON sf.id = scf.field_id
        WHERE ws.widget_id = ? AND ws.is_active = 1 AND scf.use_in_prompt = 1
        
        ORDER BY section_name
    ");
    $stmt->execute([$widget['id'], $widget['id']]);
    
    $fieldsBySection = [];
    while ($row = $stmt->fetch()) {
        $fieldsBySection[$row['section_name']][] = $row['field_name'];
    }
    
    $relevantIds = [];
    
    // Для каждого раздела ищем записи содержащие ключевые слова
    foreach ($fieldsBySection as $section => $fields) {
        if (empty($fields)) continue;
        
        // Шаг 1: Предварительная фильтрация по категориям (если есть)
        $candidateIds = null;
        if (!empty($categoriesBySection) && isset($categoriesBySection[$section]) && !empty($categoriesBySection[$section])) {
            // Получаем ID записей из релевантных категорий
            $candidateIds = getItemsByCategories($db, $widget['id'], $section, $categoriesBySection[$section]);
            
            // Если категории найдены, но записей в них нет - пропускаем раздел
            if (empty($candidateIds)) {
                continue;
            }
        }
        
        // Формируем условия LIKE для каждого поискового слова
        $likeConditions = [];
        $likeParams = [];
        
        foreach ($searchWords as $word) {
            // Используем LIKE '%word%' для учета склонений и частичных совпадений
            // MySQL с utf8mb4_unicode_ci уже делает регистронезависимое сравнение
            $likeConditions[] = "pf.field_value LIKE ?";
            $likeParams[] = '%' . mb_strtolower($word) . '%';
        }
        
        if (empty($likeConditions)) continue;
        
        // Формируем параметры в правильном порядке: widget_id, section_name, fields, like conditions
        $params = array_merge(
            [$widget['id'], $section],
            $fields,
            $likeParams
        );
        
        // Добавляем фильтр по кандидатам (если есть предварительная фильтрация по категориям)
        $candidateFilter = '';
        if ($candidateIds !== null && !empty($candidateIds)) {
            $placeholders = str_repeat('?,', count($candidateIds) - 1) . '?';
            $candidateFilter = " AND pi.id IN ($placeholders)";
            $params = array_merge($params, $candidateIds);
        }
        
        $stmt = $db->prepare("
            SELECT DISTINCT pi.id
            FROM parsed_items pi
            JOIN parsed_fields pf ON pi.id = pf.item_id
            WHERE pi.widget_id = ? 
              AND pi.section_name = ?
              AND pf.field_name IN (" . str_repeat('?,', count($fields) - 1) . "?)
              AND pi.is_duplicate = 0
              AND (" . implode(' OR ', $likeConditions) . ")
              $candidateFilter
        ");
        
        $stmt->execute($params);
        
        $sectionIds = [];
        while ($row = $stmt->fetch()) {
            $sectionIds[] = (int)$row['id'];
        }
        
        if (!empty($sectionIds)) {
            $relevantIds[$section] = array_unique($sectionIds);
        }
    }
    
    return $relevantIds;
}

/**
 * Получить ID записей по категориям
 * 
 * @param PDO $db Подключение к БД
 * @param int $widgetId ID виджета
 * @param string $sectionName Название раздела
 * @param array $categoryUrls Массив URL категорий
 * @return array Массив ID записей
 */
function getItemsByCategories($db, $widgetId, $sectionName, $categoryUrls) {
    if (empty($categoryUrls) || !is_array($categoryUrls)) {
        return [];
    }
    
    // Получаем ID записей, у которых category_id соответствует переданным категориям
    $placeholders = str_repeat('?,', count($categoryUrls) - 1) . '?';
    $stmt = $db->prepare("
        SELECT DISTINCT pi.id
        FROM parsed_items pi
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.widget_id = ?
          AND pi.section_name = ?
          AND pi.is_duplicate = 0
          AND pf.field_name = 'category_id'
          AND pf.field_value IN ($placeholders)
    ");
    $params = array_merge([$widgetId, $sectionName], $categoryUrls);
    $stmt->execute($params);
    
    $ids = [];
    while ($row = $stmt->fetch()) {
        $ids[] = (int)$row['id'];
    }
    
    return $ids;
}

/**
 * Получить все ID записей раздела
 * 
 * @param PDO $db Подключение к БД
 * @param int $widgetId ID виджета
 * @param string $sectionName Название раздела
 * @return array Массив ID записей
 */
function getAllItemIds($db, $widgetId, $sectionName) {
    $stmt = $db->prepare("
        SELECT id
        FROM parsed_items
        WHERE widget_id = ?
          AND section_name = ?
          AND is_duplicate = 0
    ");
    $stmt->execute([$widgetId, $sectionName]);
    
    $ids = [];
    while ($row = $stmt->fetch()) {
        $ids[] = (int)$row['id'];
    }
    
    return $ids;
}

/**
 * Поиск по Embedding и категориям (комбинированный подход)
 * 
 * @param PDO $db Подключение к БД
 * @param array $widget Данные виджета
 * @param string $question Вопрос пользователя
 * @param array $categoriesBySection Категории по разделам: ['section_name' => ['category_url1', ...], ...]
 * @param array $keywords Ключевые слова из этапа 2
 * @param array $specializations Специализации из этапа 2
 * @param array $serviceTypes Типы услуг из этапа 2
 * @return array Результаты по разделам: ['section_name' => [id1, id2, ...], ...]
 */
function searchByEmbeddingAndCategories($db, $widget, $question, $categoriesBySection = [], $keywords = [], $specializations = [], $serviceTypes = [], &$debugInfo = null) {
    if (empty($question)) {
        // Fallback на обычный поиск, если вопроса нет
        return searchByKeywords($db, $widget, $keywords, $categoriesBySection, $specializations, $serviceTypes);
    }
    
    // Шаг 1: Создать embedding для вопроса пользователя
    $embeddingStartTime = microtime(true);
    $questionEmbedding = createQuestionEmbedding($question);
    $embeddingTime = round((microtime(true) - $embeddingStartTime) * 1000);
    
    if (empty($questionEmbedding)) {
        // Если не удалось создать embedding, fallback на обычный поиск
        logParser("Failed to create question embedding, falling back to keyword search");
        return searchByKeywords($db, $widget, $keywords, $categoriesBySection, $specializations, $serviceTypes);
    }
    
    // Сохраняем информацию для debug
    if ($debugInfo !== null) {
        $debugInfo['embedding_search'] = [
            'model' => EMBEDDING_MODEL,
            'embedding_time_ms' => $embeddingTime,
            'candidates_by_section' => [],
            'results_by_section' => []
        ];
    }
    
    // Шаг 2: Получить список разделов виджета
    $stmt = $db->prepare("SELECT DISTINCT section_name FROM widget_sections WHERE widget_id = ? AND is_active = 1");
    $stmt->execute([$widget['id']]);
    $sections = $stmt->fetchAll();
    
    $relevantIds = [];
    // Для services увеличиваем лимит, так как там больше записей
    $baseLimit = defined('EMBEDDING_SEARCH_LIMIT') ? EMBEDDING_SEARCH_LIMIT : 20;
    $limit = $baseLimit;
    
    // Шаг 3: Для каждого раздела получить кандидатов и применить embedding поиск
    foreach ($sections as $sectionRow) {
        $sectionName = $sectionRow['section_name'];
        
        // Для services и specialists увеличиваем лимит результатов (там больше записей)
        if ($sectionName === 'services') {
            $limit = max($baseLimit * 2, 40);
        } elseif ($sectionName === 'specialists') {
            // Для specialists увеличиваем лимит, чтобы находить больше специалистов (особенно узких специализаций)
            $limit = max($baseLimit * 2, 30);
        } else {
            $limit = $baseLimit;
        }
        
        // Получаем кандидатов: сначала по категориям (если есть), иначе все записи раздела
        $candidateIds = [];
        if (!empty($categoriesBySection) && isset($categoriesBySection[$sectionName]) && !empty($categoriesBySection[$sectionName])) {
            // Фильтруем по категориям
            $candidateIds = getItemsByCategories($db, $widget['id'], $sectionName, $categoriesBySection[$sectionName]);
            
            // Для specialists: если категорий мало (< 3) или кандидатов мало (< 10), расширяем поиск
            if ($sectionName === 'specialists') {
                $categoryCount = count($categoriesBySection[$sectionName]);
                if ($categoryCount < 3 || count($candidateIds) < 10) {
                    // Добавляем дополнительные кандидаты через keyword search
                    if (!empty($keywords)) {
                        $additionalIds = searchByKeywords($db, $widget, $keywords, [], $specializations, $serviceTypes);
                        if (isset($additionalIds[$sectionName]) && !empty($additionalIds[$sectionName])) {
                            // Объединяем с существующими кандидатами, убирая дубликаты
                            $candidateIds = array_values(array_unique(array_merge($candidateIds, $additionalIds[$sectionName])));
                            logParser("Expanded specialists candidates: " . count($candidateIds) . " (added " . (count($candidateIds) - count($candidateIds)) . " from keyword search)");
                        }
                    }
                    
                    // Если все еще мало кандидатов, добавляем всех специалистов из раздела
                    if (count($candidateIds) < 10) {
                        $allSpecialists = getAllItemIds($db, $widget['id'], $sectionName);
                        $candidateIds = array_values(array_unique(array_merge($candidateIds, $allSpecialists)));
                        logParser("Expanded specialists candidates to all specialists: " . count($candidateIds) . " total");
                    }
                }
            }
        }
        
        // Если кандидатов по категориям нет или слишком много, используем предварительную фильтрацию
        // Для specialists используем более высокий порог (500 вместо 1000), чтобы не ограничивать слишком сильно
        $maxCandidatesThreshold = ($sectionName === 'specialists') ? 500 : 1000;
        
        if (empty($candidateIds) || count($candidateIds) > $maxCandidatesThreshold) {
            // Если кандидатов слишком много, используем предварительную фильтрацию по ключевым словам
            // НО для specialists, если категорий мало (< 10), не используем предварительную фильтрацию
            $usePreFilter = true;
            if ($sectionName === 'specialists' && !empty($categoriesBySection) && isset($categoriesBySection[$sectionName])) {
                $categoryCount = count($categoriesBySection[$sectionName]);
                if ($categoryCount < 10 && count($candidateIds) <= 100) {
                    // Если категорий мало и кандидатов не слишком много, не используем предварительную фильтрацию
                    $usePreFilter = false;
                }
            }
            
            if ($usePreFilter && !empty($keywords)) {
                // Получаем ID записей, содержащих ключевые слова (быстрая предварительная фильтрация)
                $preFilteredIds = searchByKeywords($db, $widget, $keywords, [], $specializations, $serviceTypes);
                if (isset($preFilteredIds[$sectionName]) && !empty($preFilteredIds[$sectionName])) {
                    // Для specialists проверяем, что найдено достаточно специалистов (минимум 10)
                    if ($sectionName === 'specialists' && count($preFilteredIds[$sectionName]) < 10) {
                        // Если найдено мало специалистов, используем все кандидаты из категорий (если есть)
                        if (!empty($candidateIds) && count($candidateIds) <= 100) {
                            // Используем кандидатов из категорий
                        } else {
                            // Используем все записи раздела
                            $candidateIds = getAllItemIds($db, $widget['id'], $sectionName);
                        }
                    } else {
                        $candidateIds = $preFilteredIds[$sectionName];
                    }
                } else {
                    // Если ничего не найдено по ключевым словам, используем все записи раздела
                    // Но только если категорий не было найдено (иначе это может быть ошибка фильтрации)
                    if (empty($categoriesBySection) || !isset($categoriesBySection[$sectionName]) || empty($categoriesBySection[$sectionName])) {
                        $candidateIds = getAllItemIds($db, $widget['id'], $sectionName);
                    }
                }
            } else {
                // Если ключевых слов нет, но есть категории - используем их
                // Если категорий нет - используем все записи раздела
                if (empty($categoriesBySection) || !isset($categoriesBySection[$sectionName]) || empty($categoriesBySection[$sectionName])) {
                    $candidateIds = getAllItemIds($db, $widget['id'], $sectionName);
                }
            }
        }
        
        // Ограничиваем количество кандидатов для производительности (максимум 500)
        // Но только если это не результат keyword search (там уже отфильтровано)
        if (count($candidateIds) > 500 && (empty($keywords) || empty($preFilteredIds[$sectionName] ?? []))) {
            $candidateIds = array_slice($candidateIds, 0, 500);
        }
        
        if (empty($candidateIds)) {
            continue;
        }
        
        // Сохраняем информацию о кандидатах для debug
        if ($debugInfo !== null && isset($debugInfo['embedding_search'])) {
            $debugInfo['embedding_search']['candidates_by_section'][$sectionName] = count($candidateIds);
        }
        
        // Проверяем, сколько кандидатов имеют embedding'и
        $placeholders = str_repeat('?,', count($candidateIds) - 1) . '?';
        $stmt = $db->prepare("
            SELECT COUNT(*) as count
            FROM item_embeddings ie
            WHERE ie.item_id IN ($placeholders)
        ");
        $stmt->execute($candidateIds);
        $hasEmbeddings = $stmt->fetch()['count'] > 0;
        
        // Шаг 4: Применяем embedding поиск среди кандидатов (только если есть embedding'и)
        if ($hasEmbeddings) {
            $candidateIdsBySection = [$sectionName => $candidateIds];
            $similarItems = findSimilarItemsByEmbedding($db, $widget['id'], $questionEmbedding, $candidateIdsBySection, $limit);
            
            if (isset($similarItems[$sectionName]) && !empty($similarItems[$sectionName])) {
                // Сохраняем результаты для debug
                if ($debugInfo !== null && isset($debugInfo['embedding_search'])) {
                    $debugInfo['embedding_search']['results_by_section'][$sectionName] = $similarItems[$sectionName];
                }
                
                // Извлекаем только ID (без similarity score)
                $relevantIds[$sectionName] = array_map(function($item) {
                    return $item['id'];
                }, $similarItems[$sectionName]);
                
                // Логируем топ-5 similarity scores для отладки
                $topSimilarities = array_slice($similarItems[$sectionName], 0, 5);
                $similarityScores = array_map(function($item) {
                    return round($item['similarity'] * 100, 1) . '%';
                }, $topSimilarities);
                logParser("Embedding search found " . count($relevantIds[$sectionName]) . " items for section $sectionName (top similarity: " . implode(', ', $similarityScores) . ")");
            } else {
                // Embedding поиск не дал результатов - используем keyword search
                logParser("Embedding search returned no results for section $sectionName (candidates: " . count($candidateIds) . "), falling back to keyword search");
                $useFallback = true;
            }
        } else {
            // У кандидатов нет embedding'ов - сразу используем keyword search
            logParser("No embeddings found for " . count($candidateIds) . " candidates in section $sectionName, using keyword search");
            $useFallback = true;
        }
        
        // Fallback: используем keyword search если embedding не дал результатов
        if (isset($useFallback) && $useFallback) {
            // Используем keyword search БЕЗ ограничения по кандидатам (чтобы найти все релевантные записи)
            // Передаем категории, если они есть, но не ограничиваемся только кандидатами
            $fallbackIds = searchByKeywords($db, $widget, $keywords, $categoriesBySection, $specializations, $serviceTypes);
            
            if (isset($fallbackIds[$sectionName]) && !empty($fallbackIds[$sectionName])) {
                $relevantIds[$sectionName] = $fallbackIds[$sectionName];
                // Ограничиваем количество результатов
                if (count($relevantIds[$sectionName]) > $limit) {
                    $relevantIds[$sectionName] = array_slice($relevantIds[$sectionName], 0, $limit);
                }
                $firstIds = array_slice($relevantIds[$sectionName], 0, 5);
                logParser("Fallback keyword search found " . count($relevantIds[$sectionName]) . " items for section $sectionName (first 5 IDs: " . implode(', ', $firstIds) . ")");
            } else {
                logParser("Fallback keyword search also returned no results for section $sectionName (keywords: " . implode(', ', array_slice($keywords, 0, 5)) . ")");
            }
        }
    }
    
    return $relevantIds;
}

/**
 * Построить промпт с данными в JSON формате
 */
/**
 * Фильтрует нерелевантные услуги на основе вопроса пользователя
 */
function filterIrrelevantServices($db, $widgetId, $serviceIds, $question) {
    if (empty($serviceIds) || empty($question)) {
        return $serviceIds;
    }
    
    $questionLower = mb_strtolower($question);
    
    // Определяем тип вопроса
    $isBackPain = (strpos($questionLower, 'спин') !== false || strpos($questionLower, 'позвоночник') !== false);
    $isJointPain = (strpos($questionLower, 'сустав') !== false || strpos($questionLower, 'колен') !== false);
    $isStomachPain = (strpos($questionLower, 'живот') !== false || strpos($questionLower, 'тошнот') !== false || strpos($questionLower, 'желудок') !== false);
    $isFootPain = (strpos($questionLower, 'стоп') !== false || strpos($questionLower, 'ногт') !== false);
    $isVaccination = (strpos($questionLower, 'вакцин') !== false || strpos($questionLower, 'прививк') !== false);
    $isPlastic = (strpos($questionLower, 'пластик') !== false);
    $isCertificate = (strpos($questionLower, 'справк') !== false || strpos($questionLower, 'комисси') !== false);
    
    // Получаем названия услуг
    $placeholders = str_repeat('?,', count($serviceIds) - 1) . '?';
    $stmt = $db->prepare("
        SELECT DISTINCT pi.id, pf.field_value as name
        FROM parsed_items pi
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.id IN ($placeholders) 
        AND pi.section_name = 'services'
        AND pf.field_name = 'name'
    ");
    $stmt->execute($serviceIds);
    
    $filteredIds = [];
    $excludedIds = [];
    
    while ($row = $stmt->fetch()) {
        $serviceId = $row['id'];
        $serviceName = mb_strtolower($row['name'] ?? '');
        
        $shouldExclude = false;
        
        // Проверка на пластическую хирургию
        if (!$isPlastic && (strpos($serviceName, 'пластический') !== false || strpos($serviceName, 'пластик') !== false)) {
            $shouldExclude = true;
        }
        
        // Проверка на подологию
        if (!$isFootPain && (strpos($serviceName, 'подолог') !== false || strpos($serviceName, 'подология') !== false)) {
            $shouldExclude = true;
        }
        
        // Проверка на вакцинацию
        if (!$isVaccination && (strpos($serviceName, 'вакцин') !== false || strpos($serviceName, 'прививк') !== false)) {
            $shouldExclude = true;
        }
        
        // Проверка на нейропсихологию (для физических болей)
        if (($isBackPain || $isJointPain || $isStomachPain) && strpos($serviceName, 'нейропсихолог') !== false) {
            $shouldExclude = true;
        }
        
        // Проверка на транспортные комиссии и справки
        if (!$isCertificate && (strpos($serviceName, 'транспортная комиссия') !== false || 
            strpos($serviceName, 'бассейн') !== false || 
            (strpos($serviceName, 'справк') !== false && strpos($serviceName, 'осмотр') !== false))) {
            $shouldExclude = true;
        }
        
        if ($shouldExclude) {
            $excludedIds[] = $serviceId;
            logParser("Excluding irrelevant service ID $serviceId: " . substr($row['name'], 0, 100));
        } else {
            $filteredIds[] = $serviceId;
        }
    }
    
    if (!empty($excludedIds)) {
        logParser("Filtered out " . count($excludedIds) . " irrelevant services: " . implode(', ', $excludedIds));
    }
    
    return $filteredIds;
}

function buildPrompt($db, $widget, $question = '', $filterIds = []) {
    $data = [];
    
    // Получаем разделы и поля для промпта (включая дочерние поля)
    $stmt = $db->prepare("
        SELECT DISTINCT ws.section_name, sf.field_name
        FROM widget_sections ws
        JOIN section_fields sf ON ws.id = sf.section_id
        WHERE ws.widget_id = ? AND ws.is_active = 1 AND sf.use_in_prompt = 1
        
        UNION
        
        SELECT DISTINCT ws.section_name, scf.child_field_name as field_name
        FROM widget_sections ws
        JOIN section_fields sf ON ws.id = sf.section_id
        JOIN section_child_fields scf ON sf.id = scf.field_id
        WHERE ws.widget_id = ? AND ws.is_active = 1 AND scf.use_in_prompt = 1
        
        ORDER BY section_name
    ");
    $stmt->execute([$widget['id'], $widget['id']]);
    
    $fieldsBySection = [];
    while ($row = $stmt->fetch()) {
        $fieldsBySection[$row['section_name']][] = $row['field_name'];
    }
    
    // Получаем все данные по разделам (включая дочерние элементы)
    foreach ($fieldsBySection as $section => $fields) {
        if (empty($fields)) continue;
        
        // Если передан фильтр по ID, используем его
        $whereClause = "WHERE pi.widget_id = ? AND pi.section_name = ? 
            AND pf.field_name IN (" . str_repeat('?,', count($fields) - 1) . "?)
            AND pi.is_duplicate = 0";
        
        $params = array_merge([$widget['id'], $section], $fields);
        
        if (!empty($filterIds) && isset($filterIds[$section]) && !empty($filterIds[$section])) {
            // Фильтруем по ID для этого раздела
            $placeholders = str_repeat('?,', count($filterIds[$section]) - 1) . '?';
            $whereClause .= " AND pi.id IN ($placeholders)";
            $params = array_merge($params, $filterIds[$section]);
        }
        
        $stmt = $db->prepare("
            SELECT pi.id, pf.field_name, pf.field_value
            FROM parsed_items pi
            JOIN parsed_fields pf ON pi.id = pf.item_id
            $whereClause
            ORDER BY pi.parent_item_id, pi.id
        ");
        $stmt->execute($params);
        
        $items = [];
        while ($row = $stmt->fetch()) {
            if (!isset($items[$row['id']])) {
                $items[$row['id']] = ['id' => (int)$row['id']];
            }
            $items[$row['id']][$row['field_name']] = $row['field_value'];
        }
        
        $data[$section] = array_values($items);
    }
    
    return json_encode($data, JSON_UNESCAPED_UNICODE);
}

/**
 * Найти услуги консультаций для выбранных специалистов
 * Использует предварительно связанные услуги из таблицы specialist_services
 * 
 * @param PDO $db Подключение к БД
 * @param int $widgetId ID виджета
 * @param array $specialistIds Массив ID специалистов
 * @return array Массив ID услуг консультаций
 */
function findConsultationServicesForSpecialists($db, $widgetId, $specialistIds) {
    if (empty($specialistIds) || !is_array($specialistIds)) {
        return [];
    }
    
    // Сначала пытаемся получить услуги из предварительно созданных связей
    $placeholders = str_repeat('?,', count($specialistIds) - 1) . '?';
    $stmt = $db->prepare("
        SELECT DISTINCT ss.service_id
        FROM specialist_services ss
        WHERE ss.widget_id = ? 
          AND ss.specialist_id IN ($placeholders)
    ");
    $params = array_merge([$widgetId], $specialistIds);
    $stmt->execute($params);
    
    $serviceIds = [];
    while ($row = $stmt->fetch()) {
        $serviceIds[] = (int)$row['service_id'];
    }
    
    // Если нашли предварительно связанные услуги - возвращаем их
    if (!empty($serviceIds)) {
        logParser("Found " . count($serviceIds) . " pre-linked consultation services for " . count($specialistIds) . " specialists");
        return $serviceIds;
    }
    
    // Fallback: если предварительных связей нет, используем старый метод поиска по именам
    // (это нужно только на время, пока не созданы связи)
    logParser("No pre-linked services found, using fallback search by names");
    
    // Получаем имена специалистов
    $stmt = $db->prepare("
        SELECT DISTINCT pi.id, pf.field_value as name
        FROM parsed_items pi
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.widget_id = ? 
          AND pi.section_name = 'specialists'
          AND pi.id IN ($placeholders)
          AND pf.field_name = 'name'
          AND pi.is_duplicate = 0
    ");
    $stmt->execute($params);
    
    $specialistNames = [];
    while ($row = $stmt->fetch()) {
        $name = trim($row['name']);
        if (!empty($name)) {
            $nameParts = explode(' ', $name);
            if (count($nameParts) > 0) {
                $specialistNames[] = $nameParts[0]; // Фамилия
            }
            $specialistNames[] = $name; // Полное имя
        }
    }
    
    if (empty($specialistNames)) {
        return [];
    }
    
    // Поиск по именам специалистов
    $nameConditions = [];
    $nameParams = [];
    foreach ($specialistNames as $name) {
        $nameConditions[] = "pf.field_value LIKE ?";
        $nameParams[] = '%' . $name . '%';
    }
    
    $stmt = $db->prepare("
        SELECT DISTINCT pi.id
        FROM parsed_items pi
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.widget_id = ?
          AND pi.section_name = 'services'
          AND pi.is_duplicate = 0
          AND pf.field_name = 'name'
          AND (
            pf.field_value LIKE '%консультация%'
            OR pf.field_value LIKE '%прием%'
            OR pf.field_value LIKE '%осмотр%'
          )
          AND (" . implode(' OR ', $nameConditions) . ")
        LIMIT 20
    ");
    $fallbackParams = array_merge([$widgetId], $nameParams);
    $stmt->execute($fallbackParams);
    
    while ($row = $stmt->fetch()) {
        $serviceIds[] = (int)$row['id'];
    }
    
    return array_unique($serviceIds);
}

/**
 * Обогащение данных полными записями из БД по ID
 */
function enrichDataWithFullRecords($db, $widgetId, $dataIds) {
    $enrichedData = [];
    
    foreach ($dataIds as $section => $ids) {
        if (empty($ids) || !is_array($ids)) {
            $enrichedData[$section] = [];
            continue;
        }
        
        // Преобразуем строковые ID в числовые для корректного сравнения
        $ids = array_map(function($id) {
            return is_string($id) ? (int)$id : $id;
        }, $ids);
        
        // Получаем полные данные по ID (только поля с show_in_widget = 1)
        $placeholders = str_repeat('?,', count($ids) - 1) . '?';
        $stmt = $db->prepare("
            SELECT pi.id, pf.field_name, pf.field_value
            FROM parsed_items pi
            JOIN parsed_fields pf ON pi.id = pf.item_id
            JOIN widget_sections ws ON ws.widget_id = pi.widget_id AND ws.section_name = pi.section_name
            JOIN section_fields sf ON sf.section_id = ws.id AND sf.field_name = pf.field_name
            WHERE pi.widget_id = ? AND pi.section_name = ? 
            AND pi.id IN ($placeholders)
            AND pi.is_duplicate = 0
            AND sf.show_in_widget = 1
        ");
        $params = array_merge([$widgetId, $section], $ids);
        $stmt->execute($params);
        
        $items = [];
        while ($row = $stmt->fetch()) {
            if (!isset($items[$row['id']])) {
                $items[$row['id']] = ['id' => (int)$row['id']];
            }
            $items[$row['id']][$row['field_name']] = $row['field_value'];
        }
        
        // Проверяем, какие ID не были найдены в БД
        $foundIds = array_keys($items);
        $missingIds = array_diff($ids, $foundIds);
        if (!empty($missingIds)) {
            logParser("WARNING: enrichDataWithFullRecords - Section '$section': " . count($missingIds) . " IDs not found in DB: " . implode(', ', array_slice($missingIds, 0, 10)) . (count($missingIds) > 10 ? '...' : ''));
            logParser("  - Requested IDs: " . count($ids) . ", Found IDs: " . count($foundIds));
        }
        
        // Сохраняем порядок ID из ответа Claude
        $orderedItems = [];
        foreach ($ids as $id) {
            $idInt = is_string($id) ? (int)$id : $id;
            if (isset($items[$idInt])) {
                $orderedItems[] = $items[$idInt];
            }
        }
        
        // Для услуг сортируем: консультации первыми
        if ($section === 'services' && !empty($orderedItems)) {
            usort($orderedItems, function($a, $b) {
                $nameA = mb_strtolower($a['name'] ?? '');
                $nameB = mb_strtolower($b['name'] ?? '');
                
                // Проверяем, является ли услуга консультацией
                $isConsultationA = (stripos($nameA, 'консультация') !== false || stripos($nameA, 'прием') !== false || stripos($nameA, 'осмотр') !== false);
                $isConsultationB = (stripos($nameB, 'консультация') !== false || stripos($nameB, 'прием') !== false || stripos($nameB, 'осмотр') !== false);
                
                // Консультации идут первыми
                if ($isConsultationA && !$isConsultationB) {
                    return -1;
                }
                if (!$isConsultationA && $isConsultationB) {
                    return 1;
                }
                
                // Если обе консультации или обе не консультации - сортируем по названию
                // Сначала первичные консультации, потом повторные
                if ($isConsultationA && $isConsultationB) {
                    $isPrimaryA = (stripos($nameA, 'первичн') !== false || stripos($nameA, 'первый') !== false);
                    $isPrimaryB = (stripos($nameB, 'первичн') !== false || stripos($nameB, 'первый') !== false);
                    
                    if ($isPrimaryA && !$isPrimaryB) {
                        return -1;
                    }
                    if (!$isPrimaryA && $isPrimaryB) {
                        return 1;
                    }
                }
                
                return strcmp($nameA, $nameB);
            });
        }
        
        $enrichedData[$section] = $orderedItems;
    }
    
    return $enrichedData;
}

/**
 * Финальная проверка релевантности услуг через AI модель
 */
function filterServicesByAI($db, $widgetId, $serviceIds, $question, $model, $customApiUrl = null, $customApiKey = null) {
    if (empty($serviceIds) || !is_array($serviceIds)) {
        return [];
    }
    
    try {
        // Получаем названия услуг
        $placeholders = str_repeat('?,', count($serviceIds) - 1) . '?';
        $stmt = $db->prepare("
            SELECT pi.id, pf.field_value as name
            FROM parsed_items pi
            JOIN parsed_fields pf ON pi.id = pf.item_id
            WHERE pi.id IN ($placeholders)
            AND pi.section_name = 'services'
            AND pi.widget_id = ?
            AND pf.field_name = 'name'
            AND pi.is_duplicate = 0
        ");
        $params = array_merge($serviceIds, [$widgetId]);
        $stmt->execute($params);
        
        $services = [];
        while ($row = $stmt->fetch()) {
            $services[$row['id']] = $row['name'];
        }
        
        if (empty($services)) {
            return $serviceIds; // Если не нашли услуги, возвращаем как есть
        }
        
        // Формируем промпт для проверки релевантности
        $servicesList = [];
        foreach ($services as $id => $name) {
            $servicesList[] = "ID $id: $name";
        }
        $servicesText = implode("\n", $servicesList);
        
        $systemPrompt = "Ты - помощник для фильтрации медицинских услуг по релевантности к запросу пользователя.

Твоя задача: из списка услуг выбрать ТОЛЬКО те, которые релевантны запросу пользователя.

КРИТИЧЕСКИ ВАЖНО - ИСКЛЮЧАЙ НЕРЕЛЕВАНТНЫЕ УСЛУГИ:
1. Косметология и эстетика (косметолог, эстетист, лифтинг, микроток, миостимуляция) - ТОЛЬКО если запрос не про косметологию
2. Пластическая хирургия - ТОЛЬКО если запрос не про пластическую хирургию
3. Подология - ТОЛЬКО если запрос не про стопы
4. Вакцинация (вакцин, прививк) - ТОЛЬКО если запрос не про вакцинацию
5. Справки (транспортная комиссия, заключение о состоянии здоровья, осмотр перед бассейном, осмотр перед вакцинацией) - ТОЛЬКО если запрос не про справки
6. Нейропсихология - ТОЛЬКО если запрос про физическую боль (не про психологические проблемы)
7. Общие консультации хирурга без указания специализации (травматолог/ортопед) - ТОЛЬКО если запрос не про общую хирургию
8. Гинекология (гинеколог, кольпо, кольпоскопия, пайпель, пайпель-тест, беременность, мазок гинеколог, межскрининговый период, видеокольпоскопия, мазок на флору гинеколог, отделяемое половых органов) - ТОЛЬКО если запрос не про гинекологию
9. Проктология (биопсия прямой кишки, рект. мазок, ректальный мазок, пцр-диагностика впч рект. мазок) - ТОЛЬКО если запрос не про проблемы с прямой кишкой
10. Урология (уролог, простата, мочеиспускание) - ТОЛЬКО если запрос не про урологию
11. Дерматология (дерматолог, кожные заболевания, акне, экзема) - ТОЛЬКО если запрос не про кожные проблемы

Формат ответа: ТОЛЬКО JSON массив ID релевантных услуг.
Пример: [1234, 5678, 9012]

НЕ добавляй никаких комментариев, объяснений или текста - ТОЛЬКО массив ID.";

        $userPrompt = "Запрос пользователя: $question\n\nСписок услуг для проверки:\n$servicesText\n\nВерни ТОЛЬКО массив ID релевантных услуг в формате JSON.";

        // Определяем провайдер
        $provider = 'openrouter';
        if (strpos($model, 'gpt://') === 0 || strpos($model, 'yandexgpt') !== false) {
            $provider = 'yandexgpt';
        } elseif (strpos($model, 'claude') !== false) {
            $provider = 'claude';
        }
        
        // Вызываем API
        $response = null;
        if ($provider === 'openrouter') {
            $response = callOpenRouterAPI($systemPrompt, $userPrompt, $question, $model);
        } elseif ($provider === 'yandexgpt') {
            $debugInfo = null;
            $response = callYandexGPTAPI($systemPrompt, $userPrompt, $question, $model, 1000, $debugInfo);
        } else {
            $response = callClaudeAPI($systemPrompt, $userPrompt, $question, false, $model);
        }
        
        if (!$response || !$response['success']) {
            logParser("AI filter failed, returning all services: " . ($response['error'] ?? 'Unknown error'));
            return $serviceIds; // При ошибке возвращаем все услуги
        }
        
        // Парсим ответ - модель может вернуть массив ID напрямую или в JSON объекте
        // Сначала проверяем data_ids - это самый надежный способ
        if (isset($response['data_ids']['services']) && is_array($response['data_ids']['services'])) {
            $filteredIds = $response['data_ids']['services'];
        } else {
            $rawAnswer = $response['raw_response'] ?? '';
            // Удаляем markdown обертку если есть
            $rawAnswer = preg_replace('/^```json\s*|\s*```$/s', '', $rawAnswer);
            $rawAnswer = preg_replace('/^```\s*|\s*```$/s', '', $rawAnswer);
            $rawAnswer = trim($rawAnswer);
            
            // Пытаемся распарсить как JSON
            $parsed = json_decode($rawAnswer, true);
            
            // Если это массив - используем его
            if (is_array($parsed) && isset($parsed[0])) {
                $filteredIds = $parsed;
            }
            // Если это объект с полем data или services
            elseif (is_array($parsed) && isset($parsed['data']) && is_array($parsed['data'])) {
                $filteredIds = $parsed['data'];
            }
            elseif (is_array($parsed) && isset($parsed['services']) && is_array($parsed['services'])) {
                $filteredIds = $parsed['services'];
            }
            // Пытаемся найти массив в тексте
            elseif (preg_match('/\[[\s\d,]*\]/', $rawAnswer, $matches)) {
                $filteredIds = json_decode($matches[0], true);
            }
            // Если все еще не массив, пытаемся извлечь ID из текста
            else {
                preg_match_all('/\b(\d+)\b/', $rawAnswer, $matches);
                $filteredIds = array_map('intval', $matches[1] ?? []);
            }
        }
        
        // Если все еще не массив, возвращаем исходный список
        if (!is_array($filteredIds)) {
            logParser("AI filter: Could not parse response as array, returning original list. Raw response: " . substr($rawAnswer, 0, 300));
            return $serviceIds;
        }
        
        // Фильтруем только те ID, которые были в исходном списке
        $filteredIds = array_map('intval', $filteredIds);
        $validFilteredIds = array_intersect($filteredIds, $serviceIds);
        
        // Если модель отфильтровала слишком много (осталось меньше 30% от исходного), возвращаем исходный список
        if (count($validFilteredIds) < count($serviceIds) * 0.3) {
            logParser("AI filter removed too many services (" . count($validFilteredIds) . " from " . count($serviceIds) . "), returning original list");
            return $serviceIds;
        }
        
        return array_values($validFilteredIds);
        
    } catch (Exception $e) {
        logParser("Error in filterServicesByAI: " . $e->getMessage());
        return $serviceIds; // При ошибке возвращаем все услуги
    }
}

/**
 * Логирование тестирования промпта в БД
 */
function logPromptTest($db, $widgetId, $question, $threeStageDebug, $stage3SystemPrompt, $promptData, $claudeResponse, $responseTime) {
    try {
        $stage1Request = null;
        $stage1Response = null;
        $stage1Prompt = null;
        $stage2Request = null;
        $stage2Response = null;
        $stage2Prompt = null;
        $stage3Request = null;
        $stage3Response = null;
        $stage3Prompt = null;
        $stage3PromptBefore = null;
        $stage3PromptAfter = null;
        
        // Этап 1 (может быть не в threeStageDebug, т.к. выполняется отдельно)
        if (isset($threeStageDebug['stage1'])) {
            $stage1 = $threeStageDebug['stage1'];
            $stage1Prompt = $stage1['prompt'] ?? null;
            $stage1Response = $stage1['response'] ?? $stage1['text'] ?? null;
            $stage1Request = json_encode([
                'model' => $stage1['model'] ?? null,
                'question' => $question
            ], JSON_UNESCAPED_UNICODE);
        }
        
        // Этап 2
        if (isset($threeStageDebug['stage2'])) {
            $stage2 = $threeStageDebug['stage2'];
            $stage2Prompt = $stage2['prompt'] ?? null;
            $stage2Response = $stage2['response'] ?? null;
            if (!$stage2Response && isset($stage2['medical_terms'])) {
                $stage2Response = json_encode($stage2['medical_terms'], JSON_UNESCAPED_UNICODE);
            }
            $stage2Request = json_encode([
                'model' => $stage2['model'] ?? null,
                'question' => $question,
                'keywords' => $stage2['keywords'] ?? null
            ], JSON_UNESCAPED_UNICODE);
        }
        
        // Этап 3
        if (isset($threeStageDebug['stage3'])) {
            $stage3 = $threeStageDebug['stage3'];
            $stage3Prompt = $stage3SystemPrompt . "\n\n" . $promptData;
            $stage3PromptBefore = $stage3['prompt_before'] ?? null;
            $stage3PromptAfter = $stage3['prompt_after'] ?? null;
            $stage3Response = $claudeResponse['raw_response'] ?? null;
            $stage3Request = json_encode([
                'model' => $stage3['model'] ?? null,
                'question' => $question,
                'system_prompt_length' => strlen($stage3SystemPrompt),
                'prompt_data_length' => strlen($promptData)
            ], JSON_UNESCAPED_UNICODE);
        }
        
        $finalResponse = json_encode([
            'text' => $claudeResponse['answer'] ?? null,
            'data_ids' => $claudeResponse['data_ids'] ?? null,
            'tokens_used' => $claudeResponse['tokens_used'] ?? null
        ], JSON_UNESCAPED_UNICODE);
        
        $debugInfo = json_encode([
            'three_stage_debug' => $threeStageDebug,
            'response_time_ms' => $responseTime,
            'provider' => $claudeResponse['provider'] ?? null,
            'model' => $claudeResponse['provider_model'] ?? null
        ], JSON_UNESCAPED_UNICODE);
        
        $stmt = $db->prepare("
            INSERT INTO prompt_test_logs 
            (widget_id, question, stage1_request, stage1_response, stage1_prompt, 
             stage2_request, stage2_response, stage2_prompt,
             stage3_request, stage3_response, stage3_prompt, stage3_prompt_before, stage3_prompt_after,
             final_response, debug_info)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $widgetId,
            $question,
            $stage1Request,
            $stage1Response,
            $stage1Prompt,
            $stage2Request,
            $stage2Response,
            $stage2Prompt,
            $stage3Request,
            $stage3Response,
            $stage3Prompt,
            $stage3PromptBefore,
            $stage3PromptAfter,
            $finalResponse,
            $debugInfo
        ]);
        
        logParser("Prompt test logged successfully. Log ID: " . $db->lastInsertId());
    } catch (Exception $e) {
        error_log("Failed to log prompt test: " . $e->getMessage());
        logParser("ERROR: Failed to log prompt test: " . $e->getMessage() . " | Trace: " . $e->getTraceAsString());
    }
}

/**
 * Отправить запрос к AI API с retry и fallback
 * Сначала пытается использовать OpenRouter, при ошибке переключается на Claude
 */
function sendClaudeRequest($systemPrompt, $promptData, $question, $useCache = true, $model = null, $debugMode = false, &$debugInfo = null, $custom_api_url = null, $custom_api_key = null, $section = null) {
    // Если указан кастомный API, используем его
    if (!empty($custom_api_url)) {
        try {
            $requestData = [
                'model' => $model ?? 'custom-model',
                'max_tokens' => 2048,
                'messages' => [
                    ['role' => 'user', 'content' => $systemPrompt . "\n\n" . $promptData]
                ],
                'temperature' => 0.3
            ];
            
            $response = callCustomAPI($custom_api_url, $custom_api_key, $requestData);
            
            // Извлекаем текст из ответа
            $text = '';
            if (isset($response['choices'][0]['message']['content'])) {
                $text = trim($response['choices'][0]['message']['content']);
            } elseif (isset($response['content'][0]['text'])) {
                $text = trim($response['content'][0]['text']);
            }
            
            // Извлекаем JSON из markdown code block, если есть
            if (preg_match('/```json\s*(.*?)\s*```/s', $text, $matches)) {
                $text = trim($matches[1]);
            } elseif (preg_match('/```\s*(.*?)\s*```/s', $text, $matches)) {
                $text = trim($matches[1]);
            }
            
            // Парсим JSON ответ
            $jsonData = json_decode($text, true);
            if (!$jsonData || json_last_error() !== JSON_ERROR_NONE) {
                // Если не JSON, возвращаем как текст
                return [
                    'success' => true,
                    'text' => $text,
                    'data_ids' => [],
                    'provider' => 'custom_api'
                ];
            }
            
            // Извлекаем data_ids из JSON
            $dataIds = [
                'specialists' => $jsonData['data']['specialists'] ?? [],
                'services' => $jsonData['data']['services'] ?? [],
                'articles' => $jsonData['data']['articles'] ?? [],
                'specializations' => $jsonData['data']['specializations'] ?? []
            ];
            
            return [
                'success' => true,
                'text' => $jsonData['text'] ?? $text,
                'data_ids' => $dataIds,
                'provider' => 'custom_api'
            ];
        } catch (Exception $e) {
            logParser("Custom API sendClaudeRequest error: " . $e->getMessage());
            throw $e;
        }
    }
    
    $useGPU = defined('GPU_LLM_ENABLED') && GPU_LLM_ENABLED;
    $useOpenRouterFirst = defined('USE_OPENROUTER_FIRST') && USE_OPENROUTER_FIRST;
    $useYandexGPTFirst = defined('USE_YANDEXGPT_FIRST') && USE_YANDEXGPT_FIRST;
    $lastError = null;
    $openRouterErrors = []; // Массив ошибок OpenRouter для debug
    
    // Определяем модель для использования
    $useModel = $model ?? OPENROUTER_MODEL;
    $isYandexGPT = (strpos($useModel, 'gpt://') === 0 || strpos($useModel, 'yandexgpt') !== false);
    // Определяем, является ли модель моделью OpenRouter (google/, qwen/, meta-llama/ и т.д.)
    $isOpenRouterModel = (strpos($useModel, 'google/') === 0 || 
                         strpos($useModel, 'qwen/') === 0 || 
                         strpos($useModel, 'meta-llama/') === 0 ||
                         strpos($useModel, 'anthropic/') === 0 ||
                         strpos($useModel, 'openai/') === 0 ||
                         (strpos($useModel, '/') !== false && strpos($useModel, 'gpt://') !== 0));
    
    // Если USE_YANDEXGPT_FIRST = true, принудительно используем Yandex GPT модель
    if ($useYandexGPTFirst && !$isYandexGPT) {
        $useModel = 'gpt://' . YANDEXGPT_FOLDER_ID . '/yandexgpt-lite/latest';
        $isYandexGPT = true;
        logParser("USE_YANDEXGPT_FIRST enabled, forcing Yandex GPT model: $useModel");
    }
    
    // Приоритет: GPU (если включен) → (fallback'и ниже)
    // Важно: для runtime-запросов виджета нам нужно отвечать ТОЛЬКО с GPU сервера.
    if ($useGPU) {
        for ($attempt = 0; $attempt < GPU_LLM_MAX_RETRIES; $attempt++) {
            try {
                logParser("Attempting GPU LLM API request (attempt " . ($attempt + 1) . "/" . GPU_LLM_MAX_RETRIES . ")");
                $response = callGPUAPI($systemPrompt, $promptData, $question, GPU_LLM_MODEL, $section);
                logParser("GPU LLM API request successful");
                
                if (!isset($response['provider'])) {
                    $response['provider'] = 'gpu';
                }
                if (!isset($response['model'])) {
                    $response['model'] = GPU_LLM_MODEL;
                }
                
                return $response;
            } catch (Exception $e) {
                $errorMsg = $e->getMessage();
                $lastError = $e;
                logParser("GPU LLM API error (attempt " . ($attempt + 1) . "): " . $errorMsg);
                
                // Если это ошибка rate limit или server error, retry
                if (strpos($errorMsg, '429') !== false || 
                    strpos($errorMsg, '500') !== false || 
                    strpos($errorMsg, '502') !== false ||
                    strpos($errorMsg, '503') !== false ||
                    strpos($errorMsg, 'timeout') !== false ||
                    strpos($errorMsg, 'Connection') !== false) {
                    
                    if ($attempt < GPU_LLM_MAX_RETRIES - 1) {
                        $delay = GPU_LLM_RETRY_DELAYS[$attempt] ?? GPU_LLM_RETRY_DELAYS[0];
                        logParser("Retry GPU LLM after {$delay}s delay");
                        sleep($delay);
                        continue;
                    }
                }
                
                // Если исчерпали попытки — не уходим во внешние LLM (OpenRouter/Claude/Yandex) для runtime.
                if ($attempt >= GPU_LLM_MAX_RETRIES - 1) {
                    throw $e;
                }
                // иначе продолжим цикл ретраев
            }
        }
    }
    
    if ($isYandexGPT && $useYandexGPTFirst) {
        for ($attempt = 0; $attempt < YANDEXGPT_MAX_RETRIES; $attempt++) {
            try {
                logParser("Attempting Yandex GPT API request (attempt " . ($attempt + 1) . "/" . YANDEXGPT_MAX_RETRIES . ")");
                $yandexDebugInfo = null;
                if ($debugMode) {
                    $response = callYandexGPTAPI($systemPrompt, $promptData, $question, $useModel, MAX_TOKENS_RESPONSE, $yandexDebugInfo);
                } else {
                    $response = callYandexGPTAPI($systemPrompt, $promptData, $question, $useModel, MAX_TOKENS_RESPONSE);
                }
                logParser("Yandex GPT API request successful");
                
                // Убеждаемся, что в ответе указан провайдер и модель
                if (!isset($response['provider'])) {
                    $response['provider'] = 'yandexgpt';
                }
                if (!isset($response['provider_model'])) {
                    $response['provider_model'] = $useModel;
                }
                
                // Добавляем debug информацию если включен режим отладки
                if ($debugMode && $yandexDebugInfo !== null) {
                    $response['yandex_debug'] = $yandexDebugInfo;
                    if ($debugInfo !== null) {
                        $debugInfo = $yandexDebugInfo;
                    }
                }
                
                return $response;
            } catch (Exception $e) {
                $errorMsg = $e->getMessage();
                $lastError = $e;
                logParser("Yandex GPT API error (attempt " . ($attempt + 1) . "): " . $errorMsg);
                
                // Если это ошибка rate limit или server error, retry
                if (strpos($errorMsg, '429') !== false || 
                    strpos($errorMsg, '500') !== false || 
                    strpos($errorMsg, '502') !== false ||
                    strpos($errorMsg, '503') !== false) {
                    
                    if ($attempt < YANDEXGPT_MAX_RETRIES - 1) {
                        $delay = YANDEXGPT_RETRY_DELAYS[$attempt] ?? YANDEXGPT_RETRY_DELAYS[0];
                        logParser("Retry Yandex GPT after {$delay}s delay");
                        sleep($delay);
                        continue;
                    }
                }
                
                // Fallback на OpenRouter или Claude
                logParser("Yandex GPT failed, switching to fallback");
                break;
            }
        }
    }
    
    // Пытаемся использовать OpenRouter, если:
    // 1. Включен USE_OPENROUTER_FIRST ИЛИ
    // 2. Модель является моделью OpenRouter (google/, qwen/, и т.д.)
    if (!$isYandexGPT && ($useOpenRouterFirst || $isOpenRouterModel)) {
        for ($attempt = 0; $attempt < OPENROUTER_MAX_RETRIES; $attempt++) {
            try {
                logParser("Attempting OpenRouter API request (attempt " . ($attempt + 1) . "/" . OPENROUTER_MAX_RETRIES . ") with model: $useModel");
                $response = callOpenRouterAPI($systemPrompt, $promptData, $question, $useModel);
                logParser("OpenRouter API request successful");
                // Добавляем информацию об успешном использовании OpenRouter
                $response['openrouter_debug'] = [
                    'attempts' => $attempt + 1,
                    'success' => true,
                    'errors' => []
                ];
                return $response;
            } catch (Exception $e) {
                $errorMsg = $e->getMessage();
                $lastError = $e;
                
                // Сохраняем информацию об ошибке
                $errorInfo = [
                    'attempt' => $attempt + 1,
                    'error' => $errorMsg,
                    'timestamp' => time()
                ];
                
                // Если в исключении есть информация о полном ответе OpenRouter, добавляем её
                if (property_exists($e, 'openrouterResponse') && isset($e->openrouterResponse)) {
                    $errorInfo['openrouter_response'] = $e->openrouterResponse;
                }
                
                $openRouterErrors[] = $errorInfo;
                
                logParser("OpenRouter API error (attempt " . ($attempt + 1) . "): " . $errorMsg);
                
                // Если это ошибка rate limit или server error, retry
                if (strpos($errorMsg, '429') !== false || 
                    strpos($errorMsg, '500') !== false || 
                    strpos($errorMsg, '502') !== false ||
                    strpos($errorMsg, '503') !== false) {
                    
                    if ($attempt < OPENROUTER_MAX_RETRIES - 1) {
                        // Для ошибки 429 (rate limit) увеличиваем задержку: 5, 10, 15 секунд
                        if (strpos($errorMsg, '429') !== false) {
                            $delay = 5 + ($attempt * 5); // 5s, 10s, 15s
                        } else {
                            $delay = OPENROUTER_RETRY_DELAYS[$attempt] ?? OPENROUTER_RETRY_DELAYS[0];
                        }
                        logParser("Retry OpenRouter after {$delay}s delay");
                        sleep($delay);
                        continue;
                    }
                }
                
                // Если это не retry-able ошибка или превышено количество попыток
                // Если модель является моделью OpenRouter, не используем Claude как fallback
                if ($isOpenRouterModel) {
                    logParser("OpenRouter failed with OpenRouter-specific model ($useModel). Cannot fallback to Claude.");
                } else {
                    logParser("OpenRouter failed, switching to Claude API as fallback");
                }
                break;
            }
        }
    }
    
    // Fallback на Claude API только если модель не является моделью OpenRouter
    // Если модель OpenRouter (google/, qwen/ и т.д.), не используем Claude API как fallback
    if ($isOpenRouterModel) {
        // Если модель OpenRouter, но OpenRouter не сработал, выбрасываем ошибку
        if ($lastError) {
            throw $lastError;
        }
        throw new Exception('OpenRouter API failed and model is OpenRouter-specific. Cannot fallback to Claude.');
    }
    
    // Fallback на Claude API
    // Если модель была Yandex GPT, используем модель Claude по умолчанию
    $claudeModel = ($isYandexGPT) ? null : $model; // null означает использование CLAUDE_MODEL по умолчанию
    logParser("Using Claude API (fallback or primary)" . ($isYandexGPT ? " with default Claude model" : ""));
    for ($attempt = 0; $attempt < CLAUDE_MAX_RETRIES; $attempt++) {
        try {
            $response = callClaudeAPI($systemPrompt, $promptData, $question, $useCache, $claudeModel);
            logParser("Claude API request successful");
            
            // Добавляем информацию об ошибках OpenRouter, если они были
            if (!empty($openRouterErrors)) {
                $response['openrouter_debug'] = [
                    'attempts' => count($openRouterErrors),
                    'success' => false,
                    'errors' => $openRouterErrors,
                    'fallback_reason' => 'OpenRouter failed, switched to Claude'
                ];
            }
            
            return $response;
        } catch (Exception $e) {
            $errorMsg = $e->getMessage();
            $lastError = $e;
            
            // Если это ошибка rate limit или server error, retry
            if (strpos($errorMsg, '429') !== false || 
                strpos($errorMsg, '500') !== false || 
                strpos($errorMsg, '502') !== false ||
                strpos($errorMsg, '503') !== false) {
                
                if ($attempt < CLAUDE_MAX_RETRIES - 1) {
                    // Для ошибки 429 (rate limit) увеличиваем задержку: 5, 10, 15 секунд
                    if (strpos($errorMsg, '429') !== false) {
                        $delay = 5 + ($attempt * 5); // 5s, 10s, 15s
                    } else {
                        $delay = CLAUDE_RETRY_DELAYS[$attempt];
                    }
                    logParser("Retry Claude attempt " . ($attempt + 1) . "/" . CLAUDE_MAX_RETRIES . " after {$delay}s delay");
                    sleep($delay);
                    continue;
                }
            }
            
            throw $e;
        }
    }
    
    // Если дошли сюда, значит все попытки исчерпаны
    if ($lastError) {
        throw $lastError;
    }
    
    throw new Exception('Max retries exceeded');
}

/**
 * Вызов Claude API с prompt caching
 */
function callClaudeAPI($systemPrompt, $promptData, $question, $useCache = true, $model = null) {
    // promptData теперь уже JSON строка
    
    // Структура запроса с prompt caching
    // Кешируем: system промпт и контекстные данные (JSON)
    // TTL управляется через константу CLAUDE_CACHE_TTL в config.php
    $systemContent = [
        'type' => 'text',
        'text' => $systemPrompt
    ];
    
    if ($useCache) {
        $systemContent['cache_control'] = ['type' => 'ephemeral', 'ttl' => CLAUDE_CACHE_TTL];
    }
    
    $promptContent = [
        'type' => 'text',
        'text' => $promptData
    ];
    
    if ($useCache) {
        $promptContent['cache_control'] = ['type' => 'ephemeral', 'ttl' => CLAUDE_CACHE_TTL];
    }
    
    $useModel = $model ?? CLAUDE_MODEL;
    $data = [
        'model' => $useModel,
        'max_tokens' => MAX_TOKENS_RESPONSE,
        'system' => [$systemContent],
        'messages' => [
            [
                'role' => 'user',
                'content' => [
                    $promptContent,
                    [
                        'type' => 'text',
                        'text' => "Вопрос пользователя: " . $question
                    ]
                ]
            ]
        ]
    ];
    
    $ch = curl_init(CLAUDE_API_URL);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, CLAUDE_TIMEOUT);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'x-api-key: ' . CLAUDE_API_KEY,
        'anthropic-version: 2023-06-01'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        throw new Exception('Claude API error: ' . curl_error($ch));
    }
    
    curl_close($ch);
    
    // Логируем полный ответ от Claude API в отдельный файл
    $claudeLogFile = LOG_DIR . '/claude_responses.log';
    $logEntry = sprintf(
        "[%s] Question: %s\nHTTP Code: %d\nResponse: %s\n%s\n",
        date('Y-m-d H:i:s'),
        substr($question, 0, 100),
        $httpCode,
        $response,
        str_repeat('=', 80)
    );
    file_put_contents($claudeLogFile, $logEntry, FILE_APPEND);
    
    if ($httpCode !== 200) {
        throw new Exception("Claude API returned HTTP $httpCode: $response");
    }
    
    $result = json_decode($response, true);
    
    if (!isset($result['content'][0]['text'])) {
        throw new Exception('Invalid Claude API response');
    }
    
    $answer = $result['content'][0]['text'];
    
    // Подсчет токенов с учетом кеширования
    $inputTokens = $result['usage']['input_tokens'] ?? 0;
    $outputTokens = $result['usage']['output_tokens'] ?? 0;
    $cacheCreationTokens = $result['usage']['cache_creation_input_tokens'] ?? 0;
    $cacheReadTokens = $result['usage']['cache_read_input_tokens'] ?? 0;
    
    // Детальная информация о кешировании (1h и 5m)
    $cache1hTokens = $result['usage']['cache_creation']['ephemeral_1h_input_tokens'] ?? 0;
    $cache5mTokens = $result['usage']['cache_creation']['ephemeral_5m_input_tokens'] ?? 0;
    
    $tokensUsed = $inputTokens + $outputTokens;
    
    // Логируем информацию о кешировании
    if ($cacheCreationTokens > 0 || $cacheReadTokens > 0) {
        $cacheInfo = "Prompt caching: created=$cacheCreationTokens (1h=$cache1hTokens, 5m=$cache5mTokens), read=$cacheReadTokens, regular=$inputTokens";
        logParser($cacheInfo);
    }
    
    // Парсим JSON ответ от Claude
    // Удаляем markdown обертку если она есть (```json ... ```)
    $answer = preg_replace('/^```json\s*|\s*```$/s', '', $answer);
    
    // Очищаем управляющие символы которые могут быть в ответе Claude
    $answer = preg_replace('/[\x00-\x1F\x7F]/u', '', $answer);
    
    // Используем флаг JSON_INVALID_UTF8_IGNORE для игнорирования проблемных символов
    $claudeResponse = json_decode($answer, true, 512, JSON_INVALID_UTF8_IGNORE);
    $jsonError = json_last_error();
    
    if ($jsonError !== JSON_ERROR_NONE) {
        $errorMsg = 'JSON decode error: ' . json_last_error_msg();
        logParser("$errorMsg. Raw response (first 500 chars): " . substr($answer, 0, 500));
        throw new Exception($errorMsg . '. Response: ' . substr($answer, 0, 200));
    }
    
    if (!$claudeResponse || !isset($claudeResponse['text']) || !isset($claudeResponse['data'])) {
        logParser("Invalid response structure. Response: " . substr($answer, 0, 500));
        throw new Exception('Invalid JSON response from Claude: ' . substr($answer, 0, 200));
    }
    
    return [
        'success' => true,
        'answer' => $claudeResponse['text'],
        'data_ids' => $claudeResponse['data'], // ID записей
        'tokens_used' => $tokensUsed,
        'raw_response' => $answer, // Сырой ответ для отладки
        'provider' => 'claude', // Указываем провайдер явно
        'cache_stats' => [
            'cache_creation_tokens' => $cacheCreationTokens,
            'cache_1h_tokens' => $cache1hTokens,
            'cache_5m_tokens' => $cache5mTokens,
            'cache_read_tokens' => $cacheReadTokens,
            'regular_tokens' => $inputTokens
        ]
    ];
}

/**
 * Вызов GPU LLM API (формат OpenAI-compatible)
 */
function callGPUAPI($systemPrompt, $promptData, $question, $model = null, $section = null) {
    // Объединяем system prompt и prompt data в один system message
    $fullSystemPrompt = $systemPrompt . "\n\n" . $promptData;
    
    // На GPU сервере доступна фиксированная модель (vLLM served model).
    // Не используем внешние/человекочитаемые названия моделей, иначе vLLM вернёт 404.
    $useModel = GPU_LLM_MODEL;
    
    logParser("=== SECTION DATA REQUEST TO GPU SERVER ===");
    logParser("URL: " . GPU_LLM_API_URL);
    logParser("Model: " . $useModel);
    logParser("Section: " . ($section ?? 'unknown'));
    
    $data = [
        'model' => $useModel,
        'max_tokens' => MAX_TOKENS_RESPONSE,
        'messages' => [
            [
                'role' => 'system',
                'content' => $fullSystemPrompt
            ],
            [
                'role' => 'user',
                'content' => "Вопрос пользователя: " . $question
            ]
        ],
        'temperature' => 0.7
    ];
    
    $ch = curl_init(GPU_LLM_API_URL);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, GPU_LLM_TIMEOUT);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $rawResponse = $response;
    
    if (curl_errno($ch)) {
        $error = curl_error($ch);
        curl_close($ch);
        throw new Exception('GPU LLM API error: ' . $error);
    }
    
    curl_close($ch);
    
    if ($httpCode !== 200) {
        throw new Exception("GPU LLM API returned HTTP $httpCode: " . substr($rawResponse, 0, 500));
    }
    
    $result = json_decode($rawResponse, true);
    
    if (!isset($result['choices'][0]['message']['content'])) {
        throw new Exception('Invalid GPU LLM API response structure. Full response: ' . substr($rawResponse, 0, 1000));
    }
    
    $answer = $result['choices'][0]['message']['content'];
    
    // Подсчет токенов
    $inputTokens = $result['usage']['prompt_tokens'] ?? 0;
    $outputTokens = $result['usage']['completion_tokens'] ?? 0;
    $tokensUsed = $inputTokens + $outputTokens;
    
    // Извлекаем JSON из markdown code block (если есть)
    $answer = extractJSONFromMarkdown($answer);
    
    // Убираем управляющие символы
    $answer = preg_replace('/[\x00-\x1F\x7F]/u', '', $answer);
    $answer = trim($answer);
    
    // Логируем для отладки
    logParser("GPU LLM answer (first 500 chars): " . substr($answer, 0, 500));
    
    // Пытаемся распарсить JSON
    $jsonData = json_decode($answer, true);
    
    if ($jsonData && json_last_error() === JSON_ERROR_NONE) {
        // Если это валидный JSON, извлекаем данные
        $text = $jsonData['text'] ?? '';
        
        // GPU может возвращать:
        // 1. {"data": {"services": [...]}}
        // 2. {"services": [...]}
        // 3. Просто массив [1,2,3] (для секций specialists/articles/specializations)
        
        $dataIds = [
            'specialists' => [],
            'services' => [],
            'articles' => [],
            'specializations' => []
        ];
        
        // Если это просто массив (не ассоциативный)
        if (isset($jsonData[0]) && is_numeric($jsonData[0])) {
            // Простой массив чисел [1,2,3]
            if ($section) {
                $dataIds[$section] = array_map('intval', $jsonData);
                logParser("GPU returned simple array for section '$section': " . count($jsonData) . " items");
            }
        } elseif (isset($jsonData[0]) && is_string($jsonData[0])) {
            // Массив строк ["123", "456"] или ["Название1", "Название2"]
            if ($section) {
                // Сначала пробуем конвертировать в числа
                $numericIds = array_map('intval', $jsonData);
                
                // Проверяем, все ли ID валидные (не нули)
                $hasZeros = in_array(0, $numericIds, true);
                
                if ($hasZeros && ($section === 'specializations' || $section === 'specialists' || $section === 'articles')) {
                    // GPU вернул названия вместо ID - ищем ID по названиям в БД
                    logParser("GPU returned names instead of IDs for section '$section', searching in DB...");
                    
                    global $db;
                    $foundIds = [];
                    
                    foreach ($jsonData as $name) {
                        $name = trim($name);
                        if (empty($name)) continue;
                        
                        // Ищем в parsed_fields по названию
                        $stmt = $db->prepare("
                            SELECT DISTINCT pi.id
                            FROM parsed_items pi
                            JOIN parsed_fields pf ON pi.id = pf.item_id
                            WHERE pi.section_name = ?
                            AND pf.field_name = 'name'
                            AND pf.field_value = ?
                            AND pi.is_duplicate = 0
                            LIMIT 1
                        ");
                        $stmt->execute([$section, $name]);
                        $row = $stmt->fetch();
                        
                        if ($row) {
                            $foundIds[] = (int)$row['id'];
                            logParser("Found ID {$row['id']} for name '$name' in section '$section'");
                        } else {
                            logParser("WARNING: Name '$name' not found in section '$section'");
                        }
                    }
                    
                    $dataIds[$section] = $foundIds;
                    logParser("GPU returned string array for section '$section': converted " . count($foundIds) . " names to IDs");
                } else {
                    // Обычные числовые строки ["123", "456"]
                    $dataIds[$section] = $numericIds;
                    logParser("GPU returned string array for section '$section': " . count($jsonData) . " items");
                }
            }
        } else {
            // Объект {"services": [...]} или {"data": {"services": [...]}}
            $data = $jsonData['data'] ?? $jsonData;
            
            $dataIds = [
                'specialists' => $data['specialists'] ?? [],
                'services' => $data['services'] ?? [],
                'articles' => $data['articles'] ?? [],
                'specializations' => $data['specializations'] ?? []
            ];
        }
        
        return [
            'success' => true,
            'text' => $text,
            'data_ids' => $dataIds,
            'provider' => 'gpu',
            'model' => $useModel,
            'tokens_used' => $tokensUsed
        ];
    }
    
    // Если не JSON, возвращаем как текст
    return [
        'success' => true,
        'text' => $answer,
        'data_ids' => [
            'specialists' => [],
            'services' => [],
            'articles' => [],
            'specializations' => []
        ],
        'provider' => 'gpu',
        'model' => $useModel,
        'tokens_used' => $tokensUsed
    ];
}

/**
 * Вызов OpenRouter API (формат OpenAI-compatible)
 */
function callOpenRouterAPI($systemPrompt, $promptData, $question, $model = null) {
    // OpenRouter использует формат OpenAI API
    // Объединяем system prompt и prompt data в один system message
    $fullSystemPrompt = $systemPrompt . "\n\n" . $promptData;
    
    // Используем указанную модель или дефолтную
    $useModel = $model ?? OPENROUTER_MODEL;
    $data = [
        'model' => $useModel,
        'max_tokens' => MAX_TOKENS_RESPONSE,
        'messages' => [
            [
                'role' => 'system',
                'content' => $fullSystemPrompt
            ],
            [
                'role' => 'user',
                'content' => "Вопрос пользователя: " . $question
            ]
        ],
        'temperature' => 0.7
    ];
    
    $ch = curl_init(OPENROUTER_API_URL);
    configureOpenRouterProxy($ch); // Настройка прокси для OpenRouter
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, 60); // Увеличено до 60 секунд для больших промптов и моделей
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: Bearer ' . OPENROUTER_API_KEY,
        'HTTP-Referer: ' . WIDGET_DOMAIN, // Опционально, для отслеживания
        'X-Title: AI Widget', // Опционально
        'X-Provider: deepinfra/fp8' // Принудительно использовать только deepinfra/fp8
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $rawResponse = $response; // Сохраняем сырой ответ для debug
    
    if (curl_errno($ch)) {
        $error = curl_error($ch);
        curl_close($ch);
        throw new Exception('OpenRouter API error: ' . $error);
    }
    
    curl_close($ch);
    
    // Логируем полный ответ от OpenRouter API
    $openrouterLogFile = LOG_DIR . '/openrouter_responses.log';
    $logEntry = sprintf(
        "[%s] Question: %s\nHTTP Code: %d\nResponse: %s\n%s\n",
        date('Y-m-d H:i:s'),
        substr($question, 0, 100),
        $httpCode,
        $rawResponse,
        str_repeat('=', 80)
    );
    file_put_contents($openrouterLogFile, $logEntry, FILE_APPEND);
    
    if ($httpCode !== 200) {
        throw new Exception("OpenRouter API returned HTTP $httpCode: " . substr($rawResponse, 0, 500));
    }
    
    $result = json_decode($rawResponse, true);
    
    if (!isset($result['choices'][0]['message']['content'])) {
        // Если ответ не в ожидаемом формате, сохраняем полный ответ для debug
        $errorMsg = 'Invalid OpenRouter API response structure. Full response: ' . substr($rawResponse, 0, 1000);
        throw new Exception($errorMsg);
    }
    
    $answer = $result['choices'][0]['message']['content'];
    
    // Подсчет токенов
    $inputTokens = $result['usage']['prompt_tokens'] ?? 0;
    $outputTokens = $result['usage']['completion_tokens'] ?? 0;
    $tokensUsed = $inputTokens + $outputTokens;
    
    // Извлекаем стоимость из ответа OpenRouter
    $cost = null;
    if (isset($result['usage']['cost'])) {
        $cost = floatval($result['usage']['cost']);
    } elseif (isset($result['cost'])) {
        $cost = floatval($result['cost']);
    } elseif (isset($result['usage']['total_cost'])) {
        $cost = floatval($result['usage']['total_cost']);
    }
    
    // Парсим JSON ответ
    // Удаляем markdown обертку если она есть (```json ... ```)
    $answer = preg_replace('/^```json\s*|\s*```$/s', '', $answer);
    
    // Очищаем управляющие символы
    $answer = preg_replace('/[\x00-\x1F\x7F]/u', '', $answer);
    
    // Используем флаг JSON_INVALID_UTF8_IGNORE для игнорирования проблемных символов
    $openrouterResponse = json_decode($answer, true, 512, JSON_INVALID_UTF8_IGNORE);
    $jsonError = json_last_error();
    
    if ($jsonError !== JSON_ERROR_NONE) {
        $errorMsg = 'JSON decode error: ' . json_last_error_msg();
        // Сохраняем полный ответ от OpenRouter для debug
        $fullOpenRouterResponse = [
            'raw_answer' => $answer,
            'full_api_response' => $rawResponse,
            'http_code' => $httpCode
        ];
        logParser("$errorMsg. Raw answer: " . substr($answer, 0, 500));
        logParser("Full OpenRouter API response: " . substr($rawResponse, 0, 1000));
        
        // Пробрасываем исключение с информацией о полном ответе
        $exception = new Exception($errorMsg . '. Response: ' . substr($answer, 0, 200));
        $exception->openrouterResponse = $fullOpenRouterResponse; // Добавляем полный ответ в исключение
        throw $exception;
    }
    
    // Проверяем формат ответа - может быть объект с text/data или просто массив ID
    if (is_array($openrouterResponse) && isset($openrouterResponse[0]) && is_numeric($openrouterResponse[0])) {
        // Это простой массив ID - используем его напрямую
        $response = [
            'success' => true,
            'text' => '',
            'data_ids' => ['services' => $openrouterResponse], // Помещаем массив в services для совместимости
            'tokens_used' => $tokensUsed,
            'raw_response' => $answer, // Сырой ответ для отладки
            'raw_response_text' => $answer, // Сырой текст ответа для отладки
            'provider' => 'openrouter',
            'provider_model' => $useModel,
            'usage' => [
                'prompt_tokens' => $inputTokens,
                'completion_tokens' => $outputTokens
            ]
        ];
    } elseif (is_array($openrouterResponse) && (
        isset($openrouterResponse['s']) || 
        isset($openrouterResponse['sv']) || 
        isset($openrouterResponse['a']) || 
        isset($openrouterResponse['sp'])
    )) {
        // Это объект с минифицированными ключами разделов (s, sv, a, sp)
        // Нормализуем формат: преобразуем минифицированные ключи в полные
        $dataIds = [
            'specialists' => [],
            'services' => [],
            'articles' => [],
            'specializations' => []
        ];
        
        // Маппинг минифицированных ключей на полные
        $minifyMap = [
            's' => 'specialists',
            'sv' => 'services',
            'a' => 'articles',
            'sp' => 'specializations'
        ];
        
        // Обрабатываем каждый минифицированный раздел
        foreach ($minifyMap as $minKey => $fullKey) {
            if (isset($openrouterResponse[$minKey])) {
                $sectionData = $openrouterResponse[$minKey];
                // Если это массив ID, используем его напрямую
                if (is_array($sectionData)) {
                    $ids = [];
                    foreach ($sectionData as $item) {
                        // Если элемент - это число, используем его напрямую
                        if (is_numeric($item)) {
                            $ids[] = (int)$item;
                        }
                        // Если элемент - это объект/массив с полем 'i' (минифицированный id), извлекаем ID
                        elseif (is_array($item) && isset($item['i']) && is_numeric($item['i'])) {
                            $ids[] = (int)$item['i'];
                        }
                        // Если элемент - это объект с числовым ключом, используем его
                        elseif (is_object($item) && isset($item->i) && is_numeric($item->i)) {
                            $ids[] = (int)$item->i;
                        }
                    }
                    // Убираем дубликаты и переиндексируем
                    $dataIds[$fullKey] = array_values(array_unique($ids));
                }
            }
        }
        
        $response = [
            'success' => true,
            'text' => $openrouterResponse['text'] ?? '',
            'data_ids' => $dataIds,
            'tokens_used' => $tokensUsed,
            'raw_response' => $answer, // Сырой ответ для отладки
            'raw_response_text' => $answer, // Сырой текст ответа для отладки
            'provider' => 'openrouter',
            'provider_model' => $useModel,
            'usage' => [
                'prompt_tokens' => $inputTokens,
                'completion_tokens' => $outputTokens
            ]
        ];
    } elseif (is_array($openrouterResponse) && (
        isset($openrouterResponse['specialists']) || 
        isset($openrouterResponse['services']) || 
        isset($openrouterResponse['articles']) || 
        isset($openrouterResponse['specializations'])
    )) {
        // Это объект с ключами разделов (specialists, services, articles, specializations)
        // Нормализуем формат: преобразуем массивы ID в нужную структуру
        $dataIds = [
            'specialists' => [],
            'services' => [],
            'articles' => [],
            'specializations' => []
        ];
        
        // Обрабатываем каждый раздел
        foreach (['specialists', 'services', 'articles', 'specializations'] as $section) {
            if (isset($openrouterResponse[$section])) {
                $sectionData = $openrouterResponse[$section];
                // Если это массив ID, используем его напрямую
                if (is_array($sectionData)) {
                    $ids = [];
                    foreach ($sectionData as $item) {
                        // Если элемент - это число, используем его напрямую
                        if (is_numeric($item)) {
                            $ids[] = (int)$item;
                        }
                        // Если элемент - это объект/массив с полем 'i' (минифицированный id), извлекаем ID
                        elseif (is_array($item) && isset($item['i']) && is_numeric($item['i'])) {
                            $ids[] = (int)$item['i'];
                        }
                        // Если элемент - это объект/массив с полем 'id', извлекаем ID
                        elseif (is_array($item) && isset($item['id']) && is_numeric($item['id'])) {
                            $ids[] = (int)$item['id'];
                        }
                        // Если элемент - это объект с полем 'i' (минифицированный id), извлекаем ID
                        elseif (is_object($item) && isset($item->i) && is_numeric($item->i)) {
                            $ids[] = (int)$item->i;
                        }
                        // Если элемент - это объект с числовым ключом, используем его
                        elseif (is_object($item) && isset($item->id) && is_numeric($item->id)) {
                            $ids[] = (int)$item->id;
                        }
                    }
                    // Убираем дубликаты и переиндексируем
                    $dataIds[$section] = array_values(array_unique($ids));
                }
            }
        }
        
        $response = [
            'success' => true,
            'text' => $openrouterResponse['text'] ?? '',
            'data_ids' => $dataIds,
            'tokens_used' => $tokensUsed,
            'raw_response' => $answer, // Сырой ответ для отладки
            'raw_response_text' => $answer, // Сырой текст ответа для отладки
            'provider' => 'openrouter',
            'provider_model' => $useModel,
            'usage' => [
                'prompt_tokens' => $inputTokens,
                'completion_tokens' => $outputTokens
            ]
        ];
    } elseif (!$openrouterResponse || !isset($openrouterResponse['text']) || !isset($openrouterResponse['data'])) {
        logParser("Invalid response structure. Response: " . substr($answer, 0, 500));
        throw new Exception('Invalid JSON response from OpenRouter: ' . substr($answer, 0, 200));
    } else {
        // Стандартный формат с text и data
        $response = [
            'success' => true,
            'text' => $openrouterResponse['text'],
            'data_ids' => $openrouterResponse['data'], // ID записей
            'tokens_used' => $tokensUsed,
            'raw_response' => $answer, // Сырой ответ для отладки
            'raw_response_text' => $answer, // Сырой текст ответа для отладки
            'provider' => 'openrouter',
            'provider_model' => $useModel,
            'usage' => [
                'prompt_tokens' => $inputTokens,
                'completion_tokens' => $outputTokens
            ]
        ];
    }
    
    // Добавляем стоимость, если она была извлечена
    if ($cost !== null) {
        $response['cost'] = $cost;
        logParser("OpenRouter API cost: $" . number_format($cost, 6));
    }
    
    return $response;
}

/**
 * Вызов Yandex GPT API (универсальная функция для этапов 2 и 3)
 * Поддерживает формат модели: gpt://{folder_id}/{model_name}/{version}
 */
function callYandexGPTAPI($systemPrompt, $promptData, $question, $model = null, $maxTokens = MAX_TOKENS_RESPONSE, &$debugInfo = null) {
    // Парсим модель формата gpt://{folder_id}/{model_name}/{version}
    $folderId = YANDEXGPT_FOLDER_ID;
    $modelUri = $model ?? 'gpt://' . YANDEXGPT_FOLDER_ID . '/yandexgpt-lite/latest';
    
    if ($model && strpos($model, 'gpt://') === 0) {
        // Формат: gpt://b1gqogohv4tgsmbj9gb4/yandexgpt-5-lite/latest
        $parts = explode('/', substr($model, 6)); // Убираем "gpt://"
        if (count($parts) >= 2) {
            $folderId = $parts[0]; // b1gqogohv4tgsmbj9gb4
            $modelName = $parts[1]; // yandexgpt-5-lite или yandexgpt-lite
            $version = $parts[2] ?? 'latest'; // latest
            $modelUri = "gpt://{$folderId}/{$modelName}/{$version}";
        }
    }
    
    // Используем новый API endpoint (rest-assistant)
    $apiUrl = 'https://rest-assistant.api.cloud.yandex.net/v1/responses';
    
    // Объединяем system prompt и prompt data
    // Если systemPrompt пустой, используем только promptData
    if (empty($systemPrompt)) {
        $fullSystemPrompt = $promptData;
    } else {
        $fullSystemPrompt = $systemPrompt . "\n\n" . $promptData;
    }
    
    // Если question пустой, не добавляем его (для этапа 2 вопрос уже в promptData)
    // Но для Yandex GPT API поле input обязательно, поэтому передаем хотя бы пустую строку
    $userPrompt = empty($question) ? "Выполни задачу." : "Вопрос пользователя: " . $question;
    
    // Формируем запрос в формате нового API
    $data = [
        'model' => $modelUri,
        'temperature' => 0.7,
        'instructions' => $fullSystemPrompt,
        'input' => $userPrompt,
        'max_output_tokens' => $maxTokens
    ];
    
    // Сохраняем информацию о запросе для debug
    $requestInfo = [
        'url' => $apiUrl,
        'method' => 'POST',
        'headers' => [
            'Content-Type: application/json',
            'Authorization: Api-Key ' . (defined('YANDEXGPT_API_KEY') ? substr(YANDEXGPT_API_KEY, 0, 10) . '...' : 'N/A'),
            'x-folder-id: ' . $folderId
        ],
        'body' => $data
    ];
    
    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, 60); // Увеличенный таймаут для больших запросов
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: Api-Key ' . YANDEXGPT_API_KEY,
        'x-folder-id: ' . $folderId
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    // Сохраняем информацию об ответе для debug
    $responseInfo = [
        'http_code' => $httpCode,
        'response' => $response
    ];
    
    if (curl_errno($ch)) {
        $error = curl_error($ch);
        curl_close($ch);
        
        // Сохраняем информацию об ошибке
        if ($debugInfo !== null) {
            $debugInfo = [
                'request' => $requestInfo,
                'response' => $responseInfo,
                'error' => $error
            ];
        }
        
        throw new Exception('Yandex GPT API error: ' . $error);
    }
    
    curl_close($ch);
    
    if ($httpCode !== 200) {
        // Сохраняем информацию об ошибке HTTP
        if ($debugInfo !== null) {
            $debugInfo = [
                'request' => $requestInfo,
                'response' => $responseInfo,
                'error' => "HTTP $httpCode: " . substr($response, 0, 500)
            ];
        }
        
        throw new Exception("Yandex GPT API returned HTTP $httpCode: " . substr($response, 0, 500));
    }
    
    $result = json_decode($response, true);
    
    if (!$result) {
        // Сохраняем информацию об ошибке парсинга
        if ($debugInfo !== null) {
            $debugInfo = [
                'request' => $requestInfo,
                'response' => $responseInfo,
                'error' => 'Invalid JSON response: ' . substr($response, 0, 500)
            ];
        }
        
        throw new Exception('Invalid Yandex GPT API response: ' . substr($response, 0, 500));
    }
    
    // Проверяем статус ответа
    if (isset($result['status']) && $result['status'] === 'failed') {
        $errorMessage = 'Yandex GPT API returned failed status';
        if (isset($result['error'])) {
            $errorMessage .= ': ' . json_encode($result['error']);
        }
        if (isset($result['message'])) {
            $errorMessage .= ' - ' . $result['message'];
        }
        logParser("Yandex GPT API failed status: " . json_encode($result));
        
        // Сохраняем информацию об ошибке статуса
        if ($debugInfo !== null) {
            $debugInfo = [
                'request' => $requestInfo,
                'response' => $responseInfo,
                'error' => $errorMessage,
                'result' => $result
            ];
        }
        
        throw new Exception($errorMessage);
    }
    
    // Обрабатываем ответ нового API формата
    // Ответ находится в output[0].content[0].text
    if (!isset($result['output'][0]['content'][0]['text'])) {
        logParser("Yandex GPT API invalid response structure: " . json_encode($result));
        
        // Сохраняем информацию об ошибке структуры
        if ($debugInfo !== null) {
            $debugInfo = [
                'request' => $requestInfo,
                'response' => $responseInfo,
                'error' => 'Invalid response structure',
                'result' => $result
            ];
        }
        
        throw new Exception('Invalid Yandex GPT API response structure: ' . json_encode($result));
    }
    
    // Сохраняем успешный ответ для debug
    if ($debugInfo !== null) {
        $debugInfo = [
            'request' => $requestInfo,
            'response' => $responseInfo,
            'result' => $result
        ];
    }
    
    $answer = trim($result['output'][0]['content'][0]['text']);
    
    // Очищаем текст от лишних символов
    $answer = preg_replace('/[\x00-\x1F\x7F]/u', '', $answer);
    
    // Удаляем markdown обертку если она есть (```json ... ```)
    $answer = preg_replace('/^```json\s*|\s*```$/s', '', $answer);
    $answer = trim($answer);
    
    // Подсчитываем токены из usage
    $tokensUsed = 0;
    if (isset($result['usage']['total_tokens'])) {
        $tokensUsed = $result['usage']['total_tokens'];
    } elseif (isset($result['usage'])) {
        $tokensUsed = ($result['usage']['input_tokens'] ?? 0) + ($result['usage']['output_tokens'] ?? 0);
    }
    
    // Парсим JSON ответ
    $parsedResponse = json_decode($answer, true, 512, JSON_INVALID_UTF8_IGNORE);
    $jsonError = json_last_error();
    
    if ($jsonError !== JSON_ERROR_NONE) {
        // Если это не JSON, возвращаем как текст (для этапа 2 может быть просто JSON без обертки)
        // Пытаемся извлечь JSON из текста
        if (preg_match('/\{[^{}]*(?:\{[^{}]*\}[^{}]*)*\}/s', $answer, $matches)) {
            $parsedResponse = json_decode($matches[0], true);
            if ($parsedResponse) {
                $answer = $matches[0];
            }
        }
        
        if (!$parsedResponse) {
            throw new Exception('Yandex GPT API returned non-JSON response: ' . substr($answer, 0, 200));
        }
    }
    
    // Для этапа 2 (извлечение терминов) возвращаем medical_terms
    // Для этапа 3 (основной запрос) возвращаем text и data
    if (isset($parsedResponse['symptoms']) || isset($parsedResponse['specializations']) || isset($parsedResponse['keywords'])) {
        // Это ответ для этапа 2 (извлечение терминов)
        return [
            'medical_terms' => $parsedResponse,
            'provider' => 'yandexgpt',
            'tokens_used' => $tokensUsed,
            'raw_response' => $answer
        ];
    } else if (isset($parsedResponse['text']) && isset($parsedResponse['data'])) {
        // Это ответ для этапа 3 (основной запрос)
        // Определяем модель из параметра функции (передается из вызова)
        // $model доступен в области видимости функции callYandexGPTAPI
        $modelUsed = $model ?? 'gpt://' . YANDEXGPT_FOLDER_ID . '/yandexgpt-lite/latest';
        
        return [
            'success' => true,
            'answer' => $parsedResponse['text'],
            'data_ids' => $parsedResponse['data'],
            'tokens_used' => $tokensUsed,
            'raw_response' => $answer,
            'provider' => 'yandexgpt',
            'provider_model' => $modelUsed,
            'usage' => [
                'input_tokens' => $result['usage']['input_tokens'] ?? 0,
                'output_tokens' => $result['usage']['output_tokens'] ?? 0
            ]
        ];
    } else {
        // Неизвестный формат ответа - логируем для отладки
        logParser("Yandex GPT API unexpected response format. Parsed keys: " . implode(', ', array_keys($parsedResponse ?? [])) . ". Raw answer (first 500): " . substr($answer, 0, 500));
        
        // Сохраняем информацию об ошибке формата для debug
        if ($debugInfo !== null) {
            $debugInfo['error'] = 'Unexpected response format';
            $debugInfo['parsed_keys'] = array_keys($parsedResponse ?? []);
            $debugInfo['raw_answer'] = substr($answer, 0, 500);
        }
        
        throw new Exception('Unexpected Yandex GPT API response format. Keys: ' . implode(', ', array_keys($parsedResponse ?? [])) . '. Response: ' . substr($answer, 0, 200));
    }
}

