<?php
/**
 * API endpoint для запросов по отдельному разделу
 * POST /api/query-section.php
 * Body: { "widget_key": "clinic-001", "question": "У меня болит спина", "section_name": "specialists", "_debug": false }
 */

// Отключаем вывод ошибок на экран
ini_set('display_errors', 0);
error_reporting(E_ALL);

// Увеличиваем время выполнения для сложных запросов
set_time_limit(120); // 2 минуты вместо стандартных 30 секунд

// Устанавливаем заголовок JSON сразу
header('Content-Type: application/json');

// Обработка фатальных ошибок для гарантии JSON ответа
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        http_response_code(500);
        echo json_encode([
            'error' => 'Internal server error: ' . $error['message'],
            'ready' => false,
            'debug' => [
                'file' => $error['file'],
                'line' => $error['line'],
                'type' => $error['type']
            ]
        ], JSON_UNESCAPED_UNICODE);
    }
});

// Функция вычисления косинусного сходства (нужна для embedding фильтрации)
if (!function_exists('calculateCosineSimilarity')) {
    function calculateCosineSimilarity($vec1, $vec2) {
        $dotProduct = 0;
        $magnitude1 = 0;
        $magnitude2 = 0;
        
        $count = min(count($vec1), count($vec2));
        
        for ($i = 0; $i < $count; $i++) {
            $dotProduct += $vec1[$i] * $vec2[$i];
            $magnitude1 += $vec1[$i] * $vec1[$i];
            $magnitude2 += $vec2[$i] * $vec2[$i];
        }
        
        $magnitude1 = sqrt($magnitude1);
        $magnitude2 = sqrt($magnitude2);
        
        if ($magnitude1 == 0 || $magnitude2 == 0) {
            return 0;
        }
        
        return $dotProduct / ($magnitude1 * $magnitude2);
    }
}

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/embedding-functions.php';

// Подключаем функции из query.php без выполнения основного кода
// Сохраняем оригинальный метод запроса
$originalMethod = $_SERVER['REQUEST_METHOD'] ?? 'POST';
// Устанавливаем флаг, чтобы основной код query.php не выполнился
$_SERVER['REQUEST_METHOD'] = 'CLI_TEST';
require_once __DIR__ . '/query.php';
// Восстанавливаем метод запроса
$_SERVER['REQUEST_METHOD'] = $originalMethod;

// Обработка preflight запроса (OPTIONS)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
    
    if ($origin) {
        $db = getDatabase();
        $stmt = $db->prepare("SELECT allowed_domains FROM widget_settings WHERE allowed_domains LIKE ?");
        $stmt->execute(['%' . $origin . '%']);
        $result = $stmt->fetch();
        
        if ($result) {
            header("Access-Control-Allow-Origin: $origin");
            header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
            header("Access-Control-Allow-Headers: Content-Type");
            header("Access-Control-Max-Age: 86400");
            http_response_code(204);
            exit;
        }
    }
    
    http_response_code(403);
    exit;
}

// CORS headers
$origin = $_SERVER['HTTP_ORIGIN'] ?? '';

// Получаем данные запроса
$input = json_decode(file_get_contents('php://input'), true);
$widget_key = $input['widget_key'] ?? '';
$question = trim($input['question'] ?? '');
$section_name = trim($input['section_name'] ?? '');
$categories = $input['categories'] ?? null; // Категории, определенные в stage1 (если переданы)
$categoriesDebug = $input['categories_debug'] ?? null; // Debug информация о категориях
$debugMode = $input['_debug'] ?? false;

if (!$widget_key || !$question || !$section_name) {
    http_response_code(400);
    echo json_encode(['error' => 'widget_key, question and section_name are required']);
    exit;
}

// ============================================================================
// БЫСТРАЯ ОБРАБОТКА АДМИНИСТРАТИВНЫХ ВОПРОСОВ (ДО обращения к базе)
// ============================================================================
$questionLowerAdmin = mb_strtolower(trim($question));

// ============================================================================
// ОБРАБОТКА ЭКСТРЕННЫХ СИТУАЦИЙ
// ============================================================================
// Ситуации, требующие немедленной медицинской помощи
// ИСПРАВЛЕНО: убрали "судорог" - судороги в ногах не экстренная ситуация (эпилептические приступы - да)
// ИСПРАВЛЕНО: добавили исключение для профилактики (инфаркт профилактика, инсульт профилактика - не экстренная ситуация)
// ИСПРАВЛЕНО: убрали "отек.*квинке" и "ангионевротическ.*отек" из экстренных - это может быть информационный запрос, нужны специалисты
if (!preg_match('/(профилактик|предотвращен|предупрежден)/ui', $question) && 
    preg_match('/(сильн.*боль.*груд|давящ.*боль.*серд|инфаркт|инсульт|потер.*сознан|эпилептическ.*припадок|остановк.*дыхан|сильн.*кровотечен|перелом.*позвоночник|травм.*голов|ожог.*больш.*площад|отек.*горл|отек.*лиц.*быстр)/ui', $question)) {
    echo json_encode([
        'text' => '⚠️ ВНИМАНИЕ! Ваши симптомы могут указывать на состояние, требующее экстренной медицинской помощи. НЕМЕДЛЕННО вызовите скорую помощь по телефону 112 или 103!',
        'data' => [
            'specialists' => [],
            'services' => [],
            'articles' => [],
            'specializations' => []
        ],
        'ready' => true,
        'section_name' => $section_name,
        'response_time_ms' => 5
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Ситуации, требующие срочной (но не экстренной) помощи
// ИСПРАВЛЕНО: сделали паттерны более точными, чтобы не ловить хронические состояния
if (preg_match('/(приступ.*удушь|задыхаюсь.*прямо.*сейчас|совсем.*не.*могу.*дышать|посинел.*губ|посинел.*лиц|острая.*синюшн|температур.*выше.*39|высок.*температур.*не.*сбива|непрекращающ.*рвот|рвота.*больше.*суток|острая.*сильн.*боль.*живот|острая.*боль.*(живот|груд)|сильн.*кровь.*мо[чж]|кровь.*стул|кровотечен.*кишечник|анафилактическ)/ui', $question)) {
    echo json_encode([
        'text' => '⚠️ ВАЖНО! Ваши симптомы требуют срочной медицинской консультации. Рекомендуем как можно скорее обратиться в приемное отделение или вызвать скорую помощь (112, 103).',
        'data' => [
            'specialists' => [],
            'services' => [],
            'articles' => [],
            'specializations' => []
        ],
        'ready' => true,
        'warning' => true,
        'section_name' => $section_name,
        'response_time_ms' => 5
    ], JSON_UNESCAPED_UNICODE);
    exit; // Выходим сразу с предупреждением
}

// ИСПРАВЛЕНО: Убрали ранний выход для административных запросов - пусть fallback логика сработает
// Для справок на оружие, больничных, медкнижек - НЕ возвращаем сразу, пусть обработается через AI и fallback
// if (preg_match('/(справк.*оружи|больничн|медицинск.*книжк|санитарн.*книжк)/ui', $question)) {
//     echo json_encode([
//         'text' => 'К сожалению, эта услуга не предоставляется нашей клиникой. Для получения данной справки обратитесь в государственное медицинское учреждение.',
//         'data' => [
//             'specialists' => [],
//             'services' => [],
//             'articles' => [],
//             'specializations' => []
//         ],
//         'ready' => true,
//         'section_name' => $section_name,
//         'response_time_ms' => 10
//     ], JSON_UNESCAPED_UNICODE);
//     exit;
// }

// ИСПРАВЛЕНО: Убрали ранний выход для выписки - пусть fallback логика сработает
// Для выписки из медкарты - административный запрос
// if (preg_match('/(выписк.*медицинск.*карт|копи.*карт|медицинск.*документ)/ui', $question)) {
//     echo json_encode([
//         'text' => 'Для получения выписки из медицинской карты обратитесь в регистратуру клиники с паспортом. Выписка готовится в течение 1-3 рабочих дней.',
//         'data' => [
//             'specialists' => [],
//             'services' => [],
//             'articles' => [],
//             'specializations' => []
//         ],
//         'ready' => true,
//         'section_name' => $section_name,
//         'response_time_ms' => 10
//     ], JSON_UNESCAPED_UNICODE);
//     exit;
// }
// ============================================================================


try {
    $db = getDatabase();
    
    // Получаем виджет и настройки
    $stmt = $db->prepare("
        SELECT w.*, ws.*
        FROM widgets w
        LEFT JOIN widget_settings ws ON w.id = ws.widget_id
        WHERE w.widget_key = ? AND w.active = 1
    ");
    $stmt->execute([$widget_key]);
    $widget = $stmt->fetch();
    
    if (!$widget) {
        http_response_code(404);
        echo json_encode(['error' => 'Widget not found']);
        exit;
    }
    
    // ============================================================================
    // ОБРАБОТКА АДМИНИСТРАТИВНЫХ ВОПРОСОВ С ДАННЫМИ ИЗ БД
    // ============================================================================
    
    // Транспортная комиссия (ГИБДД, водительские права)
    if (preg_match('/(транспортн.*комисс|гибдд|водител|прав.*категор|справка.*водител)/ui', $question)) {
        try {
            $stmtAdmin = $db->prepare("
                SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                FROM parsed_items pi
                JOIN parsed_fields pf ON pi.id = pf.item_id
                WHERE pi.widget_id = ? 
                AND pi.section_name = 'services'
                AND LOWER(pf.field_value) LIKE '%транспортная комиссия%'
                AND pi.is_duplicate = 0
                LIMIT 5
            ");
            $stmtAdmin->execute([$widget['id']]);
            
            $adminItems = [];
            while ($row = $stmtAdmin->fetch()) {
                $itemId = (int)$row['id'];
                if (!isset($adminItems[$itemId])) {
                    $adminItems[$itemId] = ['id' => $itemId];
                }
                $adminItems[$itemId][$row['field_name']] = $row['field_value'];
            }
            
            if (!empty($adminItems)) {
                $services = array_values($adminItems); // Возвращаем полные данные, а не только ID
                
                // Также ищем специалистов для комиссии
                $specialists = [];
                if ($section_name === 'specialists' || $section_name === 'services') {
                    $stmtSpecs = $db->prepare("
                        SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%терапевт%' OR LOWER(pf.field_value) LIKE '%офтальмолог%' OR LOWER(pf.field_value) LIKE '%невролог%')
                        AND pi.is_duplicate = 0
                        LIMIT 20
                    ");
                    $stmtSpecs->execute([$widget['id']]);
                    
                    $specItems = [];
                    while ($row = $stmtSpecs->fetch()) {
                        $itemId = (int)$row['id'];
                        if (!isset($specItems[$itemId])) {
                            $specItems[$itemId] = ['id' => $itemId];
                        }
                        $specItems[$itemId][$row['field_name']] = $row['field_value'];
                    }
                    $specialists = array_values($specItems);
                }
                
                // Специализация "Терапия"
                $specializations = [];
                if ($section_name === 'specializations' || $section_name === 'services') {
                    $stmtSpez = $db->prepare("
                        SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specializations'
                        AND LOWER(pf.field_value) LIKE '%терапия%'
                        AND pi.is_duplicate = 0
                        LIMIT 5
                    ");
                    $stmtSpez->execute([$widget['id']]);
                    
                    $spezItems = [];
                    while ($row = $stmtSpez->fetch()) {
                        $itemId = (int)$row['id'];
                        if (!isset($spezItems[$itemId])) {
                            $spezItems[$itemId] = ['id' => $itemId];
                        }
                        $spezItems[$itemId][$row['field_name']] = $row['field_value'];
                    }
                    $specializations = array_values($spezItems);
                }
                
                echo json_encode([
                    'text' => 'Для прохождения транспортной комиссии вам необходимо пройти осмотр у терапевта, невролога и офтальмолога. Вы можете записаться на комплексную транспортную комиссию.',
                    'data' => [
                        'specialists' => $specialists,
                        'services' => $services,
                        'articles' => [],
                        'specializations' => $specializations
                    ],
                    'ready' => true,
                    'section_name' => $section_name,
                    'response_time_ms' => round((microtime(true) - ($_SERVER['REQUEST_TIME_FLOAT'] ?? microtime(true))) * 1000)
                ], JSON_UNESCAPED_UNICODE);
                exit;
            }
        } catch (Exception $e) {
            // Продолжаем обычную обработку, если что-то пошло не так
        }
    }
    
    // Справки (учеба, бассейн, санаторий)
    if (preg_match('/(справк.*086|справк.*учеб|справк.*бассейн|справк.*санатор|справк.*лагер|справк.*070)/ui', $question)) {
        try {
            $stmtAdmin = $db->prepare("
                SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                FROM parsed_items pi
                JOIN parsed_fields pf ON pi.id = pf.item_id
                WHERE pi.widget_id = ? 
                AND pi.section_name = 'services'
                AND (LOWER(pf.field_value) LIKE '%справка%' AND (LOWER(pf.field_value) LIKE '%086%' OR LOWER(pf.field_value) LIKE '%учеб%' OR LOWER(pf.field_value) LIKE '%бассейн%' OR LOWER(pf.field_value) LIKE '%санатор%' OR LOWER(pf.field_value) LIKE '%070%'))
                AND pi.is_duplicate = 0
                LIMIT 5
            ");
            $stmtAdmin->execute([$widget['id']]);
            
            $adminItems = [];
            while ($row = $stmtAdmin->fetch()) {
                $itemId = (int)$row['id'];
                if (!isset($adminItems[$itemId])) {
                    $adminItems[$itemId] = ['id' => $itemId];
                }
                $adminItems[$itemId][$row['field_name']] = $row['field_value'];
            }
            
            if (!empty($adminItems)) {
                echo json_encode([
                    'text' => 'Мы оформляем медицинские справки для учебы, бассейна и санаториев. Запишитесь на прием к терапевту для прохождения осмотра.',
                    'data' => [
                        'specialists' => [],
                        'services' => array_values($adminItems), // Возвращаем полные данные
                        'articles' => [],
                        'specializations' => []
                    ],
                    'ready' => true,
                    'section_name' => $section_name,
                    'response_time_ms' => round((microtime(true) - ($_SERVER['REQUEST_TIME_FLOAT'] ?? microtime(true))) * 1000)
                ], JSON_UNESCAPED_UNICODE);
                exit;
            }
        } catch (Exception $e) {
            // Продолжаем обычную обработку, если что-то пошло не так
        }
    }
    // ============================================================================

    
    // Проверяем CORS
    $allowed_domains = json_decode($widget['allowed_domains'] ?? '[]', true);
    $own_domain = parse_url(WIDGET_DOMAIN, PHP_URL_HOST);
    
    $request_host = '';
    if ($origin) {
        $request_host = parse_url($origin, PHP_URL_HOST);
    } elseif (isset($_SERVER['HTTP_REFERER'])) {
        $request_host = parse_url($_SERVER['HTTP_REFERER'], PHP_URL_HOST);
    } elseif (isset($_SERVER['HTTP_HOST'])) {
        $request_host = $_SERVER['HTTP_HOST'];
    }
    
    $is_own_domain = false;
    if ($request_host) {
        $is_own_domain = (
            $request_host === $own_domain ||
            strpos($request_host, $own_domain) !== false ||
            strpos($own_domain, $request_host) !== false ||
            (preg_match('/\.medmaps\.ru$/', $request_host) && preg_match('/\.medmaps\.ru$/', $own_domain))
        );
    }
    
    $is_allowed = $debugMode || empty($origin) || $is_own_domain || in_array($origin, $allowed_domains);
    
    if (!empty($allowed_domains) && !$is_allowed && !$debugMode && !$is_own_domain) {
        http_response_code(403);
        echo json_encode(['error' => 'Domain not allowed']);
        exit;
    }
    
    // Устанавливаем CORS headers
    if ($origin) {
        if ($is_own_domain || in_array($origin, $allowed_domains)) {
            header("Access-Control-Allow-Origin: $origin");
            header("Access-Control-Allow-Methods: GET, POST");
            header("Access-Control-Allow-Headers: Content-Type");
        }
    }
    
    // Проверяем, что раздел существует и активен
    $stmt = $db->prepare("
        SELECT * FROM widget_sections 
        WHERE widget_id = ? AND section_name = ? AND is_active = 1
    ");
    $stmt->execute([$widget['id'], $section_name]);
    $section = $stmt->fetch();
    
    if (!$section) {
        http_response_code(404);
        echo json_encode(['error' => "Section '$section_name' not found or inactive"]);
        exit;
    }
    
    $startTime = microtime(true);
    
    // Получаем поля для этого раздела (включая дочерние поля)
    $stmt = $db->prepare("
        SELECT DISTINCT sf.field_name
        FROM widget_sections ws
        JOIN section_fields sf ON ws.id = sf.section_id
        WHERE ws.widget_id = ? AND ws.section_name = ? AND sf.use_in_prompt = 1
        
        UNION
        
        SELECT DISTINCT scf.child_field_name as field_name
        FROM widget_sections ws
        JOIN section_fields sf ON ws.id = sf.section_id
        JOIN section_child_fields scf ON sf.id = scf.field_id
        WHERE ws.widget_id = ? AND ws.section_name = ? AND scf.use_in_prompt = 1
    ");
    $stmt->execute([$widget['id'], $section_name, $widget['id'], $section_name]);
    $fields = [];
    while ($row = $stmt->fetch()) {
        $fields[] = $row['field_name'];
    }
    
    if (empty($fields)) {
        http_response_code(400);
        echo json_encode(['error' => "No fields configured for section '$section_name'"]);
        exit;
    }
    
    // Проверяем, есть ли включенные оптимизации для этого раздела
    $optimizationApplied = false;
    $stmtOpt = $db->prepare("
        SELECT optimization_type, optimized_data, model
        FROM widget_optimizations
        WHERE widget_id = ? AND section_name = ? AND is_enabled = 1
        ORDER BY 
            CASE optimization_type
                WHEN 'prefilter' THEN 1
                WHEN 'compress' THEN 2
                WHEN 'minify' THEN 3
                WHEN 'shorten' THEN 4
                WHEN 'group' THEN 5
            END
    ");
    $stmtOpt->execute([$widget['id'], $section_name]);
    $optimizations = $stmtOpt->fetchAll();
    
    // Определяем основную оптимизацию для отладки (первая в списке)
    $optimization = !empty($optimizations) ? $optimizations[0] : null;
    
    // НОВАЯ СИСТЕМА: Предварительная фильтрация через embeddings (БЕЗ вопроса в OpenRouter)
    $preFilteredIds = null;
    $embeddingFilterEnabled = defined('USE_EMBEDDING_SEARCH') && USE_EMBEDDING_SEARCH; // Используем глобальную настройку
    $maxItemsBeforeFilter = 500; // Если элементов больше, применяем фильтрацию
    
    // Проверяем количество элементов в разделе
    $stmtCount = $db->prepare("
        SELECT COUNT(DISTINCT pi.id) as cnt
        FROM parsed_items pi
        WHERE pi.widget_id = ? AND pi.section_name = ? AND pi.is_duplicate = 0
    ");
    $stmtCount->execute([$widget['id'], $section_name]);
    $itemCount = (int)$stmtCount->fetch()['cnt'];
    
    if ($embeddingFilterEnabled && $itemCount > $maxItemsBeforeFilter) {
        // Используем embedding-фильтрацию для больших разделов
        require_once __DIR__ . '/embedding-functions.php';
        
        logParser("Section '$section_name' has $itemCount items (>$maxItemsBeforeFilter), applying embedding filter...");
        
        // Создаем embedding для вопроса
        $questionEmbedding = createEmbedding($question);
        
        if ($questionEmbedding) {
            // Получаем все embeddings элементов
            $stmtEmb = $db->prepare("
                SELECT ie.item_id, ie.embedding_vector
                FROM item_embeddings ie
                JOIN parsed_items pi ON ie.item_id = pi.id
                WHERE pi.widget_id = ? AND pi.section_name = ? AND pi.is_duplicate = 0
            ");
            $stmtEmb->execute([$widget['id'], $section_name]);
            
            $itemsWithScores = [];
            while ($row = $stmtEmb->fetch()) {
                $itemEmbedding = json_decode($row['embedding_vector'], true);
                if ($itemEmbedding) {
                    // Вычисляем косинусное сходство
                    $similarity = calculateCosineSimilarity($questionEmbedding, $itemEmbedding);
                    $itemsWithScores[] = [
                        'id' => (int)$row['item_id'],
                        'similarity' => $similarity
                    ];
                }
            }
            
            // Сортируем по убыванию similarity
            usort($itemsWithScores, function($a, $b) {
                return $b['similarity'] <=> $a['similarity'];
            });
            
            // Берем топ 50 элементов
            $topN = 50;
            $topItems = array_slice($itemsWithScores, 0, $topN);
            $preFilteredIds = array_map(function($item) { return $item['id']; }, $topItems);
            
            if (!empty($topItems)) {
                $topSimilarities = array_map(function($item) {
                    return round($item['similarity'] * 100, 1) . '%';
                }, array_slice($topItems, 0, 5));
                
                logParser("Embedding filter reduced $itemCount items to " . count($preFilteredIds) . " items (top similarities: " . implode(', ', $topSimilarities) . ")");
            }
        } else {
            logParser("WARNING: Failed to create question embedding, loading all items");
        }
    }
    
    // Загружаем данные только этого раздела (с учетом фильтрации если есть)
    $sql = "
        SELECT pi.id, pf.field_name, pf.field_value
        FROM parsed_items pi
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.widget_id = ? AND pi.section_name = ? 
        AND pf.field_name IN (" . str_repeat('?,', count($fields) - 1) . "?)
        AND pi.is_duplicate = 0";
    
    if ($preFilteredIds !== null && !empty($preFilteredIds)) {
        $sql .= " AND pi.id IN (" . str_repeat('?,', count($preFilteredIds) - 1) . "?)";
    }
    
    $sql .= " ORDER BY pi.parent_item_id, pi.id";
    
    $stmt = $db->prepare($sql);
    
    $params = array_merge([$widget['id'], $section_name], $fields);
    if ($preFilteredIds !== null && !empty($preFilteredIds)) {
        $params = array_merge($params, $preFilteredIds);
    }
    
    $stmt->execute($params);
    
    $items = [];
    while ($row = $stmt->fetch()) {
        if (!isset($items[$row['id']])) {
            $items[$row['id']] = ['id' => (int)$row['id']];
        }
        $items[$row['id']][$row['field_name']] = $row['field_value'];
    }
    
    // Применяем оптимизации, если они включены (в порядке приоритета)
    $optimizedData = null;
    if (!empty($optimizations)) {
        $optimizationApplied = true;
        
        // Применяем все оптимизации последовательно
        foreach ($optimizations as $opt) {
            $optData = json_decode($opt['optimized_data'], true);
            
            // Применяем предварительную фильтрацию (prefilter) - должна быть первой
            if ($opt['optimization_type'] === 'prefilter') {
                // Определяем категории из вопроса пользователя через AI
                require_once __DIR__ . '/optimization-functions.php';
                
                // Сохраняем информацию о фильтрации для debug
                $prefilterInfo = [
                    'original_count' => count($items),
                    'question_categories' => [],
                    'expanded_categories' => [],
                    'filtered_count' => 0,
                    'filtered_ids' => []
                ];
                
                try {
                    // Используем категории, переданные из stage1, или получаем из кеша/определяем здесь
                    if ($categories !== null && is_array($categories)) {
                        // Категории уже определены в stage1, используем их
                        $questionCategories = $categories;
                        $prefilterInfo['question_categories'] = $questionCategories;
                        $prefilterInfo['category_extraction_debug'] = $categoriesDebug;
                        $prefilterInfo['categories_source'] = 'stage1';
                    } else {
                        // Категории не переданы, получаем из кеша или определяем через AI
                        if (!function_exists('getCachedCategories')) {
                            require_once __DIR__ . '/optimization-functions.php';
                        }
                        $categoriesResult = getCachedCategories($widget['id'], $question, $opt['model'], true);
                        $questionCategories = $categoriesResult['categories'];
                        $prefilterInfo['question_categories'] = $questionCategories;
                        $prefilterInfo['category_extraction_debug'] = $categoriesResult['debug'] ?? null;
                        $prefilterInfo['categories_source'] = $categoriesResult['from_cache'] ? 'cache' : 'query-section';
                        $prefilterInfo['from_cache'] = $categoriesResult['from_cache'] ?? false;
                    }
                    
                    if (!empty($questionCategories)) {
                        // AI уже определил все релевантные категории, включая специальности и связанные термины
                        // Просто используем их как есть, без дополнительного расширения
                        $expandedCategories = array_values(array_unique($questionCategories)); // Убираем дубликаты
                        
                        // Исключаем слишком общие категории, которые дают ложные совпадения
                        // Эти категории есть у слишком многих услуг и не помогают фильтрации
                        $excludedGeneralCategories = [
                            'диагностика', 'лечение', 'анализы', 'консультация', 'процедура',
                            'лабораторные анализы', 'обследование', 'исследование'
                        ];
                        
                        // Категории, которые требуют дополнительного подтверждения (должны совпадать с другими специфическими категориями)
                        $requiresConfirmationCategories = [
                            'массаж', 'физиотерапия', 'блокада', 'операция'
                        ];
                        
                        // Специфические категории (симптомы, части тела, специальности) - имеют приоритет
                        $specificCategories = [];
                        $generalCategories = [];
                        
                        foreach ($expandedCategories as $cat) {
                            $catLower = mb_strtolower(trim($cat));
                            // Пропускаем полностью исключенные категории
                            if (in_array($catLower, $excludedGeneralCategories)) {
                                continue;
                            }
                            // Разделяем на специфические и общие
                            if (in_array($catLower, $requiresConfirmationCategories)) {
                                $generalCategories[] = $cat;
                            } else {
                                $specificCategories[] = $cat;
                            }
                        }
                        
                        // Используем специфические категории, если они есть
                        // Общие категории ("массаж", "физиотерапия") исключаем, если есть специфические
                        // Это предотвращает ложные совпадения (например, "гинекологический массаж" при запросе "болит спина")
                        if (!empty($specificCategories)) {
                            $filteredCategories = $specificCategories;
                        } else {
                            // Если нет специфических категорий, используем общие
                            $filteredCategories = $generalCategories;
                        }
                        
                        // Если после фильтрации остались только общие категории, используем их (лучше что-то, чем ничего)
                        if (empty($filteredCategories)) {
                            $filteredCategories = $expandedCategories;
                        }
                        
                        // РАСШИРЕНИЕ КАТЕГОРИЙ для специфичных запросов
                        $questionLower = mb_strtolower($question);
                        
                        // Для бессонницы добавляем психотерапию/психологию/психиатрию
                        if (preg_match('/(бессонн|сон|засыпа|просыпа)/ui', $question)) {
                            $psychoCategories = ['психотерапия', 'психология', 'психотерапевт', 'психиатрия', 'психиатр', 'психолог'];
                            foreach ($psychoCategories as $cat) {
                                if (!in_array($cat, $filteredCategories)) {
                                    $filteredCategories[] = $cat;
                                }
                            }
                        }
                        
                        $prefilterInfo['expanded_categories'] = $expandedCategories;
                        $prefilterInfo['filtered_categories'] = $filteredCategories;
                        $prefilterInfo['specific_categories'] = $specificCategories;
                        $prefilterInfo['general_categories'] = $generalCategories;
                        $prefilterInfo['excluded_general_categories'] = $excludedGeneralCategories;
                        
                        // Получаем ID элементов, у которых есть совпадающие категории
                        // Используем нечеткое совпадение: точное совпадение ИЛИ LIKE для каждого слова из категории
                        $db = getDatabase();
                        
                        // Формируем условия для поиска: точное совпадение + LIKE для каждого слова
                        $conditions = [];
                        $params = [$widget['id'], $section_name];
                        
                        // Синонимы для категорий (для улучшения совпадений)
                        $categorySynonyms = [
                            'дерматолог' => ['дерматовенеролог', 'дерматология', 'дерматовенерология'],
                            'дерматология' => ['дерматовенерология', 'дерматолог', 'дерматовенеролог'],
                            'кожа' => ['дерматология', 'дерматовенерология', 'дерматолог', 'дерматовенеролог']
                        ];
                        
                        foreach ($filteredCategories as $cat) {
                            $cat = trim($cat);
                            if (empty($cat)) continue; // Пропускаем пустые категории
                            
                            $catLower = mb_strtolower($cat);
                            
                            // Точное совпадение
                            $conditions[] = "category = ?";
                            $params[] = $cat;
                            
                            // Добавляем синонимы, если они есть
                            if (isset($categorySynonyms[$catLower])) {
                                foreach ($categorySynonyms[$catLower] as $synonym) {
                                    $conditions[] = "category = ?";
                                    $params[] = $synonym;
                                    $conditions[] = "LOWER(category) LIKE ?";
                                    $params[] = '%' . $synonym . '%';
                                }
                            }
                            
                            // Нечеткое совпадение: разбиваем категорию на слова и ищем каждое слово
                            // НО: исключаем слишком общие слова, которые могут давать ложные совпадения
                            $words = preg_split('/[\s,]+/', mb_strtolower(trim($cat)));
                            $excludedWords = ['после', 'до', 'при', 'для', 'от', 'на', 'в', 'с', 'из', 'к', 'у', 'о', 'об', 'про', 'без', 'над', 'под'];
                            foreach ($words as $word) {
                                $word = trim($word);
                                // Игнорируем слишком короткие слова И исключенные общие слова
                                if (mb_strlen($word) > 2 && !in_array($word, $excludedWords)) {
                                    $conditions[] = "LOWER(category) LIKE ?";
                                    $params[] = '%' . $word . '%';
                                }
                            }
                        }
                        
                        if (empty($conditions)) {
                            // Если нет условий, значит нет категорий для поиска
                            $prefilterInfo['found_item_ids'] = [];
                            $prefilterInfo['found_ids_count'] = 0;
                            $prefilterInfo['note'] = 'No search conditions generated';
                            $prefilterInfo['debug_expanded_categories'] = $expandedCategories;
                            $prefilterInfo['debug_conditions_empty'] = true;
                        } else {
                            $whereClause = implode(' OR ', $conditions);
                            try {
                                $stmt = $db->prepare("
                                    SELECT DISTINCT item_id
                                    FROM item_categories
                                    WHERE widget_id = ? AND section_name = ? 
                                    AND ($whereClause)
                                ");
                                $stmt->execute($params);
                                
                                $filteredItemIds = [];
                                while ($row = $stmt->fetch()) {
                                    $filteredItemIds[] = (int)$row['item_id'];
                                }
                                
                                $prefilterInfo['found_item_ids'] = $filteredItemIds;
                                $prefilterInfo['found_ids_count'] = count($filteredItemIds);
                                $prefilterInfo['sql_conditions_count'] = count($conditions);
                                $prefilterInfo['sql_params_count'] = count($params);
                            } catch (Exception $sqlError) {
                                $prefilterInfo['sql_error'] = $sqlError->getMessage();
                                $prefilterInfo['found_item_ids'] = [];
                                $prefilterInfo['found_ids_count'] = 0;
                                error_log("Prefilter SQL error: " . $sqlError->getMessage());
                            }
                        }
                        
                        // Фильтруем элементы по найденным ID
                        if (!empty($filteredItemIds)) {
                            $filteredItems = [];
                            // $items - это ассоциативный массив, где ключ = ID элемента
                            foreach ($items as $itemId => $item) {
                                if (in_array($itemId, $filteredItemIds)) {
                                    $filteredItems[$itemId] = $item;
                                }
                            }
                            $items = $filteredItems;
                            $prefilterInfo['filtered_count'] = count($items);
                            $prefilterInfo['filtered_ids'] = array_slice($filteredItemIds, 0, 20); // Первые 20 для debug
                        } else {
                            // Если не найдено совпадений, оставляем все элементы (или можно вернуть пустой массив)
                            // Для безопасности оставляем все элементы
                            $prefilterInfo['filtered_count'] = count($items);
                            $prefilterInfo['note'] = 'No matching categories found, all items kept';
                        }
                    }
                } catch (Exception $e) {
                    // В случае ошибки определения категорий, не применяем фильтрацию
                    // Логируем ошибку для отладки
                    error_log("Prefilter optimization error: " . $e->getMessage());
                    $prefilterInfo['error'] = $e->getMessage();
                }
                
                // Сохраняем информацию о фильтрации для использования в debug
                $GLOBALS['prefilter_debug_info'] = $prefilterInfo;
            }
            
            // Применяем сжатие описаний (compress)
            if ($opt['optimization_type'] === 'compress' && isset($optData['compressed_items'])) {
                // Создаем маппинг ID -> сжатый элемент для быстрого поиска
                $compressedMap = [];
                foreach ($optData['compressed_items'] as $compressedItem) {
                    if (isset($compressedItem['id'])) {
                        $compressedMap[$compressedItem['id']] = $compressedItem;
                    }
                }
                
                // Применяем сжатые описания к существующим элементам
                foreach ($items as $itemId => &$item) {
                    if (isset($compressedMap[$itemId])) {
                        $compressed = $compressedMap[$itemId];
                        // Заменяем описание на сжатое
                        if (isset($compressed['description'])) {
                            $item['description'] = $compressed['description'];
                        }
                        if (isset($compressed['name'])) {
                            $item['name'] = $compressed['name'];
                        }
                    }
                }
                unset($item); // Снимаем ссылку
            }
            
            // Применяем минификацию JSON (minify) - это влияет на формат передачи
            if ($opt['optimization_type'] === 'minify' && isset($optData['minified_json'])) {
                // Минифицированный JSON уже готов, используем его вместо формирования нового
                // Но нужно убедиться, что данные соответствуют текущим items
                // Пока что минификация применяется позже, при формировании jsonData
            }
            
            // Применяем сокращение инструкций (shorten) - это влияет на промпт, обработаем ниже
            // Применяем группировку (group) - это влияет на формат передачи, обработаем отдельно
        }
        
        // Фильтрация специалистов по типу запроса
        if ($section_name === 'specialists') {
            $questionLower = mb_strtolower($question);
            
            // ВСЕГДА исключаем лаборантов - они не занимаются лечением пациентов
            $filteredItems = [];
            foreach ($items as $itemId => $item) {
                $description = mb_strtolower($item['description'] ?? '');
                $name = mb_strtolower($item['name'] ?? '');
                
                // Исключаем лаборантов
                if (strpos($description, 'лаборант') !== false || 
                    strpos($description, 'заведующая клинико-диагностической лабораторией') !== false ||
                    strpos($description, 'заведующий клинико-диагностической лабораторией') !== false) {
                    continue;
                }
                
                $filteredItems[$itemId] = $item;
            }
            $items = $filteredItems;
            // Исключаем гинекологов, акушеров, эндокринологов для запросов про частое мочеиспускание
            $isUrinationQuery = (strpos($questionLower, 'мочеиспускание') !== false || 
                                strpos($questionLower, 'мочеиспуска') !== false ||
                                strpos($questionLower, 'недержание мочи') !== false || 
                                strpos($questionLower, 'проблемы с мочеиспусканием') !== false);
            
            if ($isUrinationQuery) {
                $filteredItems = [];
                foreach ($items as $itemId => $item) {
                    $description = mb_strtolower($item['description'] ?? '');
                    $name = mb_strtolower($item['name'] ?? '');
                    $text = $description . ' ' . $name;
                    
                    // Исключаем гинекологов, акушеров, эндокринологов
                    if (strpos($text, 'гинеколог') !== false ||
                        strpos($text, 'акушер') !== false ||
                        strpos($text, 'эндокринолог') !== false) {
                        continue;
                    }
                    
                    $filteredItems[$itemId] = $item;
                }
                $items = $filteredItems;
            }
            
            // Исключаем подологов для дерматологических запросов
            $dermatologyKeywords = ['акне', 'псориаз', 'розацеа', 'зуд кожи', 'экзема', 'дерматит', 'крапивница', 
                                   'угри', 'прыщи', 'сыпь', 'кожа', 'дерматолог', 'дерматология', 'дерматовенеролог'];
            $isDermatologyQuery = false;
            foreach ($dermatologyKeywords as $keyword) {
                if (strpos($questionLower, $keyword) !== false) {
                    $isDermatologyQuery = true;
                    break;
                }
            }
            
            if ($isDermatologyQuery) {
                // Удаляем подологов из списка
                $filteredItems = [];
                $excludedPodologists = [];
                $beforeCount = count($items);
                
                foreach ($items as $itemId => $item) {
                    $description = mb_strtolower($item['description'] ?? '');
                    $name = mb_strtolower($item['name'] ?? '');
                    
                    // Пропускаем подологов
                    if (strpos($description, 'подолог') !== false || strpos($name, 'подолог') !== false ||
                        strpos($description, 'подология') !== false || strpos($name, 'подология') !== false) {
                        $excludedPodologists[] = $itemId;
                        continue;
                    }
                    
                    $filteredItems[$itemId] = $item;
                }
                
                $items = $filteredItems;
                
                // Если после фильтрации подологов список пуст, расширяем поиск
                // Включаем аллергологов и терапевтов из всех доступных специалистов
                if (empty($items)) {
                    // Загружаем всех специалистов виджета
                    $db = getDatabase();
                    $stmt = $db->prepare("
                        SELECT pi.id, pf.field_name, pf.field_value
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specialists'
                        AND pi.is_duplicate = 0
                    ");
                    $stmt->execute([$widget['id']]);
                    
                    $allSpecialists = [];
                    while ($row = $stmt->fetch()) {
                        if (!isset($allSpecialists[$row['id']])) {
                            $allSpecialists[$row['id']] = ['id' => (int)$row['id']];
                        }
                        $allSpecialists[$row['id']][$row['field_name']] = $row['field_value'];
                    }
                    
                    // Фильтруем: берем аллергологов и терапевтов, исключаем подологов
                    foreach ($allSpecialists as $itemId => $item) {
                        $description = mb_strtolower($item['description'] ?? '');
                        $name = mb_strtolower($item['name'] ?? '');
                        
                        // Пропускаем подологов
                        if (strpos($description, 'подолог') !== false || strpos($name, 'подолог') !== false ||
                            strpos($description, 'подология') !== false || strpos($name, 'подология') !== false) {
                            continue;
                        }
                        
                        // Берем аллергологов, иммунологов или терапевтов
                        if (strpos($description, 'аллерголог') !== false || strpos($description, 'иммунолог') !== false ||
                            strpos($description, 'терапевт') !== false) {
                            $items[$itemId] = $item;
                        }
                    }
                    
                }
            }
        }
        
        // Фильтрация услуг (services) - исключаем нерелевантные услуги ДО передачи Claude
        if ($section_name === 'services') {
            $filteredItems = [];
            
            // Список нерелевантных ключевых слов для исключения
            $excludeKeywords = [
                'транспортная комиссия',
                'для транспортной комиссии',
                'биологический риск',
                'гормональных контрацептивов',
                'для психоневрологического интерната',
                'медицинская карта для',
                'справка на оружие',
                'справка для бассейна',
                'перед вакцинацией',
                'перед проведением комплекса процедур'
            ];
            
            // Для ОРВИ исключаем нерелевантные услуги
            $questionLower = mb_strtolower($question);
            if (preg_match('/(орви|орз|простуда|грипп|респираторн.*инфекц)/ui', $questionLower)) {
                $excludeKeywords = array_merge($excludeKeywords, [
                    'герпес',
                    'ревматоидн.*артрит',
                    'лямблиоз',
                    'описторхоз',
                    'лейшманиоз',
                    'паразит',
                    'гельминт'
                ]);
            }
            
            foreach ($items as $itemId => $item) {
                $name = mb_strtolower($item['name'] ?? $item['title'] ?? '');
                
                // Проверяем наличие нерелевантных ключевых слов (поддерживаем regex)
                $shouldExclude = false;
                foreach ($excludeKeywords as $keyword) {
                    // Если ключевое слово содержит специальные символы regex, используем preg_match
                    if (preg_match('/[.*+?^${}()|[\]\\\]/', $keyword)) {
                        if (preg_match('/' . $keyword . '/ui', $name)) {
                            $shouldExclude = true;
                            break;
                        }
                    } else {
                        if (strpos($name, $keyword) !== false) {
                            $shouldExclude = true;
                            break;
                        }
                    }
                }
                
                if (!$shouldExclude) {
                    $filteredItems[$itemId] = $item;
                }
            }
            
            $items = $filteredItems;
        }
        
        // Сохраняем данные первой оптимизации для отладки
        if ($optimization) {
            $optimizedData = json_decode($optimization['optimized_data'], true);
        }
    }
    
    // Программная проверка: для "повышенная температура" всегда включаем специализацию "Терапия"
    // Для бессонницы всегда включаем специализацию "Неврология"
    if ($section_name === 'specializations') {
        $questionLower = mb_strtolower($question);
        $isTemperatureQuery = (strpos($questionLower, 'температур') !== false || 
                               strpos($questionLower, 'лихорадк') !== false ||
                               strpos($questionLower, 'температура 38') !== false ||
                               strpos($questionLower, 'высокая температура') !== false ||
                               strpos($questionLower, 'повышенная температура') !== false);
        
        $isInsomniaQuery = (strpos($questionLower, 'бессонн') !== false || 
                           strpos($questionLower, 'бессонниц') !== false ||
                           preg_match('/(проблем.*сон|нарушен.*сон|плох.*сон|не.*могу.*заснуть|трудно.*заснуть)/ui', $questionLower));
        
        $isCardiacQuery = preg_match('/(одышк|боль.*груд|сердц|давлен|гипертон|аритми|тахикарди|холестерин|экг|электрокардиограф)/ui', $question);
        $isAppetiteQuery = preg_match('/(потеря.*аппетит|аппетит.*потеря|нет.*аппетит|плох.*аппетит)/ui', $question);
        
        // Административные запросы (справки, медосмотр, комиссия, диспансеризация)
        $isAdminQuery = preg_match('/(справк|медосмотр|медицинск.*осмотр|комисс|диспансеризац|профосмотр)/ui', $question);
        
        // Онемение/покалывание - невролог + кардиология + эндокринология
        $isNumbnessQuery = preg_match('/(онемен|покалыван|мурашк|жжен|жжени)/ui', $question);
        
        // Диагностические запросы (ЭКГ, УЗИ, анализы, холтер и т.д.)
        $isDiagnosticQuery = preg_match('/(экг|электрокардиограф|холтер|узи|ультразвук|анализ|мазок|рентген|мрт|кт|компьютерн.*томограф|магнитн.*резонанс)/ui', $question);
        
        // Для административных запросов добавляем специализацию "Справки и комиссии"
        if ($isAdminQuery) {
            try {
                $db = getDatabase();
                $stmtAdmin = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%справк%' OR LOWER(pf.field_value) LIKE '%комисс%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtAdmin->execute([$widget['id']]);
                
                $adminItem = [];
                while ($row = $stmtAdmin->fetch()) {
                    if (empty($adminItem)) {
                        $adminItem = ['id' => (int)$row['id']];
                    }
                    $adminItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($adminItem) && !isset($items[$adminItem['id']])) {
                    $items[$adminItem['id']] = $adminItem;
                    logParser("DEBUG specializations: Added 'Справки и комиссии' specialization for admin query '{$question}'");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding admin specialization: " . $e->getMessage());
            }
        }
        
        // Для онемения/покалывания добавляем невролог + кардиология + эндокринология
        if ($isNumbnessQuery) {
            // Невролог
            $neurologyId = 5925;
            if (!isset($items[$neurologyId])) {
                $db = getDatabase();
                $stmtNeurology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND pi.id = ?
                    AND pi.is_duplicate = 0
                ");
                $stmtNeurology->execute([$widget['id'], $neurologyId]);
                
                $neurologyItem = [];
                while ($row = $stmtNeurology->fetch()) {
                    if (empty($neurologyItem)) {
                        $neurologyItem = ['id' => (int)$row['id']];
                    }
                    $neurologyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($neurologyItem)) {
                    $items[$neurologyId] = $neurologyItem;
                    logParser("DEBUG specializations: Added 'Невролог' specialization for numbness query '{$question}'");
                }
            }
            
            // Кардиология
            try {
                $db = getDatabase();
                $stmtCardiology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%кардиолог%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtCardiology->execute([$widget['id']]);
                
                $cardiologyItem = [];
                while ($row = $stmtCardiology->fetch()) {
                    if (empty($cardiologyItem)) {
                        $cardiologyItem = ['id' => (int)$row['id']];
                    }
                    $cardiologyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($cardiologyItem) && !isset($items[$cardiologyItem['id']])) {
                    $items[$cardiologyItem['id']] = $cardiologyItem;
                    logParser("DEBUG specializations: Added 'Кардиология' specialization for numbness query '{$question}'");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding cardiology for numbness: " . $e->getMessage());
            }
            
            // Эндокринология
            try {
                $db = getDatabase();
                $stmtEndocrinology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%эндокринолог%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtEndocrinology->execute([$widget['id']]);
                
                $endocrinologyItem = [];
                while ($row = $stmtEndocrinology->fetch()) {
                    if (empty($endocrinologyItem)) {
                        $endocrinologyItem = ['id' => (int)$row['id']];
                    }
                    $endocrinologyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($endocrinologyItem) && !isset($items[$endocrinologyItem['id']])) {
                    $items[$endocrinologyItem['id']] = $endocrinologyItem;
                    logParser("DEBUG specializations: Added 'Эндокринология' specialization for numbness query '{$question}'");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding endocrinology for numbness: " . $e->getMessage());
            }
        }
        
        // Для диагностических запросов добавляем соответствующие специализации
        if ($isDiagnosticQuery) {
            // ЭКГ, Холтер → Кардиология + Терапия
            if (preg_match('/(экг|электрокардиограф|холтер)/ui', $question)) {
                $cardiologyId = null;
                $therapyId = 5908;
                
                // Ищем Кардиологию
                $db = getDatabase();
                $stmtCardiology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%кардиолог%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtCardiology->execute([$widget['id']]);
                
                $cardiologyItem = [];
                while ($row = $stmtCardiology->fetch()) {
                    if (empty($cardiologyItem)) {
                        $cardiologyItem = ['id' => (int)$row['id']];
                    }
                    $cardiologyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($cardiologyItem) && !isset($items[$cardiologyItem['id']])) {
                    $items[$cardiologyItem['id']] = $cardiologyItem;
                    logParser("DEBUG specializations: Added 'Кардиология' specialization for ECG/Holter query '{$question}'");
                }
                
                // Добавляем Терапию
                if (!isset($items[$therapyId])) {
                    $stmtTherapy = $db->prepare("
                        SELECT pi.id, pf.field_name, pf.field_value
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND pi.id = ?
                        AND pi.is_duplicate = 0
                    ");
                    $stmtTherapy->execute([$widget['id'], $therapyId]);
                    
                    $therapyItem = [];
                    while ($row = $stmtTherapy->fetch()) {
                        if (empty($therapyItem)) {
                            $therapyItem = ['id' => (int)$row['id']];
                        }
                        $therapyItem[$row['field_name']] = $row['field_value'];
                    }
                    
                    if (!empty($therapyItem)) {
                        $items[$therapyId] = $therapyItem;
                        logParser("DEBUG specializations: Added 'Терапия' specialization for ECG/Holter query '{$question}'");
                    }
                }
            }
        }
        
        // Добавляем специализацию "Терапия" для температуры, кардиологических запросов и потери аппетита
        if ($isTemperatureQuery || $isCardiacQuery || $isAppetiteQuery) {
            // Загружаем специализацию "Терапия" (ID 5908), если её нет в списке
            $therapyId = 5908;
            if (!isset($items[$therapyId])) {
                $db = getDatabase();
                $stmt = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND pi.id = ?
                    AND pi.is_duplicate = 0
                ");
                $stmt->execute([$widget['id'], $therapyId]);
                
                $therapyItem = [];
                while ($row = $stmt->fetch()) {
                    if (empty($therapyItem)) {
                        $therapyItem = ['id' => (int)$row['id']];
                    }
                    $therapyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($therapyItem)) {
                    $items[$therapyId] = $therapyItem;
                    $queryType = $isTemperatureQuery ? 'temperature' : ($isCardiacQuery ? 'cardiac' : 'appetite');
                    logParser("DEBUG specializations: Added 'Терапия' specialization for {$queryType} query '{$question}'");
                }
            }
        }
        
        if ($isInsomniaQuery) {
            // Загружаем специализацию "Невролог" (ID 5925), всегда добавляем для бессонницы
            $neurologyId = 5925;
            // Проверяем, есть ли уже эта специализация в списке
            if (!isset($items[$neurologyId])) {
                $db = getDatabase();
                $stmt = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND pi.id = ?
                    AND pi.is_duplicate = 0
                ");
                $stmt->execute([$widget['id'], $neurologyId]);
                
                $neurologyItem = [];
                while ($row = $stmt->fetch()) {
                    if (empty($neurologyItem)) {
                        $neurologyItem = ['id' => (int)$row['id']];
                    }
                    $neurologyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($neurologyItem)) {
                    $items[$neurologyId] = $neurologyItem;
                }
            }
        }
        
        // Для "акне": добавить "Косметология", убрать "Неврология" и "Аллергология"
        $isAcneQuery = preg_match('/(акне|угр|прыщ)/ui', $question);
        if ($isAcneQuery) {
            // Добавляем "Косметология" (нужно найти ID)
            try {
                $db = getDatabase();
                $stmtCosmetology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%косметолог%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtCosmetology->execute([$widget['id']]);
                
                $cosmetologyItem = [];
                while ($row = $stmtCosmetology->fetch()) {
                    if (empty($cosmetologyItem)) {
                        $cosmetologyItem = ['id' => (int)$row['id']];
                    }
                    $cosmetologyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($cosmetologyItem) && !isset($items[$cosmetologyItem['id']])) {
                    $items[$cosmetologyItem['id']] = $cosmetologyItem;
                    logParser("DEBUG specializations: Added 'Косметология' specialization for acne query '{$question}'");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding cosmetology: " . $e->getMessage());
            }
            
            // Убираем нерелевантные специализации для акне
            foreach ($items as $itemId => $item) {
                $itemName = mb_strtolower($item['name'] ?? '');
                // Убираем Онколог, Педиатрию (если не детский вопрос), Кардиологию для акне
                if ((strpos($itemName, 'онколог') !== false || 
                     strpos($itemName, 'педиатр') !== false && !preg_match('/(ребенок|детск)/ui', $question) ||
                     strpos($itemName, 'кардиолог') !== false) &&
                    strpos($itemName, 'косметолог') === false &&
                    strpos($itemName, 'дерматолог') === false) {
                    unset($items[$itemId]);
                    logParser("DEBUG specializations: Removed '{$itemName}' specialization for acne query (irrelevant)");
                }
            }
        }
        
        // ФИЛЬТРАЦИЯ НЕРЕЛЕВАНТНЫХ СПЕЦИАЛИЗАЦИЙ: убираем "Онколог" для нерелевантных запросов
        $isOncologyQuery = preg_match('/(онколог|рак|опухол|новообразовани|метастаз|канцер|злокачественн)/ui', $question);
        if (!$isOncologyQuery) {
            // Убираем Онколог для всех запросов, кроме онкологических
            foreach ($items as $itemId => $item) {
                $itemName = mb_strtolower($item['name'] ?? '');
                if (strpos($itemName, 'онколог') !== false) {
                    unset($items[$itemId]);
                    logParser("DEBUG specializations: Removed 'Онколог' specialization for non-oncology query '{$question}'");
                }
            }
        }
        
        // ФИЛЬТРАЦИЯ: убираем нерелевантные специализации для астмы, инфаркта, сухости глаз
        if (preg_match('/(астм|бронхиальн.*астм|инфаркт|сухост.*глаз|сух.*глаз)/ui', $question)) {
            foreach ($items as $itemId => $item) {
                $itemName = mb_strtolower($item['name'] ?? '');
                // Для астмы убираем Онколог и Кардиологию (астма - легочное заболевание, не кардиологическое)
                if (preg_match('/(астм|бронхиальн.*астм)/ui', $question)) {
                    if (strpos($itemName, 'онколог') !== false) {
                        unset($items[$itemId]);
                        logParser("DEBUG specializations: Removed 'Онколог' specialization for asthma query '{$question}'");
                    }
                    if (strpos($itemName, 'кардиолог') !== false) {
                        unset($items[$itemId]);
                        logParser("DEBUG specializations: Removed 'Кардиология' specialization for asthma query '{$question}' (asthma is pulmonary, not cardiac)");
                    }
                }
                // Для инфаркта убираем Онколог
                if (preg_match('/инфаркт/ui', $question) && strpos($itemName, 'онколог') !== false) {
                    unset($items[$itemId]);
                    logParser("DEBUG specializations: Removed 'Онколог' specialization for infarction query '{$question}'");
                }
            }
        }
        
        // Для "ожог": добавить "Хирургия", убрать "Травматология"
        $isBurnQuery = preg_match('/(ожог|ожог)/ui', $question);
        if ($isBurnQuery) {
            // Добавляем "Хирургия"
            try {
                $db = getDatabase();
                $stmtSurgery = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%хирург%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtSurgery->execute([$widget['id']]);
                
                $surgeryItem = [];
                while ($row = $stmtSurgery->fetch()) {
                    if (empty($surgeryItem)) {
                        $surgeryItem = ['id' => (int)$row['id']];
                    }
                    $surgeryItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($surgeryItem) && !isset($items[$surgeryItem['id']])) {
                    $items[$surgeryItem['id']] = $surgeryItem;
                    logParser("DEBUG specializations: Added 'Хирургия' specialization for burn query '{$question}'");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding surgery: " . $e->getMessage());
            }
        }
        
        // ФИЛЬТРАЦИЯ СПЕЦИАЛИЗАЦИЙ: Убираем нерелевантные специализации
        $filteredItems = [];
        foreach ($items as $id => $item) {
            $name = mb_strtolower($item['name'] ?? $item['title'] ?? '');
            $shouldInclude = true;
            
            // Для "акне": убрать "Неврология" и "Аллергология"
            if ($isAcneQuery) {
                if (strpos($name, 'невролог') !== false || strpos($name, 'аллерголог') !== false) {
                    $shouldInclude = false;
                    logParser("DEBUG specializations filter: Excluding '{$name}' for acne query");
                }
            }
            
            // Для "ожог": убрать "Травматология"
            if ($isBurnQuery) {
                if (strpos($name, 'травматолог') !== false) {
                    $shouldInclude = false;
                    logParser("DEBUG specializations filter: Excluding '{$name}' for burn query");
                }
            }
            
            if ($shouldInclude) {
                $filteredItems[$id] = $item;
            }
        }
        
        if (count($filteredItems) !== count($items)) {
            logParser("DEBUG specializations filter: Filtered " . (count($items) - count($filteredItems)) . " specializations for query '{$question}'");
            $items = $filteredItems;
        }
    }
    
    // Функция очистки одного элемента от ссылок на изображения
    $cleanItemFromImages = function(&$item) {
        // Очищаем поле description
        if (isset($item['description']) && is_string($item['description'])) {
            $item['description'] = preg_replace('/(https?:\/\/[^\s\)]+|www\.[^\s\)]+)/i', '', $item['description']);
            $item['description'] = preg_replace('/\.(jpg|jpeg|png|gif|webp|svg|runtuit)[^\s\)]*/i', '', $item['description']);
            $item['description'] = preg_replace('/<image:[^>]*>/i', '', $item['description']);
            $item['description'] = preg_replace('/<img[^>]*>/i', '', $item['description']);
            // Удаляем пути к изображениям
            $item['description'] = preg_replace('/\/upload\/[^\s\)]+/i', '', $item['description']);
            $item['description'] = preg_replace('/\/resize_cache\/[^\s\)]+/i', '', $item['description']);
            $item['description'] = preg_replace('/ekamedcenter\.ru[^\s\)]*/i', '', $item['description']);
            // Удаляем дисклеймеры из описаний
            $item['description'] = preg_replace('/\s*Имеются\s+противопоказания[^.]*\./ui', '', $item['description']);
            $item['description'] = preg_replace('/\s*Ознакомьтесь\s+с\s+инструкцией[^.]*\./ui', '', $item['description']);
            $item['description'] = preg_replace('/\s*Проконсультируйтесь\s+у\s+специалиста[^.]*\./ui', '', $item['description']);
            $item['description'] = preg_replace('/\s*Имеются\s+противопоказания\.\s*Ознакомьтесь\s+с\s+инструкцией\s+или\s+проконсультируйтесь\s+у\s+специалиста\./ui', '', $item['description']);
            $item['description'] = trim($item['description']);
        }
        // Очищаем поле title
        if (isset($item['title']) && is_string($item['title'])) {
            $item['title'] = preg_replace('/(https?:\/\/[^\s\)]+|www\.[^\s\)]+)/i', '', $item['title']);
            $item['title'] = preg_replace('/\.(jpg|jpeg|png|gif|webp|svg|runtuit)[^\s\)]*/i', '', $item['title']);
            $item['title'] = preg_replace('/\/upload\/[^\s\)]+/i', '', $item['title']);
            $item['title'] = preg_replace('/\/resize_cache\/[^\s\)]+/i', '', $item['title']);
            $item['title'] = trim($item['title']);
        }
        // Очищаем поле name
        if (isset($item['name']) && is_string($item['name'])) {
            $item['name'] = preg_replace('/(https?:\/\/[^\s\)]+|www\.[^\s\)]+)/i', '', $item['name']);
            $item['name'] = preg_replace('/\.(jpg|jpeg|png|gif|webp|svg|runtuit)[^\s\)]*/i', '', $item['name']);
            $item['name'] = preg_replace('/\/upload\/[^\s\)]+/i', '', $item['name']);
            $item['name'] = preg_replace('/\/resize_cache\/[^\s\)]+/i', '', $item['name']);
            $item['name'] = trim($item['name']);
        }
        // Удаляем поле photo полностью (включая все варианты названий)
        foreach (['photo', 'photo_url', 'photoUrl', 'Photo', 'PHOTO'] as $field) {
            if (isset($item[$field])) {
                unset($item[$field]);
            }
        }
        // Очищаем другие поля, которые могут содержать ссылки на изображения
        foreach (['image', 'image_url', 'img', 'picture', 'thumbnail', 'preview', 'foto', 'Image', 'IMAGE', 'img_url', 'Img'] as $field) {
            if (isset($item[$field])) {
                // Если это строка, очищаем её от ссылок, иначе удаляем
                if (is_string($item[$field])) {
                    $item[$field] = preg_replace('/(https?:\/\/[^\s\)]+|www\.[^\s\)]+)/i', '', $item[$field]);
                    $item[$field] = preg_replace('/\/upload\/[^\s\)]+/i', '', $item[$field]);
                    $item[$field] = preg_replace('/\/resize_cache\/[^\s\)]+/i', '', $item[$field]);
                    $item[$field] = preg_replace('/\.(jpg|jpeg|png|gif|webp|svg|runtuit)[^\s\)]*/i', '', $item[$field]);
                    $item[$field] = trim($item[$field]);
                    // Если после очистки поле пустое, удаляем его
                    if (empty($item[$field])) {
                        unset($item[$field]);
                    }
                } else {
                    unset($item[$field]);
                }
            }
        }
    };
    
    // Применяем очистку ко всем разделам
    foreach ($items as $id => &$item) {
        $cleanItemFromImages($item);
    }
    unset($item); // Снимаем ссылку
    
    // Формируем JSON данные только для этого раздела
    $sectionData = [$section_name => array_values($items)];
    
    // ДОПОЛНИТЕЛЬНОЕ ДОБАВЛЕНИЕ СПЕЦИАЛИЗАЦИЙ: Добавляем нужные специализации даже если AI их не выбрал
    if ($section_name === 'specializations') {
        $questionLower = mb_strtolower($question);
        $existingIds = array_column($sectionData[$section_name], 'id');
        
        // Для "Одышка при нагрузке" и других кардиологических запросов: добавляем "Терапия"
        $isCardiacQuery = preg_match('/(одышк|боль.*груд|сердц|давлен|гипертон|аритми|тахикарди)/ui', $question);
        if ($isCardiacQuery && !in_array(5908, $existingIds)) {
            try {
                $db = getDatabase();
                $stmt = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND pi.id = 5908
                    AND pi.is_duplicate = 0
                ");
                $stmt->execute([$widget['id']]);
                
                $therapyItem = [];
                while ($row = $stmt->fetch()) {
                    if (empty($therapyItem)) {
                        $therapyItem = ['id' => (int)$row['id']];
                    }
                    $therapyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($therapyItem)) {
                    $cleanItemFromImages($therapyItem);
                    $sectionData[$section_name][] = $therapyItem;
                    logParser("DEBUG specializations: Added 'Терапия' specialization after sectionData formation for cardiac query");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding therapy after sectionData: " . $e->getMessage());
            }
        }
        
        // Для "акне": добавляем "Косметология", если её нет
        $isAcneQuery = preg_match('/(акне|угр|прыщ)/ui', $question);
        if ($isAcneQuery) {
            try {
                $db = getDatabase();
                $stmtCosmetology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%косметолог%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtCosmetology->execute([$widget['id']]);
                
                $cosmetologyItem = [];
                $cosmetologyId = null;
                while ($row = $stmtCosmetology->fetch()) {
                    if (empty($cosmetologyItem)) {
                        $cosmetologyId = (int)$row['id'];
                        $cosmetologyItem = ['id' => $cosmetologyId];
                    }
                    $cosmetologyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($cosmetologyItem) && !in_array($cosmetologyId, $existingIds)) {
                    $cleanItemFromImages($cosmetologyItem);
                    $sectionData[$section_name][] = $cosmetologyItem;
                    logParser("DEBUG specializations: Added 'Косметология' specialization after sectionData formation for acne query");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding cosmetology after sectionData: " . $e->getMessage());
            }
        }
        
        // Для "ожог": добавляем "Хирургия", если её нет
        $isBurnQuery = preg_match('/(ожог|ожог)/ui', $question);
        if ($isBurnQuery) {
            try {
                $db = getDatabase();
                $stmtSurgery = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%хирург%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtSurgery->execute([$widget['id']]);
                
                $surgeryItem = [];
                $surgeryId = null;
                while ($row = $stmtSurgery->fetch()) {
                    if (empty($surgeryItem)) {
                        $surgeryId = (int)$row['id'];
                        $surgeryItem = ['id' => $surgeryId];
                    }
                    $surgeryItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($surgeryItem) && !in_array($surgeryId, $existingIds)) {
                    $cleanItemFromImages($surgeryItem);
                    $sectionData[$section_name][] = $surgeryItem;
                    logParser("DEBUG specializations: Added 'Хирургия' specialization after sectionData formation for burn query");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding surgery after sectionData: " . $e->getMessage());
            }
        }
        
        // Для административных запросов добавляем специализацию "Справки и комиссии"
        $isAdminQuery = preg_match('/(справк|медосмотр|медицинск.*осмотр|комисс|диспансеризац|профосмотр)/ui', $question);
        if ($isAdminQuery) {
            try {
                $db = getDatabase();
                $stmtAdmin = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%справк%' OR LOWER(pf.field_value) LIKE '%комисс%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtAdmin->execute([$widget['id']]);
                
                $adminItem = [];
                $adminId = null;
                while ($row = $stmtAdmin->fetch()) {
                    if (empty($adminItem)) {
                        $adminId = (int)$row['id'];
                        $adminItem = ['id' => $adminId];
                    }
                    $adminItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($adminItem) && !in_array($adminId, $existingIds)) {
                    $cleanItemFromImages($adminItem);
                    $sectionData[$section_name][] = $adminItem;
                    logParser("DEBUG specializations: Added 'Справки и комиссии' specialization after sectionData formation for admin query '{$question}'");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding admin specialization after sectionData: " . $e->getMessage());
            }
        }
        
        // Для онемения/покалывания добавляем кардиология + эндокринология (невролог уже может быть от AI)
        $isNumbnessQuery = preg_match('/(онемен|покалыван|мурашк|жжен|жжени)/ui', $question);
        if ($isNumbnessQuery) {
            logParser("DEBUG specializations: Processing numbness query '{$question}', existingIds: " . implode(',', $existingIds));
            // Кардиология
            try {
                $db = getDatabase();
                $stmtCardiology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%кардиолог%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtCardiology->execute([$widget['id']]);
                
                $cardiologyItem = [];
                $cardiologyId = null;
                while ($row = $stmtCardiology->fetch()) {
                    if (empty($cardiologyItem)) {
                        $cardiologyId = (int)$row['id'];
                        $cardiologyItem = ['id' => $cardiologyId];
                    }
                    $cardiologyItem[$row['field_name']] = $row['field_value'];
                }
                
                logParser("DEBUG specializations: Found cardiology item: " . ($cardiologyId ?? 'NULL') . ", empty: " . (empty($cardiologyItem) ? 'YES' : 'NO') . ", in existingIds: " . (in_array($cardiologyId, $existingIds) ? 'YES' : 'NO'));
                
                if (!empty($cardiologyItem) && !in_array($cardiologyId, $existingIds)) {
                    $cleanItemFromImages($cardiologyItem);
                    $sectionData[$section_name][] = $cardiologyItem;
                    logParser("DEBUG specializations: Added 'Кардиология' specialization after sectionData formation for numbness query '{$question}'");
                } else {
                    logParser("DEBUG specializations: Skipped adding cardiology: empty=" . (empty($cardiologyItem) ? 'YES' : 'NO') . ", in existingIds=" . (in_array($cardiologyId, $existingIds) ? 'YES' : 'NO'));
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding cardiology after sectionData: " . $e->getMessage());
            }
            
            // Эндокринология
            try {
                $db = getDatabase();
                $stmtEndocrinology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%эндокринолог%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtEndocrinology->execute([$widget['id']]);
                
                $endocrinologyItem = [];
                $endocrinologyId = null;
                while ($row = $stmtEndocrinology->fetch()) {
                    if (empty($endocrinologyItem)) {
                        $endocrinologyId = (int)$row['id'];
                        $endocrinologyItem = ['id' => $endocrinologyId];
                    }
                    $endocrinologyItem[$row['field_name']] = $row['field_value'];
                }
                
                logParser("DEBUG specializations: Found endocrinology item: " . ($endocrinologyId ?? 'NULL') . ", empty: " . (empty($endocrinologyItem) ? 'YES' : 'NO') . ", in existingIds: " . (in_array($endocrinologyId, $existingIds) ? 'YES' : 'NO'));
                
                if (!empty($endocrinologyItem) && !in_array($endocrinologyId, $existingIds)) {
                    $cleanItemFromImages($endocrinologyItem);
                    $sectionData[$section_name][] = $endocrinologyItem;
                    logParser("DEBUG specializations: Added 'Эндокринология' specialization after sectionData formation for numbness query '{$question}'");
                } else {
                    logParser("DEBUG specializations: Skipped adding endocrinology: empty=" . (empty($endocrinologyItem) ? 'YES' : 'NO') . ", in existingIds=" . (in_array($endocrinologyId, $existingIds) ? 'YES' : 'NO'));
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding endocrinology after sectionData: " . $e->getMessage());
            }
        }
        
        // Фильтруем нерелевантные специализации
        $filteredSpecializations = [];
        foreach ($sectionData[$section_name] as $spec) {
            $name = mb_strtolower($spec['name'] ?? $spec['title'] ?? '');
            $shouldInclude = true;
            
            // Для "акне": убрать "Неврология" и "Аллергология"
            if ($isAcneQuery) {
                if (strpos($name, 'невролог') !== false || strpos($name, 'аллерголог') !== false) {
                    $shouldInclude = false;
                }
            }
            
            // Для "ожог": убрать "Травматология"
            if ($isBurnQuery) {
                if (strpos($name, 'травматолог') !== false) {
                    $shouldInclude = false;
                }
            }
            
            if ($shouldInclude) {
                $filteredSpecializations[] = $spec;
            }
        }
        
        if (count($filteredSpecializations) !== count($sectionData[$section_name])) {
            logParser("DEBUG specializations filter: Filtered " . (count($sectionData[$section_name]) - count($filteredSpecializations)) . " specializations after sectionData formation");
            $sectionData[$section_name] = $filteredSpecializations;
        }
        
        // Очищаем все специализации от ссылок на изображения после всех операций
        foreach ($sectionData[$section_name] as &$spec) {
            $cleanItemFromImages($spec);
        }
        unset($spec);
    }
    
    // Финальная очистка всех разделов от ссылок на изображения
    foreach ($sectionData as $sectionKey => &$sectionItems) {
        foreach ($sectionItems as &$item) {
            $cleanItemFromImages($item);
        }
        unset($item);
    }
    unset($sectionItems);
    
    // Применяем минификацию JSON, если она включена (после формирования данных)
    $minifyApplied = false;
    if (!empty($optimizations)) {
        foreach ($optimizations as $opt) {
            if ($opt['optimization_type'] === 'minify') {
                $minifyApplied = true;
                // Минифицируем текущие данные: заменяем длинные ключи на короткие
                $minifiedSectionData = [];
                foreach ($sectionData[$section_name] as $item) {
                    $minifiedItem = [];
                    foreach ($item as $key => $value) {
                        // Заменяем ключи на короткие
                        $shortKey = match($key) {
                            'id' => 'i',
                            'description' => 'd',
                            'name' => 'n',
                            'title' => 't',
                            default => $key
                        };
                        $minifiedItem[$shortKey] = $value;
                    }
                    $minifiedSectionData[] = $minifiedItem;
                }
                // Используем короткий ключ для раздела
                $sectionKey = match($section_name) {
                    'specialists' => 's',
                    'services' => 'sv',
                    'articles' => 'a',
                    'specializations' => 'sp',
                    default => $section_name
                };
                $sectionData = [$sectionKey => $minifiedSectionData];
                break; // Применяем только первую найденную минификацию
            }
        }
    }
    
    // ДОПОЛНИТЕЛЬНОЕ ДОБАВЛЕНИЕ СПЕЦИАЛИЗАЦИЙ ПОСЛЕ МИНИФИКАЦИИ: Добавляем нужные специализации даже если AI их не выбрал
    if ($section_name === 'specializations') {
        $questionLower = mb_strtolower($question);
        $sectionKey = $minifyApplied ? 'sp' : 'specializations';
        $existingItems = $sectionData[$sectionKey] ?? [];
        $existingIds = array_column($existingItems, $minifyApplied ? 'i' : 'id');
        
        // Для "Одышка при нагрузке" и других кардиологических запросов: добавляем "Терапия"
        $isCardiacQuery = preg_match('/(одышк|боль.*груд|сердц|давлен|гипертон|аритми|тахикарди)/ui', $question);
        if ($isCardiacQuery && !in_array(5908, $existingIds)) {
            try {
                $db = getDatabase();
                $stmt = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND pi.id = 5908
                    AND pi.is_duplicate = 0
                ");
                $stmt->execute([$widget['id']]);
                
                $therapyItem = [];
                while ($row = $stmt->fetch()) {
                    if (empty($therapyItem)) {
                        $therapyItem = ['id' => (int)$row['id']];
                    }
                    $therapyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($therapyItem)) {
                    if ($minifyApplied) {
                        $therapyItemMinified = ['i' => $therapyItem['id']];
                        if (isset($therapyItem['name'])) $therapyItemMinified['n'] = $therapyItem['name'];
                        if (isset($therapyItem['title'])) $therapyItemMinified['t'] = $therapyItem['title'];
                        if (isset($therapyItem['description'])) $therapyItemMinified['d'] = $therapyItem['description'];
                        $sectionData[$sectionKey][] = $therapyItemMinified;
                    } else {
                        $sectionData[$sectionKey][] = $therapyItem;
                    }
                    logParser("DEBUG specializations: Added 'Терапия' specialization after minification for cardiac query");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding therapy after minification: " . $e->getMessage());
            }
        }
        
        // Для "акне": добавляем "Косметология", если её нет
        $isAcneQuery = preg_match('/(акне|угр|прыщ)/ui', $question);
        if ($isAcneQuery) {
            try {
                $db = getDatabase();
                $stmtCosmetology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%косметолог%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtCosmetology->execute([$widget['id']]);
                
                $cosmetologyItem = [];
                $cosmetologyId = null;
                while ($row = $stmtCosmetology->fetch()) {
                    if (empty($cosmetologyItem)) {
                        $cosmetologyId = (int)$row['id'];
                        $cosmetologyItem = ['id' => $cosmetologyId];
                    }
                    $cosmetologyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($cosmetologyItem) && !in_array($cosmetologyId, $existingIds)) {
                    if ($minifyApplied) {
                        $cosmetologyItemMinified = ['i' => $cosmetologyItem['id']];
                        if (isset($cosmetologyItem['name'])) $cosmetologyItemMinified['n'] = $cosmetologyItem['name'];
                        if (isset($cosmetologyItem['title'])) $cosmetologyItemMinified['t'] = $cosmetologyItem['title'];
                        if (isset($cosmetologyItem['description'])) $cosmetologyItemMinified['d'] = $cosmetologyItem['description'];
                        $sectionData[$sectionKey][] = $cosmetologyItemMinified;
                    } else {
                        $sectionData[$sectionKey][] = $cosmetologyItem;
                    }
                    logParser("DEBUG specializations: Added 'Косметология' specialization after minification for acne query");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding cosmetology after minification: " . $e->getMessage());
            }
        }
        
        // Для административных запросов добавляем специализацию "Справки и комиссии"
        $isAdminQuery = preg_match('/(справк|медосмотр|медицинск.*осмотр|комисс|диспансеризац|профосмотр)/ui', $question);
        if ($isAdminQuery) {
            try {
                $db = getDatabase();
                $stmtAdmin = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%справк%' OR LOWER(pf.field_value) LIKE '%комисс%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtAdmin->execute([$widget['id']]);
                
                $adminItem = [];
                $adminId = null;
                while ($row = $stmtAdmin->fetch()) {
                    if (empty($adminItem)) {
                        $adminId = (int)$row['id'];
                        $adminItem = ['id' => $adminId];
                    }
                    $adminItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($adminItem) && !in_array($adminId, $existingIds)) {
                    if ($minifyApplied) {
                        $adminItemMinified = ['i' => $adminItem['id']];
                        if (isset($adminItem['name'])) $adminItemMinified['n'] = $adminItem['name'];
                        if (isset($adminItem['title'])) $adminItemMinified['t'] = $adminItem['title'];
                        if (isset($adminItem['description'])) $adminItemMinified['d'] = $adminItem['description'];
                        $sectionData[$sectionKey][] = $adminItemMinified;
                    } else {
                        $sectionData[$sectionKey][] = $adminItem;
                    }
                    logParser("DEBUG specializations: Added 'Справки и комиссии' specialization after minification for admin query '{$question}'");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding admin specialization after minification: " . $e->getMessage());
            }
        }
        
        // Для онемения/покалывания добавляем кардиология + эндокринология (невролог уже может быть от AI)
        $isNumbnessQuery = preg_match('/(онемен|покалыван|мурашк|жжен|жжени)/ui', $question);
        if ($isNumbnessQuery) {
            logParser("DEBUG specializations: Processing numbness query after minification '{$question}', minifyApplied: " . ($minifyApplied ? 'YES' : 'NO') . ", existingIds: " . implode(',', $existingIds));
            // Кардиология
            try {
                $db = getDatabase();
                $stmtCardiology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%кардиолог%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtCardiology->execute([$widget['id']]);
                
                $cardiologyItem = [];
                $cardiologyId = null;
                while ($row = $stmtCardiology->fetch()) {
                    if (empty($cardiologyItem)) {
                        $cardiologyId = (int)$row['id'];
                        $cardiologyItem = ['id' => $cardiologyId];
                    }
                    $cardiologyItem[$row['field_name']] = $row['field_value'];
                }
                
                logParser("DEBUG specializations: Found cardiology item after minification: " . ($cardiologyId ?? 'NULL') . ", empty: " . (empty($cardiologyItem) ? 'YES' : 'NO') . ", in existingIds: " . (in_array($cardiologyId, $existingIds) ? 'YES' : 'NO'));
                
                if (!empty($cardiologyItem) && !in_array($cardiologyId, $existingIds)) {
                    if ($minifyApplied) {
                        $cardiologyItemMinified = ['i' => $cardiologyItem['id']];
                        if (isset($cardiologyItem['name'])) $cardiologyItemMinified['n'] = $cardiologyItem['name'];
                        if (isset($cardiologyItem['title'])) $cardiologyItemMinified['t'] = $cardiologyItem['title'];
                        if (isset($cardiologyItem['description'])) $cardiologyItemMinified['d'] = $cardiologyItem['description'];
                        $sectionData[$sectionKey][] = $cardiologyItemMinified;
                    } else {
                        $sectionData[$sectionKey][] = $cardiologyItem;
                    }
                    logParser("DEBUG specializations: Added 'Кардиология' specialization after minification for numbness query '{$question}'");
                } else {
                    logParser("DEBUG specializations: Skipped adding cardiology after minification: empty=" . (empty($cardiologyItem) ? 'YES' : 'NO') . ", in existingIds=" . (in_array($cardiologyId, $existingIds) ? 'YES' : 'NO'));
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding cardiology after minification: " . $e->getMessage());
            }
            
            // Эндокринология
            try {
                $db = getDatabase();
                $stmtEndocrinology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%эндокринолог%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtEndocrinology->execute([$widget['id']]);
                
                $endocrinologyItem = [];
                $endocrinologyId = null;
                while ($row = $stmtEndocrinology->fetch()) {
                    if (empty($endocrinologyItem)) {
                        $endocrinologyId = (int)$row['id'];
                        $endocrinologyItem = ['id' => $endocrinologyId];
                    }
                    $endocrinologyItem[$row['field_name']] = $row['field_value'];
                }
                
                logParser("DEBUG specializations: Found endocrinology item after minification: " . ($endocrinologyId ?? 'NULL') . ", empty: " . (empty($endocrinologyItem) ? 'YES' : 'NO') . ", in existingIds: " . (in_array($endocrinologyId, $existingIds) ? 'YES' : 'NO'));
                
                if (!empty($endocrinologyItem) && !in_array($endocrinologyId, $existingIds)) {
                    if ($minifyApplied) {
                        $endocrinologyItemMinified = ['i' => $endocrinologyItem['id']];
                        if (isset($endocrinologyItem['name'])) $endocrinologyItemMinified['n'] = $endocrinologyItem['name'];
                        if (isset($endocrinologyItem['title'])) $endocrinologyItemMinified['t'] = $endocrinologyItem['title'];
                        if (isset($endocrinologyItem['description'])) $endocrinologyItemMinified['d'] = $endocrinologyItem['description'];
                        $sectionData[$sectionKey][] = $endocrinologyItemMinified;
                    } else {
                        $sectionData[$sectionKey][] = $endocrinologyItem;
                    }
                    logParser("DEBUG specializations: Added 'Эндокринология' specialization after minification for numbness query '{$question}'");
                } else {
                    logParser("DEBUG specializations: Skipped adding endocrinology after minification: empty=" . (empty($endocrinologyItem) ? 'YES' : 'NO') . ", in existingIds=" . (in_array($endocrinologyId, $existingIds) ? 'YES' : 'NO'));
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding endocrinology after minification: " . $e->getMessage());
            }
        }
        
        // Для "ожог": добавляем "Хирургия", если её нет
        $isBurnQuery = preg_match('/(ожог|ожог)/ui', $question);
        if ($isBurnQuery) {
            try {
                $db = getDatabase();
                $stmtSurgery = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%хирург%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtSurgery->execute([$widget['id']]);
                
                $surgeryItem = [];
                $surgeryId = null;
                while ($row = $stmtSurgery->fetch()) {
                    if (empty($surgeryItem)) {
                        $surgeryId = (int)$row['id'];
                        $surgeryItem = ['id' => $surgeryId];
                    }
                    $surgeryItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($surgeryItem) && !in_array($surgeryId, $existingIds)) {
                    if ($minifyApplied) {
                        $surgeryItemMinified = ['i' => $surgeryItem['id']];
                        if (isset($surgeryItem['name'])) $surgeryItemMinified['n'] = $surgeryItem['name'];
                        if (isset($surgeryItem['title'])) $surgeryItemMinified['t'] = $surgeryItem['title'];
                        if (isset($surgeryItem['description'])) $surgeryItemMinified['d'] = $surgeryItem['description'];
                        $sectionData[$sectionKey][] = $surgeryItemMinified;
                    } else {
                        $sectionData[$sectionKey][] = $surgeryItem;
                    }
                    logParser("DEBUG specializations: Added 'Хирургия' specialization after minification for burn query");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding surgery after minification: " . $e->getMessage());
            }
        }
        
        // Фильтруем нерелевантные специализации
        $filteredSpecializations = [];
        foreach ($sectionData[$sectionKey] as $spec) {
            $name = mb_strtolower($spec[$minifyApplied ? 'n' : 'name'] ?? $spec[$minifyApplied ? 't' : 'title'] ?? '');
            $shouldInclude = true;
            
            // Для "акне": убрать "Неврология" и "Аллергология"
            if ($isAcneQuery) {
                if (strpos($name, 'невролог') !== false || strpos($name, 'аллерголог') !== false) {
                    $shouldInclude = false;
                }
            }
            
            // Для "ожог": убрать "Травматология"
            if ($isBurnQuery) {
                if (strpos($name, 'травматолог') !== false) {
                    $shouldInclude = false;
                }
            }
            
            if ($shouldInclude) {
                $filteredSpecializations[] = $spec;
            }
        }
        
        if (count($filteredSpecializations) !== count($sectionData[$sectionKey])) {
            logParser("DEBUG specializations filter: Filtered " . (count($sectionData[$sectionKey]) - count($filteredSpecializations)) . " specializations after minification");
            $sectionData[$sectionKey] = $filteredSpecializations;
        }
    }
    
    $jsonData = json_encode($sectionData, JSON_UNESCAPED_UNICODE);
    
    // Выбираем промпт для раздела: сначала проверяем промпт конкретного раздела, потом stage3_prompt, потом claude_prompt
    $sectionPromptField = 'stage_' . $section_name . '_prompt';
    $sectionPrompt = !empty($widget[$sectionPromptField]) ? $widget[$sectionPromptField] : null;
    
    if ($sectionPrompt) {
        $stage3SystemPrompt = $sectionPrompt;
    } elseif (!empty($widget['stage3_prompt'])) {
        $stage3SystemPrompt = $widget['stage3_prompt'];
    } else {
        $stage3SystemPrompt = $widget['claude_prompt'];
    }
    
    // Применяем сокращение инструкций (shorten), если оптимизация включена
    if ($optimization && $optimization['optimization_type'] === 'shorten' && $optimizedData && isset($optimizedData['shortened_prompt'])) {
        $stage3SystemPrompt = $optimizedData['shortened_prompt'];
    }
    
    // Если применена минификация, добавляем краткую информацию о формате ТОЛЬКО если промпт не содержит эту информацию
    if ($minifyApplied) {
        if (strpos($stage3SystemPrompt, 'минифицированные ключи') === false && strpos($stage3SystemPrompt, 'ФОРМАТ') === false) {
            $minifyInfo = "\nФОРМАТ: данные используют минифицированные ключи (s/sv/a/sp, i, d). Используй тот же формат в ответе.\n";
            $stage3SystemPrompt = $minifyInfo . $stage3SystemPrompt;
        }
    }
    
    // Добавляем инструкции для одного раздела
    $sectionNameRu = [
        'specialists' => 'специалистов',
        'services' => 'услуг',
        'articles' => 'статей',
        'specializations' => 'специализаций'
    ][$section_name] ?? $section_name;
    
    $count = count($items);
    
    // Определяем имя раздела для ответа (минифицированное или полное)
    $responseSectionName = $section_name;
    if ($minifyApplied) {
        $responseSectionName = match($section_name) {
            'specialists' => 's',
            'services' => 'sv',
            'articles' => 'a',
            'specializations' => 'sp',
            default => $section_name
        };
    }
    
    // НЕ добавляем автоматические инструкции "ПРАВИЛА" - используем только промпт из настроек виджета
    // Если пользователь хочет добавить правила, он должен указать их в промпте в настройках виджета
    
    // Добавляем динамические инструкции на основе вопроса для улучшения выбора специалистов
    $dynamicInstructions = '';
    if ($section_name === 'specialists') {
        $questionLower = mb_strtolower($question);
        
        // Для дерматологических проблем (акне, псориаз, розацеа, зуд кожи и т.д.)
        $dermatologyKeywords = ['акне', 'псориаз', 'розацеа', 'зуд кожи', 'экзема', 'дерматит', 'крапивница', 
                               'угри', 'прыщи', 'сыпь', 'кожа', 'дерматолог', 'дерматология', 'дерматовенеролог'];
        $isDermatologyQuery = false;
        foreach ($dermatologyKeywords as $keyword) {
            if (strpos($questionLower, $keyword) !== false) {
                $isDermatologyQuery = true;
                break;
            }
        }
        
        if ($isDermatologyQuery) {
            $dynamicInstructions = "\n\nВАЖНО: Для проблем с кожей (акне, псориаз, розацеа, зуд, экзема и т.д.) выбирай ТОЛЬКО дерматологов или дерматовенерологов (это то же самое - врачи по кожным заболеваниям). НЕ выбирай подологов (врачей по ногтям на ногах) - они не релевантны для кожных проблем. Если дерматологов/дерматовенерологов нет в списке, выбирай аллергологов-иммунологов или терапевтов (они могут помочь с кожными проблемами), но НЕ подологов. Выбирай ВСЕХ релевантных специалистов. Если в списке есть только подологи - не выбирай никого, так как они не релевантны.\n";
        }
        
        // Для аллергий (включая аллергию на холод)
        if (preg_match('/(аллерг|аллерги|крапивниц|холод.*аллерг|аллерг.*холод|холодов.*аллерг)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для аллергических проблем (аллергия, крапивница, аллергия на холод и т.д.) выбирай ВСЕХ аллергологов-иммунологов из списка. Это ключевые специалисты для аллергий. Также можно выбрать дерматологов (если есть кожные проявления) и терапевтов, но ПРИОРИТЕТ - аллергологам-иммунологам. Выбирай ВСЕХ релевантных специалистов.\n";
        }
        
        // Для приступов удушья и проблем с дыханием
        if (preg_match('/(приступ.*удушь|удушь.*приступ|задыхаюсь|не.*могу.*дышать|затруднен.*дыхан|одышк|астм)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nКРИТИЧЕСКИ ВАЖНО: Для приступов удушья и проблем с дыханием ОБЯЗАТЕЛЬНО выбирай пульмонологов, аллергологов-иммунологов, терапевтов и кардиологов (если есть в списке). Приступ удушья - это серьезный симптом, требующий немедленной консультации. Выбирай ВСЕХ релевантных специалистов из разных специальностей. НЕ возвращай пустой список! НЕ выбирай ЛОР-врачей и психиатров для приступов удушья - они не релевантны.\n";
        }
        
        // Для прививок детям
        if (preg_match('/(прививк|вакцин|иммунизац)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для прививок и вакцинации выбирай ТОЛЬКО педиатров и аллергологов-иммунологов. НЕ выбирай хирургов, неврологов и других нерелевантных специалистов. Выбирай ВСЕХ педиатров и аллергологов-иммунологов из списка.\n";
        }
        
        // Для болей в тазобедренном суставе
        if (preg_match('/(боль.*тазобедренн|тазобедренн.*бол|тазобедренн.*сустав)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для болей в тазобедренном суставе выбирай не только травматологов-ортопедов, но также неврологов (для исключения защемления седалищного нерва), терапевтов (для первичной диагностики). Выбирай ВСЕХ релевантных специалистов.\n";
        }
        
        // Для ларингита и потери голоса
        if (preg_match('/(ларингит|потеря.*голос|голос.*потеря|охрип|осип)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nКРИТИЧЕСКИ ВАЖНО: Для ларингита и потери голоса выбирай ТОЛЬКО ЛОР-врачей (оториноларингологов). Это их прямая специализация. Терапевты и педиатры могут быть дополнительно, но ЛОР - это ПРИОРИТЕТНЫЙ специалист. Выбирай ВСЕХ ЛОР-врачей из списка.\n";
        }
        
        // Для резкой потери веса
        if (preg_match('/(резк.*потеря.*вес|потеря.*вес|снижен.*вес)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для резкой потери веса выбирай не только эндокринологов, но также гастроэнтерологов (для исключения заболеваний ЖКТ), терапевтов (для общей диагностики), психиатров (для исключения депрессии, анорексии). Выбирай ВСЕХ релевантных специалистов из разных специальностей. НЕ возвращай только одного специалиста!\n";
        }
        
        // Для нарушений ритма сердца
        if (preg_match('/(ритм.*сердц|сердц.*ритм|аритми|тахикард|брадикард|нарушен.*ритм)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для нарушений ритма сердца выбирай ТОЛЬКО кардиологов или терапевтов с кардиологической специализацией (если в описании указано 'кардиолог' или 'кардиология'). НЕ выбирай обычных терапевтов без кардиологической специализации, гастроэнтерологов, аллергологов и других нерелевантных специалистов. Выбирай ВСЕХ кардиологов из списка.\n";
        }
        
        // Для симптомов горла/носа добавляем инструкцию о выборе терапевтов и педиатров
        if (strpos($questionLower, 'горл') !== false || strpos($questionLower, 'нос') !== false ||
            strpos($questionLower, 'ангин') !== false || strpos($questionLower, 'фарингит') !== false ||
            strpos($questionLower, 'тонзиллит') !== false || strpos($questionLower, 'кашель') !== false ||
            preg_match('/(болит.*горл|трудно.*глотать|глотать.*трудно|просту|насморк|чихание)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nКРИТИЧЕСКИ ВАЖНО: Для простуды, насморка, симптомов горла/носа (боль в горле, трудно глотать, насморк, кашель и т.д.) выбирай ТЕРАПЕВТОВ - это ПРИОРИТЕТНЫЕ специалисты для первичного приема. Также выбирай ЛОР-врачей (оториноларингологов) для диагностики осложнений. НЕ выбирай неврологов - они не релевантны для простудных заболеваний. Выбирай ВСЕХ релевантных специалистов.\n";
        }
        
        // Для болей в суставах (всех суставов, бурсит)
        if (preg_match('/(боль.*всех.*сустав|всех.*сустав.*бол|бурсит|полиартрит|полиартралг)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nКРИТИЧЕСКИ ВАЖНО: Для болей во всех суставах, бурсита, полиартрита выбирай НЕ ТОЛЬКО травматологов-ортопедов, но также ревматологов, терапевтов, неврологов и хирургов. Боли во всех суставах требуют комплексной диагностики. Выбирай ВСЕХ релевантных специалистов из разных специальностей. НЕ возвращай только одного специалиста!\n";
        }
        
        // Для болей в икрах при ходьбе
        if (preg_match('/(боль.*икр|икр.*бол|икроножн.*бол|боль.*при.*ходьб|ходьб.*бол)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для болей в икрах при ходьбе выбирай не только травматологов, но также терапевтов (для общей диагностики), неврологов (для исключения невралгии), хирургов (для исключения сосудистых проблем). Выбирай ВСЕХ релевантных специалистов. НЕ выбирай узких специалистов типа рефлексотерапевтов в первую очередь.\n";
        }
        
        // Для симптомов пищеварения добавляем инструкцию о выборе терапевтов
        if (strpos($questionLower, 'желудок') !== false || strpos($questionLower, 'живот') !== false ||
            strpos($questionLower, 'тошнот') !== false || strpos($questionLower, 'изжог') !== false) {
            $dynamicInstructions = "\n\nВАЖНО: Для симптомов пищеварения выбирай не только гастроэнтерологов, но также терапевтов (если они есть в списке), так как они тоже могут помочь с такими симптомами. Выбирай ВСЕХ релевантных специалистов разных специальностей.\n";
        }
        
        // Для справок (общий fallback)
        if (preg_match('/(справк.*болезн|справк.*заболеван|справк.*здоров|справк.*медицинск|медицинск.*справк|справк.*общ)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для запросов о справках (справка о болезни, медицинская справка и т.д.) выбирай терапевтов - это основные специалисты, которые выдают такие справки. Выбирай ВСЕХ терапевтов из списка. НЕ возвращай пустой список!\n";
        }
        
        // Для болей в грудной клетке
        if (preg_match('/(боль.*грудн.*клетк|грудн.*клетк.*бол|боль.*груд|груд.*бол|боль.*сердц)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nКРИТИЧЕСКИ ВАЖНО: Для болей в грудной клетке ОБЯЗАТЕЛЬНО выбирай кардиологов (или терапевтов с кардиологической специализацией), гастроэнтерологов, пульмонологов и терапевтов. Боль в груди может быть связана с сердцем, легкими или желудком. Выбирай ВСЕХ релевантных специалистов из разных специальностей. НЕ возвращай пустой список!\n";
        }
        
        // Для УЗИ малого таза
        if (preg_match('/(узи.*мал.*таз|мал.*таз.*узи|узи.*орган.*мал.*таз|гинеколог.*узи)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nКРИТИЧЕСКИ ВАЖНО: Для УЗИ малого таза выбирай ТОЛЬКО гинекологов (акушеров-гинекологов) и врачей ультразвуковой диагностики. НЕ выбирай терапевтов, неврологов, психиатров или других нерелевантных специалистов. Выбирай ВСЕХ гинекологов и УЗИ-специалистов из списка.\n";
        }
        
        // Для хронической усталости и слабости
        if (preg_match('/(хроническ.*усталост|усталост.*слабост|слабост.*утомляемость|утомляемость.*слабость|постоянн.*усталост)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для хронической усталости и слабости выбирай не только терапевтов, но также неврологов, эндокринологов и психотерапевтов/психиатров (если есть в списке), так как усталость может быть связана с разными причинами. Выбирай ВСЕХ релевантных специалистов разных специальностей.\n";
        }
    }
    
    // Добавляем динамические инструкции для specializations
    if ($section_name === 'specializations') {
        $questionLower = mb_strtolower($question);
        
        // Для ХОБЛ, бронхита, астмы, пневмонии, одышки
        if (preg_match('/(хобл|хроническ.*обструктивн.*болезн|бронхит|астм|пневмони|одышк|затруднен.*дыхан|кашель.*хроническ|эмфизем)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для заболеваний легких (ХОБЛ, бронхит, астма, пневмония, одышка) ОБЯЗАТЕЛЬНО выбирай специализацию Пульмонология, если она есть в списке. Также можно выбрать Терапию и Кардиологию (так как легочные заболевания могут влиять на сердце), но ПРИОРИТЕТ - Пульмонология. Выбирай ВСЕ релевантные специализации.\n";
        }
        
        // Для нарушений ритма сердца, аритмии
        if (preg_match('/(нарушен.*ритм.*сердц|ритм.*сердц|аритми|тахикард|брадикард|перебо.*сердц|замира.*сердц)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для нарушений ритма сердца (аритмия, тахикардия, брадикардия) выбирай Кардиологию (это основная специализация). Также выбирай Эндокринологию (так как нарушения ритма могут быть связаны с заболеваниями щитовидной железы) и Терапию. Выбирай ВСЕ релевантные специализации.\n";
        }
        
        // Для болей в сердце, гипертонии, ишемии
        if (preg_match('/(боль.*сердц|сердц.*бол|гипертон|ишеми|стенокард|инфаркт|инсульт)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для сердечно-сосудистых заболеваний (боль в сердце, гипертония, ишемия, инфаркт, инсульт) ОБЯЗАТЕЛЬНО выбирай специализацию Кардиология, если она есть в списке. Также можно выбрать Терапию. Выбирай ВСЕ релевантные специализации.\n";
        }
        
        // Для межреберной невралгии, болей в грудной клетке
        if (preg_match('/(межреберн|невралг|грудн.*клетк.*бол|грудн.*бол)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для межреберной невралгии и болей в грудной клетке выбирай Неврологию (основная специализация) и Кардиологию (для исключения сердечных причин). НЕ выбирай специализации, не связанные с неврологией или кардиологией. Выбирай ВСЕ релевантные специализации.\n";
        }
        
        // Для проблем с желудком, гастрита, изжоги
        if (preg_match('/(желудок|гастроэнтеролог|изжог|гастрит|рефлюкс|язв)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для проблем с желудком и пищеварением (гастрит, изжога, рефлюкс, язва) ОБЯЗАТЕЛЬНО выбирай специализацию Гастроэнтерология, если она есть в списке. Также можно выбрать Терапию. Выбирай ВСЕ релевантные специализации.\n";
        }
    }
    
    // Добавляем динамические инструкции для articles
    if ($section_name === 'articles') {
        $questionLower = mb_strtolower($question);
        
        // Для ХОБЛ, бронхита, астмы
        if (preg_match('/(хобл|хроническ.*обструктивн.*болезн|бронхит|астм|пневмони)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для запросов о ХОБЛ, бронхите, астме, пневмонии выбирай статьи, которые НАПРЯМУЮ связаны с этими заболеваниями. НЕ выбирай общие статьи об одышке или сердце, если они не упоминают конкретно ХОБЛ, бронхит или астму. Приоритет - статьи с названиями, содержащими слова 'ХОБЛ', 'бронхит', 'астма', 'пневмония'. Выбирай ВСЕ релевантные статьи.\n";
        }
        
        // Для нарушений ритма сердца
        if (preg_match('/(нарушен.*ритм.*сердц|ритм.*сердц|аритми|тахикард|брадикард)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для запросов о нарушениях ритма сердца выбирай статьи, которые НАПРЯМУЮ связаны с аритмией, тахикардией, брадикардией. НЕ выбирай общие статьи о сердце или гипертонии, если они не упоминают конкретно нарушения ритма. Приоритет - статьи с названиями, содержащими слова 'аритмия', 'ритм', 'тахикардия', 'брадикардия'. Выбирай ВСЕ релевантные статьи.\n";
        }
        
        // Для межреберной невралгии
        if (preg_match('/(межреберн|невралг)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для запросов о межреберной невралгии выбирай статьи, которые НАПРЯМУЮ связаны с невралгией, межреберной болью, грудной клеткой. НЕ выбирай статьи о болях в колене, ходьбе или других частях тела, не связанных с грудной клеткой. Приоритет - статьи с названиями, содержащими слова 'невралгия', 'межреберн', 'грудн', 'ребр'. Выбирай ВСЕ релевантные статьи.\n";
        }
        
        // Для болей в сердце, гипертонии
        if (preg_match('/(боль.*сердц|сердц.*бол|гипертон)/ui', $questionLower)) {
            $dynamicInstructions = "\n\nВАЖНО: Для запросов о болях в сердце и гипертонии выбирай статьи, которые НАПРЯМУЮ связаны с сердечными заболеваниями или гипертонией. НЕ выбирай общие статьи о легких или других органах, если они не связаны с сердцем. Приоритет - статьи с названиями, содержащими слова 'сердце', 'гипертон', 'кардиолог', 'ишемия'. Выбирай ВСЕ релевантные статьи.\n";
        }
    }
    
    // Компактный список ID перед каталогом
    $idsList = implode(', ', array_slice(array_keys($items), 0, 20));
    if (count($items) > 20) {
        $idsList .= ' ... (всего ' . count($items) . ' ID)';
    }
    $sectionIdsInfo = "\n\nДоступные ID: [$idsList]\n";
    
    // Формируем финальный промпт
    $userQuestionHeader = "\n\n=== ВОПРОС ПОЛЬЗОВАТЕЛЯ ===\n" . $question . "\n\n=== КАТАЛОГ ДАННЫХ ===\n";
    $promptData = $dynamicInstructions . $sectionIdsInfo . $userQuestionHeader . $jsonData;
    
    // Выбираем модель для раздела: сначала проверяем модель конкретного раздела, потом stage3_model
    $sectionModelField = 'stage_' . $section_name . '_model';
    $sectionModel = !empty($widget[$sectionModelField]) ? $widget[$sectionModelField] : null;
    
    if ($sectionModel) {
        $stage3Model = normalizeModelName($sectionModel);
    } else {
        $stage3Model = normalizeModelName($widget['stage3_model'] ?? 'qwen/qwen3-235b-a22b-2507');
    }
    
    // Отправляем запрос к AI с повторными попытками при пустом ответе
    $stage3DebugInfo = null;
    $claudeResponse = null;
    $maxRetries = 3;
    $retryCount = 0;
    $aiDataIds = [];
    
    // Повторяем запрос до 3 раз, если AI возвращает пустой массив для услуг или ошибку парсинга
    for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
        try {
            $claudeResponse = sendClaudeRequest($stage3SystemPrompt, $promptData, $question, false, $stage3Model, $debugMode, $stage3DebugInfo, $widget['custom_api_url'] ?? null, $widget['custom_api_key'] ?? null, $section_name);
            
            logParser("sendClaudeRequest response for section '$section_name': success=" . ($claudeResponse['success'] ?? 'N/A') . ", data_ids keys: " . implode(',', array_keys($claudeResponse['data_ids'] ?? [])));
            
            if (!isset($claudeResponse) || !$claudeResponse['success']) {
                $errorMsg = isset($claudeResponse['error']) ? $claudeResponse['error'] : 'Неизвестная ошибка при обработке запроса';
                // Если это ошибка JSON decode и это услуги, пробуем повторить
                if ($section_name === 'services' && strpos($errorMsg, 'JSON decode') !== false && $attempt < $maxRetries) {
                    error_log("DEBUG services: JSON decode error on attempt {$attempt}/{$maxRetries} for query '{$question}'. Retrying...");
                    usleep(rand(100000, 300000));
                    continue;
                }
                throw new Exception($errorMsg);
            }
            
            $aiDataIds = $claudeResponse['data_ids'] ?? [];
            
            error_log("DEBUG AI RESPONSE: section='$section_name', question='{$question}', full_response=" . json_encode($claudeResponse));
            error_log("DEBUG aiDataIds: " . json_encode($aiDataIds));
            error_log("DEBUG aiDataIds['{$section_name}']: isset=" . (isset($aiDataIds[$section_name]) ? 'yes' : 'no') . ", empty=" . (empty($aiDataIds[$section_name]) ? 'yes' : 'no') . ", is_array=" . (is_array($aiDataIds[$section_name] ?? null) ? 'yes' : 'no') . ", count=" . (is_array($aiDataIds[$section_name] ?? null) ? count($aiDataIds[$section_name]) : 'N/A'));
            
            logParser("aiDataIds for section '$section_name', question '{$question}': " . json_encode($aiDataIds));
            logParser("aiDataIds['{$section_name}'] isset=" . (isset($aiDataIds[$section_name]) ? 'yes' : 'no') . ", empty=" . (empty($aiDataIds[$section_name]) ? 'yes' : 'no') . ", is_array=" . (is_array($aiDataIds[$section_name] ?? null) ? 'yes' : 'no') . ", count=" . (is_array($aiDataIds[$section_name] ?? null) ? count($aiDataIds[$section_name]) : 'N/A'));
            
            // Проверяем, есть ли данные для этого раздела
            $hasData = isset($aiDataIds[$section_name]) && !empty($aiDataIds[$section_name]) && is_array($aiDataIds[$section_name]);
            error_log("DEBUG hasData: section='$section_name', question='{$question}', hasData=" . ($hasData ? 'YES' : 'NO'));
            logParser("hasData for section '$section_name', question '{$question}': " . ($hasData ? 'YES' : 'NO'));
            
            // Если есть данные или это не услуги, выходим из цикла
            if ($hasData || $section_name !== 'services') {
                break;
            }
            
            // Если это услуги и пустой ответ, логируем и продолжаем попытки
            if ($section_name === 'services' && !$hasData && $attempt < $maxRetries) {
                error_log("DEBUG services: AI returned empty array on attempt {$attempt}/{$maxRetries} for query '{$question}'. Retrying...");
                // Небольшая задержка перед повторной попыткой (100-300ms)
                usleep(rand(100000, 300000));
            }
        } catch (Exception $e) {
            // Если это последняя попытка или не услуги, пробрасываем исключение
            if ($attempt >= $maxRetries || $section_name !== 'services') {
                throw $e;
            }
            // Если это услуги и не последняя попытка, логируем и продолжаем
            $errorMsg = $e->getMessage();
            error_log("DEBUG services: Error on attempt {$attempt}/{$maxRetries} for query '{$question}': {$errorMsg}. Retrying...");
            usleep(rand(100000, 300000));
        }
    }
    
    $responseTime = round((microtime(true) - $startTime) * 1000);
    
    // Валидация ID для этого раздела
    $validatedDataIds = [];
    $aiOrder = []; // Сохраняем порядок от AI
    
    error_log("DEBUG VALIDATION: section='$section_name', question='{$question}', aiDataIds=" . json_encode($aiDataIds));
    logParser("Validating IDs for section '$section_name'. isset=" . (isset($aiDataIds[$section_name]) ? 'yes' : 'no') . ", empty=" . (empty($aiDataIds[$section_name]) ? 'yes' : 'no') . ", is_array=" . (is_array($aiDataIds[$section_name] ?? null) ? 'yes' : 'no'));
    
    if (isset($aiDataIds[$section_name]) && is_array($aiDataIds[$section_name]) && count($aiDataIds[$section_name]) > 0) {
        error_log("DEBUG: AI HAS DATA for section '$section_name', question '{$question}', count=" . count($aiDataIds[$section_name]));
        $ids = array_map(function($id) {
            return is_string($id) ? (int)$id : $id;
        }, $aiDataIds[$section_name]);
        
        // Логируем для диагностики
        if ($section_name === 'services') {
            logParser("DEBUG services: AI selected " . count($ids) . " service IDs for query '{$question}'");
        }
        
        // Сохраняем порядок от AI
        $aiOrder = $ids;
        
        $placeholders = str_repeat('?,', count($ids) - 1) . '?';
        $stmt = $db->prepare("
            SELECT DISTINCT id
            FROM parsed_items
            WHERE widget_id = ? 
            AND section_name = ?
            AND id IN ($placeholders)
            AND is_duplicate = 0
        ");
        $params = array_merge([$widget['id'], $section_name], $ids);
        $stmt->execute($params);
        
        $validatedIdsSet = [];
        while ($row = $stmt->fetch()) {
            $validatedIdsSet[(int)$row['id']] = true;
        }
        
        // Сохраняем валидные ID в порядке от AI
        foreach ($aiOrder as $id) {
            if (isset($validatedIdsSet[$id])) {
                $validatedDataIds[$section_name][] = $id;
            }
        }
        
        // Логируем для диагностики
        if ($section_name === 'services') {
            $validatedCount = count($validatedDataIds[$section_name] ?? []);
            $notFoundCount = count($ids) - $validatedCount;
            logParser("DEBUG services: Validated {$validatedCount} IDs, {$notFoundCount} not found in DB for query '{$question}'");
        }
        
        // Логируем для specializations тоже
        if ($section_name === 'specializations') {
            $validatedCount = count($validatedDataIds[$section_name] ?? []);
            $notFoundCount = count($ids) - $validatedCount;
            logParser("DEBUG specializations: Validated {$validatedCount} IDs, {$notFoundCount} not found in DB for query '{$question}'");
            logParser("DEBUG specializations: IDs = " . json_encode($validatedDataIds[$section_name] ?? []));
            
            // ФИЛЬТРАЦИЯ НЕРЕЛЕВАНТНЫХ СПЕЦИАЛИЗАЦИЙ после получения результатов от AI
            if ($section_name === 'specializations') {
                $questionLower = mb_strtolower($question);
                $isOncologyQuery = preg_match('/(онколог|рак|опухол|новообразовани|метастаз|канцер|злокачественн)/ui', $question);
                
                // Убираем "Онколог" для не-онкологических запросов
                if (!$isOncologyQuery && !empty($validatedDataIds[$section_name])) {
                    $filteredIds = [];
                    foreach ($validatedDataIds[$section_name] as $id) {
                        // Загружаем название специализации
                        $stmtName = $db->prepare("
                            SELECT pf.field_value as name
                            FROM parsed_items pi
                            JOIN parsed_fields pf ON pi.id = pf.item_id
                            WHERE pi.widget_id = ? AND pi.section_name = 'specializations' 
                            AND pi.id = ? AND pf.field_name = 'name'
                            LIMIT 1
                        ");
                        $stmtName->execute([$widget['id'], $id]);
                        $nameRow = $stmtName->fetch();
                        
                        if ($nameRow) {
                            $name = mb_strtolower($nameRow['name'] ?? '');
                            // Исключаем "Онколог" для не-онкологических запросов
                            if (strpos($name, 'онколог') === false) {
                                $filteredIds[] = $id;
                            } else {
                                logParser("DEBUG specializations filter: Removed 'Онколог' (ID {$id}) from AI response for non-oncology query '{$question}'");
                            }
                        } else {
                            // Если не нашли название, оставляем ID (на всякий случай)
                            $filteredIds[] = $id;
                        }
                    }
                    $validatedDataIds[$section_name] = $filteredIds;
                }
            }
            
            // Инициализируем массив, если его нет (на случай, если AI вернул пустой результат)
            if (!isset($validatedDataIds[$section_name])) {
                $validatedDataIds[$section_name] = [];
            }
            
            $questionLower = mb_strtolower($question);
            
            // FALLBACK для специализаций: для бессонницы всегда добавляем "Неврология"
            $isInsomniaQuery = (strpos($questionLower, 'бессонн') !== false || 
                               strpos($questionLower, 'бессонниц') !== false ||
                               preg_match('/(проблем.*сон|нарушен.*сон|плох.*сон|не.*могу.*заснуть|трудно.*заснуть)/ui', $questionLower));
            
            if ($isInsomniaQuery) {
                try {
                    $stmtNeurology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%невролог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtNeurology->execute([$widget['id']]);
                    $neurologyRow = $stmtNeurology->fetch();
                    if ($neurologyRow) {
                        $neurologyId = (int)$neurologyRow['id'];
                        if (!in_array($neurologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $neurologyId;
                            logParser("DEBUG specializations insomnia fallback: Added neurology specialization (ID {$neurologyId}) for query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations insomnia fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // FALLBACK для пульмонологии: для ХОБЛ, бронхита, астмы, пневмонии, одышки
            $isPulmonologyQuery = preg_match('/(хобл|хроническ.*обструктивн.*болезн|бронхит|астм|пневмони|одышк|затруднен.*дыхан|кашель.*хроническ|эмфизем)/ui', $questionLower);
            
            if ($isPulmonologyQuery) {
                try {
                    $stmtPulmonology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%пульмонолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtPulmonology->execute([$widget['id']]);
                    $pulmonologyRow = $stmtPulmonology->fetch();
                    if ($pulmonologyRow) {
                        $pulmonologyId = (int)$pulmonologyRow['id'];
                        if (!in_array($pulmonologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $pulmonologyId;
                            logParser("DEBUG specializations pulmonology fallback: Added pulmonology specialization (ID {$pulmonologyId}) for query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations pulmonology fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // FALLBACK для эндокринологии: для нарушений ритма сердца, аритмии, тахикардии, брадикардии
            $isArrhythmiaQuery = preg_match('/(нарушен.*ритм.*сердц|ритм.*сердц|аритми|тахикард|брадикард|перебо.*сердц|замира.*сердц)/ui', $questionLower);
            
            if ($isArrhythmiaQuery) {
                try {
                    $stmtEndocrinology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%эндокринолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtEndocrinology->execute([$widget['id']]);
                    $endocrinologyRow = $stmtEndocrinology->fetch();
                    if ($endocrinologyRow) {
                        $endocrinologyId = (int)$endocrinologyRow['id'];
                        if (!in_array($endocrinologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $endocrinologyId;
                            logParser("DEBUG specializations endocrinology fallback: Added endocrinology specialization (ID {$endocrinologyId}) for arrhythmia query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations endocrinology fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // FALLBACK для кардиологии: для болей в сердце, гипертонии, ишемии
            $isCardiacQuery = preg_match('/(боль.*сердц|сердц.*бол|гипертон|ишеми|стенокард|инфаркт|инсульт)/ui', $questionLower);
            
            if ($isCardiacQuery) {
                try {
                    $stmtCardiology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%кардиолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtCardiology->execute([$widget['id']]);
                    $cardiologyRow = $stmtCardiology->fetch();
                    if ($cardiologyRow) {
                        $cardiologyId = (int)$cardiologyRow['id'];
                        if (!in_array($cardiologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $cardiologyId;
                            logParser("DEBUG specializations cardiology fallback: Added cardiology specialization (ID {$cardiologyId}) for cardiac query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations cardiology fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // FALLBACK для гипотонии (низкое давление): добавляем Кардиологию и Невролога
            $isHypotensionQuery = preg_match('/(низк.*давлен|гипотон|понижен.*давлен|давлен.*низк)/ui', $questionLower);
            
            if ($isHypotensionQuery) {
                // Добавляем Кардиологию
                try {
                    $stmtCardiology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%кардиолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtCardiology->execute([$widget['id']]);
                    $cardiologyRow = $stmtCardiology->fetch();
                    if ($cardiologyRow) {
                        $cardiologyId = (int)$cardiologyRow['id'];
                        if (!in_array($cardiologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $cardiologyId;
                            logParser("DEBUG specializations hypotension fallback: Added cardiology specialization (ID {$cardiologyId}) for hypotension query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations hypotension fallback: ERROR adding cardiology - " . $e->getMessage());
                }
                
                // Добавляем Невролога
                try {
                    $stmtNeurology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%невролог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtNeurology->execute([$widget['id']]);
                    $neurologyRow = $stmtNeurology->fetch();
                    if ($neurologyRow) {
                        $neurologyId = (int)$neurologyRow['id'];
                        if (!in_array($neurologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $neurologyId;
                            logParser("DEBUG specializations hypotension fallback: Added neurology specialization (ID {$neurologyId}) for hypotension query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations hypotension fallback: ERROR adding neurology - " . $e->getMessage());
                }
            }
            
            // FALLBACK для астмы: добавляем Аллергологию (если есть в БД)
            $isAsthmaQuery = preg_match('/(астм|бронхиальн.*астм)/ui', $questionLower);
            
            if ($isAsthmaQuery) {
                try {
                    $stmtAllergology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%аллерголог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtAllergology->execute([$widget['id']]);
                    $allergologyRow = $stmtAllergology->fetch();
                    if ($allergologyRow) {
                        $allergologyId = (int)$allergologyRow['id'];
                        if (!in_array($allergologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $allergologyId;
                            logParser("DEBUG specializations asthma fallback: Added allergology specialization (ID {$allergologyId}) for asthma query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations asthma fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // FALLBACK для боли внизу живота у мужчин: добавляем Проктолога и Терапию
            $isLowerAbdomenMaleQuery = preg_match('/(боль.*внизу.*живот|болит.*внизу.*живот|низ.*живот|тазов.*бол)/ui', $questionLower) && 
                                       preg_match('/(мужчин|мужск)/ui', $questionLower);
            
            if ($isLowerAbdomenMaleQuery) {
                // Добавляем Проктолога
                try {
                    $stmtProctology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%проктолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtProctology->execute([$widget['id']]);
                    $proctologyRow = $stmtProctology->fetch();
                    if ($proctologyRow) {
                        $proctologyId = (int)$proctologyRow['id'];
                        if (!in_array($proctologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $proctologyId;
                            logParser("DEBUG specializations lower abdomen male fallback: Added proctology specialization (ID {$proctologyId}) for lower abdomen male query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations lower abdomen male fallback: ERROR adding proctology - " . $e->getMessage());
                }
                
                // Добавляем Терапию
                try {
                    $stmtTherapy = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%терапи%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtTherapy->execute([$widget['id']]);
                    $therapyRow = $stmtTherapy->fetch();
                    if ($therapyRow) {
                        $therapyId = (int)$therapyRow['id'];
                        if (!in_array($therapyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $therapyId;
                            logParser("DEBUG specializations lower abdomen male fallback: Added therapy specialization (ID {$therapyId}) for lower abdomen male query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations lower abdomen male fallback: ERROR adding therapy - " . $e->getMessage());
                }
            }
            
            // FALLBACK для боли в шейно-воротниковой зоне: добавляем Невролога и Травматолога-ортопеда
            $isNeckShoulderQuery = preg_match('/(шейно.*воротников|воротников.*зон|шейн.*воротников|боль.*шейн.*воротников)/ui', $questionLower);
            
            if ($isNeckShoulderQuery) {
                // Добавляем Невролога
                try {
                    $stmtNeurology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%невролог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtNeurology->execute([$widget['id']]);
                    $neurologyRow = $stmtNeurology->fetch();
                    if ($neurologyRow) {
                        $neurologyId = (int)$neurologyRow['id'];
                        if (!in_array($neurologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $neurologyId;
                            logParser("DEBUG specializations neck shoulder fallback: Added neurology specialization (ID {$neurologyId}) for neck shoulder query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations neck shoulder fallback: ERROR adding neurology - " . $e->getMessage());
                }
                
                // Добавляем Травматолога-ортопеда
                try {
                    $stmtTraumatology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%травматолог%' OR LOWER(pf.field_value) LIKE '%ортопед%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtTraumatology->execute([$widget['id']]);
                    $traumatologyRow = $stmtTraumatology->fetch();
                    if ($traumatologyRow) {
                        $traumatologyId = (int)$traumatologyRow['id'];
                        if (!in_array($traumatologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $traumatologyId;
                            logParser("DEBUG specializations neck shoulder fallback: Added traumatology specialization (ID {$traumatologyId}) for neck shoulder query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations neck shoulder fallback: ERROR adding traumatology - " . $e->getMessage());
                }
            }
            
            // FALLBACK для слабости и усталости: добавляем все релевантные специализации
            $isWeaknessQuery = preg_match('/(слабост|усталост|утомляемост|общ.*недомоган|постоянн.*усталост|хроническ.*усталост)/ui', $questionLower);
            
            if ($isWeaknessQuery) {
                // Добавляем Терапию
                try {
                    $stmtTherapy = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%терапи%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtTherapy->execute([$widget['id']]);
                    $therapyRow = $stmtTherapy->fetch();
                    if ($therapyRow) {
                        $therapyId = (int)$therapyRow['id'];
                        if (!in_array($therapyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $therapyId;
                            logParser("DEBUG specializations weakness fallback: Added therapy specialization (ID {$therapyId}) for weakness query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations weakness fallback: ERROR adding therapy - " . $e->getMessage());
                }
                
                // Добавляем Эндокринологию
                try {
                    $stmtEndocrinology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%эндокринолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtEndocrinology->execute([$widget['id']]);
                    $endocrinologyRow = $stmtEndocrinology->fetch();
                    if ($endocrinologyRow) {
                        $endocrinologyId = (int)$endocrinologyRow['id'];
                        if (!in_array($endocrinologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $endocrinologyId;
                            logParser("DEBUG specializations weakness fallback: Added endocrinology specialization (ID {$endocrinologyId}) for weakness query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations weakness fallback: ERROR adding endocrinology - " . $e->getMessage());
                }
                
                // Добавляем Кардиологию
                try {
                    $stmtCardiology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%кардиолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtCardiology->execute([$widget['id']]);
                    $cardiologyRow = $stmtCardiology->fetch();
                    if ($cardiologyRow) {
                        $cardiologyId = (int)$cardiologyRow['id'];
                        if (!in_array($cardiologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $cardiologyId;
                            logParser("DEBUG specializations weakness fallback: Added cardiology specialization (ID {$cardiologyId}) for weakness query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations weakness fallback: ERROR adding cardiology - " . $e->getMessage());
                }
                
                // Добавляем Педиатрию (если запрос может быть про ребенка)
                if (preg_match('/(ребенок|дет|малыш|подросток)/ui', $questionLower)) {
                    try {
                        $stmtPediatrics = $db->prepare("
                            SELECT DISTINCT pi.id
                            FROM parsed_items pi
                            JOIN parsed_fields pf ON pi.id = pf.item_id
                            WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                            AND (LOWER(pf.field_value) LIKE '%педиатр%')
                            AND pi.is_duplicate = 0
                            LIMIT 1
                        ");
                        $stmtPediatrics->execute([$widget['id']]);
                        $pediatricsRow = $stmtPediatrics->fetch();
                        if ($pediatricsRow) {
                            $pediatricsId = (int)$pediatricsRow['id'];
                            if (!in_array($pediatricsId, $validatedDataIds[$section_name])) {
                                $validatedDataIds[$section_name][] = $pediatricsId;
                                logParser("DEBUG specializations weakness fallback: Added pediatrics specialization (ID {$pediatricsId}) for weakness query '{$question}'");
                            }
                        }
                    } catch (Exception $e) {
                        logParser("DEBUG specializations weakness fallback: ERROR adding pediatrics - " . $e->getMessage());
                    }
                }
            }
            
            // ФИЛЬТРАЦИЯ НЕРЕЛЕВАНТНЫХ СТАТЕЙ после получения результатов от AI
            if ($section_name === 'articles' && !empty($validatedDataIds[$section_name])) {
                $questionLower = mb_strtolower($question);
                $filteredArticleIds = [];
                
                // Список нерелевантных статей для исключения
                $excludedTitles = [
                    'программ.*лояльност',
                    'критери.*доступност',
                    'целевые критерии',
                    'правил.*подготовк',
                    'беременност.*секс',
                    'беременност.*развитие плода',
                    'чесотк.*диагностик',
                    'диагностик.*глазн.*болезн.*дет',
                    'диагностик.*варикоцеле'
                ];
                
                // Определяем ключевые слова вопроса для проверки релевантности
                $questionKeywords = [];
                if (preg_match('/(межреберн|невралг|грудн.*клетк|грудн.*бол)/ui', $questionLower)) {
                    $questionKeywords[] = 'грудн|межреберн|невралг|ребр';
                }
                if (preg_match('/(колен|коленн.*сустав|коленн.*бол)/ui', $questionLower)) {
                    $questionKeywords[] = 'колен|коленн';
                }
                if (preg_match('/(ходьб|ходьб.*бол|боль.*при.*ходьб)/ui', $questionLower)) {
                    $questionKeywords[] = 'ходьб';
                }
                if (preg_match('/(сердц|кардиолог|аритми|гипертон|ишеми)/ui', $questionLower)) {
                    $questionKeywords[] = 'сердц|кардиолог|аритми|гипертон|ишеми';
                }
                if (preg_match('/(легк|пульмонолог|бронх|хобл|астм|одышк)/ui', $questionLower)) {
                    $questionKeywords[] = 'легк|пульмонолог|бронх|хобл|астм|одышк';
                }
                if (preg_match('/(желудок|гастроэнтеролог|изжог|гастрит)/ui', $questionLower)) {
                    $questionKeywords[] = 'желудок|гастроэнтеролог|изжог|гастрит';
                }
                if (preg_match('/(спин|поясниц|позвоночник)/ui', $questionLower)) {
                    $questionKeywords[] = 'спин|поясниц|позвоночник';
                }
                
                foreach ($validatedDataIds[$section_name] as $id) {
                    // Загружаем название и текст статьи
                    $stmtArticle = $db->prepare("
                        SELECT pf.field_name, pf.field_value
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'articles' 
                        AND pi.id = ? 
                        AND (pf.field_name = 'title' OR pf.field_name = 'text')
                    ");
                    $stmtArticle->execute([$widget['id'], $id]);
                    
                    $articleData = [];
                    while ($row = $stmtArticle->fetch()) {
                        $articleData[$row['field_name']] = $row['field_value'];
                    }
                    
                    if (!empty($articleData)) {
                        $title = mb_strtolower($articleData['title'] ?? '');
                        $text = mb_strtolower($articleData['text'] ?? '');
                        $shouldExclude = false;
                        
                        // Проверяем, не является ли статья нерелевантной по общим паттернам
                        foreach ($excludedTitles as $pattern) {
                            if (preg_match('/' . $pattern . '/ui', $title)) {
                                // Исключаем только если запрос не про эту тему
                                if ($pattern === 'беременност.*секс' && !preg_match('/(беременност|женск.*здоров|гинеколог)/ui', $questionLower)) {
                                    $shouldExclude = true;
                                } elseif ($pattern === 'беременност.*развитие плода' && !preg_match('/(беременност|женск.*здоров|гинеколог)/ui', $questionLower)) {
                                    $shouldExclude = true;
                                } elseif ($pattern === 'правил.*подготовк' && !preg_match('/(подготовк|подготов.*исследован)/ui', $questionLower)) {
                                    $shouldExclude = true;
                                } elseif ($pattern === 'чесотк.*диагностик' && !preg_match('/(чесотк|кож.*заболеван)/ui', $questionLower)) {
                                    $shouldExclude = true;
                                } elseif ($pattern === 'диагностик.*глазн.*болезн.*дет' && !preg_match('/(глаз|офтальмолог|зрен)/ui', $questionLower)) {
                                    $shouldExclude = true;
                                } elseif ($pattern === 'диагностик.*варикоцеле' && !preg_match('/(варикоцеле|мошонк|яичк)/ui', $questionLower)) {
                                    $shouldExclude = true;
                                } elseif ($pattern !== 'беременност.*секс' && $pattern !== 'беременност.*развитие плода' && 
                                         $pattern !== 'правил.*подготовк' && $pattern !== 'чесотк.*диагностик' &&
                                         $pattern !== 'диагностик.*глазн.*болезн.*дет' && $pattern !== 'диагностик.*варикоцеле') {
                                    // Для остальных паттернов всегда исключаем
                                    $shouldExclude = true;
                                }
                                
                                if ($shouldExclude) {
                                    logParser("DEBUG articles filter: Removed article '{$articleData['title']}' (ID {$id}) from AI response for query '{$question}'");
                                    break;
                                }
                            }
                        }
                        
                        // Проверяем релевантность на основе ключевых слов вопроса
                        if (!$shouldExclude && !empty($questionKeywords)) {
                            $articleText = $title . ' ' . $text;
                            $hasRelevantKeyword = false;
                            
                            // Проверяем, есть ли в статье хотя бы одно ключевое слово из вопроса
                            foreach ($questionKeywords as $keywordPattern) {
                                if (preg_match('/' . $keywordPattern . '/ui', $articleText)) {
                                    $hasRelevantKeyword = true;
                                    break;
                                }
                            }
                            
                            // Если статья не содержит ключевых слов вопроса, проверяем на нерелевантность
                            if (!$hasRelevantKeyword) {
                                // Исключаем статьи о колене для запросов про грудную клетку/межреберную невралгию
                                if (preg_match('/(межреберн|невралг|грудн.*клетк|грудн.*бол)/ui', $questionLower) && 
                                    preg_match('/(колен|коленн)/ui', $articleText)) {
                                    $shouldExclude = true;
                                    logParser("DEBUG articles filter: Removed article '{$articleData['title']}' (ID {$id}) - knee article for chest/neuralgia query");
                                }
                                // Исключаем статьи о ходьбе для запросов про грудную клетку/межреберную невралгию
                                elseif (preg_match('/(межреберн|невралг|грудн.*клетк|грудн.*бол)/ui', $questionLower) && 
                                        preg_match('/(боль.*при.*ходьб|ходьб.*бол)/ui', $articleText)) {
                                    $shouldExclude = true;
                                    logParser("DEBUG articles filter: Removed article '{$articleData['title']}' (ID {$id}) - walking pain article for chest/neuralgia query");
                                }
                                // Исключаем статьи о сердце для запросов про легкие/ХОБЛ
                                elseif (preg_match('/(легк|пульмонолог|бронх|хобл|астм|одышк)/ui', $questionLower) && 
                                        preg_match('/(сердц|кардиолог|аритми|гипертон|ишеми)/ui', $articleText) &&
                                        !preg_match('/(легк|пульмонолог|бронх|хобл|астм|одышк)/ui', $articleText)) {
                                    $shouldExclude = true;
                                    logParser("DEBUG articles filter: Removed article '{$articleData['title']}' (ID {$id}) - cardiac article for pulmonary query");
                                }
                                // Исключаем статьи о головной боли, колене, сердце для гастроэнтерологических запросов
                                elseif (preg_match('/(тошнот|рвот|живот|брюшн|желудок|гастроэнтеролог|изжог|гастрит|диспепси|панкреатит|диаре|запор|метеоризм|газообразован|гореч.*рту|сухост.*рту|боль.*живот|болит.*живот|боли.*в.*животе)/ui', $questionLower)) {
                                    if (preg_match('/(головн.*бол|мигрен|головокружен|шейн|невралг|межреберн|грудн.*клетк|грудн.*бол)/ui', $articleText) && 
                                        !preg_match('/(тошнот|рвот|живот|брюшн|желудок|гастроэнтеролог|изжог|гастрит|диспепси|панкреатит|диаре|запор|метеоризм|газообразован|гореч.*рту|сухост.*рту|брюшн.*полост|кишечник)/ui', $articleText)) {
                                        $shouldExclude = true;
                                        logParser("DEBUG articles filter: Removed article '{$articleData['title']}' (ID {$id}) - neuro/headache article for gastro query");
                                    }
                                    elseif (preg_match('/(колен|коленн|ходьб)/ui', $articleText) && 
                                            !preg_match('/(тошнот|рвот|живот|брюшн|желудок|гастроэнтеролог|изжог|гастрит|диспепси|панкреатит|диаре|запор|метеоризм|газообразован|гореч.*рту|сухост.*рту|брюшн.*полост|кишечник)/ui', $articleText)) {
                                        $shouldExclude = true;
                                        logParser("DEBUG articles filter: Removed article '{$articleData['title']}' (ID {$id}) - knee/walking article for gastro query");
                                    }
                                    elseif (preg_match('/(сердц|кардиолог|аритми|гипертон|ишеми)/ui', $articleText) && 
                                            !preg_match('/(тошнот|рвот|живот|брюшн|желудок|гастроэнтеролог|изжог|гастрит|диспепси|панкреатит|диаре|запор|метеоризм|газообразован|гореч.*рту|сухост.*рту|брюшн.*полост|кишечник)/ui', $articleText)) {
                                        $shouldExclude = true;
                                        logParser("DEBUG articles filter: Removed article '{$articleData['title']}' (ID {$id}) - cardiac article for gastro query");
                                    }
                                }
                                // Исключаем статьи про хламидии для запросов про переломы/травмы
                                elseif (preg_match('/(перелом|травм|ушиб|вывих|растяжен|разрыв.*связ|разрыв.*сухожил|травматолог|мениск)/ui', $questionLower)) {
                                    if (preg_match('/(хламид|хламидиоз|синдром.*рейтер|инфекционн.*заболеван|полов.*путем)/ui', $articleText) && 
                                        !preg_match('/(перелом|травм|ушиб|вывих|растяжен|разрыв.*связ|разрыв.*сухожил|травматолог|ортопед|мениск|артроскоп|коленн.*сустав)/ui', $articleText)) {
                                        $shouldExclude = true;
                                        logParser("DEBUG articles filter: Removed article '{$articleData['title']}' (ID {$id}) - chlamydia/infection article for trauma query");
                                    }
                                }
                                // Исключаем статьи про колено для запросов про плечо
                                elseif (preg_match('/(плеч|плечев.*сустав|плечев.*бол)/ui', $questionLower)) {
                                    if (preg_match('/(колен|коленн|гонартроз|мениск.*колен)/ui', $articleText) && 
                                        !preg_match('/(плеч|плечев.*сустав|плечев.*бол)/ui', $articleText)) {
                                        $shouldExclude = true;
                                        logParser("DEBUG articles filter: Removed article '{$articleData['title']}' (ID {$id}) - knee article for shoulder query");
                                    }
                                }
                                // Исключаем статьи про простату для запросов про почки
                                elseif (preg_match('/(почк|нефролог|мочекаменн|камн.*почк|пиелонефрит)/ui', $questionLower)) {
                                    if (preg_match('/(простат|простатит|аденом.*простат|мужск.*проблем)/ui', $articleText) && 
                                        !preg_match('/(почк|нефролог|мочекаменн|камн.*почк|пиелонефрит|мочевыводящ)/ui', $articleText)) {
                                        $shouldExclude = true;
                                        logParser("DEBUG articles filter: Removed article '{$articleData['title']}' (ID {$id}) - prostate article for kidney query");
                                    }
                                }
                                // Исключаем статьи про желудок для запросов про боль внизу живота
                                elseif (preg_match('/(боль.*внизу.*живот|болит.*внизу.*живот|низ.*живот|тазов.*бол)/ui', $questionLower)) {
                                    if (preg_match('/(желудок|гастрит|язв.*желудк|анатоми.*желудк|строен.*желудк)/ui', $articleText) && 
                                        !preg_match('/(внизу.*живот|тазов.*бол|проктолог|уролог|гинеколог|мочевой.*пузыр|кишечник.*нижн)/ui', $articleText)) {
                                        $shouldExclude = true;
                                        logParser("DEBUG articles filter: Removed article '{$articleData['title']}' (ID {$id}) - stomach article for lower abdomen query");
                                    }
                                }
                                // Исключаем статьи про колено для запросов про левое подреберье
                                elseif (preg_match('/(лев.*подребер|подребер.*лев)/ui', $questionLower)) {
                                    if (preg_match('/(колен|коленн|гонартроз|мениск.*колен)/ui', $articleText) && 
                                        !preg_match('/(подребер|селезенк|желудок|панкреас|поджелудочн)/ui', $articleText)) {
                                        $shouldExclude = true;
                                        logParser("DEBUG articles filter: Removed article '{$articleData['title']}' (ID {$id}) - knee article for left hypochondrium query");
                                    }
                                }
                            }
                        }
                        
                        if (!$shouldExclude) {
                            $filteredArticleIds[] = $id;
                        }
                    } else {
                        // Если не нашли данные, оставляем ID (на всякий случай)
                        $filteredArticleIds[] = $id;
                    }
                }
                $validatedDataIds[$section_name] = $filteredArticleIds;
            }
        }
        
        // Инициализируем массив для fallback логики, если его еще нет
        if (!isset($validatedDataIds[$section_name])) {
            $validatedDataIds[$section_name] = [];
        }
        
        logParser("DEBUG fallback check: section_name='{$section_name}', question='{$question}', validatedDataIds count: " . count($validatedDataIds[$section_name] ?? []) . ", aiDataIds isset: " . (isset($aiDataIds[$section_name]) ? 'YES' : 'NO') . ", aiDataIds count: " . (isset($aiDataIds[$section_name]) && is_array($aiDataIds[$section_name]) ? count($aiDataIds[$section_name]) : 0));
        
        // FALLBACK ЛОГИКА ДЛЯ СПЕЦИАЛИЗАЦИЙ: выполняется даже если AI не вернул результатов
        if ($section_name === 'specializations') {
            $questionLower = mb_strtolower($question);
            logParser("DEBUG specializations fallback OUTSIDE: Starting fallback logic for query '{$question}', validatedDataIds count: " . count($validatedDataIds[$section_name] ?? []) . ", db available: " . (isset($db) ? 'YES' : 'NO') . ", widget available: " . (isset($widget) ? 'YES' : 'NO') . ", widget_id: " . ($widget['id'] ?? 'N/A'));
            
            // FALLBACK для боли в шейно-воротниковой зоне: добавляем Невролога и Травматолога-ортопеда
            $isNeckShoulderQuery = preg_match('/(шейно.*воротников|воротников.*зон|шейн.*воротников|боль.*шейн.*воротников)/ui', $questionLower);
            
            if ($isNeckShoulderQuery) {
                // Добавляем Невролога
                try {
                    $stmtNeurology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%невролог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtNeurology->execute([$widget['id']]);
                    $neurologyRow = $stmtNeurology->fetch();
                    if ($neurologyRow) {
                        $neurologyId = (int)$neurologyRow['id'];
                        if (!in_array($neurologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $neurologyId;
                            logParser("DEBUG specializations neck shoulder fallback OUTSIDE: Added neurology specialization (ID {$neurologyId}) for neck shoulder query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations neck shoulder fallback OUTSIDE: ERROR adding neurology - " . $e->getMessage());
                }
                
                // Добавляем Травматолога-ортопеда
                try {
                    $stmtTraumatology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%травматолог%' OR LOWER(pf.field_value) LIKE '%ортопед%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtTraumatology->execute([$widget['id']]);
                    $traumatologyRow = $stmtTraumatology->fetch();
                    if ($traumatologyRow) {
                        $traumatologyId = (int)$traumatologyRow['id'];
                        if (!in_array($traumatologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $traumatologyId;
                            logParser("DEBUG specializations neck shoulder fallback OUTSIDE: Added traumatology specialization (ID {$traumatologyId}) for neck shoulder query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations neck shoulder fallback OUTSIDE: ERROR adding traumatology - " . $e->getMessage());
                }
            }
            
            // FALLBACK для переломов, травм, менисков, плеч: добавляем Травматолога-ортопеда
            $isTraumaQuery = preg_match('/(перелом|травм|ушиб|вывих|растяжен|разрыв.*связ|разрыв.*сухожил|травматолог|мениск|плеч|плечев.*сустав|плечев.*бол|артроскоп|коленн.*сустав)/ui', $questionLower);
            
            logParser("DEBUG specializations fallback OUTSIDE: isTraumaQuery=" . ($isTraumaQuery ? 'YES' : 'NO') . " for query '{$question}'");
            
            if ($isTraumaQuery) {
                try {
                    $stmtTraumatology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%травматолог%' OR LOWER(pf.field_value) LIKE '%ортопед%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtTraumatology->execute([$widget['id']]);
                    $traumatologyRow = $stmtTraumatology->fetch();
                    if ($traumatologyRow) {
                        $traumatologyId = (int)$traumatologyRow['id'];
                        if (!in_array($traumatologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $traumatologyId;
                            logParser("DEBUG specializations trauma fallback OUTSIDE: Added traumatology specialization (ID {$traumatologyId}) for trauma query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations trauma fallback OUTSIDE: ERROR adding traumatology - " . $e->getMessage());
                }
            }
            
            // FALLBACK для отека Квинке: добавляем Аллергологию
            $isQuinckeQuery = preg_match('/(отек.*квинке|квинке|ангионевротическ.*отек|аллергическ.*отек)/ui', $questionLower);
            
            logParser("DEBUG specializations quincke fallback OUTSIDE: isQuinckeQuery=" . ($isQuinckeQuery ? 'YES' : 'NO') . " for query '{$question}'");
            
            if ($isQuinckeQuery) {
                try {
                    $stmtAllergology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%аллерголог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtAllergology->execute([$widget['id']]);
                    $allergologyRow = $stmtAllergology->fetch();
                    if ($allergologyRow) {
                        $allergologyId = (int)$allergologyRow['id'];
                        logParser("DEBUG specializations quincke fallback OUTSIDE: Found allergology ID {$allergologyId}, checking if already in array. Current validatedDataIds: " . json_encode($validatedDataIds[$section_name] ?? []));
                        if (!in_array($allergologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $allergologyId;
                            logParser("DEBUG specializations quincke fallback OUTSIDE: Added allergology specialization (ID {$allergologyId}) for quincke query '{$question}'. New count: " . count($validatedDataIds[$section_name]));
                        } else {
                            logParser("DEBUG specializations quincke fallback OUTSIDE: Allergology ID {$allergologyId} already in validatedDataIds");
                        }
                    } else {
                        logParser("DEBUG specializations quincke fallback OUTSIDE: No allergology specialization found in DB for query '{$question}'");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations quincke fallback OUTSIDE: ERROR adding allergology - " . $e->getMessage());
                }
            }
            
            // FALLBACK для больничного листа: добавляем Терапию
            $isSickLeaveQuery = preg_match('/(больничн.*лист|справк|мед.*справк|лист.*нетрудоспособност)/ui', $questionLower);
            
            if ($isSickLeaveQuery) {
                try {
                    $stmtTherapy = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%терапи%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtTherapy->execute([$widget['id']]);
                    $therapyRow = $stmtTherapy->fetch();
                    if ($therapyRow) {
                        $therapyId = (int)$therapyRow['id'];
                        if (!in_array($therapyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $therapyId;
                            logParser("DEBUG specializations sick leave fallback OUTSIDE: Added therapy specialization (ID {$therapyId}) for sick leave query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations sick leave fallback OUTSIDE: ERROR adding therapy - " . $e->getMessage());
                }
            }
            
            // FALLBACK для фарингита: добавляем Терапию и Педиатрию (если про ребенка)
            $isPharyngitisQuery = preg_match('/(фарингит|воспален.*горл|боль.*горл|першен.*горл)/ui', $questionLower);
            
            if ($isPharyngitisQuery) {
                // Добавляем Терапию
                try {
                    $stmtTherapy = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%терапи%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtTherapy->execute([$widget['id']]);
                    $therapyRow = $stmtTherapy->fetch();
                    if ($therapyRow) {
                        $therapyId = (int)$therapyRow['id'];
                        if (!in_array($therapyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $therapyId;
                            logParser("DEBUG specializations pharyngitis fallback OUTSIDE: Added therapy specialization (ID {$therapyId}) for pharyngitis query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations pharyngitis fallback OUTSIDE: ERROR adding therapy - " . $e->getMessage());
                }
                
                // Добавляем Педиатрию (если запрос может быть про ребенка)
                if (preg_match('/(ребенок|дет|малыш|подросток)/ui', $questionLower)) {
                    try {
                        $stmtPediatrics = $db->prepare("
                            SELECT DISTINCT pi.id
                            FROM parsed_items pi
                            JOIN parsed_fields pf ON pi.id = pf.item_id
                            WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                            AND (LOWER(pf.field_value) LIKE '%педиатр%')
                            AND pi.is_duplicate = 0
                            LIMIT 1
                        ");
                        $stmtPediatrics->execute([$widget['id']]);
                        $pediatricsRow = $stmtPediatrics->fetch();
                        if ($pediatricsRow) {
                            $pediatricsId = (int)$pediatricsRow['id'];
                            if (!in_array($pediatricsId, $validatedDataIds[$section_name])) {
                                $validatedDataIds[$section_name][] = $pediatricsId;
                                logParser("DEBUG specializations pharyngitis fallback OUTSIDE: Added pediatrics specialization (ID {$pediatricsId}) for pharyngitis query '{$question}'");
                            }
                        }
                    } catch (Exception $e) {
                        logParser("DEBUG specializations pharyngitis fallback OUTSIDE: ERROR adding pediatrics - " . $e->getMessage());
                    }
                }
            }
            
            // FALLBACK для слабости и усталости: добавляем Терапию, Эндокринологию, Кардиологию
            $isWeaknessQuery = preg_match('/(слабост|усталост|утомляемост|общ.*недомоган|постоянн.*усталост|хроническ.*усталост)/ui', $questionLower);
            
            if ($isWeaknessQuery) {
                // Добавляем Терапию
                try {
                    $stmtTherapy = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%терапи%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtTherapy->execute([$widget['id']]);
                    $therapyRow = $stmtTherapy->fetch();
                    if ($therapyRow) {
                        $therapyId = (int)$therapyRow['id'];
                        if (!in_array($therapyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $therapyId;
                            logParser("DEBUG specializations weakness fallback OUTSIDE: Added therapy specialization (ID {$therapyId}) for weakness query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations weakness fallback OUTSIDE: ERROR adding therapy - " . $e->getMessage());
                }
                
                // Добавляем Эндокринологию
                try {
                    $stmtEndocrinology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%эндокринолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtEndocrinology->execute([$widget['id']]);
                    $endocrinologyRow = $stmtEndocrinology->fetch();
                    if ($endocrinologyRow) {
                        $endocrinologyId = (int)$endocrinologyRow['id'];
                        if (!in_array($endocrinologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $endocrinologyId;
                            logParser("DEBUG specializations weakness fallback OUTSIDE: Added endocrinology specialization (ID {$endocrinologyId}) for weakness query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations weakness fallback OUTSIDE: ERROR adding endocrinology - " . $e->getMessage());
                }
                
                // Добавляем Кардиологию
                try {
                    $stmtCardiology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%кардиолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtCardiology->execute([$widget['id']]);
                    $cardiologyRow = $stmtCardiology->fetch();
                    if ($cardiologyRow) {
                        $cardiologyId = (int)$cardiologyRow['id'];
                        if (!in_array($cardiologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $cardiologyId;
                            logParser("DEBUG specializations weakness fallback OUTSIDE: Added cardiology specialization (ID {$cardiologyId}) for weakness query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations weakness fallback OUTSIDE: ERROR adding cardiology - " . $e->getMessage());
                }
            }
            
            // FALLBACK для повышенной потливости: добавляем Эндокринологию и Терапию
            $isSweatingQuery = preg_match('/(потливост|пот|гипергидроз|повышенн.*потливост)/ui', $questionLower);
            
            if ($isSweatingQuery) {
                // Добавляем Эндокринологию
                try {
                    $stmtEndocrinology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%эндокринолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtEndocrinology->execute([$widget['id']]);
                    $endocrinologyRow = $stmtEndocrinology->fetch();
                    if ($endocrinologyRow) {
                        $endocrinologyId = (int)$endocrinologyRow['id'];
                        if (!in_array($endocrinologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $endocrinologyId;
                            logParser("DEBUG specializations sweating fallback OUTSIDE: Added endocrinology specialization (ID {$endocrinologyId}) for sweating query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations sweating fallback OUTSIDE: ERROR adding endocrinology - " . $e->getMessage());
                }
                
                // Добавляем Терапию
                try {
                    $stmtTherapy = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%терапи%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtTherapy->execute([$widget['id']]);
                    $therapyRow = $stmtTherapy->fetch();
                    if ($therapyRow) {
                        $therapyId = (int)$therapyRow['id'];
                        if (!in_array($therapyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $therapyId;
                            logParser("DEBUG specializations sweating fallback OUTSIDE: Added therapy specialization (ID {$therapyId}) for sweating query '{$question}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations sweating fallback OUTSIDE: ERROR adding therapy - " . $e->getMessage());
                }
            }
            
            // FALLBACK для сухости во рту: добавляем Терапию и Эндокринологию
            $isDryMouthQuery = preg_match('/(сухост.*рту|сух.*во.*рту|ксеростоми|сухост.*во.*рту)/ui', $questionLower);
            
            logParser("DEBUG specializations dry mouth fallback OUTSIDE: isDryMouthQuery=" . ($isDryMouthQuery ? 'YES' : 'NO') . " for query '{$question}'");
            
            if ($isDryMouthQuery) {
                // Добавляем Терапию
                try {
                    $stmtTherapy = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%терапи%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtTherapy->execute([$widget['id']]);
                    $therapyRow = $stmtTherapy->fetch();
                    if ($therapyRow) {
                        $therapyId = (int)$therapyRow['id'];
                        logParser("DEBUG specializations dry mouth fallback OUTSIDE: Found therapy ID {$therapyId}, checking if already in array. Current validatedDataIds: " . json_encode($validatedDataIds[$section_name] ?? []));
                        if (!in_array($therapyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $therapyId;
                            logParser("DEBUG specializations dry mouth fallback OUTSIDE: Added therapy specialization (ID {$therapyId}) for dry mouth query '{$question}'. New count: " . count($validatedDataIds[$section_name]));
                        } else {
                            logParser("DEBUG specializations dry mouth fallback OUTSIDE: Therapy ID {$therapyId} already in validatedDataIds");
                        }
                    } else {
                        logParser("DEBUG specializations dry mouth fallback OUTSIDE: No therapy specialization found in DB for query '{$question}'");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations dry mouth fallback OUTSIDE: ERROR adding therapy - " . $e->getMessage());
                }
                
                // Добавляем Эндокринологию
                try {
                    $stmtEndocrinology = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                        AND (LOWER(pf.field_value) LIKE '%эндокринолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtEndocrinology->execute([$widget['id']]);
                    $endocrinologyRow = $stmtEndocrinology->fetch();
                    if ($endocrinologyRow) {
                        $endocrinologyId = (int)$endocrinologyRow['id'];
                        logParser("DEBUG specializations dry mouth fallback OUTSIDE: Found endocrinology ID {$endocrinologyId}, checking if already in array. Current validatedDataIds: " . json_encode($validatedDataIds[$section_name] ?? []));
                        if (!in_array($endocrinologyId, $validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name][] = $endocrinologyId;
                            logParser("DEBUG specializations dry mouth fallback OUTSIDE: Added endocrinology specialization (ID {$endocrinologyId}) for dry mouth query '{$question}'. New count: " . count($validatedDataIds[$section_name]));
                        } else {
                            logParser("DEBUG specializations dry mouth fallback OUTSIDE: Endocrinology ID {$endocrinologyId} already in validatedDataIds");
                        }
                    } else {
                        logParser("DEBUG specializations dry mouth fallback OUTSIDE: No endocrinology specialization found in DB for query '{$question}'");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specializations dry mouth fallback OUTSIDE: ERROR adding endocrinology - " . $e->getMessage());
                }
            }
            
            logParser("DEBUG specializations fallback OUTSIDE: Finished fallback logic for query '{$question}'. Final validatedDataIds count: " . count($validatedDataIds[$section_name] ?? []) . ", IDs: " . json_encode($validatedDataIds[$section_name] ?? []));
        }
        
        // ДОПОЛНИТЕЛЬНЫЙ FALLBACK для специалистов: добавляем всех релевантных специалистов для гастроэнтерологических запросов
        if ($section_name === 'specialists') {
            // Инициализируем массив, если его нет (на случай, если AI вернул пустой результат)
            if (!isset($validatedDataIds[$section_name])) {
                $validatedDataIds[$section_name] = [];
            }
            
            $questionLower = mb_strtolower($question);
            $isGastroQuery = preg_match('/(гастроэнтеролог|желуд|живот|тошнот|изжог|рвот|диаре|запор|метеоризм|газообразован|гореч.*рту|сухост.*рту)/ui', $question);
            
            // Всегда добавляем всех гастроэнтерологов для гастроэнтерологических запросов
            if ($isGastroQuery) {
                $currentCount = count($validatedDataIds[$section_name] ?? []);
                logParser("DEBUG specialists gastro fallback: AI selected {$currentCount} specialists for gastro query '{$question}'. Adding all gastro specialists.");
                
                try {
                    $gastroSpecIds = [];
                    $stmtGastroSpec = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%гастроэнтеролог%')
                        AND pi.is_duplicate = 0
                        LIMIT 100
                    ");
                    $stmtGastroSpec->execute([$widget['id']]);
                    
                    while ($row = $stmtGastroSpec->fetch()) {
                        $gastroSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($gastroSpecIds)) {
                        // Объединяем с уже выбранными специалистами
                        $currentIds = $validatedDataIds[$section_name] ?? [];
                        $validatedDataIds[$section_name] = array_unique(array_merge($currentIds, $gastroSpecIds));
                        logParser("DEBUG specialists gastro fallback: Added " . count($gastroSpecIds) . " gastro specialists. Total now: " . count($validatedDataIds[$section_name]));
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specialists gastro fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // ДОПОЛНИТЕЛЬНЫЙ FALLBACK для неврологов и травматологов при запросах про боль в спине/пояснице
            $isBackPainQuery = preg_match('/(острая.*боль.*спин|боль.*спин|боль.*поясниц|поясниц.*бол|болит.*поясниц|спин.*бол|люмбаго|прострел)/ui', $question);
            
            if ($isBackPainQuery) {
                $currentCount = count($validatedDataIds[$section_name] ?? []);
                logParser("DEBUG specialists back pain fallback: AI selected {$currentCount} specialists for back pain query '{$question}'. Adding all neurologists and traumatologists.");
                
                try {
                    // Добавляем всех неврологов
                    $neuroSpecIds = [];
                    $stmtNeuroSpec = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%невролог%')
                        AND pi.is_duplicate = 0
                        LIMIT 100
                    ");
                    $stmtNeuroSpec->execute([$widget['id']]);
                    
                    while ($row = $stmtNeuroSpec->fetch()) {
                        $neuroSpecIds[] = (int)$row['id'];
                    }
                    
                    // Добавляем всех травматологов
                    $traumaSpecIds = [];
                    $stmtTraumaSpec = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%травматолог%' OR LOWER(pf.field_value) LIKE '%ортопед%')
                        AND pi.is_duplicate = 0
                        LIMIT 100
                    ");
                    $stmtTraumaSpec->execute([$widget['id']]);
                    
                    while ($row = $stmtTraumaSpec->fetch()) {
                        $traumaSpecIds[] = (int)$row['id'];
                    }
                    
                    // Объединяем всех специалистов
                    if (!empty($neuroSpecIds) || !empty($traumaSpecIds)) {
                        $currentIds = $validatedDataIds[$section_name] ?? [];
                        $allBackPainSpecIds = array_merge($neuroSpecIds, $traumaSpecIds);
                        $validatedDataIds[$section_name] = array_unique(array_merge($currentIds, $allBackPainSpecIds));
                        logParser("DEBUG specialists back pain fallback: Added " . count($neuroSpecIds) . " neurologists and " . count($traumaSpecIds) . " traumatologists. Total now: " . count($validatedDataIds[$section_name]));
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specialists back pain fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // ДОПОЛНИТЕЛЬНЫЙ FALLBACK для терапевтов при кардиологических запросах и других случаях
            $isCardiacQuery = preg_match('/(одышк|боль.*груд|сердц|давлен|гипертон|аритми|тахикарди)/ui', $question);
            $isTemperatureQuery = preg_match('/(температур|лихорадк|жар|озноб)/ui', $question);
            $isAppetiteQuery = preg_match('/(потеря.*аппетит|аппетит.*потеря|нет.*аппетит|плох.*аппетит)/ui', $question);
            $isChildQuery = preg_match('/(ребенок|детск|малыш|младенец|грудн|новорожденн)/ui', $question);
            
            if ($isCardiacQuery || $isTemperatureQuery || $isAppetiteQuery) {
                $currentCount = count($validatedDataIds[$section_name] ?? []);
                $queryType = $isCardiacQuery ? 'cardiac' : ($isTemperatureQuery ? 'temperature' : 'appetite');
                
                // Если это запрос про ребенка и температуру, добавляем педиатров вместо терапевтов
                if ($isTemperatureQuery && $isChildQuery) {
                    logParser("DEBUG specialists pediatrician fallback: AI selected {$currentCount} specialists for child temperature query '{$question}'. Adding all pediatricians.");
                    
                    try {
                        $pediatricianSpecIds = [];
                        $stmtPediatricianSpec = $db->prepare("
                            SELECT DISTINCT pi.id
                            FROM parsed_items pi
                            JOIN parsed_fields pf ON pi.id = pf.item_id
                            WHERE pi.widget_id = ? 
                            AND pi.section_name = 'specialists'
                            AND (LOWER(pf.field_value) LIKE '%педиатр%')
                            AND pi.is_duplicate = 0
                            LIMIT 100
                        ");
                        $stmtPediatricianSpec->execute([$widget['id']]);
                        
                        while ($row = $stmtPediatricianSpec->fetch()) {
                            $pediatricianSpecIds[] = (int)$row['id'];
                        }
                        
                        if (!empty($pediatricianSpecIds)) {
                            // Объединяем с уже выбранными специалистами
                            $currentIds = $validatedDataIds[$section_name] ?? [];
                            $validatedDataIds[$section_name] = array_unique(array_merge($currentIds, $pediatricianSpecIds));
                            logParser("DEBUG specialists pediatrician fallback: Added " . count($pediatricianSpecIds) . " pediatricians for child temperature query. Total now: " . count($validatedDataIds[$section_name]));
                        }
                    } catch (Exception $e) {
                        logParser("DEBUG specialists pediatrician fallback: ERROR - " . $e->getMessage());
                    }
                } else {
                    // Для взрослых запросов добавляем терапевтов
                    logParser("DEBUG specialists therapist fallback: AI selected {$currentCount} specialists for {$queryType} query '{$question}'. Adding all therapists.");
                    
                    try {
                        $therapistSpecIds = [];
                        $stmtTherapistSpec = $db->prepare("
                            SELECT DISTINCT pi.id
                            FROM parsed_items pi
                            JOIN parsed_fields pf ON pi.id = pf.item_id
                            WHERE pi.widget_id = ? 
                            AND pi.section_name = 'specialists'
                            AND (LOWER(pf.field_value) LIKE '%терапевт%')
                            AND pi.is_duplicate = 0
                            LIMIT 100
                        ");
                        $stmtTherapistSpec->execute([$widget['id']]);
                        
                        while ($row = $stmtTherapistSpec->fetch()) {
                            $therapistSpecIds[] = (int)$row['id'];
                        }
                        
                        if (!empty($therapistSpecIds)) {
                            // Объединяем с уже выбранными специалистами
                            $currentIds = $validatedDataIds[$section_name] ?? [];
                            $validatedDataIds[$section_name] = array_unique(array_merge($currentIds, $therapistSpecIds));
                            logParser("DEBUG specialists therapist fallback: Added " . count($therapistSpecIds) . " therapists for {$queryType} query. Total now: " . count($validatedDataIds[$section_name]));
                        }
                    } catch (Exception $e) {
                        logParser("DEBUG specialists therapist fallback: ERROR - " . $e->getMessage());
                    }
                }
            }
            
            // ДОПОЛНИТЕЛЬНЫЙ FALLBACK для хирургов при запросах про ожоги
            $isBurnQuery = preg_match('/(ожог|ожог)/ui', $question);
            
            if ($isBurnQuery) {
                $currentCount = count($validatedDataIds[$section_name] ?? []);
                logParser("DEBUG specialists burn fallback: AI selected {$currentCount} specialists for burn query '{$question}'. Adding all surgeons.");
                
                try {
                    $surgeonSpecIds = [];
                    $stmtSurgeonSpec = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%хирург%')
                        AND pi.is_duplicate = 0
                        LIMIT 100
                    ");
                    $stmtSurgeonSpec->execute([$widget['id']]);
                    
                    while ($row = $stmtSurgeonSpec->fetch()) {
                        $surgeonSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($surgeonSpecIds)) {
                        $currentIds = $validatedDataIds[$section_name] ?? [];
                        $validatedDataIds[$section_name] = array_unique(array_merge($currentIds, $surgeonSpecIds));
                        logParser("DEBUG specialists burn fallback: Added " . count($surgeonSpecIds) . " surgeons. Total now: " . count($validatedDataIds[$section_name]));
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specialists burn fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // ДОПОЛНИТЕЛЬНЫЙ FALLBACK для косметологов и дерматологов при запросах про акне и родинки
            $isAcneQuery = preg_match('/(акне|угр|прыщ)/ui', $question);
            $isMoleQuery = preg_match('/(удал.*родинк|родинк.*удал|удален.*родинк)/ui', $question);
            $isWrinklesQuery = preg_match('/(убрать.*морщин|морщин.*убрать|разгладить.*морщин)/ui', $question);
            
            if ($isAcneQuery || $isMoleQuery || $isWrinklesQuery) {
                $currentCount = count($validatedDataIds[$section_name] ?? []);
                logParser("DEBUG specialists cosmetic fallback: AI selected {$currentCount} specialists for cosmetic query '{$question}'. Adding all cosmetologists and dermatologists.");
                
                try {
                    // Добавляем косметологов
                    $cosmetologistSpecIds = [];
                    $stmtCosmetologistSpec = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%косметолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 100
                    ");
                    $stmtCosmetologistSpec->execute([$widget['id']]);
                    
                    while ($row = $stmtCosmetologistSpec->fetch()) {
                        $cosmetologistSpecIds[] = (int)$row['id'];
                    }
                    
                    // Добавляем дерматологов и дерматовенерологов
                    $dermatologistSpecIds = [];
                    $stmtDermatologistSpec = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%дерматолог%' OR LOWER(pf.field_value) LIKE '%дерматовенеролог%')
                        AND pi.is_duplicate = 0
                        LIMIT 100
                    ");
                    $stmtDermatologistSpec->execute([$widget['id']]);
                    
                    while ($row = $stmtDermatologistSpec->fetch()) {
                        $dermatologistSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($cosmetologistSpecIds) || !empty($dermatologistSpecIds)) {
                        $currentIds = $validatedDataIds[$section_name] ?? [];
                        $allCosmeticSpecIds = array_merge($cosmetologistSpecIds, $dermatologistSpecIds);
                        $validatedDataIds[$section_name] = array_unique(array_merge($currentIds, $allCosmeticSpecIds));
                        logParser("DEBUG specialists cosmetic fallback: Added " . count($cosmetologistSpecIds) . " cosmetologists and " . count($dermatologistSpecIds) . " dermatologists. Total now: " . count($validatedDataIds[$section_name]));
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specialists cosmetic fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // ДОПОЛНИТЕЛЬНЫЙ FALLBACK для ЛОРов при кровотечении из носа
            $isNosebleedQuery = preg_match('/(кровотечен.*нос|нос.*кровотечен|кровь.*нос|нос.*кровь)/ui', $question);
            
            if ($isNosebleedQuery) {
                $currentCount = count($validatedDataIds[$section_name] ?? []);
                logParser("DEBUG specialists nosebleed fallback: AI selected {$currentCount} specialists for nosebleed query '{$question}'. Adding all ENT specialists.");
                
                try {
                    $entSpecIds = [];
                    $stmtEntSpec = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%лор%' OR LOWER(pf.field_value) LIKE '%оториноларинголог%')
                        AND pi.is_duplicate = 0
                        LIMIT 100
                    ");
                    $stmtEntSpec->execute([$widget['id']]);
                    
                    while ($row = $stmtEntSpec->fetch()) {
                        $entSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($entSpecIds)) {
                        $currentIds = $validatedDataIds[$section_name] ?? [];
                        $validatedDataIds[$section_name] = array_unique(array_merge($currentIds, $entSpecIds));
                        logParser("DEBUG specialists nosebleed fallback: Added " . count($entSpecIds) . " ENT specialists. Total now: " . count($validatedDataIds[$section_name]));
                    }
                } catch (Exception $e) {
                    logParser("DEBUG specialists nosebleed fallback: ERROR - " . $e->getMessage());
                }
            }
        }
    } else {
        error_log("DEBUG FALLBACK TRIGGERED: section='{$section_name}', question='{$question}', aiDataIds=" . json_encode($aiDataIds));
        logParser("DEBUG fallback: AI returned no data for section '{$section_name}', question '{$question}'. Initializing fallback.");
        $validatedDataIds[$section_name] = [];
        
        // ИСПРАВЛЕНИЕ 1: Для специалистов - если AI не нашел, но были найдены услуги - ищем специалистов по услугам
        if ($section_name === 'specialists') {
            error_log("DEBUG SPECIALISTS FALLBACK: Entering block for question '{$question}', validatedDataIds=" . json_encode($validatedDataIds));
            logParser("DEBUG fallback specialists: Processing fallback for specialists, question '{$question}'");
            $questionLower = mb_strtolower($question);
            
            // Для "Затрудненное дыхание" всегда проверяем fallback, даже если AI что-то нашел
            $isBreathingQuery = preg_match('/(затруднен.*дыхан|дыхан.*затруднен|тяжело.*дышать|дышать.*тяжело)/ui', $question);
            // Для гастроэнтерологических запросов проверяем fallback, если выбрано меньше 5 специалистов
            $isGastroQuery = preg_match('/(гастроэнтеролог|желуд|живот|тошнот|изжог|рвот|диаре|запор|метеоризм|газообразован|гореч.*рту|сухост.*рту)/ui', $question);
            $shouldUseFallback = empty($validatedDataIds[$section_name]) || 
                                ($isBreathingQuery && count($validatedDataIds[$section_name]) < 2) ||
                                ($isGastroQuery && count($validatedDataIds[$section_name]) < 5);
            
            if ($shouldUseFallback) {
                logParser("DEBUG specialists: " . (empty($validatedDataIds[$section_name]) ? "AI did not select any specialists" : "Found only " . count($validatedDataIds[$section_name]) . " specialist(s) for " . ($isBreathingQuery ? "breathing" : ($isGastroQuery ? "gastro" : "general")) . " query") . " for query '{$question}'. Using fallback.");
            }
            
            // Проверяем, есть ли уже выбранные услуги
            // Нужно получить список услуг из кеша stage2 или из БД
            // Для простоты - ищем специалистов по ключевым словам из вопроса
            $searchSpecialties = [];
            
            // Определяем специальность из вопроса
            if (preg_match('/(невролог|голов|мигрен|спин.*бол|боль.*спин)/ui', $question)) {
                $searchSpecialties[] = 'невролог';
            }
            if (preg_match('/(судорог|спазм.*мышц|сведен.*ног)/ui', $question)) {
                $searchSpecialties[] = 'невролог';
                $searchSpecialties[] = 'терапевт';
            }
            if (preg_match('/(терапевт|температур|просту|грипп|орви|орз|насморк|кашель|чихание|ларингит|потеря.*голос|голос.*потеря)/ui', $question)) {
                $searchSpecialties[] = 'терапевт';
                // Для простуды, ларингита также добавляем ЛОР
                if (preg_match('/(просту|насморк|кашель|чихание|орви|орз|ларингит|потеря.*голос|голос.*потеря)/ui', $question)) {
                    $searchSpecialties[] = 'оториноларинголог';
                    $searchSpecialties[] = 'лор';
                }
            }
            if (preg_match('/(кардиолог|сердц|давлен|гипертон|ритм.*сердц|сердц.*ритм|аритми|тахикард|брадикард|нарушен.*ритм|перебои.*сердц|сердц.*перебои|грудн.*клетк|боль.*груд)/ui', $question)) {
                $searchSpecialties[] = 'кардиолог';
                // Для нарушений ритма сердца и болей в груди не добавляем просто терапевтов без кардиологической специализации
            }
            if (preg_match('/(гастроэнтеролог|желуд|живот|тошнот|изжог)/ui', $question)) {
                $searchSpecialties[] = 'гастроэнтеролог';
            }
            if (preg_match('/(уролог|мочеиспускан|почк|цистит)/ui', $question)) {
                $searchSpecialties[] = 'уролог';
            }
            // ЛОР - расширенный поиск для всех ЛОР-симптомов
            if (preg_match('/(лор|оториноларинголог|нос|горл|ухо|шум.*ушах|ушах.*шум|охриплост|хрипот|насморк|ринит|синусит|гайморит|отит|ангин|фарингит|тонзиллит|кровотечен.*нос|носов.*кровотечен)/ui', $question)) {
                $searchSpecialties[] = 'оториноларинголог';
                $searchSpecialties[] = 'лор';
                $searchSpecialties[] = 'отоларинголог';
            }
            if (preg_match('/(дерматолог|кож|сыпь|акне)/ui', $question)) {
                $searchSpecialties[] = 'дерматолог';
            }
            if (preg_match('/(педиатр|ребенок|детск|скарлатин|корь|краснух|коклюш|ветрянк|паротит|температур.*ребенк|ребенк.*температур)/ui', $question)) {
                $searchSpecialties[] = 'педиатр';
            }
            // Офтальмология - расширенный поиск для всех глазных проблем
            if (preg_match('/(офтальмолог|глаз|зрен|конъюнктивит|катаракт|глауком|очк|подбор.*очков|подбор.*линз|контактн.*линз|линз.*контактн|дальнозорк|близорукость|астигматизм|диагностик.*зрен)/ui', $question)) {
                $searchSpecialties[] = 'офтальмолог';
                $searchSpecialties[] = 'окулист';
            }
            if (preg_match('/(аллерголог|аллерг|аллерги|аллергия|холод.*аллерг|аллерг.*холод|крапивниц)/ui', $question)) {
                $searchSpecialties[] = 'аллерголог';
                $searchSpecialties[] = 'иммунолог';
                $searchSpecialties[] = 'иммунолог-аллерголог';
            }
            if (preg_match('/(эндокринолог|щитовидн|диабет|гормон|резк.*потеря.*вес|потеря.*вес|снижен.*вес)/ui', $question)) {
                $searchSpecialties[] = 'эндокринолог';
                // Для резкой потери веса также добавляем гастроэнтерологов и терапевтов
                if (preg_match('/(резк.*потеря.*вес|потеря.*вес|снижен.*вес)/ui', $question)) {
                    $searchSpecialties[] = 'гастроэнтеролог';
                    $searchSpecialties[] = 'терапевт';
                    $searchSpecialties[] = 'психиатр'; // для исключения депрессии, анорексии
                }
            }
            if (preg_match('/(хирург|аппендицит|грыж|операц)/ui', $question)) {
                $searchSpecialties[] = 'хирург';
            }
            // Травматология - расширенный поиск для всех травм и проблем с суставами
            if (preg_match('/(травматолог|перелом|вывих|ушиб|растяжен|связок|сустав.*бол|боль.*сустав|плечев.*сустав|голеностоп|колен.*сустав|пяточн.*шпор|шпор.*пяточн|бурсит|боль.*всех.*сустав|всех.*сустав.*бол|остеохондроз|остеоартроз|артроз)/ui', $question)) {
                $searchSpecialties[] = 'травматолог';
                $searchSpecialties[] = 'ортопед';
                $searchSpecialties[] = 'травматолог-ортопед';
                // Для болей во всех суставах и бурсита добавляем ревматолога, терапевта, невролога
                if (preg_match('/(бурсит|боль.*всех.*сустав|всех.*сустав.*бол|полиартрит|полиартралг)/ui', $question)) {
                    $searchSpecialties[] = 'ревматолог';
                    $searchSpecialties[] = 'терапевт';
                    $searchSpecialties[] = 'невролог';
                    $searchSpecialties[] = 'хирург';
                }
                // Для остеохондроза и остеоартроза добавляем неврологов и ревматологов
                if (preg_match('/(остеохондроз|остеоартроз|артроз)/ui', $question)) {
                    $searchSpecialties[] = 'невролог';
                    $searchSpecialties[] = 'ревматолог';
                    $searchSpecialties[] = 'артролог';
                }
            }
            if (preg_match('/(ревматолог|артрит|подагр|ревматизм|волчанк|скованность.*суставах|полиартрит)/ui', $question)) {
                // Ревматолога нет, но есть терапевты с ревматологией
                $searchSpecialties[] = 'терапевт';
                $searchSpecialties[] = 'ревматолог';
            }
            if (preg_match('/(педиатр|ребенка|детей|новорожденн|корь|краснух|скарлатин|коклюш|ветрянк)/ui', $question)) {
                $searchSpecialties[] = 'педиатр';
            }
            if (preg_match('/(пульмонолог|одышк|бронхит|пневмон|хобл|астма|легких|удушь|задыхаю|приступ.*удушь|удушь.*приступ|пневмония)/ui', $question)) {
                $searchSpecialties[] = 'пульмонолог';
                $searchSpecialties[] = 'пульмонология';
                $searchSpecialties[] = 'врач-пульмонолог';
                $searchSpecialties[] = 'терапевт';
                $searchSpecialties[] = 'кардиолог';
                $searchSpecialties[] = 'аллерголог';
            }
            if (preg_match('/(инфекционист|лимфоузл|лимфатич|увеличен.*узл)/ui', $question)) {
                // Инфекциониста нет, используем терапевта (он может направить к нужному специалисту)
                $searchSpecialties[] = 'терапевт';
                // Также нужен гематолог/онколог для серьезных случаев
                $searchSpecialties[] = 'онколог';
                $searchSpecialties[] = 'гематолог';
            }
            if (preg_match('/(флеболог|отек.*ног|отеч.*ног|вен.*недостаточн|варикоз|боль.*икр|икр.*бол|икроножн.*бол|боль.*при.*ходьб|ходьб.*бол)/ui', $question)) {
                // Флеболога нет, используем терапевта + кардиолога + хирурга
                $searchSpecialties[] = 'терапевт';
                $searchSpecialties[] = 'кардиолог';
                $searchSpecialties[] = 'хирург';
                $searchSpecialties[] = 'нефролог'; // отеки могут быть из-за почек
                // Для болей в икрах при ходьбе также добавляем невролога
                if (preg_match('/(боль.*икр|икр.*бол|икроножн.*бол|боль.*при.*ходьб|ходьб.*бол)/ui', $question)) {
                    $searchSpecialties[] = 'невролог';
                }
            }
            if (preg_match('/(метеоризм|газообразован|вздут.*живот|горечь.*рту|сухость.*рту)/ui', $question)) {
                $searchSpecialties[] = 'гастроэнтеролог';
                $searchSpecialties[] = 'терапевт';
            }
            if (preg_match('/(ортопед|плоскостоп|косолап|сколиоз|нарушен.*осанк|искривлен.*позвоночник)/ui', $question)) {
                // Ортопеда может не быть, используем травматолога
                $searchSpecialties[] = 'травматолог';
                $searchSpecialties[] = 'ортопед';
            }
            if (preg_match('/(сомнолог|бессонн|апноэ|храп|нарушен.*сн)/ui', $question)) {
                // Сомнолога нет, используем невролога + терапевта
                $searchSpecialties[] = 'невролог';
                $searchSpecialties[] = 'терапевт';
            }
            if (preg_match('/(проктолог|геморрой|трещин.*анальн|кровь.*стул)/ui', $question)) {
                $searchSpecialties[] = 'проктолог';
                $searchSpecialties[] = 'хирург';
                // Для геморроя также добавляем гастроэнтерологов и терапевтов (консервативное лечение)
                if (preg_match('/(геморрой)/ui', $question)) {
                    $searchSpecialties[] = 'гастроэнтеролог';
                    $searchSpecialties[] = 'терапевт';
                }
            }
            if (preg_match('/(онколог|опухол|новообразован|рак|саркома)/ui', $question)) {
                $searchSpecialties[] = 'онколог';
                $searchSpecialties[] = 'терапевт';
            }
            if (preg_match('/(подагр|ревматолог)/ui', $question)) {
                $searchSpecialties[] = 'ревматолог';
                $searchSpecialties[] = 'хирург';
                $searchSpecialties[] = 'терапевт';
                $searchSpecialties[] = 'гастроэнтеролог';
            }
            if (preg_match('/(гинеколог|месячн|менструац|эндометриоз|миома|яичник)/ui', $question)) {
                $searchSpecialties[] = 'гинеколог';
            }
            if (preg_match('/(маммолог|грудь.*железа|уплотнен.*груд|мастопатия)/ui', $question)) {
                $searchSpecialties[] = 'маммолог';
                $searchSpecialties[] = 'гинеколог';
            }
            if (preg_match('/(нефролог|пиелонефрит|гломерулонефрит|почечн.*недостаточн)/ui', $question)) {
                $searchSpecialties[] = 'нефролог';
                $searchSpecialties[] = 'уролог';
                $searchSpecialties[] = 'терапевт';
            }
            if (preg_match('/(гематолог|анемия|лейкоз|тромбоцит|эритроцит|лейкоцит)/ui', $question)) {
                $searchSpecialties[] = 'гематолог';
                $searchSpecialties[] = 'терапевт';
            }
            if (preg_match('/(психиатр|психотерапевт|депресс|тревог|паническ|фобия|бессонниц)/ui', $question)) {
                $searchSpecialties[] = 'психотерапевт';
                $searchSpecialties[] = 'психиатр';
            }
            if (preg_match('/(косметолог|пигментн.*пятн|акне|морщин|уход.*кож)/ui', $question)) {
                $searchSpecialties[] = 'косметолог';
                $searchSpecialties[] = 'дерматолог';
            }
            if (preg_match('/(трихолог|выпадени.*волос|алопеция|перхоть)/ui', $question)) {
                $searchSpecialties[] = 'трихолог';
                $searchSpecialties[] = 'дерматолог';
            }
            if (preg_match('/(сосудист.*хирург|атеросклероз|тромбоз|эмболия)/ui', $question)) {
                $searchSpecialties[] = 'хирург';
                $searchSpecialties[] = 'кардиолог';
            }
            if (preg_match('/(остеопороз|костн.*ткан|хрупкость.*кост|перелом.*позвонк)/ui', $question)) {
                // Остеопороз - эндокринолог + травматолог + терапевт
                $searchSpecialties[] = 'эндокринолог';
                $searchSpecialties[] = 'травматолог';
                $searchSpecialties[] = 'терапевт';
            }
            if (preg_match('/(люмбаго|острая.*боль.*спин|прострел|боль.*пояснице)/ui', $question)) {
                $searchSpecialties[] = 'невролог';
                $searchSpecialties[] = 'травматолог';
            }
            if (preg_match('/(утренн.*скованн|скованн.*сустав|боль.*всех.*сустав|полиартрит)/ui', $question)) {
                // Ревматические заболевания
                $searchSpecialties[] = 'ревматолог';
                $searchSpecialties[] = 'терапевт';
            }
            if (preg_match('/(потер.*обоня|потер.*вкус|аносмия)/ui', $question)) {
                $searchSpecialties[] = 'лор';
                $searchSpecialties[] = 'невролог';
            }
            if (preg_match('/(отек.*квинке|ангионевротическ.*отек|аллергическ.*отек)/ui', $question)) {
                $searchSpecialties[] = 'аллерголог';
                $searchSpecialties[] = 'терапевт';
            }
            if (preg_match('/(мрт|кт|компьютерн.*томограф|магнитн.*резонан|рентген|диагностик)/ui', $question)) {
                // Для диагностических услуг
                $searchSpecialties[] = 'терапевт';
            }
            // УЗИ - отдельная обработка
            if (preg_match('/(узи|ультразвук|ультразвуков)/ui', $question)) {
                $searchSpecialties[] = 'ультразвуковой диагностики';
                $searchSpecialties[] = 'узи';
                // Для УЗИ малого таза добавляем гинекологов
                if (preg_match('/(узи.*мал.*таз|мал.*таз.*узи|узи.*орган.*мал.*таз|гинеколог.*узи)/ui', $question)) {
                    $searchSpecialties[] = 'гинеколог';
                    $searchSpecialties[] = 'акушер-гинеколог';
                }
            }
            if (preg_match('/(диспансеризац|медосмотр|профосмотр|чекап|обследован)/ui', $question)) {
                $searchSpecialties[] = 'терапевт';
            }
            
            // ============================================================================
            // НОВЫЕ FALLBACK для проблемных запросов
            // ============================================================================
            
            
            // Синюшность пальцев (хроническая)
            if (preg_match('/(синюшн.*пальц|синюшн.*кож|цианоз)/ui', $question)) {
                $searchSpecialties[] = 'кардиолог';
                $searchSpecialties[] = 'терапевт';
            }
            
            // Боль в правом подреберье
            if (preg_match('/(боль.*прав.*подреберь|боль.*прав.*бок|прав.*бок.*бол)/ui', $question)) {
                $searchSpecialties[] = 'гастроэнтеролог';
                $searchSpecialties[] = 'терапевт';
            }
            // Боль в левом подреберье
            if (preg_match('/(боль.*лев.*подреберь|боль.*лев.*бок|лев.*бок.*бол)/ui', $question)) {
                $searchSpecialties[] = 'гастроэнтеролог';
                $searchSpecialties[] = 'терапевт';
                $searchSpecialties[] = 'кардиолог';
                $searchSpecialties[] = 'хирург';
            }
            
            // Диарея, понос
            if (preg_match('/(диарея|понос|жидк.*стул|расстройств.*кишечник)/ui', $question)) {
                $searchSpecialties[] = 'гастроэнтеролог';
                $searchSpecialties[] = 'терапевт';
            }
            
            // Рвота и понос
            if (preg_match('/(рвот.*понос|понос.*рвот|рвот.*диарея)/ui', $question)) {
                $searchSpecialties[] = 'гастроэнтеролог';
                $searchSpecialties[] = 'терапевт';
            }
            
            // Ночная потливость
            if (preg_match('/(ночн.*потлив|пот.*ноч|потею.*ноч)/ui', $question)) {
                $searchSpecialties[] = 'эндокринолог';
                $searchSpecialties[] = 'терапевт';
            }
            
            // Слабость и утомляемость
            if (preg_match('/(слабость.*утомляемость|утомляемость.*слабость|постоянн.*усталост|хроническ.*усталост)/ui', $question)) {
                $searchSpecialties[] = 'терапевт';
                $searchSpecialties[] = 'эндокринолог';
            }
            
            // Инфаркт профилактика
            if (preg_match('/(инфаркт.*профилактик|профилактик.*инфаркт|предотврат.*инфаркт)/ui', $question)) {
                $searchSpecialties[] = 'кардиолог';
                $searchSpecialties[] = 'терапевт';
            }
            
            // ============================================================================
            // FALLBACK для проблемных запросов из теста (10/40 и ниже)
            // ============================================================================
            
            // Боль в шейно-воротниковой зоне
            if (preg_match('/(шейно.*воротников|воротников.*зон|боль.*ше.*воротник|ше.*воротник.*бол)/ui', $question)) {
                $searchSpecialties[] = 'невролог';
                $searchSpecialties[] = 'травматолог';
                $searchSpecialties[] = 'терапевт';
            }
            
            // ОРВИ симптомы и лечение
            if (preg_match('/(орви.*симптом|орви.*лечен|симптом.*орви|лечен.*орви)/ui', $question)) {
                $searchSpecialties[] = 'терапевт';
                $searchSpecialties[] = 'педиатр';
            }
            
            // Тремор рук причины
            if (preg_match('/(тремор.*рук|тремор.*причин|рук.*тремор|дрож.*рук)/ui', $question)) {
                $searchSpecialties[] = 'невролог';
                $searchSpecialties[] = 'эндокринолог';
                $searchSpecialties[] = 'терапевт';
            }
            
            // Межпозвоночная грыжа лечение
            if (preg_match('/(межпозвон.*грыж|грыж.*межпозвон|грыж.*позвоночник|позвон.*грыж)/ui', $question)) {
                $searchSpecialties[] = 'невролог';
                $searchSpecialties[] = 'травматолог';
                $searchSpecialties[] = 'ортопед';
            }
            
            // Боль в спине после сна
            if (preg_match('/(боль.*спин.*после.*сн|спин.*после.*сн.*бол|утренн.*боль.*спин|боль.*спин.*утр)/ui', $question)) {
                $searchSpecialties[] = 'невролог';
                $searchSpecialties[] = 'травматолог';
                $searchSpecialties[] = 'терапевт';
            }
            
            // Затрудненное дыхание (улучшенный fallback с пульмонологом)
            if (preg_match('/(затруднен.*дыхан|дыхан.*затруднен|тяжело.*дышать|дышать.*тяжело)/ui', $question)) {
                $searchSpecialties[] = 'пульмонолог';
                $searchSpecialties[] = 'пульмонология';
                $searchSpecialties[] = 'врач-пульмонолог';
                $searchSpecialties[] = 'терапевт';
                $searchSpecialties[] = 'кардиолог';
            }
            
            // Кашель с мокротой
            if (preg_match('/(кашель.*мокрот|мокрот.*кашель|кашель.*слизь|мокрый.*кашель)/ui', $question)) {
                $searchSpecialties[] = 'пульмонолог';
                $searchSpecialties[] = 'пульмонология';
                $searchSpecialties[] = 'врач-пульмонолог';
                $searchSpecialties[] = 'терапевт';
                $searchSpecialties[] = 'оториноларинголог';
            }
            
            // Отек Квинке (ангионевротический отек) - уже обработан выше в строке 1492
            // Дублируем здесь для fallback поиска специалистов
            if (preg_match('/(отек.*квинке|квинке.*отек|ангионевротическ.*отек|аллергическ.*отек)/ui', $question)) {
                $searchSpecialties[] = 'аллерголог';
                $searchSpecialties[] = 'терапевт';
                $searchSpecialties[] = 'иммунолог';
            }
            
            // Тяжесть в ногах
            if (preg_match('/(тяжест.*ног|ног.*тяжест|отек.*ног|ног.*отек|усталост.*ног|ног.*усталост)/ui', $question)) {
                $searchSpecialties[] = 'флеболог';
                $searchSpecialties[] = 'терапевт';
                $searchSpecialties[] = 'кардиолог';
                $searchSpecialties[] = 'нефролог';
            }
            
            // Боль в области пупка
            if (preg_match('/(боль.*пупк|пупк.*бол|боль.*област.*пупк|област.*пупк.*бол)/ui', $question)) {
                $searchSpecialties[] = 'гастроэнтеролог';
                $searchSpecialties[] = 'терапевт';
                $searchSpecialties[] = 'хирург';
                $searchSpecialties[] = 'гинеколог';
                $searchSpecialties[] = 'уролог';
            }
            
            // Ревматолог - проверяем наличие в базе перед добавлением
            // Если ревматолога нет, используем терапевта или травматолога
            if (preg_match('/(ревматолог|ревматоидн|ревматическ|полиартрит|подагр|волчанк|скованност.*сустав)/ui', $question)) {
                // Сначала проверяем, есть ли ревматолог в базе
                try {
                    $stmtCheckRheum = $db->prepare("
                        SELECT COUNT(*) as cnt
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%ревматолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtCheckRheum->execute([$widget['id']]);
                    $hasRheumatologist = $stmtCheckRheum->fetch()['cnt'] > 0;
                    
                    if ($hasRheumatologist) {
                        $searchSpecialties[] = 'ревматолог';
                    } else {
                        // Если ревматолога нет, используем альтернативы
                        $searchSpecialties[] = 'терапевт';
                        $searchSpecialties[] = 'травматолог';
                        $searchSpecialties[] = 'ортопед';
                    }
                } catch (Exception $e) {
                    // В случае ошибки используем альтернативы
                    $searchSpecialties[] = 'терапевт';
                    $searchSpecialties[] = 'травматолог';
                }
            }
            

            if (!empty($searchSpecialties)) {
                $specialistIds = [];
                foreach ($searchSpecialties as $specialty) {
                    try {
                        // Расширенный поиск для пульмонолога (включая варианты названий)
                        $searchPatterns = ['%' . $specialty . '%', '%' . mb_strtolower($specialty) . '%'];
                        if ($specialty === 'пульмонолог' || $specialty === 'пульмонология' || $specialty === 'врач-пульмонолог') {
                            $searchPatterns[] = '%пульмонолог%';
                            $searchPatterns[] = '%пульмонология%';
                            $searchPatterns[] = '%врач-пульмонолог%';
                            $searchPatterns[] = '%врач пульмонолог%';
                        }
                        // Расширенный поиск для ЛОР
                        if ($specialty === 'оториноларинголог' || $specialty === 'лор' || $specialty === 'отоларинголог') {
                            $searchPatterns[] = '%оториноларинголог%';
                            $searchPatterns[] = '%лор%';
                            $searchPatterns[] = '%отоларинголог%';
                            $searchPatterns[] = '%врач-оториноларинголог%';
                            $searchPatterns[] = '%врач оториноларинголог%';
                        }
                        // Расширенный поиск для офтальмолога
                        if ($specialty === 'офтальмолог' || $specialty === 'окулист') {
                            $searchPatterns[] = '%офтальмолог%';
                            $searchPatterns[] = '%окулист%';
                            $searchPatterns[] = '%врач-офтальмолог%';
                            $searchPatterns[] = '%врач офтальмолог%';
                        }
                        // Расширенный поиск для травматолога
                        if ($specialty === 'травматолог' || $specialty === 'ортопед' || $specialty === 'травматолог-ортопед') {
                            $searchPatterns[] = '%травматолог%';
                            $searchPatterns[] = '%ортопед%';
                            $searchPatterns[] = '%травматолог-ортопед%';
                            $searchPatterns[] = '%врач-травматолог%';
                            $searchPatterns[] = '%врач травматолог%';
                        }
                        
                        $placeholders = implode(' OR LOWER(pf.field_value) LIKE ', array_fill(0, count($searchPatterns), '?'));
                        
                        // Для кардиологических запросов (нарушение ритма сердца) фильтруем терапевтов без кардиологической специализации
                        $isCardiacRhythmQuery = preg_match('/(ритм.*сердц|сердц.*ритм|аритми|тахикард|брадикард|нарушен.*ритм)/ui', $question);
                        $additionalFilter = '';
                        if ($isCardiacRhythmQuery && $specialty === 'кардиолог') {
                            // Для кардиологических запросов исключаем терапевтов без кардиологической специализации
                            $additionalFilter = " AND NOT (LOWER(pf.field_value) LIKE '%терапевт%' AND LOWER(pf.field_value) NOT LIKE '%кардиолог%')";
                        }
                        
                        // Убираем ограничение - выбираем всех релевантных специалистов (лимит 100 для безопасности)
                        $limit = 100;
                        $stmtSpec = $db->prepare("
                            SELECT DISTINCT pi.id
                            FROM parsed_items pi
                            JOIN parsed_fields pf ON pi.id = pf.item_id
                            WHERE pi.widget_id = ? 
                            AND pi.section_name = 'specialists'
                            AND pi.is_duplicate = 0
                            AND (LOWER(pf.field_value) LIKE $placeholders)
                            $additionalFilter
                            LIMIT $limit
                        ");
                        $stmtSpec->execute(array_merge([$widget['id']], $searchPatterns));
                        
                        while ($row = $stmtSpec->fetch()) {
                            $specialistIds[] = (int)$row['id'];
                        }
                        
                        // Для кардиологических запросов дополнительно фильтруем результаты
                        if ($isCardiacRhythmQuery && !empty($specialistIds)) {
                            $filteredSpecialistIds = [];
                            foreach ($specialistIds as $specId) {
                                // Получаем описание специалиста
                                $stmtDesc = $db->prepare("
                                    SELECT pf.field_value
                                    FROM parsed_fields pf
                                    WHERE pf.item_id = ? AND pf.field_name = 'description'
                                ");
                                $stmtDesc->execute([$specId]);
                                $descRow = $stmtDesc->fetch();
                                $description = mb_strtolower($descRow['field_value'] ?? '');
                                
                                // Если это кардиолог или терапевт с кардиологической специализацией - добавляем
                                if (strpos($description, 'кардиолог') !== false || 
                                    (strpos($description, 'терапевт') !== false && strpos($description, 'кардиолог') !== false)) {
                                    $filteredSpecialistIds[] = $specId;
                                }
                            }
                            // Если после фильтрации остались специалисты - используем их
                            if (!empty($filteredSpecialistIds)) {
                                $specialistIds = $filteredSpecialistIds;
                            }
                        }
                        
                        // Для УЗИ малого таза - ищем гинекологов и УЗИ-специалистов
                        $isUziPelvisQuery = preg_match('/(узи.*мал.*таз|мал.*таз.*узи|узи.*орган.*мал.*таз|гинеколог.*узи)/ui', $question);
                        if ($isUziPelvisQuery && ($specialty === 'гинеколог' || $specialty === 'акушер-гинеколог' || $specialty === 'ультразвуковой диагностики' || $specialty === 'узи')) {
                            // Дополнительно ищем гинекологов и УЗИ-специалистов
                            try {
                                $uziPelvisIds = [];
                                $stmtUziPelvis = $db->prepare("
                                    SELECT DISTINCT pi.id
                                    FROM parsed_items pi
                                    JOIN parsed_fields pf ON pi.id = pf.item_id
                                    WHERE pi.widget_id = ? 
                                    AND pi.section_name = 'specialists'
                                    AND pi.is_duplicate = 0
                                    AND (LOWER(pf.field_value) LIKE '%гинеколог%'
                                         OR LOWER(pf.field_value) LIKE '%акушер-гинеколог%'
                                         OR LOWER(pf.field_value) LIKE '%ультразвуковой диагностики%'
                                         OR LOWER(pf.field_value) LIKE '%узи%')
                                    LIMIT 20
                                ");
                                $stmtUziPelvis->execute([$widget['id']]);
                                
                                while ($row = $stmtUziPelvis->fetch()) {
                                    $uziPelvisIds[] = (int)$row['id'];
                                }
                                
                                if (!empty($uziPelvisIds)) {
                                    $specialistIds = array_unique(array_merge($specialistIds, $uziPelvisIds));
                                }
                            } catch (Exception $e) {
                                logParser("DEBUG UZI pelvis fallback: ERROR - " . $e->getMessage());
                            }
                        }
                    } catch (Exception $e) {
                        logParser("DEBUG specialists fallback: ERROR for $specialty: " . $e->getMessage());
                    }
                }
                
                if (!empty($specialistIds) && $shouldUseFallback) {
                    // Добавляем к уже найденным специалистам (если есть)
                    if (empty($validatedDataIds[$section_name])) {
                        $validatedDataIds[$section_name] = array_unique($specialistIds);
                    } else {
                        // Для гастроэнтерологических запросов добавляем всех найденных гастроэнтерологов
                        $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $specialistIds));
                    }
                    logParser("DEBUG specialists: Found " . count($validatedDataIds[$section_name]) . " specialists by specialty (" . implode(', ', $searchSpecialties) . ") for query '{$question}'. Fallback IDs: " . json_encode($specialistIds));
                } elseif (!empty($specialistIds) && !$shouldUseFallback) {
                    logParser("DEBUG specialists: Fallback found " . count($specialistIds) . " specialists but shouldUseFallback=false for query '{$question}'");
                }
            }
            
            // ============================================================================
            // АДМИНИСТРАТИВНЫЕ ЗАПРОСЫ - fallback для специалистов (в блоке specialists)
            // ============================================================================
            
            // Медицинская книжка - специалисты
            if (preg_match('/(медицинск.*книжк|медкнижк|санитарн.*книжк)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG administrative specialists: Detected medical book query, adding specialists");
                
                try {
                    $medBookSpecIds = [];
                    $stmtMedBookSpec = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND LOWER(pf.field_value) LIKE '%терапевт%'
                        AND pi.is_duplicate = 0
                        LIMIT 3
                    ");
                    $stmtMedBookSpec->execute([$widget['id']]);
                    
                    while ($row = $stmtMedBookSpec->fetch()) {
                        $medBookSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($medBookSpecIds)) {
                        $validatedDataIds[$section_name] = array_unique($medBookSpecIds);
                        logParser("DEBUG administrative specialists: Added " . count($medBookSpecIds) . " specialists for medical book");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG administrative specialists: ERROR - " . $e->getMessage());
                }
            }
            
            // Больничный лист - специалисты
            if (preg_match('/(больничн.*лист|лист.*нетрудоспособност|больничн|нетрудоспособност)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG administrative specialists: Detected sick leave query, adding specialists");
                
                try {
                    $sickLeaveSpecIds = [];
                    $stmtSickLeaveSpec = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND LOWER(pf.field_value) LIKE '%терапевт%'
                        AND pi.is_duplicate = 0
                        LIMIT 5
                    ");
                    $stmtSickLeaveSpec->execute([$widget['id']]);
                    
                    while ($row = $stmtSickLeaveSpec->fetch()) {
                        $sickLeaveSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($sickLeaveSpecIds)) {
                        $validatedDataIds[$section_name] = array_unique($sickLeaveSpecIds);
                        logParser("DEBUG administrative specialists: Added " . count($sickLeaveSpecIds) . " specialists for sick leave");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG administrative specialists: ERROR - " . $e->getMessage());
                }
            }
            
            // Справка на оружие - специалисты
            if (preg_match('/(справк.*оружи|оружи.*справк|справк.*оруже|оруже.*справк)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG administrative specialists: Detected weapon certificate query, adding specialists");
                
                try {
                    $weaponSpecIds = [];
                    $stmtWeaponSpec = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%терапевт%'
                             OR LOWER(pf.field_value) LIKE '%психиатр%'
                             OR LOWER(pf.field_value) LIKE '%нарколог%')
                        AND pi.is_duplicate = 0
                        LIMIT 5
                    ");
                    $stmtWeaponSpec->execute([$widget['id']]);
                    
                    while ($row = $stmtWeaponSpec->fetch()) {
                        $weaponSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($weaponSpecIds)) {
                        $validatedDataIds[$section_name] = array_unique($weaponSpecIds);
                        logParser("DEBUG administrative specialists: Added " . count($weaponSpecIds) . " specialists for weapon certificate");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG administrative specialists: ERROR - " . $e->getMessage());
                }
            }
            
            // Общий fallback для справок (справка о болезни, справка и т.д.)
            if (preg_match('/(справк.*болезн|справк.*заболеван|справк.*здоров|справк.*медицинск|медицинск.*справк|справк.*общ)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG administrative specialists: Detected general certificate query, adding therapists");
                
                try {
                    $certSpecIds = [];
                    $stmtCertSpec = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND LOWER(pf.field_value) LIKE '%терапевт%'
                        AND pi.is_duplicate = 0
                        LIMIT 5
                    ");
                    $stmtCertSpec->execute([$widget['id']]);
                    
                    while ($row = $stmtCertSpec->fetch()) {
                        $certSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($certSpecIds)) {
                        $validatedDataIds[$section_name] = array_unique($certSpecIds);
                        logParser("DEBUG administrative specialists: Added " . count($certSpecIds) . " therapists for general certificate");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG administrative specialists: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для болей в грудной клетке
            if (preg_match('/(боль.*грудн.*клетк|грудн.*клетк.*бол|боль.*груд|груд.*бол|боль.*сердц)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG chest pain fallback: Detected chest pain query, adding specialists");
                
                try {
                    $chestPainSpecIds = [];
                    // Ищем кардиологов, терапевтов с кардиологической специализацией, гастроэнтерологов
                    $stmtChestPain = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%кардиолог%'
                             OR (LOWER(pf.field_value) LIKE '%терапевт%' AND LOWER(pf.field_value) LIKE '%кардиолог%')
                             OR LOWER(pf.field_value) LIKE '%гастроэнтеролог%')
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtChestPain->execute([$widget['id']]);
                    
                    while ($row = $stmtChestPain->fetch()) {
                        $chestPainSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($chestPainSpecIds)) {
                        $validatedDataIds[$section_name] = array_unique($chestPainSpecIds);
                        logParser("DEBUG chest pain fallback: Added " . count($chestPainSpecIds) . " specialists for chest pain");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG chest pain fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для УЗИ малого таза
            if (preg_match('/(узи.*мал.*таз|мал.*таз.*узи|узи.*орган.*мал.*таз|гинеколог.*узи)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG UZI pelvis fallback: Detected UZI pelvis query, adding specialists");
                
                try {
                    $uziPelvisSpecIds = [];
                    // Ищем гинекологов и УЗИ-специалистов
                    $stmtUziPelvis = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%гинеколог%'
                             OR LOWER(pf.field_value) LIKE '%акушер-гинеколог%'
                             OR LOWER(pf.field_value) LIKE '%ультразвуковой диагностики%'
                             OR LOWER(pf.field_value) LIKE '%узи%')
                        AND pi.is_duplicate = 0
                        LIMIT 15
                    ");
                    $stmtUziPelvis->execute([$widget['id']]);
                    
                    while ($row = $stmtUziPelvis->fetch()) {
                        $uziPelvisSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($uziPelvisSpecIds)) {
                        $validatedDataIds[$section_name] = array_unique($uziPelvisSpecIds);
                        logParser("DEBUG UZI pelvis fallback: Added " . count($uziPelvisSpecIds) . " specialists for UZI pelvis");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG UZI pelvis fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для болей в подреберье (левом и правом)
            if (preg_match('/(боль.*лев.*подреберь|боль.*прав.*подреберь|боль.*лев.*бок|боль.*прав.*бок|лев.*подреберь.*бол|прав.*подреберь.*бол)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG subcostal pain fallback: Detected subcostal pain query, adding specialists");
                
                try {
                    $subcostalSpecIds = [];
                    // Ищем гастроэнтерологов, терапевтов, хирургов, кардиологов
                    $stmtSubcostal = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%гастроэнтеролог%'
                             OR LOWER(pf.field_value) LIKE '%терапевт%'
                             OR LOWER(pf.field_value) LIKE '%хирург%'
                             OR LOWER(pf.field_value) LIKE '%кардиолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtSubcostal->execute([$widget['id']]);
                    
                    while ($row = $stmtSubcostal->fetch()) {
                        $subcostalSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($subcostalSpecIds)) {
                        $validatedDataIds[$section_name] = array_unique($subcostalSpecIds);
                        logParser("DEBUG subcostal pain fallback: Added " . count($subcostalSpecIds) . " specialists for subcostal pain");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG subcostal pain fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для простуды и насморка
            if (preg_match('/(просту|насморк|кашель|чихание|орви|орз)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG cold fallback: Detected cold/runny nose query, adding specialists");
                
                try {
                    $coldSpecIds = [];
                    // Ищем терапевтов и ЛОР-врачей
                    $stmtCold = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%терапевт%'
                             OR LOWER(pf.field_value) LIKE '%оториноларинголог%'
                             OR LOWER(pf.field_value) LIKE '%лор%'
                             OR LOWER(pf.field_value) LIKE '%отоларинголог%')
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtCold->execute([$widget['id']]);
                    
                    while ($row = $stmtCold->fetch()) {
                        $coldSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($coldSpecIds)) {
                        $validatedDataIds[$section_name] = array_unique($coldSpecIds);
                        logParser("DEBUG cold fallback: Added " . count($coldSpecIds) . " specialists for cold/runny nose");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG cold fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для болей во всех суставах и бурсита
            if (preg_match('/(боль.*всех.*сустав|всех.*сустав.*бол|бурсит|полиартрит|полиартралг)/ui', $question) && (empty($validatedDataIds[$section_name]) || count($validatedDataIds[$section_name]) < 3)) {
                logParser("DEBUG joint pain fallback: Detected joint pain/bursitis query, adding specialists");
                
                try {
                    $jointSpecIds = [];
                    // Ищем ревматологов, терапевтов, неврологов, травматологов, хирургов
                    $stmtJoint = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%ревматолог%'
                             OR LOWER(pf.field_value) LIKE '%терапевт%'
                             OR LOWER(pf.field_value) LIKE '%невролог%'
                             OR LOWER(pf.field_value) LIKE '%травматолог%'
                             OR LOWER(pf.field_value) LIKE '%ортопед%'
                             OR LOWER(pf.field_value) LIKE '%хирург%')
                        AND pi.is_duplicate = 0
                        LIMIT 15
                    ");
                    $stmtJoint->execute([$widget['id']]);
                    
                    while ($row = $stmtJoint->fetch()) {
                        $jointSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($jointSpecIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($jointSpecIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $jointSpecIds));
                        }
                        logParser("DEBUG joint pain fallback: Added " . count($jointSpecIds) . " specialists for joint pain/bursitis");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG joint pain fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для болей в икрах при ходьбе
            if (preg_match('/(боль.*икр|икр.*бол|икроножн.*бол|боль.*при.*ходьб|ходьб.*бол)/ui', $question) && (empty($validatedDataIds[$section_name]) || count($validatedDataIds[$section_name]) < 3)) {
                logParser("DEBUG calf pain fallback: Detected calf pain query, adding specialists");
                
                try {
                    $calfSpecIds = [];
                    // Ищем терапевтов, неврологов, хирургов, кардиологов
                    $stmtCalf = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%терапевт%'
                             OR LOWER(pf.field_value) LIKE '%невролог%'
                             OR LOWER(pf.field_value) LIKE '%хирург%'
                             OR LOWER(pf.field_value) LIKE '%кардиолог%'
                             OR LOWER(pf.field_value) LIKE '%травматолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtCalf->execute([$widget['id']]);
                    
                    while ($row = $stmtCalf->fetch()) {
                        $calfSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($calfSpecIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($calfSpecIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $calfSpecIds));
                        }
                        logParser("DEBUG calf pain fallback: Added " . count($calfSpecIds) . " specialists for calf pain");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG calf pain fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для приступов удушья (КРИТИЧНО!)
            if (preg_match('/(приступ.*удушь|удушь.*приступ|задыхаюсь|не.*могу.*дышать|совсем.*не.*могу.*дышать)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG asphyxia fallback: Detected asphyxia attack query, adding specialists CRITICAL!");
                
                try {
                    $asphyxiaSpecIds = [];
                    // Ищем терапевтов, пульмонологов, аллергологов, кардиологов
                    $stmtAsphyxia = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%терапевт%'
                             OR LOWER(pf.field_value) LIKE '%пульмонолог%'
                             OR LOWER(pf.field_value) LIKE '%аллерголог%'
                             OR LOWER(pf.field_value) LIKE '%иммунолог%'
                             OR LOWER(pf.field_value) LIKE '%кардиолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtAsphyxia->execute([$widget['id']]);
                    
                    while ($row = $stmtAsphyxia->fetch()) {
                        $asphyxiaSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($asphyxiaSpecIds)) {
                        $validatedDataIds[$section_name] = array_unique($asphyxiaSpecIds);
                        logParser("DEBUG asphyxia fallback: Added " . count($asphyxiaSpecIds) . " specialists for asphyxia attack (CRITICAL)");
                    } else {
                        logParser("DEBUG asphyxia fallback: WARNING - No specialists found for asphyxia attack!");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG asphyxia fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для затрудненного дыхания
            if (preg_match('/(затруднен.*дыхан|дыхан.*затруднен|тяжело.*дышать|дышать.*тяжело)/ui', $question) && (empty($validatedDataIds[$section_name]) || count($validatedDataIds[$section_name]) < 3)) {
                logParser("DEBUG breathing difficulty fallback: Detected breathing difficulty query, adding specialists");
                
                try {
                    $breathingSpecIds = [];
                    // Ищем пульмонологов, терапевтов, кардиологов, аллергологов
                    $stmtBreathing = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%пульмонолог%'
                             OR LOWER(pf.field_value) LIKE '%терапевт%'
                             OR LOWER(pf.field_value) LIKE '%кардиолог%'
                             OR LOWER(pf.field_value) LIKE '%аллерголог%'
                             OR LOWER(pf.field_value) LIKE '%иммунолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtBreathing->execute([$widget['id']]);
                    
                    while ($row = $stmtBreathing->fetch()) {
                        $breathingSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($breathingSpecIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($breathingSpecIds);
                        } else {
                            // Фильтруем: убираем ЛОР и психиатров, оставляем только релевантных
                            $filteredBreathingIds = [];
                            foreach ($breathingSpecIds as $specId) {
                                $stmtDesc = $db->prepare("
                                    SELECT pf.field_value
                                    FROM parsed_fields pf
                                    WHERE pf.item_id = ? AND pf.field_name = 'description'
                                ");
                                $stmtDesc->execute([$specId]);
                                $descRow = $stmtDesc->fetch();
                                $description = mb_strtolower($descRow['field_value'] ?? '');
                                
                                // Исключаем ЛОР и психиатров
                                if (strpos($description, 'оториноларинголог') === false && 
                                    strpos($description, 'лор') === false &&
                                    strpos($description, 'психиатр') === false &&
                                    strpos($description, 'психотерапевт') === false) {
                                    $filteredBreathingIds[] = $specId;
                                }
                            }
                            
                            if (!empty($filteredBreathingIds)) {
                                $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $filteredBreathingIds));
                            }
                        }
                        logParser("DEBUG breathing difficulty fallback: Added " . count($breathingSpecIds) . " specialists for breathing difficulty");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG breathing difficulty fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для прививок детям
            if (preg_match('/(прививк|вакцин|иммунизац)/ui', $question) && (empty($validatedDataIds[$section_name]) || count($validatedDataIds[$section_name]) < 2)) {
                logParser("DEBUG vaccination fallback: Detected vaccination query, adding specialists");
                
                try {
                    $vaccinationSpecIds = [];
                    // Ищем педиатров, аллергологов-иммунологов
                    $stmtVaccination = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%педиатр%'
                             OR (LOWER(pf.field_value) LIKE '%аллерголог%' AND LOWER(pf.field_value) LIKE '%иммунолог%'))
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtVaccination->execute([$widget['id']]);
                    
                    while ($row = $stmtVaccination->fetch()) {
                        $vaccinationSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($vaccinationSpecIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($vaccinationSpecIds);
                        } else {
                            // Фильтруем: убираем нерелевантных (хирургов, неврологов)
                            $filteredVaccinationIds = [];
                            foreach ($vaccinationSpecIds as $specId) {
                                $stmtDesc = $db->prepare("
                                    SELECT pf.field_value
                                    FROM parsed_fields pf
                                    WHERE pf.item_id = ? AND pf.field_name = 'description'
                                ");
                                $stmtDesc->execute([$specId]);
                                $descRow = $stmtDesc->fetch();
                                $description = mb_strtolower($descRow['field_value'] ?? '');
                                
                                // Исключаем хирургов и неврологов (если нет педиатрии или аллергологии)
                                if (strpos($description, 'педиатр') !== false || 
                                    (strpos($description, 'аллерголог') !== false && strpos($description, 'иммунолог') !== false)) {
                                    $filteredVaccinationIds[] = $specId;
                                }
                            }
                            
                            if (!empty($filteredVaccinationIds)) {
                                $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $filteredVaccinationIds));
                            }
                        }
                        logParser("DEBUG vaccination fallback: Added " . count($vaccinationSpecIds) . " specialists for vaccination");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG vaccination fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для болей в тазобедренном суставе
            if (preg_match('/(боль.*тазобедренн|тазобедренн.*бол|тазобедренн.*сустав)/ui', $question) && (empty($validatedDataIds[$section_name]) || count($validatedDataIds[$section_name]) < 3)) {
                logParser("DEBUG hip pain fallback: Detected hip pain query, adding specialists");
                
                try {
                    $hipSpecIds = [];
                    // Ищем неврологов, травматологов, терапевтов
                    $stmtHip = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%невролог%'
                             OR LOWER(pf.field_value) LIKE '%травматолог%'
                             OR LOWER(pf.field_value) LIKE '%ортопед%'
                             OR LOWER(pf.field_value) LIKE '%терапевт%')
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtHip->execute([$widget['id']]);
                    
                    while ($row = $stmtHip->fetch()) {
                        $hipSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($hipSpecIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($hipSpecIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $hipSpecIds));
                        }
                        logParser("DEBUG hip pain fallback: Added " . count($hipSpecIds) . " specialists for hip pain");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG hip pain fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для ларингита и потери голоса
            if (preg_match('/(ларингит|потеря.*голос|голос.*потеря|охрип|осип)/ui', $question) && (empty($validatedDataIds[$section_name]) || count($validatedDataIds[$section_name]) < 2)) {
                logParser("DEBUG laryngitis fallback: Detected laryngitis query, adding specialists");
                
                try {
                    $laryngitisSpecIds = [];
                    // Ищем ЛОР-врачей
                    $stmtLaryngitis = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%оториноларинголог%'
                             OR LOWER(pf.field_value) LIKE '%лор%'
                             OR LOWER(pf.field_value) LIKE '%отоларинголог%')
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtLaryngitis->execute([$widget['id']]);
                    
                    while ($row = $stmtLaryngitis->fetch()) {
                        $laryngitisSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($laryngitisSpecIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($laryngitisSpecIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $laryngitisSpecIds));
                        }
                        logParser("DEBUG laryngitis fallback: Added " . count($laryngitisSpecIds) . " specialists for laryngitis");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG laryngitis fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для резкой потери веса
            if (preg_match('/(резк.*потеря.*вес|потеря.*вес|снижен.*вес)/ui', $question) && (empty($validatedDataIds[$section_name]) || count($validatedDataIds[$section_name]) < 3)) {
                logParser("DEBUG weight loss fallback: Detected weight loss query, adding specialists");
                
                try {
                    $weightLossSpecIds = [];
                    // Ищем эндокринологов, гастроэнтерологов, терапевтов, психиатров
                    $stmtWeightLoss = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%эндокринолог%'
                             OR LOWER(pf.field_value) LIKE '%гастроэнтеролог%'
                             OR LOWER(pf.field_value) LIKE '%терапевт%'
                             OR LOWER(pf.field_value) LIKE '%психиатр%')
                        AND pi.is_duplicate = 0
                        LIMIT 15
                    ");
                    $stmtWeightLoss->execute([$widget['id']]);
                    
                    while ($row = $stmtWeightLoss->fetch()) {
                        $weightLossSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($weightLossSpecIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($weightLossSpecIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $weightLossSpecIds));
                        }
                        logParser("DEBUG weight loss fallback: Added " . count($weightLossSpecIds) . " specialists for weight loss");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG weight loss fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для варикоцеле
            if (preg_match('/(варикоцеле)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG varicocele fallback: Detected varicocele query, adding specialists");
                
                try {
                    $varicoceleSpecIds = [];
                    // Ищем урологов и андрологов
                    $stmtVaricocele = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%уролог%'
                             OR LOWER(pf.field_value) LIKE '%андролог%')
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtVaricocele->execute([$widget['id']]);
                    
                    while ($row = $stmtVaricocele->fetch()) {
                        $varicoceleSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($varicoceleSpecIds)) {
                        $validatedDataIds[$section_name] = array_unique($varicoceleSpecIds);
                        logParser("DEBUG varicocele fallback: Added " . count($varicoceleSpecIds) . " specialists for varicocele");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG varicocele fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для остеохондроза и остеоартроза
            if (preg_match('/(остеохондроз|остеоартроз|артроз)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG osteochondrosis/osteoarthritis fallback: Detected query, adding specialists");
                
                try {
                    $osteoSpecIds = [];
                    // Ищем неврологов, травматологов-ортопедов, ревматологов
                    $stmtOsteo = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%невролог%'
                             OR LOWER(pf.field_value) LIKE '%травматолог%'
                             OR LOWER(pf.field_value) LIKE '%ортопед%'
                             OR LOWER(pf.field_value) LIKE '%ревматолог%'
                             OR LOWER(pf.field_value) LIKE '%артролог%')
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtOsteo->execute([$widget['id']]);
                    
                    while ($row = $stmtOsteo->fetch()) {
                        $osteoSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($osteoSpecIds)) {
                        $validatedDataIds[$section_name] = array_unique($osteoSpecIds);
                        logParser("DEBUG osteochondrosis/osteoarthritis fallback: Added " . count($osteoSpecIds) . " specialists");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG osteochondrosis/osteoarthritis fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для геморроя
            if (preg_match('/(геморро)/ui', $question) && (empty($validatedDataIds[$section_name]) || count($validatedDataIds[$section_name]) < 3)) {
                logParser("DEBUG hemorrhoids fallback: Detected hemorrhoids query, adding specialists");
                
                try {
                    $hemorrhoidsSpecIds = [];
                    // Ищем гастроэнтерологов и терапевтов для консервативного лечения
                    $stmtHemorrhoids = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%гастроэнтеролог%'
                             OR (LOWER(pf.field_value) LIKE '%терапевт%' AND (LOWER(pf.field_value) LIKE '%гастро%' OR LOWER(pf.field_value) LIKE '%жкт%')))
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtHemorrhoids->execute([$widget['id']]);
                    
                    while ($row = $stmtHemorrhoids->fetch()) {
                        $hemorrhoidsSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($hemorrhoidsSpecIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($hemorrhoidsSpecIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $hemorrhoidsSpecIds));
                        }
                        logParser("DEBUG hemorrhoids fallback: Added " . count($hemorrhoidsSpecIds) . " specialists for hemorrhoids");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG hemorrhoids fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для увеличения лимфоузлов
            if (preg_match('/(лимфоузел|лимфоузл|увеличен.*лимфоузл|лимфоузл.*увеличен)/ui', $question) && (empty($validatedDataIds[$section_name]) || count($validatedDataIds[$section_name]) < 3)) {
                logParser("DEBUG lymph nodes fallback: Detected lymph nodes query, adding specialists");
                
                try {
                    $lymphNodesSpecIds = [];
                    // Ищем хирургов, УЗИ-специалистов, онкологов
                    $stmtLymphNodes = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%хирург%'
                             OR LOWER(pf.field_value) LIKE '%ультразвуковой диагностики%'
                             OR LOWER(pf.field_value) LIKE '%узи%'
                             OR LOWER(pf.field_value) LIKE '%онколог%')
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtLymphNodes->execute([$widget['id']]);
                    
                    while ($row = $stmtLymphNodes->fetch()) {
                        $lymphNodesSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($lymphNodesSpecIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($lymphNodesSpecIds);
                        } else {
                            // Фильтруем: убираем нерелевантных (психиатр, невролог)
                            $filteredLymphNodesIds = [];
                            foreach ($lymphNodesSpecIds as $specId) {
                                $stmtDesc = $db->prepare("
                                    SELECT pf.field_value
                                    FROM parsed_fields pf
                                    WHERE pf.item_id = ? AND pf.field_name = 'description'
                                ");
                                $stmtDesc->execute([$specId]);
                                $descRow = $stmtDesc->fetch();
                                $description = mb_strtolower($descRow['field_value'] ?? '');
                                
                                // Исключаем психиатров и неврологов
                                if (strpos($description, 'психиатр') === false && 
                                    strpos($description, 'психотерапевт') === false &&
                                    strpos($description, 'невролог') === false) {
                                    $filteredLymphNodesIds[] = $specId;
                                }
                            }
                            
                            if (!empty($filteredLymphNodesIds)) {
                                $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $filteredLymphNodesIds));
                            }
                        }
                        logParser("DEBUG lymph nodes fallback: Added " . count($lymphNodesSpecIds) . " specialists for lymph nodes");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG lymph nodes fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для повышенной температуры (фильтрация нерелевантных)
            if (preg_match('/(повышенн.*температур|температур.*повышенн|высок.*температур|температур.*высок)/ui', $question) && !empty($validatedDataIds[$section_name])) {
                logParser("DEBUG temperature fallback: Filtering irrelevant specialists for temperature query");
                
                try {
                    // Фильтруем: убираем психиатров, инструкторов ЛФК
                    $filteredTempIds = [];
                    foreach ($validatedDataIds[$section_name] as $specId) {
                        $stmtDesc = $db->prepare("
                            SELECT pf.field_value
                            FROM parsed_fields pf
                            WHERE pf.item_id = ? AND pf.field_name = 'description'
                        ");
                        $stmtDesc->execute([$specId]);
                        $descRow = $stmtDesc->fetch();
                        $description = mb_strtolower($descRow['field_value'] ?? '');
                        
                        // Исключаем психиатров, инструкторов ЛФК
                        if (strpos($description, 'психиатр') === false && 
                            strpos($description, 'психотерапевт') === false &&
                            strpos($description, 'инструктор') === false &&
                            strpos($description, 'лфк') === false) {
                            $filteredTempIds[] = $specId;
                        }
                    }
                    
                    if (!empty($filteredTempIds) && count($filteredTempIds) < count($validatedDataIds[$section_name])) {
                        $validatedDataIds[$section_name] = array_unique($filteredTempIds);
                        logParser("DEBUG temperature fallback: Filtered " . (count($validatedDataIds[$section_name]) - count($filteredTempIds)) . " irrelevant specialists");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG temperature fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для подагры
            if (preg_match('/(подагр)/ui', $question) && (empty($validatedDataIds[$section_name]) || count($validatedDataIds[$section_name]) < 3)) {
                logParser("DEBUG gout fallback: Detected gout query, adding specialists");
                
                try {
                    $goutSpecIds = [];
                    // Ищем хирургов, ревматологов, терапевтов, гастроэнтерологов, неврологов
                    $stmtGout = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%хирург%'
                             OR LOWER(pf.field_value) LIKE '%ревматолог%'
                             OR LOWER(pf.field_value) LIKE '%терапевт%'
                             OR LOWER(pf.field_value) LIKE '%гастроэнтеролог%'
                             OR LOWER(pf.field_value) LIKE '%невролог%')
                        AND pi.is_duplicate = 0
                        LIMIT 15
                    ");
                    $stmtGout->execute([$widget['id']]);
                    
                    while ($row = $stmtGout->fetch()) {
                        $goutSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($goutSpecIds)) {
                        // Фильтруем: убираем нерелевантных (УЗИ, нарколог)
                        $filteredGoutIds = [];
                        foreach ($goutSpecIds as $specId) {
                            $stmtDesc = $db->prepare("
                                SELECT pf.field_value
                                FROM parsed_fields pf
                                WHERE pf.item_id = ? AND pf.field_name = 'description'
                            ");
                            $stmtDesc->execute([$specId]);
                            $descRow = $stmtDesc->fetch();
                            $description = mb_strtolower($descRow['field_value'] ?? '');
                            
                            // Исключаем УЗИ-специалистов и наркологов
                            if (strpos($description, 'ультразвуковой диагностики') === false && 
                                strpos($description, 'узи') === false &&
                                strpos($description, 'нарколог') === false) {
                                $filteredGoutIds[] = $specId;
                            }
                        }
                        
                        if (!empty($filteredGoutIds)) {
                            if (empty($validatedDataIds[$section_name])) {
                                $validatedDataIds[$section_name] = array_unique($filteredGoutIds);
                            } else {
                                $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $filteredGoutIds));
                            }
                        }
                        logParser("DEBUG gout fallback: Added " . count($filteredGoutIds) . " specialists for gout");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG gout fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для пневмонии
            if (preg_match('/(пневмония)/ui', $question) && (empty($validatedDataIds[$section_name]) || count($validatedDataIds[$section_name]) < 2)) {
                logParser("DEBUG pneumonia fallback: Detected pneumonia query, adding specialists");
                
                try {
                    $pneumoniaSpecIds = [];
                    // Ищем терапевтов, пульмонологов, кардиологов
                    $stmtPneumonia = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%терапевт%'
                             OR LOWER(pf.field_value) LIKE '%пульмонолог%'
                             OR LOWER(pf.field_value) LIKE '%кардиолог%')
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtPneumonia->execute([$widget['id']]);
                    
                    while ($row = $stmtPneumonia->fetch()) {
                        $pneumoniaSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($pneumoniaSpecIds)) {
                        // Фильтруем: убираем ЛОР
                        $filteredPneumoniaIds = [];
                        foreach ($pneumoniaSpecIds as $specId) {
                            $stmtDesc = $db->prepare("
                                SELECT pf.field_value
                                FROM parsed_fields pf
                                WHERE pf.item_id = ? AND pf.field_name = 'description'
                            ");
                            $stmtDesc->execute([$specId]);
                            $descRow = $stmtDesc->fetch();
                            $description = mb_strtolower($descRow['field_value'] ?? '');
                            
                            // Исключаем ЛОР
                            if (strpos($description, 'оториноларинголог') === false && 
                                strpos($description, 'лор') === false &&
                                strpos($description, 'отоларинголог') === false) {
                                $filteredPneumoniaIds[] = $specId;
                            }
                        }
                        
                        if (!empty($filteredPneumoniaIds)) {
                            if (empty($validatedDataIds[$section_name])) {
                                $validatedDataIds[$section_name] = array_unique($filteredPneumoniaIds);
                            } else {
                                $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $filteredPneumoniaIds));
                            }
                        }
                        logParser("DEBUG pneumonia fallback: Added " . count($filteredPneumoniaIds) . " specialists for pneumonia");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG pneumonia fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // Fallback для температуры у ребенка
            if (preg_match('/(температур.*ребенк|ребенк.*температур)/ui', $question) && (empty($validatedDataIds[$section_name]) || count($validatedDataIds[$section_name]) < 2)) {
                logParser("DEBUG child temperature fallback: Detected child temperature query, adding specialists");
                
                try {
                    $childTempSpecIds = [];
                    // Ищем педиатров
                    $stmtChildTemp = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND LOWER(pf.field_value) LIKE '%педиатр%'
                        AND pi.is_duplicate = 0
                        LIMIT 10
                    ");
                    $stmtChildTemp->execute([$widget['id']]);
                    
                    while ($row = $stmtChildTemp->fetch()) {
                        $childTempSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($childTempSpecIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($childTempSpecIds);
                        } else {
                            // Фильтруем: оставляем только педиатров, убираем взрослых терапевтов
                            $filteredChildTempIds = [];
                            foreach ($childTempSpecIds as $specId) {
                                $stmtDesc = $db->prepare("
                                    SELECT pf.field_value
                                    FROM parsed_fields pf
                                    WHERE pf.item_id = ? AND pf.field_name = 'description'
                                ");
                                $stmtDesc->execute([$specId]);
                                $descRow = $stmtDesc->fetch();
                                $description = mb_strtolower($descRow['field_value'] ?? '');
                                
                                // Оставляем только педиатров
                                if (strpos($description, 'педиатр') !== false) {
                                    $filteredChildTempIds[] = $specId;
                                }
                            }
                            
                            if (!empty($filteredChildTempIds)) {
                                $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $filteredChildTempIds));
                            }
                        }
                        logParser("DEBUG child temperature fallback: Added " . count($childTempSpecIds) . " specialists for child temperature");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG child temperature fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // ============================================================================
            // ОТЕК КВИНКЕ - fallback для специалистов (в блоке specialists)
            // ============================================================================
            $isAngioedemaQuery = preg_match('/(отек.*квинке|квинке.*отек|ангионевротическ.*отек|аллергическ.*отек)/ui', $question);
            error_log("DEBUG ANGIOEDEMA CHECK: question='{$question}', isAngioedemaQuery=" . ($isAngioedemaQuery ? 'YES' : 'NO') . ", section_name={$section_name}, validatedDataIds[{$section_name}]=" . json_encode($validatedDataIds[$section_name] ?? []));
            logParser("DEBUG angioedema fallback specialists: Checking query '{$question}', isAngioedemaQuery=" . ($isAngioedemaQuery ? 'YES' : 'NO') . ", section_name={$section_name}, validatedDataIds empty=" . (empty($validatedDataIds[$section_name]) ? 'YES' : 'NO'));
            
            if ($isAngioedemaQuery) {
                if (empty($validatedDataIds[$section_name])) {
                    error_log("DEBUG ANGIOEDEMA FALLBACK: Adding specialists for widget_id=" . $widget['id'] . ", question='{$question}'");
                    logParser("DEBUG angioedema fallback specialists: Adding specialists for widget_id=" . $widget['id']);
                
                try {
                    $angioSpecIds = [];
                    // Ищем специалистов аллергологов, терапевтов, иммунологов
                    $stmtAngioSpec = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND (LOWER(pf.field_value) LIKE '%аллерголог%' 
                             OR LOWER(pf.field_value) LIKE '%иммунолог%'
                             OR LOWER(pf.field_value) LIKE '%терапевт%')
                        AND pi.is_duplicate = 0
                        LIMIT 5
                    ");
                    $stmtAngioSpec->execute([$widget['id']]);
                    
                    while ($row = $stmtAngioSpec->fetch()) {
                        $angioSpecIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($angioSpecIds)) {
                        $validatedDataIds[$section_name] = array_unique($angioSpecIds);
                        logParser("DEBUG angioedema fallback specialists: Added " . count($angioSpecIds) . " specialists: " . json_encode($angioSpecIds));
                    } else {
                        logParser("DEBUG angioedema fallback specialists: No specialists found in database for widget_id=" . $widget['id']);
                    }
                } catch (Exception $e) {
                    logParser("DEBUG angioedema fallback specialists: ERROR - " . $e->getMessage());
                }
                } else {
                    logParser("DEBUG angioedema fallback specialists: Skipped - already have " . count($validatedDataIds[$section_name] ?? []) . " specialists");
                }
            }
        }
        
        if ($section_name === 'services') {
            logParser("DEBUG services: AI did not select any services after {$maxRetries} attempts for query '{$question}'. Checking for specialist consultations.");
            
            $questionLower = mb_strtolower($question);
            
            // ============================================================================
            // ЛОР-ВОПРОСЫ - fallback для услуг
            // ============================================================================
            if (preg_match('/(шум.*ушах|ушах.*шум|охриплост|хрипот|насморк|ринит|синусит|гайморит|отит|ангин|фарингит|тонзиллит|кровотечен.*нос|носов.*кровотечен|горл|ухо|нос)/ui', $question)) {
                logParser("DEBUG lor fallback: Detected LOR query, adding consultations");
                
                try {
                    $lorIds = [];
                    // Ищем консультации ЛОР-врача
                    $stmtLor = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND (LOWER(pf.field_value) LIKE '%оториноларинголог%' 
                             OR LOWER(pf.field_value) LIKE '%лор%'
                             OR LOWER(pf.field_value) LIKE '%отоларинголог%')
                        AND LOWER(pf.field_value) LIKE '%консультация%'
                        AND pi.is_duplicate = 0
                        ORDER BY CASE 
                            WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 1
                            WHEN LOWER(pf.field_value) LIKE '%повторная%' THEN 2
                            ELSE 3
                        END
                        LIMIT 3
                    ");
                    $stmtLor->execute([$widget['id']]);
                    
                    while ($row = $stmtLor->fetch()) {
                        $lorIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($lorIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = $lorIds;
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $lorIds));
                        }
                        logParser("DEBUG lor fallback: Added " . count($lorIds) . " LOR consultations");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG lor fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // ============================================================================
            // ОФТАЛЬМОЛОГИЯ - fallback для услуг
            // ============================================================================
            if (preg_match('/(глаз|зрен|офтальмолог|конъюнктивит|катаракт|глауком|очк|подбор.*очков|подбор.*линз|контактн.*линз|линз.*контактн|дальнозорк|близорукость|астигматизм|диагностик.*зрен)/ui', $question)) {
                logParser("DEBUG ophthalmology fallback: Detected ophthalmology query, adding consultations");
                
                try {
                    $ophIds = [];
                    // Ищем консультации офтальмолога
                    $stmtOph = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND (LOWER(pf.field_value) LIKE '%офтальмолог%' 
                             OR LOWER(pf.field_value) LIKE '%окулист%')
                        AND LOWER(pf.field_value) LIKE '%консультация%'
                        AND pi.is_duplicate = 0
                        ORDER BY CASE 
                            WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 1
                            WHEN LOWER(pf.field_value) LIKE '%повторная%' THEN 2
                            ELSE 3
                        END
                        LIMIT 3
                    ");
                    $stmtOph->execute([$widget['id']]);
                    
                    while ($row = $stmtOph->fetch()) {
                        $ophIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($ophIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = $ophIds;
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $ophIds));
                        }
                        logParser("DEBUG ophthalmology fallback: Added " . count($ophIds) . " ophthalmology consultations");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG ophthalmology fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // ============================================================================
            // ТРАВМАТОЛОГИЯ - fallback для услуг
            // ============================================================================
            if (preg_match('/(травматолог|перелом|вывих|ушиб|растяжен|связок|сустав.*бол|боль.*сустав|плечев.*сустав|голеностоп|колен.*сустав|пяточн.*шпор|шпор.*пяточн)/ui', $question)) {
                logParser("DEBUG traumatology fallback: Detected traumatology query, adding consultations and diagnostics");
                
                try {
                    $traumaIds = [];
                    // Ищем консультации травматолога
                    $stmtTrauma = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND (LOWER(pf.field_value) LIKE '%травматолог%' 
                             OR LOWER(pf.field_value) LIKE '%ортопед%')
                        AND LOWER(pf.field_value) LIKE '%консультация%'
                        AND pi.is_duplicate = 0
                        ORDER BY CASE 
                            WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 1
                            WHEN LOWER(pf.field_value) LIKE '%повторная%' THEN 2
                            ELSE 3
                        END
                        LIMIT 3
                    ");
                    $stmtTrauma->execute([$widget['id']]);
                    
                    while ($row = $stmtTrauma->fetch()) {
                        $traumaIds[] = (int)$row['id'];
                    }
                    
                    // Также ищем диагностику (рентген, МРТ суставов)
                    $stmtDiag = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND (LOWER(pf.field_value) LIKE '%рентген%сустав%' 
                             OR LOWER(pf.field_value) LIKE '%мрт%сустав%'
                             OR LOWER(pf.field_value) LIKE '%узи%сустав%')
                        AND pi.is_duplicate = 0
                        LIMIT 2
                    ");
                    $stmtDiag->execute([$widget['id']]);
                    
                    while ($row = $stmtDiag->fetch()) {
                        $traumaIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($traumaIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = $traumaIds;
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $traumaIds));
                        }
                        logParser("DEBUG traumatology fallback: Added " . count($traumaIds) . " traumatology consultations and diagnostics");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG traumatology fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // ============================================================================
            // ОТЕК КВИНКЕ - fallback для услуг
            // ============================================================================
            if (preg_match('/(отек.*квинке|квинке.*отек|ангионевротическ.*отек|аллергическ.*отек)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG angioedema fallback: Detected angioedema query, adding consultations");
                
                // Fallback для услуг
                if ($section_name === 'services') {
                
                try {
                    $angioIds = [];
                    // Ищем консультации аллерголога
                    $stmtAngio = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND (LOWER(pf.field_value) LIKE '%аллерголог%' 
                             OR LOWER(pf.field_value) LIKE '%иммунолог%')
                        AND LOWER(pf.field_value) LIKE '%консультация%'
                        AND pi.is_duplicate = 0
                        ORDER BY CASE 
                            WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 1
                            WHEN LOWER(pf.field_value) LIKE '%повторная%' THEN 2
                            ELSE 3
                        END
                        LIMIT 3
                    ");
                    $stmtAngio->execute([$widget['id']]);
                    
                    while ($row = $stmtAngio->fetch()) {
                        $angioIds[] = (int)$row['id'];
                    }
                    
                    // Также ищем консультацию терапевта (экстренный случай)
                    $stmtTherapist = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND LOWER(pf.field_value) LIKE '%терапевт%'
                        AND LOWER(pf.field_value) LIKE '%консультация%'
                        AND pi.is_duplicate = 0
                        LIMIT 1
                    ");
                    $stmtTherapist->execute([$widget['id']]);
                    
                    while ($row = $stmtTherapist->fetch()) {
                        $angioIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($angioIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($angioIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $angioIds));
                        }
                        logParser("DEBUG angioedema fallback: Added " . count($angioIds) . " angioedema consultations");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG angioedema fallback: ERROR - " . $e->getMessage());
                }
                }
            }
            
            // ============================================================================
            // ТЯЖЕСТЬ В НОГАХ - fallback для услуг
            // ============================================================================
            if (preg_match('/(тяжест.*ног|ног.*тяжест|отек.*ног|ног.*отек|усталост.*ног|ног.*усталост)/ui', $question)) {
                logParser("DEBUG leg heaviness fallback: Detected leg heaviness query, adding consultations and diagnostics");
                
                try {
                    $legIds = [];
                    // Ищем консультации флеболога, терапевта, кардиолога
                    $stmtLeg = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND (LOWER(pf.field_value) LIKE '%флеболог%' 
                             OR LOWER(pf.field_value) LIKE '%терапевт%'
                             OR LOWER(pf.field_value) LIKE '%кардиолог%')
                        AND LOWER(pf.field_value) LIKE '%консультация%'
                        AND pi.is_duplicate = 0
                        ORDER BY CASE 
                            WHEN LOWER(pf.field_value) LIKE '%флеболог%' THEN 1
                            WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 2
                            ELSE 3
                        END
                        LIMIT 3
                    ");
                    $stmtLeg->execute([$widget['id']]);
                    
                    while ($row = $stmtLeg->fetch()) {
                        $legIds[] = (int)$row['id'];
                    }
                    
                    // Также ищем диагностику (УЗИ вен, УЗИ сосудов)
                    $stmtDiag = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND (LOWER(pf.field_value) LIKE '%узи%вен%' 
                             OR LOWER(pf.field_value) LIKE '%узи%сосуд%'
                             OR LOWER(pf.field_value) LIKE '%допплер%')
                        AND pi.is_duplicate = 0
                        LIMIT 2
                    ");
                    $stmtDiag->execute([$widget['id']]);
                    
                    while ($row = $stmtDiag->fetch()) {
                        $legIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($legIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($legIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $legIds));
                        }
                        logParser("DEBUG leg heaviness fallback: Added " . count($legIds) . " leg heaviness consultations and diagnostics");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG leg heaviness fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // ============================================================================
            // БОЛЬ В ОБЛАСТИ ПУПКА - fallback для услуг
            // ============================================================================
            if (preg_match('/(боль.*пупк|пупк.*бол|боль.*област.*пупк|област.*пупк.*бол)/ui', $question)) {
                logParser("DEBUG umbilical pain fallback: Detected umbilical pain query, adding consultations and diagnostics");
                
                try {
                    $umbilicalIds = [];
                    // Ищем консультации гастроэнтеролога, терапевта, хирурга
                    $stmtUmb = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND (LOWER(pf.field_value) LIKE '%гастроэнтеролог%' 
                             OR LOWER(pf.field_value) LIKE '%терапевт%'
                             OR LOWER(pf.field_value) LIKE '%хирург%')
                        AND LOWER(pf.field_value) LIKE '%консультация%'
                        AND pi.is_duplicate = 0
                        ORDER BY CASE 
                            WHEN LOWER(pf.field_value) LIKE '%гастроэнтеролог%' THEN 1
                            WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 2
                            ELSE 3
                        END
                        LIMIT 3
                    ");
                    $stmtUmb->execute([$widget['id']]);
                    
                    while ($row = $stmtUmb->fetch()) {
                        $umbilicalIds[] = (int)$row['id'];
                    }
                    
                    // Также ищем диагностику (УЗИ брюшной полости, ФГДС)
                    $stmtDiag = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND (LOWER(pf.field_value) LIKE '%узи%брюшн%' 
                             OR LOWER(pf.field_value) LIKE '%узи%живот%'
                             OR LOWER(pf.field_value) LIKE '%фгдс%'
                             OR LOWER(pf.field_value) LIKE '%гастроскоп%')
                        AND pi.is_duplicate = 0
                        LIMIT 2
                    ");
                    $stmtDiag->execute([$widget['id']]);
                    
                    while ($row = $stmtDiag->fetch()) {
                        $umbilicalIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($umbilicalIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($umbilicalIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $umbilicalIds));
                        }
                        logParser("DEBUG umbilical pain fallback: Added " . count($umbilicalIds) . " umbilical pain consultations and diagnostics");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG umbilical pain fallback: ERROR - " . $e->getMessage());
                }
            }
            
            // ============================================================================
            // ДЕТСКИЕ ИНФЕКЦИИ - специальная обработка
            // ============================================================================
            // Для детских инфекционных заболеваний обязательно добавляем педиатра
            if (preg_match('/(скарлатин|корь|краснух|коклюш|ветрянк|свинка|паротит)/ui', $question)) {
                logParser("DEBUG pediatric: Detected pediatric infection query");
                
                try {
                    // Ищем консультацию педиатра
                    $stmtPed = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND LOWER(pf.field_value) LIKE '%педиатр%'
                        AND LOWER(pf.field_value) LIKE '%консультация%'
                        AND pi.is_duplicate = 0
                        ORDER BY CASE 
                            WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 1
                            WHEN LOWER(pf.field_value) LIKE '%повторная%' THEN 2
                            ELSE 3
                        END
                        LIMIT 2
                    ");
                    $stmtPed->execute([$widget['id']]);
                    
                    $pedIds = [];
                    while ($row = $stmtPed->fetch()) {
                        $pedIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($pedIds)) {
                        $validatedDataIds[$section_name] = $pedIds;
                        logParser("DEBUG pediatric: Found " . count($pedIds) . " pediatric consultations");
                        
                        // Также ищем общий анализ крови
                        $stmtBlood = $db->prepare("
                            SELECT DISTINCT pi.id
                            FROM parsed_items pi
                            JOIN parsed_fields pf ON pi.id = pf.item_id
                            WHERE pi.widget_id = ? 
                            AND pi.section_name = 'services'
                            AND pf.field_name = 'name'
                            AND LOWER(pf.field_value) LIKE '%общий анализ крови%'
                            AND pi.is_duplicate = 0
                            LIMIT 1
                        ");
                        $stmtBlood->execute([$widget['id']]);
                        
                        while ($row = $stmtBlood->fetch()) {
                            $validatedDataIds[$section_name][] = (int)$row['id'];
                        }
                        
                        logParser("DEBUG pediatric: Total service IDs: " . count($validatedDataIds[$section_name]));
                    } else {
                        logParser("DEBUG pediatric: No pediatric consultations found in database");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG pediatric: ERROR - " . $e->getMessage());
                }
            }
            
            // ============================================================================
            // АДМИНИСТРАТИВНЫЕ ВОПРОСЫ - специальная обработка
            // ============================================================================
            
            // Медицинская книжка
            if (preg_match('/(медицинск.*книжк|медкнижк|санитарн.*книжк)/ui', $question)) {
                logParser("DEBUG administrative: Detected medical book query");
                
                // Fallback для специалистов (терапевт, который может выдать справки)
                if ($section_name === 'specialists' && empty($validatedDataIds[$section_name])) {
                    try {
                        $medBookSpecIds = [];
                        $stmtMedBookSpec = $db->prepare("
                            SELECT DISTINCT pi.id
                            FROM parsed_items pi
                            JOIN parsed_fields pf ON pi.id = pf.item_id
                            WHERE pi.widget_id = ? 
                            AND pi.section_name = 'specialists'
                            AND LOWER(pf.field_value) LIKE '%терапевт%'
                            AND pi.is_duplicate = 0
                            LIMIT 3
                        ");
                        $stmtMedBookSpec->execute([$widget['id']]);
                        
                        while ($row = $stmtMedBookSpec->fetch()) {
                            $medBookSpecIds[] = (int)$row['id'];
                        }
                        
                        if (!empty($medBookSpecIds)) {
                            $validatedDataIds[$section_name] = array_unique($medBookSpecIds);
                            logParser("DEBUG administrative: Added " . count($medBookSpecIds) . " specialists for medical book");
                        }
                    } catch (Exception $e) {
                        logParser("DEBUG administrative specialists: ERROR - " . $e->getMessage());
                    }
                }
                
                // Fallback для услуг
                if ($section_name === 'services' && empty($validatedDataIds[$section_name])) {
                try {
                    $medBookIds = [];
                    // Ищем услуги по медицинской книжке
                    $stmtMedBook = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND (LOWER(pf.field_value) LIKE '%медицинск%книжк%' 
                             OR LOWER(pf.field_value) LIKE '%медкнижк%'
                             OR LOWER(pf.field_value) LIKE '%санитарн%книжк%')
                        AND pi.is_duplicate = 0
                        LIMIT 5
                    ");
                    $stmtMedBook->execute([$widget['id']]);
                    
                    while ($row = $stmtMedBook->fetch()) {
                        $medBookIds[] = (int)$row['id'];
                    }
                    
                    // Если не нашли, ищем справки и комиссии
                    if (empty($medBookIds)) {
                        $stmtCom = $db->prepare("
                            SELECT DISTINCT pi.id
                            FROM parsed_items pi
                            JOIN parsed_fields pf ON pi.id = pf.item_id
                            WHERE pi.widget_id = ? 
                            AND pi.section_name = 'services'
                            AND pf.field_name = 'name'
                            AND (LOWER(pf.field_value) LIKE '%справк%' 
                                 OR LOWER(pf.field_value) LIKE '%комисси%'
                                 OR LOWER(pf.field_value) LIKE '%освидетельствован%')
                            AND pi.is_duplicate = 0
                            LIMIT 5
                        ");
                        $stmtCom->execute([$widget['id']]);
                        
                        while ($row = $stmtCom->fetch()) {
                            $medBookIds[] = (int)$row['id'];
                        }
                    }
                    
                    if (!empty($medBookIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($medBookIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $medBookIds));
                        }
                        logParser("DEBUG administrative: Added " . count($medBookIds) . " medical book services");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG administrative: ERROR - " . $e->getMessage());
                }
                }
            }
            
            // Справка для визы
            if (preg_match('/(справк.*виз|виз.*справк|справк.*загран|загран.*справк)/ui', $question)) {
                logParser("DEBUG administrative: Detected visa certificate query");
                
                try {
                    $visaIds = [];
                    $stmtVisa = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND (LOWER(pf.field_value) LIKE '%справк%виз%' 
                             OR LOWER(pf.field_value) LIKE '%виз%справк%'
                             OR LOWER(pf.field_value) LIKE '%справк%загран%'
                             OR LOWER(pf.field_value) LIKE '%медицинск%справк%')
                        AND pi.is_duplicate = 0
                        LIMIT 5
                    ");
                    $stmtVisa->execute([$widget['id']]);
                    
                    while ($row = $stmtVisa->fetch()) {
                        $visaIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($visaIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($visaIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $visaIds));
                        }
                        logParser("DEBUG administrative: Added " . count($visaIds) . " visa certificate services");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG administrative: ERROR - " . $e->getMessage());
                }
            }
            
            // Справка на оружие
            if (preg_match('/(справк.*оружи|оружи.*справк|справк.*оруже|оруже.*справк)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG administrative: Detected weapon certificate query");
                
                // Fallback для услуг
                if ($section_name === 'services') {
                try {
                    $weaponIds = [];
                    $stmtWeapon = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND (LOWER(pf.field_value) LIKE '%справк%оружи%' 
                             OR LOWER(pf.field_value) LIKE '%оружи%справк%'
                             OR LOWER(pf.field_value) LIKE '%освидетельствован%оружи%'
                             OR LOWER(pf.field_value) LIKE '%медицинск%освидетельствован%')
                        AND pi.is_duplicate = 0
                        LIMIT 5
                    ");
                    $stmtWeapon->execute([$widget['id']]);
                    
                    while ($row = $stmtWeapon->fetch()) {
                        $weaponIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($weaponIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($weaponIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $weaponIds));
                        }
                        logParser("DEBUG administrative: Added " . count($weaponIds) . " weapon certificate services");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG administrative: ERROR - " . $e->getMessage());
                }
                }
            }
            
            // Больничный лист
            if (preg_match('/(больничн.*лист|лист.*нетрудоспособност|больничн|нетрудоспособност)/ui', $question) && empty($validatedDataIds[$section_name])) {
                logParser("DEBUG administrative: Detected sick leave query");
                
                // Fallback для услуг
                if ($section_name === 'services') {
                try {
                    $sickLeaveIds = [];
                    // Ищем консультации терапевта (он выдает больничные)
                    $stmtSickLeave = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND LOWER(pf.field_value) LIKE '%терапевт%'
                        AND LOWER(pf.field_value) LIKE '%консультация%'
                        AND pi.is_duplicate = 0
                        ORDER BY CASE 
                            WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 1
                            WHEN LOWER(pf.field_value) LIKE '%повторная%' THEN 2
                            ELSE 3
                        END
                        LIMIT 3
                    ");
                    $stmtSickLeave->execute([$widget['id']]);
                    
                    while ($row = $stmtSickLeave->fetch()) {
                        $sickLeaveIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($sickLeaveIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($sickLeaveIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $sickLeaveIds));
                        }
                        logParser("DEBUG administrative: Added " . count($sickLeaveIds) . " sick leave services");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG administrative: ERROR - " . $e->getMessage());
                }
                }
            }
            
            // Выписка из медицинской карты
            if (preg_match('/(выписк.*медицинск|выписк.*карт|выписк.*истори|медицинск.*выписк)/ui', $question)) {
                logParser("DEBUG administrative: Detected medical record extract query");
                
                // Fallback для услуг
                if ($section_name === 'services' && empty($validatedDataIds[$section_name])) {
                try {
                    $extractIds = [];
                    $stmtExtract = $db->prepare("
                        SELECT DISTINCT pi.id
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND (LOWER(pf.field_value) LIKE '%выписк%' 
                             OR LOWER(pf.field_value) LIKE '%копия%карт%'
                             OR LOWER(pf.field_value) LIKE '%медицинск%документ%')
                        AND pi.is_duplicate = 0
                        LIMIT 5
                    ");
                    $stmtExtract->execute([$widget['id']]);
                    
                    while ($row = $stmtExtract->fetch()) {
                        $extractIds[] = (int)$row['id'];
                    }
                    
                    if (!empty($extractIds)) {
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = array_unique($extractIds);
                        } else {
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $extractIds));
                        }
                        logParser("DEBUG administrative: Added " . count($extractIds) . " medical record extract services");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG administrative: ERROR - " . $e->getMessage());
                }
                }
            }
            
            // Пропускаем, если уже нашли услуги (например, для детских инфекций)
            // Для "Затрудненное дыхание" всегда проверяем fallback, даже если AI что-то нашел
            $isBreathingQueryServices = preg_match('/(затруднен.*дыхан|дыхан.*затруднен|тяжело.*дышать|дышать.*тяжело)/ui', $question);
            $shouldUseFallbackServices = empty($validatedDataIds[$section_name]) || ($isBreathingQueryServices && count($validatedDataIds[$section_name]) < 2);
            
            // СПЕЦИАЛЬНЫЙ FALLBACK для "Затрудненное дыхание" - добавляем диагностические услуги
            if ($isBreathingQueryServices && $section_name === 'services' && (empty($validatedDataIds[$section_name]) || count($validatedDataIds[$section_name]) < 2)) {
                logParser("DEBUG breathing fallback: Adding diagnostic services for breathing problems");
                
                try {
                    $breathingServices = [];
                    
                    // Ищем диагностические услуги для дыхательных проблем
                    $diagnosticKeywords = [
                        ['спирометр', 'спирограф'],
                        ['рентген.*груд', 'флюорограф', 'рентген.*легк'],
                        ['экг', 'электрокардиограмм'],
                        ['общий.*анализ.*кров', 'оак']
                    ];
                    
                    foreach ($diagnosticKeywords as $keywords) {
                        foreach ($keywords as $keyword) {
                            $stmtBreathing = $db->prepare("
                                SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                                FROM parsed_items pi
                                JOIN parsed_fields pf ON pi.id = pf.item_id
                                WHERE pi.widget_id = ? 
                                AND pi.section_name = 'services'
                                AND LOWER(pf.field_value) LIKE ?
                                AND pi.is_duplicate = 0
                                LIMIT 1
                            ");
                            $stmtBreathing->execute([$widget['id'], '%' . $keyword . '%']);
                            
                            while ($row = $stmtBreathing->fetch()) {
                                $itemId = (int)$row['id'];
                                if (!isset($breathingServices[$itemId])) {
                                    $breathingServices[$itemId] = ['id' => $itemId];
                                }
                                $breathingServices[$itemId][$row['field_name']] = $row['field_value'];
                            }
                            
                            // Если нашли услугу, переходим к следующей категории
                            if (!empty($breathingServices)) {
                                break;
                            }
                        }
                    }
                    
                    if (!empty($breathingServices)) {
                        $breathingIds = array_keys($breathingServices);
                        if (empty($validatedDataIds[$section_name])) {
                            $validatedDataIds[$section_name] = $breathingIds;
                        } else {
                            // Добавляем к существующим, избегая дублей
                            $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $breathingIds));
                        }
                        logParser("DEBUG breathing fallback: Added " . count($breathingIds) . " diagnostic services for breathing problems");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG breathing fallback: ERROR - " . $e->getMessage());
                }
            }
            
            if ($shouldUseFallbackServices) {
            
            // Транспортная комиссия (ГИБДД, водительские права)
            if (preg_match('/(транспортн.*комисс|гибдд|водител|прав.*категор|справка.*водител)/ui', $question)) {
                logParser("DEBUG administrative: Detected transport commission query");
                
                try {
                    // Ищем услуги транспортной комиссии
                    $stmtAdmin = $db->prepare("
                        SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND LOWER(pf.field_value) LIKE '%транспортная комиссия%'
                        AND pi.is_duplicate = 0
                        LIMIT 5
                    ");
                    $stmtAdmin->execute([$widget['id']]);
                    
                    $adminItems = [];
                    while ($row = $stmtAdmin->fetch()) {
                        $itemId = (int)$row['id'];
                        if (!isset($adminItems[$itemId])) {
                            $adminItems[$itemId] = ['id' => $itemId];
                        }
                        $adminItems[$itemId][$row['field_name']] = $row['field_value'];
                    }
                    
                    if (!empty($adminItems)) {
                        $validatedDataIds[$section_name] = array_column($adminItems, 'id');
                        logParser("DEBUG administrative: Found " . count($validatedDataIds[$section_name]) . " transport commission services");
                        
                        // Также ищем специалистов для комиссии
                        if (isset($validatedDataIds['specialists'])) {
                            $stmtSpecs = $db->prepare("
                                SELECT DISTINCT pi.id
                                FROM parsed_items pi
                                JOIN parsed_fields pf ON pi.id = pf.item_id
                                WHERE pi.widget_id = ? 
                                AND pi.section_name = 'specialists'
                                AND (LOWER(pf.field_value) LIKE '%терапевт%' OR LOWER(pf.field_value) LIKE '%офтальмолог%' OR LOWER(pf.field_value) LIKE '%невролог%')
                                AND pi.is_duplicate = 0
                                LIMIT 3
                            ");
                            $stmtSpecs->execute([$widget['id']]);
                            $specIds = $stmtSpecs->fetchAll(PDO::FETCH_COLUMN);
                            $validatedDataIds['specialists'] = array_map('intval', $specIds);
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG administrative: ERROR - " . $e->getMessage());
                }
            }
            
            // Справки (учеба, бассейн, санаторий)
            if (preg_match('/(справк.*086|справк.*учеб|справк.*бассейн|справк.*санатор|справк.*лагер|справк.*070)/ui', $question)) {
                logParser("DEBUG administrative: Detected certificate query");
                
                try {
                    $stmtAdmin = $db->prepare("
                        SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND (LOWER(pf.field_value) LIKE '%справка%' AND (LOWER(pf.field_value) LIKE '%086%' OR LOWER(pf.field_value) LIKE '%учеб%' OR LOWER(pf.field_value) LIKE '%бассейн%' OR LOWER(pf.field_value) LIKE '%санатор%' OR LOWER(pf.field_value) LIKE '%070%'))
                        AND pi.is_duplicate = 0
                        LIMIT 5
                    ");
                    $stmtAdmin->execute([$widget['id']]);
                    
                    $adminItems = [];
                    while ($row = $stmtAdmin->fetch()) {
                        $itemId = (int)$row['id'];
                        if (!isset($adminItems[$itemId])) {
                            $adminItems[$itemId] = ['id' => $itemId];
                        }
                        $adminItems[$itemId][$row['field_name']] = $row['field_value'];
                    }
                    
                    if (!empty($adminItems)) {
                        $validatedDataIds[$section_name] = array_column($adminItems, 'id');
                        logParser("DEBUG administrative: Found " . count($validatedDataIds[$section_name]) . " certificate services");
                    }
                } catch (Exception $e) {
                    logParser("DEBUG administrative: ERROR - " . $e->getMessage());
                }
            }
            
            // Для справок на оружие, больничных, медкнижек - просто информируем что услуга отсутствует
            if (preg_match('/(справк.*оружи|больничн|медицинск.*книжк)/ui', $question)) {
                logParser("DEBUG administrative: Query about weapon certificate, sick leave, or medical book - service not available");
                // Возвращаем пустой результат, но с информативным сообщением
                $response = [
                    'text' => 'К сожалению, эта услуга не предоставляется нашей клиникой. Для получения данной справки обратитесь в государственное медицинское учреждение.',
                    'data' => [
                        'specialists' => [],
                        'services' => [],
                        'articles' => [],
                        'specializations' => []
                    ],
                    'ready' => true,
                    'section_name' => $section_name,
                    'response_time_ms' => round((microtime(true) - $startTime) * 1000)
                ];
                return $response;
            }
            
            // Для выписки из медкарты - административный запрос
            if (preg_match('/(выписк.*медицинск.*карт|копи.*карт|медицинск.*документ)/ui', $question)) {
                logParser("DEBUG administrative: Query about medical records - administrative request");
                $response = [
                    'text' => 'Для получения выписки из медицинской карты обратитесь в регистратуру клиники с паспортом. Выписка готовится в течение 1-3 рабочих дней.',
                    'data' => [
                        'specialists' => [],
                        'services' => [],
                        'articles' => [],
                        'specializations' => []
                    ],
                    'ready' => true,
                    'section_name' => $section_name,
                    'response_time_ms' => round((microtime(true) - $startTime) * 1000)
                ];
                return $response;
            }
            
            // ============================================================================
            // ДИАГНОСТИЧЕСКИЕ ИССЛЕДОВАНИЯ - специальная обработка
            // ============================================================================
            // МРТ, КТ, УЗИ, Рентген и другие диагностические процедуры
            if (preg_match('/(мрт|кт|узи|рентген|флюорография|денситометр|эндоскоп|колоноскоп|фгдс|экг|холтер|спирометр)/ui', $question)) {
                logParser("DEBUG diagnostic: Detected diagnostic procedure query");
                
                try {
                    // Извлекаем тип исследования и область
                    $diagnosticType = '';
                    $bodyPart = '';
                    
                    if (preg_match('/мрт/ui', $question)) {
                        $diagnosticType = 'мрт';
                    } else if (preg_match('/кт/ui', $question)) {
                        $diagnosticType = 'кт';
                    } else if (preg_match('/узи/ui', $question)) {
                        $diagnosticType = 'узи';
                    } else if (preg_match('/рентген/ui', $question)) {
                        $diagnosticType = 'рентген';
                    } else if (preg_match('/флюорография/ui', $question)) {
                        $diagnosticType = 'флюорография';
                    }
                    
                    // Определяем область исследования
                    if (preg_match('/(сустав|колен|локт|плеч|тазобедр)/ui', $question)) {
                        $bodyPart = 'сустав';
                    } else if (preg_match('/(позвоночник|спин|поясниц|шейн|грудн.*отдел)/ui', $question)) {
                        $bodyPart = 'позвоночник';
                    } else if (preg_match('/(голов|мозг|череп)/ui', $question)) {
                        $bodyPart = 'голов';
                    } else if (preg_match('/(брюшн.*полост|живот|печень|почк|желчн.*пузыр)/ui', $question)) {
                        $bodyPart = 'брюшн';
                    } else if (preg_match('/(грудн.*клетк|легк)/ui', $question)) {
                        $bodyPart = 'грудн';
                    }
                    
                    // Ищем диагностическую услугу
                    $searchPattern = '%' . $diagnosticType . '%';
                    if (!empty($bodyPart)) {
                        $searchPattern = '%' . $diagnosticType . '%' . $bodyPart . '%';
                    }
                    
                    $stmtDiag = $db->prepare("
                        SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND LOWER(pf.field_value) LIKE ?
                        AND pi.is_duplicate = 0
                        ORDER BY CASE 
                            WHEN LOWER(pf.field_value) LIKE ? THEN 1
                            ELSE 2
                        END
                        LIMIT 3
                    ");
                    $stmtDiag->execute([$widget['id'], $searchPattern, $searchPattern]);
                    
                    $diagItems = [];
                    while ($row = $stmtDiag->fetch()) {
                        $itemId = (int)$row['id'];
                        if (!isset($diagItems[$itemId])) {
                            $diagItems[$itemId] = ['id' => $itemId];
                        }
                        $diagItems[$itemId][$row['field_name']] = $row['field_value'];
                    }
                    
                    if (!empty($diagItems)) {
                        $validatedDataIds[$section_name] = array_column($diagItems, 'id');
                        logParser("DEBUG diagnostic: Found " . count($validatedDataIds[$section_name]) . " diagnostic services");
                        
                        // Также ищем релевантных специалистов для назначения исследования
                        $specialtySearch = [];
                        if ($bodyPart === 'сустав' || $bodyPart === 'позвоночник') {
                            $specialtySearch = ['травматолог', 'ортопед', 'невролог'];
                        } else if ($bodyPart === 'голов') {
                            $specialtySearch = ['невролог'];
                        } else if ($bodyPart === 'брюшн') {
                            $specialtySearch = ['гастроэнтеролог', 'терапевт'];
                        } else if ($bodyPart === 'грудн') {
                            $specialtySearch = ['терапевт', 'кардиолог'];
                        } else {
                            $specialtySearch = ['терапевт'];
                        }
                        
                        // Добавляем консультации специалистов
                        foreach ($specialtySearch as $specialty) {
                            $stmtSpec = $db->prepare("
                                SELECT DISTINCT pi.id
                                FROM parsed_items pi
                                JOIN parsed_fields pf ON pi.id = pf.item_id
                                WHERE pi.widget_id = ? 
                                AND pi.section_name = 'services'
                                AND pf.field_name = 'name'
                                AND LOWER(pf.field_value) LIKE ?
                                AND LOWER(pf.field_value) LIKE '%консультация%'
                                AND pi.is_duplicate = 0
                                LIMIT 1
                            ");
                            $stmtSpec->execute([$widget['id'], '%' . $specialty . '%']);
                            
                            while ($row = $stmtSpec->fetch()) {
                                $validatedDataIds[$section_name][] = (int)$row['id'];
                            }
                        }
                        
                        logParser("DEBUG diagnostic: Total with consultations: " . count($validatedDataIds[$section_name]));
                    } else {
                        logParser("DEBUG diagnostic: No matching diagnostic services found");
                        
                        // Если диагностическое исследование не найдено, все равно добавляем консультацию специалиста
                        // который может назначить это исследование в другом месте
                        $specialtySearch = [];
                        if ($bodyPart === 'сустав' || $bodyPart === 'позвоночник') {
                            $specialtySearch = ['травматолог', 'ортопед'];
                        } else if ($bodyPart === 'голов') {
                            $specialtySearch = ['невролог'];
                        } else if ($bodyPart === 'брюшн') {
                            $specialtySearch = ['гастроэнтеролог', 'терапевт'];
                        } else if ($bodyPart === 'грудн') {
                            $specialtySearch = ['терапевт', 'кардиолог'];
                        } else {
                            $specialtySearch = ['терапевт'];
                        }
                        
                        $foundConsultations = [];
                        foreach ($specialtySearch as $specialty) {
                            $stmtSpec = $db->prepare("
                                SELECT DISTINCT pi.id
                                FROM parsed_items pi
                                JOIN parsed_fields pf ON pi.id = pf.item_id
                                WHERE pi.widget_id = ? 
                                AND pi.section_name = 'services'
                                AND pf.field_name = 'name'
                                AND LOWER(pf.field_value) LIKE ?
                                AND LOWER(pf.field_value) LIKE '%консультация%'
                                AND pi.is_duplicate = 0
                                LIMIT 1
                            ");
                            $stmtSpec->execute([$widget['id'], '%' . $specialty . '%']);
                            
                            while ($row = $stmtSpec->fetch()) {
                                $foundConsultations[] = (int)$row['id'];
                            }
                        }
                        
                        if (!empty($foundConsultations)) {
                            $validatedDataIds[$section_name] = $foundConsultations;
                            logParser("DEBUG diagnostic: Added specialist consultations as alternative");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG diagnostic: ERROR - " . $e->getMessage());
                }
            }
            
            // ============================================================================
            // МЕДИЦИНСКИЕ ВОПРОСЫ - обычная обработка
            // ============================================================================
            
            // УПРОЩЕННАЯ ЛОГИКА: Ищем консультации по ключевым словам из вопроса
            // Определяем какие специальности релевантны
            $relevantSpecialties = [];
            
            // Неврология
            if (preg_match('/(бессонн|невролог|голов|головокруж|мигрен|головн.*бол|нервн|тремор|онемен|защемл|спин.*бол|боль.*спин)/ui', $question)) {
                $relevantSpecialties[] = 'невролог';
            }
            
            // Терапия
            if (preg_match('/(температур|просту|грипп|орви|орз|кашель|насморк|слабость|терапевт|аппетит)/ui', $question)) {
                $relevantSpecialties[] = 'терапевт';
            }
            
            // Травматология и ортопедия
            if (preg_match('/(травм|ортопед|перелом|вывих|растяжен|ушиб|спин.*бол|боль.*спин|сустав|колен|плеч)/ui', $question)) {
                $relevantSpecialties[] = 'травматолог';
            }
            
            // Кардиология
            if (preg_match('/(сердц|давлен|гипертон|аритми|тахикарди|кардиолог|одышк|груд.*бол|боль.*груд)/ui', $question)) {
                $relevantSpecialties[] = 'кардиолог';
            }
            
            // Гастроэнтерология
            if (preg_match('/(живот|желуд|гастрит|язв|тошнот|рвот|изжог|пищеварен|гастроэнтеролог|аппетит)/ui', $question)) {
                $relevantSpecialties[] = 'гастроэнтеролог';
            }
            
            // Дерматология
            if (preg_match('/(кож|сыпь|зуд|прыщ|акне|псориаз|розацеа|дерматолог|дерматит)/ui', $question)) {
                $relevantSpecialties[] = 'дерматолог';
            }
            
            // Урология
            if (preg_match('/(мочеиспускан|урина|уролог|моч[еа]|почк|мочевой пузырь|цистит|пиелонефрит|простат)/ui', $question)) {
                $relevantSpecialties[] = 'уролог';
            }
            
            // ЛОР (Оториноларингология)
            if (preg_match('/(нос|горл|ухо|лор|оториноларинголог|ангин|фарингит|тонзиллит|ринит|синусит|гайморит|отит|кровотечение.*нос|носов.*кровотечен)/ui', $question)) {
                $relevantSpecialties[] = 'лор';
            }
            
            // Аллергология
            if (preg_match('/(аллерг|аллерголог|аллергическ|крапивниц)/ui', $question)) {
                $relevantSpecialties[] = 'аллерголог';
            }
            
            // Офтальмология (РАСШИРЕНО: очки, подбор очков)
            if (preg_match('/(глаз|зрен|офтальмолог|конъюнктивит|катаракт|глауком|очк|подбор очков|линз)/ui', $question)) {
                $relevantSpecialties[] = 'офтальмолог';
            }
            
            // Травматология и ортопедия
            if (preg_match('/(травматолог|ортопед|перелом|вывих|ушиб|растяжение|сустав.*бол|боль.*сустав|колен.*бол|боль.*колен|спин.*бол|боль.*спин)/ui', $question)) {
                $relevantSpecialties[] = 'травматолог';
            }
            
            // Эндокринология (РАСШИРЕНО: потливость, холестерин)
            if (preg_match('/(эндокринолог|щитовидн|диабет|гормон|сахар|потливость|потеет|пот|гипергидроз|холестерин)/ui', $question)) {
                $relevantSpecialties[] = 'эндокринолог';
            }
            
            // Хирургия
            if (preg_match('/(хирург|аппендицит|грыж|операц|хирургическ)/ui', $question)) {
                $relevantSpecialties[] = 'хирург';
            }
            
            // Ревматология
            if (preg_match('/(ревматолог|артрит|подагр|ревматизм|волчанк|скованность.*суставах|полиартрит|боль.*всех.*суставах)/ui', $question)) {
                $relevantSpecialties[] = 'ревматолог';
            }
            
            // Педиатрия
            if (preg_match('/(педиатр|ребенка|детей|новорожденн|корь|краснух|скарлатин|коклюш|ветрянк|диатез)/ui', $question)) {
                $relevantSpecialties[] = 'педиатр';
                $relevantSpecialties[] = 'терапевт'; // Терапевт также может помочь
            }
            
            // Пульмонология
            if (preg_match('/(пульмонолог|одышк|бронхит|пневмон|хобл|астма|легких|удушь|задыхаю|приступ.*удушь)/ui', $question)) {
                $relevantSpecialties[] = 'пульмонолог';
                $relevantSpecialties[] = 'терапевт';
                $relevantSpecialties[] = 'кардиолог';
            }
            
            // Затрудненное дыхание (улучшенный fallback)
            if (preg_match('/(затруднен.*дыхан|дыхан.*затруднен|тяжело.*дышать|дышать.*тяжело)/ui', $question)) {
                $relevantSpecialties[] = 'пульмонолог';
                $relevantSpecialties[] = 'терапевт';
                $relevantSpecialties[] = 'кардиолог';
            }
            
            // Инфекционист + Гематолог (для лимфоузлов)
            if (preg_match('/(лимфоузл|лимфатич|увеличен.*узл)/ui', $question)) {
                // Инфекциониста нет, используем терапевта
                $relevantSpecialties[] = 'терапевт';
                $relevantSpecialties[] = 'онколог';
            }
            
            // Флебология (для отеков ног)
            if (preg_match('/(отек.*ног|отеч.*ног|вен.*недостаточн|варикоз)/ui', $question)) {
                // Флеболога нет, используем терапевта + кардиолога
                $relevantSpecialties[] = 'терапевт';
                $relevantSpecialties[] = 'кардиолог';
                $relevantSpecialties[] = 'нефролог';
            }
            
            // Гастроэнтерология (метеоризм, горечь, сухость во рту)
            if (preg_match('/(метеоризм|газообразован|вздут.*живот|горечь.*рту|сухость.*рту)/ui', $question)) {
                $relevantSpecialties[] = 'гастроэнтеролог';
            }
            
            // Пульмонология
            if (preg_match('/(пульмонолог|одышк|бронхит|пневмон|хобл|астма|легких|удушь|дых.*затруднен)/ui', $question)) {
                $relevantSpecialties[] = 'пульмонолог';
            }
            
            // Ортопедия
            if (preg_match('/(ортопед|плоскостоп|косолап|сколиоз|нарушен.*осанк|искривлен.*позвоночник)/ui', $question)) {
                $relevantSpecialties[] = 'ортопед';
                $relevantSpecialties[] = 'травматолог';
            }
            
            // Сомнология (нарушения сна)
            if (preg_match('/(сомнолог|апноэ|храп|нарушен.*сн)/ui', $question)) {
                $relevantSpecialties[] = 'невролог';
            }
            
            // Проктология
            if (preg_match('/(проктолог|геморрой|трещин.*анальн)/ui', $question)) {
                $relevantSpecialties[] = 'проктолог';
                $relevantSpecialties[] = 'хирург';
            }
            
            // Онкология
            if (preg_match('/(онколог|опухол|новообразован)/ui', $question)) {
                $relevantSpecialties[] = 'онколог';
            }
            
            // Гинекология
            if (preg_match('/(гинеколог|месячн|менструац|эндометриоз|миома|яичник)/ui', $question)) {
                $relevantSpecialties[] = 'гинеколог';
            }
            
            // Маммология
            if (preg_match('/(маммолог|грудь.*железа|уплотнен.*груд|мастопатия)/ui', $question)) {
                $relevantSpecialties[] = 'маммолог';
            }
            
            // Нефрология
            if (preg_match('/(нефролог|пиелонефрит|гломерулонефрит|почечн.*недостаточн)/ui', $question)) {
                $relevantSpecialties[] = 'нефролог';
                $relevantSpecialties[] = 'уролог';
            }
            
            // Гематология
            if (preg_match('/(гематолог|анемия|лейкоз|тромбоцит)/ui', $question)) {
                $relevantSpecialties[] = 'гематолог';
            }
            
            // Психотерапия/Психиатрия
            if (preg_match('/(психиатр|психотерапевт|депресс|тревог|паническ|фобия)/ui', $question)) {
                $relevantSpecialties[] = 'психотерапевт';
                $relevantSpecialties[] = 'психиатр';
            }
            
            // Косметология
            if (preg_match('/(косметолог|пигментн.*пятн|морщин|уход.*кож)/ui', $question)) {
                $relevantSpecialties[] = 'косметолог';
            }
            
            // Трихология
            if (preg_match('/(трихолог|выпадени.*волос|алопеция|перхоть)/ui', $question)) {
                $relevantSpecialties[] = 'трихолог';
                $relevantSpecialties[] = 'дерматолог';
            }
            
            // Остеопороз
            if (preg_match('/(остеопороз|костн.*ткан|хрупкость.*кост)/ui', $question)) {
                $relevantSpecialties[] = 'эндокринолог';
                $relevantSpecialties[] = 'травматолог';
            }
            
            // Люмбаго, острая боль в спине
            if (preg_match('/(люмбаго|острая.*боль.*спин|прострел|боль.*пояснице)/ui', $question)) {
                $relevantSpecialties[] = 'невролог';
                $relevantSpecialties[] = 'травматолог';
            }
            
            // Утренняя скованность суставов
            if (preg_match('/(утренн.*скованн|скованн.*сустав)/ui', $question)) {
                $relevantSpecialties[] = 'ревматолог';
                $relevantSpecialties[] = 'терапевт';
            }
            
            // Потеря обоняния
            if (preg_match('/(потер.*обоня|потер.*вкус|аносмия)/ui', $question)) {
                $relevantSpecialties[] = 'лор';
                $relevantSpecialties[] = 'невролог';
            }
            
            // Отек Квинке
            if (preg_match('/(отек.*квинке|ангионевротическ.*отек)/ui', $question)) {
                $relevantSpecialties[] = 'аллерголог';
                $relevantSpecialties[] = 'терапевт';
            }
            
            // Диагностические услуги (МРТ, КТ, УЗИ)
            if (preg_match('/(мрт|кт|компьютерн.*томограф|магнитн.*резонан)/ui', $question)) {
                $relevantSpecialties[] = 'терапевт';
            }
            
            // Диспансеризация
            if (preg_match('/(диспансеризац|медосмотр|профосмотр|чекап)/ui', $question)) {
                $relevantSpecialties[] = 'терапевт';
            }
            
            // Если нашли релевантные специальности, ищем их консультации
            if (!empty($relevantSpecialties)) {
                $consultationIds = [];
                
                // Маппинг сокращенных названий на полные (как в базе данных)
                $specialtyMapping = [
                    'лор' => 'оториноларинголог',
                    'невролог' => 'невролог',
                    'терапевт' => 'терапевт',
                    'кардиолог' => 'кардиолог',
                    'гастроэнтеролог' => 'гастроэнтеролог',
                    'дерматолог' => 'дерматолог',
                    'уролог' => 'уролог',
                    'аллерголог' => 'аллерголог',
                    'офтальмолог' => 'офтальмолог',
                    'травматолог' => 'травматолог',
                    'хирург' => 'хирург',
                    'эндокринолог' => 'эндокринолог',
                    'ревматолог' => 'ревматолог',
                    'педиатр' => 'педиатр',
                    'пульмонолог' => 'пульмонолог',
                    'ортопед' => 'ортопед',
                    'проктолог' => 'проктолог',
                    'онколог' => 'онколог',
                    'гинеколог' => 'гинеколог',
                    'маммолог' => 'маммолог',
                    'нефролог' => 'нефролог',
                    'гематолог' => 'гематолог',
                    'психотерапевт' => 'психотерапевт',
                    'психиатр' => 'психиатр',
                    'косметолог' => 'косметолог',
                    'трихолог' => 'трихолог'
                ];
                
                foreach ($relevantSpecialties as $specialty) {
                    // Преобразуем сокращение в полное название
                    $searchSpecialty = $specialtyMapping[$specialty] ?? $specialty;
                    
                    try {
                        $stmtConsult = $db->prepare("
                            SELECT DISTINCT pi.id,
                                CASE 
                                    WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 1
                                    WHEN LOWER(pf.field_value) LIKE '%повторная%' THEN 2
                                    ELSE 3
                                END as sort_order
                            FROM parsed_items pi
                            JOIN parsed_fields pf ON pi.id = pf.item_id
                            WHERE pi.widget_id = ? 
                            AND pi.section_name = 'services'
                            AND pf.field_name = 'name'
                            AND LOWER(pf.field_value) LIKE ?
                            AND LOWER(pf.field_value) LIKE '%консультация%'
                            AND pi.is_duplicate = 0
                            ORDER BY sort_order
                            LIMIT 2
                        ");
                        $stmtConsult->execute([$widget['id'], '%' . $searchSpecialty . '%']);
                        
                        while ($row = $stmtConsult->fetch()) {
                            $consultationIds[] = (int)$row['id'];
                        }
                    } catch (Exception $e) {
                        logParser("DEBUG services fallback: ERROR for $specialty: " . $e->getMessage());
                    }
                }
                
                if (!empty($consultationIds) && $shouldUseFallbackServices) {
                    // Добавляем к уже найденным услугам (если есть)
                    if (empty($validatedDataIds[$section_name])) {
                        $validatedDataIds[$section_name] = array_unique($consultationIds);
                    } else {
                        $validatedDataIds[$section_name] = array_unique(array_merge($validatedDataIds[$section_name], $consultationIds));
                    }
                    logParser("DEBUG services: Added " . count($validatedDataIds[$section_name]) . " specialist consultations (" . implode(', ', $relevantSpecialties) . ") for query '{$question}'");
                }
            }
            
            } // Закрытие блока: if ($shouldUseFallbackServices)
        }
    }
    
    // Загружаем полные данные по валидным ID
    logParser("DEBUG before fullData load: section_name='{$section_name}', validatedDataIds count: " . count($validatedDataIds[$section_name] ?? []) . ", IDs: " . json_encode($validatedDataIds[$section_name] ?? []));
    $fullData = [];
    if (!empty($validatedDataIds[$section_name])) {
        $placeholders = str_repeat('?,', count($validatedDataIds[$section_name]) - 1) . '?';
        $stmt = $db->prepare("
            SELECT pi.id, pf.field_name, pf.field_value
            FROM parsed_items pi
            JOIN parsed_fields pf ON pi.id = pf.item_id
            WHERE pi.widget_id = ? AND pi.section_name = ?
            AND pi.id IN ($placeholders)
            AND pi.is_duplicate = 0
        ");
        $params = array_merge([$widget['id'], $section_name], $validatedDataIds[$section_name]);
        $stmt->execute($params);
        
        $sectionItems = [];
        while ($row = $stmt->fetch()) {
            if (!isset($sectionItems[$row['id']])) {
                $sectionItems[$row['id']] = ['id' => (int)$row['id']];
            }
            $sectionItems[$row['id']][$row['field_name']] = $row['field_value'];
        }
        
        // Фильтрация нерелевантных элементов на основе вопроса
        $questionLower = mb_strtolower($question);
        
        // Определяем тип запроса
        $isThroatNoseQuery = (strpos($questionLower, 'горл') !== false || strpos($questionLower, 'нос') !== false ||
                             strpos($questionLower, 'ангин') !== false || strpos($questionLower, 'фарингит') !== false ||
                             strpos($questionLower, 'тонзиллит') !== false || strpos($questionLower, 'кашель') !== false ||
                             strpos($questionLower, 'насморк') !== false || strpos($questionLower, 'ринит') !== false ||
                             strpos($questionLower, 'синусит') !== false);
        
        $isTemperatureQuery = (strpos($questionLower, 'температур') !== false || strpos($questionLower, 'лихорадк') !== false);
        
        $isUrinationQuery = (strpos($questionLower, 'мочеиспускание') !== false || strpos($questionLower, 'мочеиспуска') !== false ||
                            strpos($questionLower, 'недержание мочи') !== false || strpos($questionLower, 'проблемы с мочеиспусканием') !== false);
        
        // Фильтруем услуги
        if ($section_name === 'services') {
            $filteredItems = [];
            $filteredIds = [];
            foreach ($sectionItems as $id => $item) {
                $name = mb_strtolower($item['name'] ?? $item['service_name'] ?? $item['title'] ?? '');
                $shouldExclude = false;
                
                // УДАЛЕНЫ проблемные фильтры (бассейн, транспортная комиссия, температура)
                // GPU LLM с embeddings делает правильный выбор услуг
                // Оставляем только базовую фильтрацию для явно нерелевантных случаев
                
                // Для запросов про мочеиспускание исключаем нерелевантные услуги
                if ($isUrinationQuery && !$shouldExclude) {
                    if (strpos($name, 'микоплазм') !== false ||
                        strpos($name, 'уреаплазм') !== false ||
                        strpos($name, 'коронавирус') !== false ||
                        strpos($name, 'covid') !== false ||
                        strpos($name, 'кондилом') !== false ||
                        strpos($name, 'аллергочип') !== false ||
                        strpos($name, 'циркулирующие иммунные комплексы') !== false) {
                        $shouldExclude = true;
                    }
                }
                
                // Для розацеа исключаем услуги ДЭК и удаления новообразований
                $isRosaceaQuery = (strpos($questionLower, 'розацеа') !== false);
                if ($isRosaceaQuery && !$shouldExclude) {
                    if (strpos($name, 'дэк образований') !== false ||
                        strpos($name, 'удаление новообразований') !== false ||
                        strpos($name, 'удаление кожного новообразования') !== false) {
                        $shouldExclude = true;
                    }
                }
                
                // Для псориаза исключаем нерелевантные услуги (аналогично розацеа)
                $isPsoriasisQuery = (strpos($questionLower, 'псориаз') !== false);
                if ($isPsoriasisQuery && !$shouldExclude) {
                    if (strpos($name, 'дэк образований') !== false ||
                        strpos($name, 'удаление новообразований') !== false ||
                        strpos($name, 'удаление кожного новообразования') !== false ||
                        strpos($name, 'шейки матки') !== false ||
                        strpos($name, 'гинеколог') !== false) {
                        $shouldExclude = true;
                    }
                }
                
                // Для зуд кожи исключаем услуги удаления кондилом и новообразований
                $isItchingQuery = (strpos($questionLower, 'зуд кожи') !== false || 
                                  (strpos($questionLower, 'зуд') !== false && strpos($questionLower, 'кожа') !== false));
                if ($isItchingQuery && !$shouldExclude) {
                    if (strpos($name, 'удаление кондилом') !== false ||
                        strpos($name, 'удаление новообразований') !== false ||
                        strpos($name, 'удаление кожного новообразования') !== false) {
                        $shouldExclude = true;
                    }
                }
                
                // Для "высокое давление" исключаем услуги гастроэнтеролога и онколога
                $isHighPressureQuery = (strpos($questionLower, 'высокое давление') !== false ||
                                       strpos($questionLower, 'высокое артериальное давление') !== false ||
                                       strpos($questionLower, 'гипертония') !== false ||
                                       strpos($questionLower, 'гипертензия') !== false ||
                                       strpos($questionLower, 'повышенное давление') !== false);
                if ($isHighPressureQuery && !$shouldExclude) {
                    if (strpos($name, 'гастроэнтеролог') !== false ||
                        strpos($name, 'онколог') !== false) {
                        $shouldExclude = true;
                    }
                }
                
                // Для "защемление нерва" исключаем услуги про анальную трещину
                $isNervePinchQuery = (strpos($questionLower, 'защемлен') !== false && 
                                     (strpos($questionLower, 'нерв') !== false || strpos($questionLower, 'спин') !== false));
                if ($isNervePinchQuery && !$shouldExclude) {
                    if (strpos($name, 'анальн') !== false ||
                        strpos($name, 'трещин') !== false ||
                        strpos($name, 'проктолог') !== false) {
                        $shouldExclude = true;
                    }
                }
                
                // Для "межпозвоночная грыжа" исключаем услуги про грыжу брюшной стенки
                $isSpinalHerniaQuery = (strpos($questionLower, 'межпозвон') !== false && strpos($questionLower, 'грыж') !== false) ||
                                      (strpos($questionLower, 'грыж') !== false && strpos($questionLower, 'позвон') !== false);
                if ($isSpinalHerniaQuery && !$shouldExclude) {
                    if (strpos($name, 'брюшн') !== false ||
                        strpos($name, 'вентральн') !== false ||
                        (strpos($name, 'грыж') !== false && strpos($name, 'брюшн') !== false)) {
                        $shouldExclude = true;
                    }
                }
                
                if (!$shouldExclude) {
                    $filteredItems[$id] = $item;
                    $filteredIds[] = $id;
                }
            }
            $sectionItems = $filteredItems;
            // Обновляем validatedDataIds после фильтрации, сохраняя порядок от AI
            $originalValidatedCount = count($validatedDataIds[$section_name]);
            $validatedDataIds[$section_name] = array_values(array_intersect($validatedDataIds[$section_name], $filteredIds));
            $filteredValidatedCount = count($validatedDataIds[$section_name]);
            
            // Логируем для диагностики
            error_log("DEBUG services filter: Query '{$question}' - Original validated: {$originalValidatedCount}, After filter: {$filteredValidatedCount}, Filtered items available: " . count($filteredItems));
            
            // FALLBACK: Если после фильтрации все услуги были исключены, но есть отфильтрованные услуги,
            // используем их (это может произойти при очень строгой фильтрации, например для температуры)
            if (empty($validatedDataIds[$section_name]) && !empty($filteredItems)) {
                if ($originalValidatedCount > 0) {
                    // Логируем предупреждение
                    error_log("Warning: All AI-selected services were filtered out for query '{$question}'. Using filtered items as fallback. Original: {$originalValidatedCount}, Filtered: " . count($filteredItems));
                } else {
                    // AI не выбрал услуги, но есть отфильтрованные - используем их
                    error_log("Warning: AI did not select services, but filtered items available for query '{$question}'. Using filtered items. Count: " . count($filteredItems));
                }
                // Используем все отфильтрованные услуги (ограничиваем до 20)
                $validatedDataIds[$section_name] = array_slice(array_keys($filteredItems), 0, 20);
                error_log("DEBUG services fallback: Using " . count($validatedDataIds[$section_name]) . " filtered items for query '{$question}'");
            }
        }
        
        // Фильтруем статьи
        if ($section_name === 'articles') {
            $filteredItems = [];
            foreach ($sectionItems as $id => $item) {
                $title = mb_strtolower($item['title'] ?? $item['name'] ?? '');
                $shouldExclude = false;
                
                // УНИВЕРСАЛЬНАЯ ФИЛЬТРАЦИЯ: исключаем нерелевантные статьи для ВСЕХ запросов
                if (preg_match('/(программ.*лояльност|лояльност.*пациент)/ui', $title) || 
                    preg_match('/(критери.*доступност|доступност.*качеств)/ui', $title) ||
                    strpos($title, 'целевые критерии') !== false ||
                    (strpos($title, 'правил') !== false && strpos($title, 'подготовк') !== false && !preg_match('/(подготовк|подготов.*исследован)/ui', $questionLower)) ||
                    (strpos($title, 'беременност') !== false && strpos($title, 'секс') !== false && !preg_match('/(беременност|женск.*здоров|гинеколог)/ui', $questionLower)) ||
                    (strpos($title, 'беременност') !== false && strpos($title, 'развитие плода') !== false && !preg_match('/(беременност|женск.*здоров|гинеколог)/ui', $questionLower)) ||
                    (strpos($title, 'чесотк') !== false && strpos($title, 'диагностик') !== false && !preg_match('/(чесотк|кож.*заболеван)/ui', $questionLower)) ||
                    (strpos($title, 'диагностик') !== false && strpos($title, 'глазн') !== false && strpos($title, 'болезн') !== false && strpos($title, 'дет') !== false && !preg_match('/(глаз|офтальмолог|зрен)/ui', $questionLower)) ||
                    (strpos($title, 'диагностик') !== false && strpos($title, 'варикоцеле') !== false && !preg_match('/(варикоцеле|мошонк|яичк)/ui', $questionLower))) {
                    $shouldExclude = true;
                    logParser("DEBUG articles filter sectionItems: Excluding article '{$item['title']}' (ID {$id}) for query '{$question}' (loyalty program/accessibility criteria/general rules)");
                }
                
                // Для запросов про горло/нос исключаем нерелевантные статьи
                if ($isThroatNoseQuery && !$shouldExclude) {
                    if (strpos($title, 'запор') !== false ||
                        strpos($title, 'массаж головы') !== false ||
                        strpos($title, 'лактостаз') !== false ||
                        strpos($title, 'мастит') !== false ||
                        strpos($title, 'колики') !== false ||
                        strpos($title, 'косоглазие') !== false ||
                        strpos($title, 'головная боль') !== false ||
                        strpos($title, 'головной боли') !== false ||
                        strpos($title, 'магнитотерапи') !== false ||
                        strpos($title, 'аппарат магнитотерапии') !== false) {
                        $shouldExclude = true;
                    }
                }
                
                // Для псориаза исключаем статьи про боль в колене
                $isPsoriasisQuery = (strpos($questionLower, 'псориаз') !== false);
                if ($isPsoriasisQuery) {
                    if (strpos($title, 'боль в колене') !== false ||
                        strpos($title, 'боли в колене') !== false ||
                        strpos($title, 'коленном суставе') !== false) {
                        $shouldExclude = true;
                    }
                }
                
                // Для "боль в груди" исключаем статьи про другие виды боли
                $isChestPainQuery = (strpos($questionLower, 'боль в груди') !== false ||
                                    strpos($questionLower, 'боли в груди') !== false ||
                                    strpos($questionLower, 'болит грудь') !== false ||
                                    strpos($questionLower, 'болит в груди') !== false);
                if ($isChestPainQuery) {
                    if (strpos($title, 'боль в колене') !== false ||
                        strpos($title, 'боли в колене') !== false ||
                        strpos($title, 'боль в спине') !== false ||
                        strpos($title, 'боли в спине') !== false ||
                        strpos($title, 'боль в суставах') !== false ||
                        strpos($title, 'боли в суставах') !== false ||
                        strpos($title, 'боль в горле') !== false ||
                        strpos($title, 'боли в горле') !== false ||
                        strpos($title, 'боль в ухе') !== false ||
                        strpos($title, 'боли в ухе') !== false ||
                        strpos($title, 'боль в животе') !== false ||
                        strpos($title, 'боли в животе') !== false ||
                        strpos($title, 'боль в пояснице') !== false ||
                        strpos($title, 'боли в пояснице') !== false) {
                        $shouldExclude = true;
                    }
                }
                
                if (!$shouldExclude) {
                    $filteredItems[$id] = $item;
                }
            }
            $sectionItems = $filteredItems;
            
            // Обновляем validatedDataIds после фильтрации
            $validatedDataIds[$section_name] = array_keys($filteredItems);
        }
        
        // Фильтруем специализации
        if ($section_name === 'specializations') {
            $filteredItems = [];
            foreach ($sectionItems as $id => $item) {
                $name = mb_strtolower($item['name'] ?? '');
                $shouldExclude = false;
                
                // Для "бородавка" исключаем специализацию аллергология
                $isWartQuery = (strpos($questionLower, 'бородавк') !== false ||
                               strpos($questionLower, 'папиллом') !== false ||
                               strpos($questionLower, 'кондилом') !== false);
                if ($isWartQuery) {
                    if (strpos($name, 'аллерголог') !== false) {
                        $shouldExclude = true;
                    }
                }
                
                if (!$shouldExclude) {
                    $filteredItems[$id] = $item;
                }
            }
            $sectionItems = $filteredItems;
            
            // Обновляем validatedDataIds после фильтрации
            $validatedDataIds[$section_name] = array_keys($filteredItems);
            
            // ИСПРАВЛЕНИЕ 2: Если специализаций нет, но AI вернул пустой массив - автоопределяем из вопроса
            if (empty($validatedDataIds[$section_name])) {
                logParser("DEBUG specializations: AI did not select specializations for query '{$question}'. Auto-detecting from keywords.");
                
                $autoSpecializations = [];
                
                // Карта специальностей к их ID в БД
                $specialtyNameToSearch = [
                    'Невролог' => ['невролог'],
                    'Терапия' => ['терапия', 'терапевт'],
                    'Кардиология' => ['кардиология', 'кардиолог'],
                    'Гастроэнтерология' => ['гастроэнтерология', 'гастроэнтеролог'],
                    'Урология' => ['урология', 'уролог'],
                    'Оториноларингология (ЛОР)' => ['оториноларингология', 'лор'],
                    'Дерматология' => ['дерматология', 'дерматолог'],
                    'Офтальмология' => ['офтальмология', 'офтальмолог'],
                    'Аллергология' => ['аллергология', 'аллерголог'],
                    'Эндокринология' => ['эндокринология', 'эндокринолог'],
                    'Травматология и ортопедия' => ['травматология', 'ортопедия', 'травматолог', 'ортопед'],
                    'Хирургия' => ['хирургия', 'хирург'],
                    'Ревматология' => ['ревматология', 'ревматолог'],
                    'Педиатрия' => ['педиатрия', 'педиатр'],
                    'Пульмонология' => ['пульмонология', 'пульмонолог'],
                    'Справки и комиссии' => ['справки', 'комиссии', 'медосмотр', 'профосмотр', 'диспансеризация']
                ];
                
                $questionLower = mb_strtolower($question);
                
                // Определяем специальности из вопроса
                // Онемение/покалывание - невролог + кардиология + эндокринология
                if (preg_match('/(онемен|покалыван|мурашк|жжен|жжени)/ui', $question)) {
                    $autoSpecializations[] = 'невролог';
                    $autoSpecializations[] = 'кардиология';
                    $autoSpecializations[] = 'эндокринология';
                }
                // Бессонница - невролог + психотерапия (если есть)
                if (preg_match('/(бессонн|инсомн|проблем.*сон|не.*сплю|не.*засыпаю)/ui', $question)) {
                    $autoSpecializations[] = 'невролог';
                }
                if (preg_match('/(невролог|голов|мигрен|спин.*бол|боль.*спин)/ui', $question)) {
                    $autoSpecializations[] = 'невролог';
                }
                // Административные запросы (справки, медосмотр, комиссия)
                if (preg_match('/(справк|медосмотр|медицинск.*осмотр|комисс|диспансеризац|профосмотр)/ui', $question)) {
                    // Ищем специализацию "Справки и комиссии"
                    $autoSpecializations[] = 'справки и комиссии';
                }
                if (preg_match('/(терапевт|температур|просту|грипп|орви|слабость|усталость|утомляемость)/ui', $question)) {
                    $autoSpecializations[] = 'терапия';
                }
                if (preg_match('/(кардиолог|сердц|давлен|гипертон|аритми)/ui', $question)) {
                    $autoSpecializations[] = 'кардиология';
                }
                if (preg_match('/(гастроэнтеролог|желуд|живот|тошнот|изжог|пищеварен)/ui', $question)) {
                    $autoSpecializations[] = 'гастроэнтерология';
                }
                if (preg_match('/(уролог|мочеиспускан|почк|цистит|простат)/ui', $question)) {
                    $autoSpecializations[] = 'урология';
                }
                if (preg_match('/(лор|оториноларинголог|нос|горл|ухо|ангин|отит)/ui', $question)) {
                    $autoSpecializations[] = 'оториноларингология';
                }
                if (preg_match('/(дерматолог|кож|сыпь|акне|псориаз|экзема)/ui', $question)) {
                    $autoSpecializations[] = 'дерматология';
                }
                if (preg_match('/(офтальмолог|глаз|зрен|конъюнктивит|очк|подбор очков)/ui', $question)) {
                    $autoSpecializations[] = 'офтальмология';
                }
                if (preg_match('/(аллерголог|аллерг|крапивниц)/ui', $question)) {
                    $autoSpecializations[] = 'аллергология';
                }
                if (preg_match('/(эндокринолог|щитовидн|диабет|гормон|сахар|потливость|холестерин)/ui', $question)) {
                    $autoSpecializations[] = 'эндокринология';
                }
                if (preg_match('/(травматолог|ортопед|перелом|вывих|сустав)/ui', $question)) {
                    $autoSpecializations[] = 'травматология';
                }
                if (preg_match('/(хирург|аппендицит|грыж|операц|хирургическ)/ui', $question)) {
                    $autoSpecializations[] = 'хирургия';
                }
                if (preg_match('/(ревматолог|артрит|подагр|ревматизм|волчанк|скованность.*суставах|полиартрит|боль.*всех.*суставах)/ui', $question)) {
                    $autoSpecializations[] = 'ревматология';
                }
                if (preg_match('/(педиатр|ребенка|детей|новорожденн|корь|краснух|скарлатин|коклюш|ветрянк|диатез)/ui', $question)) {
                    $autoSpecializations[] = 'педиатрия';
                }
                if (preg_match('/(пульмонолог|одышк|бронхит|пневмон|хобл|астма|легких|удушь|дых.*затруднен)/ui', $question)) {
                    $autoSpecializations[] = 'пульмонология';
                }
                
                if (!empty($autoSpecializations)) {
                    // Ищем ID специализаций в БД
                    foreach ($autoSpecializations as $specialty) {
                        $searchTerms = $specialtyNameToSearch[$specialty] ?? [$specialty];
                        
                        foreach ($searchTerms as $term) {
                            try {
                                $stmtSpec = $db->prepare("
                                    SELECT DISTINCT pi.id
                                    FROM parsed_items pi
                                    JOIN parsed_fields pf ON pi.id = pf.item_id
                                    WHERE pi.widget_id = ? 
                                    AND pi.section_name = 'specializations'
                                    AND pi.is_duplicate = 0
                                    AND LOWER(pf.field_value) LIKE ?
                                    LIMIT 1
                                ");
                                $stmtSpec->execute([$widget['id'], '%' . mb_strtolower($term) . '%']);
                                
                                $row = $stmtSpec->fetch();
                                if ($row) {
                                    $validatedDataIds[$section_name][] = (int)$row['id'];
                                    
                                    // Загружаем данные специализации
                                    $stmtLoad = $db->prepare("
                                        SELECT pi.id, pf.field_name, pf.field_value
                                        FROM parsed_items pi
                                        JOIN parsed_fields pf ON pi.id = pf.item_id
                                        WHERE pi.id = ?
                                    ");
                                    $stmtLoad->execute([(int)$row['id']]);
                                    
                                    $specItem = ['id' => (int)$row['id']];
                                    while ($rowData = $stmtLoad->fetch()) {
                                        $specItem[$rowData['field_name']] = $rowData['field_value'];
                                    }
                                    $sectionItems[(int)$row['id']] = $specItem;
                                    break; // Нашли, переходим к следующей специальности
                                }
                            } catch (Exception $e) {
                                logParser("DEBUG specializations auto-detect: ERROR for $term: " . $e->getMessage());
                            }
                        }
                    }
                    
                    if (!empty($validatedDataIds[$section_name])) {
                        $validatedDataIds[$section_name] = array_unique($validatedDataIds[$section_name]);
                        logParser("DEBUG specializations: Auto-detected " . count($validatedDataIds[$section_name]) . " specializations (" . implode(', ', $autoSpecializations) . ") for query '{$question}'");
                    }
                }
            }
        }
        
        // Сортируем согласно порядку AI, но с учетом правил для консультаций
        if ($section_name === 'services') {
            // ЛОГИКА: Добавляем консультации специалистов, даже если AI выбрал другие услуги
            // Определяем релевантные специальности из вопроса
            $relevantSpecialties = [];
            
            if (preg_match('/(бессонн|невролог|голов|головокруж|мигрен|головн.*бол|нервн|тремор|онемен|защемл|спин.*бол|боль.*спин)/ui', $question)) {
                $relevantSpecialties[] = 'невролог';
            }
            if (preg_match('/(температур|просту|грипп|орви|орз|кашель|насморк|слабость|терапевт|одышк|аппетит)/ui', $question)) {
                $relevantSpecialties[] = 'терапевт';
            }
            if (preg_match('/(травм|ортопед|перелом|вывих|растяжен|ушиб|спин.*бол|боль.*спин|сустав|колен|плеч)/ui', $question)) {
                $relevantSpecialties[] = 'травматолог';
            }
            if (preg_match('/(сердц|давлен|гипертон|аритми|тахикарди|кардиолог|одышк|груд.*бол|боль.*груд)/ui', $question)) {
                $relevantSpecialties[] = 'кардиолог';
            }
            if (preg_match('/(живот|желуд|гастрит|язв|тошнот|рвот|изжог|пищеварен|гастроэнтеролог|аппетит)/ui', $question)) {
                $relevantSpecialties[] = 'гастроэнтеролог';
            }
            if (preg_match('/(кож|сыпь|зуд|прыщ|акне|псориаз|розацеа|дерматолог|дерматит)/ui', $question)) {
                $relevantSpecialties[] = 'дерматолог';
            }
            if (preg_match('/(мочеиспускан|урина|уролог|моч[еа]|почк|мочевой пузырь|цистит|пиелонефрит|простат)/ui', $question)) {
                $relevantSpecialties[] = 'уролог';
            }
            if (preg_match('/(нос|горл|ухо|лор|оториноларинголог|ангин|фарингит|тонзиллит|ринит|синусит|гайморит|отит|кровотечение.*нос|носов.*кровотечен)/ui', $question)) {
                $relevantSpecialties[] = 'лор';
            }
            if (preg_match('/(аллерг|аллерголог|аллергическ|крапивниц)/ui', $question)) {
                $relevantSpecialties[] = 'аллерголог';
            }
            if (preg_match('/(глаз|зрен|офтальмолог|конъюнктивит|катаракт|глауком)/ui', $question)) {
                $relevantSpecialties[] = 'офтальмолог';
            }
            if (preg_match('/(хирург|аппендицит|грыж|операц|хирургическ)/ui', $question)) {
                $relevantSpecialties[] = 'хирург';
            }
            if (preg_match('/(ревматолог|артрит|подагр|ревматизм|волчанк|скованность.*суставах|полиартрит|боль.*всех.*суставах)/ui', $question)) {
                $relevantSpecialties[] = 'ревматолог';
            }
            if (preg_match('/(педиатр|ребенка|детей|новорожденн|корь|краснух|скарлатин|коклюш|ветрянк|диатез)/ui', $question)) {
                $relevantSpecialties[] = 'педиатр';
            }
            if (preg_match('/(пульмонолог|одышк|бронхит|пневмон|хобл|астма|легких|удушь|дых.*затруднен|приступ.*удушь|задыхаю)/ui', $question)) {
                $relevantSpecialties[] = 'пульмонолог';
                $relevantSpecialties[] = 'терапевт';
                $relevantSpecialties[] = 'кардиолог';
            }
            if (preg_match('/(лимфоузл|лимфатич|увеличен.*узл)/ui', $question)) {
                // Инфекциониста нет, используем терапевта
                $relevantSpecialties[] = 'терапевт';
                $relevantSpecialties[] = 'онколог';
            }
            if (preg_match('/(отек.*ног|отеч.*ног|вен.*недостаточн|варикоз)/ui', $question)) {
                // Флеболога нет, используем терапевта + кардиолога
                $relevantSpecialties[] = 'терапевт';
                $relevantSpecialties[] = 'кардиолог';
                $relevantSpecialties[] = 'нефролог';
            }
            if (preg_match('/(метеоризм|газообразован|вздут.*живот|горечь.*рту|сухость.*рту)/ui', $question)) {
                $relevantSpecialties[] = 'гастроэнтеролог';
            }
            
            // Если есть релевантные специальности, добавляем их консультации
            $consultationIdsToAdd = [];
            if (!empty($relevantSpecialties)) {
                // Маппинг сокращенных названий на полные (как в базе данных)
                $specialtyMapping = [
                    'лор' => 'оториноларинголог',
                    'невролог' => 'невролог',
                    'терапевт' => 'терапевт',
                    'кардиолог' => 'кардиолог',
                    'гастроэнтеролог' => 'гастроэнтеролог',
                    'дерматолог' => 'дерматолог',
                    'уролог' => 'уролог',
                    'аллерголог' => 'аллерголог',
                    'офтальмолог' => 'офтальмолог',
                    'травматолог' => 'травматолог',
                    'хирург' => 'хирург',
                    'эндокринолог' => 'эндокринолог',
                    'ревматолог' => 'ревматолог',
                    'педиатр' => 'педиатр',
                    'пульмонолог' => 'пульмонолог',
                    'ортопед' => 'ортопед',
                    'проктолог' => 'проктолог',
                    'онколог' => 'онколог',
                    'гинеколог' => 'гинеколог',
                    'маммолог' => 'маммолог',
                    'нефролог' => 'нефролог',
                    'гематолог' => 'гематолог',
                    'психотерапевт' => 'психотерапевт',
                    'психиатр' => 'психиатр',
                    'косметолог' => 'косметолог',
                    'трихолог' => 'трихолог'
                ];
                
                foreach ($relevantSpecialties as $specialty) {
                    // Преобразуем сокращение в полное название
                    $searchSpecialty = $specialtyMapping[$specialty] ?? $specialty;
                    
                    try {
                        $stmtConsult = $db->prepare("
                            SELECT DISTINCT pi.id,
                                CASE 
                                    WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 1
                                    WHEN LOWER(pf.field_value) LIKE '%повторная%' THEN 2
                                    ELSE 3
                                END as sort_order
                            FROM parsed_items pi
                            JOIN parsed_fields pf ON pi.id = pf.item_id
                            WHERE pi.widget_id = ? 
                            AND pi.section_name = 'services'
                            AND pf.field_name = 'name'
                            AND LOWER(pf.field_value) LIKE ?
                            AND LOWER(pf.field_value) LIKE '%консультация%'
                            AND pi.is_duplicate = 0
                            ORDER BY sort_order
                            LIMIT 2
                        ");
                        $stmtConsult->execute([$widget['id'], '%' . $searchSpecialty . '%']);
                        
                        while ($row = $stmtConsult->fetch()) {
                            $cid = (int)$row['id'];
                            // Добавляем только если этой консультации ещё нет в списке
                            if (!in_array($cid, $validatedDataIds[$section_name] ?? [])) {
                                $consultationIdsToAdd[] = $cid;
                                
                                // Загружаем данные консультации
                                if (!isset($sectionItems[$cid])) {
                                    $stmtLoad = $db->prepare("
                                        SELECT pi.id, pf.field_name, pf.field_value
                                        FROM parsed_items pi
                                        JOIN parsed_fields pf ON pi.id = pf.item_id
                                        WHERE pi.id = ?
                                    ");
                                    $stmtLoad->execute([$cid]);
                                    
                                    $consultItem = ['id' => $cid];
                                    while ($rowData = $stmtLoad->fetch()) {
                                        $consultItem[$rowData['field_name']] = $rowData['field_value'];
                                    }
                                    $sectionItems[$cid] = $consultItem;
                                }
                            }
                        }
                    } catch (Exception $e) {
                        logParser("DEBUG services: ERROR adding consultation for $specialty: " . $e->getMessage());
                    }
                }
                
                if (!empty($consultationIdsToAdd)) {
                    // Добавляем консультации в НАЧАЛО списка
                    $validatedDataIds[$section_name] = array_merge($consultationIdsToAdd, $validatedDataIds[$section_name] ?? []);
                    logParser("DEBUG services: Added " . count($consultationIdsToAdd) . " consultations to existing services for query '{$question}'");
                }
            }
            
            // Разделяем на консультации и остальные
            $consultations = [];
            $otherServices = [];
            
            // FALLBACK: Если validatedDataIds пустой, но есть отфильтрованные услуги, используем их
            if (empty($validatedDataIds[$section_name]) && !empty($sectionItems)) {
                error_log("Warning: validatedDataIds is empty for services, using all filtered items. Query: '{$question}'");
                // Используем все отфильтрованные услуги
                $validatedDataIds[$section_name] = array_keys($sectionItems);
            }
            
            // ФИЛЬТРАЦИЯ: Исключаем психиатра/психотерапевта для физических симптомов
            $questionLower = mb_strtolower($question);
            $isPhysicalSymptom = preg_match('/(температур|боль|болит|кашел|насморк|одышк|тошнот|рвот|понос|запор|головокруж|слабость|усталость|кровотечен|кровь|травм|перелом|вывих|ушиб|отёк|опух|сыпь|зуд|покраснен)/ui', $question);
            
            $filteredValidatedIds = [];
            foreach ($validatedDataIds[$section_name] as $id) {
                if (isset($sectionItems[$id])) {
                    $name = mb_strtolower($sectionItems[$id]['name'] ?? $sectionItems[$id]['title'] ?? '');
                    $isConsultation = (strpos($name, 'консультация') !== false || strpos($name, 'осмотр') !== false);
                    
                    // Исключаем психиатра/психотерапевта для физических симптомов
                    $isPsychiatrist = (strpos($name, 'психиатр') !== false || strpos($name, 'психотерапевт') !== false);
                    if ($isPsychiatrist && $isPhysicalSymptom) {
                        logParser("DEBUG filter: Excluded psychiatrist/psychotherapist for physical symptom '{$question}': {$name}");
                        continue;
                    }
                    
                    $filteredValidatedIds[] = $id;
                    
                    if ($isConsultation) {
                        $consultations[] = $sectionItems[$id];
                    } else {
                        $otherServices[] = $sectionItems[$id];
                    }
                }
            }
            
            // Обновляем список валидированных ID после фильтрации
            $validatedDataIds[$section_name] = $filteredValidatedIds;
            
            // УЛУЧШЕНИЕ 1: Автодобавление диагностических услуг на основе симптомов
            // Добавляем, если:
            // - Только консультации (нет диагностики)
            // - Мало услуг (≤ 3)
            // - Вообще нет услуг
            $totalServices = count($consultations) + count($otherServices);
            $onlyConsultations = $totalServices > 0 && count($otherServices) == 0;
            
            // Проверяем, есть ли уже диагностические услуги
            $hasDiagnostics = false;
            $diagnosticKeywords = ['анализ', 'рентген', 'узи', 'экг', 'мрт', 'кт', 'флюорография', 'спирометр', 'денситометр'];
            foreach ($otherServices as $service) {
                $name = mb_strtolower($service['name'] ?? $service['title'] ?? '');
                foreach ($diagnosticKeywords as $keyword) {
                    if (strpos($name, $keyword) !== false) {
                        $hasDiagnostics = true;
                        break 2;
                    }
                }
            }
            
            // Добавляем диагностику, если ее нет или услуг мало
            if ($onlyConsultations || $totalServices <= 3 || !$hasDiagnostics) {
                $questionLower = mb_strtolower($question);
                $diagnosticServices = [];
                
                // Определяем какие диагностические услуги нужны на основе симптома
                if (preg_match('/(температур|лихорадк|жар|озноб|грипп|простуд|орз)/ui', $question)) {
                    // Для температуры: анализы крови
                    $diagnosticServices[] = ['keywords' => ['общий анализ крови', 'оак'], 'specialty' => 'терапевт'];
                    $diagnosticServices[] = ['keywords' => ['биохимический анализ крови'], 'specialty' => 'терапевт'];
                } elseif (preg_match('/(бессонн|сон|засыпа|просыпа)/ui', $question)) {
                    // Для бессонницы: ЭЭГ, полисомнография, анализы
                    $diagnosticServices[] = ['keywords' => ['ээг', 'электроэнцефалография'], 'specialty' => 'невролог'];
                    $diagnosticServices[] = ['keywords' => ['полисомнография'], 'specialty' => 'невролог'];
                    $diagnosticServices[] = ['keywords' => ['анализ.*гормон', 'гормональ'], 'specialty' => 'эндокринолог'];
                } elseif (preg_match('/(одышк|дыхан|задыха|удушь|приступ.*удушь|хобл|бронхит|пневмон|легких|затруднен.*дыхан|кашель.*мокрот)/ui', $question)) {
                    // Для одышки/ХОБЛ/пневмонии/затрудненного дыхания: ЭКГ, спирометрия, рентген, анализы
                    $diagnosticServices[] = ['keywords' => ['экг', 'электрокардиограмма'], 'specialty' => 'кардиолог'];
                    $diagnosticServices[] = ['keywords' => ['спирометрия', 'спирография'], 'specialty' => 'пульмонолог'];
                    $diagnosticServices[] = ['keywords' => ['рентген.*груд', 'флюорография', 'рентген.*легк'], 'specialty' => 'терапевт'];
                    $diagnosticServices[] = ['keywords' => ['общий анализ крови'], 'specialty' => 'терапевт'];
                } elseif (preg_match('/(межпозвон.*грыж|грыж.*межпозвон|грыж.*позвон|позвон.*грыж|защемлен.*нерв.*спин)/ui', $question)) {
                    // Для межпозвоночной грыжи: МРТ позвоночника, рентген
                    $diagnosticServices[] = ['keywords' => ['мрт.*позвон', 'мрт.*спин', 'магнитно.*резонансн.*позвон'], 'specialty' => 'невролог'];
                    $diagnosticServices[] = ['keywords' => ['рентген.*позвон', 'рентген.*спин'], 'specialty' => 'травматолог'];
                } elseif (preg_match('/(шейно.*воротников|воротников.*зон|боль.*ше.*воротник)/ui', $question)) {
                    // Для шейно-воротниковой зоны: МРТ/рентген шейного отдела
                    $diagnosticServices[] = ['keywords' => ['мрт.*шейн', 'мрт.*ше', 'магнитно.*резонансн.*шейн'], 'specialty' => 'невролог'];
                    $diagnosticServices[] = ['keywords' => ['рентген.*шейн', 'рентген.*ше'], 'specialty' => 'травматолог'];
                } elseif (preg_match('/(тремор.*рук|тремор.*причин|рук.*тремор|дрож.*рук)/ui', $question)) {
                    // Для тремора рук: МРТ головного мозга, анализы на гормоны
                    $diagnosticServices[] = ['keywords' => ['мрт.*головн', 'мрт.*мозг', 'магнитно.*резонансн.*головн'], 'specialty' => 'невролог'];
                    $diagnosticServices[] = ['keywords' => ['гормон.*щитовидн', 'ттг', 'т4', 'т3'], 'specialty' => 'эндокринолог'];
                } elseif (preg_match('/(профилактическ|профосмотр|диспансеризац|чекап|общий.*анализ.*кров.*сдать)/ui', $question)) {
                    // Для профосмотра: общий анализ крови, биохимия
                    $diagnosticServices[] = ['keywords' => ['общий анализ крови', 'оак'], 'specialty' => 'терапевт'];
                    $diagnosticServices[] = ['keywords' => ['биохимический анализ крови'], 'specialty' => 'терапевт'];
                } elseif (preg_match('/(горечь|сухость.*рту|метеоризм|газообразован|вздут|тошнот|рвот|изжог)/ui', $question)) {
                    // Для ЖКТ: УЗИ, ФГДС, анализы
                    $diagnosticServices[] = ['keywords' => ['узи.*брюшн', 'узи.*печен', 'узи органов брюшной'], 'specialty' => 'гастроэнтеролог'];
                    $diagnosticServices[] = ['keywords' => ['фгдс', 'гастроскопия', 'эндоскопия'], 'specialty' => 'гастроэнтеролог'];
                    $diagnosticServices[] = ['keywords' => ['общий анализ крови'], 'specialty' => 'терапевт'];
                } elseif (preg_match('/(подагра|мочевая кислота|сустав.*бол|боль.*сустав|скованн.*сустав)/ui', $question)) {
                    // Для подагры/суставов: анализы крови (мочевая кислота), рентген
                    $diagnosticServices[] = ['keywords' => ['мочев.*кислот', 'биохимический анализ'], 'specialty' => 'терапевт'];
                    $diagnosticServices[] = ['keywords' => ['общий анализ крови', 'оак'], 'specialty' => 'терапевт'];
                    $diagnosticServices[] = ['keywords' => ['рентген.*сустав'], 'specialty' => 'травматолог'];
                    $diagnosticServices[] = ['keywords' => ['узи.*сустав'], 'specialty' => 'травматолог'];
                } elseif (preg_match('/(остеопороз|костн.*ткан|хрупкость.*кост|денситометр)/ui', $question)) {
                    // Для остеопороза: денситометрия, анализы (кальций, витамин D, фосфор)
                    $diagnosticServices[] = ['keywords' => ['денситометр'], 'specialty' => 'эндокринолог'];
                    $diagnosticServices[] = ['keywords' => ['кальций', 'биохимический анализ'], 'specialty' => 'эндокринолог'];
                    $diagnosticServices[] = ['keywords' => ['витамин.*d', 'витамин d'], 'specialty' => 'эндокринолог'];
                    $diagnosticServices[] = ['keywords' => ['общий анализ крови'], 'specialty' => 'терапевт'];
                } elseif (preg_match('/(скарлатин|корь|краснух|коклюш|ветрянк|детск.*инфекц)/ui', $question)) {
                    // Для детских инфекций: мазок из зева, общий анализ крови
                    $diagnosticServices[] = ['keywords' => ['мазок.*зев', 'мазок на стрептококк'], 'specialty' => 'педиатр'];
                    $diagnosticServices[] = ['keywords' => ['общий анализ крови', 'оак'], 'specialty' => 'терапевт'];
                }
                
                // Ищем эти услуги в БД и добавляем
                foreach ($diagnosticServices as $diagService) {
                    $keywordsPattern = '%' . implode('%', $diagService['keywords']) . '%';
                    
                    try {
                        $stmtDiag = $db->prepare("
                            SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                            FROM parsed_items pi
                            JOIN parsed_fields pf ON pi.id = pf.item_id
                            WHERE pi.widget_id = ? 
                            AND pi.section_name = 'services'
                            AND LOWER(pf.field_value) REGEXP ?
                            AND pi.is_duplicate = 0
                            LIMIT 1
                        ");
                        
                        // Создаем regex паттерн из ключевых слов
                        $regexPattern = implode('|', array_map(function($kw) {
                            return preg_quote($kw, '/');
                        }, $diagService['keywords']));
                        
                        $stmtDiag->execute([$widget['id'], $regexPattern]);
                        
                        $diagItem = null;
                        while ($row = $stmtDiag->fetch()) {
                            if (!$diagItem) {
                                $diagItem = ['id' => (int)$row['id']];
                            }
                            $diagItem[$row['field_name']] = $row['field_value'];
                        }
                        
                        if ($diagItem && !isset($sectionItems[$diagItem['id']])) {
                            $otherServices[] = $diagItem;
                            $sectionItems[$diagItem['id']] = $diagItem;
                            $validatedDataIds[$section_name][] = $diagItem['id'];
                            logParser("DEBUG diagnostic: Added diagnostic service '{$diagItem['name']}' for query '{$question}'");
                        }
                    } catch (Exception $e) {
                        logParser("DEBUG diagnostic: ERROR adding diagnostic service: " . $e->getMessage());
                    }
                }
            }
            
            // УЛУЧШЕНИЕ 2: Если нет консультаций, но есть только анализы/процедуры → добавляем консультацию специалиста
            if (empty($consultations) && !empty($otherServices)) {
                // Проверяем, все ли услуги - это анализы/диагностика
                $allDiagnostic = true;
                $diagnosticKeywords = ['анализ', 'исследование', 'узи', 'экг', 'холтер', 'рентген', 'мрт', 'кт', 'флюорография', 'томография', 'диагностика'];
                
                foreach ($otherServices as $service) {
                    $name = mb_strtolower($service['name'] ?? $service['title'] ?? '');
                    $isDiagnostic = false;
                    foreach ($diagnosticKeywords as $keyword) {
                        if (strpos($name, $keyword) !== false) {
                            $isDiagnostic = true;
                            break;
                        }
                    }
                    if (!$isDiagnostic) {
                        $allDiagnostic = false;
                        break;
                    }
                }
                
                // Если все услуги - диагностика, добавляем консультацию специалиста
                if ($allDiagnostic) {
                    $questionLower = mb_strtolower($question);
                    $consultationSpecialty = null;
                    
                    // Определяем специальность из вопроса или названия услуг
                    if (preg_match('/(холестерин|сахар|диабет|щитовидн|гормон|потливость)/ui', $question)) {
                        $consultationSpecialty = 'эндокринолог';
                    } elseif (preg_match('/(сердц|давлен|гипертон|экг|холтер)/ui', $question)) {
                        $consultationSpecialty = 'кардиолог';
                    } elseif (preg_match('/(желуд|живот|гастр)/ui', $question)) {
                        $consultationSpecialty = 'гастроэнтеролог';
                    } elseif (preg_match('/(мочеиспускан|почк|цистит|уролог)/ui', $question)) {
                        $consultationSpecialty = 'уролог';
                    } elseif (preg_match('/(невролог|голов|спин)/ui', $question)) {
                        $consultationSpecialty = 'невролог';
                    } elseif (preg_match('/(бессонн|сон)/ui', $question)) {
                        $consultationSpecialty = 'невролог';
                    } elseif (preg_match('/(температур|лихорадк)/ui', $question)) {
                        $consultationSpecialty = 'терапевт';
                    } else {
                        // По умолчанию - терапевт
                        $consultationSpecialty = 'терапевт';
                    }
                    
                    // Ищем консультацию специалиста
                    if ($consultationSpecialty) {
                        try {
                            $stmtConsult = $db->prepare("
                                SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                                FROM parsed_items pi
                                JOIN parsed_fields pf ON pi.id = pf.item_id
                                WHERE pi.widget_id = ? 
                                AND pi.section_name = 'services'
                                AND pf.field_name = 'name'
                                AND LOWER(pf.field_value) LIKE ?
                                AND LOWER(pf.field_value) LIKE '%консультация%'
                                AND pi.is_duplicate = 0
                                ORDER BY CASE 
                                    WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 1
                                    WHEN LOWER(pf.field_value) LIKE '%повторная%' THEN 2
                                    ELSE 3
                                END
                                LIMIT 1
                            ");
                            $stmtConsult->execute([$widget['id'], '%' . $consultationSpecialty . '%']);
                            
                            $consultItem = null;
                            while ($row = $stmtConsult->fetch()) {
                                if (!$consultItem) {
                                    $consultItem = ['id' => (int)$row['id']];
                                }
                                $consultItem[$row['field_name']] = $row['field_value'];
                            }
                            
                            if ($consultItem && !isset($sectionItems[$consultItem['id']])) {
                                $consultations[] = $consultItem;
                                $sectionItems[$consultItem['id']] = $consultItem;
                                logParser("DEBUG diagnostic: Added consultation of $consultationSpecialty to diagnostic services for query '{$question}'");
                            }
                        } catch (Exception $e) {
                            logParser("DEBUG diagnostic: ERROR adding consultation: " . $e->getMessage());
                        }
                    }
                }
            }
            
            // Сортируем консультации: первичные перед повторными
            $validatedIds = $validatedDataIds[$section_name];
            usort($consultations, function($a, $b) use ($validatedIds) {
                $nameA = mb_strtolower($a['name'] ?? $a['title'] ?? '');
                $nameB = mb_strtolower($b['name'] ?? $b['title'] ?? '');
                
                $isPrimaryA = (strpos($nameA, 'первичн') !== false || strpos($nameA, 'первый') !== false);
                $isPrimaryB = (strpos($nameB, 'первичн') !== false || strpos($nameB, 'первый') !== false);
                
                if ($isPrimaryA === $isPrimaryB) {
                    // Обе первичные или обе повторные - сохраняем порядок от AI
                    $orderA = array_search($a['id'], $validatedIds);
                    $orderB = array_search($b['id'], $validatedIds);
                    return $orderA <=> $orderB;
                }
                
                // Первичная перед повторной
                return $isPrimaryA ? -1 : 1;
            });
            
            // Остальные услуги сохраняем в порядке от AI
            usort($otherServices, function($a, $b) use ($validatedIds) {
                $orderA = array_search($a['id'], $validatedIds);
                $orderB = array_search($b['id'], $validatedIds);
                return $orderA <=> $orderB;
            });
            
            $sortedItems = array_merge($consultations, $otherServices);
            
            // Логируем для диагностики
            error_log("DEBUG services sort: Query '{$question}' - Consultations: " . count($consultations) . ", Other: " . count($otherServices) . ", Total sorted: " . count($sortedItems));
            
            // FALLBACK: Если после сортировки ничего не осталось, но есть отфильтрованные услуги, используем их
            if (empty($sortedItems) && !empty($sectionItems)) {
                error_log("Warning: No services after sorting, using filtered items. Query: '{$question}', Filtered count: " . count($sectionItems));
                $sortedItems = array_slice(array_values($sectionItems), 0, 20);
                error_log("DEBUG services fallback after sort: Using " . count($sortedItems) . " filtered items for query '{$question}'");
            }
        } else {
            // Для других разделов сохраняем порядок от AI
            $sortedItems = [];
            foreach ($validatedDataIds[$section_name] as $id) {
                if (isset($sectionItems[$id])) {
                    $sortedItems[] = $sectionItems[$id];
                }
            }
        }
        
        // Для раздела "services" ограничиваем максимум 20 услугами
        if ($section_name === 'services' && count($sortedItems) > 20) {
            // Приоритет: сначала все консультации, затем остальные услуги
            $sortedItems = array_slice($sortedItems, 0, 20);
        }
        
        // ФИНАЛЬНАЯ ПРОВЕРКА: Если для услуг ничего не осталось, но есть отфильтрованные услуги, используем их
        if ($section_name === 'services' && empty($sortedItems) && !empty($sectionItems)) {
            error_log("CRITICAL: No services in final response for query '{$question}', but filtered items available. Count: " . count($sectionItems));
            // Используем первые 20 отфильтрованных услуг
            $sortedItems = array_slice(array_values($sectionItems), 0, 20);
            error_log("CRITICAL FIX: Using " . count($sortedItems) . " filtered items as fallback for query '{$question}'");
        }
        
        // ФИНАЛЬНАЯ ФИЛЬТРАЦИЯ: Исключаем психиатра/психотерапевта для физических симптомов
        if ($section_name === 'services') {
            $questionLower = mb_strtolower($question);
            $isPhysicalSymptom = preg_match('/(температур|боль|болит|кашел|насморк|одышк|тошнот|рвот|понос|запор|головокруж|слабость|усталость|кровотечен|кровь|травм|перелом|вывих|ушиб|отёк|опух|сыпь|зуд|покраснен)/ui', $question);
            
            if ($isPhysicalSymptom &&  count($sortedItems) > 0) {
                $filteredItems = [];
                foreach ($sortedItems as $item) {
                    $name = mb_strtolower($item['name'] ?? $item['title'] ?? '');
                    $isPsychiatrist = (strpos($name, 'психиатр') !== false || strpos($name, 'психотерапевт') !== false);
                    
                    if (!$isPsychiatrist) {
                        $filteredItems[] = $item;
                    }
                }
                $sortedItems = $filteredItems;
            }
        }
        
        $fullData[$section_name] = $sortedItems;
    } else {
        $fullData[$section_name] = [];
    }
    
    // ФИНАЛЬНАЯ ФИЛЬТРАЦИЯ СПЕЦИАЛИСТОВ: Исключаем/включаем специалистов для конкретных запросов
    if ($section_name === 'specialists' && !empty($fullData['specialists'])) {
        $questionLower = mb_strtolower($question);
        $filteredSpecialists = [];
        
        foreach ($fullData['specialists'] as $specialist) {
            $name = mb_strtolower($specialist['name'] ?? $specialist['title'] ?? '');
            $specialty = mb_strtolower($specialist['specialty'] ?? $specialist['description'] ?? '');
            $fullText = $name . ' ' . $specialty;
            $shouldInclude = true;
            
            // Для "Кровотечение из носа": исключить терапевтов, добавить ЛОРов
            if (preg_match('/(кровотечен.*нос|нос.*кровотечен|кровь.*нос|нос.*кровь)/ui', $question)) {
                if (strpos($fullText, 'терапевт') !== false && strpos($fullText, 'лор') === false) {
                    $shouldInclude = false;
                    logParser("DEBUG specialists filter: Excluding therapist '{$name}' for nosebleed query");
                }
                // ЛОРы уже должны быть включены через fallback, но убедимся
            }
            
            // Для "акне": исключить неверных специалистов, добавить косметологов и дерматологов
            if (preg_match('/(акне|угр|прыщ)/ui', $question)) {
                // Исключаем нерелевантных специалистов
                if (strpos($fullText, 'невролог') !== false || 
                    strpos($fullText, 'аллерголог') !== false ||
                    (strpos($fullText, 'терапевт') !== false && strpos($fullText, 'косметолог') === false && strpos($fullText, 'дерматолог') === false)) {
                    $shouldInclude = false;
                    logParser("DEBUG specialists filter: Excluding '{$name}' for acne query (not dermatologist/cosmetologist)");
                }
            }
            
            // Для "удалить родинку": исключить терапевтов, психологов, массажистов; оставить только косметологов и дерматовенерологов
            if (preg_match('/(удал.*родинк|родинк.*удал|удален.*родинк)/ui', $question)) {
                if (strpos($fullText, 'терапевт') !== false ||
                    strpos($fullText, 'психолог') !== false ||
                    strpos($fullText, 'массаж') !== false ||
                    strpos($fullText, 'перманентн') !== false) {
                    $shouldInclude = false;
                    logParser("DEBUG specialists filter: Excluding '{$name}' for mole removal query");
                }
                // Косметологи и дерматовенерологи уже должны быть включены
            }
            
            // Для "убрать морщины": исключить гастроэнтеролога Иванцову М.А.
            if (preg_match('/(убрать.*морщин|морщин.*убрать|разгладить.*морщин)/ui', $question)) {
                if (strpos($name, 'иванцова') !== false && strpos($fullText, 'гастроэнтеролог') !== false) {
                    $shouldInclude = false;
                    logParser("DEBUG specialists filter: Excluding Ivantsova M.A. (gastroenterologist) for wrinkles query");
                }
            }
            
            // Для "болит ухо": исключить терапевтов и педиатров, оставить только ЛОРов
            if (preg_match('/(ухо.*бол|болит.*ухо|боль.*ухо|ушах.*бол)/ui', $question)) {
                if ((strpos($fullText, 'терапевт') !== false || strpos($fullText, 'педиатр') !== false) &&
                    strpos($fullText, 'лор') === false && strpos($fullText, 'оториноларинголог') === false) {
                    $shouldInclude = false;
                    logParser("DEBUG specialists filter: Excluding therapist/pediatrician '{$name}' for ear pain query");
                }
            }
            
            // Для "болит горло": выводить терапевтов вместо ЛОРов (так как в большинстве случаев это простуда)
            if (preg_match('/(горл.*бол|болит.*горл|боль.*горл|ангин|фарингит)/ui', $question)) {
                // Исключаем ЛОРов, если есть терапевты
                if (strpos($fullText, 'лор') !== false || strpos($fullText, 'оториноларинголог') !== false) {
                    // Проверяем, есть ли терапевты в списке
                    $hasTherapists = false;
                    foreach ($fullData['specialists'] as $otherSpec) {
                        $otherName = mb_strtolower($otherSpec['name'] ?? $otherSpec['title'] ?? '');
                        $otherSpecialty = mb_strtolower($otherSpec['specialty'] ?? $otherSpec['description'] ?? '');
                        $otherFullText = $otherName . ' ' . $otherSpecialty;
                        if (strpos($otherFullText, 'терапевт') !== false) {
                            $hasTherapists = true;
                            break;
                        }
                    }
                    if ($hasTherapists) {
                        $shouldInclude = false;
                        logParser("DEBUG specialists filter: Excluding ENT '{$name}' for throat pain query (preferring therapists)");
                    }
                }
            }
            
            // Для кардиологических запросов (одышка, боль в груди, гипертония): исключить нерелевантных специалистов
            if (preg_match('/(одышк|боль.*груд|сердц|давлен|гипертон|аритми|тахикарди|кардиолог)/ui', $question)) {
                if (strpos($fullText, 'офтальмолог') !== false || 
                    strpos($fullText, 'окулист') !== false ||
                    (strpos($fullText, 'психиатр') !== false && strpos($fullText, 'кардиолог') === false) ||
                    // Для гипертонии исключаем педиатров, аллергологов, гастроэнтерологов (если нет кардиологической специализации)
                    (preg_match('/(гипертон|давлен)/ui', $question) && 
                     ((strpos($fullText, 'педиатр') !== false && strpos($fullText, 'кардиолог') === false) ||
                      (strpos($fullText, 'аллерголог') !== false && strpos($fullText, 'кардиолог') === false) ||
                      (strpos($fullText, 'гастроэнтеролог') !== false && strpos($fullText, 'кардиолог') === false)))) {
                    $shouldInclude = false;
                    logParser("DEBUG specialists filter: Excluding '{$name}' for cardiac query (not relevant)");
                }
            }
            
            // Для запросов про температуру: исключить офтальмологов и психиатров
            if (preg_match('/(температур|лихорадк|жар|озноб)/ui', $question)) {
                if (strpos($fullText, 'офтальмолог') !== false || 
                    strpos($fullText, 'окулист') !== false ||
                    (strpos($fullText, 'психиатр') !== false && strpos($fullText, 'терапевт') === false)) {
                    $shouldInclude = false;
                    logParser("DEBUG specialists filter: Excluding '{$name}' for temperature query (not relevant)");
                }
            }
            
            if ($shouldInclude) {
                $filteredSpecialists[] = $specialist;
            }
        }
        
        // Для "болит поясница": добавить неврологов и травматологов (уже есть fallback, но убедимся)
        if (preg_match('/(поясниц.*бол|болит.*поясниц|боль.*поясниц)/ui', $question)) {
            // Fallback уже должен добавить неврологов и травматологов, но проверим наличие
        }
        
        // Для "ожог": добавить хирургов (добавим через fallback ниже)
        if (preg_match('/(ожог|ожог)/ui', $question)) {
            // Добавим fallback для хирургов ниже
        }
        
        if (count($filteredSpecialists) !== count($fullData['specialists'])) {
            logParser("DEBUG specialists filter: Filtered " . (count($fullData['specialists']) - count($filteredSpecialists)) . " specialists for query '{$question}'");
            $fullData['specialists'] = $filteredSpecialists;
        }
    }
    
    // ДОПОЛНИТЕЛЬНЫЙ FALLBACK ДЛЯ СПЕЦИАЛИСТОВ В fullData: Добавляем специалистов для "болит поясница"
    if ($section_name === 'specialists') {
        $questionLower = mb_strtolower($question);
        $isBackPainQuery = preg_match('/(острая.*боль.*спин|боль.*спин|боль.*поясниц|поясниц.*бол|болит.*поясниц|спин.*бол|люмбаго|прострел)/ui', $question);
        
        if ($isBackPainQuery && (empty($fullData['specialists']) || count($fullData['specialists']) < 3)) {
            try {
                $db = getDatabase();
                
                // Добавляем всех неврологов
                $stmtNeuroSpec = $db->prepare("
                    SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? 
                    AND pi.section_name = 'specialists'
                    AND (LOWER(pf.field_value) LIKE '%невролог%')
                    AND pi.is_duplicate = 0
                    LIMIT 100
                ");
                $stmtNeuroSpec->execute([$widget['id']]);
                
                $neuroSpecs = [];
                while ($row = $stmtNeuroSpec->fetch()) {
                    if (!isset($neuroSpecs[$row['id']])) {
                        $neuroSpecs[$row['id']] = ['id' => (int)$row['id']];
                    }
                    $neuroSpecs[$row['id']][$row['field_name']] = $row['field_value'];
                }
                
                // Добавляем всех травматологов
                $stmtTraumaSpec = $db->prepare("
                    SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? 
                    AND pi.section_name = 'specialists'
                    AND (LOWER(pf.field_value) LIKE '%травматолог%' OR LOWER(pf.field_value) LIKE '%ортопед%')
                    AND pi.is_duplicate = 0
                    LIMIT 100
                ");
                $stmtTraumaSpec->execute([$widget['id']]);
                
                $traumaSpecs = [];
                while ($row = $stmtTraumaSpec->fetch()) {
                    if (!isset($traumaSpecs[$row['id']])) {
                        $traumaSpecs[$row['id']] = ['id' => (int)$row['id']];
                    }
                    $traumaSpecs[$row['id']][$row['field_name']] = $row['field_value'];
                }
                
                // Объединяем всех специалистов
                $allBackPainSpecs = array_merge(array_values($neuroSpecs), array_values($traumaSpecs));
                
                if (!empty($allBackPainSpecs)) {
                    $existingIds = array_column($fullData['specialists'] ?? [], 'id');
                    foreach ($allBackPainSpecs as $spec) {
                        if (!in_array($spec['id'], $existingIds)) {
                            $fullData['specialists'][] = $spec;
                        }
                    }
                    logParser("DEBUG specialists back pain fallback in fullData: Added " . count($neuroSpecs) . " neurologists and " . count($traumaSpecs) . " traumatologists. Total now: " . count($fullData['specialists']));
                }
            } catch (Exception $e) {
                logParser("DEBUG specialists back pain fallback in fullData: ERROR - " . $e->getMessage());
            }
        }
    }
    
    // ФИЛЬТРАЦИЯ СТАТЕЙ: Исключаем нерелевантные статьи и очищаем технические артефакты
    if ($section_name === 'articles' && !empty($fullData['articles'])) {
        $questionLower = mb_strtolower($question);
        $filteredArticles = [];
        
        foreach ($fullData['articles'] as $article) {
            // Очищаем технические артефакты из статей
            foreach ($article as $key => $value) {
                if (is_string($value)) {
                    // Удаляем HTML-комментарии (<!-- -->)
                    $value = preg_replace('/<!--.*?-->/s', '', $value);
                    // Удаляем HTML-теги
                    $value = preg_replace('/<[^>]+>/', '', $value);
                    // Удаляем HTML-сущности
                    $value = html_entity_decode($value, ENT_QUOTES | ENT_HTML5, 'UTF-8');
                    // Удаляем ссылки на изображения
                    $value = preg_replace('/<img[^>]*>/i', '', $value);
                    // Удаляем ссылки в формате <image:https://...>
                    $value = preg_replace('/<image:[^>]*>/i', '', $value);
                    // Удаляем URL (включая ссылки на изображения .jpg, .png и т.д.)
                    $value = preg_replace('/(https?:\/\/[^\s]+|www\.[^\s]+)/i', '', $value);
                    // Удаляем ссылки на изображения в формате .jpg, .png, .jpeg и т.д.
                    $value = preg_replace('/\.(jpg|jpeg|png|gif|webp|svg)[^\s]*/i', '', $value);
                    // Удаляем технические коды категорий
                    $value = preg_replace('/category:\/\/[^\s]+/i', '', $value);
                    // Удаляем дисклеймеры и юридические фразы
                    $value = preg_replace('/Имеются противопоказания\.\s*Ознакомьтесь с инструкцией или проконсультируйтесь у специалиста\./ui', '', $value);
                    $value = preg_replace('/Противопоказания\.\s*Ознакомьтесь с инструкцией/ui', '', $value);
                    $value = preg_replace('/Имеются противопоказания[^.]*\./ui', '', $value);
                    $value = preg_replace('/Проконсультируйтесь у специалиста[^.]*\./ui', '', $value);
                    
                    // Обрезаем незавершенные предложения (если текст обрывается без точки в конце)
                    // Для длинных текстов (описания статей) обрезаем на последнем завершенном предложении
                    if (mb_strlen($value) > 100 && !preg_match('/[.!?]\s*$/', trim($value))) {
                        // Ищем последнее завершенное предложение
                        if (preg_match('/^(.+[.!?])\s*[^.!?]*$/u', $value, $matches)) {
                            $value = trim($matches[1]);
                        } else {
                            // Если нет завершенных предложений, обрезаем на последней запятой или точке
                            if (preg_match('/^(.+[,.])\s*[^,.]*$/u', $value, $matches)) {
                                $value = trim($matches[1]);
                            }
                        }
                    }
                    
                    // Удаляем множественные пробелы
                    $value = preg_replace('/\s+/u', ' ', $value);
                    $value = trim($value);
                    $article[$key] = $value;
                }
            }
            
            // Удаляем поля со ссылками на изображения из JSON-данных (photo, foto, image_url, image и т.д.)
            $imageFields = ['photo', 'foto', 'image', 'image_url', 'img', 'picture', 'thumbnail', 'preview'];
            foreach ($imageFields as $field) {
                if (isset($article[$field])) {
                    unset($article[$field]);
                }
            }
            
            $title = mb_strtolower($article['title'] ?? $article['name'] ?? '');
            $shouldInclude = true;
            
            // УНИВЕРСАЛЬНАЯ ФИЛЬТРАЦИЯ: исключаем нерелевантные статьи для ВСЕХ запросов
            // Исключаем статьи про программы лояльности, критерии доступности, общие правила клиники
            if (preg_match('/(программ.*лояльност|лояльност.*пациент)/ui', $title) || 
                preg_match('/(критери.*доступност|доступност.*качеств)/ui', $title) ||
                strpos($title, 'целевые критерии') !== false ||
                (strpos($title, 'правил') !== false && strpos($title, 'подготовк') !== false && !preg_match('/(подготовк|подготов.*исследован)/ui', $questionLower)) ||
                (strpos($title, 'беременност') !== false && strpos($title, 'секс') !== false && !preg_match('/(беременност|женск.*здоров|гинеколог)/ui', $questionLower)) ||
                (strpos($title, 'беременност') !== false && strpos($title, 'развитие плода') !== false && !preg_match('/(беременност|женск.*здоров|гинеколог)/ui', $questionLower)) ||
                (strpos($title, 'чесотк') !== false && strpos($title, 'диагностик') !== false && !preg_match('/(чесотк|кож.*заболеван)/ui', $questionLower)) ||
                (strpos($title, 'диагностик') !== false && strpos($title, 'глазн') !== false && strpos($title, 'болезн') !== false && strpos($title, 'дет') !== false && !preg_match('/(глаз|офтальмолог|зрен)/ui', $questionLower)) ||
                (strpos($title, 'диагностик') !== false && strpos($title, 'варикоцеле') !== false && !preg_match('/(варикоцеле|мошонк|яичк)/ui', $questionLower))) {
                $shouldInclude = false;
                logParser("DEBUG articles filter fullData: Excluding article '{$article['title']}' (ID {$article['id']}) for query '{$question}' (loyalty program/accessibility criteria/general rules)");
            }
            
            // Для урологических запросов исключаем нерелевантные статьи
            if (preg_match('/(уролог|мочеиспускан|почк|цистит|простат|аденом|моч)/ui', $question)) {
                // Исключаем статьи про желчный пузырь, нейропсихолога, если они не про урологию
                if (strpos($title, 'желчн') !== false && strpos($title, 'моч') === false && 
                    strpos($title, 'урол') === false && strpos($title, 'простат') === false) {
                    $shouldInclude = false;
                    logParser("DEBUG articles filter: Excluding article '{$title}' for urination query (желчный пузырь)");
                }
                if (strpos($title, 'нейропсихолог') !== false && strpos($title, 'моч') === false) {
                    $shouldInclude = false;
                    logParser("DEBUG articles filter: Excluding article '{$title}' for urination query (нейропсихолог)");
                }
                // Исключаем онкологические статьи, если они не про простату/мочеиспускание
                if ((strpos($title, 'онкол') !== false || strpos($title, 'опухол') !== false || strpos($title, 'рак') !== false) &&
                    strpos($title, 'простат') === false && strpos($title, 'моч') === false && 
                    strpos($title, 'урол') === false && strpos($title, 'цистит') === false) {
                    $shouldInclude = false;
                    logParser("DEBUG articles filter: Excluding article '{$title}' for urination query (онкология не по теме)");
                }
                // Исключаем статьи про кондиломы/папилломы, если они не про мочеиспускание
                if ((strpos($title, 'кондилом') !== false || strpos($title, 'папиллом') !== false) &&
                    strpos($title, 'моч') === false && strpos($title, 'урол') === false) {
                    $shouldInclude = false;
                    logParser("DEBUG articles filter: Excluding article '{$title}' for urination query (кондиломы не по теме)");
                }
            }
            
            // Для неврологических запросов исключаем нерелевантные
            if (preg_match('/(невролог|голов|мигрен|спин.*бол|боль.*спин|бессонн)/ui', $question)) {
                if (strpos($title, 'желчн') !== false && strpos($title, 'голов') === false) {
                    $shouldInclude = false;
                    logParser("DEBUG articles filter: Excluding article '{$title}' for neurology query (желчный пузырь)");
                }
            }
            
            // Для "болит ухо": исключить статьи по косметологии
            if (preg_match('/(ухо.*бол|болит.*ухо|боль.*ухо|ушах.*бол)/ui', $question)) {
                if (strpos($title, 'косметолог') !== false || strpos($title, 'космет') !== false) {
                    $shouldInclude = false;
                    logParser("DEBUG articles filter: Excluding article '{$title}' for ear pain query (cosmetology)");
                }
            }
            
            // Для кровотечения из носа: исключить нерелевантные статьи
            if (preg_match('/(кровотечен.*нос|носов.*кровотечен|кровь.*нос)/ui', $question)) {
                if (strpos($title, 'чесотк') !== false || 
                    strpos($title, 'беременност') !== false ||
                    strpos($title, 'офтальмолог') !== false) {
                    $shouldInclude = false;
                    logParser("DEBUG articles filter: Excluding article '{$title}' for nosebleed query (irrelevant: scabies/pregnancy/ophthalmology)");
                }
            }
            
            // Для акне: исключить нерелевантные статьи
            if (preg_match('/(акне|угр|прыщ)/ui', $question)) {
                if (strpos($title, 'гемангиом') !== false || 
                    strpos($title, 'новообразовани') !== false ||
                    (strpos($title, 'родинк') !== false && strpos($title, 'акне') === false) ||
                    (strpos($title, 'бородавк') !== false && strpos($title, 'акне') === false)) {
                    $shouldInclude = false;
                    logParser("DEBUG articles filter: Excluding article '{$title}' for acne query (irrelevant: hemangioma/new growths/moles/warts)");
                }
            }
            
            // Для детских запросов (ребенок, детск, малыш) исключаем статьи про взрослые темы
            $isChildQuery = preg_match('/(ребенок|детск|малыш|младенец|грудн|новорожденн)/ui', $question);
            if ($isChildQuery) {
                // Исключаем статьи про бесплодие, гинекологию, ректальную температуру (для взрослых)
                if (strpos($title, 'бесплоди') !== false || 
                    strpos($title, 'гинеколог') !== false || 
                    strpos($title, 'ректальн') !== false ||
                    strpos($title, 'яичник') !== false ||
                    strpos($title, 'менструац') !== false ||
                    strpos($title, 'беременност') !== false ||
                    strpos($title, 'гормон.*женск') !== false) {
                    $shouldInclude = false;
                    logParser("DEBUG articles filter: Excluding article '{$title}' for child query (adult topics: infertility/gynecology)");
                }
            }
            
            if ($shouldInclude) {
                $filteredArticles[] = $article;
            }
        }
        
        // Для "ребенок простыл": используем те же ключевые слова, что для "ребенок температура"
        if (preg_match('/(ребенок.*простыл|простыл.*ребенок|дет.*простыл|простыл.*дет)/ui', $question)) {
            // Используем те же ключевые слова, что для температуры у ребенка
            $temperatureKeywords = ['%температур%', '%просту%', '%грипп%', '%орви%', '%педиатр%'];
            try {
                $placeholders = implode(' OR LOWER(pf.field_value) LIKE ', array_fill(0, count($temperatureKeywords), '?'));
                $stmtArticles = $db->prepare("
                    SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? 
                    AND pi.section_name = 'articles'
                    AND (pf.field_name = 'title' OR pf.field_name = 'name')
                    AND (LOWER(pf.field_value) LIKE $placeholders)
                    AND pi.is_duplicate = 0
                    LIMIT 5
                ");
                $stmtArticles->execute(array_merge([$widget['id']], $temperatureKeywords));
                
                $articlesData = [];
                while ($row = $stmtArticles->fetch()) {
                    if (!isset($articlesData[$row['id']])) {
                        $articlesData[$row['id']] = ['id' => (int)$row['id']];
                    }
                    $articlesData[$row['id']][$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($articlesData)) {
                    $existingIds = array_column($filteredArticles, 'id');
                    foreach ($articlesData as $article) {
                        if (!in_array($article['id'], $existingIds)) {
                            $filteredArticles[] = $article;
                        }
                    }
                    logParser("DEBUG articles filter: Added " . count($articlesData) . " articles for 'ребенок простыл' query (same as temperature)");
                }
            } catch (Exception $e) {
                logParser("DEBUG articles filter: ERROR for 'ребенок простыл': " . $e->getMessage());
            }
        }
        
        // Для "ожог": добавляем статью про ожоги по URL
        if (preg_match('/(ожог|ожог)/ui', $question)) {
            try {
                $stmtBurnArticle = $db->prepare("
                    SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? 
                    AND pi.section_name = 'articles'
                    AND pi.is_duplicate = 0
                    AND (pf.field_name = 'url' AND pf.field_value LIKE '%ozhogi%')
                    LIMIT 1
                ");
                $stmtBurnArticle->execute([$widget['id']]);
                
                $burnArticleId = null;
                while ($row = $stmtBurnArticle->fetch()) {
                    $burnArticleId = (int)$row['id'];
                    break;
                }
                
                if ($burnArticleId) {
                    // Загружаем полные данные статьи
                    $stmtBurnArticleFull = $db->prepare("
                        SELECT pi.id, pf.field_name, pf.field_value
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? AND pi.section_name = 'articles'
                        AND pi.id = ?
                        AND pi.is_duplicate = 0
                    ");
                    $stmtBurnArticleFull->execute([$widget['id'], $burnArticleId]);
                    
                    $burnArticle = ['id' => $burnArticleId];
                    while ($row = $stmtBurnArticleFull->fetch()) {
                        $burnArticle[$row['field_name']] = $row['field_value'];
                    }
                    
                    if (!empty($burnArticle) && !in_array($burnArticleId, array_column($filteredArticles, 'id'))) {
                        $filteredArticles[] = $burnArticle;
                        logParser("DEBUG articles filter: Added burn article (ID {$burnArticleId}) for burn query");
                    }
                }
            } catch (Exception $e) {
                logParser("DEBUG articles filter: ERROR adding burn article: " . $e->getMessage());
            }
        }
        
        if (count($filteredArticles) !== count($fullData['articles'])) {
            logParser("DEBUG articles filter: Filtered " . (count($fullData['articles']) - count($filteredArticles)) . " irrelevant articles for query '{$question}'");
            $fullData['articles'] = $filteredArticles;
        }
    }
    
    // CROSS-SECTION LOGIC: Если раздел specialists пустой, но есть services - ищем специалистов по услугам
    if ($section_name === 'specialists' && empty($fullData['specialists'])) {
        // Проверяем, были ли найдены услуги для этого же вопроса
        // Пытаемся извлечь специальность из названия услуг
        
        $questionLower = mb_strtolower($question);
        $foundSpecialists = [];
        
        // Определяем специальность из вопроса (используем те же паттерны что и для fallback)
        $specialtyKeywords = [];
        
        if (preg_match('/(бессонн|невролог|голов|мигрен|спин.*бол|боль.*спин)/ui', $question)) {
            $specialtyKeywords[] = 'невролог';
        }
        if (preg_match('/(терапевт|температур|просту|грипп|орви|слабость|усталость)/ui', $question)) {
            $specialtyKeywords[] = 'терапевт';
        }
        if (preg_match('/(офтальмолог|глаз|зрен|очк|подбор очков)/ui', $question)) {
            $specialtyKeywords[] = 'офтальмолог';
        }
        if (preg_match('/(эндокринолог|щитовидн|диабет|гормон|сахар|потливость|холестерин)/ui', $question)) {
            $specialtyKeywords[] = 'эндокринолог';
        }
        if (preg_match('/(кардиолог|сердц|давлен|гипертон)/ui', $question)) {
            $specialtyKeywords[] = 'кардиолог';
        }
        if (preg_match('/(гастроэнтеролог|желуд|живот|тошнот|изжог|запор)/ui', $question)) {
            $specialtyKeywords[] = 'гастроэнтеролог';
        }
        if (preg_match('/(уролог|мочеиспускан|почк|цистит)/ui', $question)) {
            $specialtyKeywords[] = 'уролог';
        }
        if (preg_match('/(лор|оториноларинголог|нос|горл|ухо)/ui', $question)) {
            $specialtyKeywords[] = 'оториноларинголог';
        }
        if (preg_match('/(дерматолог|кож|сыпь|акне)/ui', $question)) {
            $specialtyKeywords[] = 'дерматолог';
        }
        if (preg_match('/(аллерголог|аллерг)/ui', $question)) {
            $specialtyKeywords[] = 'аллерголог';
        }
        
        if (!empty($specialtyKeywords)) {
            foreach ($specialtyKeywords as $specialty) {
                try {
                    $stmtSpec = $db->prepare("
                        SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'specialists'
                        AND pi.is_duplicate = 0
                        AND (LOWER(pf.field_value) LIKE ? OR LOWER(pf.field_value) LIKE ?)
                        LIMIT 5
                    ");
                    $stmtSpec->execute([$widget['id'], '%' . $specialty . '%', '%' . mb_strtolower($specialty) . '%']);
                    
                    $specialistsData = [];
                    while ($row = $stmtSpec->fetch()) {
                        if (!isset($specialistsData[$row['id']])) {
                            $specialistsData[$row['id']] = ['id' => (int)$row['id']];
                        }
                        $specialistsData[$row['id']][$row['field_name']] = $row['field_value'];
                    }
                    
                    if (!empty($specialistsData)) {
                        $foundSpecialists = array_merge($foundSpecialists, array_values($specialistsData));
                    }
                } catch (Exception $e) {
                    logParser("DEBUG cross-section specialists: ERROR for $specialty: " . $e->getMessage());
                }
            }
            
            if (!empty($foundSpecialists)) {
                // Ограничиваем до 5 специалистов
                $fullData['specialists'] = array_slice($foundSpecialists, 0, 5);
                logParser("DEBUG cross-section: Found " . count($fullData['specialists']) . " specialists for query '{$question}' via services analysis");
            }
        }
    }
    
    // CROSS-SECTION LOGIC: Если раздел services пустой, но есть specialists - добавляем консультации
    if ($section_name === 'services' && empty($fullData['services'])) {
        // Проверяем, можем ли определить специальность из вопроса
        $questionLower = mb_strtolower($question);
        $consultationServices = [];
        
        // Определяем специальности (используем те же паттерны)
        $specialtyMapping = [
            'невролог' => 'невролог',
            'терапевт' => 'терапевт',
            'кардиолог' => 'кардиолог',
            'гастроэнтеролог' => 'гастроэнтеролог',
            'уролог' => 'уролог',
            'лор' => 'оториноларинголог',
            'дерматолог' => 'дерматолог',
            'офтальмолог' => 'офтальмолог',
            'аллерголог' => 'аллерголог',
            'эндокринолог' => 'эндокринолог',
            'травматолог' => 'травматолог',
            'пульмонолог' => 'пульмонолог',
            'ортопед' => 'ортопед',
            'проктолог' => 'проктолог',
            'онколог' => 'онколог',
            'гинеколог' => 'гинеколог',
            'маммолог' => 'маммолог',
            'нефролог' => 'нефролог',
            'гематолог' => 'гематолог',
            'психотерапевт' => 'психотерапевт',
            'психиатр' => 'психиатр',
            'косметолог' => 'косметолог',
            'трихолог' => 'трихолог'
        ];
        
        $foundSpecialties = [];
        
        if (preg_match('/(невролог|голов|мигрен|спин.*бол|боль.*спин|бессонн)/ui', $question)) {
            $foundSpecialties[] = 'невролог';
        }
        if (preg_match('/(терапевт|температур|просту|грипп|орви|слабость)/ui', $question)) {
            $foundSpecialties[] = 'терапевт';
        }
        if (preg_match('/(кардиолог|сердц|давлен|гипертон)/ui', $question)) {
            $foundSpecialties[] = 'кардиолог';
        }
        if (preg_match('/(гастроэнтеролог|желуд|живот|тошнот|изжог|запор)/ui', $question)) {
            $foundSpecialties[] = 'гастроэнтеролог';
        }
        if (preg_match('/(уролог|мочеиспускан|почк|цистит)/ui', $question)) {
            $foundSpecialties[] = 'уролог';
        }
        if (preg_match('/(лор|оториноларинголог|нос|горл|ухо)/ui', $question)) {
            $foundSpecialties[] = 'лор';
        }
        if (preg_match('/(дерматолог|кож|сыпь|акне)/ui', $question)) {
            $foundSpecialties[] = 'дерматолог';
        }
        if (preg_match('/(офтальмолог|глаз|зрен|очк|подбор очков)/ui', $question)) {
            $foundSpecialties[] = 'офтальмолог';
        }
        if (preg_match('/(аллерголог|аллерг)/ui', $question)) {
            $foundSpecialties[] = 'аллерголог';
        }
        if (preg_match('/(эндокринолог|щитовидн|диабет|гормон|сахар|потливость|холестерин)/ui', $question)) {
            $foundSpecialties[] = 'эндокринолог';
        }
        if (preg_match('/(травматолог|ортопед|перелом|вывих|сустав)/ui', $question)) {
            $foundSpecialties[] = 'травматолог';
        }
        
        if (!empty($foundSpecialties)) {
            foreach ($foundSpecialties as $specialty) {
                $searchSpecialty = $specialtyMapping[$specialty] ?? $specialty;
                
                try {
                    $stmtConsult = $db->prepare("
                        SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND LOWER(pf.field_value) LIKE ?
                        AND LOWER(pf.field_value) LIKE '%консультация%'
                        AND pi.is_duplicate = 0
                        ORDER BY CASE 
                            WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 1
                            WHEN LOWER(pf.field_value) LIKE '%повторная%' THEN 2
                            ELSE 3
                        END
                        LIMIT 2
                    ");
                    $stmtConsult->execute([$widget['id'], '%' . $searchSpecialty . '%']);
                    
                    $servicesData = [];
                    while ($row = $stmtConsult->fetch()) {
                        if (!isset($servicesData[$row['id']])) {
                            $servicesData[$row['id']] = ['id' => (int)$row['id']];
                        }
                        $servicesData[$row['id']][$row['field_name']] = $row['field_value'];
                    }
                    
                    if (!empty($servicesData)) {
                        $consultationServices = array_merge($consultationServices, array_values($servicesData));
                    }
                } catch (Exception $e) {
                    logParser("DEBUG cross-section services: ERROR for $specialty: " . $e->getMessage());
                }
            }
            
            if (!empty($consultationServices)) {
                $fullData['services'] = array_slice($consultationServices, 0, 5);
                logParser("DEBUG cross-section: Found " . count($fullData['services']) . " consultation services for query '{$question}'");
            }
        }
    }
    
    // СПЕЦИАЛЬНАЯ ЛОГИКА ДЛЯ "УДАЛИТЬ РОДИНКУ": Принудительно добавляем консультации дерматологов и косметологов
    if ($section_name === 'services' && preg_match('/(удал.*родинк|родинк.*удал|удален.*родинк|родинк.*удален)/ui', $question)) {
        $existingServiceIds = array_column($fullData['services'] ?? [], 'id');
        $moleConsultationServices = [];
        
        // Проверяем, есть ли уже консультации в списке
        $hasConsultations = false;
        foreach ($fullData['services'] ?? [] as $service) {
            $serviceName = mb_strtolower($service['name'] ?? $service['title'] ?? '');
            if (strpos($serviceName, 'консультация') !== false && 
                (strpos($serviceName, 'дерматолог') !== false || 
                 strpos($serviceName, 'косметолог') !== false ||
                 strpos($serviceName, 'дерматовенеролог') !== false)) {
                $hasConsultations = true;
                break;
            }
        }
        
        // Если консультаций нет, добавляем их принудительно
        if (!$hasConsultations) {
            logParser("DEBUG mole removal: No consultations found, adding them forcibly");
            
            $specialties = ['косметолог', 'дерматолог', 'дерматовенеролог'];
            foreach ($specialties as $specialty) {
                try {
                    $stmtConsult = $db->prepare("
                        SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                        FROM parsed_items pi
                        JOIN parsed_fields pf ON pi.id = pf.item_id
                        WHERE pi.widget_id = ? 
                        AND pi.section_name = 'services'
                        AND pf.field_name = 'name'
                        AND LOWER(pf.field_value) LIKE ?
                        AND LOWER(pf.field_value) LIKE '%консультация%'
                        AND pi.is_duplicate = 0
                        ORDER BY CASE 
                            WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 1
                            WHEN LOWER(pf.field_value) LIKE '%повторная%' THEN 2
                            ELSE 3
                        END
                        LIMIT 3
                    ");
                    $stmtConsult->execute([$widget['id'], '%' . $specialty . '%']);
                    
                    $consultData = [];
                    while ($row = $stmtConsult->fetch()) {
                        if (!isset($consultData[$row['id']])) {
                            $consultData[$row['id']] = ['id' => (int)$row['id']];
                        }
                        $consultData[$row['id']][$row['field_name']] = $row['field_value'];
                    }
                    
                    foreach ($consultData as $consult) {
                        if (!in_array($consult['id'], $existingServiceIds)) {
                            $moleConsultationServices[] = $consult;
                            $existingServiceIds[] = $consult['id'];
                            logParser("DEBUG mole removal: Added consultation service ID {$consult['id']} for specialty '{$specialty}'");
                        }
                    }
                } catch (Exception $e) {
                    logParser("DEBUG mole removal consultations: ERROR for $specialty: " . $e->getMessage());
                }
            }
            
            // Добавляем консультации в начало списка
            if (!empty($moleConsultationServices)) {
                $currentServices = $fullData['services'] ?? [];
                $fullData['services'] = array_merge($moleConsultationServices, $currentServices);
                logParser("DEBUG mole removal: Added " . count($moleConsultationServices) . " consultation services forcibly");
            }
        }
    }
    
    // АВТОМАТИЧЕСКОЕ ДОБАВЛЕНИЕ КОНСУЛЬТАЦИЙ: Всегда добавляем консультации выбранных специалистов
    if ($section_name === 'services' && !empty($fullData['specialists'])) {
        $questionLower = mb_strtolower($question);
        $existingServiceIds = array_column($fullData['services'] ?? [], 'id');
        $consultationServices = [];
        
        // Определяем специальности выбранных специалистов
        $specialtiesToAdd = [];
        foreach ($fullData['specialists'] as $specialist) {
            $name = mb_strtolower($specialist['name'] ?? $specialist['title'] ?? '');
            $specialty = mb_strtolower($specialist['specialty'] ?? $specialist['description'] ?? '');
            $fullText = $name . ' ' . $specialty;
            
            if (strpos($fullText, 'дерматолог') !== false || strpos($fullText, 'дерматовенеролог') !== false) {
                $specialtiesToAdd['дерматолог'] = true;
            }
            if (strpos($fullText, 'косметолог') !== false) {
                $specialtiesToAdd['косметолог'] = true;
            }
            if (strpos($fullText, 'педиатр') !== false) {
                $specialtiesToAdd['педиатр'] = true;
            }
            if (strpos($fullText, 'терапевт') !== false) {
                $specialtiesToAdd['терапевт'] = true;
            }
            if (strpos($fullText, 'невролог') !== false) {
                $specialtiesToAdd['невролог'] = true;
            }
            if (strpos($fullText, 'травматолог') !== false || strpos($fullText, 'ортопед') !== false) {
                $specialtiesToAdd['травматолог'] = true;
            }
            if (strpos($fullText, 'хирург') !== false) {
                $specialtiesToAdd['хирург'] = true;
            }
        }
        
        // Добавляем консультации для определенных запросов
        if (preg_match('/(акне|угр|прыщ)/ui', $question)) {
            $specialtiesToAdd['дерматолог'] = true;
            $specialtiesToAdd['косметолог'] = true;
        }
        // Для "удалить родинку" ОБЯЗАТЕЛЬНО добавляем консультации дерматологов и косметологов
        if (preg_match('/(удал.*родинк|родинк.*удал|удален.*родинк|родинк.*удален)/ui', $question)) {
            $specialtiesToAdd['косметолог'] = true;
            $specialtiesToAdd['дерматолог'] = true;
            $specialtiesToAdd['дерматовенеролог'] = true;
            logParser("DEBUG mole removal: Adding consultations for dermatologists and cosmetologists");
        }
        if (preg_match('/(убрать.*морщин|морщин.*убрать)/ui', $question)) {
            $specialtiesToAdd['косметолог'] = true;
        }
        if (preg_match('/(ребенок.*температур|температур.*ребенок|дет.*температур)/ui', $question)) {
            $specialtiesToAdd['педиатр'] = true;
        }
        if (preg_match('/(поясниц.*бол|болит.*поясниц|боль.*поясниц)/ui', $question)) {
            $specialtiesToAdd['невролог'] = true;
            $specialtiesToAdd['травматолог'] = true;
        }
        if (preg_match('/(ожог|ожог)/ui', $question)) {
            $specialtiesToAdd['хирург'] = true;
        }
        
        // Ищем консультации для каждой специальности
        foreach (array_keys($specialtiesToAdd) as $specialty) {
            try {
                $stmtConsult = $db->prepare("
                    SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? 
                    AND pi.section_name = 'services'
                    AND pf.field_name = 'name'
                    AND LOWER(pf.field_value) LIKE ?
                    AND LOWER(pf.field_value) LIKE '%консультация%'
                    AND pi.is_duplicate = 0
                    ORDER BY CASE 
                        WHEN LOWER(pf.field_value) LIKE '%первичная%' THEN 1
                        WHEN LOWER(pf.field_value) LIKE '%повторная%' THEN 2
                        ELSE 3
                    END
                    LIMIT 2
                ");
                $stmtConsult->execute([$widget['id'], '%' . $specialty . '%']);
                
                $consultData = [];
                while ($row = $stmtConsult->fetch()) {
                    if (!isset($consultData[$row['id']])) {
                        $consultData[$row['id']] = ['id' => (int)$row['id']];
                    }
                    $consultData[$row['id']][$row['field_name']] = $row['field_value'];
                }
                
                // Для "удалить родинку" добавляем все найденные консультации
                $isMoleRemovalQuery = preg_match('/(удал.*родинк|родинк.*удал|удален.*родинк|родинк.*удален)/ui', $question);
                foreach ($consultData as $consult) {
                    if (!in_array($consult['id'], $existingServiceIds)) {
                        $consultationServices[] = $consult;
                        $existingServiceIds[] = $consult['id'];
                        if ($isMoleRemovalQuery) {
                            logParser("DEBUG mole removal: Added consultation service ID {$consult['id']} for specialty '{$specialty}'");
                        }
                    }
                }
            } catch (Exception $e) {
                logParser("DEBUG auto consultations: ERROR for $specialty: " . $e->getMessage());
            }
        }
        
        // Добавляем консультации в начало списка услуг
        if (!empty($consultationServices)) {
            $currentServices = $fullData['services'] ?? [];
            // Для "удалить родинку" обязательно добавляем консультации в начало
            $isMoleRemovalQuery = preg_match('/(удал.*родинк|родинк.*удал|удален.*родинк|родинк.*удален)/ui', $question);
            if ($isMoleRemovalQuery) {
                // Убеждаемся, что консультации есть в списке
                $hasConsultations = false;
                foreach ($currentServices as $service) {
                    $serviceName = mb_strtolower($service['name'] ?? $service['title'] ?? '');
                    if (strpos($serviceName, 'консультация') !== false && 
                        (strpos($serviceName, 'дерматолог') !== false || 
                         strpos($serviceName, 'косметолог') !== false ||
                         strpos($serviceName, 'дерматовенеролог') !== false)) {
                        $hasConsultations = true;
                        break;
                    }
                }
                if (!$hasConsultations) {
                    logParser("DEBUG mole removal: No consultations found in current services, adding " . count($consultationServices) . " consultation services");
                }
            }
            $fullData['services'] = array_merge($consultationServices, $currentServices);
            logParser("DEBUG auto consultations: Added " . count($consultationServices) . " consultation services for query '{$question}'");
        }
    }
    
    // ФИЛЬТРАЦИЯ УСЛУГ: Исключаем нерелевантные услуги для конкретных запросов
    if ($section_name === 'services' && !empty($fullData['services'])) {
        $questionLower = mb_strtolower($question);
        $filteredServices = [];
        
        foreach ($fullData['services'] as $service) {
            $name = mb_strtolower($service['name'] ?? $service['title'] ?? '');
            $shouldInclude = true;
            
            // Для "акне": убрать неверные позиции
            if (preg_match('/(акне|угр|прыщ)/ui', $question)) {
                if (strpos($name, 'удаление новообразований кожи') !== false ||
                    strpos($name, 'одномоментное удаление более 3 образований') !== false ||
                    strpos($name, 'местная аппликационная анестезия') !== false ||
                    strpos($name, 'исследование кожи на микозы') !== false ||
                    (strpos($name, 'удаление новообразований') !== false && strpos($name, 'консультация') === false)) {
                    $shouldInclude = false;
                    logParser("DEBUG services filter: Excluding service '{$name}' for acne query");
                }
            }
            
            // Для "болит горло": выводить только консультации терапевтов (убрать позиции по носу)
            if (preg_match('/(горл.*бол|болит.*горл|боль.*горл|ангин|фарингит)/ui', $question)) {
                if (strpos($name, 'нос') !== false && strpos($name, 'консультация') === false) {
                    $shouldInclude = false;
                    logParser("DEBUG services filter: Excluding service '{$name}' for throat pain query (nose-related)");
                }
            }
            
            if ($shouldInclude) {
                $filteredServices[] = $service;
            }
        }
        
        if (count($filteredServices) !== count($fullData['services'])) {
            logParser("DEBUG services filter: Filtered " . (count($fullData['services']) - count($filteredServices)) . " services for query '{$question}'");
            $fullData['services'] = $filteredServices;
        }
    }
    
    // FALLBACK ДЛЯ СТАТЕЙ: Если статьи не выбраны или их мало, добавляем релевантные статьи по ключевым словам
    if ($section_name === 'articles' && (empty($fullData['articles']) || count($fullData['articles']) < 3)) {
        $questionLower = mb_strtolower($question);
        $articleKeywords = [];
        
        // Определяем ключевые слова для поиска статей на основе вопроса
        // Бессонница - ищем статьи про бессонницу/сон, а не про спину
        if (preg_match('/(бессонн|сон|засыпа|просыпа|инсомн|не.*сплю|не.*засыпаю)/ui', $question)) {
            $articleKeywords[] = '%бессонн%';
            $articleKeywords[] = '%сон%';
            $articleKeywords[] = '%инсомн%';
            $articleKeywords[] = '%гигиен.*сон%';
            $articleKeywords[] = '%стресс%';
        }
        // Онемение/покалывание - неврология + кардиология + эндокринология
        if (preg_match('/(онемен|покалыван|мурашк|жжен|жжени)/ui', $question)) {
            $articleKeywords[] = '%онемен%';
            $articleKeywords[] = '%невропат%';
            $articleKeywords[] = '%полинейропат%';
            $articleKeywords[] = '%туннельн%';
            $articleKeywords[] = '%диабет%';
            $articleKeywords[] = '%эндокрин%';
            $articleKeywords[] = '%кардиол%';
            $articleKeywords[] = '%сердц%';
        }
        if (preg_match('/(невролог|голов|мигрен|спин.*бол|боль.*спин|головокруж)/ui', $question)) {
            $articleKeywords[] = '%голов%';
            $articleKeywords[] = '%мигрен%';
            $articleKeywords[] = '%неврол%';
            $articleKeywords[] = '%спин%';
            $articleKeywords[] = '%позвон%';
        }
        if (preg_match('/(гастроэнтеролог|желуд|живот|тошнот|изжог|рвот|диаре|запор|метеоризм)/ui', $question)) {
            $articleKeywords[] = '%желуд%';
            $articleKeywords[] = '%живот%';
            $articleKeywords[] = '%гастро%';
        }
        // Урология: мочеиспускание, цистит, простата, почки
        if (preg_match('/(уролог|мочеиспускан|почк|цистит|простат|аденом|моч)/ui', $question)) {
            $articleKeywords[] = '%мочеиспуск%';
            $articleKeywords[] = '%цистит%';
            $articleKeywords[] = '%простат%';
            $articleKeywords[] = '%аденом%';
            $articleKeywords[] = '%урол%';
            $articleKeywords[] = '%моч%';
            $articleKeywords[] = '%почк%';
        }
        // Кардиология: сердце, давление, одышка
        if (preg_match('/(кардиолог|сердц|давлен|гипертон|одышк|аритми)/ui', $question)) {
            $articleKeywords[] = '%сердц%';
            $articleKeywords[] = '%давлен%';
            $articleKeywords[] = '%кардиол%';
        }
        // ЛОР: горло, нос, ухо
        if (preg_match('/(лор|оториноларинголог|нос|горл|ухо|ангин|фарингит|тонзиллит)/ui', $question)) {
            $articleKeywords[] = '%горл%';
            $articleKeywords[] = '%нос%';
            $articleKeywords[] = '%ухо%';
            $articleKeywords[] = '%ангин%';
        }
        // Дерматология: кожа, сыпь, акне
        if (preg_match('/(дерматолог|кож|сыпь|акне|псориаз|экзема|дерматит)/ui', $question)) {
            $articleKeywords[] = '%кож%';
            $articleKeywords[] = '%сыпь%';
            $articleKeywords[] = '%акне%';
            $articleKeywords[] = '%псориаз%';
        }
        // Терапия: температура, простуда, ОРВИ
        if (preg_match('/(терапевт|температур|просту|грипп|орви|орз|кашель)/ui', $question)) {
            $articleKeywords[] = '%температур%';
            $articleKeywords[] = '%просту%';
            $articleKeywords[] = '%грипп%';
            $articleKeywords[] = '%орви%';
        }
        
        if (!empty($articleKeywords)) {
            try {
                $placeholders = implode(' OR LOWER(pf.field_value) LIKE ', array_fill(0, count($articleKeywords), '?'));
                $stmtArticles = $db->prepare("
                    SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? 
                    AND pi.section_name = 'articles'
                    AND (pf.field_name = 'title' OR pf.field_name = 'name')
                    AND (LOWER(pf.field_value) LIKE $placeholders)
                    AND pi.is_duplicate = 0
                    LIMIT 10
                ");
                $stmtArticles->execute(array_merge([$widget['id']], $articleKeywords));
                
                $articlesData = [];
                while ($row = $stmtArticles->fetch()) {
                    if (!isset($articlesData[$row['id']])) {
                        $articlesData[$row['id']] = ['id' => (int)$row['id']];
                    }
                    $articlesData[$row['id']][$row['field_name']] = $row['field_value'];
                }
                
                // Фильтруем нерелевантные статьи на основе вопроса
                if (!empty($articlesData)) {
                    $filteredArticles = [];
                    $questionLower = mb_strtolower($question);
                    foreach ($articlesData as $article) {
                        $title = mb_strtolower($article['title'] ?? $article['name'] ?? '');
                        $shouldInclude = true;
                        
                        // УНИВЕРСАЛЬНАЯ ФИЛЬТРАЦИЯ: исключаем нерелевантные статьи для ВСЕХ запросов
                        if (preg_match('/(программ.*лояльност|лояльност.*пациент)/ui', $title) || 
                            preg_match('/(критери.*доступност|доступност.*качеств)/ui', $title) ||
                            strpos($title, 'целевые критерии') !== false ||
                            (strpos($title, 'правил') !== false && strpos($title, 'подготовк') !== false && !preg_match('/(подготовк|подготов.*исследован)/ui', $questionLower)) ||
                            (strpos($title, 'беременност') !== false && strpos($title, 'секс') !== false && !preg_match('/(беременност|женск.*здоров|гинеколог)/ui', $questionLower)) ||
                            (strpos($title, 'беременност') !== false && strpos($title, 'развитие плода') !== false && !preg_match('/(беременност|женск.*здоров|гинеколог)/ui', $questionLower)) ||
                            (strpos($title, 'чесотк') !== false && strpos($title, 'диагностик') !== false && !preg_match('/(чесотк|кож.*заболеван)/ui', $questionLower)) ||
                            (strpos($title, 'диагностик') !== false && strpos($title, 'глазн') !== false && strpos($title, 'болезн') !== false && strpos($title, 'дет') !== false && !preg_match('/(глаз|офтальмолог|зрен)/ui', $questionLower)) ||
                            (strpos($title, 'диагностик') !== false && strpos($title, 'варикоцеле') !== false && !preg_match('/(варикоцеле|мошонк|яичк)/ui', $questionLower))) {
                            $shouldInclude = false;
                            logParser("DEBUG articles fallback filter: Excluding article '{$article['title']}' (ID {$article['id']}) for query '{$question}' (loyalty program/accessibility criteria/general rules)");
                        }
                        
                        // Для урологических запросов исключаем нерелевантные статьи
                        if ($shouldInclude && preg_match('/(уролог|мочеиспускан|почк|цистит|простат|аденом|моч)/ui', $question)) {
                            // Исключаем статьи про желчный пузырь, нейропсихолога, если они не про урологию
                            if (strpos($title, 'желчн') !== false && strpos($title, 'моч') === false && 
                                strpos($title, 'урол') === false && strpos($title, 'простат') === false) {
                                $shouldInclude = false;
                            }
                            if (strpos($title, 'нейропсихолог') !== false && strpos($title, 'моч') === false) {
                                $shouldInclude = false;
                            }
                        }
                        
                        // Для неврологических запросов исключаем нерелевантные
                        if ($shouldInclude && preg_match('/(невролог|голов|мигрен|спин.*бол|боль.*спин)/ui', $question)) {
                            if (strpos($title, 'желчн') !== false && strpos($title, 'голов') === false) {
                                $shouldInclude = false;
                            }
                        }
                        // Для бессонницы исключаем статьи про спину и голову (если они не про бессонницу)
                        if ($shouldInclude && preg_match('/(бессонн|инсомн|не.*сплю|не.*засыпаю)/ui', $question)) {
                            if ((strpos($title, 'спин') !== false || strpos($title, 'позвон') !== false || strpos($title, 'голов') !== false) && 
                                strpos($title, 'бессонн') === false && strpos($title, 'сон') === false) {
                                $shouldInclude = false;
                                logParser("DEBUG articles fallback filter: Excluding article '{$article['title']}' (ID {$article['id']}) for insomnia query (irrelevant: spine/head)");
                            }
                        }
                        
                        if ($shouldInclude) {
                            $filteredArticles[] = $article;
                        }
                    }
                    
                    if (!empty($filteredArticles)) {
                        // Если уже есть статьи от AI, объединяем, иначе заменяем
                        if (!empty($fullData['articles'])) {
                            $existingIds = array_column($fullData['articles'], 'id');
                            foreach ($filteredArticles as $article) {
                                if (!in_array($article['id'], $existingIds)) {
                                    $fullData['articles'][] = $article;
                                }
                            }
                        } else {
                            $fullData['articles'] = array_values($filteredArticles);
                        }
                        logParser("DEBUG articles fallback: Found " . count($fullData['articles']) . " articles for query '{$question}'");
                    }
                }
            } catch (Exception $e) {
                logParser("DEBUG articles fallback: ERROR - " . $e->getMessage());
            }
        }
    }
    
    // ФИНАЛЬНАЯ ДОБАВКА ДИАГНОСТИКИ: Если очень мало услуг (только консультации), добавляем базовые диагностические услуги
    if ($section_name === 'services' && !empty($fullData['services']) && count($fullData['services']) <= 3) {
        $questionLower = mb_strtolower($question);
        $diagnosticNeeded = [];
        
        // Определяем какие диагностические услуги нужны
        if (preg_match('/(температур|лихорадк|жар|озноб|грипп|простуд|орз)/ui', $question)) {
            $diagnosticNeeded[] = ['pattern' => 'общ.*анализ.*кров', 'label' => 'Общий анализ крови'];
            $diagnosticNeeded[] = ['pattern' => 'биохимич.*анализ.*кров', 'label' => 'Биохимический анализ крови'];
        } elseif (preg_match('/(бессонн|сон|засыпа|просыпа)/ui', $question)) {
            $diagnosticNeeded[] = ['pattern' => 'ээг|электроэнцефалограф', 'label' => 'ЭЭГ'];
            $diagnosticNeeded[] = ['pattern' => 'полисомнограф', 'label' => 'Полисомнография'];
        }
        
        // Ищем и добавляем диагностические услуги
        foreach ($diagnosticNeeded as $diag) {
            try {
                $stmtDiag = $db->prepare("
                    SELECT DISTINCT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? 
                    AND pi.section_name = 'services'
                    AND LOWER(pf.field_value) REGEXP ?
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtDiag->execute([$widget['id'], $diag['pattern']]);
                
                $diagItem = null;
                while ($row = $stmtDiag->fetch()) {
                    if (!$diagItem) {
                        $diagItem = ['id' => (int)$row['id']];
                    }
                    $diagItem[$row['field_name']] = $row['field_value'];
                }
                
                // Проверяем что такой услуги ещё нет
                if ($diagItem) {
                    $alreadyExists = false;
                    foreach ($fullData['services'] as $existing) {
                        if ($existing['id'] == $diagItem['id']) {
                            $alreadyExists = true;
                            break;
                        }
                    }
                    if (!$alreadyExists) {
                        $fullData['services'][] = $diagItem;
                    }
                }
            } catch (Exception $e) {
                // Игнорируем ошибки
            }
        }
    }
    
    // ФИНАЛЬНАЯ ФИНАЛЬНАЯ ФИЛЬТРАЦИЯ: Исключаем нерелевантные услуги
    // Применяется после всех cross-section логик
    if ($section_name === 'services' && !empty($fullData['services'])) {
        $questionLower = mb_strtolower($question);
        
        // Определяем физические симптомы (где психиатр НЕ нужен)
        $isPhysicalSymptom = preg_match('/(температур|боль|болит|кашел|насморк|одышк|тошнот|рвот|понос|запор|головокруж|слабость|усталость|кровотечен|кровь|травм|перелом|вывих|ушиб|отёк|опух|сыпь|зуд|покраснен)/ui', $question);
        
        // Определяем психологические/неврологические симптомы (где психотерапевт МОЖЕТ быть нужен)
        $isPsychologicalSymptom = preg_match('/(бессонн|сон|депресс|тревог|паник|стресс|страх|фоб|настроен)/ui', $question);
        
        $filteredServices = [];
        foreach ($fullData['services'] as $item) {
            $name = mb_strtolower($item['name'] ?? $item['title'] ?? '');
            
            // 1. Исключаем психиатра ТОЛЬКО для чисто физических симптомов (не психологических)
            $isPsychiatrist = (strpos($name, 'психиатр') !== false || strpos($name, 'психотерапевт') !== false);
            if ($isPsychiatrist && $isPhysicalSymptom && !$isPsychologicalSymptom) {
                continue;
            }
            
            // 2. Исключаем нерелевантные справки и комиссии
            $isIrrelevantCertificate = (
                strpos($name, 'транспортная комиссия') !== false ||
                strpos($name, 'для транспортной комиссии') !== false ||
                strpos($name, 'справка на оружие') !== false ||
                strpos($name, 'справка для бассейна') !== false ||
                strpos($name, 'перед вакцинацией') !== false ||
                strpos($name, 'для психоневрологического интерната') !== false ||
                strpos($name, 'медицинская карта для') !== false
            );
            if ($isIrrelevantCertificate) {
                continue;
            }
            
            // 3. Исключаем специфичные нерелевантные услуги
            $isIrrelevantSpecific = (
                strpos($name, 'биологический риск') !== false && strpos($name, 'контрацептивов') !== false
            );
            if ($isIrrelevantSpecific) {
                continue;
            }
            
            // 4. Исключаем онкомаркеры если вопрос не про онкологию
            $isOncomarker = (
                strpos($name, 'онкомаркер') !== false ||
                strpos($name, 'nse') !== false ||
                strpos($name, 'cea') !== false ||
                strpos($name, 'ca 125') !== false ||
                strpos($name, 'ca 19') !== false ||
                strpos($name, 'psa') !== false ||
                preg_match('/\b(нейрон специфическая энолаза|раковоэмбриональный антиген)\b/ui', $name)
            );
            $isOncologyQuestion = preg_match('/(онкол|рак|опухол|новообразован|метастаз)/ui', $question);
            if ($isOncomarker && !$isOncologyQuestion) {
                continue;
            }
            
            $filteredServices[] = $item;
        }
        $fullData['services'] = $filteredServices;
    }
    
    // ДОБАВЛЕНИЕ СПЕЦИАЛИЗАЦИЙ В fullData: Добавляем нужные специализации в финальный ответ
    if ($section_name === 'specializations') {
        logParser("DEBUG specializations: ENTERED fullData block for query '{$question}', section_name='{$section_name}'");
        $questionLower = mb_strtolower($question);
        // Инициализируем массив если его нет
        if (!isset($fullData['specializations'])) {
            $fullData['specializations'] = [];
            logParser("DEBUG specializations: Initialized empty specializations array");
        }
        $existingIds = array_column($fullData['specializations'], 'id');
        logParser("DEBUG specializations: Current existingIds: " . implode(',', $existingIds) . ", count: " . count($fullData['specializations']));
        
        // Для "Одышка при нагрузке" и других кардиологических запросов: добавляем "Терапия"
        $isCardiacQuery = preg_match('/(одышк|боль.*груд|сердц|давлен|гипертон|аритми|тахикарди)/ui', $question);
        if ($isCardiacQuery && !in_array(5908, $existingIds)) {
            try {
                $db = getDatabase();
                $stmt = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND pi.id = 5908
                    AND pi.is_duplicate = 0
                ");
                $stmt->execute([$widget['id']]);
                
                $therapyItem = [];
                while ($row = $stmt->fetch()) {
                    if (empty($therapyItem)) {
                        $therapyItem = ['id' => (int)$row['id']];
                    }
                    $therapyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($therapyItem)) {
                    $fullData['specializations'][] = $therapyItem;
                    logParser("DEBUG specializations: Added 'Терапия' to fullData for cardiac query");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding therapy to fullData: " . $e->getMessage());
            }
        }
        
        // Для "акне": добавляем "Косметология", если её нет
        $isAcneQuery = preg_match('/(акне|угр|прыщ)/ui', $question);
        if ($isAcneQuery) {
            try {
                $db = getDatabase();
                $stmtCosmetology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%косметолог%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtCosmetology->execute([$widget['id']]);
                
                $cosmetologyItem = [];
                $cosmetologyId = null;
                while ($row = $stmtCosmetology->fetch()) {
                    if (empty($cosmetologyItem)) {
                        $cosmetologyId = (int)$row['id'];
                        $cosmetologyItem = ['id' => $cosmetologyId];
                    }
                    $cosmetologyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($cosmetologyItem) && !in_array($cosmetologyId, $existingIds)) {
                    $fullData['specializations'][] = $cosmetologyItem;
                    logParser("DEBUG specializations: Added 'Косметология' to fullData for acne query");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding cosmetology to fullData: " . $e->getMessage());
            }
        }
        
        // Для "ожог": добавляем "Хирургия", если её нет
        $isBurnQuery = preg_match('/(ожог|ожог)/ui', $question);
        if ($isBurnQuery) {
            try {
                $db = getDatabase();
                $stmtSurgery = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%хирург%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtSurgery->execute([$widget['id']]);
                
                $surgeryItem = [];
                $surgeryId = null;
                while ($row = $stmtSurgery->fetch()) {
                    if (empty($surgeryItem)) {
                        $surgeryId = (int)$row['id'];
                        $surgeryItem = ['id' => $surgeryId];
                    }
                    $surgeryItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($surgeryItem) && !in_array($surgeryId, $existingIds)) {
                    $fullData['specializations'][] = $surgeryItem;
                    logParser("DEBUG specializations: Added 'Хирургия' to fullData for burn query");
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding surgery to fullData: " . $e->getMessage());
            }
        }
        
        // Для административных запросов добавляем специализацию "Справки и комиссии"
        $isAdminQuery = preg_match('/(справк|медосмотр|медицинск.*осмотр|комисс|диспансеризац|профосмотр)/ui', $question);
        logParser("DEBUG specializations: Checking admin query for '{$question}', isAdminQuery=" . ($isAdminQuery ? 'YES' : 'NO') . ", widget_id=" . ($widget['id'] ?? 'NULL'));
        if ($isAdminQuery) {
            try {
                $db = getDatabase();
                $stmtAdmin = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND pi.is_duplicate = 0
                    AND pi.id IN (
                        SELECT DISTINCT pf2.item_id 
                        FROM parsed_fields pf2 
                        WHERE pf2.field_name = 'name' 
                        AND (LOWER(pf2.field_value) LIKE '%справк%' OR LOWER(pf2.field_value) LIKE '%комисс%')
                    )
                    LIMIT 1
                ");
                $stmtAdmin->execute([$widget['id']]);
                
                $adminItem = [];
                $adminId = null;
                while ($row = $stmtAdmin->fetch()) {
                    if (empty($adminItem)) {
                        $adminId = (int)$row['id'];
                        $adminItem = ['id' => $adminId];
                    }
                    $adminItem[$row['field_name']] = $row['field_value'];
                }
                
                // Проверяем, что элемент имеет нормальную структуру (есть name или title)
                $hasValidName = !empty($adminItem['name']) || !empty($adminItem['title']);
                $hasBlockSpez = !empty($adminItem['block_spez']);
                
                logParser("DEBUG specializations: Found admin specialization: " . (!empty($adminItem) ? "YES (ID: {$adminId})" : "NO") . ", hasValidName=" . ($hasValidName ? 'YES' : 'NO') . ", hasBlockSpez=" . ($hasBlockSpez ? 'YES' : 'NO') . ", in existingIds=" . (in_array($adminId, $existingIds) ? 'YES' : 'NO'));
                
                // Добавляем только если есть валидное имя и нет block_spez, или если есть name/title
                if (!empty($adminItem) && $hasValidName && !$hasBlockSpez && !in_array($adminId, $existingIds)) {
                    $fullData['specializations'][] = $adminItem;
                    $existingIds[] = $adminId; // Обновляем existingIds чтобы избежать дублей
                    logParser("DEBUG specializations: Added 'Справки и комиссии' (ID: {$adminId}) to fullData for admin query '{$question}', total count: " . count($fullData['specializations']));
                } else {
                    logParser("DEBUG specializations: Skipped adding admin specialization: empty=" . (empty($adminItem) ? 'YES' : 'NO') . ", hasValidName=" . ($hasValidName ? 'YES' : 'NO') . ", hasBlockSpez=" . ($hasBlockSpez ? 'YES' : 'NO') . ", in existingIds=" . (in_array($adminId, $existingIds) ? 'YES' : 'NO') . ", adminId=" . ($adminId ?? 'NULL'));
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding admin specialization to fullData: " . $e->getMessage());
            }
        }
        
        // Для онемения/покалывания добавляем кардиология + эндокринология (невролог уже может быть от AI)
        $isNumbnessQuery = preg_match('/(онемен|покалыван|мурашк|жжен|жжени)/ui', $question);
        if ($isNumbnessQuery) {
            logParser("DEBUG specializations: Processing numbness query '{$question}' in fullData block, existingIds: " . implode(',', $existingIds) . ", current count: " . count($fullData['specializations']));
            // Кардиология
            try {
                $db = getDatabase();
                $stmtCardiology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%кардиолог%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtCardiology->execute([$widget['id']]);
                
                $cardiologyItem = [];
                $cardiologyId = null;
                while ($row = $stmtCardiology->fetch()) {
                    if (empty($cardiologyItem)) {
                        $cardiologyId = (int)$row['id'];
                        $cardiologyItem = ['id' => $cardiologyId];
                    }
                    $cardiologyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($cardiologyItem) && !in_array($cardiologyId, $existingIds)) {
                    $fullData['specializations'][] = $cardiologyItem;
                    $existingIds[] = $cardiologyId; // Обновляем existingIds чтобы избежать дублей
                    logParser("DEBUG specializations: Added 'Кардиология' (ID: {$cardiologyId}) to fullData for numbness query '{$question}', total count: " . count($fullData['specializations']));
                } else {
                    logParser("DEBUG specializations: Skipped adding cardiology: empty=" . (empty($cardiologyItem) ? 'YES' : 'NO') . ", in existingIds=" . (in_array($cardiologyId, $existingIds) ? 'YES' : 'NO') . ", cardiologyId=" . ($cardiologyId ?? 'NULL'));
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding cardiology to fullData: " . $e->getMessage());
            }
            
            // Эндокринология
            try {
                $db = getDatabase();
                $stmtEndocrinology = $db->prepare("
                    SELECT pi.id, pf.field_name, pf.field_value
                    FROM parsed_items pi
                    JOIN parsed_fields pf ON pi.id = pf.item_id
                    WHERE pi.widget_id = ? AND pi.section_name = 'specializations'
                    AND (LOWER(pf.field_value) LIKE '%эндокринолог%')
                    AND pi.is_duplicate = 0
                    LIMIT 1
                ");
                $stmtEndocrinology->execute([$widget['id']]);
                
                $endocrinologyItem = [];
                $endocrinologyId = null;
                while ($row = $stmtEndocrinology->fetch()) {
                    if (empty($endocrinologyItem)) {
                        $endocrinologyId = (int)$row['id'];
                        $endocrinologyItem = ['id' => $endocrinologyId];
                    }
                    $endocrinologyItem[$row['field_name']] = $row['field_value'];
                }
                
                if (!empty($endocrinologyItem) && !in_array($endocrinologyId, $existingIds)) {
                    $fullData['specializations'][] = $endocrinologyItem;
                    $existingIds[] = $endocrinologyId; // Обновляем existingIds чтобы избежать дублей
                    logParser("DEBUG specializations: Added 'Эндокринология' (ID: {$endocrinologyId}) to fullData for numbness query '{$question}', total count: " . count($fullData['specializations']));
                } else {
                    logParser("DEBUG specializations: Skipped adding endocrinology: empty=" . (empty($endocrinologyItem) ? 'YES' : 'NO') . ", in existingIds=" . (in_array($endocrinologyId, $existingIds) ? 'YES' : 'NO') . ", endocrinologyId=" . ($endocrinologyId ?? 'NULL'));
                }
            } catch (Exception $e) {
                logParser("DEBUG specializations: ERROR adding endocrinology to fullData: " . $e->getMessage());
            }
        }
        
        // Фильтруем нерелевантные специализации и некорректные элементы
        $filteredSpecializations = [];
        foreach ($fullData['specializations'] as $spec) {
            // Пропускаем некорректные элементы (без name/title или с block_spez)
            if (empty($spec['name']) && empty($spec['title']) && !empty($spec['block_spez'])) {
                logParser("DEBUG specializations filter: Skipping invalid item with ID " . ($spec['id'] ?? 'NULL') . " (has block_spez but no name/title)");
                continue;
            }
            $name = mb_strtolower($spec['name'] ?? $spec['title'] ?? '');
            $shouldInclude = true;
            
            // Исключаем специализацию "Услуги" - это нерелевантно
            if (strpos($name, 'услуг') !== false || $name === 'услуги' || (isset($spec['block_spez']) && !empty($spec['block_spez']))) {
                $shouldInclude = false;
                logParser("DEBUG specializations filter: Excluding '{$name}' from fullData (irrelevant 'Услуги' specialization or block_spez)");
            }
            
            // Для "акне": убрать "Неврология" и "Аллергология"
            if ($isAcneQuery) {
                if (strpos($name, 'невролог') !== false || strpos($name, 'аллерголог') !== false) {
                    $shouldInclude = false;
                    logParser("DEBUG specializations filter: Excluding '{$name}' from fullData for acne query");
                }
            }
            
            // Для "ожог": убрать "Травматология"
            if ($isBurnQuery) {
                if (strpos($name, 'травматолог') !== false) {
                    $shouldInclude = false;
                    logParser("DEBUG specializations filter: Excluding '{$name}' from fullData for burn query");
                }
            }
            
            if ($shouldInclude) {
                $filteredSpecializations[] = $spec;
            }
        }
        
        if (count($filteredSpecializations) !== count($fullData['specializations'])) {
            logParser("DEBUG specializations filter: Filtered " . (count($fullData['specializations']) - count($filteredSpecializations)) . " specializations in fullData (before: " . count($fullData['specializations']) . ", after: " . count($filteredSpecializations) . ")");
            $fullData['specializations'] = $filteredSpecializations;
        }
        logParser("DEBUG specializations: Final count in fullData after all operations: " . count($fullData['specializations']) . " for query '{$question}'");
    }
    
    // ============================================================================
    // ГЕНЕРАЦИЯ ИНФОРМАТИВНЫХ ТЕКСТОВ ДЛЯ ЗАПРОСОВ О СИМПТОМАХ
    // ============================================================================
    // Если запрос содержит слово "симптомы", "признаки" или "проявления", добавляем информативный текст
    if (empty($claudeResponse['text']) || $claudeResponse['text'] === '[]') {
        $informativeText = '';
        
        // Словарь информативных текстов для частых заболеваний
        $symptomTexts = [
            'пневмония' => 'Пневмония — это воспаление легочной ткани. Основные симптомы: высокая температура (38-40°C), кашель с мокротой, одышка, боль в груди при дыхании, слабость. При подозрении на пневмонию необходима консультация терапевта, рентген грудной клетки и анализы крови.',
            
            'скарлатина' => 'Скарлатина — острое инфекционное заболевание, вызываемое стрептококком. Симптомы: высокая температура, ангина (боль в горле, покраснение), мелкая точечная сыпь на теле (появляется на 1-2 день), ярко-красный язык ("малиновый язык"). Требуется консультация педиатра и лечение антибиотиками.',
            
            'хобл' => 'ХОБЛ (хроническая обструктивная болезнь легких) — прогрессирующее заболевание легких. Основные симптомы: хронический кашель, одышка при физической нагрузке, выделение мокроты. Для диагностики необходимы: спирометрия, рентген/КТ легких, консультация пульмонолога или терапевта.',
            
            'хроническая обструктивная' => 'ХОБЛ (хроническая обструктивная болезнь легких) — прогрессирующее заболевание легких. Основные симптомы: хронический кашель, одышка при физической нагрузке, выделение мокроты. Для диагностики необходимы: спирометрия, рентген/КТ легких, консультация пульмонолога или терапевта.',
            
            'остеопороз' => 'Остеопороз — заболевание, при котором кости становятся хрупкими. Часто протекает бессимптомно до первого перелома. Для диагностики и профилактики необходимы: денситометрия (измерение плотности костей), анализы на кальций и витамин D, консультация эндокринолога.',
            
            'подагра' => 'Подагра — заболевание, связанное с нарушением обмена мочевой кислоты. Симптомы приступа: острая боль в суставе (часто большой палец ноги), покраснение, отек, повышение температуры над суставом. Для диагностики нужны: анализ крови на мочевую кислоту, УЗИ суставов, консультация ревматолога.',
            
            'отек квинке' => 'Отек Квинке (ангионевротический отек) — опасное аллергическое состояние с быстрым отеком кожи, слизистых, особенно лица и гортани. При отеке языка, губ, затруднении дыхания НЕМЕДЛЕННО вызывайте скорую помощь! Это жизнеугрожающее состояние.',
            
            'инфаркт' => 'Инфаркт миокарда — острое состояние, требующее НЕМЕДЛЕННОЙ медицинской помощи. Симптомы: сильная давящая боль в груди (более 15 минут), боль отдает в руку, шею, челюсть, одышка, холодный пот, страх смерти. СРОЧНО вызывайте скорую 103!',
            
            'инсульт' => 'Инсульт — острое нарушение мозгового кровообращения. Признаки (тест "УЗП"): У — улыбка асимметричная, З — заторможенная речь, П — поднять руки (одна не поднимается). При любом из этих признаков НЕМЕДЛЕННО вызывайте скорую 103!',
            
            'аппендицит' => 'Аппендицит — воспаление червеобразного отростка. Симптомы: боль в животе (сначала около пупка, затем справа внизу), тошнота, рвота, повышение температуры, напряжение мышц живота. При подозрении на аппендицит необходима срочная консультация хирурга.',
            
            'гастрит' => 'Гастрит — воспаление слизистой оболочки желудка. Симптомы: боль в верхней части живота, изжога, тошнота, отрыжка, чувство переполнения. Для диагностики необходимы: ФГДС (гастроскопия), тест на Helicobacter pylori, консультация гастроэнтеролога.',
            
            'цистит' => 'Цистит — воспаление мочевого пузыря. Симптомы: частые болезненные мочеиспускания, боль внизу живота, иногда кровь в моче. Необходимы: общий анализ мочи, УЗИ мочевого пузыря, консультация уролога или терапевта.',
            
            'артрит' => 'Артрит — воспаление суставов. Симптомы: боль в суставах, отечность, покраснение, утренняя скованность, ограничение движений. Для диагностики нужны: анализы крови (РФ, СОЭ), рентген суставов, УЗИ, консультация ревматолога.',
            
            'мигрень' => 'Мигрень — интенсивная пульсирующая головная боль, часто с одной стороны. Может сопровождаться тошнотой, рвотой, светобоязнью. Длится от 4 до 72 часов. Для диагностики и лечения необходима консультация невролога.',
            
            'лимфоузл' => 'Увеличение лимфоузлов (лимфаденопатия) может быть признаком инфекции, воспаления или других заболеваний. При увеличении лимфоузлов необходимы: консультация терапевта, УЗИ лимфоузлов, общий и биохимический анализ крови. При длительном увеличении (более 2 недель) или быстром росте — обязательная консультация онколога.',
            
            'остеопороз профилактика' => 'Профилактика остеопороза включает: достаточное потребление кальция (1000-1200 мг/день) и витамина D (800-1000 МЕ/день), регулярные физические нагрузки (особенно с весом), отказ от курения и избыточного алкоголя. Для оценки риска рекомендуется денситометрия костей, анализы на кальций и витамин D, консультация эндокринолога или терапевта.',
            
            'мрт сустав' => 'МРТ сустава — высокоточный метод диагностики, позволяющий увидеть связки, сухожилия, хрящи, мениски и другие мягкие ткани. Показания: травмы, боли неясного происхождения, подозрение на разрывы связок или повреждения хрящей. Для назначения МРТ и интерпретации результатов необходима консультация травматолога-ортопеда.',
            
            'судорог' => 'Судороги в ногах (мышечные спазмы) — непроизвольное сокращение мышц, часто возникающее ночью. Причины: дефицит магния и калия, обезвоживание, физические перегрузки, варикоз, неврологические нарушения. Для диагностики необходимы: анализы крови (магний, калий, кальций), консультация невролога или терапевта. При частых судорогах рекомендуется УЗИ сосудов ног.',
            
            // Информативные тексты для проблемных запросов из теста
            'шейно.*воротников' => 'Боль в шейно-воротниковой зоне может быть связана с остеохондрозом, мышечным напряжением, невралгией или травмой. Симптомы: боль в шее и плечах, скованность движений, головная боль. Для диагностики необходимы: рентген или МРТ шейного отдела позвоночника, консультация невролога или травматолога-ортопеда.',
            'орви.*симптом' => 'ОРВИ (острая респираторная вирусная инфекция) — группа вирусных заболеваний верхних дыхательных путей. Симптомы: повышение температуры, насморк, кашель, боль в горле, слабость, головная боль. Лечение: постельный режим, обильное питье, жаропонижающие при температуре выше 38,5°C, симптоматическая терапия. При осложнениях или температуре выше 39°C необходима консультация терапевта или педиатра.',
            'орви.*лечен' => 'ОРВИ (острая респираторная вирусная инфекция) — группа вирусных заболеваний верхних дыхательных путей. Симптомы: повышение температуры, насморк, кашель, боль в горле, слабость, головная боль. Лечение: постельный режим, обильное питье, жаропонижающие при температуре выше 38,5°C, симптоматическая терапия. При осложнениях или температуре выше 39°C необходима консультация терапевта или педиатра.',
            'тремор.*рук' => 'Тремор рук (дрожание) может быть физиологическим (при волнении, усталости) или патологическим (при неврологических, эндокринных заболеваниях, приеме лекарств). Причины: болезнь Паркинсона, эссенциальный тремор, тиреотоксикоз, алкогольная интоксикация, стресс. Для диагностики необходимы: консультация невролога, анализы на гормоны щитовидной железы, МРТ головного мозга.',
            'межпозвон.*грыж' => 'Межпозвоночная грыжа — выпячивание межпозвонкового диска, сдавливающее нервные корешки или спинной мозг. Симптомы: боль в спине, отдающая в ногу или руку, онемение, слабость мышц. Для диагностики необходимы: МРТ позвоночника (наиболее информативно), КТ, рентген. Лечение: консервативное (медикаменты, физиотерапия, ЛФК) или хирургическое. Консультация невролога или травматолога-ортопеда обязательна.',
            'боль.*спин.*после.*сн' => 'Боль в спине после сна может быть связана с неудобной позой, остеохондрозом, мышечным напряжением, грыжей диска. Утренняя скованность характерна для воспалительных заболеваний (анкилозирующий спондилит). Для диагностики необходимы: рентген или МРТ позвоночника, консультация невролога или травматолога. Рекомендуется также проверить матрас и подушку.'
        ];
        
        // Проверяем, содержит ли вопрос ключевые слова о симптомах, профилактике или диагностике
        // ИСПРАВЛЕНО: добавили "что это|что такое" для запросов типа "ХОБЛ что это"
        // ИСПРАВЛЕНО: добавили "судорог|спазм" для запросов о судорогах
        // ИСПРАВЛЕНО: добавили паттерны для проблемных запросов
        if (preg_match('/(симптом|признак|проявлен|как.*проявляется|что.*бывает|что.*это|что.*такое|характерн|профилактик|предотвращ|мрт|кт|узи|рентген|диагностик|увеличен|боль|болит|приступ|судорог|спазм|шейно.*воротников|воротников.*зон|орви|тремор|межпозвон.*грыж|грыж.*позвон|лечен|причин)/ui', $question)) {
            // Ищем заболевание в вопросе
            $questionLower = mb_strtolower($question);
            foreach ($symptomTexts as $disease => $text) {
                // Проверяем, является ли паттерн regex (содержит специальные символы regex)
                $isRegexPattern = (strpos($disease, '.*') !== false || 
                                  strpos($disease, '.+') !== false ||
                                  strpos($disease, '|') !== false ||
                                  preg_match('/[.*+?^${}()|[\]\\\]/', $disease));
                
                if ($isRegexPattern) {
                    // Это regex паттерн - используем preg_match (паттерн уже готов для использования)
                    try {
                        if (preg_match('/' . $disease . '/ui', $questionLower)) {
                            $informativeText = $text;
                            break;
                        }
                    } catch (Exception $e) {
                        // Если паттерн некорректный, пробуем как обычную строку
                        if (strpos($questionLower, $disease) !== false) {
                            $informativeText = $text;
                            break;
                        }
                    }
                } else {
                    // Это обычная строка - используем strpos
                    if (strpos($questionLower, $disease) !== false) {
                        $informativeText = $text;
                        break;
                    }
                }
            }
            
            // Если не нашли специфический текст, даем общую рекомендацию
            if (empty($informativeText) && !empty($fullData['specialists'])) {
                $informativeText = 'Для точной диагностики и определения тактики лечения рекомендуем обратиться к специалисту. Врач проведет осмотр, назначит необходимые анализы и исследования.';
            }
        }
        
        // Добавляем информативный текст к ответу Claude (если есть)
        if (!empty($informativeText)) {
            $claudeResponse['text'] = $informativeText;
        }
    }
    
    // ФИНАЛЬНАЯ ФИЛЬТРАЦИЯ СТАТЕЙ: применяем ко всем статьям перед формированием ответа
    if (!empty($fullData['articles'])) {
        $questionLower = mb_strtolower($question);
        $finalFilteredArticles = [];
        
        // Определяем ключевые слова вопроса для проверки релевантности
        $questionKeywords = [];
        $questionTopics = [];
        
        // Гастроэнтерология
        if (preg_match('/(тошнот|рвот|живот|брюшн|желудок|гастроэнтеролог|изжог|гастрит|диспепси|панкреатит|диаре|запор|метеоризм|газообразован|гореч.*рту|сухост.*рту|боль.*живот|болит.*живот|боли.*в.*животе)/ui', $questionLower)) {
            $questionKeywords[] = 'тошнот|рвот|живот|брюшн|желудок|гастроэнтеролог|изжог|гастрит|диспепси|панкреатит|диаре|запор|метеоризм|газообразован|гореч.*рту|сухост.*рту|брюшн.*полост|кишечник';
            $questionTopics[] = 'gastro';
        }
        
        // Неврология / Головная боль
        if (preg_match('/(головн.*бол|мигрен|головокружен|шейн|невралг|межреберн|грудн.*клетк|грудн.*бол)/ui', $questionLower)) {
            $questionKeywords[] = 'головн.*бол|мигрен|головокружен|шейн|невралг|межреберн|грудн.*клетк|грудн.*бол';
            $questionTopics[] = 'neuro';
        }
        
        // Ортопедия / Колено
        if (preg_match('/(колен|коленн.*сустав|коленн.*бол|ходьб|ходьб.*бол|боль.*при.*ходьб)/ui', $questionLower)) {
            $questionKeywords[] = 'колен|коленн|ходьб';
            $questionTopics[] = 'ortho';
        }
        
        // Кардиология
        if (preg_match('/(сердц|кардиолог|аритми|гипертон|ишеми|стенокард|инфаркт|инсульт)/ui', $questionLower)) {
            $questionKeywords[] = 'сердц|кардиолог|аритми|гипертон|ишеми|стенокард|инфаркт|инсульт';
            $questionTopics[] = 'cardio';
        }
        
        // Пульмонология
        if (preg_match('/(легк|пульмонолог|бронх|хобл|астм|одышк|кашель)/ui', $questionLower)) {
            $questionKeywords[] = 'легк|пульмонолог|бронх|хобл|астм|одышк|кашель';
            $questionTopics[] = 'pulmo';
        }
        
        // Офтальмология
        if (preg_match('/(глаз|офтальмолог|зрен|зрительн)/ui', $questionLower)) {
            $questionKeywords[] = 'глаз|офтальмолог|зрен|зрительн';
            $questionTopics[] = 'ophthalmo';
        }
        
        // Спина / Поясница
        if (preg_match('/(спин|поясниц|позвоночник|люмбаго|прострел)/ui', $questionLower)) {
            $questionKeywords[] = 'спин|поясниц|позвоночник|люмбаго|прострел';
            $questionTopics[] = 'spine';
        }
        
        // Травмы / Переломы
        if (preg_match('/(перелом|травм|ушиб|вывих|растяжен|разрыв.*связ|разрыв.*сухожил|травматолог)/ui', $questionLower)) {
            $questionKeywords[] = 'перелом|травм|ушиб|вывих|растяжен|разрыв.*связ|разрыв.*сухожил|травматолог|ортопед';
            $questionTopics[] = 'trauma';
        }
        
        // Плечо
        if (preg_match('/(плеч|плечев.*сустав|плечев.*бол)/ui', $questionLower)) {
            $questionKeywords[] = 'плеч|плечев.*сустав|плечев.*бол';
            $questionTopics[] = 'shoulder';
        }
        
        // Почки / Нефрология
        if (preg_match('/(почк|нефролог|мочекаменн|камн.*почк|пиелонефрит)/ui', $questionLower)) {
            $questionKeywords[] = 'почк|нефролог|мочекаменн|камн.*почк|пиелонефрит|мочевыводящ';
            $questionTopics[] = 'kidney';
        }
        
        // Нижняя часть живота
        if (preg_match('/(боль.*внизу.*живот|болит.*внизу.*живот|низ.*живот|тазов.*бол)/ui', $questionLower)) {
            $questionKeywords[] = 'внизу.*живот|тазов.*бол|проктолог|уролог|гинеколог';
            $questionTopics[] = 'lower_abdomen';
        }
        
        // Мениск
        if (preg_match('/(мениск|разрыв.*мениск)/ui', $questionLower)) {
            $questionKeywords[] = 'мениск|разрыв.*мениск|артроскоп|коленн.*сустав';
            $questionTopics[] = 'meniscus';
        }
        
        foreach ($fullData['articles'] as $article) {
            $title = mb_strtolower($article['title'] ?? $article['name'] ?? '');
            $text = mb_strtolower($article['text'] ?? '');
            $articleText = $title . ' ' . $text;
            $shouldInclude = true;
            
            // УНИВЕРСАЛЬНАЯ ФИЛЬТРАЦИЯ: исключаем нерелевантные статьи для ВСЕХ запросов
            if (preg_match('/(программ.*лояльност|лояльност.*пациент)/ui', $title) || 
                preg_match('/(критери.*доступност|доступност.*качеств)/ui', $title) ||
                strpos($title, 'целевые критерии') !== false ||
                (strpos($title, 'правил') !== false && strpos($title, 'подготовк') !== false && !preg_match('/(подготовк|подготов.*исследован)/ui', $questionLower)) ||
                (strpos($title, 'беременност') !== false && strpos($title, 'секс') !== false && !preg_match('/(беременност|женск.*здоров|гинеколог)/ui', $questionLower)) ||
                (strpos($title, 'беременност') !== false && strpos($title, 'развитие плода') !== false && !preg_match('/(беременност|женск.*здоров|гинеколог)/ui', $questionLower)) ||
                (strpos($title, 'чесотк') !== false && strpos($title, 'диагностик') !== false && !preg_match('/(чесотк|кож.*заболеван)/ui', $questionLower)) ||
                (strpos($title, 'диагностик') !== false && strpos($title, 'глазн') !== false && strpos($title, 'болезн') !== false && strpos($title, 'дет') !== false && !preg_match('/(глаз|офтальмолог|зрен)/ui', $questionLower)) ||
                (strpos($title, 'диагностик') !== false && strpos($title, 'варикоцеле') !== false && !preg_match('/(варикоцеле|мошонк|яичк)/ui', $questionLower))) {
                $shouldInclude = false;
                logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) for query '{$question}' (loyalty program/accessibility criteria/general rules)");
            }
            
            // СТРОГАЯ ФИЛЬТРАЦИЯ ПО ТЕМАМ: исключаем статьи из других тем
            if ($shouldInclude && !empty($questionTopics)) {
                // Для гастроэнтерологических запросов исключаем статьи о головной боли, колене, сердце
                if (in_array('gastro', $questionTopics)) {
                    if (preg_match('/(головн.*бол|мигрен|головокружен|шейн|невралг|межреберн|грудн.*клетк|грудн.*бол)/ui', $articleText) && 
                        !preg_match('/(тошнот|рвот|живот|брюшн|желудок|гастроэнтеролог|изжог|гастрит|диспепси|панкреатит|диаре|запор|метеоризм|газообразован|гореч.*рту|сухост.*рту|брюшн.*полост|кишечник)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - neuro/headache article for gastro query '{$question}'");
                    }
                    if (preg_match('/(колен|коленн|ходьб)/ui', $articleText) && 
                        !preg_match('/(тошнот|рвот|живот|брюшн|желудок|гастроэнтеролог|изжог|гастрит|диспепси|панкреатит|диаре|запор|метеоризм|газообразован|гореч.*рту|сухост.*рту|брюшн.*полост|кишечник)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - knee/walking article for gastro query '{$question}'");
                    }
                    if (preg_match('/(сердц|кардиолог|аритми|гипертон|ишеми)/ui', $articleText) && 
                        !preg_match('/(тошнот|рвот|живот|брюшн|желудок|гастроэнтеролог|изжог|гастрит|диспепси|панкреатит|диаре|запор|метеоризм|газообразован|гореч.*рту|сухост.*рту|брюшн.*полост|кишечник)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - cardiac article for gastro query '{$question}'");
                    }
                }
                
                // Для неврологических запросов (головная боль, межреберная невралгия) исключаем статьи о колене, ходьбе
                if (in_array('neuro', $questionTopics)) {
                    if (preg_match('/(колен|коленн|ходьб)/ui', $articleText) && 
                        !preg_match('/(головн.*бол|мигрен|головокружен|шейн|невралг|межреберн|грудн.*клетк|грудн.*бол)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - knee/walking article for neuro query '{$question}'");
                    }
                }
                
                // Для ортопедических запросов (колено) исключаем статьи о головной боли, межреберной невралгии, сердце
                if (in_array('ortho', $questionTopics)) {
                    if (preg_match('/(головн.*бол|мигрен|головокружен|шейн|невралг|межреберн|грудн.*клетк|грудн.*бол)/ui', $articleText) && 
                        !preg_match('/(колен|коленн|ходьб)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - neuro/headache article for ortho query '{$question}'");
                    }
                    if (preg_match('/(сердц|кардиолог|аритми|гипертон|ишеми)/ui', $articleText) && 
                        !preg_match('/(колен|коленн|ходьб)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - cardiac article for ortho query '{$question}'");
                    }
                }
                
                // Для кардиологических запросов исключаем статьи о легких/ХОБЛ (если они не упоминают сердце)
                if (in_array('cardio', $questionTopics)) {
                    if (preg_match('/(легк|пульмонолог|бронх|хобл|астм|одышк)/ui', $articleText) && 
                        !preg_match('/(сердц|кардиолог|аритми|гипертон|ишеми|стенокард|инфаркт|инсульт)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - pulmonary article for cardiac query '{$question}'");
                    }
                }
                
                // Для пульмонологических запросов исключаем статьи о сердце (если они не упоминают легкие)
                if (in_array('pulmo', $questionTopics)) {
                    if (preg_match('/(сердц|кардиолог|аритми|гипертон|ишеми)/ui', $articleText) && 
                        !preg_match('/(легк|пульмонолог|бронх|хобл|астм|одышк|кашель)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - cardiac article for pulmonary query '{$question}'");
                    }
                }
                
                // Для травматологических запросов (переломы, травмы) исключаем статьи про хламидии, инфекции
                if (in_array('trauma', $questionTopics)) {
                    if (preg_match('/(хламид|хламидиоз|синдром.*рейтер|инфекционн.*заболеван|полов.*путем)/ui', $articleText) && 
                        !preg_match('/(перелом|травм|ушиб|вывих|растяжен|разрыв.*связ|разрыв.*сухожил|травматолог|ортопед)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - chlamydia/infection article for trauma query '{$question}'");
                    }
                }
                
                // Для запросов про плечо исключаем статьи про колено
                if (in_array('shoulder', $questionTopics)) {
                    if (preg_match('/(колен|коленн|гонартроз|мениск.*колен)/ui', $articleText) && 
                        !preg_match('/(плеч|плечев.*сустав|плечев.*бол)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - knee article for shoulder query '{$question}'");
                    }
                }
                
                // Для запросов про почки исключаем статьи про простату (если они не упоминают почки)
                if (in_array('kidney', $questionTopics)) {
                    if (preg_match('/(простат|простатит|аденом.*простат|мужск.*проблем)/ui', $articleText) && 
                        !preg_match('/(почк|нефролог|мочекаменн|камн.*почк|пиелонефрит|мочевыводящ)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - prostate article for kidney query '{$question}'");
                    }
                }
                
                // Для запросов про боль внизу живота исключаем статьи про желудок (верхняя часть живота)
                if (in_array('lower_abdomen', $questionTopics)) {
                    if (preg_match('/(желудок|гастрит|язв.*желудк|анатоми.*желудк|строен.*желудк)/ui', $articleText) && 
                        !preg_match('/(внизу.*живот|тазов.*бол|проктолог|уролог|гинеколог|мочевой.*пузыр|кишечник.*нижн)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - stomach article for lower abdomen query '{$question}'");
                    }
                }
                
                // Для запросов про мениск исключаем статьи про хламидии
                if (in_array('meniscus', $questionTopics)) {
                    if (preg_match('/(хламид|хламидиоз|синдром.*рейтер|инфекционн.*заболеван|полов.*путем)/ui', $articleText) && 
                        !preg_match('/(мениск|разрыв.*мениск|артроскоп|коленн.*сустав)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - chlamydia article for meniscus query '{$question}'");
                    }
                }
                
                // Для запросов про левое подреберье исключаем статьи про колено
                if (preg_match('/(лев.*подребер|подребер.*лев)/ui', $questionLower)) {
                    if (preg_match('/(колен|коленн|гонартроз|мениск.*колен)/ui', $articleText) && 
                        !preg_match('/(подребер|селезенк|желудок|панкреас|поджелудочн)/ui', $articleText)) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - knee article for left hypochondrium query '{$question}'");
                    }
                }
            }
            
            // Проверка релевантности: статья должна содержать хотя бы одно ключевое слово из вопроса
            if ($shouldInclude && !empty($questionKeywords)) {
                $hasRelevantKeyword = false;
                foreach ($questionKeywords as $keywordPattern) {
                    if (preg_match('/' . $keywordPattern . '/ui', $articleText)) {
                        $hasRelevantKeyword = true;
                        break;
                    }
                }
                
                // Если статья не содержит ключевых слов вопроса, исключаем её (кроме случаев, когда это общая медицинская статья)
                if (!$hasRelevantKeyword) {
                    // Разрешаем общие медицинские статьи только если они действительно общие
                    $isGeneralMedical = preg_match('/(общ.*медицинск|здоров.*образ.*жизн|профилактик|диспансеризац)/ui', $title);
                    if (!$isGeneralMedical) {
                        $shouldInclude = false;
                        logParser("DEBUG articles final filter: Excluding article '{$article['title']}' (ID {$article['id']}) - no relevant keywords for query '{$question}'");
                    }
                }
            }
            
            if ($shouldInclude) {
                $finalFilteredArticles[] = $article;
            }
        }
        
        if (count($finalFilteredArticles) !== count($fullData['articles'])) {
            logParser("DEBUG articles final filter: Filtered " . (count($fullData['articles']) - count($finalFilteredArticles)) . " irrelevant articles before response for query '{$question}'");
            $fullData['articles'] = $finalFilteredArticles;
        }
    }
    
    // Формируем ответ
    $response = [
        'text' => $claudeResponse['text'] ?? '',
        'data' => $fullData,
        'ready' => true,
        'section_name' => $section_name,
        'response_time_ms' => $responseTime
    ];
    
    // Добавляем debug информацию если включен режим отладки
    if ($debugMode) {
        $fullPrompt = $stage3SystemPrompt . "\n\n" . $promptData;
        
        // Подсчитываем общее количество записей в БД для этого виджета (для сравнения)
        $stmtTotal = $db->prepare("SELECT COUNT(DISTINCT id) as total FROM parsed_items WHERE widget_id = ? AND is_duplicate = 0");
        $stmtTotal->execute([$widget['id']]);
        $totalItemsInDb = $stmtTotal->fetch()['total'] ?? 0;
        
        // Подсчитываем количество записей по каждому разделу (для сравнения)
        $stmtBySection = $db->prepare("SELECT section_name, COUNT(DISTINCT id) as cnt FROM parsed_items WHERE widget_id = ? AND is_duplicate = 0 GROUP BY section_name");
        $stmtBySection->execute([$widget['id']]);
        $itemsBySection = [];
        while ($row = $stmtBySection->fetch()) {
            $itemsBySection[$row['section_name']] = (int)$row['cnt'];
        }
        
        $response['debug'] = [
            'section_name' => $section_name,
            'items_count' => $count,
            'items_count_in_db_total' => $totalItemsInDb,
            'items_count_by_section' => $itemsBySection,
            'filter_applied' => "WHERE widget_id = {$widget['id']} AND section_name = '{$section_name}' AND is_duplicate = 0",
            'validated_ids' => $validatedDataIds[$section_name] ?? [],
            'prompt_size' => mb_strlen($promptData),
            'prompt_tokens' => estimateTokens($promptData),
            'full_prompt' => $fullPrompt,
            'system_prompt' => $stage3SystemPrompt,
            'user_prompt' => $promptData,
            'section_model' => $sectionModel ?? null,
            'used_model' => $stage3Model,
            'provider' => $claudeResponse['provider'] ?? 'unknown',
            'provider_model' => $claudeResponse['provider_model'] ?? $stage3Model,
            'claude_raw_response' => $claudeResponse['raw_response'] ?? null,
            'claude_raw_response_text' => $claudeResponse['raw_response_text'] ?? null,
            'yandex_debug' => $stage3DebugInfo,
            'optimization_applied' => $optimizationApplied,
            'optimization_type' => $optimization['optimization_type'] ?? null,
            'prefilter_info' => $GLOBALS['prefilter_debug_info'] ?? null
        ];
    }
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    $errorMessage = $e->getMessage();
    $errorTrace = $e->getTraceAsString();
    
    // Логируем ошибку с дополнительной информацией
    $logContext = [
        'widget_key' => $widgetKey ?? 'unknown',
        'question' => $question ?? 'unknown',
        'section_name' => $section_name ?? 'unknown',
        'error' => $errorMessage,
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ];
    logError("Query Section API error: " . json_encode($logContext, JSON_UNESCAPED_UNICODE) . "\nTrace: " . $errorTrace);
    
    http_response_code(500);
    echo json_encode([
        'error' => 'Сервис временно недоступен: ' . $errorMessage,
        'ready' => false,
        'section_name' => $section_name ?? null,
        'debug' => $debugMode ? [
            'message' => $errorMessage,
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $errorTrace
        ] : null
    ], JSON_UNESCAPED_UNICODE);
}

