<?php
/**
 * API endpoint для выполнения оптимизаций промптов
 * POST /api/optimize-prompt.php
 * Body: { "widget_id": 1, "section_name": "specialists", "question": "У меня болит спина", "optimization_type": "prefilter", "model": "google/gemini-2.5-flash-lite", "save_enabled": false }
 */

ini_set('display_errors', 0);
error_reporting(E_ALL);

// Увеличиваем время выполнения для больших оптимизаций
set_time_limit(600); // 10 минут
ini_set('max_execution_time', 600);

header('Content-Type: application/json');

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/optimization-functions.php';

// Получаем данные запроса
$input = json_decode(file_get_contents('php://input'), true);
$widgetId = (int)($input['widget_id'] ?? 0);
$sectionName = trim($input['section_name'] ?? '');
$question = trim($input['question'] ?? '');
$optimizationType = trim($input['optimization_type'] ?? '');
$model = trim($input['model'] ?? 'google/gemini-2.5-flash-lite');
$saveEnabled = isset($input['save_enabled']) ? (bool)$input['save_enabled'] : false;

if (!$widgetId || !$sectionName || !$optimizationType) {
    http_response_code(400);
    echo json_encode(['error' => 'widget_id, section_name and optimization_type are required']);
    exit;
}

// Для prefilter вопрос не требуется (новая логика с категоризацией)
if ($optimizationType !== 'prefilter' && empty($question)) {
    http_response_code(400);
    echo json_encode(['error' => 'question is required for this optimization type']);
    exit;
}

$validTypes = ['prefilter', 'compress', 'minify', 'shorten', 'group'];
if (!in_array($optimizationType, $validTypes)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid optimization_type. Must be one of: ' . implode(', ', $validTypes)]);
    exit;
}

try {
    $db = getDatabase();
    
    // Проверяем существование виджета
    $stmt = $db->prepare("SELECT id FROM widgets WHERE id = ?");
    $stmt->execute([$widgetId]);
    if (!$stmt->fetch()) {
        http_response_code(404);
        echo json_encode(['error' => 'Widget not found']);
        exit;
    }
    
    // Выполняем оптимизацию в зависимости от типа
    $result = null;
    $startTime = microtime(true);
    
    switch ($optimizationType) {
        case 'prefilter':
            // Для prefilter просто сохраняем настройки (категоризация выполняется через categorize-items.php)
            // Проверяем, есть ли категории для этого раздела
            $db = getDatabase();
            $stmt = $db->prepare("SELECT COUNT(*) as total FROM item_categories WHERE widget_id = ? AND section_name = ?");
            $stmt->execute([$widgetId, $sectionName]);
            $categoriesCount = (int)$stmt->fetch()['total'];
            
            $result = [
                'categories_count' => $categoriesCount,
                'message' => $categoriesCount > 0 
                    ? "Найдено $categoriesCount категорий. Используйте кнопку 'Создать категории' для категоризации элементов."
                    : "Категории не найдены. Используйте кнопку 'Создать категории' для начала категоризации."
            ];
            break;
        case 'compress':
            $result = executeCompressOptimization($widgetId, $sectionName, $question, $model);
            break;
        case 'minify':
            $result = executeMinifyOptimization($widgetId, $sectionName, $question, $model);
            break;
        case 'shorten':
            $result = executeShortenOptimization($widgetId, $sectionName, $question, $model);
            break;
        case 'group':
            $result = executeGroupOptimization($widgetId, $sectionName, $question, $model);
            break;
    }
    
    $executionTime = round((microtime(true) - $startTime) * 1000, 2);
    
    // Сохраняем результат в БД
    $saveId = saveOptimizationResult($widgetId, $sectionName, $optimizationType, $model, $result, $question, $saveEnabled);
    
    // Формируем ответ
    $response = [
        'success' => true,
        'optimization_type' => $optimizationType,
        'section_name' => $sectionName,
        'widget_id' => $widgetId,
        'model' => $model,
        'execution_time_ms' => $executionTime,
        'result' => $result,
        'saved' => true,
        'save_id' => $saveId
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'error' => $e->getMessage(),
        'success' => false
    ], JSON_UNESCAPED_UNICODE);
}

