<?php
/**
 * Функции для оптимизации промптов через AI
 * Все оптимизации выполняются через OpenRouter API
 */

require_once __DIR__ . '/../config.php';

// Временно устанавливаем REQUEST_METHOD в CLI_TEST, чтобы query.php не выполнял основной код
$originalRequestMethod = $_SERVER['REQUEST_METHOD'] ?? null;
$_SERVER['REQUEST_METHOD'] = 'CLI_TEST';
require_once __DIR__ . '/query.php';
// Восстанавливаем оригинальный REQUEST_METHOD
if ($originalRequestMethod !== null) {
    $_SERVER['REQUEST_METHOD'] = $originalRequestMethod;
} else {
    unset($_SERVER['REQUEST_METHOD']);
}

/**
 * Загружает данные раздела из БД
 */
function loadSectionData($widgetId, $sectionName) {
    $db = getDatabase();
    
    // Получаем поля для этого раздела
    $stmt = $db->prepare("
        SELECT DISTINCT sf.field_name
        FROM widget_sections ws
        JOIN section_fields sf ON ws.id = sf.section_id
        WHERE ws.widget_id = ? AND ws.section_name = ? AND sf.use_in_prompt = 1
        
        UNION
        
        SELECT DISTINCT scf.child_field_name as field_name
        FROM widget_sections ws
        JOIN section_fields sf ON ws.id = sf.section_id
        JOIN section_child_fields scf ON sf.id = scf.field_id
        WHERE ws.widget_id = ? AND ws.section_name = ? AND scf.use_in_prompt = 1
    ");
    $stmt->execute([$widgetId, $sectionName, $widgetId, $sectionName]);
    $fields = [];
    while ($row = $stmt->fetch()) {
        $fields[] = $row['field_name'];
    }
    
    if (empty($fields)) {
        return [];
    }
    
    // Загружаем данные раздела
    $stmt = $db->prepare("
        SELECT pi.id, pf.field_name, pf.field_value
        FROM parsed_items pi
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.widget_id = ? AND pi.section_name = ? 
        AND pf.field_name IN (" . str_repeat('?,', count($fields) - 1) . "?)
        AND pi.is_duplicate = 0
        ORDER BY pi.parent_item_id, pi.id
    ");
    $params = array_merge([$widgetId, $sectionName], $fields);
    $stmt->execute($params);
    
    $items = [];
    while ($row = $stmt->fetch()) {
        if (!isset($items[$row['id']])) {
            $items[$row['id']] = ['id' => (int)$row['id']];
        }
        $items[$row['id']][$row['field_name']] = $row['field_value'];
    }
    
    return array_values($items);
}

/**
 * Выполняет оптимизацию через AI (общая функция)
 */
function executeOptimizationViaAI($systemPrompt, $userPrompt, $model = 'google/gemini-2.5-flash-lite') {
    try {
        // Если GPU включен, используем его для категоризации
        if (defined('GPU_LLM_ENABLED') && GPU_LLM_ENABLED) {
            logParser("=== OPTIMIZATION REQUEST TO GPU SERVER ===");
            logParser("URL: " . GPU_LLM_API_URL);
            logParser("Model: " . GPU_LLM_MODEL);
            
            $data = [
                'model' => GPU_LLM_MODEL,
                'max_tokens' => MAX_TOKENS_RESPONSE,
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => $systemPrompt
                    ],
                    [
                        'role' => 'user',
                        'content' => $userPrompt
                    ]
                ],
                'temperature' => 0.7
            ];
            
            $ch = curl_init(GPU_LLM_API_URL);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_TIMEOUT, GPU_LLM_TIMEOUT);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json'
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            
            if (curl_errno($ch)) {
                $error = curl_error($ch);
                curl_close($ch);
                throw new Exception('GPU LLM API error: ' . $error);
            }
            
            curl_close($ch);
            
            if ($httpCode !== 200) {
                throw new Exception("GPU LLM API returned HTTP $httpCode: " . substr($response, 0, 500));
            }
            
            $result = json_decode($response, true);
            
            if (!isset($result['choices'][0]['message']['content'])) {
                throw new Exception('Invalid GPU LLM API response structure');
            }
            
            $text = trim($result['choices'][0]['message']['content']);
            
            // Извлекаем JSON из markdown code block (если есть)
            $text = extractJSONFromMarkdown($text);
            
            // Очищаем текст от лишних символов
            $text = preg_replace('/[\x00-\x1F\x7F]/u', '', $text);
            $text = trim($text);
            
            return [
                'text' => $text,
                'raw_response' => $response,
                'provider' => 'gpu',
                'tokens_used' => ($result['usage']['prompt_tokens'] ?? 0) + ($result['usage']['completion_tokens'] ?? 0)
            ];
        }
        
        // Fallback: используем прямой вызов OpenRouter API для оптимизаций
        $fullSystemPrompt = $systemPrompt;
        
        logParser("=== OPTIMIZATION REQUEST TO OPENROUTER ===");
        logParser("URL: " . OPENROUTER_API_URL);
        logParser("Model: " . $model);
        
        $data = [
            'model' => $model,
            'max_tokens' => MAX_TOKENS_RESPONSE,
            'messages' => [
                [
                    'role' => 'system',
                    'content' => $fullSystemPrompt
                ],
                [
                    'role' => 'user',
                    'content' => $userPrompt
                ]
            ],
            'temperature' => 0.7
        ];
        
        $ch = curl_init(OPENROUTER_API_URL);
        configureOpenRouterProxy($ch); // Настройка прокси для OpenRouter
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_TIMEOUT, 30); // Таймаут 30 секунд для категоризации
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10); // Таймаут подключения 10 секунд
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . OPENROUTER_API_KEY,
            'HTTP-Referer: ' . WIDGET_DOMAIN,
            'X-Title: AI Widget Optimization'
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if (curl_errno($ch)) {
            $error = curl_error($ch);
            curl_close($ch);
            throw new Exception('OpenRouter API error: ' . $error);
        }
        
        curl_close($ch);
        
        if ($httpCode !== 200) {
            throw new Exception("OpenRouter API returned HTTP $httpCode: " . substr($response, 0, 500));
        }
        
        $result = json_decode($response, true);
        
        if (!isset($result['choices'][0]['message']['content'])) {
            throw new Exception('Invalid OpenRouter API response structure');
        }
        
        $text = $result['choices'][0]['message']['content'];
        
        return [
            'success' => true,
            'text' => $text,
            'raw_response' => $response
        ];
    } catch (Exception $e) {
        throw new Exception('AI optimization error: ' . $e->getMessage());
    }
}

/**
 * Определяет категории из вопроса пользователя через AI
 * @param string $question Вопрос пользователя
 * @param string $model Модель AI для использования
 * @param bool $returnDebugInfo Если true, возвращает массив с категориями и debug информацией
 * @return array|array Массив категорий или массив с категориями и debug информацией
 */
/**
 * Получить категории из кеша или определить через AI
 */
function getCachedCategories($widgetId, $question, $model, $returnDebugInfo = false) {
    $cacheKey = 'categories_' . md5($widgetId . '_' . $question . '_' . $model);
    $cacheFile = '/root/cache/' . $cacheKey . '.json';
    $cacheTTL = 3600; // 1 час
    
    // Проверяем кеш
    if (file_exists($cacheFile)) {
        $cacheData = json_decode(file_get_contents($cacheFile), true);
        if ($cacheData && isset($cacheData['timestamp']) && isset($cacheData['categories'])) {
            $age = time() - $cacheData['timestamp'];
            if ($age < $cacheTTL) {
                // Кеш валиден
                $result = [
                    'categories' => $cacheData['categories'],
                    'from_cache' => true
                ];
                if ($returnDebugInfo && isset($cacheData['debug'])) {
                    $result['debug'] = $cacheData['debug'];
                }
                return $result;
            }
        }
    }
    
    // Кеш не найден или устарел, определяем через AI
    $result = extractCategoriesFromQuestion($question, $model, $returnDebugInfo);
    
    // Сохраняем в кеш
    $cacheData = [
        'timestamp' => time(),
        'categories' => $result['categories'],
        'model' => $model
    ];
    if ($returnDebugInfo && isset($result['debug'])) {
        $cacheData['debug'] = $result['debug'];
    }
    
    // Создаем директорию кеша, если её нет
    if (!is_dir('/root/cache')) {
        mkdir('/root/cache', 0755, true);
    }
    
    // Сохраняем в кеш с правильными правами доступа
    $writeResult = file_put_contents($cacheFile, json_encode($cacheData, JSON_UNESCAPED_UNICODE));
    if ($writeResult !== false) {
        chmod($cacheFile, 0644); // Права доступа для чтения всем, запись владельцу
    }
    
    $result['from_cache'] = false;
    return $result;
}

function extractCategoriesFromQuestion($question, $model = 'google/gemini-2.5-flash-lite', $returnDebugInfo = false) {
    $systemPrompt = "Ты - медицинский эксперт. Твоя задача - определить все релевантные категории из вопроса пользователя для поиска медицинских услуг, специалистов или статей.

Определяй категории в следующих типах:
1. СИМПТОМЫ: конкретные симптомы (тошнота, головная боль, боль в спине, кашель, изжога, острая боль, хроническая боль)
2. ЧАСТИ ТЕЛА: органы и части тела (желудок, спина, голова, нос, горло, живот, позвоночник, суставы)
3. СПЕЦИАЛЬНОСТИ: медицинские специальности (гастроэнтерология, неврология, хирургия, оториноларингология, кардиология, дерматология, травматология, ортопедия и т.д.)
4. ЗАБОЛЕВАНИЯ: названия болезней (гастрит, язва, гайморит, остеохондроз, радикулит, грыжа)
5. ТИПЫ УСЛУГ: консультация, диагностика, лечение, процедура, массаж, физиотерапия, блокада, операция
6. МЕТОДЫ: рентген, УЗИ, МРТ, КТ, эндоскопия, лабораторные анализы
7. АДМИНИСТРАТИВНЫЕ ЗАПРОСЫ: справки, комиссии, медосмотры, документы

ВАЖНО:
- Для симптомов пищеварения (тошнота, рвота, изжога, отрыжка, боль в животе, диспепсия) ВСЕГДА добавляй: \"гастроэнтерология\", \"гастроэнтеролог\", \"желудок\", \"пищеварение\", \"жкт\"
- Для симптомов головы (головная боль, головокружение) ВСЕГДА добавляй: \"неврология\", \"невролог\"
- Для симптомов спины (боль в спине, острая боль в спине, остеохондроз) ВСЕГДА добавляй: \"неврология\", \"травматология\", \"ортопедия\", \"позвоночник\", \"массаж\", \"физиотерапия\", \"блокада\" (если боль острая)
- Для симптомов носа/горла (насморк, боль в горле, кашель) ВСЕГДА добавляй: \"оториноларингология\", \"лор\", \"терапевт\", \"педиатр\", \"терапия\", \"педиатрия\"
- Для проблем с кожей (акне, псориаз, розацеа, зуд кожи, экзема, дерматит, крапивница, угри, прыщи, сыпь) ВСЕГДА добавляй: \"дерматология\", \"дерматолог\", \"кожа\"
- Для симптомов сердца (боль в сердце, аритмия) ВСЕГДА добавляй: \"кардиология\", \"кардиолог\"
- Для ОСТРОЙ боли ВСЕГДА добавляй: \"блокада\", \"обезболивание\", \"экстренная помощь\", \"консультация\"
- Для ХРОНИЧЕСКОЙ боли ВСЕГДА добавляй: \"лечение\", \"физиотерапия\", \"массаж\", \"реабилитация\"
- Для АДМИНИСТРАТИВНЫХ ЗАПРОСОВ (транспортная комиссия, водительская справка, справка в бассейн, справка на оружие, медосмотр, диспансеризация, справка для работы, справка в школу/детский сад) ВСЕГДА добавляй: \"справка\", \"комиссия\", \"медосмотр\", \"осмотр\", название конкретной комиссии/справки
- Используй как оригинальные формулировки из вопроса, так и их медицинские эквиваленты
- Добавляй синонимы и связанные термины
- Учитывай контекст: для \"острая боль\" добавляй категории, связанные с экстренной помощью и обезболиванием

ПРИМЕРЫ:
- \"Транспортная комиссия\" → [\"транспортная комиссия\", \"водительская комиссия\", \"справка\", \"комиссия\", \"медосмотр\", \"осмотр\", \"водитель\", \"права\"]
- \"Справка в бассейн\" → [\"справка в бассейн\", \"справка\", \"бассейн\", \"осмотр\", \"медосмотр\", \"терапевт\", \"дерматолог\"]
- \"Острая боль в спине\" → [\"острая боль\", \"боль в спине\", \"спина\", \"позвоночник\", \"неврология\", \"невролог\", \"травматология\", \"травматолог\", \"ортопедия\", \"ортопед\", \"блокада\", \"обезболивание\", \"консультация\", \"диагностика\", \"массаж\", \"физиотерапия\"]

Верни JSON в формате: {\"categories\": [\"категория1\", \"категория2\", ...]}.
Используй короткие, конкретные категории.";

    $userPrompt = "Вопрос пользователя: $question\n\nОпредели ВСЕ релевантные категории из этого вопроса для поиска медицинских услуг. Верни JSON с массивом категорий, включая симптомы, части тела, специальности и связанные термины.";

    $response = executeOptimizationViaAI($systemPrompt, $userPrompt, $model);
    
    // Парсим ответ AI
    $text = $response['text'] ?? '';
    $rawResponse = $response['raw_response'] ?? '';
    
    // Удаляем markdown обертку если есть
    $text = preg_replace('/^```json\s*|\s*```$/s', '', $text);
    $text = preg_replace('/[\x00-\x1F\x7F]/u', '', $text);
    $parsed = json_decode($text, true);
    
    if (!$parsed || !isset($parsed['categories']) || !is_array($parsed['categories'])) {
        // Пробуем найти массив категорий напрямую в тексте
        if (preg_match('/\[.*?\]/', $text, $matches)) {
            $categories = json_decode($matches[0], true);
            if (is_array($categories)) {
                if ($returnDebugInfo) {
                    return [
                        'categories' => $categories,
                        'debug' => [
                            'system_prompt' => $systemPrompt,
                            'user_prompt' => $userPrompt,
                            'full_prompt' => $systemPrompt . "\n\n" . $userPrompt,
                            'ai_response' => $text,
                            'raw_response' => $rawResponse,
                            'model' => $model
                        ]
                    ];
                }
                return $categories;
            }
        }
        throw new Exception('Failed to parse AI response for categories. Response: ' . substr($text, 0, 200));
    }
    
    if ($returnDebugInfo) {
        return [
            'categories' => $parsed['categories'],
            'debug' => [
                'system_prompt' => $systemPrompt,
                'user_prompt' => $userPrompt,
                'full_prompt' => $systemPrompt . "\n\n" . $userPrompt,
                'ai_response' => $text,
                'raw_response' => $rawResponse,
                'model' => $model
            ]
        ];
    }
    
    return $parsed['categories'];
}

/**
 * Определяет категории для одного элемента данных через AI
 * @param array $item Элемент данных (с полями id, description, name и т.д.)
 * @param string $sectionName Название раздела
 * @param string $model Модель AI для использования
 * @return array Массив категорий
 */
function categorizeItem($item, $sectionName, $model = 'google/gemini-2.5-flash-lite') {
    // Формируем описание элемента
    $description = '';
    if (isset($item['description'])) {
        $description = $item['description'];
    } elseif (isset($item['name'])) {
        $description = $item['name'];
    } elseif (isset($item['title'])) {
        $description = $item['title'];
    }
    
    $itemData = json_encode($item, JSON_UNESCAPED_UNICODE);
    
    $systemPrompt = "Ты должен проанализировать описание специалиста/услуги/статьи и определить категории, к которым он относится. 
Категории могут быть: симптомы (болит спина, головная боль), части тела (нос, горло, спина), специальности (неврология, хирургия), заболевания.
Верни JSON в формате: {\"categories\": [\"категория1\", \"категория2\", ...]}.
Используй короткие, конкретные категории. Например: \"болит спина\", \"нос\", \"неврология\".
Один элемент может иметь несколько категорий.";

    $userPrompt = "Описание элемента:\n$description\n\nПолные данные элемента:\n$itemData\n\nОпредели категории для этого элемента и верни JSON с массивом категорий.";

    $response = executeOptimizationViaAI($systemPrompt, $userPrompt, $model);
    
    // Парсим ответ AI
    $text = $response['text'] ?? '';
    
    // Удаляем markdown обертку если есть
    $text = preg_replace('/^```json\s*|\s*```$/s', '', $text);
    $text = preg_replace('/[\x00-\x1F\x7F]/u', '', $text);
    $parsed = json_decode($text, true);
    
    if (!$parsed || !isset($parsed['categories']) || !is_array($parsed['categories'])) {
        // Пробуем найти массив категорий напрямую в тексте
        if (preg_match('/\[.*?\]/', $text, $matches)) {
            $categories = json_decode($matches[0], true);
            if (is_array($categories)) {
                return $categories;
            }
        }
        // Если не удалось распарсить, возвращаем пустой массив
        return [];
    }
    
    return $parsed['categories'];
}

/**
 * Сохраняет категории элемента в базу данных
 * @param int $widgetId ID виджета
 * @param string $sectionName Название раздела
 * @param int $itemId ID элемента
 * @param array $categories Массив категорий
 */
function saveItemCategories($widgetId, $sectionName, $itemId, $categories) {
    $db = getDatabase();
    
    // Удаляем старые категории для этого элемента
    $stmt = $db->prepare("DELETE FROM item_categories WHERE widget_id = ? AND section_name = ? AND item_id = ?");
    $stmt->execute([$widgetId, $sectionName, $itemId]);
    
    // Сохраняем новые категории
    $stmt = $db->prepare("INSERT INTO item_categories (widget_id, section_name, item_id, category) VALUES (?, ?, ?, ?)");
    foreach ($categories as $category) {
        $category = trim($category);
        if (!empty($category)) {
            try {
                $stmt->execute([$widgetId, $sectionName, $itemId, $category]);
            } catch (PDOException $e) {
                // Игнорируем ошибки дубликатов (UNIQUE constraint)
                if (strpos($e->getMessage(), 'Duplicate entry') === false) {
                    throw $e;
                }
            }
        }
    }
}

/**
 * Вариант 1: Предварительная фильтрация (новая версия с категоризацией)
 * Категоризирует элементы данных (вызывается для пакетной обработки)
 * @param int $widgetId ID виджета
 * @param string $sectionName Название раздела
 * @param array $items Массив элементов для категоризации
 * @param string $model Модель AI для использования
 * @return array Статистика категоризации
 */
function executePrefilterOptimization($widgetId, $sectionName, $items = null, $model = 'google/gemini-2.5-flash-lite') {
    if ($items === null) {
        $items = loadSectionData($widgetId, $sectionName);
    }
    
    if (empty($items)) {
        throw new Exception("No data found for section '$sectionName'");
    }
    
    $totalCategories = 0;
    $processed = 0;
    
    foreach ($items as $item) {
        if (!isset($item['id'])) {
            continue;
        }
        
        $categories = categorizeItem($item, $sectionName, $model);
        if (!empty($categories)) {
            saveItemCategories($widgetId, $sectionName, $item['id'], $categories);
            $totalCategories += count($categories);
        }
        $processed++;
    }
    
    return [
        'processed' => $processed,
        'total' => count($items),
        'categories_created' => $totalCategories,
        'items_count' => count($items)
    ];
}

/**
 * Вариант 2: Сжатие описаний
 * AI сжимает описания, оставляя только ключевые специальности
 */
function executeCompressOptimization($widgetId, $sectionName, $question, $model = 'google/gemini-2.5-flash-lite') {
    $items = loadSectionData($widgetId, $sectionName);
    
    if (empty($items)) {
        throw new Exception("No data found for section '$sectionName'");
    }
    
    $totalItems = count($items);
    $batchSize = 50; // Обрабатываем по 50 записей за раз для больших разделов
    $compressedItems = [];
    $allErrors = [];
    
    // Если записей меньше 100, обрабатываем все сразу
    if ($totalItems <= 100) {
        $jsonData = json_encode([$sectionName => $items], JSON_UNESCAPED_UNICODE);
        
        $systemPrompt = "Ты должен сжать описания записей, оставив только ключевые специальности. 
Убери лишнее: 'высшей категории', 'кандидат наук', названия клиник, должности, лишние слова.
Верни JSON в том же формате, но со сжатыми описаниями. Сохрани все поля записей (id, name, description, title и т.д.), измени только description/name/title (в зависимости от того, какое поле есть в записи).";
        
        $userPrompt = "Вопрос пользователя: $question\n\nКаталог данных:\n$jsonData\n\nВерни тот же JSON, но со сжатыми описаниями. Сохрани все поля и структуру.";
        
        try {
            $response = executeOptimizationViaAI($systemPrompt, $userPrompt, $model);
            
            // Парсим ответ AI
            $text = $response['text'] ?? '';
            $rawText = $text; // Сохраняем оригинальный текст для ошибок
            $text = preg_replace('/^```json\s*|\s*```$/s', '', $text);
            $text = trim($text);
            
            // Пробуем найти JSON в тексте
            $parsed = json_decode($text, true);
            
            if (!$parsed || !isset($parsed[$sectionName]) || !is_array($parsed[$sectionName])) {
                // Пробуем найти JSON объект в тексте
                if (preg_match('/\{[^{]*"?' . preg_quote($sectionName, '/') . '"?\s*:\s*\[.*?\]\s*\}/s', $text, $matches)) {
                    $parsed = json_decode($matches[0], true);
                }
                
                if (!$parsed || !isset($parsed[$sectionName]) || !is_array($parsed[$sectionName])) {
                    $errorMsg = 'Failed to parse AI response for compression. ';
                    $errorMsg .= 'Response length: ' . strlen($text) . ' chars. ';
                    $errorMsg .= 'JSON error: ' . json_last_error_msg() . '. ';
                    $errorMsg .= 'First 1000 chars: ' . substr($text, 0, 1000);
                    error_log("Compress optimization error: " . $errorMsg);
                    error_log("Raw AI response (first 2000 chars): " . substr($rawText, 0, 2000));
                    throw new Exception($errorMsg);
                }
            }
            
            $compressedItems = $parsed[$sectionName];
        } catch (Exception $e) {
            throw new Exception('Compression failed: ' . $e->getMessage());
        }
    } else {
        // Для больших разделов обрабатываем батчами
        $batches = array_chunk($items, $batchSize, true); // true сохраняет ключи массива
        $totalBatches = count($batches);
        
        $systemPrompt = "Ты должен сжать описания записей, оставив только ключевые специальности. 
Убери лишнее: 'высшей категории', 'кандидат наук', названия клиник, должности, лишние слова.
Верни JSON в том же формате, но со сжатыми описаниями. Сохрани все поля записей (id, name, description, title и т.д.), измени только description/name/title (в зависимости от того, какое поле есть в записи).";
        
        foreach ($batches as $batchNum => $batch) {
            try {
                $jsonData = json_encode([$sectionName => $batch], JSON_UNESCAPED_UNICODE);
                
                $userPrompt = "Вопрос пользователя: $question\n\nКаталог данных (батч " . ($batchNum + 1) . " из $totalBatches):\n$jsonData\n\nВерни тот же JSON, но со сжатыми описаниями. Сохрани все поля и структуру.";
                
                $response = executeOptimizationViaAI($systemPrompt, $userPrompt, $model);
                
                // Парсим ответ AI
                $text = $response['text'] ?? '';
                $rawText = $text; // Сохраняем оригинальный текст для ошибок
                $text = preg_replace('/^```json\s*|\s*```$/s', '', $text);
                $text = trim($text);
                
                $parsed = json_decode($text, true);
                
                if (!$parsed || !isset($parsed[$sectionName]) || !is_array($parsed[$sectionName])) {
                    // Пробуем найти JSON в тексте
                    if (preg_match('/\{[^{]*"?' . preg_quote($sectionName, '/') . '"?\s*:\s*\[.*?\]\s*\}/s', $text, $matches)) {
                        $parsed = json_decode($matches[0], true);
                    }
                    
                    if (!$parsed || !isset($parsed[$sectionName]) || !is_array($parsed[$sectionName])) {
                        $errorMsg = 'Failed to parse AI response for compression (batch ' . ($batchNum + 1) . '). ';
                        $errorMsg .= 'Response length: ' . strlen($text) . ' chars. ';
                        $errorMsg .= 'JSON error: ' . json_last_error_msg() . '. ';
                        $errorMsg .= 'First 500 chars: ' . substr($text, 0, 500);
                        $allErrors[] = $errorMsg;
                        error_log("Compress optimization error (batch " . ($batchNum + 1) . "): " . $errorMsg);
                        error_log("Raw AI response (first 1000 chars): " . substr($rawText, 0, 1000));
                        continue; // Пропускаем этот батч
                    }
                }
                
                // Объединяем результаты
                foreach ($parsed[$sectionName] as $item) {
                    $compressedItems[] = $item;
                }
            } catch (Exception $e) {
                $allErrors[] = "Batch " . ($batchNum + 1) . " error: " . $e->getMessage();
                error_log("Compress optimization batch " . ($batchNum + 1) . " exception: " . $e->getMessage());
                continue; // Продолжаем обработку следующих батчей
            }
        }
        
        if (empty($compressedItems)) {
            $errorDetails = !empty($allErrors) ? implode('; ', array_slice($allErrors, 0, 3)) : 'No batches processed successfully';
            throw new Exception('Failed to compress any items. Errors: ' . $errorDetails);
        }
        
        // Если обработано меньше элементов, чем было, добавляем оригинальные для недостающих
        if (count($compressedItems) < $totalItems) {
            $processedIds = [];
            foreach ($compressedItems as $item) {
                if (isset($item['id'])) {
                    $processedIds[$item['id']] = true;
                }
            }
            
            // Добавляем оригинальные элементы для тех, что не были обработаны
            foreach ($items as $item) {
                if (isset($item['id']) && !isset($processedIds[$item['id']])) {
                    $compressedItems[] = $item;
                }
            }
        }
    }
    
    // Вычисляем токены для каждого описания
    $originalDescTokens = 0;
    $compressedDescTokens = 0;
    
    foreach ($items as $idx => $item) {
        $desc = $item['description'] ?? $item['name'] ?? $item['title'] ?? '';
        $originalDescTokens += estimateTokens($desc);
    }
    
    foreach ($compressedItems as $item) {
        $compDesc = $item['description'] ?? $item['name'] ?? $item['title'] ?? '';
        $compressedDescTokens += estimateTokens($compDesc);
    }
    
    $result = [
        'original_items' => $items,
        'compressed_items' => $compressedItems,
        'original_count' => $totalItems,
        'optimized_count' => count($compressedItems),
        'original_tokens' => estimateTokens(json_encode([$sectionName => $items], JSON_UNESCAPED_UNICODE)),
        'optimized_tokens' => estimateTokens(json_encode([$sectionName => $compressedItems], JSON_UNESCAPED_UNICODE)),
        'original_desc_tokens' => $originalDescTokens,
        'compressed_desc_tokens' => $compressedDescTokens
    ];
    
    // Добавляем информацию о батчах, если они использовались
    if ($totalItems > 100) {
        $result['batches_processed'] = isset($totalBatches) ? $totalBatches : 1;
        $result['batches_successful'] = isset($totalBatches) ? ($totalBatches - count($allErrors)) : 1;
        if (!empty($allErrors)) {
            $result['warnings'] = array_slice($allErrors, 0, 5); // Первые 5 ошибок
        }
    }
    
    return $result;
}

/**
 * Вариант 3: Минификация JSON
 * AI минифицирует JSON структуру, используя короткие ключи или массивы
 */
function executeMinifyOptimization($widgetId, $sectionName, $question, $model = 'google/gemini-2.5-flash-lite') {
    $items = loadSectionData($widgetId, $sectionName);
    
    if (empty($items)) {
        throw new Exception("No data found for section '$sectionName'");
    }
    
    $totalItems = count($items);
    $jsonData = json_encode([$sectionName => $items], JSON_UNESCAPED_UNICODE);
    
    $systemPrompt = "Ты должен минифицировать JSON структуру, используя короткие ключи.
Используй следующие сокращения:
- \"id\" → \"i\"
- \"description\" → \"d\"
- \"name\" → \"n\"
- \"title\" → \"t\"
- \"specialists\" → \"s\"
- \"services\" → \"sv\"
- \"articles\" → \"a\"
- \"specializations\" → \"sp\"
Верни минифицированный JSON в том же формате (объект с ключом раздела и массивом элементов).";
    
    // Для больших разделов (>100 записей) обрабатываем батчами
    if ($totalItems > 100) {
        $batchSize = 50;
        $batches = array_chunk($items, $batchSize, true); // true сохраняет ключи массива
        $totalBatches = count($batches);
        $minifiedItems = [];
        $allErrors = [];
        $rawResponses = [];
        
        foreach ($batches as $batchNum => $batch) {
            try {
                $batchJson = json_encode([$sectionName => $batch], JSON_UNESCAPED_UNICODE);
                $userPrompt = "Вопрос пользователя: $question\n\nОригинальный JSON (батч " . ($batchNum + 1) . " из $totalBatches):\n$batchJson\n\nВерни минифицированный JSON в том же формате.";
                
                $response = executeOptimizationViaAI($systemPrompt, $userPrompt, $model);
                
                // Парсим ответ AI
                $text = $response['text'] ?? '';
                $rawText = $text;
                $rawResponses[] = "Batch " . ($batchNum + 1) . ": " . substr($text, 0, 500);
                
                $text = preg_replace('/^```json\s*|\s*```$/s', '', $text);
                $text = trim($text);
                
                // Пробуем найти JSON в тексте
                $parsed = json_decode($text, true);
                
                if (!$parsed || !isset($parsed[$sectionName]) || !is_array($parsed[$sectionName])) {
                    // Пробуем найти JSON объект в тексте
                    if (preg_match('/\{[^{]*"?' . preg_quote($sectionName, '/') . '"?\s*:\s*\[.*?\]\s*\}/s', $text, $matches)) {
                        $parsed = json_decode($matches[0], true);
                    }
                    
                    if (!$parsed || !isset($parsed[$sectionName]) || !is_array($parsed[$sectionName])) {
                        $errorMsg = 'Failed to parse AI response for minification (batch ' . ($batchNum + 1) . '). ';
                        $errorMsg .= 'Response length: ' . strlen($text) . ' chars. ';
                        $errorMsg .= 'JSON error: ' . json_last_error_msg() . '. ';
                        $errorMsg .= 'First 500 chars: ' . substr($text, 0, 500);
                        $allErrors[] = $errorMsg;
                        error_log("Minify optimization error (batch " . ($batchNum + 1) . "): " . $errorMsg);
                        error_log("Raw AI response (first 1000 chars): " . substr($rawText, 0, 1000));
                        continue; // Пропускаем этот батч
                    }
                }
                
                // Объединяем результаты
                foreach ($parsed[$sectionName] as $item) {
                    $minifiedItems[] = $item;
                }
            } catch (Exception $e) {
                $allErrors[] = "Batch " . ($batchNum + 1) . " error: " . $e->getMessage();
                error_log("Minify optimization batch " . ($batchNum + 1) . " exception: " . $e->getMessage());
                continue; // Продолжаем обработку следующих батчей
            }
        }
        
        if (empty($minifiedItems)) {
            $errorDetails = !empty($allErrors) ? implode('; ', array_slice($allErrors, 0, 3)) : 'No batches processed successfully';
            throw new Exception('Failed to minify any items. Errors: ' . $errorDetails);
        }
        
        // Если обработано меньше элементов, добавляем оригинальные для недостающих
        if (count($minifiedItems) < $totalItems) {
            $processedIds = [];
            foreach ($minifiedItems as $item) {
                if (isset($item['i']) || isset($item['id'])) {
                    $processedIds[($item['i'] ?? $item['id'])] = true;
                }
            }
            
            // Добавляем оригинальные элементы для тех, что не были обработаны
            foreach ($items as $item) {
                $itemId = $item['id'] ?? null;
                if ($itemId && !isset($processedIds[$itemId])) {
                    $minifiedItems[] = $item; // Оставляем оригинальный формат
                }
            }
        }
        
        $minifiedJson = json_encode([$sectionName => $minifiedItems], JSON_UNESCAPED_UNICODE);
        
        $result = [
            'original_json' => substr($jsonData, 0, 1000) . '... (truncated, ' . strlen($jsonData) . ' chars total)',
            'minified_json' => substr($minifiedJson, 0, 1000) . '... (truncated, ' . strlen($minifiedJson) . ' chars total)',
            'original_count' => $totalItems,
            'optimized_count' => count($minifiedItems),
            'original_tokens' => estimateTokens($jsonData),
            'optimized_tokens' => estimateTokens($minifiedJson),
            'batches_processed' => $totalBatches,
            'batches_successful' => $totalBatches - count($allErrors),
            'ai_response' => implode("\n\n", $rawResponses)
        ];
        
        if (!empty($allErrors)) {
            $result['warnings'] = array_slice($allErrors, 0, 5); // Первые 5 ошибок
        }
        
        return $result;
    } else {
        // Для небольших разделов обрабатываем все сразу
        $userPrompt = "Вопрос пользователя: $question\n\nОригинальный JSON:\n$jsonData\n\nВерни минифицированный JSON в том же формате.";
        
        $response = executeOptimizationViaAI($systemPrompt, $userPrompt, $model);
        
        // Парсим ответ AI
        $text = $response['text'] ?? '';
        $rawText = $text;
        $text = preg_replace('/^```json\s*|\s*```$/s', '', $text);
        $text = trim($text);
        
        $parsed = json_decode($text, true);
        
        if (!$parsed || !isset($parsed[$sectionName]) || !is_array($parsed[$sectionName])) {
            // Пробуем найти JSON объект в тексте
            if (preg_match('/\{[^{]*"?' . preg_quote($sectionName, '/') . '"?\s*:\s*\[.*?\]\s*\}/s', $text, $matches)) {
                $parsed = json_decode($matches[0], true);
            }
            
            if (!$parsed || !isset($parsed[$sectionName]) || !is_array($parsed[$sectionName])) {
                $errorMsg = 'Failed to parse AI response for minification. ';
                $errorMsg .= 'Response length: ' . strlen($text) . ' chars. ';
                $errorMsg .= 'JSON error: ' . json_last_error_msg() . '. ';
                $errorMsg .= 'First 1000 chars: ' . substr($text, 0, 1000);
                error_log("Minify optimization error: " . $errorMsg);
                error_log("Raw AI response (first 2000 chars): " . substr($rawText, 0, 2000));
                throw new Exception($errorMsg);
            }
        }
        
        $minifiedJson = json_encode($parsed, JSON_UNESCAPED_UNICODE);
        
        return [
            'original_json' => $jsonData,
            'minified_json' => $minifiedJson,
            'original_count' => count($items),
            'optimized_count' => count($parsed[$sectionName]),
            'original_tokens' => estimateTokens($jsonData),
            'optimized_tokens' => estimateTokens($minifiedJson),
            'ai_response' => $text
        ];
    }
}

/**
 * Вариант 4: Сокращение инструкций
 * AI сокращает промпт, убирая повторения и лишние акценты
 */
function executeShortenOptimization($widgetId, $sectionName, $question, $model = 'google/gemini-2.5-flash-lite') {
    $db = getDatabase();
    
    // Получаем промпт для раздела
    $stmt = $db->prepare("
        SELECT w.*, ws.*
        FROM widgets w
        LEFT JOIN widget_settings ws ON w.id = ws.widget_id
        WHERE w.id = ?
    ");
    $stmt->execute([$widgetId]);
    $widget = $stmt->fetch();
    
    if (!$widget) {
        throw new Exception("Widget not found");
    }
    
    // Выбираем промпт для раздела
    $sectionPromptField = 'stage_' . $sectionName . '_prompt';
    $sectionPrompt = !empty($widget[$sectionPromptField]) ? $widget[$sectionPromptField] : null;
    
    if ($sectionPrompt) {
        $originalPrompt = $sectionPrompt;
    } elseif (!empty($widget['stage3_prompt'])) {
        $originalPrompt = $widget['stage3_prompt'];
    } else {
        $originalPrompt = $widget['claude_prompt'];
    }
    
    if (empty($originalPrompt)) {
        throw new Exception("No prompt found for section '$sectionName'");
    }
    
    $systemPrompt = "Ты должен сократить промпт, убрав повторения, лишние акценты ('ВАЖНО', 'КРИТИЧЕСКИ ВАЖНО'), объединив похожие инструкции.
Сохрани смысл и важные требования, но сделай промпт более кратким и структурированным.
Верни только сокращенный промпт без дополнительных комментариев.";
    
    $userPrompt = "Вопрос пользователя: $question\n\nОригинальный промпт:\n$originalPrompt\n\nВерни сокращенный промпт.";
    
    $response = executeOptimizationViaAI($systemPrompt, $userPrompt, $model);
    
    $shortenedPrompt = trim($response['text'] ?? '');
    
    if (empty($shortenedPrompt)) {
        throw new Exception('Failed to get shortened prompt from AI');
    }
    
    return [
        'original_prompt' => $originalPrompt,
        'shortened_prompt' => $shortenedPrompt,
        'original_tokens' => estimateTokens($originalPrompt),
        'optimized_tokens' => estimateTokens($shortenedPrompt),
        'ai_response' => $shortenedPrompt
    ];
}

/**
 * Вариант 5: Группировка по категориям
 * AI группирует записи по категориям (специальностям)
 */
function executeGroupOptimization($widgetId, $sectionName, $question, $model = 'google/gemini-2.5-flash-lite') {
    $items = loadSectionData($widgetId, $sectionName);
    
    if (empty($items)) {
        throw new Exception("No data found for section '$sectionName'");
    }
    
    $jsonData = json_encode([$sectionName => $items], JSON_UNESCAPED_UNICODE);
    
    $systemPrompt = "Ты должен сгруппировать записи по категориям (специальностям) на основе вопроса пользователя.
Верни JSON где ключи - названия категорий, значения - массивы ID записей.
Формат: {\"категория1\": [id1, id2, ...], \"категория2\": [id3, id4, ...]}.
Группируй только релевантные категории к вопросу пользователя.";
    
    $userPrompt = "Вопрос пользователя: $question\n\nКаталог данных:\n$jsonData\n\nВерни сгруппированные данные по категориям.";
    
    $response = executeOptimizationViaAI($systemPrompt, $userPrompt, $model);
    
    // Парсим ответ AI
    $text = $response['text'] ?? '';
    $text = preg_replace('/^```json\s*|\s*```$/s', '', $text);
    $text = preg_replace('/[\x00-\x1F\x7F]/u', '', $text);
    $parsed = json_decode($text, true);
    
    if (!$parsed || !is_array($parsed)) {
        throw new Exception('Failed to parse AI response for grouping');
    }
    
    // Подсчитываем общее количество ID в группах
    $totalGroupedIds = 0;
    foreach ($parsed as $category => $ids) {
        if (is_array($ids)) {
            $totalGroupedIds += count($ids);
        }
    }
    
    $groupedJson = json_encode($parsed, JSON_UNESCAPED_UNICODE);
    
    return [
        'original_items' => $items,
        'grouped_data' => $parsed,
        'original_count' => count($items),
        'optimized_count' => $totalGroupedIds,
        'original_tokens' => estimateTokens($jsonData),
        'optimized_tokens' => estimateTokens($groupedJson),
        'ai_response' => $text
    ];
}

/**
 * Сохраняет результат оптимизации в БД
 */
function saveOptimizationResult($widgetId, $sectionName, $optimizationType, $model, $result, $testQuestion, $isEnabled = false) {
    $db = getDatabase();
    
    $optimizedData = json_encode($result, JSON_UNESCAPED_UNICODE);
    
    $stmt = $db->prepare("
        INSERT INTO widget_optimizations 
        (widget_id, section_name, optimization_type, model, optimized_data, original_count, optimized_count, original_tokens, optimized_tokens, test_question, is_enabled)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE
            model = VALUES(model),
            optimized_data = VALUES(optimized_data),
            original_count = VALUES(original_count),
            optimized_count = VALUES(optimized_count),
            original_tokens = VALUES(original_tokens),
            optimized_tokens = VALUES(optimized_tokens),
            test_question = VALUES(test_question),
            is_enabled = VALUES(is_enabled),
            updated_at = CURRENT_TIMESTAMP
    ");
    
    $stmt->execute([
        $widgetId,
        $sectionName,
        $optimizationType,
        $model,
        $optimizedData,
        $result['original_count'] ?? 0,
        $result['optimized_count'] ?? 0,
        $result['original_tokens'] ?? 0,
        $result['optimized_tokens'] ?? 0,
        $testQuestion,
        $isEnabled ? 1 : 0
    ]);
    
    return $db->lastInsertId();
}

