<?php
/**
 * Функции для компрессии данных перед отправкой в GPU
 * Используется OpenRouter для предварительной обработки БЕЗ вопроса пользователя
 */

require_once __DIR__ . '/../config.php';

// Загружаем функцию configureOpenRouterProxy если прокси включен
if (defined('OPENROUTER_PROXY_ENABLED') && OPENROUTER_PROXY_ENABLED) {
    if (!function_exists('configureOpenRouterProxy')) {
        // Определяем функцию прямо здесь
        function configureOpenRouterProxy($ch) {
            if (defined('OPENROUTER_PROXY_ENABLED') && OPENROUTER_PROXY_ENABLED) {
                curl_setopt($ch, CURLOPT_PROXY, OPENROUTER_PROXY_HOST . ':' . OPENROUTER_PROXY_PORT);
                curl_setopt($ch, CURLOPT_PROXYTYPE, CURLPROXY_HTTP); // Для HTTPS прокси используется HTTP тип
                curl_setopt($ch, CURLOPT_PROXYUSERPWD, OPENROUTER_PROXY_USER . ':' . OPENROUTER_PROXY_PASS);
                curl_setopt($ch, CURLOPT_HTTPPROXYTUNNEL, 1); // Туннелирование для HTTPS
            }
        }
    }
}

/**
 * Создать компактное описание элемента через OpenRouter
 * Извлекает только ключевые слова БЕЗ контекста вопроса
 * 
 * @param array $item Данные элемента (специалист/услуга/статья)
 * @param string $sectionName Тип раздела
 * @return array ['keywords' => '...', 'compact' => '...']
 */
function compressItemData($item, $sectionName) {
    // Формируем текст для сжатия
    $fullText = json_encode($item, JSON_UNESCAPED_UNICODE);
    
    // Системный промпт для извлечения ключевых слов
    $systemPrompt = "Ты - эксперт по медицинским данным. Твоя задача: извлечь ТОЛЬКО ключевые медицинские термины и понятия из предоставленного текста.

ВАЖНО:
- НЕ добавляй свои интерпретации
- НЕ пиши полные предложения  
- Только ключевые слова через запятую
- Максимум 15-20 ключевых слов
- Фокус на: симптомы, диагнозы, процедуры, специализации, органы/системы

Формат ответа:
keywords: <список ключевых слов через запятую>
compact: <очень краткое описание в 1 предложение>";

    $userPrompt = "Извлеки ключевые термины из этих данных раздела '$sectionName':\n\n" . substr($fullText, 0, 2000);
    
    $data = [
        'model' => 'google/gemini-2.0-flash-exp:free', // Быстрая бесплатная модель
        'max_tokens' => 200,
        'messages' => [
            ['role' => 'system', 'content' => $systemPrompt],
            ['role' => 'user', 'content' => $userPrompt]
        ],
        'temperature' => 0.3
    ];
    
    try {
        $ch = curl_init(OPENROUTER_API_URL);
        if (defined('OPENROUTER_PROXY_ENABLED') && OPENROUTER_PROXY_ENABLED) {
            configureOpenRouterProxy($ch);
        }
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . OPENROUTER_API_KEY,
            'HTTP-Referer: ' . (defined('WIDGET_DOMAIN') ? WIDGET_DOMAIN : 'https://wtest.medmaps.ru'),
            'X-Title: AI Widget Compressor'
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            throw new Exception("OpenRouter returned HTTP $httpCode");
        }
        
        $result = json_decode($response, true);
        $content = $result['choices'][0]['message']['content'] ?? '';
        
        // Парсим ответ
        $keywords = '';
        $compact = '';
        
        if (preg_match('/keywords:\s*(.+)/im', $content, $matches)) {
            $keywords = trim($matches[1]);
        }
        if (preg_match('/compact:\s*(.+)/im', $content, $matches)) {
            $compact = trim($matches[1]);
        }
        
        // Если не удалось распарсить, берем весь текст как keywords
        if (empty($keywords)) {
            $keywords = substr($content, 0, 500);
        }
        
        return [
            'keywords' => $keywords,
            'compact' => $compact ?: substr($content, 0, 200)
        ];
        
    } catch (Exception $e) {
        logParser("ERROR: Failed to compress item data: " . $e->getMessage());
        
        // Fallback: простое извлечение ключевых полей
        $keywords = '';
        if (isset($item['title'])) $keywords .= $item['title'] . ', ';
        if (isset($item['specialization'])) $keywords .= $item['specialization'] . ', ';
        if (isset($item['description'])) $keywords .= substr($item['description'], 0, 100);
        
        return [
            'keywords' => rtrim($keywords, ', '),
            'compact' => $keywords
        ];
    }
}

/**
 * Обработать все элементы раздела для виджета
 * Создает компактные описания для всех элементов
 * 
 * @param int $widgetId ID виджета
 * @param string $sectionName Название раздела
 * @param int $limit Максимальное количество элементов для обработки (0 = все)
 * @return array Статистика обработки
 */
function processWidgetSectionData($widgetId, $sectionName, $limit = 0) {
    $db = getDatabase();
    
    // Получаем элементы без компактных описаний
    $sql = "
        SELECT DISTINCT pi.id
        FROM parsed_items pi
        LEFT JOIN item_compact_data icd ON pi.id = icd.item_id
        WHERE pi.widget_id = ? 
        AND pi.section_name = ?
        AND pi.is_duplicate = 0
        AND icd.id IS NULL
    ";
    
    if ($limit > 0) {
        $sql .= " LIMIT ?";
        $stmt = $db->prepare($sql);
        $stmt->execute([$widgetId, $sectionName, $limit]);
    } else {
        $stmt = $db->prepare($sql);
        $stmt->execute([$widgetId, $sectionName]);
    }
    
    $itemIds = [];
    while ($row = $stmt->fetch()) {
        $itemIds[] = $row['id'];
    }
    
    if (empty($itemIds)) {
        return ['processed' => 0, 'message' => 'No items to process'];
    }
    
    $processed = 0;
    $failed = 0;
    
    foreach ($itemIds as $itemId) {
        // Получаем все поля элемента
        $stmt = $db->prepare("
            SELECT pf.field_name, pf.field_value
            FROM parsed_fields pf
            WHERE pf.item_id = ?
        ");
        $stmt->execute([$itemId]);
        
        $item = ['id' => $itemId];
        while ($row = $stmt->fetch()) {
            $item[$row['field_name']] = $row['field_value'];
        }
        
        // Создаем компактное описание
        $compressed = compressItemData($item, $sectionName);
        
        if (!empty($compressed['keywords'])) {
            // Сохраняем в БД
            $stmt = $db->prepare("
                INSERT INTO item_compact_data 
                (item_id, widget_id, section_name, compact_description, keywords)
                VALUES (?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    compact_description = VALUES(compact_description),
                    keywords = VALUES(keywords),
                    updated_at = CURRENT_TIMESTAMP
            ");
            
            $stmt->execute([
                $itemId,
                $widgetId,
                $sectionName,
                $compressed['compact'],
                $compressed['keywords']
            ]);
            
            $processed++;
            
            // Небольшая задержка чтобы не перегрузить API
            usleep(100000); // 0.1 сек
        } else {
            $failed++;
        }
        
        // Логируем прогресс каждые 10 элементов
        if ($processed % 10 === 0) {
            logParser("Processed $processed items for section '$sectionName'...");
        }
    }
    
    logParser("Finished processing section '$sectionName': $processed processed, $failed failed");
    
    return [
        'processed' => $processed,
        'failed' => $failed,
        'total' => count($itemIds)
    ];
}

/**
 * Получить элементы с компактными описаниями для фильтрации
 * 
 * @param int $widgetId ID виджета
 * @param string $sectionName Название раздела
 * @return array Массив элементов с embeddings и компактными описаниями
 */
function getCompactItemsForFiltering($widgetId, $sectionName) {
    $db = getDatabase();
    
    $stmt = $db->prepare("
        SELECT 
            pi.id,
            icd.keywords,
            icd.compact_description,
            ie.embedding
        FROM parsed_items pi
        LEFT JOIN item_compact_data icd ON pi.id = icd.item_id
        LEFT JOIN item_embeddings ie ON pi.id = ie.item_id
        WHERE pi.widget_id = ? 
        AND pi.section_name = ?
        AND pi.is_duplicate = 0
    ");
    
    $stmt->execute([$widgetId, $sectionName]);
    
    $items = [];
    while ($row = $stmt->fetch()) {
        $items[] = [
            'id' => (int)$row['id'],
            'keywords' => $row['keywords'],
            'compact' => $row['compact_description'],
            'embedding' => $row['embedding'] ? json_decode($row['embedding'], true) : null
        ];
    }
    
    return $items;
}

/**
 * Фильтровать элементы по embedding similarity
 * Использует ТОЛЬКО embeddings, БЕЗ отправки вопроса в OpenRouter
 * 
 * @param string $question Вопрос пользователя
 * @param array $items Массив элементов с embeddings
 * @param int $topN Количество элементов для возврата
 * @return array Топ N элементов с наивысшей схожестью
 */
function filterItemsByEmbedding($question, $items, $topN = 50) {
    require_once __DIR__ . '/embedding-functions.php';
    
    // Создаем embedding для вопроса
    $questionEmbedding = createEmbedding($question);
    
    if (!$questionEmbedding) {
        logParser("WARNING: Failed to create question embedding, returning all items");
        return array_slice($items, 0, $topN);
    }
    
    // Считаем similarity для каждого элемента
    $itemsWithScores = [];
    
    foreach ($items as $item) {
        if (empty($item['embedding'])) {
            continue; // Пропускаем элементы без embedding
        }
        
        $similarity = cosineSimilarity($questionEmbedding, $item['embedding']);
        $itemsWithScores[] = [
            'id' => $item['id'],
            'keywords' => $item['keywords'],
            'compact' => $item['compact'],
            'similarity' => $similarity
        ];
    }
    
    // Сортируем по убыванию similarity
    usort($itemsWithScores, function($a, $b) {
        return $b['similarity'] <=> $a['similarity'];
    });
    
    // Возвращаем топ N
    $topItems = array_slice($itemsWithScores, 0, $topN);
    
    if (!empty($topItems)) {
        $topSimilarities = array_map(function($item) {
            return round($item['similarity'] * 100, 1) . '%';
        }, array_slice($topItems, 0, 5));
        
        logParser("Embedding filter: selected top $topN items (top similarities: " . implode(', ', $topSimilarities) . ")");
    }
    
    // Возвращаем только ID
    return array_map(function($item) {
        return $item['id'];
    }, $topItems);
}

/**
 * Вычислить косинусное сходство между двумя векторами
 */
function cosineSimilarity($vec1, $vec2) {
    $dotProduct = 0;
    $magnitude1 = 0;
    $magnitude2 = 0;
    
    $count = min(count($vec1), count($vec2));
    
    for ($i = 0; $i < $count; $i++) {
        $dotProduct += $vec1[$i] * $vec2[$i];
        $magnitude1 += $vec1[$i] * $vec1[$i];
        $magnitude2 += $vec2[$i] * $vec2[$i];
    }
    
    $magnitude1 = sqrt($magnitude1);
    $magnitude2 = sqrt($magnitude2);
    
    if ($magnitude1 == 0 || $magnitude2 == 0) {
        return 0;
    }
    
    return $dotProduct / ($magnitude1 * $magnitude2);
}

