<?php
/**
 * API endpoint для получения конфигурации виджета
 * GET /api/config.php?widget_key=clinic-001
 */

require_once __DIR__ . '/../config.php';

header('Content-Type: application/json');

// CORS headers
$origin = $_SERVER['HTTP_ORIGIN'] ?? '';
$widget_key = $_GET['widget_key'] ?? '';

if (!$widget_key) {
    http_response_code(400);
    echo json_encode(['error' => 'widget_key is required']);
    exit;
}

try {
    $db = getDatabase();
    
    // Получаем виджет и настройки
    $stmt = $db->prepare("
        SELECT w.*, ws.*
        FROM widgets w
        LEFT JOIN widget_settings ws ON w.id = ws.widget_id
        WHERE w.widget_key = ? AND w.active = 1
    ");
    $stmt->execute([$widget_key]);
    $widget = $stmt->fetch();
    
    if (!$widget) {
        http_response_code(404);
        echo json_encode(['error' => 'Widget not found']);
        exit;
    }
    
    // Проверяем CORS
    $allowed_domains = json_decode($widget['allowed_domains'] ?? '[]', true);
    
    // Определяем домен запроса (из origin или из Referer)
    $request_host = '';
    if ($origin) {
        $request_host = parse_url($origin, PHP_URL_HOST);
    } elseif (isset($_SERVER['HTTP_REFERER'])) {
        $request_host = parse_url($_SERVER['HTTP_REFERER'], PHP_URL_HOST);
    } elseif (isset($_SERVER['HTTP_HOST'])) {
        $request_host = $_SERVER['HTTP_HOST'];
    }
    
    // Проверяем, является ли запрос с того же домена (включая админку)
    $own_domain = parse_url(WIDGET_DOMAIN, PHP_URL_HOST);
    $is_own_domain = false;
    if ($request_host) {
        // Проверяем совпадение домена или поддомена
        $is_own_domain = (
            $request_host === $own_domain ||
            strpos($request_host, $own_domain) !== false ||
            strpos($own_domain, $request_host) !== false ||
            // Проверяем общий корневой домен (medmaps.ru)
            (preg_match('/\.medmaps\.ru$/', $request_host) && preg_match('/\.medmaps\.ru$/', $own_domain))
        );
    }
    
    // Разрешаем доступ если:
    // 1. Origin пустой (запрос с того же домена или прямой доступ)
    // 2. Origin = наш домен (w2.medmaps.ru или w.medmaps.ru)
    // 3. Origin в списке разрешенных доменов
    // 4. Запрос с того же домена (включая админку)
    $is_allowed = empty($origin) || $is_own_domain || in_array($origin, $allowed_domains);
    
    if (!empty($allowed_domains) && !$is_allowed && !$is_own_domain) {
        http_response_code(403);
        echo json_encode(['error' => 'Domain not allowed']);
        exit;
    }
    
    // Устанавливаем CORS headers
    if ($origin) {
        if ($is_own_domain || in_array($origin, $allowed_domains)) {
            header("Access-Control-Allow-Origin: $origin");
            header("Access-Control-Allow-Methods: GET, POST");
            header("Access-Control-Allow-Headers: Content-Type");
        }
    }
    
    // Проверяем наличие кастомных файлов
    $customPath = WIDGET_ROOT . '/widget/custom/' . $widget['widget_key'];
    $hasCustomTemplate = file_exists($customPath . '/template.html');
    $hasCustomStyles = file_exists($customPath . '/styles.css');
    $hasCustomJS = file_exists($customPath . '/custom.js');
    
    // Генерируем версию на основе времени изменения файла (для сброса кеша браузера)
    // Используем время изменения файла шаблона, если он существует
    $templateFile = $hasCustomTemplate ? $customPath . '/template.html' : WIDGET_ROOT . '/widget/template.html';
    $fileMtime = file_exists($templateFile) ? filemtime($templateFile) : time();
    $version = '?v=' . $fileMtime;
    
    // Возвращаем конфигурацию
    $config = [
        'widget_key' => $widget['widget_key'],
        'name' => $widget['name'],
        'primary_color' => $widget['primary_color'] ?? '#007bff',
        'welcome_message' => $widget['welcome_message'] ?? 'Привет! Я ваш личный AI-помощник',
        'suggestions' => json_decode($widget['suggestions'] ?? '[]', true),
        'custom_files' => [
            'template' => $hasCustomTemplate ? "/widget/custom/{$widget['widget_key']}/template.html{$version}" : "/widget/template.html{$version}",
            'styles' => $hasCustomStyles ? "/widget/custom/{$widget['widget_key']}/styles.css{$version}" : "/widget/widget.css{$version}",
            'js' => $hasCustomJS ? "/widget/custom/{$widget['widget_key']}/custom.js{$version}" : null
        ]
    ];
    
    echo json_encode($config);
    
} catch (Exception $e) {
    logError("Config API error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error']);
}

