#!/bin/bash

# Скрипт создания бэкапа проекта и базы данных
# Использование: ./backup.sh

set -e

# Конфигурация
PROJECT_ROOT="/root"
BACKUP_DIR="/root/ai-widget/backups"
TIMESTAMP=$(date +%Y%m%d_%H%M%S)
BACKUP_NAME="ai-widget-backup-${TIMESTAMP}"

# Данные БД
DB_HOST="localhost"
DB_NAME="ai_widget"
DB_USER="telegram_bot"
DB_PASS="bot_password_2025"

# Создаем директорию для бэкапов
mkdir -p "${BACKUP_DIR}"

echo "[$(date '+%Y-%m-%d %H:%M:%S')] Начало создания бэкапа..."

# 1. Создаем дамп базы данных
echo "[$(date '+%Y-%m-%d %H:%M:%S')] Создание дампа базы данных..."
DB_DUMP="${BACKUP_DIR}/${BACKUP_NAME}.sql"
mysqldump -h"${DB_HOST}" -u"${DB_USER}" -p"${DB_PASS}" \
    --single-transaction \
    --routines \
    --triggers \
    --events \
    "${DB_NAME}" > "${DB_DUMP}"

if [ $? -eq 0 ]; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] Дамп БД создан: ${DB_DUMP}"
else
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] ОШИБКА: Не удалось создать дамп БД!"
    exit 1
fi

# 2. Создаем архив проекта (исключаем ненужные папки)
echo "[$(date '+%Y-%m-%d %H:%M:%S')] Создание архива проекта..."
cd "${PROJECT_ROOT}"

# Собираем список тестовых PHP файлов
TEST_FILES=$(find . -maxdepth 1 -name "test-*.php" -type f 2>/dev/null | sed 's|^\./||')
TEST_COUNT=$(echo "${TEST_FILES}" | grep -c . || echo "0")

# Исключаем из бэкапа:
# - vendor/ (можно восстановить через composer install)
# - cache/ (временные файлы)
# - ai-widget/backups/ (чтобы не бэкапить бэкапы)
# - node_modules/ (если есть)
# - .git/ (если есть)
# Включаем все PHP тестовые файлы из /root
if [ "${TEST_COUNT}" -gt 0 ]; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] Найдено тестовых PHP файлов: ${TEST_COUNT}"
    # Создаем временный файл со списком файлов для включения
    FILE_LIST=$(mktemp)
    echo "api/" >> "${FILE_LIST}"
    echo "admin/" >> "${FILE_LIST}"
    echo "config.php" >> "${FILE_LIST}"
    echo "ai-widget/" >> "${FILE_LIST}"
    echo "${TEST_FILES}" >> "${FILE_LIST}"
    
    tar -czf "${BACKUP_DIR}/${BACKUP_NAME}.tar.gz" \
        --exclude='vendor' \
        --exclude='cache' \
        --exclude='ai-widget/backups' \
        --exclude='node_modules' \
        --exclude='.git' \
        --exclude='*.log' \
        --exclude='.env' \
        --exclude='test-*.json' \
        --exclude='test-*.txt' \
        --exclude='*.tmp' \
        --files-from="${FILE_LIST}"
    
    rm -f "${FILE_LIST}"
else
    tar -czf "${BACKUP_DIR}/${BACKUP_NAME}.tar.gz" \
        --exclude='vendor' \
        --exclude='cache' \
        --exclude='ai-widget/backups' \
        --exclude='node_modules' \
        --exclude='.git' \
        --exclude='*.log' \
        --exclude='.env' \
        --exclude='test-*.json' \
        --exclude='test-*.txt' \
        --exclude='*.tmp' \
        api/ admin/ config.php ai-widget/
fi

if [ $? -eq 0 ]; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] Архив проекта создан: ${BACKUP_DIR}/${BACKUP_NAME}.tar.gz"
else
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] ОШИБКА: Не удалось создать архив проекта!"
    exit 1
fi

# 3. Создаем общий архив (проект + БД)
echo "[$(date '+%Y-%m-%d %H:%M:%S')] Создание общего архива..."
cd "${BACKUP_DIR}"
tar -czf "${BACKUP_NAME}-full.tar.gz" \
    "${BACKUP_NAME}.tar.gz" \
    "${BACKUP_NAME}.sql"

if [ $? -eq 0 ]; then
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] Общий архив создан: ${BACKUP_DIR}/${BACKUP_NAME}-full.tar.gz"
    
    # Удаляем промежуточные файлы
    rm -f "${BACKUP_NAME}.tar.gz" "${BACKUP_NAME}.sql"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] Промежуточные файлы удалены"
else
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] ОШИБКА: Не удалось создать общий архив!"
    exit 1
fi

# Выводим информацию о размере
FULL_BACKUP="${BACKUP_DIR}/${BACKUP_NAME}-full.tar.gz"
BACKUP_SIZE=$(du -h "${FULL_BACKUP}" | cut -f1)
echo "[$(date '+%Y-%m-%d %H:%M:%S')] Бэкап успешно создан!"
echo "[$(date '+%Y-%m-%d %H:%M:%S')] Файл: ${FULL_BACKUP}"
echo "[$(date '+%Y-%m-%d %H:%M:%S')] Размер: ${BACKUP_SIZE}"

exit 0

