<?php
$page_title = 'Тестер виджета';
require_once __DIR__ . '/includes/header.php';

// Получаем список активных виджетов
$db = getDatabase();
$widgets = $db->query("
    SELECT id, name, widget_key, site_domain
    FROM widgets
    WHERE active = 1
    ORDER BY name ASC
")->fetchAll();

// Предустановленные фразы для тестирования
$defaultPhrases = [
    'У меня болит голова',
    'Нужен хороший кардиолог',
    'Где можно сделать МРТ?',
    'Болит спина после тренировки',
    'Нужна консультация невролога',
    'Где принимает эндокринолог?',
    'Болит зуб, нужен стоматолог',
    'Нужна консультация гинеколога',
    'Где можно сдать анализы крови?',
    'Болит горло и температура',
    'Нужен хороший офтальмолог',
    'Где делают УЗИ брюшной полости?',
    'Болит колено при ходьбе',
    'Нужна консультация дерматолога',
    'Где можно сделать рентген?',
    'Болит живот после еды',
    'Нужен хороший уролог',
    'Где принимает психолог?',
    'Болит ухо, нужен ЛОР',
    'Нужна консультация гастроэнтеролога'
];
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
    <h1 class="h2">Тестер виджета</h1>
</div>

<!-- Статистика (скрыта до начала тестирования) -->
<div id="stats-section" class="row mb-4" style="display: none;">
    <div class="col-md-3">
        <div class="card stat-card">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="text-muted mb-1">Всего запросов</h6>
                        <h3 class="mb-0" id="stat-total">0</h3>
                    </div>
                    <div class="text-primary">
                        <i class="bi bi-list-check" style="font-size: 2rem;"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-md-3">
        <div class="card stat-card">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="text-muted mb-1">Успешных</h6>
                        <h3 class="mb-0 text-success" id="stat-success">0</h3>
                    </div>
                    <div class="text-success">
                        <i class="bi bi-check-circle" style="font-size: 2rem;"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-md-3">
        <div class="card stat-card">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="text-muted mb-1">Ошибок</h6>
                        <h3 class="mb-0 text-danger" id="stat-errors">0</h3>
                    </div>
                    <div class="text-danger">
                        <i class="bi bi-x-circle" style="font-size: 2rem;"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-md-3">
        <div class="card stat-card">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="text-muted mb-1">Среднее время</h6>
                        <h3 class="mb-0 text-info" id="stat-avg-time">0 мс</h3>
                    </div>
                    <div class="text-info">
                        <i class="bi bi-clock" style="font-size: 2rem;"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Распределение по провайдерам -->
<div id="providers-section" class="row mb-4" style="display: none;">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h6 class="mb-0">Распределение по провайдерам</h6>
            </div>
            <div class="card-body">
                <div id="providers-list" class="d-flex flex-wrap gap-2"></div>
            </div>
        </div>
    </div>
</div>

<!-- Настройки тестирования -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">Настройки тестирования</h5>
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6 mb-3">
                <label for="widget-select" class="form-label">Выберите виджет</label>
                <select class="form-select" id="widget-select">
                    <option value="">-- Выберите виджет --</option>
                    <?php foreach ($widgets as $widget): ?>
                        <option value="<?php echo htmlspecialchars($widget['widget_key']); ?>" 
                                data-id="<?php echo $widget['id']; ?>">
                            <?php echo htmlspecialchars($widget['name']); ?> 
                            (<?php echo htmlspecialchars($widget['site_domain']); ?>)
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
        </div>
        
        <div class="mb-3">
            <label for="phrases-input" class="form-label">Тестовые фразы (по одной на строку)</label>
            <textarea class="form-control" id="phrases-input" rows="10" style="font-family: monospace;"><?php echo implode("\n", $defaultPhrases); ?></textarea>
            <small class="text-muted">Каждая строка будет отправлена как отдельный запрос к виджету</small>
        </div>
        
        <div class="d-flex gap-2">
            <button type="button" class="btn btn-primary" id="btn-run-test" disabled>
                <i class="bi bi-play-circle"></i> Запустить тестирование
            </button>
            <button type="button" class="btn btn-secondary" id="btn-stop-test" disabled style="display: none;">
                <i class="bi bi-stop-circle"></i> Остановить
            </button>
            <button type="button" class="btn btn-outline-secondary" id="btn-reset">
                <i class="bi bi-arrow-counterclockwise"></i> Сбросить
            </button>
        </div>
        
        <div id="progress-section" class="mt-3" style="display: none;">
            <div class="progress mb-2">
                <div class="progress-bar progress-bar-striped progress-bar-animated" 
                     role="progressbar" 
                     id="progress-bar" 
                     style="width: 0%"></div>
            </div>
            <small class="text-muted" id="progress-text">Ожидание...</small>
        </div>
    </div>
</div>

<!-- Таблица результатов -->
<div class="card">
    <div class="card-header">
        <h5 class="mb-0">Результаты тестирования</h5>
    </div>
    <div class="card-body">
        <div id="results-container">
            <div class="text-center py-5 text-muted">
                <i class="bi bi-clipboard-data" style="font-size: 4rem; opacity: 0.3;"></i>
                <p class="mt-3">Результаты тестирования появятся здесь</p>
            </div>
        </div>
    </div>
</div>

<script>
let testResults = [];
let isRunning = false;
let shouldStop = false;

// Загрузка сохраненных фраз из localStorage
document.addEventListener('DOMContentLoaded', function() {
    const savedPhrases = localStorage.getItem('widget_tester_phrases');
    if (savedPhrases) {
        document.getElementById('phrases-input').value = savedPhrases;
    }
    
    // Сохранение фраз при изменении
    document.getElementById('phrases-input').addEventListener('input', function() {
        localStorage.setItem('widget_tester_phrases', this.value);
    });
    
    // Проверка выбора виджета
    document.getElementById('widget-select').addEventListener('change', function() {
        document.getElementById('btn-run-test').disabled = !this.value;
    });
    
    // Кнопка запуска
    document.getElementById('btn-run-test').addEventListener('click', runTest);
    
    // Кнопка остановки
    document.getElementById('btn-stop-test').addEventListener('click', function() {
        shouldStop = true;
    });
    
    // Кнопка сброса
    document.getElementById('btn-reset').addEventListener('click', function() {
        testResults = [];
        document.getElementById('results-container').innerHTML = `
            <div class="text-center py-5 text-muted">
                <i class="bi bi-clipboard-data" style="font-size: 4rem; opacity: 0.3;"></i>
                <p class="mt-3">Результаты тестирования появятся здесь</p>
            </div>
        `;
        document.getElementById('stats-section').style.display = 'none';
        document.getElementById('providers-section').style.display = 'none';
    });
});

async function runTest() {
    const widgetKey = document.getElementById('widget-select').value;
    const phrasesText = document.getElementById('phrases-input').value.trim();
    
    if (!widgetKey) {
        showAlert('Выберите виджет', 'warning');
        return;
    }
    
    if (!phrasesText) {
        showAlert('Введите тестовые фразы', 'warning');
        return;
    }
    
    const phrases = phrasesText.split('\n')
        .map(p => p.trim())
        .filter(p => p.length > 0);
    
    if (phrases.length === 0) {
        showAlert('Введите хотя бы одну фразу', 'warning');
        return;
    }
    
    isRunning = true;
    shouldStop = false;
    testResults = [];
    
    // UI изменения
    document.getElementById('btn-run-test').disabled = true;
    document.getElementById('btn-stop-test').disabled = false;
    document.getElementById('btn-stop-test').style.display = 'inline-block';
    document.getElementById('progress-section').style.display = 'block';
    document.getElementById('widget-select').disabled = true;
    document.getElementById('phrases-input').disabled = true;
    
    // Очистка результатов
    document.getElementById('results-container').innerHTML = '';
    document.getElementById('stats-section').style.display = 'none';
    
    // Выполнение запросов последовательно
    for (let i = 0; i < phrases.length; i++) {
        if (shouldStop) {
            console.log('Тестирование остановлено пользователем');
            break;
        }
        
        const phrase = phrases[i];
        console.log(`[${i + 1}/${phrases.length}] Обработка фразы: "${phrase}"`);
        updateProgress(i + 1, phrases.length, phrase);
        
        try {
            const startTime = Date.now();
            const response = await fetch('/api/query.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    widget_key: widgetKey,
                    question: phrase,
                    _debug: true,
                    _no_cache: true
                })
            });
            
            const responseTime = Date.now() - startTime;
            console.log(`[${i + 1}/${phrases.length}] Получен ответ за ${responseTime}мс, статус: ${response.status}`);
            
            // Проверяем статус ответа
            if (!response.ok) {
                console.error(`[${i + 1}/${phrases.length}] Ошибка HTTP: ${response.status} ${response.statusText}`);
                let errorText = '';
                try {
                    const errorData = await response.json();
                    errorText = errorData.error || `HTTP ${response.status}: ${response.statusText}`;
                } catch (e) {
                    errorText = `HTTP ${response.status}: ${response.statusText}`;
                }
                
                const result = {
                    index: i + 1,
                    phrase: phrase,
                    success: false,
                    error: errorText,
                    responseTime: responseTime,
                    response: null,
                    provider: 'error',
                    providerModel: 'N/A',
                    tokens: null,
                    answer: null,
                    medicalTerms: [],
                    specialists: [],
                    services: [],
                    articles: [],
                    specializations: []
                };
                
                testResults.push(result);
                renderResult(result);
                continue;
            }
            
            // Парсим JSON ответ
            let data;
            try {
                data = await response.json();
                console.log(`[${i + 1}/${phrases.length}] JSON распарсен успешно`, data.error ? 'Ошибка в ответе: ' + data.error : 'Успех');
            } catch (parseError) {
                console.error(`[${i + 1}/${phrases.length}] Ошибка парсинга JSON:`, parseError);
                throw new Error('Ошибка парсинга JSON ответа: ' + parseError.message);
            }
            
            const result = {
                index: i + 1,
                phrase: phrase,
                success: !data.error,
                error: data.error || null,
                responseTime: responseTime,
                response: data,
                provider: data.debug?.provider || 'unknown',
                providerModel: data.debug?.provider_model || 'N/A',
                tokens: data.debug?.cost ? 
                    `${(data.debug.cost.input_tokens || 0) + (data.debug.cost.output_tokens || 0)} токенов` : 
                    null,
                answer: data.text || null,
                medicalTerms: data.debug?.three_stage_search?.stage2?.all_extracted_terms || [],
                categories: data.debug?.three_stage_search?.stage3?.relevant_categories || {},
                itemsFound: data.debug?.three_stage_search?.stage3?.items_found || {},
                totalItemsFound: data.debug?.three_stage_search?.stage3?.total_items_found || 0,
                specialists: data.data?.specialists || [],
                services: data.data?.services || [],
                articles: data.data?.articles || [],
                specializations: data.data?.specializations || []
            };
            
            testResults.push(result);
            renderResult(result);
            console.log(`[${i + 1}/${phrases.length}] Результат добавлен в таблицу`);
            
        } catch (error) {
            console.error(`[${i + 1}/${phrases.length}] Исключение при обработке:`, error);
            const result = {
                index: i + 1,
                phrase: phrase,
                success: false,
                error: error.message || 'Ошибка запроса',
                responseTime: 0,
                response: null,
                provider: 'error',
                providerModel: 'N/A',
                tokens: null,
                answer: null,
                medicalTerms: [],
                specialists: [],
                services: [],
                articles: [],
                specializations: []
            };
            
            testResults.push(result);
            renderResult(result);
        }
        
        // Обновляем статистику после каждого запроса
        updateStats();
        
        // Небольшая задержка между запросами
        if (i < phrases.length - 1 && !shouldStop) {
            await new Promise(resolve => setTimeout(resolve, 500));
        }
    }
    
    // Завершение
    isRunning = false;
    shouldStop = false;
    document.getElementById('btn-run-test').disabled = false;
    document.getElementById('btn-stop-test').disabled = true;
    document.getElementById('btn-stop-test').style.display = 'none';
    document.getElementById('progress-section').style.display = 'none';
    document.getElementById('widget-select').disabled = false;
    document.getElementById('phrases-input').disabled = false;
    
    // Финальное обновление статистики
    updateStats();
    
    // Показываем сообщение о завершении
    if (testResults.length > 0) {
        const successCount = testResults.filter(r => r.success).length;
        const totalCount = testResults.length;
        showAlert(`Тестирование завершено: ${successCount} из ${totalCount} запросов успешно`, successCount === totalCount ? 'success' : 'warning');
    }
}

function updateProgress(current, total, currentPhrase) {
    const percent = Math.round((current / total) * 100);
    document.getElementById('progress-bar').style.width = percent + '%';
    document.getElementById('progress-text').textContent = 
        `Выполнено ${current} из ${total} запросов: "${currentPhrase.substring(0, 50)}${currentPhrase.length > 50 ? '...' : ''}"`;
}

function renderResult(result) {
    try {
        const container = document.getElementById('results-container');
        
        if (!container) {
            console.error('Контейнер результатов не найден!');
            return;
        }
        
        // Создаем таблицу если её еще нет
        if (container.querySelector('.text-center') || !container.querySelector('table')) {
            container.innerHTML = '<table class="table table-hover"><thead><tr><th style="width: 50px">№</th><th>Фраза</th><th style="width: 100px">Статус</th><th style="width: 150px">Провайдер</th><th style="width: 100px">Время</th><th style="width: 120px">Токены</th><th style="width: 100px">Детали</th></tr></thead><tbody id="results-tbody"></tbody></table>';
        }
        
        const tbody = document.getElementById('results-tbody');
        
        if (!tbody) {
            console.error('Tbody не найден!');
            return;
        }
    const row = document.createElement('tr');
    row.className = result.success ? '' : 'table-danger';
    
    const statusBadge = result.success ? 
        '<span class="badge bg-success">Успех</span>' : 
        '<span class="badge bg-danger">Ошибка</span>';
    
    const providerBadge = getProviderBadge(result.provider, result.providerModel);
    
    row.innerHTML = `
        <td>${result.index}</td>
        <td><small>${escapeHtml(result.phrase)}</small></td>
        <td>${statusBadge}</td>
        <td>${providerBadge}</td>
        <td><small>${result.responseTime} мс</small></td>
        <td><small>${result.tokens || '-'}</small></td>
        <td>
            <button class="btn btn-sm btn-outline-primary" 
                    type="button" 
                    data-bs-toggle="collapse" 
                    data-bs-target="#details-${result.index}"
                    onclick="this.innerHTML = this.innerHTML.includes('Показать') ? '<i class=\\'bi bi-chevron-up\\'></i> Скрыть' : '<i class=\\'bi bi-chevron-down\\'></i> Показать'">
                <i class="bi bi-chevron-down"></i> Показать
            </button>
        </td>
    `;
    
    tbody.appendChild(row);
    
    // Детальная информация
    const detailsRow = document.createElement('tr');
    detailsRow.className = 'collapse';
    detailsRow.id = `details-${result.index}`;
    detailsRow.innerHTML = `
        <td colspan="7">
            <div class="p-3 bg-dark">
                ${renderDetails(result)}
            </div>
        </td>
    `;
    tbody.appendChild(detailsRow);
    } catch (error) {
        console.error('Ошибка при отображении результата:', error, result);
    }
}

function renderDetails(result) {
    let html = '<div class="row g-3">';
    
    // Ошибка
    if (result.error) {
        html += `
            <div class="col-12">
                <div class="alert alert-danger mb-0">
                    <strong>Ошибка:</strong> ${escapeHtml(result.error)}
                </div>
            </div>
        `;
    }
    
    // Ответ от помощника
    if (result.answer) {
        html += `
            <div class="col-12">
                <h6 class="text-primary mb-2"><i class="bi bi-chat-dots"></i> Ответ от помощника</h6>
                <div class="bg-secondary p-3 rounded">
                    <p class="mb-0">${escapeHtml(result.answer)}</p>
                </div>
            </div>
        `;
    }
    
    // Извлеченные термины
    if (result.medicalTerms && result.medicalTerms.length > 0) {
        html += `
            <div class="col-12">
                <h6 class="text-info mb-2"><i class="bi bi-tags"></i> Извлеченные термины (Этап 2)</h6>
                <div class="d-flex flex-wrap gap-2">
                    ${result.medicalTerms.map(term => 
                        `<span class="badge bg-info">${escapeHtml(term)}</span>`
                    ).join('')}
                </div>
            </div>
        `;
    }
    
    // Релевантные категории и статистика поиска
    if (result.categories && Object.keys(result.categories).length > 0) {
        html += `
            <div class="col-12">
                <h6 class="text-primary mb-2"><i class="bi bi-folder"></i> Релевантные категории (Этап 3)</h6>
                <div class="mb-2">
                    <strong>Всего найдено записей:</strong> <span class="badge bg-success">${result.totalItemsFound}</span>
                </div>
                ${Object.entries(result.categories).map(([section, categories]) => `
                    <div class="mb-2">
                        <strong>${escapeHtml(section)}:</strong> 
                        <span class="badge bg-secondary">${result.itemsFound[section] || 0} записей</span>
                        <div class="mt-1 d-flex flex-wrap gap-1">
                            ${categories.map(cat => 
                                `<span class="badge bg-primary">${escapeHtml(cat)}</span>`
                            ).join('')}
                        </div>
                    </div>
                `).join('')}
            </div>
        `;
    } else if (result.totalItemsFound > 0) {
        html += `
            <div class="col-12">
                <h6 class="text-primary mb-2"><i class="bi bi-search"></i> Результаты поиска (Этап 3)</h6>
                <div class="mb-2">
                    <strong>Всего найдено записей:</strong> <span class="badge bg-success">${result.totalItemsFound}</span>
                </div>
                ${Object.entries(result.itemsFound || {}).map(([section, count]) => `
                    <div class="mb-1">
                        <strong>${escapeHtml(section)}:</strong> <span class="badge bg-secondary">${count} записей</span>
                    </div>
                `).join('')}
            </div>
        `;
    }
    
    // Специалисты
    if (result.specialists && result.specialists.length > 0) {
        html += `
            <div class="col-12">
                <h6 class="text-success mb-2"><i class="bi bi-people"></i> Выбранные специалисты (${result.specialists.length})</h6>
                <div class="list-group">
                    ${result.specialists.map(spec => `
                        <div class="list-group-item bg-secondary">
                            <strong>${escapeHtml(spec.name || 'Без имени')}</strong>
                            ${spec.specialization ? `<br><small class="text-muted">${escapeHtml(spec.specialization)}</small>` : ''}
                            ${spec.description ? `<br><small>${escapeHtml(spec.description)}</small>` : ''}
                        </div>
                    `).join('')}
                </div>
            </div>
        `;
    }
    
    // Услуги
    if (result.services && result.services.length > 0) {
        html += `
            <div class="col-12">
                <h6 class="text-warning mb-2"><i class="bi bi-briefcase"></i> Выбранные услуги (${result.services.length})</h6>
                <div class="list-group">
                    ${result.services.map(service => `
                        <div class="list-group-item bg-secondary">
                            <strong>${escapeHtml(service.name || service.title || 'Без названия')}</strong>
                            ${service.description ? `<br><small>${escapeHtml(service.description)}</small>` : ''}
                            ${service.price ? `<br><small class="text-success">Цена: ${escapeHtml(service.price)}</small>` : ''}
                        </div>
                    `).join('')}
                </div>
            </div>
        `;
    }
    
    // Статьи
    if (result.articles && result.articles.length > 0) {
        html += `
            <div class="col-12">
                <h6 class="text-info mb-2"><i class="bi bi-file-text"></i> Выбранные статьи (${result.articles.length})</h6>
                <div class="list-group">
                    ${result.articles.map(article => `
                        <div class="list-group-item bg-secondary">
                            <strong>${escapeHtml(article.title || article.name || 'Без названия')}</strong>
                            ${article.description ? `<br><small>${escapeHtml(article.description)}</small>` : ''}
                        </div>
                    `).join('')}
                </div>
            </div>
        `;
    }
    
    // Специализации
    if (result.specializations && result.specializations.length > 0) {
        html += `
            <div class="col-12">
                <h6 class="text-primary mb-2"><i class="bi bi-bookmark"></i> Выбранные специализации (${result.specializations.length})</h6>
                <div class="d-flex flex-wrap gap-2">
                    ${result.specializations.map(spec => 
                        `<span class="badge bg-primary">${escapeHtml(spec.name || spec.title || 'Без названия')}</span>`
                    ).join('')}
                </div>
            </div>
        `;
    }
    
    html += '</div>';
    return html;
}

function getProviderBadge(provider, model) {
    const badges = {
        'claude': '<span class="badge bg-primary">Claude</span>',
        'yandexgpt': '<span class="badge bg-success">Yandex GPT</span>',
        'openrouter': '<span class="badge bg-warning text-dark">OpenRouter</span>',
        'cached': '<span class="badge bg-secondary">Кеш</span>',
        'error': '<span class="badge bg-danger">Ошибка</span>',
        'unknown': '<span class="badge bg-dark">Неизвестно</span>'
    };
    
    let badge = badges[provider] || badges['unknown'];
    if (model && model !== 'N/A' && provider !== 'cached') {
        badge += `<br><small class="text-muted">${escapeHtml(model.substring(0, 30))}${model.length > 30 ? '...' : ''}</small>`;
    }
    
    return badge;
}

function updateStats() {
    try {
        if (testResults.length === 0) return;
        
        const total = testResults.length;
        const success = testResults.filter(r => r.success).length;
        const errors = total - success;
        const successfulResults = testResults.filter(r => r.success);
        const avgTime = successfulResults.length > 0 ? 
            Math.round(successfulResults.reduce((sum, r) => sum + r.responseTime, 0) / successfulResults.length) : 0;
    
    // Распределение по провайдерам
    const providers = {};
    testResults.forEach(r => {
        if (r.success) {
            providers[r.provider] = (providers[r.provider] || 0) + 1;
        }
    });
    
    document.getElementById('stat-total').textContent = total;
    document.getElementById('stat-success').textContent = success;
    document.getElementById('stat-errors').textContent = errors;
    document.getElementById('stat-avg-time').textContent = avgTime + ' мс';
    
    // Отображение распределения по провайдерам
    const providersList = document.getElementById('providers-list');
    if (Object.keys(providers).length > 0 && success > 0) {
        providersList.innerHTML = '';
        Object.entries(providers).forEach(([provider, count]) => {
            const badge = getProviderBadge(provider, '');
            const percent = Math.round((count / success) * 100);
            providersList.innerHTML += `
                <div class="d-flex align-items-center gap-2 p-2 bg-dark rounded">
                    ${badge}
                    <span class="text-white">${count} запросов (${percent}%)</span>
                </div>
            `;
        });
        document.getElementById('providers-section').style.display = 'block';
    } else {
        document.getElementById('providers-section').style.display = 'none';
    }
    
    document.getElementById('stat-total').textContent = total;
    document.getElementById('stat-success').textContent = success;
    document.getElementById('stat-errors').textContent = errors;
    document.getElementById('stat-avg-time').textContent = avgTime + ' мс';
    
    document.getElementById('stats-section').style.display = 'flex';
    } catch (error) {
        console.error('Ошибка при обновлении статистики:', error);
    }
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>

