<?php
$page_title = 'Предпросмотр промпта';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/includes/auth.php';

$widget_id = $_GET['widget_id'] ?? 0;

// Получаем виджет и настройки
$db = getDatabase();
$stmt = $db->prepare("
    SELECT w.*, ws.*
    FROM widgets w
    LEFT JOIN widget_settings ws ON w.id = ws.widget_id
    WHERE w.id = ?
");
$stmt->execute([$widget_id]);
$widget = $stmt->fetch();

if (!$widget) {
    die('Виджет не найден');
}

// Трёхэтапный режим всегда включен
$stage1_model = $widget['stage1_model'] ?? 'meta-llama/llama-3.2-1b-instruct';
$stage2_model = $widget['stage2_model'] ?? 'qwen/qwen2.5-14b-instruct';
$stage3_model = $widget['stage3_model'] ?? 'qwen/qwen3-235b-a22b-2507';
$stage1_prompt = $widget['stage1_prompt'] ?? 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей кратко и по делу (2-3 предложения).';
$stage2_prompt = $widget['stage2_prompt'] ?? null;
$stage3_prompt = $widget['stage3_prompt'] ?? null;

// Получаем разделы и поля (включая дочерние)
$stmt = $db->prepare("
    SELECT DISTINCT
        ws.section_name,
        sf.field_name,
        sf.use_in_prompt
    FROM widget_sections ws
    JOIN section_fields sf ON ws.id = sf.section_id
    WHERE ws.widget_id = ? AND ws.is_active = 1 AND sf.use_in_prompt = 1
    
    UNION
    
    SELECT DISTINCT
        ws.section_name,
        scf.child_field_name as field_name,
        scf.use_in_prompt
    FROM widget_sections ws
    JOIN section_fields sf ON ws.id = sf.section_id
    JOIN section_child_fields scf ON sf.id = scf.field_id
    WHERE ws.widget_id = ? AND ws.is_active = 1 AND scf.use_in_prompt = 1
    
    ORDER BY section_name
");
$stmt->execute([$widget_id, $widget_id]);
$fields_by_section = [];
while ($row = $stmt->fetch()) {
    $fields_by_section[$row['section_name']][] = $row['field_name'];
}

// Получаем ВСЕ данные из каждого раздела
$prompt_data = [];
$section_totals = [];

// Устанавливаем timeout для php скриптов (увеличено для больших запросов)
set_time_limit(120);
ini_set('memory_limit', '512M');

foreach ($fields_by_section as $section_name => $fields) {
    $stmt = $db->prepare("
        SELECT pi.id, pf.field_name, pf.field_value
        FROM parsed_items pi
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.widget_id = ? AND pi.section_name = ? AND pf.field_name IN (" . str_repeat('?,', count($fields) - 1) . "?)
        AND pi.is_duplicate = 0
        ORDER BY pi.id
    ");
    $params = array_merge([$widget_id, $section_name], $fields);
    
    try {
        $stmt->execute($params);
    } catch (Exception $e) {
        die('Ошибка при загрузке данных для раздела ' . htmlspecialchars($section_name) . ': ' . $e->getMessage());
    }
    
    $items = [];
    while ($row = $stmt->fetch()) {
        if (!isset($items[$row['id']])) {
            $items[$row['id']] = ['id' => (int)$row['id']];
        }
        $items[$row['id']][$row['field_name']] = $row['field_value'];
    }
    
    $prompt_data[$section_name] = array_values($items);
    $section_totals[$section_name] = count($items);
}

// Названия разделов на русском
$section_names = [
    'specialists' => 'Специалисты',
    'services' => 'Услуги',
    'specializations' => 'Специализации',
    'articles' => 'Статьи'
];

// Получаем список активных разделов для меню шагов
$stmt = $db->prepare("
    SELECT section_name 
    FROM widget_sections 
    WHERE widget_id = ? AND is_active = 1 
    ORDER BY 
        CASE section_name
            WHEN 'specialists' THEN 1
            WHEN 'services' THEN 2
            WHEN 'specializations' THEN 3
            WHEN 'articles' THEN 4
        END
");
$stmt->execute([$widget_id]);
$active_sections = [];
while ($row = $stmt->fetch()) {
    $active_sections[] = $row['section_name'];
}

// Формируем JSON данные для Claude (как в query.php)
$json_data = json_encode($prompt_data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

// Формируем полный промпт
$full_prompt = $widget['claude_prompt'] . "\n\n" . $json_data;

$tokens_estimate = estimateTokens($full_prompt);

// Подсчитываем общее количество элементов
$total_items = array_sum($section_totals);

// Функция расчета стоимости для Claude Haiku 4.5
function calculateCost($inputTokens, $outputTokens, $useCache = false, $cacheTTL = '1h') {
    // Цены для Claude Haiku 4.5 (за миллион токенов)
    $prices = [
        'input_base' => 1.0,           // $1 / MTok
        'input_cache_5m' => 1.25,     // $1.25 / MTok
        'input_cache_1h' => 2.0,       // $2 / MTok
        'input_cache_hit' => 0.1,      // $0.10 / MTok
        'output' => 5.0                // $5 / MTok
    ];
    
    $inputCost = 0;
    if ($useCache) {
        if ($cacheTTL === '5m') {
            $inputCost = ($inputTokens / 1000000) * $prices['input_cache_5m'];
        } else {
            $inputCost = ($inputTokens / 1000000) * $prices['input_cache_1h'];
        }
    } else {
        $inputCost = ($inputTokens / 1000000) * $prices['input_base'];
    }
    
    $outputCost = ($outputTokens / 1000000) * $prices['output'];
    
    return [
        'input_cost' => $inputCost,
        'output_cost' => $outputCost,
        'total_cost' => $inputCost + $outputCost,
        'input_tokens' => $inputTokens,
        'output_tokens' => $outputTokens
    ];
}

// Расчет стоимости для обычного режима (с кешированием)
$cost_normal = calculateCost($tokens_estimate, 1000, true, CLAUDE_CACHE_TTL);

// Расчет стоимости для трёхэтапного режима
// Этап 1: быстрый текстовый ответ (простая модель)
$stage1_input_tokens = 200;
$stage1_output_tokens = 100;
$cost_stage1 = calculateCost($stage1_input_tokens, $stage1_output_tokens, false);

// Этап 2: извлечение терминов (средняя модель)
$stage2_input_tokens = 500;
$stage2_output_tokens = 200;
$cost_stage2 = calculateCost($stage2_input_tokens, $stage2_output_tokens, false);

// Этап 3: основной запрос (мощная модель, промпт сокращается на 80-90%)
$optimized_tokens = round($tokens_estimate * 0.15); // ~15% от исходного
$stage3_input_tokens = $optimized_tokens;
$stage3_output_tokens = 1000;
$cost_stage3 = calculateCost($stage3_input_tokens, $stage3_output_tokens, false);

$cost_three_stage = [
    'stage1' => $cost_stage1,
    'stage2' => $cost_stage2,
    'stage3' => $cost_stage3,
    'total' => [
        'input_tokens' => $stage1_input_tokens + $stage2_input_tokens + $stage3_input_tokens,
        'output_tokens' => $stage1_output_tokens + $stage2_output_tokens + $stage3_output_tokens,
        'total_cost' => $cost_stage1['total_cost'] + $cost_stage2['total_cost'] + $cost_stage3['total_cost']
    ]
];
?>
<!DOCTYPE html>
<html lang="ru" data-bs-theme="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Полный промпт - <?php echo htmlspecialchars($widget['name']); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
</head>
<body class="p-4">
    <div class="container-fluid">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div class="d-flex justify-content-between align-items-center">
                <h1 class="mb-0">Предпросмотр промпта</h1>
                <div>
                    <a href="test-prompt.php?widget_id=<?php echo $widget_id; ?>" class="btn btn-success me-2">
                        <i class="bi bi-play-circle"></i> Тестирование промпта
                    </a>
                    <a href="prompt-test-logs.php?widget_id=<?php echo $widget_id; ?>" class="btn btn-info">
                        <i class="bi bi-list-ul"></i> Логи тестирования
                    </a>
                </div>
            </div>
            <div class="btn-group" role="group">
                <button class="btn btn-warning" id="clear-cache-btn">
                    <i class="bi bi-arrow-clockwise"></i> Очистить кеш
                </button>
                <button class="btn btn-secondary" onclick="window.close()">Закрыть</button>
            </div>
        </div>
        
        <div class="alert alert-info">
            <div class="row">
                <div class="col-md-3">
                    <strong>Виджет:</strong> <?php echo htmlspecialchars($widget['name']); ?>
                </div>
                <div class="col-md-3">
                    <strong>Токенов (примерно):</strong> <?php echo number_format($tokens_estimate, 0, '', ' '); ?>
                </div>
                <div class="col-md-3">
                    <strong>Символов:</strong> <?php echo number_format(mb_strlen($full_prompt), 0, '', ' '); ?>
                </div>
                <div class="col-md-3">
                    <strong>Разделов:</strong> <?php echo count($prompt_data); ?>
                </div>
            </div>
            <hr>
            <div class="row">
                <div class="col-md-12">
                    <?php 
                    $isYandexGPTStage1 = (strpos($stage1_model, 'gpt://') === 0 || strpos($stage1_model, 'yandexgpt') !== false);
                    $useYandexGPT = $isYandexGPTStage1 && defined('USE_YANDEXGPT_FIRST') && USE_YANDEXGPT_FIRST;
                    ?>
                    <?php if ($isYandexGPTStage1): ?>
                    <div class="alert alert-info mb-0">
                        <i class="bi bi-info-circle"></i> 
                        <strong>Этап 1: Yandex GPT 5 Lite</strong>
                        <div class="mt-2">
                            <small>Модель: <code><?php echo htmlspecialchars($stage1_model); ?></code></small><br>
                            <?php if ($useYandexGPT): ?>
                            <small class="text-success"><i class="bi bi-check-circle"></i> Yandex GPT включен (USE_YANDEXGPT_FIRST = true)</small><br>
                            <small>Резервные провайдеры: OpenRouter → Claude Haiku 4.5 (используются при ошибках Yandex GPT)</small>
                            <?php else: ?>
                            <small class="text-warning"><i class="bi bi-exclamation-triangle"></i> Yandex GPT отключен. Для использования установите <code>USE_YANDEXGPT_FIRST = true</code> в config.php</small><br>
                            <small>Текущий провайдер: <?php echo (defined('USE_OPENROUTER_FIRST') && USE_OPENROUTER_FIRST) ? 'OpenRouter' : 'Claude Haiku 4.5'; ?></small>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php elseif (defined('USE_OPENROUTER_FIRST') && USE_OPENROUTER_FIRST): ?>
                    <div class="alert alert-success mb-0">
                        <i class="bi bi-check-circle"></i> 
                        <strong>Основной провайдер: OpenRouter</strong>
                        <div class="mt-2">
                            <small>Модель: <code><?php echo htmlspecialchars(OPENROUTER_MODEL); ?></code></small><br>
                            <small>Резервный провайдер: Claude Haiku 4.5 (используется при ошибках OpenRouter)</small>
                        </div>
                    </div>
                    <?php else: ?>
                    <div class="alert alert-warning mb-0">
                        <i class="bi bi-info-circle"></i> 
                        <strong>Провайдер: Claude Haiku 4.5</strong>
                        <div class="mt-2">
                            <small>OpenRouter отключен. Для включения установите <code>USE_OPENROUTER_FIRST = true</code> в config.php</small>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <hr>
            <div class="alert alert-success mb-0">
                <i class="bi bi-check-circle"></i> 
                <strong>Трёхэтапный режим включен</strong>
                <div class="mt-2">
                    <small><strong>Этап 1:</strong> <?php echo htmlspecialchars($stage1_model); ?> - быстрый текстовый ответ</small><br>
                    <small><strong>Этап 2:</strong> <?php echo htmlspecialchars($stage2_model); ?> - извлечение медицинских терминов</small><br>
                    <small><strong>Этап 3:</strong> <?php echo htmlspecialchars($stage3_model); ?> - основной запрос с данными</small>
                </div>
            </div>
            <?php if (defined('USE_EMBEDDING_SEARCH') && USE_EMBEDDING_SEARCH): ?>
            <div class="alert alert-info mb-0 mt-2">
                <i class="bi bi-vector-pen"></i> 
                <strong>Embedding поиск включен</strong>
                <div class="mt-2">
                    <small>Модель: <code><?php echo htmlspecialchars(EMBEDDING_MODEL); ?></code></small><br>
                    <small>Лимит результатов: <?php echo EMBEDDING_SEARCH_LIMIT; ?> записей на раздел</small><br>
                    <small class="text-muted">Используется семантический поиск для точной фильтрации записей и сокращения промпта</small>
                </div>
            </div>
            <?php else: ?>
            <div class="alert alert-warning mb-0 mt-2">
                <i class="bi bi-exclamation-triangle"></i> 
                <strong>Embedding поиск отключен</strong>
                <div class="mt-2">
                    <small>Используется обычный поиск по ключевым словам (LIKE). Для включения embedding поиска установите <code>USE_EMBEDDING_SEARCH = true</code> в config.php</small>
                </div>
            </div>
            <?php endif; ?>
            <hr>
            <div class="row mt-2">
                <div class="col-md-12">
                    <h6><i class="bi bi-calculator"></i> Стоимость запросов (Claude Haiku 4.5):</h6>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="card bg-dark mb-2">
                                <div class="card-body">
                                    <strong>Обычный режим (с кешированием):</strong>
                                    <div class="mt-2">
                                        <div>Входные токены: <?php echo number_format($cost_normal['input_tokens'], 0, '', ' '); ?></div>
                                        <div>Выходные токены: ~<?php echo number_format($cost_normal['output_tokens'], 0, '', ' '); ?> (оценка)</div>
                                        <div class="mt-2">
                                            <strong>Стоимость:</strong> 
                                            <span class="text-warning">$<?php echo number_format($cost_normal['total_cost'], 6); ?></span>
                                            <small class="text-muted">(≈ $<?php echo number_format($cost_normal['total_cost'] * 1000, 3); ?> за 1000 запросов)</small>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="card bg-dark mb-2">
                                <div class="card-body">
                                    <strong>Трёхэтапный режим (без кеширования):</strong>
                                    <div class="mt-2">
                                        <div><small>Этап 1 (текст): <?php echo number_format($cost_three_stage['stage1']['input_tokens'], 0, '', ' '); ?> токенов → $<?php echo number_format($cost_three_stage['stage1']['total_cost'], 6); ?></small></div>
                                        <div><small>Этап 2 (термины): <?php echo number_format($cost_three_stage['stage2']['input_tokens'], 0, '', ' '); ?> токенов → $<?php echo number_format($cost_three_stage['stage2']['total_cost'], 6); ?></small></div>
                                        <div><small>Этап 3 (основной): ~<?php echo number_format($cost_three_stage['stage3']['input_tokens'], 0, '', ' '); ?> токенов → $<?php echo number_format($cost_three_stage['stage3']['total_cost'], 6); ?></small></div>
                                        <div class="mt-2">
                                            <strong>Общая стоимость:</strong> 
                                            <span class="text-success">$<?php echo number_format($cost_three_stage['total']['total_cost'], 6); ?></span>
                                            <small class="text-muted">(≈ $<?php echo number_format($cost_three_stage['total']['total_cost'] * 1000, 3); ?> за 1000 запросов)</small>
                                        </div>
                                        <div class="mt-2">
                                            <strong>Экономия:</strong> 
                                            <span class="text-success"><?php echo number_format((1 - $cost_three_stage['total']['total_cost'] / $cost_normal['total_cost']) * 100, 1); ?>%</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <hr>
            <div class="row mt-2">
                <?php foreach ($section_totals as $section => $count): ?>
                    <div class="col-md-3">
                        <strong><?php echo $section_names[$section] ?? $section; ?>:</strong> <?php echo $count; ?> элементов
                    </div>
                <?php endforeach; ?>
            </div>
            
            <?php if ($tokens_estimate > 50000): ?>
                <hr>
                <div class="alert alert-warning mb-0 mt-2">
                    <i class="bi bi-exclamation-triangle"></i> 
                    <strong>Внимание:</strong> Большой объем данных (>50k токенов). Тестовый запрос может работать медленнее обычного.
                    Timeout: 60 секунд.
                </div>
            <?php endif; ?>
        </div>
        
        <div class="card bg-dark">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Промпт с полными данными (JSON формат)</h5>
                <button class="btn btn-sm btn-outline-light" onclick="copyToClipboard()">
                    <i class="bi bi-clipboard"></i> Копировать
                </button>
            </div>
            <div class="card-body">
                <pre id="prompt-content" style="white-space: pre-wrap; color: #e0e0e0; font-size: 13px; line-height: 1.5; max-height: 80vh; overflow-y: auto;"><?php echo htmlspecialchars($full_prompt); ?></pre>
            </div>
        </div>
        
        <!-- Секция тестирования -->
        <div class="card mt-4">
            <div class="card-header">
                <h5 class="mb-0">Тестовый запрос</h5>
            </div>
            <div class="card-body">
                <div class="alert alert-info mb-3">
                    <i class="bi bi-info-circle"></i> 
                    <strong>Важно:</strong> Запрос отправляется с реальными данными виджета (<?php echo number_format($total_items); ?> записей = ~87,000 токенов).
                    При превышении лимита Claude API (100,000 токенов/мин) может появиться ошибка 429. 
                    <strong>Рекомендуется делать запросы с интервалом 5-10 минут.</strong>
                </div>
                
                <!-- Меню шагов -->
                <div class="mb-3">
                    <label class="form-label">Выберите шаг для тестирования:</label>
                    <div class="btn-group" role="group" id="steps-menu">
                        <button type="button" class="btn btn-outline-primary active" data-step="1">
                            <i class="bi bi-1-circle"></i> Шаг 1: Текстовый ответ
                        </button>
                        <?php 
                        $stepNum = 2;
                        foreach ($active_sections as $section): 
                        ?>
                        <button type="button" class="btn btn-outline-primary" data-step="<?php echo $stepNum; ?>" data-section="<?php echo htmlspecialchars($section); ?>">
                            <i class="bi bi-<?php echo $stepNum; ?>-circle"></i> Шаг <?php echo $stepNum; ?>: <?php echo htmlspecialchars($section_names[$section] ?? $section); ?>
                        </button>
                        <?php 
                        $stepNum++;
                        endforeach; 
                        ?>
                        <button type="button" class="btn btn-success" id="test-all-steps-btn">
                            <i class="bi bi-play-circle"></i> Тестировать все шаги
                        </button>
                    </div>
                </div>
                
                <form id="test-form">
                    <div class="mb-3">
                        <label class="form-label">Вопрос пользователя:</label>
                        <textarea class="form-control" id="test-question" rows="3" 
                                  placeholder="Например: Где найти терапевта?">У меня болит спина</textarea>
                    </div>
                    <button type="submit" class="btn btn-primary" id="test-step-btn">
                        <i class="bi bi-send"></i> Отправить запрос
                    </button>
                </form>
                
                <div id="test-results" style="display: none;" class="mt-4">
                    <!-- Трёхэтапный режим: Этап 1 -->
                    <div id="three-stage-stage1" class="mb-3" style="display: none;">
                        <h6><i class="bi bi-1-circle"></i> Этап 1: Быстрый текстовый ответ</h6>
                        <div class="card bg-dark">
                            <div class="card-body">
                    <div class="mb-3">
                                    <strong>Модель:</strong> <span class="text-info"><?php echo htmlspecialchars($stage1_model); ?></span>
                                    <span id="stage1-provider" class="badge bg-secondary ms-2"></span>
                                </div>
                                <div class="mb-3">
                                    <strong>Запрос:</strong>
                                    <div class="mb-3">
                                        <strong>Полный промпт (отправляется в AI):</strong>
                                        <pre id="stage1-full-prompt" class="bg-dark text-warning p-2 mt-2" style="max-height: 500px; overflow-y: auto; font-size: 11px; white-space: pre-wrap; border-left: 3px solid #ffc107;"></pre>
                                    </div>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                                <strong>System Prompt:</strong>
                                                <pre id="stage1-system-prompt" class="mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap;"></pre>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                                <strong>User Prompt:</strong>
                                                <pre id="stage1-user-prompt" class="mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap;"></pre>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="mb-3">
                                    <strong>Ответ:</strong>
                                    <div id="stage1-text" class="bg-success bg-opacity-25 p-2 mt-2 mb-0" style="max-height: 200px; overflow-y: auto; font-size: 12px; white-space: pre-wrap; border-left: 3px solid #198754;"></div>
                                </div>
                                <div class="mb-2">
                                    <strong>Время выполнения:</strong> <span id="stage1-time" class="text-success"></span> мс
                                </div>
                                <div class="mb-3">
                                    <strong>Полный JSON запроса:</strong>
                                    <pre id="stage1-request-json" class="bg-dark text-info p-2 mt-2" style="max-height: 300px; overflow-y: auto; font-size: 11px; white-space: pre-wrap;"></pre>
                                </div>
                                <div class="mb-3">
                                    <strong>Полный JSON ответа:</strong>
                                    <pre id="stage1-response-json" class="bg-dark text-success p-2 mt-2" style="max-height: 300px; overflow-y: auto; font-size: 11px; white-space: pre-wrap;"></pre>
                                </div>
                                <div id="stage1-yandex-debug" class="mt-3" style="display: none;">
                                    <strong>Yandex GPT API запрос/ответ:</strong>
                                    <details class="mt-2">
                                        <summary style="cursor: pointer; color: #0dcaf0;">Показать детали запроса/ответа</summary>
                                        <div class="mt-2">
                                            <div class="mb-2">
                                                <strong>Запрос:</strong>
                                                <pre id="stage1-yandex-request" class="bg-dark text-light p-2 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 300px; overflow-y: auto;"></pre>
                                            </div>
                                            <div class="mb-2">
                                                <strong>Ответ (HTTP код):</strong> <span id="stage1-yandex-http-code" class="badge"></span>
                                            </div>
                                            <div class="mb-2">
                                                <strong>Ответ (тело):</strong>
                                                <pre id="stage1-yandex-response" class="bg-dark text-light p-2 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 300px; overflow-y: auto;"></pre>
                                            </div>
                                            <div id="stage1-yandex-error" class="alert alert-danger mt-2" style="display: none;">
                                                <strong>Ошибка:</strong> <span id="stage1-yandex-error-text"></span>
                                            </div>
                                        </div>
                                    </details>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Трёхэтапный режим: Этап 2 -->
                    <div id="three-stage-stage2" class="mb-3" style="display: none;">
                        <h6><i class="bi bi-2-circle"></i> Этап 2: Извлечение медицинских терминов</h6>
                        <div class="card bg-dark">
                            <div class="card-body">
                                <div class="mb-3">
                                    <strong>Модель:</strong> <span class="text-info"><?php echo htmlspecialchars($stage2_model); ?></span>
                                    <span id="stage2-provider" class="badge bg-secondary ms-2"></span>
                                </div>
                                <div class="mb-3">
                                    <strong>Запрос:</strong>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                                <strong>System Prompt:</strong>
                                                <pre id="stage2-system-prompt" class="mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 200px; overflow-y: auto;"></pre>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                                <strong>User Prompt:</strong>
                                                <pre id="stage2-user-prompt" class="mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 200px; overflow-y: auto;"></pre>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="mb-3">
                                    <strong>Ответ:</strong>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="bg-success bg-opacity-25 p-2 mt-2" style="font-size: 12px; border-left: 3px solid #198754;">
                                                <strong>Извлеченные ключевые слова:</strong>
                                                <pre id="stage2-keywords" class="mb-0 mt-1" style="font-size: 11px; max-height: 150px; overflow-y: auto;"></pre>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                                <strong>Полный JSON ответ:</strong>
                                                <pre id="stage2-full-response" class="mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 200px; overflow-y: auto;"></pre>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="mb-2">
                                    <strong>Вопрос:</strong> <span id="stage2-question" class="text-info"></span>
                                </div>
                                <div class="mb-2">
                                    <strong>Время выполнения:</strong> <span id="stage2-time" class="text-success"></span> мс
                                    <span id="stage2-cached" class="badge bg-success ms-2" style="display: none;">Из кеша</span>
                                </div>
                                <div id="stage2-yandex-debug" class="mt-3" style="display: none;">
                                    <strong>Yandex GPT API запрос/ответ:</strong>
                                    <details class="mt-2">
                                        <summary style="cursor: pointer; color: #0dcaf0;">Показать детали запроса/ответа</summary>
                                        <div class="mt-2">
                                            <div class="mb-2">
                                                <strong>Запрос:</strong>
                                                <pre id="stage2-yandex-request" class="bg-dark text-light p-2 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 300px; overflow-y: auto;"></pre>
                                            </div>
                                            <div class="mb-2">
                                                <strong>Ответ (HTTP код):</strong> <span id="stage2-yandex-http-code" class="badge"></span>
                                            </div>
                                            <div class="mb-2">
                                                <strong>Ответ (тело):</strong>
                                                <pre id="stage2-yandex-response" class="bg-dark text-light p-2 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 300px; overflow-y: auto;"></pre>
                                            </div>
                                            <div id="stage2-yandex-error" class="alert alert-danger mt-2" style="display: none;">
                                                <strong>Ошибка:</strong> <span id="stage2-yandex-error-text"></span>
                                            </div>
                                        </div>
                                    </details>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Трёхэтапный режим: Этап 3 -->
                    <div id="three-stage-stage3" class="mb-3" style="display: none;">
                        <h6><i class="bi bi-3-circle"></i> Этап 3: Основной запрос с данными</h6>
                        <div class="card bg-dark">
                            <div class="card-body">
                                <div class="mb-3">
                                    <strong>Модель:</strong> <span class="text-info"><?php echo htmlspecialchars($stage3_model); ?></span>
                                    <span id="stage3-provider" class="badge bg-secondary ms-2"></span>
                                </div>
                                <div class="mb-3">
                                    <strong>Запрос:</strong>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                                <strong>System Prompt:</strong>
                                                <pre id="stage3-system-prompt" class="mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 200px; overflow-y: auto;"></pre>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                                <strong>User Prompt (после оптимизации):</strong>
                                                <pre id="stage3-user-prompt" class="mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 200px; overflow-y: auto;"></pre>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="mb-3">
                                    <strong>Оптимизация промпта:</strong>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="bg-warning bg-opacity-25 p-2 mt-2" style="font-size: 12px; border-left: 3px solid #ffc107;">
                                                <strong>До оптимизации:</strong>
                                                <div class="mt-1">Символов: <span id="stage3-size-before" class="text-warning"></span></div>
                                                <div>Токенов: <span id="stage3-tokens-before" class="text-warning"></span></div>
                                                <details class="mt-2">
                                                    <summary style="cursor: pointer; font-size: 11px;">Показать полный промпт</summary>
                                                    <pre id="stage3-prompt-before" class="mt-2 mb-0" style="font-size: 10px; white-space: pre-wrap; word-wrap: break-word; max-height: 300px; overflow-y: auto;"></pre>
                                                </details>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="bg-success bg-opacity-25 p-2 mt-2" style="font-size: 12px; border-left: 3px solid #198754;">
                                                <strong>После оптимизации:</strong>
                                                <div class="mt-1">Символов: <span id="stage3-size-after" class="text-success"></span></div>
                                                <div>Токенов: <span id="stage3-tokens-after" class="text-success"></span></div>
                                                <div class="mt-1"><strong>Сокращение:</strong> <span id="stage3-reduction" class="text-success fw-bold"></span>%</div>
                                                <details class="mt-2">
                                                    <summary style="cursor: pointer; font-size: 11px;">Показать оптимизированный промпт</summary>
                                                    <pre id="stage3-prompt-after" class="mt-2 mb-0" style="font-size: 10px; white-space: pre-wrap; word-wrap: break-word; max-height: 300px; overflow-y: auto;"></pre>
                                                </details>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="mb-3">
                                    <strong>Ответ:</strong>
                                    <div class="bg-success bg-opacity-25 p-2 mt-2" style="font-size: 12px; border-left: 3px solid #198754;">
                                        <pre id="stage3-response" class="mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 300px; overflow-y: auto;"></pre>
                                    </div>
                                </div>
                                <div class="mb-2">
                                    <strong>Время выполнения:</strong> <span id="stage3-time" class="text-success"></span> мс
                                </div>
                                <div id="stage3-yandex-debug" class="mt-3" style="display: none;">
                                    <strong>Yandex GPT API запрос/ответ:</strong>
                                    <details class="mt-2">
                                        <summary style="cursor: pointer; color: #0dcaf0;">Показать детали запроса/ответа</summary>
                                        <div class="mt-2">
                                            <div class="mb-2">
                                                <strong>Запрос:</strong>
                                                <pre id="stage3-yandex-request" class="bg-dark text-light p-2 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 300px; overflow-y: auto;"></pre>
                                            </div>
                                            <div class="mb-2">
                                                <strong>Ответ (HTTP код):</strong> <span id="stage3-yandex-http-code" class="badge"></span>
                                            </div>
                                            <div class="mb-2">
                                                <strong>Ответ (тело):</strong>
                                                <pre id="stage3-yandex-response" class="bg-dark text-light p-2 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 300px; overflow-y: auto;"></pre>
                                            </div>
                                            <div id="stage3-yandex-error" class="alert alert-danger mt-2" style="display: none;">
                                                <strong>Ошибка:</strong> <span id="stage3-yandex-error-text"></span>
                                            </div>
                                        </div>
                                    </details>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Шаги 2-5: Запросы по разделам -->
                    <?php 
                    $stepNum = 2;
                    foreach ($active_sections as $section): 
                    ?>
                    <div id="section-stage-<?php echo $stepNum; ?>" class="mb-3" style="display: none;">
                        <h6><i class="bi bi-<?php echo $stepNum; ?>-circle"></i> Шаг <?php echo $stepNum; ?>: <?php echo htmlspecialchars($section_names[$section] ?? $section); ?></h6>
                        <div class="card bg-dark">
                            <div class="card-body">
                                <div class="mb-3">
                                    <strong>Раздел:</strong> <span class="text-info"><?php echo htmlspecialchars($section); ?></span>
                                    <span id="section-<?php echo $stepNum; ?>-provider" class="badge bg-secondary ms-2"></span>
                                </div>
                                <div class="mb-3">
                                    <strong>Запрос:</strong>
                                    <div class="mb-3">
                                        <strong>Полный промпт (отправляется в AI):</strong>
                                        <pre id="section-<?php echo $stepNum; ?>-full-prompt" class="bg-dark text-warning p-2 mt-2" style="max-height: 500px; overflow-y: auto; font-size: 11px; white-space: pre-wrap; border-left: 3px solid #ffc107;"></pre>
                                    </div>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                                <strong>System Prompt:</strong>
                                                <pre id="section-<?php echo $stepNum; ?>-system-prompt" class="mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 200px; overflow-y: auto;"></pre>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                                <strong>User Prompt:</strong>
                                                <pre id="section-<?php echo $stepNum; ?>-user-prompt" class="mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 200px; overflow-y: auto;"></pre>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="mb-3">
                                    <strong>Ответ:</strong>
                                    <div id="section-<?php echo $stepNum; ?>-text" class="bg-success bg-opacity-25 p-2 mt-2 mb-0" style="max-height: 300px; overflow-y: auto; font-size: 12px; white-space: pre-wrap; border-left: 3px solid #198754;"></div>
                                </div>
                                <div class="mb-3">
                                    <strong>Данные:</strong>
                                    <pre id="section-<?php echo $stepNum; ?>-data" class="bg-dark text-light p-2 mt-2" style="max-height: 300px; overflow-y: auto; font-size: 11px; white-space: pre-wrap;"></pre>
                                </div>
                                <div class="mb-2">
                                    <strong>Время выполнения:</strong> <span id="section-<?php echo $stepNum; ?>-time" class="text-success"></span> мс
                                </div>
                                <div class="mb-3">
                                    <strong>Полный JSON запроса:</strong>
                                    <pre id="section-<?php echo $stepNum; ?>-request-json" class="bg-dark text-info p-2 mt-2" style="max-height: 300px; overflow-y: auto; font-size: 11px; white-space: pre-wrap;"></pre>
                                </div>
                                <div class="mb-3">
                                    <strong>Полный JSON ответа:</strong>
                                    <pre id="section-<?php echo $stepNum; ?>-response-json" class="bg-dark text-success p-2 mt-2" style="max-height: 300px; overflow-y: auto; font-size: 11px; white-space: pre-wrap;"></pre>
                                </div>
                                
                                <!-- Информация о категоризации (prefilter) -->
                                <div id="section-<?php echo $stepNum; ?>-category-info" class="mb-3" style="display: none;">
                                    <div class="card bg-secondary bg-opacity-25">
                                        <div class="card-header">
                                            <strong><i class="bi bi-tags"></i> Определение категорий из вопроса (prefilter оптимизация)</strong>
                                        </div>
                                        <div class="card-body">
                                            <div class="mb-2">
                                                <strong>System Prompt:</strong>
                                                <pre id="section-<?php echo $stepNum; ?>-category-system-prompt" class="bg-dark text-warning p-2 mt-1" style="max-height: 200px; overflow-y: auto; font-size: 11px; white-space: pre-wrap;"></pre>
                                            </div>
                                            <div class="mb-2">
                                                <strong>User Prompt:</strong>
                                                <pre id="section-<?php echo $stepNum; ?>-category-user-prompt" class="bg-dark text-info p-2 mt-1" style="max-height: 200px; overflow-y: auto; font-size: 11px; white-space: pre-wrap;"></pre>
                                            </div>
                                            <div class="mb-2">
                                                <strong>Полный промпт:</strong>
                                                <pre id="section-<?php echo $stepNum; ?>-category-full-prompt" class="bg-dark text-light p-2 mt-1" style="max-height: 200px; overflow-y: auto; font-size: 11px; white-space: pre-wrap;"></pre>
                                            </div>
                                            <div class="mb-2">
                                                <strong>Ответ AI:</strong>
                                                <pre id="section-<?php echo $stepNum; ?>-category-ai-response" class="bg-dark text-success p-2 mt-1" style="max-height: 200px; overflow-y: auto; font-size: 11px; white-space: pre-wrap;"></pre>
                                            </div>
                                            <div class="mb-2">
                                                <strong>Определенные категории:</strong>
                                                <div id="section-<?php echo $stepNum; ?>-category-categories" class="mt-1"></div>
                                            </div>
                                            <div class="mb-2">
                                                <strong>Расширенные категории:</strong>
                                                <div id="section-<?php echo $stepNum; ?>-category-expanded" class="mt-1"></div>
                                            </div>
                                            <div class="mb-2">
                                                <strong>Статистика фильтрации:</strong>
                                                <div id="section-<?php echo $stepNum; ?>-category-stats" class="mt-1"></div>
                                            </div>
                                            <div class="mb-2">
                                                <strong>Полный JSON ответа (определение категорий):</strong>
                                                <pre id="section-<?php echo $stepNum; ?>-category-raw-response" class="bg-dark text-success p-2 mt-1" style="max-height: 200px; overflow-y: auto; font-size: 11px; white-space: pre-wrap;"></pre>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php 
                    $stepNum++;
                    endforeach; 
                    ?>
                    
                    <!-- 0. Сырой RAW ответ от AI (до парсинга) -->
                    <div class="mb-3" id="raw-text-section" style="display: none;">
                        <h6><i class="bi bi-4-circle"></i> Этап 4: Сырой текст от AI API (до парсинга):</h6>
                        <pre id="claude-raw-text" class="bg-dark text-warning p-3 border border-warning" style="max-height: 300px; overflow-y: auto; font-size: 11px; white-space: pre-wrap;"></pre>
                    </div>
                    
                    <!-- 1. Сырой ответ AI -->
                    <div class="mb-3">
                        <h6>5. Сырой ответ AI (распарсенный JSON):</h6>
                        <pre id="raw-response" class="bg-dark text-light p-3 border border-secondary" style="max-height: 300px; overflow-y: auto; font-size: 12px;"></pre>
                    </div>
                    
                    <!-- 2. Распарсенный JSON -->
                    <div class="mb-3">
                        <h6>6. Распарсенный ответ:</h6>
                        <div class="row">
                            <div class="col-md-6">
                                <strong>Текстовый ответ:</strong>
                                <div id="parsed-text" class="border border-secondary p-2 mt-2 bg-dark" style="min-height: 100px;"></div>
                            </div>
                            <div class="col-md-6">
                                <strong>ID записей:</strong>
                                <pre id="parsed-ids" class="bg-dark text-light border border-secondary p-2 mt-2" style="max-height: 200px; overflow-y: auto; font-size: 12px;"></pre>
                            </div>
                        </div>
                    </div>
                    
                    <!-- 3. Данные из БД -->
                    <div class="mb-3">
                        <h6>7. Полные данные из БД по ID:</h6>
                        <pre id="db-data" class="bg-dark text-light border border-secondary p-3" style="max-height: 400px; overflow-y: auto; font-size: 12px;"></pre>
                    </div>
                    
                    <!-- 4. Итоговый JSON для виджета -->
                    <div class="mb-3">
                        <h6>8. Итоговый JSON для виджета:</h6>
                        <pre id="final-json" class="bg-dark text-light border border-secondary p-3" style="max-height: 400px; overflow-y: auto; font-size: 12px;"></pre>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="mt-3 text-center">
            <button class="btn btn-primary" onclick="window.close()">Закрыть</button>
        </div>
    </div>
    
    <script>
    // Данные о разделах из PHP
    const activeSections = <?php echo json_encode($active_sections, JSON_UNESCAPED_UNICODE); ?>;
    const sectionNames = <?php echo json_encode($section_names, JSON_UNESCAPED_UNICODE); ?>;
    const widgetKey = '<?php echo $widget['widget_key']; ?>';
    let currentStep = 1;
    
    function copyToClipboard() {
        const content = document.getElementById('prompt-content').textContent;
        navigator.clipboard.writeText(content).then(() => {
            alert('Промпт скопирован в буфер обмена!');
        }).catch(err => {
            console.error('Ошибка копирования:', err);
        });
    }
    
    // Обработка выбора шага
    document.querySelectorAll('#steps-menu button[data-step]').forEach(btn => {
        btn.addEventListener('click', function() {
            // Убираем активный класс со всех кнопок
            document.querySelectorAll('#steps-menu button[data-step]').forEach(b => {
                b.classList.remove('active');
            });
            // Добавляем активный класс к выбранной кнопке
            this.classList.add('active');
            currentStep = parseInt(this.dataset.step);
            
            // Обновляем текст кнопки отправки
            const testBtn = document.getElementById('test-step-btn');
            if (currentStep === 1) {
                testBtn.innerHTML = '<i class="bi bi-send"></i> Отправить запрос (Шаг 1)';
            } else {
                const section = this.dataset.section;
                const sectionName = sectionNames[section] || section;
                testBtn.innerHTML = `<i class="bi bi-send"></i> Отправить запрос (Шаг ${currentStep}: ${sectionName})`;
            }
        });
    });
    
    // Обработка тестирования всех шагов
    document.getElementById('test-all-steps-btn').addEventListener('click', async function() {
        const question = document.getElementById('test-question').value.trim();
        if (!question) {
            alert('Введите вопрос');
            return;
        }
        
        const btn = this;
        const originalHTML = btn.innerHTML;
        btn.disabled = true;
        btn.innerHTML = '<i class="bi bi-hourglass-split"></i> Тестирование...';
        
        const resultsDiv = document.getElementById('test-results');
        resultsDiv.style.display = 'block';
        
        try {
            // Шаг 1: Быстрый текстовый ответ
            document.getElementById('three-stage-stage1').style.display = 'block';
            document.getElementById('stage1-text').textContent = 'Загрузка...';
            
            const stage1Prompt = <?php echo json_encode($stage1_prompt, JSON_UNESCAPED_UNICODE); ?>;
            const stage1SystemPromptText = stage1Prompt || 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей кратко и по делу (2-3 предложения).';
            const stage1UserPromptText = 'Вопрос пользователя: ' + question;
            
            document.getElementById('stage1-system-prompt').textContent = stage1SystemPromptText;
            document.getElementById('stage1-user-prompt').textContent = stage1UserPromptText;
            
            // Формируем и показываем полный промпт
            const stage1FullPrompt = stage1SystemPromptText + '\n\n' + stage1UserPromptText;
            document.getElementById('stage1-full-prompt').textContent = stage1FullPrompt;
            
            const stage1RequestData = {
                widget_key: widgetKey,
                question: question,
                _debug: true
            };
            
            // Показываем полный JSON запроса
            document.getElementById('stage1-request-json').textContent = JSON.stringify(stage1RequestData, null, 2);
            
            const stage1Response = await fetch(window.location.origin + '/api/query-stage1.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify(stage1RequestData)
            });
            
            const stage1Data = await stage1Response.json();
            
            // Показываем полный JSON ответа
            document.getElementById('stage1-response-json').textContent = JSON.stringify(stage1Data, null, 2);
            
            if (stage1Data.error) {
                document.getElementById('stage1-text').textContent = 'Ошибка: ' + stage1Data.error;
                document.getElementById('stage1-text').className = 'bg-danger bg-opacity-25 p-2 mt-2 mb-0';
            } else {
                document.getElementById('stage1-text').textContent = stage1Data.text || 'Нет ответа';
                document.getElementById('stage1-time').textContent = stage1Data.response_time_ms || 0;
            }
            
            // Извлекаем категории из ответа stage1 (если они были определены)
            const categories = stage1Data.categories || null;
            const categoriesDebug = stage1Data.categories_debug || null;
            
            // Шаги 2-5: Запросы по разделам
            let stepNum = 2;
            for (const section of activeSections) {
                const sectionDiv = document.getElementById(`section-stage-${stepNum}`);
                if (sectionDiv) {
                    sectionDiv.style.display = 'block';
                    document.getElementById(`section-${stepNum}-text`).textContent = 'Загрузка...';
                    
                    try {
                        const sectionRequestData = {
                            widget_key: widgetKey,
                            question: question,
                            section_name: section,
                            _debug: true
                        };
                        
                        // Передаем категории, если они были определены в stage1
                        if (categories) {
                            sectionRequestData.categories = categories;
                            if (categoriesDebug) {
                                sectionRequestData.categories_debug = categoriesDebug;
                            }
                        }
                        
                        // Показываем полный JSON запроса
                        document.getElementById(`section-${stepNum}-request-json`).textContent = JSON.stringify(sectionRequestData, null, 2);
                        
                        const sectionResponse = await fetch(window.location.origin + '/api/query-section.php', {
                            method: 'POST',
                            headers: {'Content-Type': 'application/json'},
                            body: JSON.stringify(sectionRequestData)
                        });
                        
                        const sectionData = await sectionResponse.json();
                        
                        // Показываем полный JSON ответа
                        document.getElementById(`section-${stepNum}-response-json`).textContent = JSON.stringify(sectionData, null, 2);
                        
                        if (sectionData.error) {
                            document.getElementById(`section-${stepNum}-text`).textContent = 'Ошибка: ' + sectionData.error;
                            document.getElementById(`section-${stepNum}-text`).className = 'bg-danger bg-opacity-25 p-2 mt-2 mb-0';
                        } else {
                            document.getElementById(`section-${stepNum}-text`).textContent = sectionData.text || 'Нет ответа';
                            document.getElementById(`section-${stepNum}-data`).textContent = JSON.stringify(sectionData.data || {}, null, 2);
                            document.getElementById(`section-${stepNum}-time`).textContent = sectionData.response_time_ms || 0;
                            
                            if (sectionData.debug) {
                                // Показываем полный промпт из debug информации
                                if (sectionData.debug.full_prompt) {
                                    document.getElementById(`section-${stepNum}-full-prompt`).textContent = sectionData.debug.full_prompt;
                                }
                                
                                // Показываем system и user промпты отдельно
                                // Определяем промпт для текущего раздела
                                const sectionPromptsAll = {
                                    'specialists': <?php echo json_encode($widget['stage_specialists_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>,
                                    'services': <?php echo json_encode($widget['stage_services_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>,
                                    'articles': <?php echo json_encode($widget['stage_articles_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>,
                                    'specializations': <?php echo json_encode($widget['stage_specializations_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>
                                };
                                
                                const stage3Prompt = sectionPromptsAll[section] || 
                                                   <?php echo json_encode($widget['stage3_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?> || 
                                                   <?php echo json_encode($widget['claude_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>;
                                
                                if (sectionData.debug.system_prompt) {
                                    document.getElementById(`section-${stepNum}-system-prompt`).textContent = sectionData.debug.system_prompt;
                                } else {
                                    document.getElementById(`section-${stepNum}-system-prompt`).textContent = stage3Prompt.substring(0, 500) + '...';
                                }
                                
                                if (sectionData.debug.user_prompt) {
                                    document.getElementById(`section-${stepNum}-user-prompt`).textContent = sectionData.debug.user_prompt.substring(0, 1000) + (sectionData.debug.user_prompt.length > 1000 ? '...' : '');
                                } else {
                                    document.getElementById(`section-${stepNum}-user-prompt`).textContent = 'Вопрос: ' + question + '\n\nДанные раздела: ' + (sectionData.debug.items_count || 0) + ' записей';
                                }
                                
                                // Показываем информацию о категоризации (prefilter)
                                if (sectionData.debug.prefilter_info && sectionData.debug.prefilter_info.category_extraction_debug) {
                                    const categoryInfo = sectionData.debug.prefilter_info.category_extraction_debug;
                                    const categoryDiv = document.getElementById(`section-${stepNum}-category-info`);
                                    if (categoryDiv) {
                                        categoryDiv.style.display = 'block';
                                        
                                        if (categoryInfo.system_prompt) {
                                            document.getElementById(`section-${stepNum}-category-system-prompt`).textContent = categoryInfo.system_prompt;
                                        }
                                        if (categoryInfo.user_prompt) {
                                            document.getElementById(`section-${stepNum}-category-user-prompt`).textContent = categoryInfo.user_prompt;
                                        }
                                        if (categoryInfo.full_prompt) {
                                            document.getElementById(`section-${stepNum}-category-full-prompt`).textContent = categoryInfo.full_prompt;
                                        }
                                        if (categoryInfo.ai_response) {
                                            document.getElementById(`section-${stepNum}-category-ai-response`).textContent = categoryInfo.ai_response;
                                        }
                                        if (categoryInfo.raw_response) {
                                            try {
                                                const rawJson = JSON.parse(categoryInfo.raw_response);
                                                document.getElementById(`section-${stepNum}-category-raw-response`).textContent = JSON.stringify(rawJson, null, 2);
                                            } catch (e) {
                                                document.getElementById(`section-${stepNum}-category-raw-response`).textContent = categoryInfo.raw_response;
                                            }
                                        }
                                        if (sectionData.debug.prefilter_info.question_categories) {
                                            const categories = sectionData.debug.prefilter_info.question_categories;
                                            const escapeHtml = (text) => {
                                                const div = document.createElement('div');
                                                div.textContent = text;
                                                return div.innerHTML;
                                            };
                                            document.getElementById(`section-${stepNum}-category-categories`).innerHTML = 
                                                categories.map(cat => `<span class="badge bg-primary me-1 mb-1">${escapeHtml(cat)}</span>`).join(' ');
                                        }
                                        if (sectionData.debug.prefilter_info.expanded_categories) {
                                            const expanded = sectionData.debug.prefilter_info.expanded_categories;
                                            const escapeHtml = (text) => {
                                                const div = document.createElement('div');
                                                div.textContent = text;
                                                return div.innerHTML;
                                            };
                                            document.getElementById(`section-${stepNum}-category-expanded`).innerHTML = 
                                                expanded.map(cat => `<span class="badge bg-success me-1 mb-1">${escapeHtml(cat)}</span>`).join(' ');
                                        }
                                        const stats = sectionData.debug.prefilter_info;
                                        let statsHtml = '';
                                        if (stats.original_count !== undefined) {
                                            statsHtml += `<div>Исходное количество элементов: <strong>${stats.original_count}</strong></div>`;
                                        }
                                        if (stats.filtered_count !== undefined) {
                                            statsHtml += `<div>После фильтрации: <strong>${stats.filtered_count}</strong></div>`;
                                        }
                                        if (stats.filtered_ids && stats.filtered_ids.length > 0) {
                                            statsHtml += `<div>Отфильтрованные ID (первые 20): <code>${stats.filtered_ids.join(', ')}</code></div>`;
                                        }
                                        document.getElementById(`section-${stepNum}-category-stats`).innerHTML = statsHtml;
                                    }
                                } else {
                                    // Скрываем блок категоризации, если нет информации
                                    const categoryDiv = document.getElementById(`section-${stepNum}-category-info`);
                                    if (categoryDiv) {
                                        categoryDiv.style.display = 'none';
                                    }
                                }
                            } else {
                                // Если нет debug информации, показываем базовые промпты
                                const sectionPromptsFallback = {
                                    'specialists': <?php echo json_encode($widget['stage_specialists_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>,
                                    'services': <?php echo json_encode($widget['stage_services_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>,
                                    'articles': <?php echo json_encode($widget['stage_articles_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>,
                                    'specializations': <?php echo json_encode($widget['stage_specializations_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>
                                };
                                
                                const stage3PromptFallback = sectionPromptsFallback[section] || 
                                                           <?php echo json_encode($widget['stage3_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?> || 
                                                           <?php echo json_encode($widget['claude_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>;
                                
                                document.getElementById(`section-${stepNum}-system-prompt`).textContent = stage3PromptFallback.substring(0, 500) + '...';
                                document.getElementById(`section-${stepNum}-user-prompt`).textContent = 'Вопрос: ' + question;
                            }
                        }
                    } catch (error) {
                        document.getElementById(`section-${stepNum}-text`).textContent = 'Ошибка: ' + error.message;
                        document.getElementById(`section-${stepNum}-text`).className = 'bg-danger bg-opacity-25 p-2 mt-2 mb-0';
                    }
                }
                stepNum++;
            }
            
        } catch (error) {
            alert('Ошибка при тестировании: ' + error.message);
        } finally {
            btn.disabled = false;
            btn.innerHTML = originalHTML;
        }
    });
    
    // Обработчик для кнопки очистки кеша
    document.getElementById('clear-cache-btn').addEventListener('click', async () => {
        if (!confirm('Вы уверены, что хотите очистить кеш для этого виджета?')) {
            return;
        }
        
        const btn = document.getElementById('clear-cache-btn');
        const originalHTML = btn.innerHTML;
        btn.disabled = true;
        btn.innerHTML = '<i class="bi bi-hourglass-split"></i> Очистка...';
        
        try {
            const response = await fetch(window.location.origin + '/api/clear-cache.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    widget_id: <?php echo $widget_id; ?>
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                const msg = `✓ Кеш успешно очищен!\n` +
                    `Удалено логов: ${data.logs_cleared}\n` +
                    `Удалено файлов rate limit: ${data.rate_limit_files_cleared}\n` +
                    `Удалено файлов кеша этапа 2: ${data.term_extraction_cache_files_cleared || 0}`;
                alert(msg);
                btn.innerHTML = '<i class="bi bi-check-circle"></i> Очищено!';
                setTimeout(() => {
                    btn.disabled = false;
                    btn.innerHTML = originalHTML;
                }, 2000);
            } else {
                throw new Error(data.error || 'Неизвестная ошибка');
            }
        } catch (error) {
            alert('Ошибка: ' + error.message);
            btn.disabled = false;
            btn.innerHTML = originalHTML;
        }
    });
    
    document.getElementById('test-form').addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const question = document.getElementById('test-question').value.trim();
        if (!question) {
            alert('Введите вопрос');
            return;
        }
        
        const resultsDiv = document.getElementById('test-results');
        resultsDiv.style.display = 'block';
        
        // Скрываем все секции результатов
        document.getElementById('three-stage-stage1').style.display = 'none';
        document.getElementById('three-stage-stage2').style.display = 'none';
        document.getElementById('three-stage-stage3').style.display = 'none';
        document.getElementById('raw-text-section').style.display = 'none';
        activeSections.forEach((section, index) => {
            const stepNum = index + 2;
            const sectionDiv = document.getElementById(`section-stage-${stepNum}`);
            if (sectionDiv) {
                sectionDiv.style.display = 'none';
            }
        });
        
        // Тестируем выбранный шаг
        if (currentStep === 1) {
            // Шаг 1: Быстрый текстовый ответ
            document.getElementById('three-stage-stage1').style.display = 'block';
            document.getElementById('stage1-text').textContent = 'Загрузка...';
            document.getElementById('stage1-text').className = 'bg-secondary bg-opacity-25 p-2 mt-2 mb-0';
            document.getElementById('stage1-time').textContent = '...';
            
            const stage1Prompt = <?php echo json_encode($stage1_prompt, JSON_UNESCAPED_UNICODE); ?>;
            const stage1SystemPromptText = stage1Prompt || 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей кратко и по делу (2-3 предложения).';
            const stage1UserPromptText = 'Вопрос пользователя: ' + question;
            
            document.getElementById('stage1-system-prompt').textContent = stage1SystemPromptText;
            document.getElementById('stage1-user-prompt').textContent = stage1UserPromptText;
            
            // Формируем и показываем полный промпт
            const stage1FullPrompt = stage1SystemPromptText + '\n\n' + stage1UserPromptText;
            document.getElementById('stage1-full-prompt').textContent = stage1FullPrompt;
            
            try {
                const stage1RequestData = {
                    widget_key: widgetKey,
                    question: question,
                    _debug: true
                };
                
                // Показываем полный JSON запроса
                document.getElementById('stage1-request-json').textContent = JSON.stringify(stage1RequestData, null, 2);
                
                const stage1Response = await fetch(window.location.origin + '/api/query-stage1.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(stage1RequestData)
                });
                
                const stage1Data = await stage1Response.json();
                
                // Показываем полный JSON ответа
                document.getElementById('stage1-response-json').textContent = JSON.stringify(stage1Data, null, 2);
                
                if (stage1Data.error) {
                    document.getElementById('stage1-text').textContent = 'Ошибка: ' + stage1Data.error;
                    document.getElementById('stage1-text').className = 'bg-danger bg-opacity-25 p-2 mt-2 mb-0';
                } else {
                    document.getElementById('stage1-text').textContent = stage1Data.text || 'Нет ответа';
                    document.getElementById('stage1-text').className = 'bg-success bg-opacity-25 p-2 mt-2 mb-0';
                    document.getElementById('stage1-time').textContent = stage1Data.response_time_ms || 0;
                    
                    if (stage1Data.provider) {
                        const providerBadge = document.getElementById('stage1-provider');
                        let providerText = 'Claude';
                        let providerClass = 'badge bg-warning ms-2';
                        if (stage1Data.provider === 'openrouter') {
                            providerText = 'OpenRouter';
                            providerClass = 'badge bg-success ms-2';
                        } else if (stage1Data.provider === 'yandexgpt') {
                            providerText = 'Yandex GPT';
                            providerClass = 'badge bg-info ms-2';
                        }
                        providerBadge.textContent = providerText;
                        providerBadge.className = providerClass;
                    }
                }
            } catch (error) {
                document.getElementById('stage1-text').textContent = 'Ошибка: ' + error.message;
                document.getElementById('stage1-text').className = 'bg-danger bg-opacity-25 p-2 mt-2 mb-0';
            }
        } else {
            // Шаги 2-5: Запрос по конкретному разделу
            const sectionIndex = currentStep - 2;
            const section = activeSections[sectionIndex];
            const stepNum = currentStep;
            
            if (section) {
                const sectionDiv = document.getElementById(`section-stage-${stepNum}`);
                if (sectionDiv) {
                    sectionDiv.style.display = 'block';
                    document.getElementById(`section-${stepNum}-text`).textContent = 'Загрузка...';
                    document.getElementById(`section-${stepNum}-text`).className = 'bg-secondary bg-opacity-25 p-2 mt-2 mb-0';
                    document.getElementById(`section-${stepNum}-time`).textContent = '...';
                    
                    // Выбираем промпт для раздела
                    const sectionPromptField = 'stage_' + section + '_prompt';
                    let sectionPrompt = <?php echo json_encode($widget['stage_specialists_prompt'] ?? $widget['stage_services_prompt'] ?? $widget['stage_articles_prompt'] ?? $widget['stage_specializations_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>;
                    
                    // Определяем промпт для текущего раздела
                    const sectionPrompts = {
                        'specialists': <?php echo json_encode($widget['stage_specialists_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>,
                        'services': <?php echo json_encode($widget['stage_services_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>,
                        'articles': <?php echo json_encode($widget['stage_articles_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>,
                        'specializations': <?php echo json_encode($widget['stage_specializations_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>
                    };
                    
                    const stage3Prompt = sectionPrompts[section] || 
                                       <?php echo json_encode($widget['stage3_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?> || 
                                       <?php echo json_encode($widget['claude_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>;
                    
                    // Получаем категории из stage1, если они были определены
                    // Для этого нужно сначала получить ответ stage1
                    let categories = null;
                    let categoriesDebug = null;
                    try {
                        const stage1CheckResponse = await fetch(window.location.origin + '/api/query-stage1.php', {
                            method: 'POST',
                            headers: {'Content-Type': 'application/json'},
                            body: JSON.stringify({
                                widget_key: widgetKey,
                                question: question,
                                _debug: true
                            })
                        });
                        const stage1CheckData = await stage1CheckResponse.json();
                        if (stage1CheckData.categories) {
                            categories = stage1CheckData.categories;
                            categoriesDebug = stage1CheckData.categories_debug || null;
                        }
                    } catch (e) {
                        // Игнорируем ошибку, продолжаем без категорий
                    }
                    
                    try {
                        const sectionRequestData = {
                            widget_key: widgetKey,
                            question: question,
                            section_name: section,
                            _debug: true
                        };
                        
                        // Передаем категории, если они были определены
                        if (categories) {
                            sectionRequestData.categories = categories;
                            if (categoriesDebug) {
                                sectionRequestData.categories_debug = categoriesDebug;
                            }
                        }
                        
                        // Показываем полный JSON запроса
                        document.getElementById(`section-${stepNum}-request-json`).textContent = JSON.stringify(sectionRequestData, null, 2);
                        
                        const sectionResponse = await fetch(window.location.origin + '/api/query-section.php', {
                            method: 'POST',
                            headers: {'Content-Type': 'application/json'},
                            body: JSON.stringify(sectionRequestData)
                        });
                        
                        const sectionData = await sectionResponse.json();
                        
                        // Показываем полный JSON ответа
                        document.getElementById(`section-${stepNum}-response-json`).textContent = JSON.stringify(sectionData, null, 2);
                        
                        if (sectionData.error) {
                            document.getElementById(`section-${stepNum}-text`).textContent = 'Ошибка: ' + sectionData.error;
                            document.getElementById(`section-${stepNum}-text`).className = 'bg-danger bg-opacity-25 p-2 mt-2 mb-0';
                        } else {
                            document.getElementById(`section-${stepNum}-text`).textContent = sectionData.text || 'Нет ответа';
                            document.getElementById(`section-${stepNum}-text`).className = 'bg-success bg-opacity-25 p-2 mt-2 mb-0';
                            document.getElementById(`section-${stepNum}-data`).textContent = JSON.stringify(sectionData.data || {}, null, 2);
                            document.getElementById(`section-${stepNum}-time`).textContent = sectionData.response_time_ms || 0;
                            
                            if (sectionData.debug) {
                                // Показываем полный промпт из debug информации
                                if (sectionData.debug.full_prompt) {
                                    document.getElementById(`section-${stepNum}-full-prompt`).textContent = sectionData.debug.full_prompt;
                                }
                                
                                // Показываем system и user промпты отдельно
                                if (sectionData.debug.system_prompt) {
                                    document.getElementById(`section-${stepNum}-system-prompt`).textContent = sectionData.debug.system_prompt;
                                } else {
                                    document.getElementById(`section-${stepNum}-system-prompt`).textContent = stage3Prompt.substring(0, 500) + '...';
                                }
                                
                                if (sectionData.debug.user_prompt) {
                                    document.getElementById(`section-${stepNum}-user-prompt`).textContent = sectionData.debug.user_prompt.substring(0, 1000) + (sectionData.debug.user_prompt.length > 1000 ? '...' : '');
                                } else {
                                    document.getElementById(`section-${stepNum}-user-prompt`).textContent = 'Вопрос: ' + question;
                                }
                                
                                // Показываем информацию о категоризации (prefilter)
                                if (sectionData.debug.prefilter_info && sectionData.debug.prefilter_info.category_extraction_debug) {
                                    const categoryInfo = sectionData.debug.prefilter_info.category_extraction_debug;
                                    const categoryDiv = document.getElementById(`section-${stepNum}-category-info`);
                                    if (categoryDiv) {
                                        categoryDiv.style.display = 'block';
                                        
                                        if (categoryInfo.system_prompt) {
                                            document.getElementById(`section-${stepNum}-category-system-prompt`).textContent = categoryInfo.system_prompt;
                                        }
                                        if (categoryInfo.user_prompt) {
                                            document.getElementById(`section-${stepNum}-category-user-prompt`).textContent = categoryInfo.user_prompt;
                                        }
                                        if (categoryInfo.full_prompt) {
                                            document.getElementById(`section-${stepNum}-category-full-prompt`).textContent = categoryInfo.full_prompt;
                                        }
                                        if (categoryInfo.ai_response) {
                                            document.getElementById(`section-${stepNum}-category-ai-response`).textContent = categoryInfo.ai_response;
                                        }
                                        if (categoryInfo.raw_response) {
                                            try {
                                                const rawJson = JSON.parse(categoryInfo.raw_response);
                                                document.getElementById(`section-${stepNum}-category-raw-response`).textContent = JSON.stringify(rawJson, null, 2);
                                            } catch (e) {
                                                document.getElementById(`section-${stepNum}-category-raw-response`).textContent = categoryInfo.raw_response;
                                            }
                                        }
                                        if (sectionData.debug.prefilter_info.question_categories) {
                                            const categories = sectionData.debug.prefilter_info.question_categories;
                                            const escapeHtml = (text) => {
                                                const div = document.createElement('div');
                                                div.textContent = text;
                                                return div.innerHTML;
                                            };
                                            document.getElementById(`section-${stepNum}-category-categories`).innerHTML = 
                                                categories.map(cat => `<span class="badge bg-primary me-1 mb-1">${escapeHtml(cat)}</span>`).join(' ');
                                        }
                                        if (sectionData.debug.prefilter_info.expanded_categories) {
                                            const expanded = sectionData.debug.prefilter_info.expanded_categories;
                                            const escapeHtml = (text) => {
                                                const div = document.createElement('div');
                                                div.textContent = text;
                                                return div.innerHTML;
                                            };
                                            document.getElementById(`section-${stepNum}-category-expanded`).innerHTML = 
                                                expanded.map(cat => `<span class="badge bg-success me-1 mb-1">${escapeHtml(cat)}</span>`).join(' ');
                                        }
                                        const stats = sectionData.debug.prefilter_info;
                                        let statsHtml = '';
                                        if (stats.original_count !== undefined) {
                                            statsHtml += `<div>Исходное количество элементов: <strong>${stats.original_count}</strong></div>`;
                                        }
                                        if (stats.filtered_count !== undefined) {
                                            statsHtml += `<div>После фильтрации: <strong>${stats.filtered_count}</strong></div>`;
                                        }
                                        if (stats.filtered_ids && stats.filtered_ids.length > 0) {
                                            statsHtml += `<div>Отфильтрованные ID (первые 20): <code>${stats.filtered_ids.join(', ')}</code></div>`;
                                        }
                                        document.getElementById(`section-${stepNum}-category-stats`).innerHTML = statsHtml;
                                    }
                                } else {
                                    // Скрываем блок категоризации, если нет информации
                                    const categoryDiv = document.getElementById(`section-${stepNum}-category-info`);
                                    if (categoryDiv) {
                                        categoryDiv.style.display = 'none';
                                    }
                                }
                                
                                if (sectionData.debug.provider) {
                                    const providerBadge = document.getElementById(`section-${stepNum}-provider`);
                                    let providerText = 'Claude';
                                    let providerClass = 'badge bg-warning ms-2';
                                    if (sectionData.debug.provider === 'openrouter') {
                                        providerText = 'OpenRouter';
                                        providerClass = 'badge bg-success ms-2';
                                    } else if (sectionData.debug.provider === 'yandexgpt') {
                                        providerText = 'Yandex GPT';
                                        providerClass = 'badge bg-info ms-2';
                                    }
                                    providerBadge.textContent = providerText;
                                    providerBadge.className = providerClass;
                                }
                            } else {
                                // Если нет debug информации, показываем базовые промпты
                                document.getElementById(`section-${stepNum}-system-prompt`).textContent = stage3Prompt.substring(0, 500) + '...';
                                document.getElementById(`section-${stepNum}-user-prompt`).textContent = 'Вопрос: ' + question;
                            }
                        }
                    } catch (error) {
                        document.getElementById(`section-${stepNum}-text`).textContent = 'Ошибка: ' + error.message;
                        document.getElementById(`section-${stepNum}-text`).className = 'bg-danger bg-opacity-25 p-2 mt-2 mb-0';
                    }
                }
            }
        }
    });
    
    // Обработчик для кнопки очистки кеша
    document.getElementById('clear-cache-btn').addEventListener('click', async () => {
        if (!confirm('Вы уверены, что хотите очистить кеш для этого виджета?')) {
            return;
        }
        
        const btn = document.getElementById('clear-cache-btn');
        
        // Показываем индикатор загрузки для Stage 1
        document.getElementById('stage1-text').textContent = 'Ожидание ответа...';
        document.getElementById('stage1-text').className = 'bg-secondary bg-opacity-25 p-2 mt-2 mb-0';
        document.getElementById('stage1-time').textContent = '...';
        
        // Сбрасываем провайдер
        document.getElementById('stage1-provider').textContent = '';
        document.getElementById('stage1-provider').className = 'badge bg-secondary ms-2';
        
        try {
            // Запрос 1: Этап 1 - быстрый текстовый ответ
            const stage1Promise = fetch(window.location.origin + '/api/query-stage1.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    widget_key: '<?php echo $widget['widget_key']; ?>',
                    question: question,
                    _debug: true // Включаем режим отладки
                })
            }).then(async (response) => {
                // Клонируем response для возможности повторного чтения
                const responseClone = response.clone();
                
                let responseData;
                let responseText = '';
                try {
                    // Сначала пытаемся прочитать как текст
                    responseText = await responseClone.text();
                    // Затем пытаемся распарсить как JSON
                    try {
                        responseData = JSON.parse(responseText);
                    } catch (parseError) {
                        // Если не JSON, создаем объект с текстом ответа
                        responseData = {
                            error: 'Invalid JSON response',
                            raw_response: responseText.substring(0, 500)
                        };
                    }
                } catch (readError) {
                    throw new Error('Stage1 failed: HTTP ' + response.status + '. Не удалось прочитать ответ: ' + readError.message);
                }
                
                if (!response.ok) {
                    // Если есть ошибка в JSON ответе, используем её
                    const errorMsg = responseData.error || responseData.message || responseText.substring(0, 200) || 'HTTP ' + response.status;
                    throw new Error(errorMsg);
                }
                
                return responseData;
            }).catch(error => {
                console.error('Stage1 error:', error);
                return {error: error.message};
            });
            
            // Запрос 2: Этапы 2-3 - данные (специалисты, услуги и т.д.)
            const apiUrl = window.location.origin + '/api/query.php';
            
            // Устанавливаем timeout 60 секунд (увеличено для Yandex GPT и больших запросов)
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), 60000);
            
            const stage23Promise = fetch(apiUrl, {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                signal: controller.signal,
                body: JSON.stringify({
                    widget_key: '<?php echo $widget['widget_key']; ?>',
                    question: question,
                    _debug: true, // Специальный флаг для отладки
                    _no_cache: true // Отключаем кеш для тестирования (чтобы всегда выполнялся embedding поиск)
                })
            });
            
            // Обрабатываем этап 1 сразу как только получен
            stage1Promise.then((stage1Data) => {
                console.log('Stage1 response:', stage1Data);
                
                // Всегда показываем секцию Stage 1 (она уже показана, но на всякий случай)
                document.getElementById('three-stage-stage1').style.display = 'block';
                
                if (stage1Data.error) {
                    document.getElementById('stage1-text').textContent = 'Ошибка: ' + stage1Data.error;
                    document.getElementById('stage1-text').className = 'bg-danger bg-opacity-25 p-2 mt-2 mb-0';
                    document.getElementById('stage1-time').textContent = stage1Data.response_time_ms || 'N/A';
                    
                    // Показываем debug информацию об ошибке если есть
                    if (stage1Data.debug) {
                        const debugInfo = 'Файл: ' + (stage1Data.debug.file || 'N/A') + 
                            ', Строка: ' + (stage1Data.debug.line || 'N/A') + 
                            ', Сообщение: ' + (stage1Data.debug.message || stage1Data.error);
                        document.getElementById('stage1-text').textContent += '\n\nДетали: ' + debugInfo;
                    }
                } else {
                    document.getElementById('stage1-text').textContent = stage1Data.text || 'Нет ответа';
                    document.getElementById('stage1-text').className = 'bg-success bg-opacity-25 p-2 mt-2 mb-0';
                    document.getElementById('stage1-time').textContent = stage1Data.response_time_ms || 0;
                    
                    if (stage1Data.provider) {
                        const providerBadge = document.getElementById('stage1-provider');
                        let providerText = 'Claude';
                        let providerClass = 'badge bg-warning ms-2';
                        if (stage1Data.provider === 'openrouter') {
                            providerText = 'OpenRouter';
                            providerClass = 'badge bg-success ms-2';
                        } else if (stage1Data.provider === 'yandexgpt') {
                            providerText = 'Yandex GPT';
                            providerClass = 'badge bg-info ms-2';
                        }
                        providerBadge.textContent = providerText;
                        providerBadge.className = providerClass;
                    }
                    
                    // Отображаем debug информацию о запросах/ответах Yandex GPT
                    if (stage1Data.yandex_debug) {
                        const debugDiv = document.getElementById('stage1-yandex-debug');
                        debugDiv.style.display = 'block';
                        
                        const debug = stage1Data.yandex_debug;
                        
                        // Запрос
                        if (debug.request) {
                            document.getElementById('stage1-yandex-request').textContent = 
                                JSON.stringify(debug.request, null, 2);
                        }
                        
                        // Ответ
                        if (debug.response) {
                            const httpCode = debug.response.http_code || 'N/A';
                            const httpCodeBadge = document.getElementById('stage1-yandex-http-code');
                            httpCodeBadge.textContent = httpCode;
                            httpCodeBadge.className = httpCode === 200 ? 'badge bg-success' : 'badge bg-danger';
                            
                            // Парсим и форматируем ответ
                            let responseText = debug.response.response || '';
                            try {
                                const parsedResponse = JSON.parse(responseText);
                                responseText = JSON.stringify(parsedResponse, null, 2);
                            } catch (e) {
                                // Если не JSON, оставляем как есть
                            }
                            document.getElementById('stage1-yandex-response').textContent = responseText;
                        }
                        
                        // Ошибка
                        if (debug.error) {
                            document.getElementById('stage1-yandex-error').style.display = 'block';
                            document.getElementById('stage1-yandex-error-text').textContent = debug.error;
                        } else {
                            document.getElementById('stage1-yandex-error').style.display = 'none';
                        }
                    } else {
                        document.getElementById('stage1-yandex-debug').style.display = 'none';
                    }
                }
            }).catch((error) => {
                console.error('Stage1 promise error:', error);
                // Показываем секцию даже при ошибке промиса
                document.getElementById('three-stage-stage1').style.display = 'block';
                document.getElementById('stage1-text').textContent = 'Ошибка: ' + error.message;
                document.getElementById('stage1-text').className = 'bg-danger bg-opacity-25 p-2 mt-2 mb-0';
                document.getElementById('stage1-time').textContent = 'N/A';
            });
            
            // Обрабатываем этапы 2-3 после их завершения
            const response = await stage23Promise;
            clearTimeout(timeoutId);
            
            // Парсим ответ сначала
            let data;
            let responseText = '';
            try {
                // Клонируем response для возможности повторного чтения
                const responseClone = response.clone();
                
                try {
                    // Сначала читаем как текст
                    responseText = await responseClone.text();
                    // Затем пытаемся распарсить как JSON
                    try {
                        data = JSON.parse(responseText);
                    } catch (parseError) {
                        // Если не JSON, создаем объект с ошибкой
                        console.error("Failed to parse JSON response. Status:", response.status, "Text:", responseText.substring(0, 500));
                        data = {
                            error: 'Invalid JSON response',
                            raw_response: responseText.substring(0, 500),
                            http_status: response.status
                        };
                    }
                } catch (readError) {
                    throw new Error('HTTP ' + response.status + ': Не удалось прочитать ответ: ' + readError.message);
                }
            } catch (parseError) {
                // Если это уже наша ошибка, пробрасываем дальше
                if (parseError.message && parseError.message.includes('Не удалось')) {
                    throw parseError;
                }
                throw new Error('HTTP ' + response.status + ': Не удалось прочитать ответ: ' + parseError.message);
            }
            
            // Проверяем статус после парсинга
            if (!response.ok) {
                // Показываем детальную ошибку если она есть
                const errorMsg = data.error || response.statusText;
                const errorType = data.type || 'unknown';
                
                // Специальная обработка для Rate Limit
                if (response.status === 429 || errorType === 'rate_limit') {
                    document.getElementById('raw-response').innerHTML = 
                        '<div class="alert alert-warning mb-0">' +
                        '<h6><i class="bi bi-exclamation-triangle"></i> Rate Limit (429)</h6>' +
                        '<p class="mb-2"><strong>Причина:</strong> Превышен лимит запросов к Claude API (100,000 токенов/минуту)</p>' +
                        '<p class="mb-2"><strong>Что делать:</strong></p>' +
                        '<ul class="mb-0">' +
                        '<li>Подождите 5-10 минут перед следующим запросом</li>' +
                        '<li>Уменьшите количество данных в промпте (сейчас ~87,000 токенов из 1855 услуг)</li>' +
                        '<li>Рекомендуется ограничить до 200-300 услуг (~10,000 токенов)</li>' +
                        '</ul>' +
                        '<p class="mt-2 mb-0"><strong>Сообщение:</strong> ' + errorMsg + '</p>' +
                        (data.debug ? '<details class="mt-2"><summary style="cursor: pointer;">Детали ошибки (debug)</summary><pre class="mt-2 mb-0" style="font-size: 11px; white-space: pre-wrap;">' + JSON.stringify(data.debug, null, 2) + '</pre></details>' : '') +
                        '</div>';
                    document.getElementById('parsed-text').textContent = errorMsg;
                    return;
                }
                
                // Показываем детальную ошибку с debug информацией
                let errorDisplay = 'HTTP ' + response.status + ': ' + errorMsg;
                if (data.debug) {
                    errorDisplay += '\n\nДетали ошибки:\n';
                    errorDisplay += 'Файл: ' + (data.debug.file || 'N/A') + '\n';
                    errorDisplay += 'Строка: ' + (data.debug.line || 'N/A') + '\n';
                    errorDisplay += 'Сообщение: ' + (data.debug.message || errorMsg) + '\n';
                    if (data.debug.trace) {
                        errorDisplay += '\nTrace:\n' + data.debug.trace;
                    }
                }
                
                document.getElementById('raw-response').textContent = errorDisplay;
                document.getElementById('parsed-text').textContent = errorMsg;
                throw new Error('HTTP ' + response.status + ': ' + errorMsg);
            }
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            // Отображаем информацию о трёхэтапном режиме
            if (data.debug && data.debug.three_stage_search) {
                const threeStage = data.debug.three_stage_search || {};
                
                // Этап 2: Извлечение терминов
                if (threeStage.stage2) {
                    document.getElementById('three-stage-stage2').style.display = 'block';
                    document.getElementById('stage2-question').textContent = threeStage.stage2.question || '';
                    document.getElementById('stage2-time').textContent = threeStage.stage2.time_ms || 0;
                    
                    // Показываем промпт для этапа 2
                    const stage2Prompt = <?php echo json_encode($stage2_prompt, JSON_UNESCAPED_UNICODE); ?>;
                    if (stage2Prompt) {
                        // Используем кастомный промпт из настроек
                        const stage2PromptWithQuestion = stage2Prompt.replace(/{QUESTION}/g, threeStage.stage2.question || '');
                        // Разделяем на system и user промпты (если есть разделение)
                        // Ищем первое вхождение "ВАЖНО:" или "Твоя задача" для разделения
                        const importantIndex = stage2PromptWithQuestion.indexOf('ВАЖНО:');
                        if (importantIndex > 0) {
                            document.getElementById('stage2-system-prompt').textContent = stage2PromptWithQuestion.substring(0, importantIndex).trim();
                            document.getElementById('stage2-user-prompt').textContent = stage2PromptWithQuestion.substring(importantIndex).trim();
                        } else {
                            // Если нет явного разделения, показываем весь промпт в user prompt
                            document.getElementById('stage2-system-prompt').textContent = '';
                            document.getElementById('stage2-user-prompt').textContent = stage2PromptWithQuestion;
                        }
                    } else {
                        // Дефолтный промпт
                        const stage2SystemPrompt = 'Ты - помощник для извлечения медицинских терминов из вопроса пользователя.';
                        const stage2UserPrompt = 'ВАЖНО: Верни ТОЛЬКО валидный JSON, без дополнительного текста до или после JSON.\n\n' +
                            'Твоя задача - максимально полно извлечь все релевантные медицинские термины из вопроса пользователя, включая:\n' +
                            '- Прямо упомянутые симптомы, заболевания, состояния\n' +
                            '- Связанные симптомы и состояния, которые могут быть связаны с упомянутыми\n' +
                            '- Все возможные специализации и специалистов, которые могут помочь\n' +
                            '- Типы услуг, диагностики и лечения, которые могут быть релевантны\n' +
                            '- Ключевые слова для поиска, включая синонимы, медицинские термины и разговорные названия\n\n' +
                            'Думай широко: если пользователь упоминает симптом, подумай о возможных причинах, связанных состояниях и специалистах, которые могут помочь.\n\n' +
                            'Из вопроса извлеки и верни в формате JSON:\n' +
                            '{\n' +
                            '  "symptoms": [массив всех симптомов, включая связанные и возможные],\n' +
                            '  "specializations": [массив всех релевантных специализаций и специалистов],\n' +
                            '  "service_types": [массив всех типов услуг, диагностики и лечения],\n' +
                            '  "keywords": [объединенный массив всех релевантных ключевых слов, включая синонимы и связанные термины]\n' +
                            '}\n\n' +
                            'Вопрос пользователя: ' + (threeStage.stage2.question || '') + '\n\n' +
                            'Верни ТОЛЬКО JSON объект, без объяснений и дополнительного текста.';
                        document.getElementById('stage2-system-prompt').textContent = stage2SystemPrompt;
                        document.getElementById('stage2-user-prompt').textContent = stage2UserPrompt;
                    }
                    
                    // Показываем информацию о кеше
                    if (threeStage.stage2.cached) {
                        document.getElementById('stage2-cached').style.display = 'inline';
                        if (threeStage.stage2.cache_timestamp) {
                            const cacheDate = new Date(threeStage.stage2.cache_timestamp * 1000);
                            if (cacheDate && !isNaN(cacheDate.getTime())) {
                                document.getElementById('stage2-cached').title = 'Закешировано: ' + cacheDate.toLocaleString('ru-RU');
                            }
                        }
                    } else {
                        document.getElementById('stage2-cached').style.display = 'none';
                    }
                    
                    // Показываем все извлеченные термины (объединенные из всех массивов)
                    const allTerms = threeStage.stage2.all_extracted_terms || threeStage.stage2.keywords || [];
                    document.getElementById('stage2-keywords').textContent = JSON.stringify(allTerms, null, 2);
                    document.getElementById('stage2-full-response').textContent = JSON.stringify(threeStage.stage2.medical_terms || {}, null, 2);
                    
                    // Показываем провайдера
                    if (data.debug.provider) {
                        const providerBadge = document.getElementById('stage2-provider');
                        let providerText = 'Claude';
                        let providerClass = 'badge bg-warning ms-2';
                        if (data.debug.provider === 'openrouter') {
                            providerText = 'OpenRouter';
                            providerClass = 'badge bg-success ms-2';
                        } else if (data.debug.provider === 'yandexgpt') {
                            providerText = 'Yandex GPT';
                            providerClass = 'badge bg-info ms-2';
                        }
                        providerBadge.textContent = providerText;
                        providerBadge.className = providerClass;
                    }
                    
                    // Отображаем debug информацию о запросах/ответах Yandex GPT для этапа 2
                    if (threeStage.stage2 && threeStage.stage2.yandex_debug) {
                        const debugDiv = document.getElementById('stage2-yandex-debug');
                        debugDiv.style.display = 'block';
                        
                        const debug = threeStage.stage2.yandex_debug;
                        
                        // Запрос
                        if (debug.request) {
                            document.getElementById('stage2-yandex-request').textContent = 
                                JSON.stringify(debug.request, null, 2);
                        }
                        
                        // Ответ
                        if (debug.response) {
                            const httpCode = debug.response.http_code || 'N/A';
                            const httpCodeBadge = document.getElementById('stage2-yandex-http-code');
                            httpCodeBadge.textContent = httpCode;
                            httpCodeBadge.className = httpCode === 200 ? 'badge bg-success' : 'badge bg-danger';
                            
                            // Парсим и форматируем ответ
                            let responseText = debug.response.response || '';
                            try {
                                const parsedResponse = JSON.parse(responseText);
                                responseText = JSON.stringify(parsedResponse, null, 2);
                            } catch (e) {
                                // Если не JSON, оставляем как есть
                            }
                            document.getElementById('stage2-yandex-response').textContent = responseText;
                        }
                        
                        // Ошибка
                        if (debug.error) {
                            document.getElementById('stage2-yandex-error').style.display = 'block';
                            document.getElementById('stage2-yandex-error-text').textContent = debug.error;
                        } else {
                            document.getElementById('stage2-yandex-error').style.display = 'none';
                        }
                    } else {
                        document.getElementById('stage2-yandex-debug').style.display = 'none';
                    }
                }
                
                // Этап 3: Основной запрос
                if (threeStage.stage3) {
                    document.getElementById('three-stage-stage3').style.display = 'block';
                    const sizeBefore = threeStage.stage3.prompt_size_before ?? 0;
                    const tokensBefore = threeStage.stage3.prompt_tokens_before ?? 0;
                    const sizeAfter = threeStage.stage3.prompt_size_after ?? 0;
                    const tokensAfter = threeStage.stage3.prompt_tokens_after ?? 0;
                    const reduction = threeStage.stage3.reduction_percent ?? 0;
                    
                    document.getElementById('stage3-size-before').textContent = (sizeBefore || 0).toLocaleString();
                    document.getElementById('stage3-tokens-before').textContent = (tokensBefore || 0).toLocaleString();
                    document.getElementById('stage3-size-after').textContent = (sizeAfter || 0).toLocaleString();
                    document.getElementById('stage3-tokens-after').textContent = (tokensAfter || 0).toLocaleString();
                    document.getElementById('stage3-reduction').textContent = reduction;
                    
                    // Показываем промпты
                    const stage3Prompt = <?php echo json_encode($stage3_prompt, JSON_UNESCAPED_UNICODE); ?>;
                    const stage3SystemPromptText = stage3Prompt || <?php echo json_encode($widget['claude_prompt'] ?? '', JSON_UNESCAPED_UNICODE); ?>;
                    
                    if (threeStage.stage3.prompt_before) {
                        const promptBefore = threeStage.stage3.prompt_before;
                        const promptParts = promptBefore.split('\n\n');
                        if (promptParts.length >= 2) {
                            document.getElementById('stage3-system-prompt').textContent = promptParts[0];
                            document.getElementById('stage3-user-prompt').textContent = promptParts.slice(1).join('\n\n');
                        } else {
                            document.getElementById('stage3-system-prompt').textContent = stage3SystemPromptText || promptBefore.substring(0, 500) + '...';
                            document.getElementById('stage3-user-prompt').textContent = promptBefore.length > 500 ? 'См. полный промпт ниже' : promptBefore;
                        }
                        document.getElementById('stage3-prompt-before').textContent = promptBefore;
                    } else {
                        // Если нет prompt_before, показываем system prompt из настроек
                        document.getElementById('stage3-system-prompt').textContent = stage3SystemPromptText || 'Промпт не указан';
                    }
                    
                    if (threeStage.stage3.prompt_after) {
                        document.getElementById('stage3-prompt-after').textContent = threeStage.stage3.prompt_after;
                        // Обновляем user prompt если есть оптимизированный промпт
                        const promptAfter = threeStage.stage3.prompt_after;
                        const promptParts = promptAfter.split('\n\n');
                        if (promptParts.length >= 2) {
                            document.getElementById('stage3-user-prompt').textContent = promptParts.slice(1).join('\n\n');
                        }
                    }
                    
                    // Показываем информацию об Embedding поиске
                    if (threeStage.stage3.embedding_search) {
                        const embeddingInfo = document.createElement('div');
                        embeddingInfo.className = 'mb-3 p-2 bg-primary bg-opacity-25 rounded';
                        embeddingInfo.style.borderLeft = '3px solid #0d6efd';
                        const emb = threeStage.stage3.embedding_search;
                        let embeddingHtml = '<h6><i class="bi bi-vector-pen"></i> Embedding поиск:</h6>';
                        embeddingHtml += '<div class="mt-2">';
                        embeddingHtml += '<small><strong>Модель:</strong> <code>' + (emb.model || 'intfloat/multilingual-e5-large') + '</code></small><br>';
                        embeddingHtml += '<small><strong>Время создания embedding:</strong> ' + (emb.embedding_time_ms || 0) + ' мс</small><br>';
                        if (emb.candidates_by_section && Object.keys(emb.candidates_by_section).length > 0) {
                            embeddingHtml += '<div class="mt-2"><strong>Кандидаты для поиска:</strong></div>';
                            Object.entries(emb.candidates_by_section).forEach(([section, count]) => {
                                embeddingHtml += '<small>' + section + ': <span class="badge bg-secondary">' + count + ' кандидатов</span></small><br>';
                            });
                        }
                        if (emb.results_by_section && Object.keys(emb.results_by_section).length > 0) {
                            embeddingHtml += '<div class="mt-2"><strong>Результаты embedding поиска:</strong></div>';
                            Object.entries(emb.results_by_section).forEach(([section, results]) => {
                                if (results && results.length > 0) {
                                    const topSimilarity = results[0].similarity || 0;
                                    embeddingHtml += '<small>' + section + ': <span class="badge bg-success">' + results.length + ' записей</span> (макс. similarity: <span class="badge bg-info">' + (topSimilarity * 100).toFixed(1) + '%</span>)</small><br>';
                                }
                            });
                        }
                        embeddingHtml += '</div>';
                        embeddingInfo.innerHTML = embeddingHtml;
                        document.getElementById('three-stage-stage3').querySelector('.card-body').insertBefore(embeddingInfo, document.getElementById('three-stage-stage3').querySelector('.card-body').querySelector('.mb-3:last-of-type'));
                    }
                    
                    // Показываем информацию о категориях и найденных записях
                    if (threeStage.stage3.relevant_categories && Object.keys(threeStage.stage3.relevant_categories).length > 0) {
                        const categoriesInfo = document.createElement('div');
                        categoriesInfo.className = 'mb-3 p-2 bg-info bg-opacity-25 rounded';
                        categoriesInfo.style.borderLeft = '3px solid #0dcaf0';
                        let categoriesHtml = '<strong>Релевантные категории:</strong><br>';
                        categoriesHtml += '<small>Всего найдено записей: <span class="badge bg-success">' + (threeStage.stage3.total_items_found || 0) + '</span></small><br>';
                        Object.entries(threeStage.stage3.relevant_categories).forEach(([section, categories]) => {
                            const count = threeStage.stage3.items_found && threeStage.stage3.items_found[section] ? threeStage.stage3.items_found[section] : 0;
                            categoriesHtml += '<div class="mt-2"><strong>' + section + ':</strong> <span class="badge bg-secondary">' + count + ' записей</span><br>';
                            categoriesHtml += '<small>' + categories.map(cat => '<span class="badge bg-primary me-1">' + cat + '</span>').join('') + '</small></div>';
                        });
                        categoriesInfo.innerHTML = categoriesHtml;
                        document.getElementById('three-stage-stage3').querySelector('.card-body').insertBefore(categoriesInfo, document.getElementById('three-stage-stage3').querySelector('.card-body').querySelector('.mb-3:last-of-type'));
                    } else if (threeStage.stage3.total_items_found > 0) {
                        const itemsInfo = document.createElement('div');
                        itemsInfo.className = 'mb-3 p-2 bg-success bg-opacity-25 rounded';
                        itemsInfo.style.borderLeft = '3px solid #198754';
                        let itemsHtml = '<strong>Результаты поиска:</strong><br>';
                        itemsHtml += '<small>Всего найдено записей: <span class="badge bg-success">' + threeStage.stage3.total_items_found + '</span></small><br>';
                        if (threeStage.stage3.items_found) {
                            Object.entries(threeStage.stage3.items_found).forEach(([section, count]) => {
                                itemsHtml += '<div class="mt-1"><strong>' + section + ':</strong> <span class="badge bg-secondary">' + count + ' записей</span></div>';
                            });
                        }
                        itemsInfo.innerHTML = itemsHtml;
                        document.getElementById('three-stage-stage3').querySelector('.card-body').insertBefore(itemsInfo, document.getElementById('three-stage-stage3').querySelector('.card-body').querySelector('.mb-3:last-of-type'));
                    }
                    
                    // Показываем ответ этапа 3
                    if (data.debug && data.debug.claude_raw_response_text) {
                        document.getElementById('stage3-response').textContent = data.debug.claude_raw_response_text;
                    } else if (data.text) {
                        document.getElementById('stage3-response').textContent = data.text;
                    }
                    
                    // Показываем время выполнения этапа 3 (общее время минус время этапа 2)
                    if (data.debug && data.debug.response_time_ms && threeStage.stage2 && threeStage.stage2.time_ms) {
                        const stage3Time = data.debug.response_time_ms - threeStage.stage2.time_ms;
                        document.getElementById('stage3-time').textContent = stage3Time > 0 ? stage3Time : data.debug.response_time_ms;
                    } else if (data.debug && data.debug.response_time_ms) {
                        document.getElementById('stage3-time').textContent = data.debug.response_time_ms;
                    }
                    
                    // Показываем провайдера
                    if (data.debug.provider) {
                        const providerBadge = document.getElementById('stage3-provider');
                        let providerText = 'Claude';
                        let providerClass = 'badge bg-warning ms-2';
                        if (data.debug.provider === 'openrouter') {
                            providerText = 'OpenRouter';
                            providerClass = 'badge bg-success ms-2';
                        } else if (data.debug.provider === 'yandexgpt') {
                            providerText = 'Yandex GPT';
                            providerClass = 'badge bg-info ms-2';
                        }
                        providerBadge.textContent = providerText;
                        providerBadge.className = providerClass;
                    }
                    
                    // Отображаем debug информацию о запросах/ответах Yandex GPT для этапа 3
                    if (threeStage.stage3 && threeStage.stage3.yandex_debug) {
                        const debugDiv = document.getElementById('stage3-yandex-debug');
                        debugDiv.style.display = 'block';
                        
                        const debug = threeStage.stage3.yandex_debug;
                        
                        // Запрос
                        if (debug.request) {
                            document.getElementById('stage3-yandex-request').textContent = 
                                JSON.stringify(debug.request, null, 2);
                        }
                        
                        // Ответ
                        if (debug.response) {
                            const httpCode = debug.response.http_code || 'N/A';
                            const httpCodeBadge = document.getElementById('stage3-yandex-http-code');
                            httpCodeBadge.textContent = httpCode;
                            httpCodeBadge.className = httpCode === 200 ? 'badge bg-success' : 'badge bg-danger';
                            
                            // Парсим и форматируем ответ
                            let responseText = debug.response.response || '';
                            try {
                                const parsedResponse = JSON.parse(responseText);
                                responseText = JSON.stringify(parsedResponse, null, 2);
                            } catch (e) {
                                // Если не JSON, оставляем как есть
                            }
                            document.getElementById('stage3-yandex-response').textContent = responseText;
                        }
                        
                        // Ошибка
                        if (debug.error) {
                            document.getElementById('stage3-yandex-error').style.display = 'block';
                            document.getElementById('stage3-yandex-error-text').textContent = debug.error;
                        } else {
                            document.getElementById('stage3-yandex-error').style.display = 'none';
                        }
                    } else {
                        document.getElementById('stage3-yandex-debug').style.display = 'none';
                    }
                    
                    // Показываем стоимость если есть
                    if (threeStage.stage3.cost) {
                        const cost = threeStage.stage3.cost;
                        const costInfo = document.createElement('div');
                        costInfo.className = 'mt-3 p-2 bg-secondary rounded';
                        costInfo.innerHTML = '<strong>Стоимость запроса:</strong> $' + cost.total_cost.toFixed(6) + 
                            ' (вход: $' + cost.input_cost.toFixed(6) + ', выход: $' + cost.output_cost.toFixed(6) + ')';
                        document.getElementById('three-stage-stage3').querySelector('.card-body').appendChild(costInfo);
                    }
                }
                
                // Показываем ошибку если была
                if (threeStage.error) {
                    const errorDiv = document.createElement('div');
                    errorDiv.className = 'alert alert-danger';
                    errorDiv.innerHTML = '<strong>Ошибка трёхэтапного режима:</strong> ' + threeStage.error + '<br>Использован fallback на обычный режим.';
                    document.getElementById('test-results').insertBefore(errorDiv, document.getElementById('test-results').firstChild);
                }
            }
            
            // Показываем полный debug ответ для отладки
            if (data.debug) {
                const debugDiv = document.createElement('div');
                debugDiv.className = 'alert alert-secondary mb-3';
                debugDiv.innerHTML = '<h6><i class="bi bi-bug"></i> Debug информация (полный ответ):</h6>' +
                    '<pre style="max-height: 300px; overflow-y: auto; font-size: 11px; white-space: pre-wrap; word-wrap: break-word;">' + 
                    JSON.stringify(data.debug, null, 2) + '</pre>';
                document.getElementById('test-results').insertBefore(debugDiv, document.getElementById('test-results').firstChild);
            }
            
            // Показываем информацию о провайдере
            if (data.debug && data.debug.provider) {
                const provider = data.debug.provider;
                const providerModel = data.debug.provider_model || '';
                let providerName = 'Claude';
                let providerClass = 'alert alert-warning mb-3';
                let providerIcon = 'info-circle';
                
                if (provider === 'openrouter') {
                    providerName = 'OpenRouter';
                    providerClass = 'alert alert-success mb-3';
                    providerIcon = 'check-circle';
                } else if (provider === 'yandexgpt') {
                    providerName = 'Yandex GPT';
                    providerClass = 'alert alert-info mb-3';
                    providerIcon = 'info-circle';
                }
                
                const providerDiv = document.createElement('div');
                providerDiv.className = providerClass;
                let cacheInfo = '';
                if (provider === 'claude' && data.debug && data.debug.cache_stats) {
                    const cacheReadTokens = (data.debug.cache_stats.cache_read_tokens || 0);
                    cacheInfo = '<div class="mt-2"><small>Кеширование: ' + 
                        (cacheReadTokens > 0 ? 
                            'Использован кеш (' + cacheReadTokens.toLocaleString() + ' токенов)' : 
                            'Кеш не использован') + 
                        '</small></div>';
                }
                providerDiv.innerHTML = '<h6><i class="bi bi-' + providerIcon + '"></i> Использованный провайдер: <strong>' + providerName + '</strong></h6>' +
                    '<div>Модель: <code>' + providerModel + '</code></div>' + cacheInfo;
                document.getElementById('test-results').insertBefore(providerDiv, document.getElementById('test-results').firstChild);
                
                // Показываем информацию об ошибках OpenRouter, если был fallback на Claude
                if (provider === 'claude' && data.debug.openrouter_debug && !data.debug.openrouter_debug.success) {
                    const openrouterErrorDiv = document.createElement('div');
                    openrouterErrorDiv.className = 'alert alert-danger mb-3';
                    let errorHtml = '<h6><i class="bi bi-exclamation-triangle"></i> Ошибки OpenRouter (переключение на Claude):</h6>';
                    errorHtml += '<div class="mt-2"><strong>Причина:</strong> ' + (data.debug.openrouter_debug.fallback_reason || 'Неизвестная ошибка') + '</div>';
                    errorHtml += '<div class="mt-2"><strong>Попыток:</strong> ' + data.debug.openrouter_debug.attempts + '</div>';
                    if (data.debug.openrouter_debug.errors && data.debug.openrouter_debug.errors.length > 0) {
                        errorHtml += '<div class="mt-2"><strong>Ошибки:</strong></div>';
                        errorHtml += '<ul class="mb-0">';
                        data.debug.openrouter_debug.errors.forEach(function(error, index) {
                            const errorTime = error.timestamp ? new Date(error.timestamp * 1000).toLocaleTimeString('ru-RU') : '';
                            errorHtml += '<li><strong>Попытка ' + error.attempt + ':</strong> ' + error.error + 
                                (errorTime ? ' <small class="text-muted">(' + errorTime + ')</small>' : '') + '</li>';
                            
                            // Показываем полный ответ OpenRouter, если он есть
                            if (error.openrouter_response) {
                                errorHtml += '<div class="mt-2 ms-4">';
                                errorHtml += '<strong>Полный ответ от OpenRouter API:</strong>';
                                errorHtml += '<pre class="bg-dark text-light p-2 mt-1 mb-2" style="max-height: 300px; overflow-y: auto; font-size: 11px; white-space: pre-wrap; word-wrap: break-word;">';
                                if (error.openrouter_response.full_api_response) {
                                    errorHtml += 'HTTP ' + (error.openrouter_response.http_code || 'N/A') + ':\n';
                                    errorHtml += error.openrouter_response.full_api_response;
                                } else if (error.openrouter_response.raw_answer) {
                                    errorHtml += 'Raw answer:\n' + error.openrouter_response.raw_answer;
                                }
                                errorHtml += '</pre>';
                                errorHtml += '</div>';
                            }
                        });
                        errorHtml += '</ul>';
                    }
                    openrouterErrorDiv.innerHTML = errorHtml;
                    document.getElementById('test-results').insertBefore(openrouterErrorDiv, document.getElementById('test-results').firstChild);
                }
            } else if (data.debug) {
                // Если провайдер не указан в debug, но debug есть - показываем предупреждение
                const providerDiv = document.createElement('div');
                providerDiv.className = 'alert alert-warning mb-3';
                providerDiv.innerHTML = '<h6><i class="bi bi-exclamation-triangle"></i> Информация о провайдере отсутствует</h6>' +
                    '<div>Провайдер не указан в debug ответе. Проверьте логи сервера.</div>';
                document.getElementById('test-results').insertBefore(providerDiv, document.getElementById('test-results').firstChild);
            }
            
            // Показываем стоимость запроса если есть
            if (data.debug && data.debug.cost) {
                const cost = data.debug.cost;
                const inputTokens = cost.input_tokens ?? 0;
                const outputTokens = cost.output_tokens ?? 0;
                const inputCost = cost.input_cost ?? 0;
                const outputCost = cost.output_cost ?? 0;
                const totalCost = cost.total_cost ?? 0;
                
                const costDiv = document.createElement('div');
                costDiv.className = 'alert alert-info mb-3';
                costDiv.innerHTML = '<h6><i class="bi bi-calculator"></i> Стоимость запроса:</h6>' +
                    '<div>Входные токены: ' + inputTokens.toLocaleString() + ' → $' + inputCost.toFixed(6) + 
                    (cost.use_cache ? ' (с кешированием ' + cost.cache_ttl + ')' : ' (без кеширования)') + '</div>' +
                    '<div>Выходные токены: ' + outputTokens.toLocaleString() + ' → $' + outputCost.toFixed(6) + '</div>' +
                    '<div class="mt-2"><strong>Общая стоимость: $' + totalCost.toFixed(6) + '</strong></div>';
                document.getElementById('test-results').insertBefore(costDiv, document.getElementById('test-results').firstChild);
            }
            
            // 0. Сырой текст от AI (если есть)
            if (data.debug && data.debug.claude_raw_response_text) {
                document.getElementById('claude-raw-text').textContent = data.debug.claude_raw_response_text;
            } else {
                document.getElementById('claude-raw-text').textContent = 'Debug mode не включен или данные недоступны';
            }
            
            // 1. Сырой ответ
            document.getElementById('raw-response').textContent = 
                JSON.stringify(data.debug ? data.debug.claude_raw_response : data, null, 2);
            
            // 2. Распарсенный
            document.getElementById('parsed-text').textContent = data.text || data.data?.text || 'Нет ответа';
            document.getElementById('parsed-ids').textContent = 
                JSON.stringify(data.debug ? data.debug.data_ids : {}, null, 2);
            
            // 3. Данные из БД
            document.getElementById('db-data').textContent = 
                JSON.stringify(data.debug ? data.debug.db_data : {}, null, 2);
            
            // 4. Итоговый JSON
            document.getElementById('final-json').textContent = 
                JSON.stringify({text: data.text || data.data?.text, data: data.data}, null, 2);
                
        } catch (error) {
            if (error.name === 'AbortError') {
                document.getElementById('raw-response').textContent = 
                    'Ошибка: Timeout 60 секунд истек. Возможно, слишком много данных или сервис работает медленно.';
            } else {
                document.getElementById('raw-response').textContent = 
                    'Ошибка: ' + error.message;
            }
            document.getElementById('parsed-text').textContent = 
                'Ошибка: ' + error.message;
        }
    });
    </script>
</body>
</html>

