<?php
$page_title = 'Создать виджет';
require_once __DIR__ . '/includes/header.php';

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $widget_key = trim($_POST['widget_key'] ?? '');
    $site_domain = trim($_POST['site_domain'] ?? '');
    $sitemap_url = trim($_POST['sitemap_url'] ?? '');
    $claude_prompt = trim($_POST['claude_prompt'] ?? '');
    $client_id = !empty($_POST['client_id']) ? (int)$_POST['client_id'] : null;
    
    // Валидация
    if (empty($name) || empty($widget_key) || empty($site_domain) || empty($sitemap_url)) {
        $error = 'Заполните все обязательные поля';
    } elseif (!preg_match('/^[a-z0-9-]+$/', $widget_key)) {
        $error = 'Widget Key может содержать только латинские буквы, цифры и дефис';
    } else {
        try {
            $db = getDatabase();
            $db->beginTransaction();
            
            // Проверяем уникальность widget_key
            $stmt = $db->prepare("SELECT id FROM widgets WHERE widget_key = ?");
            $stmt->execute([$widget_key]);
            if ($stmt->fetch()) {
                throw new Exception('Widget Key уже используется');
            }
            
            // Создаем виджет
            $stmt = $db->prepare("
                INSERT INTO widgets (client_id, widget_key, name, site_domain, sitemap_url, claude_prompt, claude_model)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $client_id,
                $widget_key,
                $name,
                $site_domain,
                $sitemap_url,
                $claude_prompt ?: 'Ты - AI-ассистент медицинской клиники. Помогай пользователям находить нужных специалистов и услуги.',
                CLAUDE_MODEL
            ]);
            $widget_id = $db->lastInsertId();
            
            // Создаем настройки виджета
            $stmt = $db->prepare("
                INSERT INTO widget_settings (widget_id, welcome_message, suggestions, allowed_domains)
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute([
                $widget_id,
                'Привет! Я ваш личный AI-помощник',
                json_encode(['У меня болит спина', 'Записаться к врачу', 'Стоимость услуг'], JSON_UNESCAPED_UNICODE),
                json_encode([$site_domain], JSON_UNESCAPED_UNICODE)
            ]);
            
            // Создаем 4 дефолтных раздела с полями
            $sections = [
                'specialists' => [
                    'url_pattern' => '/specialists/',
                    'fields' => [
                        ['name', '.doctor-name, h1', 1, 1, 0],
                        ['photo', 'img.doctor-photo, .avatar img', 0, 1, 1],
                        ['description', '.doctor-bio, .about', 1, 1, 2],
                        ['url', 'current', 0, 1, 3]
                    ]
                ],
                'services' => [
                    'url_pattern' => '/services/',
                    'fields' => [
                        ['name', 'h1, .service-title', 1, 1, 0],
                        ['price', '.price, .cost', 1, 1, 1],
                        ['description', '.service-description, meta[name="description"]', 1, 1, 2],
                        ['url', 'current', 0, 1, 3]
                    ]
                ],
                'articles' => [
                    'url_pattern' => '/articles/',
                    'fields' => [
                        ['title', 'h1, .article-title', 1, 1, 0],
                        ['content', '.article-content, article', 0, 0, 1],
                        ['date', '.article-date, time', 0, 1, 2],
                        ['url', 'current', 0, 1, 3]
                    ]
                ],
                'specializations' => [
                    'url_pattern' => '/specializations/',
                    'fields' => [
                        ['name', 'h1, .specialization-title', 1, 1, 0],
                        ['description', '.specialization-description, meta[name="description"]', 1, 1, 1],
                        ['url', 'current', 0, 1, 2]
                    ]
                ]
            ];
            
            foreach ($sections as $section_name => $section_data) {
                // Создаем раздел
                $stmt = $db->prepare("
                    INSERT INTO widget_sections (widget_id, section_name, url_pattern, test_limit)
                    VALUES (?, ?, ?, 10)
                ");
                $stmt->execute([$widget_id, $section_name, $section_data['url_pattern']]);
                $section_id = $db->lastInsertId();
                
                // Создаем поля
                foreach ($section_data['fields'] as $field) {
                    $stmt = $db->prepare("
                        INSERT INTO section_fields (section_id, field_name, css_selectors, use_in_prompt, show_in_widget, field_order)
                        VALUES (?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$section_id, $field[0], $field[1], $field[2], $field[3], $field[4]]);
                }
            }
            
            // Загружаем sitemap (асинхронно, позже)
            // Пока просто логируем
            logInfo("Widget created: $widget_key (ID: $widget_id)");
            
            // Создаем папку для кастомных шаблонов
            $customPath = WIDGET_ROOT . '/widget/custom/' . $widget_key;
            if (!is_dir($customPath)) {
                mkdir($customPath, 0755, true);
            }
            
            // Копируем дефолтные файлы
            $defaultTemplate = WIDGET_ROOT . '/widget/template.html';
            $defaultStyles = WIDGET_ROOT . '/widget/widget.css';
            
            if (file_exists($defaultTemplate)) {
                copy($defaultTemplate, $customPath . '/template.html');
            }
            if (file_exists($defaultStyles)) {
                copy($defaultStyles, $customPath . '/styles.css');
            }
            
            // Создаем пустой custom.js с комментариями
            $customJsContent = <<<'JS'
/**
 * Кастомная логика виджета
 * Эта функция полностью заменяет стандартную логику widget.js
 * 
 * @param {Object} config - Конфигурация виджета из API
 * @param {string} API_BASE - Базовый URL API (https://w2.medmaps.ru/api)
 */
function initCustomWidget(config, API_BASE) {
    console.log('Custom widget initialized:', config);
    
    // Ваш код здесь
    // Пример: создать кастомный UI, обработчики событий и т.д.
    
    // Доступные параметры config:
    // - config.widget_key
    // - config.name
    // - config.primary_color
    // - config.welcome_message
    // - config.suggestions (массив)
    // - config.custom_files (пути к файлам)
}
JS;
            file_put_contents($customPath . '/custom.js', $customJsContent);
            
            $db->commit();
            $success = 'Виджет успешно создан!';
            
            // Перенаправляем на настройки
            header("Location: /admin/widget-settings.php?id=$widget_id");
            exit;
            
        } catch (Exception $e) {
            $db->rollBack();
            $error = 'Ошибка создания виджета: ' . $e->getMessage();
            logError("Widget creation error: " . $e->getMessage());
        }
    }
}

// Автогенерация widget_key из названия
$suggested_key = '';
if (!empty($_POST['name']) && empty($_POST['widget_key'])) {
    $suggested_key = generateSlug($_POST['name']);
}
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
    <h1 class="h2">Создать виджет</h1>
    <div class="btn-toolbar mb-2 mb-md-0">
        <a href="/admin/" class="btn btn-sm btn-outline-secondary">
            <i class="bi bi-arrow-left"></i> Назад
        </a>
    </div>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
<?php endif; ?>

<?php if ($success): ?>
    <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
<?php endif; ?>

<div class="row">
    <div class="col-md-8">
        <div class="card">
            <div class="card-body">
                <form method="POST" id="createWidgetForm">
                    <div class="mb-3">
                        <label for="name" class="form-label">Название виджета *</label>
                        <input type="text" class="form-control" id="name" name="name" required value="<?php echo htmlspecialchars($_POST['name'] ?? ''); ?>" placeholder="Клиника ЕМЦ">
                        <div class="form-text">Название для идентификации виджета в админке</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="widget_key" class="form-label">Widget Key *</label>
                        <div class="input-group">
                            <input type="text" class="form-control" id="widget_key" name="widget_key" required value="<?php echo htmlspecialchars($_POST['widget_key'] ?? $suggested_key); ?>" pattern="[a-z0-9-]+" placeholder="emc-clinic-001">
                            <button class="btn btn-outline-secondary" type="button" id="generateKey">
                                <i class="bi bi-arrow-clockwise"></i> Авто
                            </button>
                        </div>
                        <div class="form-text">Уникальный идентификатор (только латиница, цифры, дефис)</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="site_domain" class="form-label">Домен сайта *</label>
                        <input type="url" class="form-control" id="site_domain" name="site_domain" required value="<?php echo htmlspecialchars($_POST['site_domain'] ?? ''); ?>" placeholder="https://www.example.com">
                        <div class="form-text">Полный URL сайта клиники (будет добавлен в allowed_domains)</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="sitemap_url" class="form-label">URL Sitemap *</label>
                        <input type="url" class="form-control" id="sitemap_url" name="sitemap_url" required value="<?php echo htmlspecialchars($_POST['sitemap_url'] ?? ''); ?>" placeholder="https://www.example.com/sitemap.xml">
                        <div class="form-text">Адрес файла sitemap.xml для парсинга страниц</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="claude_prompt" class="form-label">Промпт для Claude (опционально)</label>
                        <textarea class="form-control" id="claude_prompt" name="claude_prompt" rows="4" placeholder="Оставьте пустым для использования дефолтного промпта"><?php echo htmlspecialchars($_POST['claude_prompt'] ?? ''); ?></textarea>
                        <div class="form-text">Можно будет отредактировать позже в настройках</div>
                    </div>
                    
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-primary btn-lg">
                            <i class="bi bi-plus-circle"></i> Создать виджет
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <div class="card">
            <div class="card-header">
                <h6 class="mb-0">Что будет создано?</h6>
            </div>
            <div class="card-body">
                <p class="small text-muted">После создания виджета автоматически будут созданы:</p>
                <ul class="small">
                    <li>4 раздела:
                        <ul>
                            <li><strong>Специалисты</strong> (name, photo, description, url)</li>
                            <li><strong>Услуги</strong> (name, price, description, url)</li>
                            <li><strong>Статьи</strong> (title, content, date, url)</li>
                            <li><strong>Специализации</strong> (name, description, url)</li>
                        </ul>
                    </li>
                    <li>Дефолтные CSS селекторы для каждого поля</li>
                    <li>Настройки виджета (цвета, подсказки, CORS)</li>
                </ul>
                <hr>
                <p class="small mb-0"><strong>Следующие шаги:</strong></p>
                <ol class="small mb-0">
                    <li>Настроить URL patterns для разделов</li>
                    <li>Настроить CSS селекторы</li>
                    <li>Протестировать парсинг</li>
                    <li>Запустить полный парсинг</li>
                </ol>
            </div>
        </div>
    </div>
</div>

<script>
// Автогенерация widget_key
document.getElementById('name').addEventListener('input', function(e) {
    if (!document.getElementById('widget_key').value) {
        document.getElementById('widget_key').value = generateSlug(e.target.value);
    }
});

document.getElementById('generateKey').addEventListener('click', function() {
    const name = document.getElementById('name').value;
    if (name) {
        document.getElementById('widget_key').value = generateSlug(name);
    }
});

// Автозаполнение sitemap_url
document.getElementById('site_domain').addEventListener('blur', function(e) {
    const domain = e.target.value;
    if (domain && !document.getElementById('sitemap_url').value) {
        document.getElementById('sitemap_url').value = domain.replace(/\/$/, '') + '/sitemap.xml';
    }
});
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>

