<?php
$page_title = 'Тестирование промпта';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/includes/auth.php';

$widget_id = $_GET['widget_id'] ?? 0;

// Получаем виджет
$db = getDatabase();
$stmt = $db->prepare("SELECT id, widget_key, name FROM widgets WHERE id = ?");
$stmt->execute([$widget_id]);
$widget = $stmt->fetch();

if (!$widget) {
    die('Виджет не найден');
}

// Получаем последний лог
$stmt = $db->query("SELECT * FROM prompt_test_logs WHERE widget_id = $widget_id ORDER BY created_at DESC LIMIT 1");
$lastLog = $stmt->fetch();

// Обработка AJAX запроса для тестирования
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'test') {
    header('Content-Type: application/json');
    
    $prompt = $_POST['prompt'] ?? '';
    $question = $_POST['question'] ?? 'У меня болит спина';
    
    if (empty($prompt)) {
        echo json_encode(['error' => 'Промпт не может быть пустым']);
        exit;
    }
    
    // Получаем данные каталога из последнего лога или из БД
    $catalogData = '';
    if ($lastLog) {
        // Извлекаем данные каталога из промпта лога
        $promptParts = explode("\n\n=== КАТАЛОГ ДАННЫХ ===\n", $lastLog['stage3_prompt_after'] ?? $lastLog['stage3_prompt'] ?? '');
        $catalogData = $promptParts[1] ?? '';
    }
    
    // Если каталог не найден, получаем из БД
    if (empty($catalogData) && $lastLog) {
        // Временно меняем REQUEST_METHOD, чтобы обойти проверки в query.php
        $oldMethod = $_SERVER['REQUEST_METHOD'] ?? null;
        $_SERVER['REQUEST_METHOD'] = 'CLI_TEST';
        
        require_once __DIR__ . '/../api/query.php';
        
        $stmt = $db->prepare("SELECT * FROM widgets WHERE id = ?");
        $stmt->execute([$widget_id]);
        $widgetForPrompt = $stmt->fetch();
        
        if ($widgetForPrompt) {
            $finalResponse = json_decode($lastLog['final_response'], true);
            $relevantIds = $finalResponse['data_ids'] ?? [];
            $catalogData = buildPrompt($db, $widgetForPrompt, $lastLog['question'], $relevantIds);
        }
        
        // Восстанавливаем REQUEST_METHOD
        if ($oldMethod !== null) {
            $_SERVER['REQUEST_METHOD'] = $oldMethod;
        }
    }
    
    // Получаем модель из лога или используем дефолт
    $debugInfo = json_decode($lastLog['debug_info'] ?? '{}', true);
    $model = $debugInfo['model'] ?? 'google/gemini-2.5-flash-lite';
    
    // Отправляем запрос к API
    $startTime = microtime(true);
    
    try {
        // Временно меняем REQUEST_METHOD, чтобы обойти проверки в query.php
        $oldMethod2 = $_SERVER['REQUEST_METHOD'] ?? null;
        $_SERVER['REQUEST_METHOD'] = 'CLI_TEST';
        
        require_once __DIR__ . '/../api/query.php';
        
        // Определяем провайдер
        $provider = 'openrouter';
        if (strpos($model, 'gpt://') === 0 || strpos($model, 'yandexgpt') !== false) {
            $provider = 'yandexgpt';
        } elseif (strpos($model, 'claude') !== false) {
            $provider = 'claude';
        }
        
        $response = null;
        if ($provider === 'openrouter') {
            $response = callOpenRouterAPI($prompt, $catalogData, $question, $model);
        } elseif ($provider === 'yandexgpt') {
            $debugInfo = null;
            $response = callYandexGPTAPI($prompt, $catalogData, $question, $model, MAX_TOKENS_RESPONSE, $debugInfo);
        } else {
            $response = callClaudeAPI($prompt, $catalogData, $question, false, $model);
        }
        
        // Восстанавливаем REQUEST_METHOD
        if ($oldMethod2 !== null) {
            $_SERVER['REQUEST_METHOD'] = $oldMethod2;
        }
        
        $responseTime = round((microtime(true) - $startTime) * 1000);
        
        // Анализируем услуги
        $analysis = [];
        if (isset($response['data_ids']['services']) && !empty($response['data_ids']['services'])) {
            $serviceIds = $response['data_ids']['services'];
            $placeholders = str_repeat('?,', count($serviceIds) - 1) . '?';
            $stmt = $db->prepare("
                SELECT pi.id, pf.field_value as name
                FROM parsed_items pi
                JOIN parsed_fields pf ON pi.id = pf.item_id
                WHERE pi.id IN ($placeholders)
                AND pi.section_name = 'services'
                AND pf.field_name = 'name'
            ");
            $stmt->execute($serviceIds);
            
            $services = [];
            while ($row = $stmt->fetch()) {
                $services[$row['id']] = $row['name'];
            }
            
            $irrelevant = [];
            $relevant = [];
            
            foreach ($serviceIds as $id) {
                $name = mb_strtolower($services[$id] ?? '');
                $isIrrelevant = false;
                $reason = '';
                
                // Проверки на нерелевантность
                if (stripos($name, 'косметолог') !== false || stripos($name, 'эстетист') !== false || stripos($name, 'лифтинг') !== false || stripos($name, 'микроток') !== false || stripos($name, 'миостимуляция') !== false) {
                    $isIrrelevant = true;
                    $reason = 'косметология';
                }
                if (stripos($name, 'пластический') !== false) {
                    $isIrrelevant = true;
                    $reason = 'пластическая хирургия';
                }
                if (stripos($name, 'подолог') !== false) {
                    $isIrrelevant = true;
                    $reason = 'подология';
                }
                if (stripos($name, 'вакцин') !== false || stripos($name, 'прививк') !== false) {
                    $isIrrelevant = true;
                    $reason = 'вакцинация';
                }
                if (stripos($name, 'транспортная комиссия') !== false || stripos($name, 'заключение о состоянии здоровья') !== false || stripos($name, 'оформлением заключения') !== false) {
                    $isIrrelevant = true;
                    $reason = 'справка';
                }
                if (stripos($name, 'нейропсихолог') !== false) {
                    $isIrrelevant = true;
                    $reason = 'нейропсихология';
                }
                if (stripos($name, 'консультация хирурга') !== false && stripos($name, 'травматолог') === false && stripos($name, 'ортопед') === false && stripos($name, 'пластический') === false) {
                    $isIrrelevant = true;
                    $reason = 'общий хирург без специализации';
                }
                if (stripos($name, 'гинеколог') !== false || stripos($name, 'кольпо') !== false || stripos($name, 'пайпель') !== false || stripos($name, 'беременность') !== false || stripos($name, 'мазок гинеколог') !== false || stripos($name, 'межскрининговый период') !== false) {
                    $isIrrelevant = true;
                    $reason = 'гинекология';
                }
                if (stripos($name, 'биопсия прямой кишки') !== false || stripos($name, 'прямой кишки') !== false || stripos($name, 'рект. мазок') !== false) {
                    $isIrrelevant = true;
                    $reason = 'проктология (без указания проблемы с прямой кишкой)';
                }
                if (stripos($name, 'осмотр врача восстановительной медицины') !== false && stripos($name, 'травматолог') === false) {
                    $isIrrelevant = true;
                    $reason = 'общий осмотр без специализации';
                }
                if (stripos($name, 'осмотр перед проведением комплекса процедур') !== false && stripos($name, 'травматолог') === false && stripos($name, 'ортопед') === false) {
                    $isIrrelevant = true;
                    $reason = 'общий осмотр без специализации';
                }
                
                if ($isIrrelevant) {
                    $irrelevant[] = ['id' => $id, 'name' => $services[$id] ?? 'N/A', 'reason' => $reason];
                } else {
                    $relevant[] = ['id' => $id, 'name' => $services[$id] ?? 'N/A'];
                }
            }
            
            $analysis = [
                'total' => count($serviceIds),
                'relevant' => count($relevant),
                'irrelevant' => count($irrelevant),
                'relevant_list' => $relevant,
                'irrelevant_list' => $irrelevant
            ];
        }
        
        echo json_encode([
            'success' => true,
            'response' => $response,
            'response_time_ms' => $responseTime,
            'analysis' => $analysis,
            'prompt_length' => strlen($prompt),
            'catalog_length' => strlen($catalogData)
        ], JSON_UNESCAPED_UNICODE);
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

require_once __DIR__ . '/includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <h2>Тестирование промпта</h2>
            <p><strong>Виджет:</strong> <?php echo htmlspecialchars($widget['name']); ?> (<?php echo htmlspecialchars($widget['widget_key']); ?>)</p>
            
            <?php if ($lastLog): ?>
            <div class="alert alert-info">
                <i class="bi bi-info-circle"></i> Используются данные из последнего лога (<?php echo date('d.m.Y H:i:s', strtotime($lastLog['created_at'])); ?>)
            </div>
            <?php endif; ?>
            
            <div class="row">
                <div class="col-md-6">
                    <div class="card mb-3">
                        <div class="card-header">
                            <h5>Промпт</h5>
                        </div>
                        <div class="card-body">
                            <form id="test-form">
                                <div class="mb-3">
                                    <label for="question" class="form-label">Вопрос для тестирования:</label>
                                    <input type="text" class="form-control" id="question" name="question" value="<?php echo htmlspecialchars($lastLog['question'] ?? 'У меня болит спина'); ?>">
                                </div>
                                
                                <div class="mb-3">
                                    <label for="prompt" class="form-label">Системный промпт:</label>
                                    <textarea class="form-control font-monospace" id="prompt" name="prompt" rows="25" style="font-size: 11px;"><?php 
                                        if ($lastLog) {
                                            $promptParts = explode("\n\n=== КАТАЛОГ ДАННЫХ ===\n", $lastLog['stage3_prompt_after'] ?? $lastLog['stage3_prompt'] ?? '');
                                            echo htmlspecialchars($promptParts[0] ?? '');
                                        }
                                    ?></textarea>
                                    <small class="form-text text-muted">
                                        Длина: <span id="prompt-length">0</span> символов
                                    </small>
                                </div>
                                
                                <div class="d-flex gap-2">
                                    <button type="submit" class="btn btn-primary" id="test-btn">
                                        <i class="bi bi-play-circle"></i> Запустить тест
                                    </button>
                                    <button type="button" class="btn btn-secondary" onclick="loadPromptFromLog()">
                                        <i class="bi bi-arrow-clockwise"></i> Загрузить из лога
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
                
                <div class="col-md-6">
                    <div class="card mb-3">
                        <div class="card-header">
                            <h5>Результаты теста</h5>
                        </div>
                        <div class="card-body">
                            <div id="test-results" style="display: none;">
                                <div id="loading" class="text-center" style="display: none;">
                                    <div class="spinner-border" role="status">
                                        <span class="visually-hidden">Загрузка...</span>
                                    </div>
                                    <p>Отправка запроса к API...</p>
                                </div>
                                
                                <div id="results-content"></div>
                            </div>
                            <div id="no-results" class="text-muted text-center">
                                <p>Запустите тест, чтобы увидеть результаты</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.getElementById('prompt').addEventListener('input', function() {
    document.getElementById('prompt-length').textContent = this.value.length.toLocaleString();
});

// Обновляем длину при загрузке
document.getElementById('prompt-length').textContent = document.getElementById('prompt').value.length.toLocaleString();

function loadPromptFromLog() {
    <?php if ($lastLog): ?>
    const promptParts = <?php echo json_encode(explode("\n\n=== КАТАЛОГ ДАННЫХ ===\n", $lastLog['stage3_prompt_after'] ?? $lastLog['stage3_prompt'] ?? '')); ?>;
    document.getElementById('prompt').value = promptParts[0] || '';
    document.getElementById('prompt-length').textContent = (promptParts[0] || '').length.toLocaleString();
    <?php endif; ?>
}

document.getElementById('test-form').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const btn = document.getElementById('test-btn');
    const loading = document.getElementById('loading');
    const resultsContent = document.getElementById('results-content');
    const testResults = document.getElementById('test-results');
    const noResults = document.getElementById('no-results');
    
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Тестирование...';
    loading.style.display = 'block';
    testResults.style.display = 'block';
    noResults.style.display = 'none';
    resultsContent.innerHTML = '';
    
    const formData = new FormData();
    formData.append('action', 'test');
    formData.append('prompt', document.getElementById('prompt').value);
    formData.append('question', document.getElementById('question').value);
    
    try {
        const response = await fetch('?widget_id=<?php echo $widget_id; ?>', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        loading.style.display = 'none';
        
        if (data.success) {
            let html = '<div class="alert alert-success"><strong>Тест выполнен успешно!</strong></div>';
            html += '<p><strong>Время ответа:</strong> ' + data.response_time_ms + ' мс</p>';
            html += '<p><strong>Токенов использовано:</strong> ' + (data.response.tokens_used || 'N/A') + '</p>';
            html += '<p><strong>Длина промпта:</strong> ' + data.prompt_length.toLocaleString() + ' символов</p>';
            html += '<p><strong>Длина каталога:</strong> ' + data.catalog_length.toLocaleString() + ' символов</p>';
            
            html += '<hr><h6>Текстовый ответ:</h6>';
            html += '<div class="bg-dark text-light p-3 rounded mb-3">' + escapeHtml(data.response.answer) + '</div>';
            
            html += '<h6>ID записей:</h6>';
            const dataIds = data.response.data_ids || {};
            for (const [section, ids] of Object.entries(dataIds)) {
                html += '<p><strong>' + section + ':</strong> ' + ids.length + ' ID';
                if (ids.length <= 20) {
                    html += '<br><code>' + ids.join(', ') + '</code>';
                } else {
                    html += '<br><code>' + ids.slice(0, 20).join(', ') + ' ... (еще ' + (ids.length - 20) + ')</code>';
                }
                html += '</p>';
            }
            
            if (data.analysis && data.analysis.total > 0) {
                html += '<hr><h6>Анализ услуг:</h6>';
                const analysis = data.analysis;
                const relevantPercent = (analysis.relevant / analysis.total * 100).toFixed(1);
                const irrelevantPercent = (analysis.irrelevant / analysis.total * 100).toFixed(1);
                
                html += '<div class="row mb-3">';
                html += '<div class="col-md-6"><div class="card bg-success bg-opacity-25"><div class="card-body"><strong>Релевантных:</strong> ' + analysis.relevant + ' (' + relevantPercent + '%)</div></div></div>';
                html += '<div class="col-md-6"><div class="card bg-danger bg-opacity-25"><div class="card-body"><strong>Нерелевантных:</strong> ' + analysis.irrelevant + ' (' + irrelevantPercent + '%)</div></div></div>';
                html += '</div>';
                
                if (analysis.irrelevant > 0) {
                    html += '<div class="alert alert-danger"><strong>Нерелевантные услуги:</strong><ul class="mb-0">';
                    analysis.irrelevant_list.forEach(item => {
                        html += '<li>ID ' + item.id + ': ' + escapeHtml(item.name) + ' <span class="badge bg-secondary">' + item.reason + '</span></li>';
                    });
                    html += '</ul></div>';
                }
                
                if (analysis.relevant > 0) {
                    html += '<details><summary class="text-success"><strong>Релевантные услуги (' + analysis.relevant + ')</strong></summary><ul>';
                    analysis.relevant_list.forEach(item => {
                        html += '<li>ID ' + item.id + ': ' + escapeHtml(item.name) + '</li>';
                    });
                    html += '</ul></details>';
                }
            }
            
            html += '<hr><h6>Сырой ответ API:</h6>';
            html += '<pre class="bg-dark text-light p-3 rounded" style="max-height: 300px; overflow-y: auto; font-size: 11px;">' + escapeHtml(data.response.raw_response || '').substring(0, 2000) + '</pre>';
            
            resultsContent.innerHTML = html;
        } else {
            resultsContent.innerHTML = '<div class="alert alert-danger"><strong>Ошибка:</strong> ' + escapeHtml(data.error || 'Неизвестная ошибка') + '</div>';
        }
    } catch (error) {
        loading.style.display = 'none';
        resultsContent.innerHTML = '<div class="alert alert-danger"><strong>Ошибка:</strong> ' + escapeHtml(error.message) + '</div>';
    } finally {
        btn.disabled = false;
        btn.innerHTML = '<i class="bi bi-play-circle"></i> Запустить тест';
    }
});

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
</script>

<?php
include __DIR__ . '/includes/footer.php';

