<?php
require_once __DIR__ . '/../../config.php';

// Проверка авторизации
session_start();
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    die('Unauthorized');
}

$widgetId = isset($_GET['id']) ? (int)$_GET['id'] : 1;

try {
    $db = getDatabase();
    
    // Получаем все категории с ключевыми словами
    $stmt = $db->prepare("
        SELECT section_name, category_name, keywords
        FROM category_keywords
        WHERE widget_id = ?
        ORDER BY section_name, category_name
    ");
    $stmt->execute([$widgetId]);
    $categories = $stmt->fetchAll();
    
    $allKeywords = [];
    $keywordsBySection = [];
    $keywordsByCategory = [];
    
    foreach ($categories as $category) {
        $section = $category['section_name'];
        $categoryName = $category['category_name'];
        $keywordsJson = $category['keywords'];
        
        // Парсим JSON с ключевыми словами
        $keywords = json_decode($keywordsJson, true);
        
        if (is_array($keywords)) {
            foreach ($keywords as $keyword) {
                $keyword = trim($keyword);
                if (!empty($keyword)) {
                    // Добавляем в общий список
                    $allKeywords[] = $keyword;
                    
                    // Добавляем в список по разделам
                    if (!isset($keywordsBySection[$section])) {
                        $keywordsBySection[$section] = [];
                    }
                    $keywordsBySection[$section][] = $keyword;
                    
                    // Добавляем в список по категориям
                    $key = $section . '::' . $categoryName;
                    if (!isset($keywordsByCategory[$key])) {
                        $keywordsByCategory[$key] = [];
                    }
                    $keywordsByCategory[$key][] = $keyword;
                }
            }
        }
    }
    
    // Убираем дубликаты
    $uniqueKeywords = array_unique($allKeywords);
    $uniqueKeywords = array_values($uniqueKeywords); // Переиндексируем массив
    
    // Формируем текст для промпта
    $promptText = implode(', ', $uniqueKeywords);
    
    // Подсчитываем размеры
    $textSizeBytes = strlen($promptText);
    $textSizeChars = mb_strlen($promptText, 'UTF-8');
    
    // Используем функцию estimateTokens из config.php для более точной оценки
    // Функция уже доступна через require_once config.php в начале файла
    
    // Примерная оценка токенов (1 токен ≈ 4 символа для русского текста)
    // Более точная оценка: примерно 0.75 токена на слово для русского языка
    $words = preg_split('/[\s,]+/u', $promptText);
    $wordCount = count(array_filter($words, function($w) { return mb_strlen(trim($w)) > 0; }));
    $estimatedTokens = round($wordCount * 0.75);
    
    // Альтернативная оценка: примерно 1 токен на 3-4 символа для русского текста
    $estimatedTokensAlt = round($textSizeChars / 3.5);
    
    // Используем функцию estimateTokens из query.php
    $estimatedTokensPrecise = estimateTokens($promptText);
    
    // Статистика по разделам
    $statsBySection = [];
    foreach ($keywordsBySection as $section => $keywords) {
        $uniqueSectionKeywords = array_unique($keywords);
        $statsBySection[$section] = [
            'total' => count($keywords),
            'unique' => count($uniqueSectionKeywords)
        ];
    }
    
    // Выводим результаты
    header('Content-Type: text/html; charset=utf-8');
    ?>
    <!DOCTYPE html>
    <html lang="ru">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Анализ ключевых слов - Виджет #<?php echo $widgetId; ?></title>
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
        <style>
            .keyword-list {
                max-height: 400px;
                overflow-y: auto;
                background: #f8f9fa;
                padding: 15px;
                border-radius: 8px;
                font-family: monospace;
                font-size: 0.9em;
            }
            .stats-box {
                background: #e9ecef;
                padding: 15px;
                border-radius: 8px;
                margin-bottom: 20px;
            }
        </style>
    </head>
    <body>
        <div class="container mt-4">
            <h1>Анализ ключевых слов - Виджет #<?php echo $widgetId; ?></h1>
            
            <div class="stats-box">
                <h3>Общая статистика</h3>
                <ul>
                    <li><strong>Всего категорий:</strong> <?php echo count($categories); ?></li>
                    <li><strong>Всего ключевых слов (с дубликатами):</strong> <?php echo count($allKeywords); ?></li>
                    <li><strong>Уникальных ключевых слов:</strong> <?php echo count($uniqueKeywords); ?></li>
                    <li><strong>Размер текста (байты):</strong> <?php echo number_format($textSizeBytes); ?></li>
                    <li><strong>Размер текста (символы UTF-8):</strong> <?php echo number_format($textSizeChars); ?></li>
                    <li><strong>Примерное количество слов:</strong> <?php echo number_format($wordCount); ?></li>
                    <li><strong>Примерное количество токенов (оценка 1):</strong> <?php echo number_format($estimatedTokens); ?> (0.75 токена на слово)</li>
                    <li><strong>Примерное количество токенов (оценка 2):</strong> <?php echo number_format($estimatedTokensAlt); ?> (1 токен на 3.5 символа)</li>
                    <li><strong>Точная оценка токенов (estimateTokens):</strong> <span class="text-success"><strong><?php echo number_format($estimatedTokensPrecise); ?></strong></span></li>
                    <li><strong>Рекомендуемая оценка токенов:</strong> <span class="text-primary"><strong><?php echo number_format($estimatedTokensPrecise); ?></strong></span> (используется функция estimateTokens)</li>
                </ul>
            </div>
            
            <div class="stats-box">
                <h3>Статистика по разделам</h3>
                <table class="table table-sm">
                    <thead>
                        <tr>
                            <th>Раздел</th>
                            <th>Всего ключевых слов</th>
                            <th>Уникальных</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($statsBySection as $section => $stats): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($section); ?></td>
                            <td><?php echo $stats['total']; ?></td>
                            <td><?php echo $stats['unique']; ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <div class="mb-4">
                <h3>Все уникальные ключевые слова (<?php echo count($uniqueKeywords); ?>)</h3>
                <div class="keyword-list">
                    <?php echo htmlspecialchars($promptText); ?>
                </div>
            </div>
            
            <div class="mb-4">
                <h3>Текст для промпта (первые 500 символов)</h3>
                <div class="keyword-list">
                    <?php echo htmlspecialchars(mb_substr($promptText, 0, 500)) . (mb_strlen($promptText) > 500 ? '...' : ''); ?>
                </div>
            </div>
            
            <div class="alert alert-info">
                <strong>Примечание:</strong> Оценка токенов является приблизительной. 
                Точное количество токенов зависит от модели токенизации (например, tiktoken для GPT моделей).
                Для более точной оценки рекомендуется использовать API токенизации конкретной модели.
            </div>
        </div>
    </body>
    </html>
    <?php
    
} catch (Exception $e) {
    die('Ошибка: ' . $e->getMessage());
}

