/**
 * URL Tree Manager
 * Древовидное отображение и управление URL
 */

class URLTreeManager {
    constructor(widgetId, containerId) {
        this.widgetId = widgetId;
        this.container = document.getElementById(containerId);
        this.urls = [];
        this.tree = {};
    }
    
    /**
     * Загрузить URL из базы
     */
    async loadURLs() {
        try {
            const response = await fetch(`ajax/url-get-tree.php?widget_id=${this.widgetId}`);
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            this.urls = data.urls;
            this.buildTree();
            this.render();
            
        } catch (error) {
            console.error('Error loading URLs:', error);
            alert('Ошибка загрузки URL: ' + error.message);
        }
    }
    
    /**
     * Построить древовидную структуру из плоского списка URL
     */
    buildTree() {
        this.tree = {};
        
        // Группируем URL по нормализованному пути
        const groups = {};
        
        this.urls.forEach(url => {
            const normalized = url.normalized_url;
            
            if (!groups[normalized]) {
                groups[normalized] = {
                    canonical: null,
                    duplicates: []
                };
            }
            
            if (url.is_canonical) {
                groups[normalized].canonical = url;
            } else {
                groups[normalized].duplicates.push(url);
            }
        });
        
        // Строим дерево по структуре URL
        Object.values(groups).forEach(group => {
            if (!group.canonical) return;
            
            const url = group.canonical;
            const path = this.getPathFromURL(url.url);
            const parts = path.split('/').filter(p => p.length > 0);
            
            let current = this.tree;
            let currentPath = '';
            
            parts.forEach((part, index) => {
                currentPath += '/' + part;
                
                if (!current[part]) {
                    current[part] = {
                        name: part,
                        path: currentPath,
                        children: {},
                        urls: []
                    };
                }
                
                // Добавляем URL на последнем уровне
                if (index === parts.length - 1) {
                    current[part].urls.push({
                        ...url,
                        duplicates: group.duplicates
                    });
                }
                
                current = current[part].children;
            });
            
            // URL без пути (корень)
            if (parts.length === 0) {
                if (!this.tree['_root']) {
                    this.tree['_root'] = {
                        name: '/',
                        path: '/',
                        children: {},
                        urls: []
                    };
                }
                this.tree['_root'].urls.push({
                    ...url,
                    duplicates: group.duplicates
                });
            }
        });
    }
    
    /**
     * Получить путь из URL
     */
    getPathFromURL(url) {
        try {
            const parsed = new URL(url);
            return parsed.pathname;
        } catch (e) {
            return '/';
        }
    }
    
    /**
     * Отрендерить дерево
     */
    render() {
        this.container.innerHTML = '';
        
        if (Object.keys(this.tree).length === 0) {
            this.container.innerHTML = '<div class="alert alert-info">Нет найденных URL. Запустите краулинг.</div>';
            return;
        }
        
        const treeHTML = this.renderNode(this.tree, 0);
        this.container.innerHTML = treeHTML;
        
        // Привязываем обработчики событий
        this.attachEventHandlers();
    }
    
    /**
     * Отрендерить узел дерева
     */
    renderNode(node, level) {
        let html = '';
        
        Object.values(node).forEach(item => {
            const hasChildren = Object.keys(item.children).length > 0;
            const hasUrls = item.urls.length > 0;
            
            if (!hasChildren && !hasUrls) return;
            
            const indent = level * 20;
            
            // Заголовок папки
            if (hasChildren || (hasUrls && level > 0)) {
                // Подсчитываем количество URL в папке (рекурсивно)
                const urlCount = this.countUrlsInFolder(item.path);
                
                html += `
                    <div class="tree-folder" style="margin-left: ${indent}px">
                        <input type="checkbox" class="form-check-input folder-checkbox me-2" data-path="${item.path}" title="Выбрать все URL в папке">
                        <span class="folder-toggle" data-path="${item.path}">
                            <i class="bi bi-chevron-right"></i>
                        </span>
                        <i class="bi bi-folder"></i>
                        <strong>${item.name}</strong>
                        <span class="badge bg-secondary ms-2">${urlCount}</span>
                        <select class="form-select form-select-sm folder-section-select ms-2" data-path="${item.path}" title="Назначить раздел всей папке">
                            <option value="">-- Назначить папке --</option>
                            <option value="specialists">Специалисты</option>
                            <option value="services">Услуги</option>
                            <option value="articles">Статьи</option>
                            <option value="specializations">Специализации</option>
                            <option value="excluded">Исключить</option>
                        </select>
                    </div>
                `;
            }
            
            // URL в этой папке
            if (hasUrls) {
                item.urls.forEach(url => {
                    html += this.renderURL(url, level + 1);
                });
            }
            
            // Дочерние папки
            if (hasChildren) {
                html += `<div class="tree-children collapsed" data-parent="${item.path}">`;
                html += this.renderNode(item.children, level + 1);
                html += '</div>';
            }
        });
        
        return html;
    }
    
    /**
     * Отрендерить один URL
     */
    renderURL(url, level) {
        const indent = level * 20;
        const statusClass = this.getStatusClass(url.status);
        const duplicateCount = url.duplicates.length;
        const duplicateBadge = duplicateCount > 0 ? `<span class="badge bg-secondary duplicate-badge" data-url-id="${url.id}">(+${duplicateCount})</span>` : '';
        
        return `
            <div class="tree-url ${statusClass}" style="margin-left: ${indent}px" data-url-id="${url.id}">
                <div class="url-row">
                    <input type="checkbox" class="form-check-input url-checkbox" data-url-id="${url.id}">
                    <a href="${url.url}" target="_blank" class="url-link">${this.truncateURL(url.url)}</a>
                    ${duplicateBadge}
                    <select class="form-select form-select-sm url-section-select" data-url-id="${url.id}">
                        <option value="">-- Не назначен --</option>
                        <option value="specialists" ${url.assigned_section === 'specialists' ? 'selected' : ''}>Специалисты</option>
                        <option value="services" ${url.assigned_section === 'services' ? 'selected' : ''}>Услуги</option>
                        <option value="articles" ${url.assigned_section === 'articles' ? 'selected' : ''}>Статьи</option>
                        <option value="specializations" ${url.assigned_section === 'specializations' ? 'selected' : ''}>Специализации</option>
                        <option value="excluded" ${url.assigned_section === 'excluded' ? 'selected' : ''}>Исключить</option>
                    </select>
                    <button class="btn btn-sm btn-primary parse-single-btn" data-url-id="${url.id}" ${!url.assigned_section || url.assigned_section === 'excluded' ? 'disabled' : ''}>
                        <i class="bi bi-play-circle"></i>
                    </button>
                </div>
                ${duplicateCount > 0 ? this.renderDuplicates(url.duplicates) : ''}
            </div>
        `;
    }
    
    /**
     * Отрендерить список дублей
     */
    renderDuplicates(duplicates) {
        let html = '<div class="duplicate-list" style="display: none;">';
        duplicates.forEach(dup => {
            html += `<div class="duplicate-item">${dup.url}</div>`;
        });
        html += '</div>';
        return html;
    }
    
    /**
     * Усечь URL для отображения
     */
    truncateURL(url) {
        const maxLength = 80;
        if (url.length <= maxLength) return url;
        return url.substring(0, maxLength) + '...';
    }
    
    /**
     * Получить класс статуса
     */
    getStatusClass(status) {
        const classes = {
            'new': 'status-new',
            'assigned': 'status-assigned',
            'parsing': 'status-parsing',
            'parsed': 'status-parsed',
            'parse_error': 'status-error'
        };
        return classes[status] || '';
    }
    
    /**
     * Привязать обработчики событий
     */
    attachEventHandlers() {
        // Сворачивание/разворачивание папок
        document.querySelectorAll('.folder-toggle').forEach(toggle => {
            toggle.addEventListener('click', (e) => {
                const path = e.currentTarget.dataset.path;
                const children = document.querySelector(`.tree-children[data-parent="${path}"]`);
                const icon = e.currentTarget.querySelector('i');
                
                if (children) {
                    children.classList.toggle('collapsed');
                    icon.classList.toggle('bi-chevron-down');
                    icon.classList.toggle('bi-chevron-right');
                }
            });
        });
        
        // Показ/скрытие дублей
        document.querySelectorAll('.duplicate-badge').forEach(badge => {
            badge.addEventListener('click', (e) => {
                const urlId = e.currentTarget.dataset.urlId;
                const duplicateList = document.querySelector(`.tree-url[data-url-id="${urlId}"] .duplicate-list`);
                if (duplicateList) {
                    duplicateList.style.display = duplicateList.style.display === 'none' ? 'block' : 'none';
                }
            });
        });
        
        // Изменение раздела (автосохранение)
        document.querySelectorAll('.url-section-select').forEach(select => {
            select.addEventListener('change', async (e) => {
                const urlId = e.currentTarget.dataset.urlId;
                const section = e.currentTarget.value;
                
                await this.assignSection(urlId, section);
            });
        });
        
        // Парсинг одной страницы
        document.querySelectorAll('.parse-single-btn').forEach(btn => {
            btn.addEventListener('click', async (e) => {
                const urlId = e.currentTarget.dataset.urlId;
                await this.parseSingle(urlId, e.currentTarget);
            });
        });
        
        // Отслеживание изменения чекбоксов для активации кнопки краулинга
        document.querySelectorAll('.url-checkbox').forEach(checkbox => {
            checkbox.addEventListener('change', () => {
                this.updateCrawlButtonState();
            });
        });
        
        // Массовое назначение раздела папке
        document.querySelectorAll('.folder-section-select').forEach(select => {
            select.addEventListener('change', async (e) => {
                const path = e.currentTarget.dataset.path;
                const section = e.currentTarget.value;
                
                if (section) {
                    await this.assignSectionToFolder(path, section, e.currentTarget);
                }
            });
        });
        
        // Массовое выделение/снятие выделения URL в папке
        document.querySelectorAll('.folder-checkbox').forEach(checkbox => {
            checkbox.addEventListener('change', (e) => {
                const path = e.currentTarget.dataset.path;
                const isChecked = e.currentTarget.checked;
                
                this.toggleFolderSelection(path, isChecked);
            });
        });
    }
    
    /**
     * Обновить состояние кнопки краулинга
     */
    updateCrawlButtonState() {
        const selectedUrls = this.getSelectedUrls();
        
        // Обновляем кнопку краулинга
        const crawlBtn = document.getElementById('crawl-from-selected-btn');
        if (crawlBtn) {
            crawlBtn.disabled = selectedUrls.length === 0;
            const countSpan = crawlBtn.querySelector('.selected-count');
            if (countSpan) {
                countSpan.textContent = selectedUrls.length;
            }
        }
        
        // Обновляем кнопку парсинга (только для назначенных URL, не excluded)
        const assignedUrls = selectedUrls.filter(urlId => {
            const select = document.querySelector(`.url-section-select[data-url-id="${urlId}"]`);
            if (select) {
                const section = select.value;
                return section && section !== 'excluded' && section !== '';
            }
            return false;
        });
        
        const parseBtn = document.getElementById('parse-selected-btn');
        if (parseBtn) {
            parseBtn.disabled = assignedUrls.length === 0;
            const countSpan = parseBtn.querySelector('.selected-parse-count');
            if (countSpan) {
                countSpan.textContent = assignedUrls.length;
            }
        }
    }
    
    /**
     * Получить выбранные URL
     */
    getSelectedUrls() {
        const checkboxes = document.querySelectorAll('.url-checkbox:checked');
        return Array.from(checkboxes).map(cb => cb.dataset.urlId);
    }
    
    /**
     * Переключить выделение всех URL в папке
     */
    toggleFolderSelection(path, isChecked) {
        // Получаем все ID URL в папке
        const urlIds = this.getUrlIdsInFolder(path);
        
        // Отмечаем/снимаем чекбоксы для всех URL
        urlIds.forEach(urlId => {
            const checkbox = document.querySelector(`.url-checkbox[data-url-id="${urlId}"]`);
            if (checkbox) {
                checkbox.checked = isChecked;
            }
        });
        
        // Обновляем состояние кнопок
        this.updateCrawlButtonState();
    }
    
    /**
     * Массовый парсинг выбранных URL
     */
    async parseSelected() {
        const selectedUrls = this.getSelectedUrls();
        
        if (selectedUrls.length === 0) {
            alert('Выберите хотя бы один URL для парсинга');
            return;
        }
        
        // Фильтруем только назначенные URL (не excluded)
        const urlsToParse = selectedUrls.filter(urlId => {
            const select = document.querySelector(`.url-section-select[data-url-id="${urlId}"]`);
            if (select) {
                const section = select.value;
                return section && section !== 'excluded' && section !== '';
            }
            return false;
        });
        
        if (urlsToParse.length === 0) {
            alert('Нет назначенных URL для парсинга.\n\nСначала назначьте раздел выбранным URL.');
            return;
        }
        
        const confirmMsg = `Запустить парсинг ${urlsToParse.length} выбранных URL?\n\nЭто может занять некоторое время.`;
        if (!confirm(confirmMsg)) {
            return;
        }
        
        // Отключаем кнопку и показываем прогресс
        const parseBtn = document.getElementById('parse-selected-btn');
        if (parseBtn) {
            parseBtn.disabled = true;
            parseBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Парсинг...';
        }
        
        let successCount = 0;
        let errorCount = 0;
        const errors = [];
        
        // Парсим последовательно
        for (let i = 0; i < urlsToParse.length; i++) {
            const urlId = urlsToParse[i];
            
            try {
                const formData = new FormData();
                formData.append('url_id', urlId);
                
                const response = await fetch('ajax/url-parse-single.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (data.error) {
                    throw new Error(data.error);
                }
                
                // Обновляем статус в UI
                const urlRow = document.querySelector(`.tree-url[data-url-id="${urlId}"]`);
                if (urlRow) {
                    urlRow.classList.remove('status-new', 'status-assigned', 'status-parsing', 'status-error');
                    urlRow.classList.add('status-parsed');
                }
                
                successCount++;
                
                // Обновляем кнопку с прогрессом
                if (parseBtn) {
                    parseBtn.innerHTML = `<span class="spinner-border spinner-border-sm"></span> ${i + 1}/${urlsToParse.length}`;
                }
                
            } catch (error) {
                console.error(`Error parsing URL ${urlId}:`, error);
                errorCount++;
                errors.push({urlId, error: error.message});
                
                // Обновляем статус в UI
                const urlRow = document.querySelector(`.tree-url[data-url-id="${urlId}"]`);
                if (urlRow) {
                    urlRow.classList.remove('status-new', 'status-assigned', 'status-parsing');
                    urlRow.classList.add('status-error');
                }
            }
            
            // Небольшая задержка между запросами
            await new Promise(resolve => setTimeout(resolve, 100));
        }
        
        // Восстанавливаем кнопку
        if (parseBtn) {
            parseBtn.innerHTML = '<i class="bi bi-play-circle"></i> Спарсить выделенные (<span class="selected-parse-count">0</span>)';
            parseBtn.disabled = false;
        }
        
        // Снимаем все чекбоксы
        document.querySelectorAll('.url-checkbox:checked').forEach(cb => cb.checked = false);
        this.updateCrawlButtonState();
        
        // Показываем результат
        let resultMsg = `Парсинг завершен!\n\n`;
        resultMsg += `Успешно: ${successCount}\n`;
        if (errorCount > 0) {
            resultMsg += `Ошибок: ${errorCount}\n\n`;
            resultMsg += `Первые ошибки:\n`;
            errors.slice(0, 3).forEach(err => {
                resultMsg += `- URL ID ${err.urlId}: ${err.error}\n`;
            });
        }
        
        alert(resultMsg);
    }
    
    /**
     * Открыть модальное окно для краулинга
     */
    openCrawlModal() {
        const selectedUrls = this.getSelectedUrls();
        if (selectedUrls.length === 0) {
            alert('Выберите хотя бы один URL для краулинга');
            return;
        }
        
        const modal = new bootstrap.Modal(document.getElementById('crawlFromUrlsModal'));
        modal.show();
    }
    
    /**
     * Запустить краулинг с выбранных URL
     */
    async startCrawlFromSelected() {
        const selectedUrls = this.getSelectedUrls();
        
        if (selectedUrls.length === 0) {
            alert('Выберите хотя бы один URL');
            return;
        }
        
        // Получаем параметры из формы
        const maxDepth = parseInt(document.getElementById('crawl_max_depth').value);
        const maxUrls = parseInt(document.getElementById('crawl_max_urls').value);
        const timeout = parseInt(document.getElementById('crawl_timeout').value);
        
        // Валидация
        if (maxDepth < 1 || maxDepth > 20) {
            alert('Глубина должна быть от 1 до 20');
            return;
        }
        
        if (maxUrls < 10 || maxUrls > 50000) {
            alert('Количество URL должно быть от 10 до 50000');
            return;
        }
        
        if (timeout < 60 || timeout > 3600) {
            alert('Таймаут должен быть от 60 до 3600 секунд');
            return;
        }
        
        // Закрываем модальное окно
        const modal = bootstrap.Modal.getInstance(document.getElementById('crawlFromUrlsModal'));
        modal.hide();
        
        // Показываем прогресс
        document.getElementById('crawler-progress').style.display = 'block';
        document.getElementById('crawler-status').className = 'alert alert-info';
        document.getElementById('crawler-status').textContent = 'Запуск краулинга...';
        
        try {
            const formData = new FormData();
            formData.append('widget_id', this.widgetId);
            selectedUrls.forEach(urlId => {
                formData.append('start_urls[]', urlId);
            });
            formData.append('max_depth', maxDepth);
            formData.append('max_urls', maxUrls);
            formData.append('timeout_seconds', timeout);
            
            const response = await fetch('ajax/crawler-start-from-urls.php', {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            console.log('Crawler started:', data);
            
            // Устанавливаем глобальный job_id (без window, чтобы обновить переменную в widget-urls.php)
            if (typeof currentCrawlerJobId !== 'undefined') {
                currentCrawlerJobId = data.job_id;
            } else {
                // Fallback: устанавливаем через window
                window.currentCrawlerJobId = data.job_id;
            }
            
            // Вызываем глобальную функцию polling (определена в widget-urls.php)
            if (typeof pollCrawlerStatus === 'function') {
                pollCrawlerStatus();
            } else {
                console.error('pollCrawlerStatus function not found');
                document.getElementById('crawler-status').className = 'alert alert-warning';
                document.getElementById('crawler-status').textContent = 'Краулинг запущен (job_id: ' + data.job_id + '), но автообновление недоступно. Обновите страницу вручную.';
            }
            
        } catch (error) {
            console.error('Error starting crawler:', error);
            document.getElementById('crawler-status').className = 'alert alert-danger';
            document.getElementById('crawler-status').textContent = 'Ошибка: ' + error.message;
        }
    }
    
    /**
     * Назначить раздел
     */
    async assignSection(urlId, section) {
        try {
            const formData = new FormData();
            formData.append('url_id', urlId);
            formData.append('section', section);
            
            const response = await fetch('ajax/url-assign-section.php', {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            // Обновляем кнопку парсинга
            const btn = document.querySelector(`.parse-single-btn[data-url-id="${urlId}"]`);
            if (btn) {
                btn.disabled = !section || section === 'excluded';
            }
            
            console.log('Section assigned successfully');
            
        } catch (error) {
            console.error('Error assigning section:', error);
            alert('Ошибка назначения раздела: ' + error.message);
        }
    }
    
    /**
     * Парсить одну страницу
     */
    async parseSingle(urlId, button) {
        const originalHTML = button.innerHTML;
        button.innerHTML = '<span class="spinner-border spinner-border-sm"></span>';
        button.disabled = true;
        
        try {
            const formData = new FormData();
            formData.append('url_id', urlId);
            
            const response = await fetch('ajax/url-parse-single.php', {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            // Обновляем статус
            const urlRow = document.querySelector(`.tree-url[data-url-id="${urlId}"]`);
            if (urlRow) {
                urlRow.classList.remove('status-new', 'status-assigned', 'status-parsing', 'status-error');
                urlRow.classList.add('status-parsed');
            }
            
            let successMsg = 'Страница успешно спарсена!';
            if (data.child_items_count && data.child_items_count > 0) {
                successMsg += `\n\nСоздано дочерних элементов: ${data.child_items_count}`;
            }
            alert(successMsg);
            
        } catch (error) {
            console.error('Error parsing page:', error);
            alert('Ошибка парсинга: ' + error.message);
        } finally {
            button.innerHTML = originalHTML;
            button.disabled = false;
        }
    }
    
    /**
     * Массовое назначение раздела всем URL в папке
     */
    async assignSectionToFolder(path, section, selectElement) {
        // Собираем все URL в этой папке (включая вложенные)
        const urlIds = this.getUrlIdsInFolder(path);
        
        if (urlIds.length === 0) {
            alert('В папке нет URL для назначения');
            selectElement.value = '';
            return;
        }
        
        const confirmMsg = `Назначить раздел "${this.getSectionName(section)}" для ${urlIds.length} URL в папке?\n\nПуть: ${path}`;
        if (!confirm(confirmMsg)) {
            selectElement.value = '';
            return;
        }
        
        // Показываем индикатор загрузки
        const originalHTML = selectElement.innerHTML;
        selectElement.disabled = true;
        selectElement.innerHTML = '<option>Обновление...</option>';
        
        try {
            const formData = new FormData();
            formData.append('widget_id', this.widgetId);
            formData.append('path', path);
            formData.append('section', section);
            
            const response = await fetch('ajax/url-assign-section-bulk.php', {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            
            if (data.error) {
                throw new Error(data.error);
            }
            
            // Обновляем UI всех затронутых URL
            urlIds.forEach(urlId => {
                const select = document.querySelector(`.url-section-select[data-url-id="${urlId}"]`);
                if (select) {
                    select.value = section;
                }
                
                const btn = document.querySelector(`.parse-single-btn[data-url-id="${urlId}"]`);
                if (btn) {
                    btn.disabled = !section || section === 'excluded';
                }
            });
            
            alert(`Успешно назначен раздел для ${data.updated_count} URL`);
            
        } catch (error) {
            console.error('Error assigning section to folder:', error);
            alert('Ошибка назначения раздела: ' + error.message);
        } finally {
            selectElement.innerHTML = originalHTML;
            selectElement.value = '';
            selectElement.disabled = false;
        }
    }
    
    /**
     * Получить все ID URL в папке (рекурсивно)
     */
    getUrlIdsInFolder(path) {
        const urlIds = [];
        
        // Ищем все URL в дереве, чей путь начинается с указанного
        this.urls.forEach(url => {
            const urlPath = this.getPathFromURL(url.url);
            if (urlPath.startsWith(path) && url.is_canonical) {
                urlIds.push(url.id);
            }
        });
        
        return urlIds;
    }
    
    /**
     * Подсчитать количество URL в папке (рекурсивно)
     */
    countUrlsInFolder(path) {
        let count = 0;
        
        // Ищем все URL в дереве, чей путь начинается с указанного
        this.urls.forEach(url => {
            const urlPath = this.getPathFromURL(url.url);
            if (urlPath.startsWith(path) && url.is_canonical) {
                count++;
            }
        });
        
        return count;
    }
    
    /**
     * Получить название раздела
     */
    getSectionName(section) {
        const names = {
            'specialists': 'Специалисты',
            'services': 'Услуги',
            'articles': 'Статьи',
            'specializations': 'Специализации',
            'excluded': 'Исключить'
        };
        return names[section] || section;
    }
}

// Глобальная переменная для доступа из других скриптов
window.URLTreeManager = URLTreeManager;

