<?php
/**
 * AJAX: Парсинг одной страницы
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../../vendor/autoload.php';

use AIWidget\Parser\PageParser;
use AIWidget\Parser\ListItemsParser;

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    $urlId = $_POST['url_id'] ?? null;
    
    if (!$urlId) {
        throw new Exception('URL ID is required');
    }
    
    $db = getDatabase();
    
    // Получаем URL
    $stmt = $db->prepare("SELECT * FROM discovered_urls WHERE id = ?");
    $stmt->execute([$urlId]);
    $urlData = $stmt->fetch();
    
    if (!$urlData) {
        throw new Exception('URL not found');
    }
    
    if (!$urlData['assigned_section'] || $urlData['assigned_section'] === 'excluded') {
        throw new Exception('URL must be assigned to a section before parsing');
    }
    
    // Получаем виджет и настройки
    $stmt = $db->prepare("SELECT * FROM widgets WHERE id = ?");
    $stmt->execute([$urlData['widget_id']]);
    $widget = $stmt->fetch();
    
    $stmt = $db->prepare("SELECT * FROM widget_settings WHERE widget_id = ?");
    $stmt->execute([$urlData['widget_id']]);
    $settings = $stmt->fetch();
    
    // Получаем раздел и поля
    $stmt = $db->prepare("SELECT * FROM widget_sections WHERE widget_id = ? AND section_name = ?");
    $stmt->execute([$urlData['widget_id'], $urlData['assigned_section']]);
    $section = $stmt->fetch();
    
    if (!$section) {
        throw new Exception('Section not found');
    }
    
    $stmt = $db->prepare("SELECT * FROM section_fields WHERE section_id = ? ORDER BY field_order");
    $stmt->execute([$section['id']]);
    $fields = $stmt->fetchAll();
    
    // Обновляем статус
    $stmt = $db->prepare("UPDATE discovered_urls SET status = 'parsing' WHERE id = ?");
    $stmt->execute([$urlId]);
    
    // Загружаем страницу
    $context = stream_context_create([
        'http' => [
            'timeout' => 10,
            'user_agent' => 'AIWidget/1.0 (+https://w2.medmaps.ru)',
            'follow_location' => true
        ],
        'ssl' => [
            'verify_peer' => false,
            'verify_peer_name' => false
        ]
    ]);
    
    $html = @file_get_contents($urlData['url'], false, $context);
    
    if ($html === false) {
        throw new Exception('Failed to fetch URL');
    }
    
    // Парсим страницу
    $parser = new PageParser($html, $urlData['url'], $settings['encoding']);
    
    // Извлекаем данные
    $data = [];
    foreach ($fields as $field) {
        $value = $parser->extract($field['css_selectors']);
        if ($value) {
            $data[$field['field_name']] = $value;
        }
    }
    
    // Сохраняем в parsed_items
    $stmt = $db->prepare("
        INSERT INTO parsed_items 
        (widget_id, section_name, source_url)
        VALUES (?, ?, ?)
    ");
    $stmt->execute([
        $urlData['widget_id'],
        $urlData['assigned_section'],
        $urlData['url']
    ]);
    
    $itemId = $db->lastInsertId();
    
    // Сохраняем поля
    foreach ($data as $fieldName => $fieldValue) {
        $stmt = $db->prepare("
            INSERT INTO parsed_fields 
            (item_id, field_name, field_value)
            VALUES (?, ?, ?)
        ");
        $stmt->execute([$itemId, $fieldName, $fieldValue]);
    }
    
    // ===== ПАРСИНГ ДОЧЕРНИХ ЭЛЕМЕНТОВ =====
    // Проверяем есть ли поля с включенным split_into_items
    $splitFields = array_filter($fields, function($field) {
        return !empty($field['split_into_items']) && !empty($field['item_container_selector']);
    });
    
    $totalChildItems = 0;
    if (!empty($splitFields)) {
        $listParser = new ListItemsParser($db);
        
        foreach ($splitFields as $splitField) {
            // Получаем дочерние поля для этого поля
            $stmt = $db->prepare("
                SELECT * FROM section_child_fields 
                WHERE field_id = ? 
                ORDER BY field_order
            ");
            $stmt->execute([$splitField['id']]);
            $childFields = $stmt->fetchAll();
            
            if (empty($childFields)) {
                continue;
            }
            
            // Парсим элементы списка
            $items = $listParser->parseItemsFromHtml(
                $html, 
                $splitField['item_container_selector'], 
                $childFields, 
                $urlData['url']
            );
            
            if (!empty($items)) {
                $saved = $listParser->saveChildItems(
                    $urlData['widget_id'],
                    $urlData['assigned_section'],
                    $itemId,
                    $urlData['url'],
                    $items,
                    $childFields
                );
                $totalChildItems += $saved;
            }
        }
    }
    // ===== КОНЕЦ ПАРСИНГА ДОЧЕРНИХ ЭЛЕМЕНТОВ =====
    
    // Обновляем статус URL
    $stmt = $db->prepare("
        UPDATE discovered_urls 
        SET status = 'parsed', parsed_at = NOW()
        WHERE id = ?
    ");
    $stmt->execute([$urlId]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Page parsed successfully',
        'data' => $data,
        'child_items_count' => $totalChildItems
    ]);
    
} catch (Exception $e) {
    // Обновляем статус на ошибку
    if (isset($urlId) && $urlId) {
        try {
            $stmt = $db->prepare("
                UPDATE discovered_urls 
                SET status = 'parse_error', error_message = ?
                WHERE id = ?
            ");
            $stmt->execute([$e->getMessage(), $urlId]);
        } catch (Exception $dbError) {
            // Игнорируем ошибку обновления
        }
    }
    
    http_response_code(500);
    echo json_encode([
        'error' => $e->getMessage()
    ]);
}

