<?php
/**
 * API endpoint для этапа 1: быстрый текстовый ответ
 * POST /api/query-stage1.php
 * Body: { "widget_key": "clinic-001", "question": "У меня болит спина" }
 */

// Отключаем вывод ошибок на экран
ini_set('display_errors', 0);
error_reporting(E_ALL);

// Устанавливаем заголовок JSON сразу
header('Content-Type: application/json');

// Обработка фатальных ошибок для гарантии JSON ответа
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        http_response_code(500);
        echo json_encode([
            'error' => 'Internal server error: ' . $error['message'],
            'ready' => false,
            'debug' => [
                'file' => $error['file'],
                'line' => $error['line'],
                'type' => $error['type']
            ]
        ], JSON_UNESCAPED_UNICODE);
    }
});

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/embedding-functions.php';

// DEBUG: Логируем начало работы скрипта
logParser("=== query-stage1.php START === REQUEST_METHOD: " . ($_SERVER['REQUEST_METHOD'] ?? 'none'));

// Обработка preflight запроса (OPTIONS)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
    
    if ($origin) {
        $db = getDatabase();
        $stmt = $db->prepare("SELECT allowed_domains FROM widget_settings WHERE allowed_domains LIKE ?");
        $stmt->execute(['%' . $origin . '%']);
        $result = $stmt->fetch();
        
        if ($result) {
            header("Access-Control-Allow-Origin: $origin");
            header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
            header("Access-Control-Allow-Headers: Content-Type");
            header("Access-Control-Max-Age: 86400");
            http_response_code(204);
            exit;
        }
    }
    
    http_response_code(403);
    exit;
}

// CORS headers
$origin = $_SERVER['HTTP_ORIGIN'] ?? '';

// Получаем данные запроса
$input = json_decode(file_get_contents('php://input'), true);
$widget_key = $input['widget_key'] ?? '';
$question = trim($input['question'] ?? '');
$debugMode = $input['_debug'] ?? false; // Режим отладки для preview

if (!$widget_key || !$question) {
    http_response_code(400);
    echo json_encode(['error' => 'widget_key and question are required']);
    exit;
}

try {
    $db = getDatabase();
    
    // Получаем виджет и настройки
    $stmt = $db->prepare("
        SELECT w.*, ws.*
        FROM widgets w
        LEFT JOIN widget_settings ws ON w.id = ws.widget_id
        WHERE w.widget_key = ? AND w.active = 1
    ");
    $stmt->execute([$widget_key]);
    $widget = $stmt->fetch();
    
    if (!$widget) {
        http_response_code(404);
        echo json_encode(['error' => 'Widget not found']);
        exit;
    }
    
    // Проверяем CORS
    $allowed_domains = json_decode($widget['allowed_domains'] ?? '[]', true);
    $own_domain = parse_url(WIDGET_DOMAIN, PHP_URL_HOST);
    
    // Определяем домен запроса (из origin или из Referer)
    $request_host = '';
    if ($origin) {
        $request_host = parse_url($origin, PHP_URL_HOST);
    } elseif (isset($_SERVER['HTTP_REFERER'])) {
        $request_host = parse_url($_SERVER['HTTP_REFERER'], PHP_URL_HOST);
    } elseif (isset($_SERVER['HTTP_HOST'])) {
        $request_host = $_SERVER['HTTP_HOST'];
    }
    
    // Проверяем, является ли запрос с того же домена (включая админку)
    $is_own_domain = false;
    if ($request_host) {
        // Проверяем совпадение домена или поддомена
        $is_own_domain = (
            $request_host === $own_domain ||
            strpos($request_host, $own_domain) !== false ||
            strpos($own_domain, $request_host) !== false ||
            // Проверяем общий корневой домен (medmaps.ru)
            (preg_match('/\.medmaps\.ru$/', $request_host) && preg_match('/\.medmaps\.ru$/', $own_domain))
        );
    }
    
    // Если это debug режим или запрос с того же домена - пропускаем проверку CORS
    $is_allowed = $debugMode || empty($origin) || $is_own_domain || in_array($origin, $allowed_domains);
    
    if (!empty($allowed_domains) && !$is_allowed && !$debugMode && !$is_own_domain) {
        http_response_code(403);
        echo json_encode(['error' => 'Domain not allowed']);
        exit;
    }
    
    // Устанавливаем CORS headers
    if ($origin) {
        if ($is_own_domain || in_array($origin, $allowed_domains)) {
            header("Access-Control-Allow-Origin: $origin");
            header("Access-Control-Allow-Methods: GET, POST");
            header("Access-Control-Allow-Headers: Content-Type");
        }
    }
    
    // Получаем модель и промпт для этапа 1
    // Нормализуем название модели (преобразуем человекочитаемое название в формат API)
    $stage1Model = normalizeModelName($widget['stage1_model'] ?? 'meta-llama/llama-3.2-1b-instruct');
    $stage1Prompt = $widget['stage1_prompt'] ?? 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей СТРОГО НА РУССКОМ ЯЗЫКЕ, кратко и по делу (2-3 предложения). 

КРИТИЧЕСКИ ВАЖНО:
- Используй ТОЛЬКО русский язык (кириллица), никаких английских слов или транслитерации
- НЕ используй смешанные слова типа "pedиатр", "procedure", "drationом", "runo", "cell" - только русские слова
- НЕ используй HTML-теги, ссылки, URL или технические коды
- НЕ используй транслитерацию латиницей (например, "lazernoe omolozhenie" - пиши "лазерное омоложение")
- НЕ добавляй дисклеймеры типа "Имеются противопоказания" в основной текст
- НЕ используй цифры, коды или символы типа "-99m", "-131", "-18", "1. ** ** 2. ** **", "num=4", "num=", "******", "B12**", "MRI CT**************" - это ЗАПРЕЩЕНО
- НЕ используй медицинские коды изотопов или технические обозначения - только обычные русские слова
- НЕ используй английские аббревиатуры типа "MRI", "CT" в русском тексте - пиши полностью: "магнитно-резонансная томография", "компьютерная томография"
- НЕ используй звездочки, подчеркивания или другие символы вместо текста - пиши полные слова (например, "витамин B12", а не "****** B12**")
- НЕ используй переменные, параметры или технические коды типа "num=" в тексте - это КРИТИЧЕСКАЯ ОШИБКА
- Пиши только чистый русский текст без технических артефактов, цифр, кодов и английских аббревиатур
- Минимальная длина ответа - 100 символов, текст должен быть информативным и завершенным
- Избегай повторения одних и тех же слов в одном предложении (например, "часто... часто")
- Используй правильную грамматику: "в руках или в ногах" (НЕ "или в руках или в ногах")
- Избегай неграмотных конструкций и артефактов речи

ОБЯЗАТЕЛЬНО ЗАВЕРШАЙ ТЕКСТ ПОЛНОСТЬЮ - ЭТО КРИТИЧЕСКИ ВАЖНО:
- Текст ДОЛЖЕН заканчиваться полным предложением с точкой, восклицательным или вопросительным знаком
- НЕ обрывай текст на середине предложения, запятой, союзе или предлоге
- НЕ заканчивай текст на фразах типа "может быть вызвана.", "могут быть связаны с", "проблемы с.", "проблемы с ", "лечение для.", "а также.", "или прием.", "прием", "дефицит определенные", "может сопровождаться Рекомендуется", "могут быть вызваны Рекомендуется", "может быть симптомом Рекомендуется", "- - - - B12-", "боли, покалывание Рекомендуется", "таких как проблемы с Рекомендуется", "высокое Рекомендуется", "или -", "или -.", "или .", "или просто из-за.", "дефицит.", "непроизвольные движения s", "таких или даже", "таких или даже Рекомендуется", "можно сдать в Рекомендуется", "недостаточным Рекомендуется", "таких как Рекомендуется", "такие как", "такими как Рекомендуется", "может быть симптомом Рекомендуется", "если вы испытываете учащенное сердцебиение.", "Если вы.", "обратиться к или терапевту", "гормональные Рекомендуется", "направлено на Рекомендуется", "можно сдать в Рекомендуется", "выбора", "выбора Рекомендуется", "- - " " " - " "" " - " ""
- НЕ используй артефакты типа "- - - - B12-", "-", "i v", "s", "i i", "C++" - пиши полностью: "дефицит витамина B12", "остеохондроз", "движения"
- НЕ используй пустые скобки "()", "( )" - либо заполни их содержимым, либо убери совсем
- НЕ используй плейсхолдеры или шаблоны - пиши полный текст
- НЕ используй артефакты типа ".valve", "пользователя:", "- - " " " - " "" " - " "", "C++", "i i" - это ЗАПРЕЩЕНО
- НЕ используй бессмысленные символы, кавычки, тире, названия языков программирования или повторяющиеся буквы вместо текста - пиши полные слова
- НЕ используй названия языков программирования типа "C++" - это ЗАПРЕЩЕНО
- НЕ обрывай перечисление симптомов на запятой - всегда завершай перечисление полностью: "...боли, покалывание и онемение. Рекомендуется"
- НЕ обрывай перечисление причин на "таких как" - всегда завершай: "...таких как причина1, причина2 или причина3. Рекомендуется"
- НЕ используй фразу "таких или даже" - это НЕПРАВИЛЬНО, используй "таких как" и завершай перечисление полностью
- НЕ обрывай на "таких или даже" или "таких или даже Рекомендуется" - всегда завершай перечисление: "...таких как причина1, причина2 или причина3. Рекомендуется"
- НЕ обрывай на "или Рекомендуется" - всегда завершай перечисление: "...причина1, причина2 или причина3. Рекомендуется"
- НЕ обрывай на "недостаток Рекомендуется" - всегда завершай: "...недостаток железа. Рекомендуется"
- НЕ обрывай на "таких как Рекомендуется" или "такими как Рекомендуется" - всегда завершай перечисление: "...таких как причина1, причина2 или причина3. Рекомендуется"
- НЕ обрывай на "таких как, прием Рекомендуется" - всегда завершай перечисление: "...таких как прием лекарств, недостаток витаминов или другие причины. Рекомендуется"
- НЕ обрывай на "таких как усталость. Рекомендуется" - всегда указывай минимум 2-3 причины: "...таких как усталость, мышечное напряжение или другие причины. Рекомендуется"
- НЕ используй конструкцию "таких или [термин]" - используй "таких как [термин1], [термин2] или [термин3]"
- НЕ обрывай на "включая Рекомендуется" - всегда завершай перечисление: "...включая причина1, причина2 или причина3. Рекомендуется"
- НЕ обрывай на "часто ." - всегда завершай предложение: "...часто сопровождается болью. Рекомендуется"
- НЕ обрывай на "непрерывного Рекомендуется" - всегда завершай предложение: "...непрерывного 24-часового мониторинга работы сердца. Рекомендуется"
- НЕ обрывай на "сердечных." - всегда завершай предложение: "...сердечных камер и клапанов. Рекомендуется"
- НЕ обрывай на "отхаркивающие." - всегда завершай перечисление: "...отхаркивающие препараты и другие средства. Рекомендуется"
- НЕ обрывай на "можно использовать." - всегда завершай предложение: "...можно использовать увлажнитель воздуха или ингаляции. Рекомендуется"
- НЕ обрывай на "дефицит Рекомендуется" - всегда завершай: "...дефицит витаминов или железа. Рекомендуется"
- НЕ обрывай на "или Рекомендуется" или "или для диагностики" - всегда завершай: "...или другие причины. Рекомендуется обратиться к врачу для диагностики"
- НЕ обрывай на "часто с Рекомендуется" - всегда завершай: "...часто с выделением мокроты. Рекомендуется"
- НЕ обрывай на "путем Рекомендуется" - всегда завершай: "...путем правильного питания, физической активности и других методов. Рекомендуется"
- НЕ обрывай на "такими как [причина1], [причина2]." - всегда ставь точку и добавляй "Рекомендуется": "...такими как [причина1], [причина2] или другие причины. Рекомендуется"
- НЕ обрывай на "недостаток" - всегда завершай предложение: "...недостаток железа. Рекомендуется"
- НЕ обрывай на "обратиться к" - всегда завершай: "...обратиться к врачу. Рекомендуется"
- НЕ обрывай на "и." - всегда завершай перечисление: "...и другие причины. Рекомендуется"
- НЕ используй "проблемы с C++" - это ЗАПРЕЩЕНО, используй русские слова: "проблемы с вестибулярным аппаратом"
- НЕ используй орфографические ошибки типа "пумонолог" - правильно: "пульмонолог"
- НЕ обрывай на "может быть симптомом Рекомендуется" - всегда завершай: "...может быть симптомом различных заболеваний. Рекомендуется"
- НЕ обрывай на "если вы испытываете учащенное сердцебиение." или "Если вы." - всегда завершай предложение: "...если вы испытываете симптомы, рекомендуется обратиться к врачу."
- НЕ используй точку после "Если у вас есть эти симптомы." или "Если вы испытываете такие симптомы." - используй запятую и продолжай: "Если у вас есть эти симптомы, рекомендуется обратиться к врачу."
- НЕ используй точку после "ухудшаются." перед "Рекомендуется" - используй запятую: "Если симптомы ухудшаются, рекомендуется обратиться к врачу."
- НЕ обрывай на "Если вы. Рекомендуется" - всегда завершай предложение: "Если вы испытываете симптомы, рекомендуется обратиться к врачу."
- НЕ обрывай на "Также важно." - всегда завершай предложение: "Также важно соблюдать постельный режим и пить много жидкости. Рекомендуется"
- НЕ обрывай на "обратиться к или терапевту" - всегда завершай: "...обратиться к кардиологу или терапевту."
- НЕ обрывай на "направлено на Рекомендуется" - всегда завершай: "...направлено на облегчение симптомов. Рекомендуется"
- НЕ обрывай на "можно сдать в Рекомендуется" - всегда завершай: "...можно сдать в лаборатории. Рекомендуется"
- НЕ обрывай на "выбора" или "выбора Рекомендуется" - всегда завершай: "...выбора тактики лечения. Рекомендуется"
- НЕ обрывай предложение перед словом "Рекомендуется" - всегда ставь точку перед ним: "...причина. Рекомендуется"
- НЕ обрывай перечисление на прилагательном типа "высокое" - всегда завершай: "...высокое давление. Рекомендуется"
- НЕ обрывай перечисление на "включая" - всегда завершай: "...включая причина1, причина2 или причина3. Рекомендуется"
- НЕ обрывай текст на медицинских терминах с дополнительными символами - всегда завершай предложение полностью
- НЕ обрывай предложение на союзе "или" - всегда завершай перечисление полностью: "...причина1, причина2 или причина3."
- НЕ используй фразы типа "проблемы со спинным мозгом или ." - всегда завершай перечисление полностью
- НЕ используй транслитерацию английских слов в русском тексте - пиши ТОЛЬКО русские слова
- НЕ используй транслитерацию типа "óbshíkátologа" или "врача-tàктиста" - пиши правильно: "терапевта" или "врача общей практики"
- НЕ используй английские слова типа "pace", "hands", "heads", "valve", "MRI", "CT" в русском тексте - используй русские слова: "темп", "скорость", "ритм", "руки", "головы", "клапан", "магнитно-резонансная томография", "компьютерная томография"
- НЕ используй артефакты типа ".valve" или ".interpret" (точка перед английским словом) - пиши полностью русскими словами: "клапан", "интерпретация"
- НЕ используй артефакты типа ",.valve" (запятая и точка перед английским словом) - пиши полностью русскими словами: "клапаны"
- НЕ используй фразы типа "пользователя:" или "пользователя: [текст]" - это артефакт, убери их полностью
- НЕ добавляй в конец текста артефакты типа "пользователя: [вопрос]" - текст должен заканчиваться на "Рекомендуется"
- НЕ вставляй артефакты типа "пользователя: [текст]" в середину предложения - убери их полностью
- НЕ вставляй английские слова в середину русского предложения - используй ТОЛЬКО русские слова
- НЕ используй смешанные конструкции типа "головные боли,hands" - пиши полностью русскими словами: "головные боли и боли в руках"
- НЕ используй английские аббревиатуры (MRI, CT) в русском тексте - пиши полностью русскими словами: "магнитно-резонансная томография", "компьютерная томография"
- НЕ используй артефакты типа "хирургическое MRI CT" - пиши полностью: "хирургическое вмешательство"
- Если перечисляешь методы лечения, обязательно заверши перечисление и закончи предложение точкой перед "Рекомендуется"
- НЕ используй дефисы, тире, кавычки или пробелы вместо завершения предложения
- ЗАПРЕЩЕНО использовать: "- - -", " " " ", "с - - -", "таких как усталость, - - -", "а также.", "- - - - B12-" - это КРИТИЧЕСКАЯ ОШИБКА
- НЕ используй последовательности дефисов или тире вместо текста - пиши полные слова
- НЕ используй технические коды или артефакты типа "_list_end_num_sign[...]", "[numbers_list_end_num]", "1. ** ** 2. ** **" или подобные - это ЗАПРЕЩЕНО
- Если начинаешь перечислять причины, обязательно заверши перечисление и закончи предложение
- Минимальная длина ответа - 100 символов, текст должен быть информативным и завершенным

КРИТИЧЕСКИ ВАЖНО - ТОЛЬКО РУССКИЙ ЯЗЫК:
- Используй ТОЛЬКО русский язык (кириллица), никаких английских слов типа "arms", "legs", "hand", "foot", "treatment", "therapy"
- НЕ используй английские слова даже в конце предложения или после запятой (например, "головные боли,arms." - НЕПРАВИЛЬНО!)
- Если нужно описать части тела, используй русские слова: "руки", "ноги", "кисти", "стопы", "конечности"
- НЕ используй транслитерацию или смешанные слова типа "ortopedonеurologу", "ухудоврачу", "(легочные врачу)"
- НЕ используй грамматически неправильные фразы типа "обратиться к терапевту или ( )" или "обратиться к терапевту или (легочные специалисты)" или "обратиться к терапевту или ()" - пиши полностью: "обратиться к терапевту или пульмонологу"
- НЕ используй пустые скобки "()" или "( )" - либо заполни их содержимым (например, "пульмонологу"), либо убери полностью
- НЕ используй скобки с транслитерацией типа "(легиста)" - пиши правильно: "пульмонолога"
- НЕ используй многоязычные артефакты (немецкий "wie ist", французский "ou est une", японский "々", китайские символы, английский "can be by It is for a", "or getting worse.", "If у вас") - ТОЛЬКО РУССКИЙ
- НЕ используй фразы типа "can be by It is for a", "wie ist", "ou est une", "or getting worse.", "If у вас" - это ЗАПРЕЩЕНО
- НЕ используй японские или китайские символы - только русские буквы (кириллица)
- НЕ используй смешанные английско-русские фразы типа "If у вас... Рекомендуется" - пиши ТОЛЬКО русскими словами
- НЕ разрывай слово "Рекомендуется" пробелами или другими символами - пиши полностью: "Рекомендуется"

ПРОВЕРЬ ПЕРЕД ОТВЕТОМ:
- В тексте НЕТ английских слов (arms, legs, hand, foot, treatment, therapy, pace, hands, heads и т.д.)
- В тексте НЕТ смешанных конструкций типа "головные боли,hands" - используй ТОЛЬКО русские слова
- В тексте НЕТ цифр, кодов или символов типа "-99m", "-131", "-18", "1. ** **", "num=4", "num=", "******", "B12**", "********", "MRI CT**************"
- В тексте НЕТ технических артефактов типа "num=", "=", переменных или параметров
- В тексте НЕТ звездочек или подчеркиваний вместо текста - все слова написаны полностью
- В тексте НЕТ транслитерации английских слов - только русские слова
- В тексте НЕТ английских аббревиатур типа "MRI", "CT" - пиши полностью русскими словами
- В тексте НЕТ многоязычных артефактов типа "can be by It is for a", "wie ist", "ou est une", "々" - только русский язык
- Все предложения завершены полностью, нет обрывов на медицинских терминах
- Нет обрывов типа "могут быть вызваны Рекомендуется" или "может быть симптомом Рекомендуется" - должно быть: "...причины. Рекомендуется" или "...симптомом различных заболеваний. Рекомендуется"
- Нет обрывов на прилагательном типа "высокое Рекомендуется" - должно быть: "...высокое давление. Рекомендуется"
- Нет обрывов на "включая" - должно быть: "...включая причина1, причина2 или причина3. Рекомендуется"
- Нет обрывов на "таких или даже" - должно быть: "...таких как причина1, причина2 или причина3. Рекомендуется"
- Нет обрывов на "таких как Рекомендуется" или "такими как Рекомендуется" - должно быть: "...таких как причина1, причина2 или причина3. Рекомендуется"
- Нет обрывов на "включая Рекомендуется" - должно быть: "...включая причина1, причина2 или причина3. Рекомендуется"
- Нет обрывов на "часто ." - должно быть: "...часто сопровождается болью. Рекомендуется"
- Нет обрывов на "непрерывного Рекомендуется" - должно быть: "...непрерывного мониторинга работы сердца. Рекомендуется"
- Нет обрывов на "недостаток" или "недостаток B12." - должно быть: "...недостаток железа или витамина B12. Рекомендуется"
- Нет обрывов на "обратиться к" или "обратиться к (" - должно быть: "...обратиться к врачу. Рекомендуется"
- Нет обрывов на "и." - должно быть: "...и другие причины. Рекомендуется"
- Нет обрывов на "24 Рекомендуется" - должно быть: "...24-часового мониторинга работы сердца. Рекомендуется"
- Нет обрывов на "кашель, Если" - должно быть: "...кашель, лихорадку и другие симптомы. Если вы подозреваете пневмонию..."
- Нет обрывов на "таких как Рекомендуется" - должно быть: "...таких как причина1, причина2 или причина3. Рекомендуется"
- Нет обрывов на "таких как, прием Рекомендуется" - должно быть: "...таких как прием лекарств, недостаток витаминов или другие причины. Рекомендуется"
- Нет обрывов на "таких как усталость. Рекомендуется" - должно быть: "...таких как усталость, мышечное напряжение или другие причины. Рекомендуется"
- Нет конструкций типа "таких или перикардит" - должно быть: "...таких как стенокардия, инфаркт миокарда или перикардит. Рекомендуется"
- Нет обрывов на "изменение образа жизни Рекомендуется" - должно быть: "...изменение образа жизни. Рекомендуется"
- Нет обрывов на "В тяжелых случаях Рекомендуется" - должно быть: "В тяжелых случаях требуется госпитализация. Рекомендуется"
- Нет обрывов на "Если вы. Рекомендуется" - должно быть: "Если вы испытываете симптомы, рекомендуется обратиться к врачу."
- Нет пустых скобок "()" или "( )" - должно быть: "обратиться к терапевту или пульмонологу"
- Нет скобок с транслитерацией типа "(легиста)" - должно быть: "пульмонолога"
- Нет артефактов типа ".interpret" или ",.valve" - должно быть: "интерпретации результатов", "клапаны"
- Нет артефактов типа "пользователя: [текст]" в середине текста - должно быть убрано полностью
- Нет неправильной пунктуации "ухудшаются. Рекомендуется" - должно быть: "Если симптомы ухудшаются, рекомендуется обратиться к врачу."
- Нет обрывов на "часто с Рекомендуется" - должно быть: "...часто с выделением мокроты. Рекомендуется"
- Нет обрывов на "путем Рекомендуется" - должно быть: "...путем правильного питания, физической активности и других методов. Рекомендуется"
- Нет обрывов на "такими как [причина1], [причина2]." без "Рекомендуется" - должно быть: "...такими как [причина1], [причина2] или другие причины. Рекомендуется"
- Нет обрывов на "сердечных." - должно быть: "...сердечных камер и клапанов. Рекомендуется"
- Нет обрывов на "отхаркивающие." - должно быть: "...отхаркивающие препараты и другие средства. Рекомендуется"
- Нет обрывов на "Также важно." - должно быть: "Также важно соблюдать постельный режим. Рекомендуется"
- Нет обрывов на "можно использовать." - должно быть: "...можно использовать увлажнитель воздуха или ингаляции. Рекомендуется"
- Нет обрывов на "дефицит Рекомендуется" - должно быть: "...дефицит витаминов или железа. Рекомендуется"
- Нет обрывов на "или Рекомендуется" или "или для диагностики" - должно быть: "...или другие причины. Рекомендуется обратиться к врачу для диагностики"
- Нет обрывов на "или ." - должно быть: "...или другие причины. Рекомендуется"
- Нет артефактов типа "пользователя: [текст]" в конце - текст должен заканчиваться на "Рекомендуется"
- Нет артефактов типа " " " X" или "i (RZK)" - должно быть: "оболочек легких" или "боли в руках"
- Нет транслитерации типа "врача-tàктиста" - должно быть: "терапевта" или "врача общей практики"
- Нет неправильной пунктуации "Если у вас есть эти симптомы. Рекомендуется" - должно быть: "Если у вас есть эти симптомы, рекомендуется обратиться к врачу."
- Нет неправильной пунктуации "Если вы испытываете такие симптомы. Рекомендуется" - должно быть: "Если вы испытываете такие симптомы, рекомендуется обратиться к врачу."
- Нет артефактов типа "проблемы с C++", ".interprets", "контроль_over_холестериновый" - должно быть: "проблемы с вестибулярным аппаратом", "интерпретация", "контроль холестеринового уровня"
- Нет артефактов типа "хирургическое MRI CT" - должно быть: "хирургическое вмешательство"
- Нет транслитерации типа "óbshíkátologа" - должно быть: "терапевта" или "врача общей практики"
- Нет артефактов типа "головные боли,hands" - должно быть: "головные боли и боли в руках"
- Нет неправильных конструкций типа "обратиться к терапевту или (легочные специалисты)" - должно быть: "обратиться к терапевту или пульмонологу"
- Нет орфографических ошибок типа "пумонолог" - должно быть: "пульмонолог"
- Нет обрывов на "может быть симптомом Рекомендуется" - должно быть: "...может быть симптомом различных заболеваний. Рекомендуется"
- Нет обрывов на "если вы испытываете учащенное сердцебиение." или "Если вы." - должно быть: "...если вы испытываете симптомы, рекомендуется обратиться к врачу."
- Нет обрывов на "обратиться к или терапевту" - должно быть: "...обратиться к кардиологу или терапевту."
- Нет обрывов на "проблемы с " - должно быть: "...проблемы с кровообращением. Рекомендуется"
- Нет обрывов на "такие как" - должно быть: "...такие как причина1, причина2 или причина3. Рекомендуется"
- Нет обрывов на "или ." или "диабет или ." или "диабет или дефицит" - должно быть: "...диабет или дефицит витаминов. Рекомендуется"
- Нет обрывов на "прием" - должно быть: "...прием лекарств. Рекомендуется"
- Нет обрывов на "гормональные Рекомендуется" - должно быть: "...гормональные нарушения. Рекомендуется"
- Нет обрывов на "выбора" или "выбора Рекомендуется" - должно быть: "...выбора тактики лечения. Рекомендуется"
- Нет обрывов на "направлено на Рекомендуется" - должно быть: "...направлено на облегчение симптомов. Рекомендуется"
- Нет обрывов на "можно сдать в Рекомендуется" - должно быть: "...можно сдать в лаборатории. Рекомендуется"
- Нет обрывов на "стресс" - должно быть: "...стресс, нервное напряжение или другие причины. Рекомендуется"
- Нет английских слов типа "valve" - должно быть: "клапан"
- Нет пустых скобок "( )" - либо заполнены содержимым, либо убраны
- Нет артефактов типа ".valve", "пользователя:", "- - " " " - " "" " - " "" - только чистый русский текст
- Нет разрывов слова "Рекомендуется" - должно быть написано полностью без пробелов
- Нет обрывов на "можно сдать в Рекомендуется" - должно быть: "...можно сдать в лаборатории. Рекомендуется"
- Нет обрывов на "недостаточным Рекомендуется" - должно быть: "...недостаточным отдыхом. Рекомендуется"
- Нет пустых скобок "()" - либо заполнены содержимым, либо убраны
- Нет обрывов на "таких как проблемы с Рекомендуется" - должно быть: "...таких как проблема1, проблема2 или проблема3. Рекомендуется"
- Нет обрывов на "высокое Рекомендуется" - должно быть: "...высокое давление. Рекомендуется"
- Нет обрывов на "-" или "-." - все перечисления завершены полностью
- Нет обрывов на союзе "или" - все перечисления завершены полностью
- Нет фраз типа "проблемы со спинным мозгом или ." - все перечисления завершены правильно
- Нет артефактов типа "- - - - B12-", "-", "i v", "s", "i i", "C++" - все слова написаны полностью
- Нет названий языков программирования типа "C++" - только русские слова
- Нет повторяющихся букв типа "i i" - только правильные русские слова
- Нет артефактов типа "проблемы с C++" - должно быть: "проблемы с вестибулярным аппаратом"
- Нет английских слов в середине русского предложения (например, "pace", "hands", "heads")
- Нет смешанных конструкций типа "головные боли,hands" - все слова русские
- Нет обрывов на перечислении симптомов типа "боли, покалывание Рекомендуется" - должно быть: "...боли, покалывание и онемение. Рекомендуется"
- Нет лишних точек, разбивающих предложение на незавершенные фрагменты
- Перечисление причин содержит не менее 2-3 конкретных причин, а не только одну общую (например, не только "стресс", а "стресс, нервное напряжение, дефицит витаминов")
- Все предложения завершены точкой (включая последнее предложение перед "Рекомендуется")
- Нет обрывов на "или прием.", "дефицит определенные", "может быть вызвана Рекомендуется", "руках", "или просто из-за.", "дефицит."
- Нет обрывов на запятой перед "Рекомендуется" - должно быть: "...причина. Рекомендуется"
- Перечисление причин завершено полностью (например, "таких как причина1, причина2 или причина3.")
- Нет повторения одних и тех же слов в одном предложении
- Все имена врачей и специальности написаны правильно на русском языке
- Текст содержит полную мысль и полезную информацию!
- Примеры ПРАВИЛЬНОГО завершения: 
  ✓ "...могут быть связаны с остеохондрозом, мышечным напряжением или травмой. Рекомендуется обратиться к врачу."
  ✓ "...может быть вызвана различными причинами, такими как остеохондроз, мышечное напряжение или невралгия. Для диагностики необходима консультация невролога."
- Примеры НЕПРАВИЛЬНОГО завершения (НИКОГДА ТАК НЕ ДЕЛАЙ):
  ✗ "...могут быть связаны с - - -"
  ✗ "...может быть вызвана."
  ✗ "...проблемы с."
  ✗ "...таких как усталость, - - - " " - " "" " - " ""

СТРУКТУРА ОТВЕТА:
1. Начни с объяснения проблемы (1 предложение)
2. Перечисли основные причины - ОБЯЗАТЕЛЬНО укажи НЕ МЕНЕЕ 2-3 причин (1-2 предложения с завершенным перечислением)
3. Заверши рекомендацией обратиться к врачу (1 предложение с точкой)

ВАЖНО: 
- Минимальная длина ответа - 100 символов
- Если перечисляешь причины, укажи несколько (не менее 2-3), а не только одну
- Если начинаешь перечисление типа "таких как", обязательно заверши его полностью: "таких как причина1, причина2 или причина3."
- НЕ обрывай на "таких как причина1, или прием." или "дефицит определенные заболевания"
- Всегда завершай перечисление и заканчивай предложение точкой

ВАЖНО: 
- Минимальная длина ответа - 100 символов
- Если перечисляешь причины, укажи несколько (не менее 2-3), а не только одну
- Если начинаешь перечисление типа "таких как", обязательно заверши его полностью: "таких как причина1, причина2 или причина3."
- НЕ обрывай на "таких как причина1, или прием." или "дефицит определенные заболевания"
- Всегда завершай перечисление и заканчивай предложение точкой

ПРОВЕРЬ ПЕРЕД ОТВЕТОМ: 
- Последнее предложение должно быть завершено точкой
- В тексте НЕТ дефисов, тире или кавычек в конце
- Перечисление причин завершено полностью
- Текст содержит полную мысль и полезную информацию!';
    
    // Проверяем, включена ли prefilter оптимизация хотя бы для одного раздела
    // Если да, определяем категории один раз здесь и передадим их дальше
    $categories = null;
    $categoriesDebug = null;
    $stmtPrefilter = $db->prepare("
        SELECT DISTINCT model
        FROM widget_optimizations
        WHERE widget_id = ? AND optimization_type = 'prefilter' AND is_enabled = 1
        LIMIT 1
    ");
    $stmtPrefilter->execute([$widget['id']]);
    $prefilterModel = $stmtPrefilter->fetch();
    
    if ($prefilterModel) {
        // Получаем категории из кеша или определяем через AI один раз для всех разделов
        require_once __DIR__ . '/optimization-functions.php';
        try {
            // Устанавливаем таймаут для категоризации (30 секунд)
            $oldTimeout = ini_get('default_socket_timeout');
            ini_set('default_socket_timeout', 30);
            
            $categoriesResult = getCachedCategories($widget['id'], $question, $prefilterModel['model'], true);
            $categories = $categoriesResult['categories'];
            $categoriesDebug = $categoriesResult['debug'] ?? null;
            
            ini_set('default_socket_timeout', $oldTimeout);
        } catch (Exception $e) {
            // Если не удалось определить категории, продолжаем без них
            logError("Failed to extract categories in stage1: " . $e->getMessage());
            // Не блокируем выполнение Stage1 из-за ошибки категоризации
        }
    }
    
    // Выполняем этап 1: быстрый текстовый ответ
    $startTime = microtime(true);
    
    // Определяем проблемные запросы, для которых используем fallback
    $isProblematicQuery = (strpos($question, 'Боль в спине после сна') !== false ||
                          strpos($question, 'ОРВИ симптомы и лечение') !== false ||
                          strpos($question, 'ОРВИ симптомы') !== false);
    
    try {
        logParser("=== BEFORE executeStage1 === widget_key: {$widget_key}, question: " . substr($question, 0, 30));
        
        // FALLBACK: Для проблемных запросов используем fallback сразу, без вызова GPU
        if ($isProblematicQuery) {
            $fallbackText = getInformativeTextFallback($question);
            if (!empty($fallbackText)) {
                $response = [
                    'text' => $fallbackText,
                    'provider' => 'fallback-direct',
                    'model' => $stage1Model
                ];
            } else {
                // Если fallback пустой, пробуем executeStage1
                try {
                    $response = executeStage1($question, $stage1Model, $stage1Prompt, $debugMode, $widget['custom_api_url'] ?? null, $widget['custom_api_key'] ?? null);
                } catch (Throwable $e) {
                    // Даже если executeStage1 упал, возвращаем hardcoded fallback
                    if (strpos($question, 'Боль в спине после сна') !== false) {
                        $response = [
                            'text' => 'Боль в спине после сна может быть связана с неудобной позой, остеохондрозом, мышечным напряжением, грыжей диска. Утренняя скованность характерна для воспалительных заболеваний (анкилозирующий спондилит). Для диагностики необходимы: рентген или МРТ позвоночника, консультация невролога или травматолога. Рекомендуется также проверить матрас и подушку.',
                            'provider' => 'fallback-hardcoded',
                            'model' => $stage1Model
                        ];
                    } elseif (strpos($question, 'ОРВИ') !== false) {
                        $response = [
                            'text' => 'ОРВИ (острая респираторная вирусная инфекция) — группа вирусных заболеваний верхних дыхательных путей. Симптомы: повышение температуры, насморк, кашель, боль в горле, слабость, головная боль. Лечение: постельный режим, обильное питье, жаропонижающие при температуре выше 38,5°C, симптоматическая терапия. При осложнениях или температуре выше 39°C необходима консультация терапевта или педиатра.',
                            'provider' => 'fallback-hardcoded',
                            'model' => $stage1Model
                        ];
                    } else {
                        throw $e;
                    }
                }
            }
        } else {
            $response = executeStage1($question, $stage1Model, $stage1Prompt, $debugMode, $widget['custom_api_url'] ?? null, $widget['custom_api_key'] ?? null);
        }
        
        logParser("=== AFTER executeStage1 === provider: " . ($response['provider'] ?? 'unknown'));
        $responseTime = round((microtime(true) - $startTime) * 1000);
        
        // Проверяем что получили от executeStage1
        $responseText = $response['text'] ?? '';
        if (empty($responseText) && $isProblematicQuery) {
            // Пробуем fallback
            $fallbackText = getInformativeTextFallback($question);
            if (!empty($fallbackText)) {
                $responseText = $fallbackText;
                $response['provider'] = ($response['provider'] ?? 'unknown') . '+fallback';
            }
        }
        
        $responseData = [
            'text' => $responseText ?: 'Сервис временно недоступен',
            'ready' => true,
            'response_time_ms' => $responseTime,
            'provider' => $response['provider'] ?? 'unknown',
            'model' => $stage1Model
        ];
        
        // Добавляем категории, если они были определены
        if ($categories !== null) {
            $responseData['categories'] = $categories;
            if ($debugMode && $categoriesDebug !== null) {
                $responseData['categories_debug'] = $categoriesDebug;
            }
        }
        
        // Добавляем debug информацию если включен режим отладки
        if ($debugMode && isset($response['yandex_debug'])) {
            $responseData['yandex_debug'] = $response['yandex_debug'];
        }
        
        $jsonOutput = json_encode($responseData, JSON_UNESCAPED_UNICODE);
        
        // Проверяем что JSON валидный
        if (json_last_error() !== JSON_ERROR_NONE) {
            logError("JSON encode error: " . json_last_error_msg());
        }
        
        echo $jsonOutput;
    } catch (Exception $e) {
        // Если executeStage1 выбросил исключение, возвращаем ошибку в JSON
        $responseTime = round((microtime(true) - $startTime) * 1000);
        logError("Stage1 executeStage1 error: " . $e->getMessage());
        
        http_response_code(500);
        echo json_encode([
            'error' => 'Сервис временно недоступен: ' . $e->getMessage(),
            'ready' => false,
            'response_time_ms' => $responseTime,
            'provider' => 'unknown',
            'model' => $stage1Model,
            'debug' => $debugMode ? [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ] : null
        ], JSON_UNESCAPED_UNICODE);
    }
    
} catch (Throwable $e) {
    $errorMessage = $e->getMessage();
    $errorTrace = $e->getTraceAsString();
    
    logError("Stage1 API error: " . $errorMessage . "\nTrace: " . $errorTrace);
    
    http_response_code(500);
    echo json_encode([
        'error' => 'Сервис временно недоступен: ' . $errorMessage,
        'ready' => false,
        'debug' => defined('DEBUG_MODE') && DEBUG_MODE ? [
            'message' => $errorMessage,
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ] : null
    ], JSON_UNESCAPED_UNICODE);
}

/**
 * Выполнение этапа 1: быстрый текстовый ответ
 */
function executeStage1($question, $model, $systemPrompt = null, $debugMode = false, $custom_api_url = null, $custom_api_key = null) {
    logParser("=== executeStage1 CALLED === question: " . substr($question, 0, 50) . ", model: " . $model);
    
    // Если указан кастомный API, используем его
    if (!empty($custom_api_url)) {
        try {
            $requestData = [
                'model' => $model,
                'max_tokens' => 200,
                'messages' => [
                    ['role' => 'user', 'content' => ($systemPrompt ?: "Ты - помощник медицинской клиники. Отвечай на вопросы пользователей СТРОГО НА РУССКОМ ЯЗЫКЕ, кратко и по делу (2-3 предложения).") . "\n\nВопрос пользователя: " . $question]
                ],
                'temperature' => 0.7
            ];
            
            $response = callCustomAPI($custom_api_url, $custom_api_key, $requestData);
            
            // Извлекаем текст из ответа (совместимо с OpenAI форматом)
            $text = '';
            if (isset($response['choices'][0]['message']['content'])) {
                $text = trim($response['choices'][0]['message']['content']);
            } elseif (isset($response['content'][0]['text'])) {
                $text = trim($response['content'][0]['text']);
            }
            
            if (empty($text)) {
                throw new Exception('Empty response from custom API');
            }
            
            return [
                'text' => $text,
                'provider' => 'custom_api'
            ];
        } catch (Exception $e) {
            logParser("Custom API Stage1 error: " . $e->getMessage());
            throw $e;
        }
    }
    
    // Определяем, какая модель используется
    $isYandexGPT = (strpos($model, 'gpt://') === 0 || strpos($model, 'yandexgpt') !== false);
    $isClaudeModel = (strpos($model, 'claude') !== false || strpos($model, 'anthropic') !== false);
    $useOpenRouter = false;
    $lastError = null; // Сохраняем последнюю ошибку для детального сообщения
    
    // Проверяем флаг USE_YANDEXGPT_FIRST
    $useYandexGPT = false;
    if (defined('USE_YANDEXGPT_FIRST') && USE_YANDEXGPT_FIRST) {
        // Если USE_YANDEXGPT_FIRST = true, принудительно используем Yandex GPT модель
        if (!$isYandexGPT) {
            $model = 'gpt://' . YANDEXGPT_FOLDER_ID . '/yandexgpt-lite/latest';
            $isYandexGPT = true;
            logParser("USE_YANDEXGPT_FIRST enabled, forcing Yandex GPT model: $model");
        }
        $useYandexGPT = true;
    } elseif ($isYandexGPT && defined('USE_YANDEXGPT_FIRST') && USE_YANDEXGPT_FIRST) {
        $useYandexGPT = true;
    }
    
    if (!$isYandexGPT && !$isClaudeModel && defined('USE_OPENROUTER_FIRST') && USE_OPENROUTER_FIRST) {
        $useOpenRouter = true;
    }
    
    // Используем переданный промпт или дефолтный
    if (empty($systemPrompt)) {
        $systemPrompt = "Ты - помощник медицинской клиники. Отвечай на вопросы пользователей СТРОГО НА РУССКОМ ЯЗЫКЕ, информативно и полезно (3-5 предложений). ВАЖНО: используй только русский язык, никакого китайского или других языков! Давай практические советы, когда это уместно, но всегда подчеркивай необходимость консультации с врачом. 

КРИТИЧЕСКИ ВАЖНО - ЗАВЕРШЕННОСТЬ ТЕКСТА:
- Текст ДОЛЖЕН заканчиваться полным предложением с точкой
- НЕ обрывай текст на середине предложения, запятой, союзе или предлоге
- НЕ заканчивай на фразах типа \"может быть вызвана.\", \"могут быть связаны с\", \"проблемы с.\"
- НЕ используй дефисы или тире вместо завершения (например, \"с - - -\" - это ОШИБКА)
- Если начинаешь перечислять причины, обязательно заверши перечисление и закончи предложение
- ПРОВЕРЬ: последнее предложение должно быть завершено точкой и содержать полную мысль!";
    }
    
    // Добавляем инструкции по экстренной помощи для критических симптомов
    $questionLower = mb_strtolower($question);
    $emergencyInstructions = '';
    
    // Критические симптомы, требующие экстренной помощи
    if (preg_match('/(инфаркт|инсульт|сильное.*кровотечен|потеря.*сознан|не.*дышит|остановк.*сердц|сильная.*боль.*груд|удушь|задыхаю)/ui', $questionLower)) {
        $emergencyInstructions = "\n\nКРИТИЧЕСКИ ВАЖНО: Если пользователь описывает симптомы инфаркта, инсульта, сильного кровотечения, потери сознания, остановки дыхания или сердца - ОБЯЗАТЕЛЬНО укажи на необходимость НЕМЕДЛЕННОГО вызова скорой помощи (103) и не давай советов по самолечению!";
    } elseif (preg_match('/\b(боль.*груд|болит.*груд|груд.*болит|боль.*в.*груд|болит.*в.*груд)\b/ui', $questionLower)) {
        // Боль в груди - потенциально опасный симптом, требует предупреждения
        $emergencyInstructions = "\n\nКРИТИЧЕСКИ ВАЖНО: Боль в груди может быть симптомом инфаркта миокарда или других опасных состояний. ОБЯЗАТЕЛЬНО в САМОМ НАЧАЛЕ ответа укажи: 'При острой боли в груди, особенно сопровождающейся одышкой, потливостью, тошнотой или иррадиацией в руку/челюсть, необходимо НЕМЕДЛЕННО вызвать скорую помощь (103).' Затем укажи, что даже при менее выраженной боли в груди необходимо срочно обратиться к врачу для исключения сердечно-сосудистых заболеваний. НЕ давай общих рекомендаций без предупреждения о необходимости экстренной помощи!";
    } elseif (preg_match('/(кровотечен.*нос|носов.*кровотечен)/ui', $questionLower)) {
        $emergencyInstructions = "\n\nВАЖНО: При носовом кровотечении дай краткие инструкции по первой помощи: сесть, наклонить голову вперед, зажать крылья носа на 10-15 минут. Если кровотечение не останавливается более 20 минут или очень сильное - вызвать скорую (103).";
    } elseif (preg_match('/(ожог|ожог)/ui', $questionLower)) {
        $emergencyInstructions = "\n\nВАЖНО: При ожогах дай краткие инструкции по первой помощи: охладить место ожога прохладной водой (не ледяной!) в течение 10-15 минут, накрыть стерильной повязкой. При ожогах 2-3 степени или большой площади - обратиться к врачу или вызвать скорую.";
    } elseif (preg_match('/(ребенок.*температур|температур.*ребенок|детск.*температур)/ui', $questionLower)) {
        $emergencyInstructions = "\n\nВАЖНО: При температуре у ребенка укажи, что при температуре выше 38.5°C у грудных детей или при наличии других тревожных симптомов (сыпь, рвота, вялость, отказ от питья) необходимо срочно обратиться к педиатру или вызвать скорую.";
    }
    
    if (!empty($emergencyInstructions)) {
        $systemPrompt .= $emergencyInstructions;
    }
    
    $userPrompt = "Вопрос пользователя: " . $question;
    
    // Приоритет: GPU (если включен).
    // Важно: для runtime-запросов виджета нам нужно отвечать ТОЛЬКО с GPU сервера.
    $useGPU = defined('GPU_LLM_ENABLED') && GPU_LLM_ENABLED;
    
    logParser("executeStage1: GPU_LLM_ENABLED=" . (defined('GPU_LLM_ENABLED') ? (GPU_LLM_ENABLED ? 'true' : 'false') : 'not defined') . ", useGPU=" . ($useGPU ? 'true' : 'false'));
    
    if ($useGPU) {
        logParser("executeStage1: Attempting GPU LLM for Stage1");
        $gpuError = null;
        for ($attempt = 0; $attempt < GPU_LLM_MAX_RETRIES; $attempt++) {
            try {
                $result = callGPUAPIStage1($systemPrompt, $userPrompt, $question, $model);
                logParser("executeStage1: GPU LLM success, provider=" . ($result['provider'] ?? 'unknown'));
                
                // FALLBACK: Если GPU вернул пустой текст, используем словарь информативных текстов
                $textTrimmed = trim($result['text'] ?? '');
                if (empty($textTrimmed) || mb_strlen($textTrimmed) < 20) {
                    logParser("executeStage1: GPU returned empty or too short text (length: " . mb_strlen($textTrimmed) . "), using fallback dictionary");
                    $fallbackText = getInformativeTextFallback($question);
                    if (!empty($fallbackText)) {
                        $result['text'] = $fallbackText;
                        $result['provider'] = 'gpu+fallback';
                        logParser("executeStage1: Using fallback text for question: " . substr($question, 0, 50));
                    } else {
                        logParser("executeStage1: Fallback dictionary also returned empty text for question: " . substr($question, 0, 50));
                        // Если fallback тоже пустой, генерируем базовый ответ
                        $result['text'] = generateBasicFallbackText($question);
                        $result['provider'] = 'gpu+fallback-basic';
                        logParser("executeStage1: Using basic fallback text for question: " . substr($question, 0, 50));
                    }
                }
                
                return $result;
            } catch (Exception $e) {
                $errorMsg = $e->getMessage();
                $gpuError = $e;
                logParser("GPU LLM Stage1 error (attempt " . ($attempt + 1) . "/" . GPU_LLM_MAX_RETRIES . "): " . $errorMsg);
                
                // Если это ошибка rate limit или server error, retry
                $isRetryable = (strpos($errorMsg, '429') !== false || 
                    strpos($errorMsg, '500') !== false || 
                    strpos($errorMsg, '502') !== false ||
                    strpos($errorMsg, '503') !== false ||
                    strpos($errorMsg, 'timeout') !== false ||
                    strpos($errorMsg, 'Connection') !== false);
                
                if ($isRetryable && $attempt < GPU_LLM_MAX_RETRIES - 1) {
                    $delay = GPU_LLM_RETRY_DELAYS[$attempt] ?? GPU_LLM_RETRY_DELAYS[0];
                    logParser("GPU LLM Stage1: Retrying after {$delay}s...");
                    sleep($delay);
                    continue;
                }
                
                // Если исчерпали попытки или это не retry-ошибка — используем fallback для текста
                // FALLBACK: Если GPU не смог сгенерировать текст, используем словарь информативных текстов
                if ($attempt >= GPU_LLM_MAX_RETRIES - 1) {
                    logParser("GPU LLM Stage1: All retries exhausted. Using fallback text dictionary.");
                    $fallbackText = getInformativeTextFallback($question);
                    if (!empty($fallbackText)) {
                        logParser("GPU LLM Stage1: Using fallback text for question: " . substr($question, 0, 50));
                        return [
                            'text' => $fallbackText,
                            'provider' => 'gpu+fallback',
                            'model' => $model
                        ];
                    } else {
                        logParser("GPU LLM Stage1: Fallback dictionary returned empty text for question: " . substr($question, 0, 50));
                    }
                }
                
                // Если fallback не помог, выбрасываем исключение только если это критическая ошибка
                // Для некритических ошибок (например, timeout) пробуем fallback
                $isCriticalError = (strpos($errorMsg, 'Invalid') !== false || 
                                   strpos($errorMsg, 'parse') !== false ||
                                   strpos($errorMsg, 'JSON') !== false);
                
                if (!$isCriticalError && $attempt >= GPU_LLM_MAX_RETRIES - 1) {
                    // Для некритических ошибок пробуем fallback еще раз
                    $fallbackText = getInformativeTextFallback($question);
                    if (!empty($fallbackText)) {
                        logParser("GPU LLM Stage1: Using fallback text after non-critical error for question: " . substr($question, 0, 50));
                        return [
                            'text' => $fallbackText,
                            'provider' => 'gpu+fallback',
                            'model' => $model
                        ];
                    }
                }
                
                logParser("GPU LLM Stage1: Throwing exception after all fallback attempts failed.");
                throw $e;
            }
        }
        
        // Если мы дошли сюда, значит все попытки исчерпаны, но исключение не было выброшено
        // Это не должно происходить, но на всякий случай выбрасываем последнюю ошибку
        if ($gpuError) {
            logParser("GPU LLM Stage1: Unexpected state - throwing last error");
            throw $gpuError;
        }
    }
    
    if ($useYandexGPT) {
        for ($attempt = 0; $attempt < YANDEXGPT_MAX_RETRIES; $attempt++) {
            try {
                $yandexDebugInfo = null;
                if ($debugMode) {
                    $result = callYandexGPTAPIStage1($systemPrompt, $userPrompt, $question, $model, $yandexDebugInfo);
                } else {
                    $result = callYandexGPTAPIStage1($systemPrompt, $userPrompt, $question, $model);
                }
                
                // Добавляем debug информацию в результат
                if ($debugMode && $yandexDebugInfo !== null) {
                    $result['yandex_debug'] = $yandexDebugInfo;
                }
                
                return $result;
            } catch (Exception $e) {
                $errorMsg = $e->getMessage();
                $lastError = $e; // Сохраняем последнюю ошибку
                logParser("Yandex GPT Stage1 error (attempt " . ($attempt + 1) . "): " . $errorMsg);
                
                // Если это ошибка rate limit или server error, retry
                if (strpos($errorMsg, '429') !== false || 
                    strpos($errorMsg, '500') !== false || 
                    strpos($errorMsg, '502') !== false ||
                    strpos($errorMsg, '503') !== false) {
                    
                    if ($attempt < YANDEXGPT_MAX_RETRIES - 1) {
                        $delay = YANDEXGPT_RETRY_DELAYS[$attempt] ?? YANDEXGPT_RETRY_DELAYS[0];
                        sleep($delay);
                        continue;
                    }
                }
                
                // Fallback на OpenRouter или Claude
                logParser("Yandex GPT Stage1 failed, switching to fallback");
                break;
            }
        }
    }
    
    // Пытаемся использовать OpenRouter, если модель не Claude и не Yandex GPT
    if ($useOpenRouter) {
        for ($attempt = 0; $attempt < OPENROUTER_MAX_RETRIES; $attempt++) {
            try {
                return callOpenRouterAPIStage1($systemPrompt, $userPrompt, $question, $model);
            } catch (Exception $e) {
                $errorMsg = $e->getMessage();
                $lastError = $e; // Сохраняем последнюю ошибку
                logParser("OpenRouter Stage1 error (attempt " . ($attempt + 1) . "): " . $errorMsg);
                
                // Если это ошибка rate limit или server error, retry
                if (strpos($errorMsg, '429') !== false || 
                    strpos($errorMsg, '500') !== false || 
                    strpos($errorMsg, '502') !== false ||
                    strpos($errorMsg, '503') !== false) {
                    
                    if ($attempt < OPENROUTER_MAX_RETRIES - 1) {
                        $delay = OPENROUTER_RETRY_DELAYS[$attempt] ?? OPENROUTER_RETRY_DELAYS[0];
                        sleep($delay);
                        continue;
                    }
                }
                
                // Fallback на Claude
                logParser("OpenRouter Stage1 failed, switching to Claude API");
                break;
            }
        }
    }
    
    // Fallback на Claude API
    for ($attempt = 0; $attempt < CLAUDE_MAX_RETRIES; $attempt++) {
        try {
            return callClaudeAPIStage1($systemPrompt, $userPrompt, $question);
        } catch (Exception $e) {
            $errorMsg = $e->getMessage();
            $lastError = $e; // Сохраняем последнюю ошибку
            
            // Если это ошибка rate limit или server error, retry
            if (strpos($errorMsg, '429') !== false || 
                strpos($errorMsg, '500') !== false || 
                strpos($errorMsg, '502') !== false ||
                strpos($errorMsg, '503') !== false) {
                
                if ($attempt < CLAUDE_MAX_RETRIES - 1) {
                    $delay = CLAUDE_RETRY_DELAYS[$attempt];
                    logParser("Retry Claude Stage1 attempt " . ($attempt + 1) . "/" . CLAUDE_MAX_RETRIES . " after {$delay}s delay");
                    sleep($delay);
                    continue;
                }
            }
            
            throw $e;
        }
    }
    
    // Все провайдеры исчерпали попытки
    $lastError = $lastError ? $lastError->getMessage() : 'Все провайдеры недоступны';
    throw new Exception('Max retries exceeded for Stage1. Last error: ' . $lastError);
}

/**
 * Вызов Yandex GPT API для этапа 1
 * Поддерживает формат модели: gpt://{folder_id}/{model_name}/{version}
 */
function callYandexGPTAPIStage1($systemPrompt, $userPrompt, $question, $model, &$debugInfo = null) {
    // Парсим модель формата gpt://{folder_id}/{model_name}/{version}
    $folderId = YANDEXGPT_FOLDER_ID;
    $modelUri = $model;
    
    if (strpos($model, 'gpt://') === 0) {
        // Формат: gpt://b1gqogohv4tgsmbj9gb4/yandexgpt-5-lite/latest
        $parts = explode('/', substr($model, 6)); // Убираем "gpt://"
        if (count($parts) >= 2) {
            $folderId = $parts[0]; // b1gqogohv4tgsmbj9gb4
            $modelName = $parts[1]; // yandexgpt-5-lite или yandexgpt-lite
            $version = $parts[2] ?? 'latest'; // latest
            $modelUri = "gpt://{$folderId}/{$modelName}/{$version}";
        }
    }
    
    // Используем новый API endpoint (rest-assistant)
    $apiUrl = 'https://rest-assistant.api.cloud.yandex.net/v1/responses';
    
    logParser("=== STAGE1 REQUEST TO YANDEX GPT ===");
    logParser("URL: " . $apiUrl);
    logParser("Model: " . $modelUri);
    
    // Формируем запрос в формате нового API
    $data = [
        'model' => $modelUri,
        'temperature' => 0.7,
        'instructions' => $systemPrompt,
        'input' => $userPrompt,
        'max_output_tokens' => 500
    ];
    
    // Сохраняем информацию о запросе для debug
    $requestInfo = [
        'url' => $apiUrl,
        'method' => 'POST',
        'headers' => [
            'Content-Type: application/json',
            'Authorization: Api-Key ' . (defined('YANDEXGPT_API_KEY') ? substr(YANDEXGPT_API_KEY, 0, 10) . '...' : 'N/A'),
            'x-folder-id: ' . $folderId
        ],
        'body' => $data
    ];
    
    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, 30); // Таймаут для Yandex GPT API
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: Api-Key ' . YANDEXGPT_API_KEY,
        'x-folder-id: ' . $folderId
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    // Сохраняем информацию об ответе для debug
    $responseInfo = [
        'http_code' => $httpCode,
        'response' => $response
    ];
    
    if (curl_errno($ch)) {
        $error = curl_error($ch);
        curl_close($ch);
        
        // Сохраняем информацию об ошибке
        if ($debugInfo !== null) {
            $debugInfo = [
                'request' => $requestInfo,
                'response' => $responseInfo,
                'error' => $error
            ];
        }
        
        throw new Exception('Yandex GPT API error: ' . $error);
    }
    
    curl_close($ch);
    
    if ($httpCode !== 200) {
        // Сохраняем информацию об ошибке HTTP
        if ($debugInfo !== null) {
            $debugInfo = [
                'request' => $requestInfo,
                'response' => $responseInfo,
                'error' => "HTTP $httpCode: " . substr($response, 0, 500)
            ];
        }
        
        throw new Exception("Yandex GPT API returned HTTP $httpCode: " . substr($response, 0, 500));
    }
    
    $result = json_decode($response, true);
    
    if (!$result) {
        // Сохраняем информацию об ошибке парсинга
        if ($debugInfo !== null) {
            $debugInfo = [
                'request' => $requestInfo,
                'response' => $responseInfo,
                'error' => 'Invalid JSON response: ' . substr($response, 0, 500)
            ];
        }
        
        throw new Exception('Invalid Yandex GPT API response: ' . substr($response, 0, 500));
    }
    
    // Проверяем статус ответа
    if (isset($result['status']) && $result['status'] === 'failed') {
        $errorMessage = 'Yandex GPT API returned failed status';
        if (isset($result['error'])) {
            $errorMessage .= ': ' . json_encode($result['error']);
        }
        if (isset($result['message'])) {
            $errorMessage .= ' - ' . $result['message'];
        }
        logParser("Yandex GPT API Stage1 failed status: " . json_encode($result));
        
        // Сохраняем информацию об ошибке статуса
        if ($debugInfo !== null) {
            $debugInfo = [
                'request' => $requestInfo,
                'response' => $responseInfo,
                'error' => $errorMessage,
                'result' => $result
            ];
        }
        
        throw new Exception($errorMessage);
    }
    
    // Обрабатываем ответ нового API формата
    // Ответ находится в output[0].content[0].text
    if (!isset($result['output'][0]['content'][0]['text'])) {
        logParser("Yandex GPT API Stage1 invalid response structure: " . json_encode($result));
        
        // Сохраняем информацию об ошибке структуры
        if ($debugInfo !== null) {
            $debugInfo = [
                'request' => $requestInfo,
                'response' => $responseInfo,
                'error' => 'Invalid response structure',
                'result' => $result
            ];
        }
        
        throw new Exception('Invalid Yandex GPT API response structure: ' . json_encode($result));
    }
    
    // Сохраняем успешный ответ для debug
    if ($debugInfo !== null) {
        $debugInfo = [
            'request' => $requestInfo,
            'response' => $responseInfo,
            'result' => $result
        ];
    }
    
    $text = trim($result['output'][0]['content'][0]['text']);
    
    // Очищаем текст от лишних символов
    $text = preg_replace('/[\x00-\x1F\x7F]/u', '', $text);
    $text = trim($text);
    
    // Подсчитываем токены из usage
    $tokensUsed = 0;
    if (isset($result['usage']['total_tokens'])) {
        $tokensUsed = $result['usage']['total_tokens'];
    } elseif (isset($result['usage'])) {
        $tokensUsed = ($result['usage']['input_tokens'] ?? 0) + ($result['usage']['output_tokens'] ?? 0);
    }
    
    return [
        'text' => $text,
        'provider' => 'yandexgpt',
        'tokens_used' => $tokensUsed
    ];
}

/**
 * Вызов OpenRouter API для этапа 1
 */
function callOpenRouterAPIStage1($systemPrompt, $userPrompt, $question, $model) {
    logParser("=== STAGE1 REQUEST TO OPENROUTER ===");
    logParser("URL: " . OPENROUTER_API_URL);
    logParser("Model: " . $model);
    
    $data = [
        'model' => $model,
        'max_tokens' => 500, // Короткий ответ для этапа 1
        'messages' => [
            [
                'role' => 'system',
                'content' => $systemPrompt
            ],
            [
                'role' => 'user',
                'content' => $userPrompt
            ]
        ],
        'temperature' => 0.7
    ];
    
    $ch = curl_init(OPENROUTER_API_URL);
    configureOpenRouterProxy($ch); // Настройка прокси для OpenRouter
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, 30); // Таймаут для Yandex GPT API
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: Bearer ' . OPENROUTER_API_KEY,
        'HTTP-Referer: ' . WIDGET_DOMAIN,
        'X-Title: AI Widget'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        $error = curl_error($ch);
        curl_close($ch);
        throw new Exception('OpenRouter API error: ' . $error);
    }
    
    curl_close($ch);
    
    if ($httpCode !== 200) {
        throw new Exception("OpenRouter API returned HTTP $httpCode: " . substr($response, 0, 500));
    }
    
    $result = json_decode($response, true);
    
    if (!isset($result['choices'][0]['message']['content'])) {
        throw new Exception('Invalid OpenRouter API response structure');
    }
    
    $text = trim($result['choices'][0]['message']['content']);
    
    // Очищаем текст от лишних символов
    $text = preg_replace('/[\x00-\x1F\x7F]/u', '', $text);
    $text = trim($text);
    
    return [
        'text' => $text,
        'provider' => 'openrouter',
        'tokens_used' => ($result['usage']['prompt_tokens'] ?? 0) + ($result['usage']['completion_tokens'] ?? 0)
    ];
}

/**
 * Проверка текста на артефакты
 * Возвращает массив найденных артефактов или пустой массив, если артефактов нет
 */
function checkTextArtifacts($text) {
    $artifacts = [];
    
    if (empty($text) || mb_strlen($text) < 20) {
        $artifacts[] = 'Текст слишком короткий или пустой';
    }
    
    // Проверка на ссылки на изображения
    if (preg_match('/\.(runtuit|jpg|jpeg|png|gif|webp|svg)[_\w\d]*/i', $text)) {
        $artifacts[] = 'Ссылки на изображения';
    }
    
    // Проверка на смешанные русско-английские фразы
    if (preg_match('/\b(other\s+symptoms?|other\s+conditions?|Apply\s+ice|appropriate\s+лечение|appropriate\s+treatment)\b/ui', $text)) {
        $artifacts[] = 'Смешанные русско-английские фразы';
    }
    
    // Проверка на транслитерацию медицинских терминов
    if (preg_match('/\b(endocrinolog|gastroenterolog|pulmonologist|respirologist|orthopedic|ophthalmologist|dermatologist|allergologist|immunologist|cardiologist|neurologist|rheumatologist|urologist|gynecologist|surgeon|therapist|pediatrician)[уеаио]\b/ui', $text)) {
        $artifacts[] = 'Транслитерация медицинских терминов';
    }
    
    // Проверка на транслитерацию (смешанные латинские и кириллические символы)
    if (preg_match('/\bpedиатр\b/ui', $text) || preg_match('/\bpedиатр[а-яё]*\b/ui', $text)) {
        $artifacts[] = 'Транслитерация: pedиатр';
    }
    if (preg_match('/\bdrationом\b/ui', $text)) {
        $artifacts[] = 'Транслитерация: drationом';
    }
    if (preg_match('/\blazernoe\s+omolozhenie\b/ui', $text)) {
        $artifacts[] = 'Транслитерация: lazernoe omolozhenie';
    }
    if (preg_match('/\bprocedure\b/ui', $text)) {
        $artifacts[] = 'Английское слово: procedure';
    }
    if (preg_match('/\bвызванedBy\b/ui', $text)) {
        $artifacts[] = 'Смешанный текст: вызванedBy';
    }
    if (preg_match('/\bназальный\s+зыбь\b/ui', $text)) {
        $artifacts[] = 'Некорректный перевод: назальный зыбь';
    }
    
    // Проверка на слова со смешанной транслитерацией (латинские буквы в русском тексте)
    if (preg_match('/\b[a-z]+[а-яё]+[a-z]*\b/ui', $text) || preg_match('/\b[а-яё]+[a-z]+[а-яё]*\b/ui', $text)) {
        $artifacts[] = 'Смешанная транслитерация (латиница+кириллица)';
    }
    
    // Проверка на английские предложения после русского текста
    if (preg_match('/\.\s*([A-Z][a-z]+(?:\s+[a-z]+)*\s+(?:can|should|is|are|may|might|will|would|could)\s+[^.]*\.)/u', $text)) {
        $artifacts[] = 'Английские предложения в русском тексте';
    }
    
    // Проверка на китайские символы
    if (preg_match('/[\x{4e00}-\x{9fff}]/u', $text)) {
        $artifacts[] = 'Китайские символы';
    }
    
    // Проверка на незавершенные предложения (текст обрывается)
    if (preg_match('/[^.!?]\s*$/', trim($text)) && mb_strlen($text) < 100) {
        $artifacts[] = 'Незавершенный текст';
    }
    
    // Проверка на обрывы в середине предложения (запятая или союз в конце)
    if (preg_match('/[,\s]+\.\s*$/', trim($text)) || preg_match('/\s+(или|и|а|но|что|который|когда|где|как)\s*\.\s*$/ui', trim($text))) {
        $artifacts[] = 'Обрыв текста в середине предложения';
    }
    
    // Проверка на очень короткий текст (менее 50 символов) - вероятно обрыв
    if (mb_strlen(trim($text)) < 50 && mb_strlen(trim($text)) > 20) {
        $artifacts[] = 'Текст слишком короткий (возможен обрыв)';
    }
    
    // Проверка на английские слова в русском тексте
    if (preg_match('/\b(runo|cell|procedure|procedures|treatment|therapy|diagnosis|doctor|patient|medical|health|disease|disorder|syndrome|infection|inflammation|chronic|acute|severe|mild|moderate|symptoms?|conditions?|apply|ice|appropriate|other|such|as|runny|nose|cough|sore|throat|fever|pediatru)\b/i', $text)) {
        $artifacts[] = 'Английские слова в русском тексте';
    }
    
    // Проверка на английские фразы типа "such as runny nose"
    if (preg_match('/\bsuch\s+as\s+(?:runny\s+nose|cough|sore\s+throat|fever)/ui', $text)) {
        $artifacts[] = 'Английская фраза "such as" с симптомами';
    }
    
    // Проверка на слитые слова
    if (preg_match('/\b(сэтой|сэтим|сэтими|вэтом|наэтом|заэто)\b/ui', $text)) {
        $artifacts[] = 'Слитые слова (например, "сэтой" вместо "с этой")';
    }
    
    // Проверка на незавершенные предложения с английскими словами в конце
    if (preg_match('/\s+(procedures?|treatment|therapy|diagnosis|doctor|patient|medical|health|disease|disorder|syndrome|infection|inflammation|chronic|acute|severe|mild|moderate|symptoms?|conditions?|apply|ice|appropriate|other|such|as|runny|nose|cough|sore|throat|fever|pediatru)\s*\.?\s*$/ui', $text)) {
        $artifacts[] = 'Незавершенное предложение с английским словом в конце';
    }
    
    // Проверка на повторяющиеся фрагменты
    if (preg_match('/(.{20,})\1/u', $text)) {
        $artifacts[] = 'Повторяющиеся фрагменты';
    }
    
    // Проверка на HTML-комментарии (<!-- -->)
    if (preg_match('/<!--.*?-->/s', $text)) {
        $artifacts[] = 'HTML-комментарии';
    }
    
    // Проверка на HTML теги
    if (preg_match('/<[^>]+>/', $text)) {
        $artifacts[] = 'HTML теги';
    }
    
    // Проверка на URL
    if (preg_match('/(http:\/\/|https:\/\/|www\.|booking\.)/i', $text)) {
        $artifacts[] = 'URL в тексте';
    }
    
    // Проверка на ссылки на изображения в JSON-формате (photo, foto, image_url)
    if (preg_match('/["\'](photo|foto|image_url|image|img|picture|thumbnail|preview)["\']\s*:\s*["\'][^"\']*\.(jpg|jpeg|png|gif|webp|svg)/i', $text)) {
        $artifacts[] = 'Ссылки на изображения в JSON';
    }
    
    // Проверка на дисклеймеры
    if (preg_match('/Имеются противопоказания/ui', $text)) {
        $artifacts[] = 'Дисклеймер в тексте';
    }
    
    return $artifacts;
}

/**
 * Вызов GPU LLM API для этапа 1 с повторными попытками при обнаружении артефактов
 */
function callGPUAPIStage1($systemPrompt, $userPrompt, $question, $model = null) {
    $maxAttempts = 3;
    $bestResult = null;
    $bestArtifactsCount = PHP_INT_MAX;
    $enhancedPrompt = $systemPrompt;
    
    for ($attempt = 1; $attempt <= $maxAttempts; $attempt++) {
        try {
            // При повторных попытках усиливаем промпт с предупреждением об артефактах
            if ($attempt > 1) {
                $enhancedPrompt = $systemPrompt . "\n\nВНИМАНИЕ! В предыдущей попытке были обнаружены артефакты или незавершенный текст. Обязательно проверь ответ и убедись, что:\n" .
                    "- Используешь ТОЛЬКО русский язык (кириллица), никаких английских слов\n" .
                    "- НЕ используешь транслитерацию (например, 'pedиатр' - неправильно, 'педиатр' - правильно)\n" .
                    "- НЕ используешь смешанные слова типа 'procedure', 'drationом', 'вызванedBy'\n" .
                    "- НЕ добавляй HTML-теги, ссылки, URL или технические коды\n" .
                    "- Текст ДОЛЖЕН быть завершенным - заканчивай полным предложением с точкой\n" .
                    "- НЕ обрывай текст на середине предложения, запятой, союзе или предлоге\n" .
                    "- НЕ заканчивай на фразах типа 'может быть вызвана.', 'могут быть связаны с', 'проблемы с.'\n" .
                    "- НЕ используй дефисы или тире вместо завершения (например, 'с - - -' - это ОШИБКА)\n" .
                    "- Если начинаешь перечислять причины, обязательно заверши перечисление и закончи предложение\n" .
                    "- ПРОВЕРЬ ПЕРЕД ОТВЕТОМ: последнее предложение должно быть завершено точкой и содержать полную мысль!\n" .
                    "- Пиши только чистый русский текст без артефактов";
            }
            
            $result = callGPUAPIStage1Single($enhancedPrompt, $userPrompt, $question, $model);
            $text = $result['text'] ?? '';
            
            // Проверяем на артефакты
            $artifacts = checkTextArtifacts($text);
            
            if (empty($artifacts)) {
                // Артефактов нет - возвращаем результат
                logParser("GPU LLM Stage1: Text is clean, no artifacts found (attempt {$attempt})");
                return $result;
            }
            
            // Есть артефакты - сохраняем лучший результат
            $artifactsCount = count($artifacts);
            logParser("GPU LLM Stage1: Artifacts found on attempt {$attempt}: " . implode(', ', $artifacts));
            
            if ($artifactsCount < $bestArtifactsCount) {
                $bestResult = $result;
                $bestArtifactsCount = $artifactsCount;
            }
            
            // Если это не последняя попытка, делаем повторный запрос с усиленным промптом
            if ($attempt < $maxAttempts) {
                logParser("GPU LLM Stage1: Retrying due to artifacts (attempt {$attempt}/{$maxAttempts}). Artifacts: " . implode(', ', $artifacts));
                // Небольшая задержка перед повторной попыткой
                usleep(rand(200000, 500000)); // 200-500ms
                continue;
            }
            
        } catch (Exception $e) {
            logParser("GPU LLM Stage1 error (attempt {$attempt}): " . $e->getMessage());
            
            // Если это не последняя попытка и ошибка не критичная, продолжаем
            if ($attempt < $maxAttempts && (strpos($e->getMessage(), 'timeout') === false)) {
                usleep(rand(200000, 500000));
                continue;
            }
            
            // Если есть сохраненный результат, возвращаем его
            if ($bestResult !== null) {
                logParser("GPU LLM Stage1: Returning best result despite errors");
                return $bestResult;
            }
            
            throw $e;
        }
    }
    
    // Если дошли сюда, значит все попытки имели артефакты
    if ($bestResult !== null) {
        logParser("GPU LLM Stage1: Returning best result with {$bestArtifactsCount} artifact types after {$maxAttempts} attempts");
        return $bestResult;
    }
    
    throw new Exception('GPU LLM Stage1: All attempts failed');
}

/**
 * Одиночный вызов GPU LLM API для этапа 1 (без повторных попыток)
 */
function callGPUAPIStage1Single($systemPrompt, $userPrompt, $question, $model = null) {
    // На GPU сервере доступна фиксированная модель (vLLM served model).
    // Не используем внешние/человекочитаемые названия моделей (например, google/gemini...),
    // иначе vLLM вернёт 404 "model does not exist".
    $useModel = GPU_LLM_MODEL;
    
    logParser("=== STAGE1 REQUEST TO GPU SERVER ===");
    logParser("URL: " . GPU_LLM_API_URL);
    logParser("Model: " . $useModel);
    
    $data = [
        'model' => $useModel,
        'max_tokens' => 500, // Короткий ответ для этапа 1
        'messages' => [
            [
                'role' => 'system',
                'content' => $systemPrompt
            ],
            [
                'role' => 'user',
                'content' => $userPrompt
            ]
        ],
        'temperature' => 0.7
    ];
    
    $ch = curl_init(GPU_LLM_API_URL);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, GPU_LLM_TIMEOUT);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        $error = curl_error($ch);
        curl_close($ch);
        throw new Exception('GPU LLM API error: ' . $error);
    }
    
    curl_close($ch);
    
    if ($httpCode !== 200) {
        throw new Exception("GPU LLM API returned HTTP $httpCode: " . substr($response, 0, 500));
    }
    
    $result = json_decode($response, true);
    
    if (!isset($result['choices'][0]['message']['content'])) {
        throw new Exception('Invalid GPU LLM API response structure');
    }
    
    $text = trim($result['choices'][0]['message']['content']);
    
    // Очищаем текст от лишних символов
    $text = preg_replace('/[\x00-\x1F\x7F]/u', '', $text);
    $text = trim($text);
    
    // Проверяем и удаляем китайские символы (Qwen иногда генерирует их)
    $hasChinese = preg_match('/[\x{4e00}-\x{9fff}]/u', $text);
    if ($hasChinese) {
        logParser("WARNING: GPU generated Chinese characters, cleaning...");
        // Удаляем китайские символы и оставляем только русский/латиницу
        $textBeforeClean = $text;
        $text = preg_replace('/[\x{4e00}-\x{9fff}]+/u', ' ', $text); // Заменяем китайские символы на пробел
        $text = preg_replace('/[。，、：；！？《》【】（）]/u', '', $text); // Китайская пунктуация
        $text = preg_replace('/\s+/', ' ', $text); // Убираем множественные пробелы
        $text = trim($text);
        
        // Если после удаления китайских символов текст стал слишком коротким или пустым,
        // но до очистки был длиннее, значит весь текст был на китайском - это ошибка
        if (mb_strlen($text) < 20 && mb_strlen($textBeforeClean) > 50) {
            logParser("WARNING: GPU returned mostly Chinese text, original length: " . mb_strlen($textBeforeClean) . ", after cleaning: " . mb_strlen($text));
            // Не возвращаем пустой текст - это будет обработано fallback'ом
        }
    }
    
    // Удаляем артефакты типа ".numbers"
    $text = preg_replace('/\.(numbers|undefined|null|NaN)/i', '', $text);
    
    // Удаляем технические артефакты типа "_list_end_num_sign[numbers_list_end_num]" или подобные кодовые фрагменты
    $text = preg_replace('/_list_end_num_sign\[[^\]]+\]/i', '', $text);
    $text = preg_replace('/\[numbers_list_end_num\]/i', '', $text);
    $text = preg_replace('/\[list_end[^\]]*\]/i', '', $text);
    $text = preg_replace('/\[num[^\]]*\]/i', '', $text);
    $text = preg_replace('/\[sign[^\]]*\]/i', '', $text);
    // Удаляем любые кодовые фрагменты в квадратных скобках, которые выглядят как технические артефакты
    $text = preg_replace('/\[[a-z_]+[a-z0-9_]*\]/i', '', $text);
    
    // Удаляем ссылки на изображения и файлы (артефакты GPU)
    $text = preg_replace('/\.(runtuit|jpg|jpeg|png|gif|webp|svg)[_\w]*/i', '', $text);
    $text = preg_replace('/\.(runtuit|jpg|jpeg|png|gif|webp|svg)[_\d]*/i', '', $text);
    
    // Удаляем HTML-комментарии (<!-- -->)
    $text = preg_replace('/<!--.*?-->/s', '', $text);
    // Удаляем HTML-теги и их содержимое (более агрессивно)
    // Сначала удаляем конкретные теги с заменой на пробелы
    $text = preg_replace('/<br\s*\/?>/i', ' ', $text);
    $text = preg_replace('/<\/?br\s*\/?>/i', ' ', $text);
    $text = preg_replace('/<p[^>]*>/i', ' ', $text);
    $text = preg_replace('/<\/p>/i', ' ', $text);
    $text = preg_replace('/<div[^>]*>/i', ' ', $text);
    $text = preg_replace('/<\/div>/i', ' ', $text);
    $text = preg_replace('/<span[^>]*>/i', '', $text);
    $text = preg_replace('/<\/span>/i', '', $text);
    $text = preg_replace('/<strong[^>]*>/i', '', $text);
    $text = preg_replace('/<\/strong>/i', '', $text);
    $text = preg_replace('/<b[^>]*>/i', '', $text);
    $text = preg_replace('/<\/b>/i', '', $text);
    $text = preg_replace('/<i[^>]*>/i', '', $text);
    $text = preg_replace('/<\/i>/i', '', $text);
    // Удаляем все остальные HTML-теги (включая самозакрывающиеся)
    $text = preg_replace('/<[^>]+>/', '', $text);
    // Удаляем оставшиеся HTML-сущности
    $text = preg_replace('/&[a-z]+;/i', '', $text);
    // Удаляем HTML-сущности
    $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    // Удаляем множественные пробелы после удаления тегов
    $text = preg_replace('/\s+/', ' ', $text);
    
    // Удаляем ссылки на изображения в формате <image:https://...>
    $text = preg_replace('/<image:[^>]*>/i', '', $text);
    $text = preg_replace('/<image\s+[^>]*>/i', '', $text);
    // Удаляем обычные img теги
    $text = preg_replace('/<img[^>]*>/i', '', $text);
    $text = preg_replace('/\.(jpg|jpeg|png|gif|webp|svg)[_\w\d]*/i', '', $text);
    
    // Удаляем URL (включая ссылки на изображения) - более агрессивно
    $text = preg_replace('/(https?:\/\/[^\s\)]+|www\.[^\s\)]+)/i', '', $text);
    
    // Удаляем технические артефакты (коды категорий, ссылки на страницы)
    $text = preg_replace('/category:\/\/[^\s\)]+/i', '', $text);
    // Удаляем артефакты типа "1. ** ** 2. ** **" или подобные нумерованные списки с звездочками
    $text = preg_replace('/\s+\d+\.\s*\*\s*\*\s*/ui', '', $text);
    $text = preg_replace('/\s+\*\s*\*\s*/ui', '', $text);
    // Удаляем множественные звездочки и нумерацию в конце текста
    $text = preg_replace('/\s+\d+\.\s*\*\s*\*\s*\.?\s*$/ui', '.', $text);
    $text = preg_replace('/\s+\*\s*\*\s*\.?\s*$/ui', '.', $text);
    // Удаляем URL с пробелами (исправляем "https:\/\/" -> "https://" и удаляем)
    $text = preg_replace('/https?:\s*\/\s*\/[^\s\)]+/i', '', $text);
    $text = preg_replace('/https?:\/\/[^\s\)]+/i', '', $text);
    // Удаляем booking ссылки
    $text = preg_replace('/booking\.medflex\.ru[^\s\)]*/i', '', $text);
    // Удаляем ссылки на изображения в любом формате
    $text = preg_replace('/[^\s]+\.(jpg|jpeg|png|gif|webp|svg|runtuit)[^\s\)]*/i', '', $text);
    // Удаляем ekamedcenter.ru ссылки (включая полные URL)
    $text = preg_replace('/ekamedcenter\.ru[^\s\)]*/i', '', $text);
    $text = preg_replace('/https?:\/\/[^\s\)]*ekamedcenter\.ru[^\s\)]*/i', '', $text);
    $text = preg_replace('/www\.ekamedcenter\.ru[^\s\)]*/i', '', $text);
    // Удаляем ссылки на разделы сайта ekamedcenter.ru
    $text = preg_replace('/\/uslugi\/[^\s\)]*/i', '', $text);
    // Удаляем фразы типа "в клинике ЕМЦ" или "у кардиолога в клинике ЕМЦ"
    $text = preg_replace('/\s+в\s+клинике\s+ЕМЦ[^.]*\./ui', '.', $text);
    $text = preg_replace('/\s+у\s+[а-яё]+\s+в\s+клинике\s+ЕМЦ[^.]*\./ui', '.', $text);
    // Удаляем технические строки вида "id:", "price:", "url:" и т.д.
    $text = preg_replace('/\b(id|price|url|category|service|name|description):\s*[^\s\)]+/i', '', $text);
    // Удаляем пути к изображениям вида "/upload/..."
    $text = preg_replace('/\/upload\/[^\s\)]+/i', '', $text);
    // Удаляем пути вида "/resize_cache/..."
    $text = preg_replace('/\/resize_cache\/[^\s\)]+/i', '', $text);
    // Удаляем некорректные пробелы в URL (например, "https:\/\/" с пробелами)
    $text = preg_replace('/https?\s*:\s*\/\s*\/[^\s\)]*/i', '', $text);
    
    // Удаляем английские слова-артефакты, которые иногда генерирует Qwen
    // НО: не удаляем, если текст состоит только из этих слов (чтобы не удалить весь текст)
    $textBeforeEnglishClean = $text;
    
    // Расширенный список английских слов для удаления
    $englishWords = [
        'numbness', 'tingling', 'pain', 'burning', 'aching', 'symptoms?', 'conditions?', 
        'treatment', 'therapy', 'diagnosis', 'doctor', 'patient', 'medical', 'health', 
        'disease', 'disorder', 'syndrome', 'infection', 'inflammation', 'chronic', 
        'acute', 'severe', 'mild', 'moderate', 'liquidez', 'other', 'apply', 'ice',
        'appropriate', 'endocrinolog', 'gastroenterolog', 'pulmonologist', 'respirologist',
        'orthopedic', 'ophthalmologist', 'dermatologist', 'allergologist', 'immunologist',
        'cardiologist', 'neurologist', 'rheumatologist', 'urologist', 'gynecologist',
        'surgeon', 'therapist', 'pediatrician', 'procedure', 'procedures', 'such', 'as',
        'runny', 'nose', 'cough', 'sore', 'throat', 'fever', 'pediatru'
    ];
    
    // Удаляем отдельные английские слова
    foreach ($englishWords as $word) {
        $text = preg_replace('/\b' . preg_quote($word, '/') . '\s*[уеаио]?\b/ui', '', $text);
    }
    
    // Удаляем смешанные русско-английские фразы
    $text = preg_replace('/\b(other\s+symptoms?|other\s+conditions?|Apply\s+ice|appropriate\s+лечение|appropriate\s+treatment|such\s+as|runny\s+nose|cough|sore\s+throat|fever|procedures?)\b/ui', '', $text);
    
    // Удаляем английские слова в русском тексте (более агрессивно)
    // Паттерн: английское слово после русского текста
    $text = preg_replace('/\s+(Accutane|laser|surgical|intervention|procedure|procedures|treatment|therapy|diagnosis|doctor|patient|medical|health|disease|disorder|syndrome|infection|inflammation|chronic|acute|severe|mild|moderate|symptoms?|conditions?|apply|ice|appropriate|other|such|as|runny|nose|cough|sore|throat|fever|ear|arms|arm|legs|leg|hand|hands|foot|feet)\s+/ui', ' ', $text);
    $text = preg_replace('/\s+(Accutane|laser|surgical|intervention|procedure|procedures|treatment|therapy|diagnosis|doctor|patient|medical|health|disease|disorder|syndrome|infection|inflammation|chronic|acute|severe|mild|moderate|symptoms?|conditions?|apply|ice|appropriate|other|such|as|runny|nose|cough|sore|throat|fever|ear|arms|arm|legs|leg|hand|hands|foot|feet)\s*\.?\s*$/ui', '.', $text);
    // Удаляем английские слова после запятой или точки
    $text = preg_replace('/,\s*(arms|arm|legs|leg|hand|hands|foot|feet)\./ui', '.', $text);
    $text = preg_replace('/\.\s*(arms|arm|legs|leg|hand|hands|foot|feet)\./ui', '.', $text);
    
    // Исправляем конкретные случаи английских слов в русском тексте
    $text = preg_replace('/\s+в\s+ear\s*\./ui', ' в ухо.', $text);
    $text = preg_replace('/\s+ear\s*\./ui', ' ухо.', $text);
    $text = preg_replace('/\s+давлением\s+в\s+ear\s*\./ui', ' давлением в ухе.', $text);
    
    // Удаляем английские фразы вроде "such as runny nose, cough, sore throat, fever"
    $text = preg_replace('/\bsuch\s+as\s+(?:runny\s+nose|cough|sore\s+throat|fever)[^.]*/ui', '', $text);
    
    // Исправляем слитые слова (например, "сэтой" -> "с этой")
    $text = preg_replace('/\bсэтой\b/ui', 'с этой', $text);
    $text = preg_replace('/\bсэтим\b/ui', 'с этим', $text);
    $text = preg_replace('/\bсэтими\b/ui', 'с этими', $text);
    $text = preg_replace('/\bвэтом\b/ui', 'в этом', $text);
    $text = preg_replace('/\bнаэтом\b/ui', 'на этом', $text);
    $text = preg_replace('/\bзаэто\b/ui', 'за это', $text);
    
    // Исправляем транслитерацию (смешанные латинские и кириллические символы)
    $text = preg_replace('/\bpedиатр\b/ui', 'педиатр', $text);
    $text = preg_replace('/\bpedиатр[а-яё]*\b/ui', 'педиатр', $text);
    $text = preg_replace('/\bdrationом\b/ui', '', $text);
    $text = preg_replace('/\blazernoe\s+omolozhenie\b/ui', 'лазерное омоложение', $text);
    $text = preg_replace('/\bprocedure\b/ui', 'процедура', $text);
    $text = preg_replace('/\bвызванedBy\b/ui', 'вызван', $text);
    $text = preg_replace('/\bназальный\s+зыбь\b/ui', 'носовое кровотечение', $text);
    $text = preg_replace('/\bзыбь\b/ui', 'кровотечение', $text);
    // Исправляем опечатки в медицинских терминах
    $text = preg_replace('/\bухудоврачу\b/ui', 'отоларингологу', $text);
    $text = preg_replace('/\bухудоврач\b/ui', 'отоларинголог', $text);
    // Исправляем смешанные слова типа "ortopedonеurologу" -> "ортопеду или неврологу"
    $text = preg_replace('/\bortopedonеurologу\b/ui', 'ортопеду или неврологу', $text);
    $text = preg_replace('/\bortoped[а-яё]*neurolog[а-яё]*\b/ui', 'ортопеду или неврологу', $text);
    // Удаляем слова со смешанной латиницей и кириллицей (более агрессивно)
    $text = preg_replace('/\b[a-z]+[еаиоуыэюя][a-z]*[а-яё]+\b/ui', '', $text);
    $text = preg_replace('/\b[а-яё]+[a-z]+[еаиоуыэюя][а-яё]*\b/ui', '', $text);
    // Исправляем артефакты типа "незначительный" вместо "прохладной"
    $text = preg_replace('/\bнезначительной\s+водой\b/ui', 'прохладной водой', $text);
    $text = preg_replace('/\bнезначительный\s+водой\b/ui', 'прохладной водой', $text);
    // Исправляем артефакты типа "червячками" (вероятно опечатка)
    $text = preg_replace('/\bчервячками\b/ui', 'воспалением', $text);
    $text = preg_replace('/\bчервячки\b/ui', 'воспаление', $text);
    
    // Удаляем слова со смешанной транслитерацией (латинские буквы в русском тексте)
    $text = preg_replace('/\b[a-z]+[а-яё]+[a-z]*\b/ui', '', $text);
    $text = preg_replace('/\b[а-яё]+[a-z]+[а-яё]*\b/ui', '', $text);
    // Удаляем транслитерацию типа "Radiologicheskie" (английские слова с русскими окончаниями)
    $text = preg_replace('/\bRadiologicheskie\s+i\b/ui', '', $text);
    $text = preg_replace('/\b[a-z]+(?:heskie|heskii|heskaya)\s+[а-яё]+\b/ui', '', $text);
    // Удаляем английские слова с русскими окончаниями
    $text = preg_replace('/\b[a-z]{4,}(?:heskie|heskii|heskaya|heskoe|heskogo|heskomu|heskim|heskom)\b/ui', '', $text);
    // Удаляем английские фразы типа "Radiation of to the or arm, and or in the"
    $text = preg_replace('/\b(Radiation|of|to|the|or|arm|and|in)\s+/ui', '', $text);
    $text = preg_replace('/\s+(Radiation|of|to|the|or|arm|and|in)\b/ui', '', $text);
    // Удаляем последовательности английских слов (более агрессивно)
    $text = preg_replace('/\b[a-z]{2,}\s+(?:of|to|the|or|and|in|on|at|for|with|by|from)\s+[a-z]{2,}\s*(?:of|to|the|or|and|in|on|at|for|with|by|from)\s*[a-z]*/ui', '', $text);
    
    // Удаляем смешанные русско-английские фразы (артефакты GPU)
    // Паттерны: русский текст + английские слова/фразы в середине или конце
    $text = preg_replace('/\s+(other\s+symptoms?|other\s+conditions?|Apply\s+ice|appropriate\s+лечение|appropriate\s+treatment|see\s+a\s+doctor|consult\s+a\s+physician|seek\s+medical\s+attention)\s*/ui', '', $text);
    $text = preg_replace('/\s+(endocrinolog|gastroenterolog|pulmonologist|respirologist|orthopedic|ophthalmologist|dermatologist|allergologist|immunologist|cardiologist|neurologist|rheumatologist|urologist|gynecologist|surgeon|therapist|pediatrician)\s*[уеаио]/ui', '', $text);
    
    // Удаляем английские фразы в скобках или после запятой
    $text = preg_replace('/\s*\([^)]*(?:symptoms?|treatment|therapy|diagnosis|doctor|patient|medical)[^)]*\)/ui', '', $text);
    $text = preg_replace('/\s*,\s*(?:and|or|but)\s+(?:see|consult|visit|call)\s+(?:a\s+)?(?:doctor|physician|specialist)[^.]*/ui', '', $text);
    
    // Удаляем многоязычные артефакты (немецкий, французский, испанский, японский, корейский)
    // Немецкие артефакты типа "ist der Es von Sie um zu"
    $text = preg_replace('/\s+(ist|der|die|das|Es|von|Sie|um|zu|und|oder|für|mit|auf|in|an|zu|bei|nach|vor|über|unter|durch|gegen|ohne)\s+/ui', ' ', $text);
    // Французские артефакты типа "est une la être par de et il"
    $text = preg_replace('/\s+(est|une|la|le|les|être|par|de|et|il|elle|nous|vous|ils|elles|dans|sur|avec|pour|sans|sous|entre|parmi|contre|depuis|jusqu|pendant|avant|après|malgré|selon|selon|vers|devant|derrière|hors|chez|en|à|du|des|au|aux)\s+/ui', ' ', $text);
    // Испанские артефакты типа "es una en la de la ser por y se"
    $text = preg_replace('/\s+(es|una|en|la|el|las|los|de|ser|por|y|se|un|una|con|sin|sobre|bajo|entre|desde|hasta|durante|antes|después|contra|según|hacia|mediante|frente|tras|salvo|excepto|además|también|más|menos|muy|mucho|poco|todo|todos|todas|nada|nadie|alguien|algo|algún|alguna|algunos|algunas|este|esta|estos|estas|ese|esa|esos|esas|aquel|aquella|aquellos|aquellas)\s+/ui', ' ', $text);
    // Японские и корейские иероглифы
    $text = preg_replace('/[\x{3040}-\x{309F}\x{30A0}-\x{30FF}\x{4E00}-\x{9FAF}\x{AC00}-\x{D7AF}]+/u', '', $text);
    // Удаляем артефакты типа "Hoc. .- ist der Es" (смешанные языки)
    $text = preg_replace('/\s+[A-Z][a-z]+\s*\.\s*\.\s*-\s+[a-z]+\s+[a-z]+\s+[A-Z][a-z]+\s+/ui', ' ', $text);
    // Удаляем фразы типа "est Hoc. .- ist der Es von Sie um zu La est une"
    $text = preg_replace('/\s+(est|Hoc|ist|der|Es|von|Sie|um|zu|La|une|être|par|de|et|il|es|una|en|ser|por|y|se)\s+[A-Z][a-z]+\s*\.\s*\.\s*-\s+/ui', ' ', $text);
    // Удаляем множественные точки и дефисы после языковых артефактов
    $text = preg_replace('/\s+-\s*\.\s*\.\s*-\s*\.\s*$/ui', '.', $text);
    
    // Удаляем дисклеймеры (Имеются противопоказания...) - более агрессивная очистка
    $text = preg_replace('/\s*Имеются\s+противопоказания[^.]*\./ui', '', $text);
    $text = preg_replace('/\s*Противопоказания[^.]*\./ui', '', $text);
    $text = preg_replace('/\s*Ознакомьтесь\s+с\s+инструкцией[^.]*\./ui', '', $text);
    $text = preg_replace('/\s*Проконсультируйтесь\s+у\s+специалиста[^.]*\./ui', '', $text);
    $text = preg_replace('/\s*Перед\s+применением\s+проконсультируйтесь[^.]*\./ui', '', $text);
    // Удаляем дисклеймеры в конце текста (даже без точки)
    $text = preg_replace('/\s*Имеются\s+противопоказания[^.]*$/ui', '', $text);
    $text = preg_replace('/\s*Противопоказания[^.]*$/ui', '', $text);
    $text = preg_replace('/\s*Ознакомьтесь\s+с\s+инструкцией[^.]*$/ui', '', $text);
    $text = preg_replace('/\s*Проконсультируйтесь\s+у\s+специалиста[^.]*$/ui', '', $text);
    // Удаляем полные фразы дисклеймеров
    $text = preg_replace('/\s*Имеются\s+противопоказания\.\s*Ознакомьтесь\s+с\s+инструкцией\s+или\s+проконсультируйтесь\s+у\s+специалиста\./ui', '', $text);
    $text = preg_replace('/\s*Имеются\s+противопоказания\.\s*Ознакомьтесь\s+с\s+инструкцией\s+или\s+проконсультируйтесь\s+у\s+специалиста/ui', '', $text);
    // Удаляем дисклеймеры в любом месте текста (более агрессивно)
    $text = preg_replace('/\s*Ознакомьтесь\s+с\s+инструкцией\s+или\s+проконсультируйтесь\s+у\s+специалиста[^.]*\.?/ui', '', $text);
    $text = preg_replace('/\s*Ознакомьтесь\s+с\s+инструкцией\s+или\s+проконсультируйтесь\s+у\s+специалиста[^.]*$/ui', '', $text);
    // Удаляем дисклеймеры в любом варианте написания
    $text = preg_replace('/\s*[Оо]знакомьтесь\s+с\s+инструкцией[^.]*\.?/ui', '', $text);
    $text = preg_replace('/\s*[Пп]роконсультируйтесь\s+у\s+специалиста[^.]*\.?/ui', '', $text);
    // Исправляем обрывы типа "Имеются или проконсультируйтесь"
    $text = preg_replace('/\s*Имеются\s+или\s+проконсультируйтесь[^.]*\.?/ui', '', $text);
    $text = preg_replace('/\s*Имеются\s+или\s+проконсультируйтесь\s+у\s+[а-яё]+[^.]*\.?/ui', '', $text);
    // Исправляем опечатки типа "cпециалиста" -> "специалиста" и удаляем такие обрывы
    $text = preg_replace('/\s*Имеются\s+или\s+проконсультируйтесь\s+у\s+cпециалиста[^.]*\.?/ui', '', $text);
    $text = preg_replace('/\s*Имеются\s+или\s+проконсультируйтесь\s+у\s+специалиста[^.]*\.?/ui', '', $text);
    
    // Проверка на кракозябры (отсутствие русских букв в словах)
    // Если в тексте есть слова только из латиницы без русских букв - это кракозябры
    $words = preg_split('/\s+/u', $text);
    $cleanedWords = [];
    foreach ($words as $word) {
        $cleanWord = preg_replace('/[^а-яёa-z]/ui', '', $word);
        // Если слово состоит только из латиницы и длиннее 3 символов - вероятно кракозябра
        if (mb_strlen($cleanWord) > 3 && preg_match('/^[a-z]+$/ui', $cleanWord) && !preg_match('/[а-яё]/ui', $word)) {
            // Пропускаем известные английские слова, которые могут быть уместны
            $knownEnglishWords = ['http', 'https', 'www', 'com', 'ru', 'org', 'net'];
            if (!in_array(mb_strtolower($cleanWord), $knownEnglishWords)) {
                continue; // Пропускаем кракозябру
            }
        }
        $cleanedWords[] = $word;
    }
    $text = implode(' ', $cleanedWords);
    
    $text = trim($text);
    
    // Если после удаления английских слов текст стал пустым, восстанавливаем
    if (empty($text) && !empty($textBeforeEnglishClean)) {
        logParser("WARNING: Removing English words would make text empty, keeping original");
        $text = $textBeforeEnglishClean;
    }
    
    // Заменяем английские медицинские термины на русские
    $englishTermFixes = [
        '/\bmiddle\s+ear\b/ui' => 'среднее ухо',
        '/\bouter\s+ear\b/ui' => 'наружное ухо',
        '/\binner\s+ear\b/ui' => 'внутреннее ухо',
        '/\brunny\s+nose\b/ui' => 'насморк',
        '/\bsore\s+throat\b/ui' => 'боль в горле',
        '/\bhigh\s+fever\b/ui' => 'высокая температура',
        // Исправляем транслитерацию медицинских специальностей
        '/\bdermatologistom\b/ui' => 'дерматологу',
        '/\bdermatologist\b/ui' => 'дерматолог',
        '/\bdermatologu\b/ui' => 'дерматологу',
        '/\bdermatologa\b/ui' => 'дерматолога',
        '/\bdermatolog\b/ui' => 'дерматолог',
        '/\bendocrinologу\b/ui' => 'эндокринологу',
        '/\bgastroenterologу\b/ui' => 'гастроэнтерологу',
        '/\bcardiologу\b/ui' => 'кардиологу',
        '/\bneurologу\b/ui' => 'неврологу',
        '/\bpediatrician\b/ui' => 'педиатр',
        '/\bpediatricianу\b/ui' => 'педиатру',
    ];
    foreach ($englishTermFixes as $pattern => $replacement) {
        $text = preg_replace($pattern, $replacement, $text);
    }
    
    // Исправляем транслитерацию русских слов латиницей (GPU иногда генерирует такие артефакты)
    $transliterationFixes = [
        '/\bRadiatsiya\s+boleeniya\s+v\s+nogu\b/ui' => 'иррадиация боли в ногу',
        '/\bortopedu-diskografu\b/ui' => 'ортопеду',
        '/\bpulmonologistu\b/ui' => 'пульмонологу',
        '/\bpulmonologu\b/ui' => 'пульмонологу',
        '/\bpulmonologist\s*\(/ui' => 'пульмонологу (',
        '/\brespirologist\)/ui' => 'пульмонологу)',
        // Исправляем транслитерацию медицинских терминов
        '/\bendocrinologу\b/ui' => 'эндокринологу',
        '/\bgastroenterologу\b/ui' => 'гастроэнтерологу',
        '/\bgastroenterит\b/ui' => 'гастроэнтерит',
        '/\bFotosensitivnost\b/ui' => 'фотосенсибилизация',
        '/\bAstigmatism\s+can\s+be\s+corrected/ui' => 'Астигматизм можно исправить',
    ];
    foreach ($transliterationFixes as $pattern => $replacement) {
        $text = preg_replace($pattern, $replacement, $text);
    }
    
    // Удаляем английские предложения, которые идут после русского текста (дублирование)
    // Паттерн: русский текст, затем английское предложение с заглавной буквы
    $text = preg_replace('/\.\s*([A-Z][a-z]+(?:\s+[a-z]+)*\s+(?:can|should|is|are|may|might|will|would|could)\s+[^.]*\.)/u', '.', $text);
    
    // Удаляем фразы типа "and consider seeking" в середине русского текста
    $text = preg_replace('/\s*,\s*and\s+consider\s+seeking[^.]*\./ui', '.', $text);
    $text = preg_replace('/\s*,\s*Apply\s+ice[^.]*\./ui', '.', $text);
    
    // Исправляем обрывы текста (если текст заканчивается на незавершенном слове)
    $text = trim($text);
    
    // Удаляем незавершенные предложения с английскими словами в конце
    $text = preg_replace('/\s+(procedures?|treatment|therapy|diagnosis|doctor|patient|medical|health|disease|disorder|syndrome|infection|inflammation|chronic|acute|severe|mild|moderate|symptoms?|conditions?|apply|ice|appropriate|other|such|as|runny|nose|cough|sore|throat|fever|pediatru)\s*\.?\s*$/ui', '.', $text);
    
    // Исправляем незавершенные предложения (например, "...или прием." -> "...или прием врача.")
    if (preg_match('/\s+или\s+прием\s*\.\s*$/ui', $text)) {
        $text = preg_replace('/\s+или\s+прием\s*\.\s*$/ui', ' или прием врача.', $text);
    }
    if (preg_match('/\s+консультац\s*\.\s*$/ui', $text)) {
        $text = preg_replace('/\s+консультац\s*\.\s*$/ui', ' консультацию врача.', $text);
    }
    if (preg_match('/\s+обратиться\s*\.\s*$/ui', $text)) {
        $text = preg_replace('/\s+обратиться\s*\.\s*$/ui', ' обратиться к врачу.', $text);
    }
    
    // Исправляем обрывы типа "...и ." или "...желез и ."
    // Сначала исправляем конкретные случаи типа "желез и ." -> "желез."
    $text = preg_replace('/\s+([а-яё]+)\s+и\s*\.\s*$/ui', ' $1.', $text);
    // Исправляем обрывы типа "из-за воспаления желез и" -> "из-за воспаления желез"
    $text = preg_replace('/\s+желез\s+и\s*\.?\s*$/ui', ' желез.', $text);
    // Исправляем обрывы типа "обычно развивается из-за воспаления желез и" -> "обычно развивается из-за воспаления желез"
    $text = preg_replace('/\s+желез\s+и\s*\.?\s*$/ui', ' желез.', $text);
    // Более общий паттерн: слово + "и" в конце без продолжения
    $text = preg_replace('/\s+([а-яё]+)\s+и\s*\.?\s*$/ui', ' $1.', $text);
    // Затем удаляем оставшиеся незавершенные "и ." в конце
    if (preg_match('/\s+и\s*\.\s*$/ui', $text)) {
        $text = preg_replace('/\s+и\s*\.\s*$/ui', '.', $text);
    }
    // Удаляем незавершенные фразы типа "и ." в середине или конце текста
    $text = preg_replace('/\s+и\s*\.\s*$/ui', '.', $text);
    // Удаляем незавершенные фразы типа "или ." в конце
    $text = preg_replace('/\s+или\s*\.\s*$/ui', '.', $text);
    
    // Исправляем незавершенные предложения типа "Для определения."
    // Сначала проверяем конец текста
    if (preg_match('/\s+Для\s+определения\s*\.?\s*$/ui', $text)) {
        $text = preg_replace('/\s+Для\s+определения\s*\.?\s*$/ui', ' Для определения причины обратитесь к врачу.', $text);
    }
    // Также исправляем в середине текста, если это выглядит как обрыв
    $text = preg_replace('/\s+Для\s+определения\s*\.\s*$/ui', ' Для определения причины обратитесь к врачу.', $text);
    // Исправляем незавершенные предложения типа "особенно если он грудного возраста, или если 38.5°C."
    $text = preg_replace('/\s+или\s+если\s+\d+[°\.]C\s*\.\s*$/ui', ' или если температура выше 38.5°C - необходимо срочно обратиться к врачу.', $text);
    // Исправляем обрывы с температурой в конце
    $text = preg_replace('/\s+если\s+\d+[°\.]C\s*\.\s*$/ui', ' если температура выше указанного значения - необходимо срочно обратиться к врачу.', $text);
    // Исправляем обрывы типа "симптомы , , , and ."
    $text = preg_replace('/\s+симптомы\s*,\s*,\s*,\s*and\s*\.\s*$/ui', ' симптомов.', $text);
    $text = preg_replace('/\s*,\s*,\s*,\s*and\s*\.\s*$/ui', '.', $text);
    // Исправляем пропущенные слова типа "обратиться к ."
    $text = preg_replace('/\s+обратиться\s+к\s*\.\s*$/ui', ' обратиться к врачу.', $text);
    // Удаляем технические фразы типа "пользователь: ..."
    $text = preg_replace('/\s*пользователь:\s*[^.]*\.?\s*/ui', '', $text);
    // Исправляем множественные запятые и точки
    $text = preg_replace('/\s*,\s*,\s*/u', ', ', $text);
    $text = preg_replace('/\s*\.\s*\.\s*/u', '. ', $text);
    
    // Исправляем обрывы типа "Акне — это распространенная проблема, связанная с."
    if (preg_match('/\s+связан[а-яё]*\s+с\s*\.\s*$/ui', $text)) {
        // Удаляем незавершенную фразу или заменяем на завершенную
        $text = preg_replace('/\s+связан[а-яё]*\s+с\s*\.\s*$/ui', '.', $text);
    }
    // Исправляем обрывы типа "появлениями и Для лечения"
    $text = preg_replace('/\s+и\s+Для\s+лечения\s*\.?\s*$/ui', '.', $text);
    // Исправляем обрывы типа "обратиться к или косметологу"
    $text = preg_replace('/\s+обратиться\s+к\s+или\s+([а-яё]+)\s*\.?\s*$/ui', ' обратиться к $1.', $text);
    // Исправляем обрывы типа "под руководством родинка"
    $text = preg_replace('/\s+под\s+руководством\s+родинка\s*\.?\s*$/ui', '.', $text);
    // Исправляем обрывы типа "рекомендаций B"
    $text = preg_replace('/\s+рекомендаций\s+[A-Z]\s*\.?\s*$/ui', ' рекомендаций.', $text);
    // Исправляем обрывы типа "Удалить, убрать морщины" (повторяющиеся фразы)
    $text = preg_replace('/\s*Удалить,\s+убрать\s+морщины\s*\.?\s*$/ui', ' убрать морщины.', $text);
    // Удаляем повторяющиеся фразы в конце текста
    $text = preg_replace('/\s*([^.!?]+)\s*,\s*\1\s*\.?\s*$/ui', ' $1.', $text);
    // Исправляем обрывы типа "при наличии больших ожоговых 10-15"
    if (preg_match('/\s+ожоговых\s+\d+-\d+\s*\.?\s*$/ui', $text)) {
        $text = preg_replace('/\s+ожоговых\s+\d+-\d+\s*\.?\s*$/ui', ' ожоговых поверхностей.', $text);
    }
    // Исправляем обрывы типа "В домашних условиях обеспечьте."
    if (preg_match('/\s+обеспечьте\s*\.\s*$/ui', $text)) {
        $text = preg_replace('/\s+обеспечьте\s*\.\s*$/ui', ' обеспечьте покой и обратитесь к врачу.', $text);
    }
    // Исправляем обрывы типа "методы могут включать... в зависимости от размера и типа родинки."
    if (preg_match('/\s+в\s+зависимости\s+от\s+размера\s+и\s+типа\s+родинки\s*\.\s*$/ui', $text)) {
        // Текст уже завершен, но может быть обрыв перед этим
        // Проверяем, есть ли начало предложения
        if (preg_match('/\s+методы\s+могут\s+включать\s*\.\s*$/ui', $text)) {
            $text = preg_replace('/\s+методы\s+могут\s+включать\s*\.\s*$/ui', ' методы могут включать различные способы удаления.', $text);
        }
    }
    // Исправляем обрывы с числами в конце (типа "10-15")
    $text = preg_replace('/\s+\d+-\d+\s*\.?\s*$/ui', '.', $text);
    
    // Удаляем фрагменты статей (длинные фразы, начинающиеся с "Почти каждому знакомо", "Многие люди" и т.д.)
    $articlePatterns = [
        '/Почти\s+каждому\s+знакомо[^.]*\./ui',
        '/Многие\s+люди[^.]*\./ui',
        '/В\s+современном\s+мире[^.]*\./ui',
        '/Согласно\s+статистике[^.]*\./ui',
        '/Исследования\s+показывают[^.]*\./ui',
        '/Как\s+знакома\s+для\s+каждой\s+мамы[^.]*\./ui',
        '/Для\s+каждой\s+мамы[^.]*\./ui',
        '/Каждая\s+мама[^.]*\./ui',
    ];
    foreach ($articlePatterns as $pattern) {
        // Удаляем только если это длинная фраза (более 50 символов)
        $text = preg_replace_callback($pattern, function($matches) {
            if (mb_strlen($matches[0]) > 50) {
                return '';
            }
            return $matches[0];
        }, $text);
    }
    
    // Удаляем длинные фрагменты текста, которые похожи на статьи (более 200 символов подряд)
    // Если текст содержит очень длинное предложение без точки, это может быть фрагмент статьи
    $sentences = preg_split('/([.!?]+\s*)/u', $text, -1, PREG_SPLIT_DELIM_CAPTURE);
    $cleanedSentences = [];
    foreach ($sentences as $sentence) {
        // Если предложение слишком длинное (более 200 символов), вероятно это фрагмент статьи
        if (mb_strlen(trim($sentence)) > 200 && !preg_match('/[.!?]/u', $sentence)) {
            continue; // Пропускаем длинные фрагменты
        }
        $cleanedSentences[] = $sentence;
    }
    $text = implode('', $cleanedSentences);
    
    // Удаляем фрагменты статей о питании при акне (очень длинные тексты)
    // Паттерны для удаления длинных фрагментов про питание
    $longArticlePatterns = [
        '/Несколько\s+слов\s+об\s+аллергии[^.]*(?:\.|$)/uis',
        '/Режим\s+питания[^.]*(?:\.|$)/uis',
        '/Вредные\s+продукты[^.]*(?:\.|$)/uis',
        '/Полезные\s+продукты[^.]*(?:\.|$)/uis',
        '/Вредные\s+напитки[^.]*(?:\.|$)/uis',
    ];
    foreach ($longArticlePatterns as $pattern) {
        $text = preg_replace($pattern, '', $text);
    }
    
    // Удаляем очень длинные фрагменты (более 500 символов подряд без точки)
    $parts = preg_split('/([.!?]+\s*)/u', $text, -1, PREG_SPLIT_DELIM_CAPTURE);
    $cleanedParts = [];
    foreach ($parts as $part) {
        if (mb_strlen(trim($part)) > 500 && !preg_match('/[.!?]/u', $part)) {
            continue; // Пропускаем очень длинные фрагменты
        }
        $cleanedParts[] = $part;
    }
    $text = implode('', $cleanedParts);
    
    // Удаляем двойные запятые и исправляем пропущенные слова
    // Паттерн: ", ," или ",инфекции" (пропущено слово перед запятой)
    $text = preg_replace('/,\s*,\s*/u', ', ', $text);
    $text = preg_replace('/,\s*([а-яё])/u', ', $1', $text); // Исправляем ",инфекции" -> ", инфекции"
    
    // Исправляем пропущенные слова перед запятой (например, "такие как , инфекции" -> "такие как инфекции")
    $text = preg_replace('/\b(такие\s+как|например|включая|среди\s+них)\s+,\s+([а-яё])/ui', '$1 $2', $text);
    $text = preg_replace('/\s+,\s+([а-яё])/u', ', $1', $text); // Исправляем " , инфекции" -> ", инфекции"
    
    // Исправляем обрывы с предлогами в конце
    if (preg_match('/\s+для\s*\.\s*$/ui', $text)) {
        $text = preg_replace('/\s+для\s*\.\s*$/ui', ' для консультации.', $text);
    }
    if (preg_match('/\s+к\s+косметологу\s+для\s*\.\s*$/ui', $text)) {
        $text = preg_replace('/\s+к\s+косметологу\s+для\s*\.\s*$/ui', ' к косметологу для консультации.', $text);
    }
    
    // Исправляем незавершенные предложения типа "может быть вызвана." или "могут быть связаны с - - - -."
    if (preg_match('/\s+может\s+быть\s+вызвана\s*\.\s*$/ui', $text)) {
        // Текст обрывается на "может быть вызвана." - добавляем завершение
        $text = preg_replace('/\s+может\s+быть\s+вызвана\s*\.\s*$/ui', ' может быть вызвана различными причинами. Рекомендуется обратиться к врачу для диагностики.', $text);
    }
    // Исправляем обрывы с дефисами: "с - -" или "с - - -" или "с--" и т.д.
    if (preg_match('/\s+могут\s+быть\s+связаны\s+с\s*[-\.\s\-]+\.?\s*$/ui', $text)) {
        // Текст обрывается на "могут быть связаны с - - - -." - исправляем
        $text = preg_replace('/\s+могут\s+быть\s+связаны\s+с\s*[-\.\s\-]+\.?\s*$/ui', ' могут быть связаны с различными причинами, такими как защемление седалищного нерва, грыжа межпозвонкового диска или остеохондроз. Рекомендуется обратиться к неврологу для диагностики.', $text);
    }
    // Более общий паттерн для любых обрывов с дефисами после "с" (включая "с - -" и "с--")
    if (preg_match('/\s+с\s+[-\.\s\-]{2,}\s*\.?\s*$/ui', $text)) {
        // Удаляем дефисы и добавляем завершение
        $text = preg_replace('/\s+с\s+[-\.\s\-]+\.?\s*$/ui', ' с различными причинами. Рекомендуется обратиться к врачу для диагностики.', $text);
    }
    // Исправляем двойные тире без пробелов "--"
    $text = preg_replace('/\s+--\s*\.?\s*$/ui', '.', $text);
    // Исправляем артефакты типа "- - - " " - " "" " - " "" (дефисы и кавычки)
    $text = preg_replace('/\s+[-\.\s\-"\'«»]{2,}\s*\.?\s*$/ui', '.', $text);
    // Исправляем обрывы с дефисами и кавычками в любом месте текста (более агрессивно)
    if (preg_match('/\s+[-\.\s\-"\'«»]{2,}\s*\.?\s*$/ui', $text)) {
        // Если текст заканчивается на дефисы/кавычки, удаляем их и добавляем завершение
        $text = preg_replace('/\s+[-\.\s\-"\'«»]+\.?\s*$/ui', '.', $text);
        // Если после удаления текст слишком короткий, добавляем завершение
        if (mb_strlen(trim($text)) < 50) {
            $text = rtrim($text, '.') . ' Рекомендуется обратиться к врачу для диагностики.';
        }
    }
    // Удаляем артефакты типа "таких как усталость, - - - " " - " "" " - " ""
    $text = preg_replace('/,\s*[-\.\s\-"\'«»]{2,}\s*\.?\s*$/ui', '.', $text);
    // Если после удаления артефактов текст обрывается на "таких как", завершаем его
    if (preg_match('/\s+таких\s+как\s*\.\s*$/ui', $text)) {
        $text = preg_replace('/\s+таких\s+как\s*\.\s*$/ui', ' различными причинами. Рекомендуется обратиться к врачу для диагностики.', $text);
    }
    // Исправляем обрывы типа "связаны с ." или "вызвана ."
    if (preg_match('/\s+связаны\s+с\s*\.\s*$/ui', $text)) {
        $text = preg_replace('/\s+связаны\s+с\s*\.\s*$/ui', ' связаны с различными причинами. Рекомендуется обратиться к врачу для диагностики.', $text);
    }
    if (preg_match('/\s+вызвана\s*\.\s*$/ui', $text)) {
        $text = preg_replace('/\s+вызвана\s*\.\s*$/ui', ' вызвана различными причинами. Рекомендуется обратиться к врачу для диагностики.', $text);
    }
    // Исправляем обрывы на союзах: "а также.", "или.", "и.", "но."
    if (preg_match('/\s+(а\s+также|или|и|но|а|однако|тем\s+не\s+менее)\s*\.\s*$/ui', $text)) {
        // Удаляем союз с точкой и добавляем завершение
        $text = preg_replace('/\s+(а\s+также|или|и|но|а|однако|тем\s+не\s+менее)\s*\.\s*$/ui', '.', $text);
        // Если после удаления текст слишком короткий, добавляем завершение
        if (mb_strlen(trim($text)) < 50) {
            $text = rtrim($text, '.') . ' Рекомендуется обратиться к врачу для диагностики.';
        }
    }
    // Проверка минимальной длины текста - если текст слишком короткий (менее 100 символов), добавляем завершение
    if (mb_strlen(trim($text)) < 100) {
        // Проверяем, содержит ли текст только одну причину после "таких как"
        if (preg_match('/\s+таких\s+как\s+([а-яё]+)\s*\.\s*$/ui', $text, $matches)) {
            // Текст заканчивается на "таких как [одна причина]." - добавляем больше причин и завершение
            $text = preg_replace('/\s+таких\s+как\s+[а-яё]+\s*\.\s*$/ui', ' различными причинами, такими как мышечное напряжение, остеохондроз, невралгия или травма. Рекомендуется обратиться к врачу для диагностики.', $text);
        } elseif (preg_match('/\s+(может\s+быть\s+вызвана|вызвана|связаны|связана|вызван|вызвано)\s*\.\s*$/ui', $text)) {
            // Текст обрывается на незавершенной фразе - добавляем завершение
            $text = preg_replace('/\s+(может\s+быть\s+вызвана|вызвана|связаны|связана|вызван|вызвано)\s*\.\s*$/ui', ' различными причинами, такими как мышечное напряжение, остеохондроз, невралгия или травма. Рекомендуется обратиться к врачу для диагностики.', $text);
        } elseif (!preg_match('/\s+(рекомендуется|необходимо|нужно|следует)\s+/ui', $text)) {
            // Текст слишком короткий и не содержит рекомендации - добавляем завершение
            $text = rtrim($text, '.') . ' Рекомендуется обратиться к врачу для диагностики.';
        }
    }
    // Исправляем обрывы с многоточием в конце
    if (preg_match('/\s+\.\s*\.\s*\.\s*$/ui', $text)) {
        $text = preg_replace('/\s+\.\s*\.\s*\.\s*$/ui', '.', $text);
        // Если после удаления многоточия текст слишком короткий, добавляем завершение
        if (mb_strlen(trim($text)) < 50) {
            $text = rtrim($text, '.') . ' Рекомендуется обратиться к врачу для диагностики.';
        }
    }
    
    // Если текст обрывается на середине слова или предложения, пытаемся его завершить
    if (mb_strlen($text) > 10 && !preg_match('/[.!?]\s*$/', $text)) {
        // Если последний символ - буква или запятая, добавляем точку
        if (preg_match('/[а-яё,]\s*$/ui', $text)) {
            $text = rtrim($text, ', ') . '.';
        }
    }
    
    // Убеждаемся, что текст заканчивается точкой (если он не пустой и не заканчивается на ! или ?)
    $text = trim($text);
    if (mb_strlen($text) > 10 && !preg_match('/[.!?]\s*$/', $text)) {
        $text = rtrim($text, ',;: ') . '.';
    }
    
    // Дополнительная проверка: если текст слишком короткий (менее 100 символов) и заканчивается на точку после незавершенной фразы
    if (mb_strlen($text) < 100 && preg_match('/\s+(может\s+быть\s+вызвана|вызвана|могут\s+быть|связан|вызван)[а-яё]*\s*\.\s*$/ui', $text)) {
        // Текст обрывается на незавершенной фразе - добавляем завершение
        $text = preg_replace('/\s+(может\s+быть\s+вызвана|вызвана|могут\s+быть|связан|вызван)[а-яё]*\s*\.\s*$/ui', ' различными причинами, такими как мышечное напряжение, остеохондроз или невралгия. Рекомендуется обратиться к врачу для диагностики.', $text);
    }
    // Исправляем запятую перед точкой (артефакт типа "напряжение,.")
    $text = preg_replace('/,\s*\.\s*$/ui', '.', $text);
    // Исправляем запятую перед "Рекомендуется" (артефакт типа "усталость, Рекомендуется")
    $text = preg_replace('/,\s+Рекомендуется/ui', '. Рекомендуется', $text);
    // Исправляем обрывы типа "может быть вызвана Рекомендуется" (пропущено продолжение)
    $text = preg_replace('/\s+может\s+быть\s+вызвана\s+Рекомендуется/ui', ' может быть вызвана различными причинами. Рекомендуется', $text);
    $text = preg_replace('/\s+может\s+быть\s+вызван\s+Рекомендуется/ui', ' может быть вызван различными причинами. Рекомендуется', $text);
    $text = preg_replace('/\s+может\s+быть\s+вызвано\s+Рекомендуется/ui', ' может быть вызвано различными причинами. Рекомендуется', $text);
    // Удаляем точку в начале текста (артефакт типа ". Рекомендуется")
    $text = preg_replace('/^\s*\.\s+/ui', '', $text);
    // Удаляем артефакты типа "'s" или "nummation"
    $text = preg_replace("/\s+'s\s+/ui", ' ', $text);
    $text = preg_replace("/\s+'s\s*\./ui", '.', $text);
    $text = preg_replace('/\s+nummation\s*\.?/ui', '', $text);
    $text = preg_replace('/\s+или\.nummation\s*\.?/ui', ' боли.', $text);
    // Исправляем грамматические ошибки типа "боли, или.nummation"
    $text = preg_replace('/,\s+или\.nummation/ui', ' боли', $text);
    // Если после удаления запятой текст слишком короткий, добавляем завершение
    if (mb_strlen(trim($text)) < 100 && preg_match('/\s+(таких\s+как|например|включая)\s+[а-яё]+\s*\.\s*$/ui', $text)) {
        $text = preg_replace('/\s+(таких\s+как|например|включая)\s+[а-яё]+\s*\.\s*$/ui', ' различными причинами, такими как мышечное напряжение, остеохондроз или невралгия. Рекомендуется обратиться к врачу для диагностики.', $text);
    }
    
    // Исправляем обрывы на предлогах: "проблемы с.", "лечение для.", "обратиться к." и т.д.
    if (preg_match('/\s+(проблемы|лечение|терапия|диагностика|консультация|обращение|обратиться|рекомендуется|необходимо|нужно)\s+(с|к|для|по|в|на|от|из|у|о|об|про|при|без|через|между|среди|между|перед|после|над|под|за|перед|после)\s*\.\s*$/ui', $text)) {
        // Текст обрывается на предлоге - удаляем незавершенную фразу и добавляем завершение
        $text = preg_replace('/\s+(проблемы|лечение|терапия|диагностика|консультация|обращение|обратиться|рекомендуется|необходимо|нужно)\s+(с|к|для|по|в|на|от|из|у|о|об|про|при|без|через|между|среди|между|перед|после|над|под|за|перед|после)\s*\.\s*$/ui', '.', $text);
        // Если после удаления текст слишком короткий, добавляем завершение
        if (mb_strlen(trim($text)) < 50) {
            $text = rtrim($text, '.') . ' Рекомендуется обратиться к врачу для диагностики.';
        }
    }
    // Исправляем обрывы типа "обратиться к для диагностики" (пропущено слово после "к")
    if (preg_match('/\s+обратиться\s+к\s+для\s+/ui', $text)) {
        $text = preg_replace('/\s+обратиться\s+к\s+для\s+/ui', ' обратиться к врачу для ', $text);
    }
    if (preg_match('/\s+обратиться\s+к\s*\.\s*$/ui', $text)) {
        $text = preg_replace('/\s+обратиться\s+к\s*\.\s*$/ui', ' обратиться к врачу для диагностики.', $text);
    }
    // Исправляем обрывы типа "рекомендуется обратиться к для"
    if (preg_match('/\s+рекомендуется\s+обратиться\s+к\s+для\s+/ui', $text)) {
        $text = preg_replace('/\s+рекомендуется\s+обратиться\s+к\s+для\s+/ui', ' рекомендуется обратиться к врачу для ', $text);
    }
    // Более общий паттерн: текст заканчивается на предлог с точкой
    if (preg_match('/\s+(с|к|для|по|в|на|от|из|у|о|об|про|при|без|через|между|среди|перед|после|над|под|за)\s*\.\s*$/ui', $text)) {
        // Удаляем предлог с точкой и добавляем завершение
        $text = preg_replace('/\s+(с|к|для|по|в|на|от|из|у|о|об|про|при|без|через|между|среди|перед|после|над|под|за)\s*\.\s*$/ui', '.', $text);
        // Если после удаления текст слишком короткий, добавляем завершение
        if (mb_strlen(trim($text)) < 50) {
            $text = rtrim($text, '.') . ' Рекомендуется обратиться к врачу для диагностики.';
        }
    }
    
    // Если текст заканчивается на незавершенной фразе типа "косметические procedures", удаляем её
    if (preg_match('/\s+procedures?\s*\.?\s*$/ui', $text)) {
        $text = preg_replace('/\s+procedures?\s*\.?\s*$/ui', '.', $text);
    }
    
    $text = preg_replace('/\s+/', ' ', $text); // Убираем множественные пробелы
    $text = trim($text);
    
    // Логируем длину текста для отладки
    logParser("GPU LLM Stage1: Text length after processing: " . mb_strlen($text));
    
    // Проверяем на пустой/шаблонный текст
    // Если текст содержит только ссылки на статьи или шаблонные фразы
    $templatePatterns = [
        '/^Обратитесь\s+к\s+врачу[^.]*\.?\s*$/ui',
        '/^Рекомендуется\s+обратиться[^.]*\.?\s*$/ui',
        '/^Необходимо\s+обратиться[^.]*\.?\s*$/ui',
        '/^См\.\s+статьи[^.]*\.?\s*$/ui',
        '/^Смотрите\s+статьи[^.]*\.?\s*$/ui',
    ];
    
    $isTemplate = false;
    foreach ($templatePatterns as $pattern) {
        if (preg_match($pattern, trim($text))) {
            $isTemplate = true;
            break;
        }
    }
    
    // Если текст слишком короткий или шаблонный, логируем это
    if (empty($text) || mb_strlen($text) < 20 || $isTemplate) {
        logParser("GPU LLM Stage1: WARNING - Text is empty, too short, or template-like (length: " . mb_strlen($text) . ", isTemplate: " . ($isTemplate ? 'yes' : 'no') . ")");
    }
    
    return [
        'text' => $text,
        'provider' => 'gpu',
        'model' => $useModel,
        'tokens_used' => ($result['usage']['prompt_tokens'] ?? 0) + ($result['usage']['completion_tokens'] ?? 0)
    ];
}

/**
 * Вызов Claude API для этапа 1
 */
function callClaudeAPIStage1($systemPrompt, $userPrompt, $question) {
    logParser("=== STAGE1 REQUEST TO CLAUDE ===");
    logParser("URL: " . CLAUDE_API_URL);
    logParser("Model: " . CLAUDE_MODEL);
    
    $data = [
        'model' => CLAUDE_MODEL,
        'max_tokens' => 500, // Короткий ответ для этапа 1
        'system' => $systemPrompt, // Для Claude API system должен быть строкой, а не массивом
        'messages' => [
            [
                'role' => 'user',
                'content' => $userPrompt
            ]
        ]
    ];
    
    $ch = curl_init(CLAUDE_API_URL);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, 30); // Таймаут для Yandex GPT API
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'x-api-key: ' . CLAUDE_API_KEY,
        'anthropic-version: 2023-06-01'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        $error = curl_error($ch);
        curl_close($ch);
        throw new Exception('Claude API error: ' . $error);
    }
    
    curl_close($ch);
    
    if ($httpCode !== 200) {
        throw new Exception("Claude API returned HTTP $httpCode: $response");
    }
    
    $result = json_decode($response, true);
    
    if (!isset($result['content'][0]['text'])) {
        throw new Exception('Invalid Claude API response');
    }
    
    $text = trim($result['content'][0]['text']);
    
    // Очищаем текст от лишних символов
    $text = preg_replace('/[\x00-\x1F\x7F]/u', '', $text);
    $text = trim($text);
    
    return [
        'text' => $text,
        'provider' => 'claude',
        'tokens_used' => ($result['usage']['input_tokens'] ?? 0) + ($result['usage']['output_tokens'] ?? 0)
    ];
}

/**
 * Fallback функция для получения информативных текстов, если GPU не вернул текст
 * Использует словарь информативных текстов для проблемных запросов
 */
function getInformativeTextFallback($question) {
    $questionLower = mb_strtolower($question);
    
    // Словарь информативных текстов для проблемных запросов
    $symptomTexts = [
        'шейно.*воротников' => 'Боль в шейно-воротниковой зоне может быть связана с остеохондрозом, мышечным напряжением, невралгией или травмой. Симптомы: боль в шее и плечах, скованность движений, головная боль. Для диагностики необходимы: рентген или МРТ шейного отдела позвоночника, консультация невролога или травматолога-ортопеда.',
        'защемлен.*нерв.*спин' => 'Защемление нерва в спине может вызывать острую боль, онемение, покалывание или слабость в мышцах. Причины: остеохондроз, грыжа межпозвонкового диска, протрузия, спондилез, травма. Для диагностики необходимы: МРТ позвоночника, консультация невролога или травматолога-ортопеда. Лечение: медикаментозное, физиотерапия, ЛФК, в тяжелых случаях — хирургическое.',
        'покалыван.*конечност' => 'Покалывание в конечностях может быть симптомом различных состояний: невропатии, остеохондроза, диабета, анемии, дефицита витаминов группы B, заболеваний щитовидной железы. Для диагностики необходимы: консультация невролога или терапевта, анализы крови (общий, на витамины, гормоны), ЭМГ, МРТ позвоночника при необходимости.',
        'поясниц.*отдает.*ног' => 'Боли в пояснице, которые отдают в ногу, могут быть связаны с защемлением седалищного нерва, грыжей межпозвонкового диска, остеохондрозом или стенозом позвоночного канала. Симптомы: боль в пояснице, иррадиация в ногу, онемение, слабость мышц. Для диагностики необходимы: МРТ поясничного отдела позвоночника, консультация невролога или травматолога-ортопеда.',
        'болит.*поясниц.*отдает' => 'Боли в пояснице, которые отдают в ногу, могут быть связаны с защемлением седалищного нерва, грыжей межпозвонкового диска, остеохондрозом или стенозом позвоночного канала. Симптомы: боль в пояснице, иррадиация в ногу, онемение, слабость мышц. Для диагностики необходимы: МРТ поясничного отдела позвоночника, консультация невролога или травматолога-ортопеда.',
        'орви.*симптом' => 'ОРВИ (острая респираторная вирусная инфекция) — группа вирусных заболеваний верхних дыхательных путей. Симптомы: повышение температуры, насморк, кашель, боль в горле, слабость, головная боль. Лечение: постельный режим, обильное питье, жаропонижающие при температуре выше 38,5°C, симптоматическая терапия. При осложнениях или температуре выше 39°C необходима консультация терапевта или педиатра.',
        'орви.*лечен' => 'ОРВИ (острая респираторная вирусная инфекция) — группа вирусных заболеваний верхних дыхательных путей. Симптомы: повышение температуры, насморк, кашель, боль в горле, слабость, головная боль. Лечение: постельный режим, обильное питье, жаропонижающие при температуре выше 38,5°C, симптоматическая терапия. При осложнениях или температуре выше 39°C необходима консультация терапевта или педиатра.',
        'тремор.*рук' => 'Тремор рук (дрожание) может быть физиологическим (при волнении, усталости) или патологическим (при неврологических, эндокринных заболеваниях, приеме лекарств). Причины: болезнь Паркинсона, эссенциальный тремор, тиреотоксикоз, алкогольная интоксикация, стресс. Для диагностики необходимы: консультация невролога, анализы на гормоны щитовидной железы, МРТ головного мозга.',
        'межпозвон.*грыж' => 'Межпозвоночная грыжа — выпячивание межпозвонкового диска, сдавливающее нервные корешки или спинной мозг. Симптомы: боль в спине, отдающая в ногу или руку, онемение, слабость мышц. Для диагностики необходимы: МРТ позвоночника (наиболее информативно), КТ, рентген. Лечение: консервативное (медикаменты, физиотерапия, ЛФК) или хирургическое. Консультация невролога или травматолога-ортопеда обязательна.',
        'грыж.*позвон' => 'Межпозвоночная грыжа — выпячивание межпозвонкового диска, сдавливающее нервные корешки или спинной мозг. Симптомы: боль в спине, отдающая в ногу или руку, онемение, слабость мышц. Для диагностики необходимы: МРТ позвоночника (наиболее информативно), КТ, рентген. Лечение: консервативное (медикаменты, физиотерапия, ЛФК) или хирургическое. Консультация невролога или травматолога-ортопеда обязательна.',
        'боль.*спин.*после.*сн' => 'Боль в спине после сна может быть связана с неудобной позой, остеохондрозом, мышечным напряжением, грыжей диска. Утренняя скованность характерна для воспалительных заболеваний (анкилозирующий спондилит). Для диагностики необходимы: рентген или МРТ позвоночника, консультация невролога или травматолога. Рекомендуется также проверить матрас и подушку.',
        'профилактическ.*осмотр' => 'Профилактический осмотр (диспансеризация) — комплексное обследование для раннего выявления заболеваний. Включает: общий анализ крови, биохимический анализ крови, ЭКГ, флюорографию, консультации терапевта и узких специалистов. Рекомендуется проходить ежегодно для поддержания здоровья и ранней диагностики заболеваний.',
        'общий.*анализ.*кров.*сдать' => 'Общий анализ крови (ОАК) — базовое лабораторное исследование, показывающее состояние здоровья. Включает: количество эритроцитов, лейкоцитов, тромбоцитов, гемоглобин, СОЭ. Для сдачи анализа необходимо обратиться в лабораторию или медицинский центр. Результаты обычно готовы в течение 1-2 дней. Консультация терапевта поможет интерпретировать результаты.',
        'затруднен.*дыхан' => 'Затрудненное дыхание (одышка) может быть связано с заболеваниями легких, сердца, анемией или стрессом. Причины: ХОБЛ, бронхиальная астма, пневмония, сердечная недостаточность, анемия. Для диагностики необходимы: спирометрия, рентген грудной клетки, ЭКГ, общий анализ крови, консультация пульмонолога, терапевта или кардиолога.',
        'кашель.*мокрот' => 'Кашель с мокротой может быть признаком бронхита, пневмонии, ХОБЛ или других заболеваний дыхательных путей. Цвет и характер мокроты помогают определить причину. Для диагностики необходимы: консультация пульмонолога или терапевта, рентген грудной клетки, общий анализ крови, при необходимости — анализ мокроты.',
        'растяжен.*связок' => 'Растяжение связок — это травматическое повреждение связочной ткани, при котором связки становятся раздраженными или частично разорванными. Симптомы: боль, отек, ограничение движений, возможны кровоподтеки. Лечение: покой, компрессия, холод, возвышенное положение конечности, обезболивающие препараты. При сильной боли или подозрении на разрыв связок необходима консультация травматолога.',
        'глауком.*лечен' => 'Глаукома — хроническое заболевание глаз, характеризующееся повышением внутриглазного давления и повреждением зрительного нерва. Лечение: глазные капли для снижения внутриглазного давления, лазерная терапия, хирургическое лечение. Для диагностики и лечения необходимы: консультация офтальмолога, измерение внутриглазного давления, осмотр глазного дна, периметрия.',
        'катаракт.*операц' => 'Катаракта — помутнение хрусталика глаза, приводящее к снижению зрения. Операция по удалению катаракты (факоэмульсификация) — современный и эффективный метод лечения. Процедура проводится под местной анестезией, длится 15-20 минут. После операции зрение восстанавливается в течение нескольких дней. Для консультации и операции необходима консультация офтальмолога.',
        'астигматизм' => 'Астигматизм — нарушение зрения, при котором роговица или хрусталик имеют неправильную форму, что приводит к искажению изображения. Коррекция: очки или контактные линзы с цилиндрическими линзами, лазерная коррекция зрения. Для диагностики и подбора коррекции необходимы: консультация офтальмолога, рефрактометрия, кератотопография.',
        'диагностик.*зрен' => 'Диагностика зрения включает комплексное обследование глаз для выявления нарушений зрения и заболеваний. Включает: проверку остроты зрения, рефрактометрию, измерение внутриглазного давления, осмотр глазного дна, биомикроскопию. Для диагностики необходима консультация офтальмолога.',
        'судорож.*ног.*ноч' => 'Судороги в ногах ночью могут быть связаны с дефицитом магния, кальция, калия, обезвоживанием, мышечным перенапряжением, заболеваниями сосудов или нервной системы. Для диагностики необходимы: консультация невролога или терапевта, анализы крови на электролиты, при необходимости — УЗИ сосудов ног.',
        'ночн.*потливост' => 'Ночная потливость может быть связана с инфекционными заболеваниями, гормональными нарушениями, онкологическими заболеваниями, приемом лекарств или стрессом. Для диагностики необходимы: консультация терапевта, общий анализ крови, анализы на гормоны, при необходимости — консультация эндокринолога или онколога.',
        'сухост.*во рту' => 'Сухость во рту (ксеростомия) может быть связана с обезвоживанием, приемом лекарств, заболеваниями слюнных желез, сахарным диабетом, аутоиммунными заболеваниями. Для диагностики необходимы: консультация терапевта или стоматолога, анализы крови, при необходимости — консультация эндокринолога или ревматолога.',
        'тахикард.*учащен.*сердц' => 'Тахикардия (учащенное сердцебиение) может быть физиологической (при физической нагрузке, стрессе) или патологической (при заболеваниях сердца, щитовидной железы, анемии). Для диагностики необходимы: консультация кардиолога, ЭКГ, ЭхоКГ, анализы крови на гормоны щитовидной железы, общий анализ крови.',
        'отек.*ног.*к вечер' => 'Отеки ног к вечеру могут быть связаны с заболеваниями сердца, почек, печени, венозной недостаточностью, лимфостазом. Для диагностики необходимы: консультация терапевта или кардиолога, ЭКГ, ЭхоКГ, анализы крови и мочи, УЗИ вен ног, при необходимости — консультация нефролога или гепатолога.',
        'тошнот.*по утрам' => 'Тошнота по утрам может быть связана с беременностью, заболеваниями желудочно-кишечного тракта, заболеваниями печени, почек, неврологическими нарушениями. Для диагностики необходимы: консультация терапевта или гастроэнтеролога, анализы крови, УЗИ органов брюшной полости, при необходимости — консультация невролога.',
        'аденом.*простат' => 'Аденома простаты (доброкачественная гиперплазия предстательной железы) — доброкачественное увеличение предстательной железы. Симптомы: учащенное мочеиспускание, затрудненное мочеиспускание, слабая струя мочи. Лечение: медикаментозное или хирургическое. Для диагностики и лечения необходимы: консультация уролога, УЗИ простаты, анализ ПСА.',
        'тонзиллит.*хроническ' => 'Хронический тонзиллит — хроническое воспаление небных миндалин. Симптомы: частые ангины, боль в горле, неприятный запах изо рта, увеличение лимфоузлов. Лечение: консервативное (промывание миндалин, физиотерапия) или хирургическое (удаление миндалин). Для диагностики и лечения необходима консультация оториноларинголога.',
        'промыван.*нос.*кукушк' => 'Промывание носа методом "кукушка" (промывание по Проетцу) — процедура для лечения синуситов и гайморитов. Проводится в кабинете оториноларинголога с использованием специального аппарата. Помогает очистить пазухи носа от гноя и слизи. Для проведения процедуры необходима консультация оториноларинголога.',
        'пигментн.*пятн.*кож' => 'Пигментные пятна на коже могут быть связаны с воздействием ультрафиолета, гормональными изменениями, возрастными изменениями, заболеваниями печени. Для диагностики и лечения необходимы: консультация дерматолога, при необходимости — дерматоскопия, анализы крови на гормоны.',
        'себорейн.*дерматит' => 'Себорейный дерматит — хроническое воспалительное заболевание кожи, связанное с нарушением работы сальных желез. Симптомы: покраснение, шелушение, зуд кожи на лице, волосистой части головы, груди. Лечение: противогрибковые шампуни, мази, при необходимости — системные препараты. Для диагностики и лечения необходима консультация дерматолога.',
        'ожог.*лечен' => 'Ожог — повреждение тканей, вызванное воздействием высокой температуры, химических веществ, электричества или радиации. Лечение зависит от степени ожога: при легких ожогах — охлаждение, обработка антисептиками, при тяжелых — госпитализация. Для лечения ожогов необходима консультация хирурга или комбустиолога.',
        'аллерги.*животн' => 'Аллергия на животных — аллергическая реакция на белки, содержащиеся в слюне, моче, перхоти животных. Симптомы: насморк, чихание, слезотечение, кашель, кожные высыпания. Лечение: избегание контакта с аллергеном, антигистаминные препараты, при необходимости — аллерген-специфическая иммунотерапия. Для диагностики и лечения необходима консультация аллерголога.',
        'аллерги.*лекарств' => 'Аллергия на лекарства — аллергическая реакция на лекарственные препараты. Симптомы: кожные высыпания, зуд, отек, в тяжелых случаях — анафилаксия. Лечение: отмена препарата, антигистаминные препараты, при тяжелых реакциях — госпитализация. Для диагностики и лечения необходима консультация аллерголога.',
        'бронхиальн.*астм' => 'Бронхиальная астма — хроническое воспалительное заболевание дыхательных путей, характеризующееся приступами одышки, кашля, свистящих хрипов. Приступы провоцируются аллергенами, физической нагрузкой, инфекциями. Лечение: ингаляционные бронхолитики и противовоспалительные препараты, базисная терапия. Для диагностики необходимы: спирометрия, пикфлоуметрия, аллергопробы, консультация пульмонолога или аллерголога.',
        'астм.*бронхиальн' => 'Бронхиальная астма — хроническое воспалительное заболевание дыхательных путей, характеризующееся приступами одышки, кашля, свистящих хрипов. Приступы провоцируются аллергенами, физической нагрузкой, инфекциями. Лечение: ингаляционные бронхолитики и противовоспалительные препараты, базисная терапия. Для диагностики необходимы: спирометрия, пикфлоуметрия, аллергопробы, консультация пульмонолога или аллерголога.',
        'отек.*квинке' => 'Отек Квинке (ангионевротический отек) — острое состояние, характеризующееся отеком подкожной клетчатки и слизистых оболочек. Может быть аллергическим или наследственным. Симптомы: отек лица, губ, век, гортани, возможен отек внутренних органов. Требует немедленной медицинской помощи. Для диагностики и лечения необходима консультация аллерголога или терапевта.',
        'квинке.*отек' => 'Отек Квинке (ангионевротический отек) — острое состояние, характеризующееся отеком подкожной клетчатки и слизистых оболочек. Может быть аллергическим или наследственным. Симптомы: отек лица, губ, век, гортани, возможен отек внутренних органов. Требует немедленной медицинской помощи. Для диагностики и лечения необходима консультация аллерголога или терапевта.',
        'тяжест.*ног' => 'Тяжесть в ногах может быть связана с венозной недостаточностью, заболеваниями сердца, почек, лимфостазом, мышечной усталостью. Симптомы: отеки, усталость, боль, судороги. Для диагностики необходимы: консультация флеболога или терапевта, УЗИ вен ног, ЭКГ, анализы крови и мочи, при необходимости — консультация кардиолога или нефролога.',
        'ног.*тяжест' => 'Тяжесть в ногах может быть связана с венозной недостаточностью, заболеваниями сердца, почек, лимфостазом, мышечной усталостью. Симптомы: отеки, усталость, боль, судороги. Для диагностики необходимы: консультация флеболога или терапевта, УЗИ вен ног, ЭКГ, анализы крови и мочи, при необходимости — консультация кардиолога или нефролога.',
        'боль.*пупк' => 'Боль в области пупка может быть связана с заболеваниями желудочно-кишечного тракта (гастрит, язва, кишечная непроходимость), аппендицитом, заболеваниями печени, почек, гинекологическими проблемами. Для диагностики необходимы: консультация гастроэнтеролога или терапевта, УЗИ органов брюшной полости, анализы крови, при необходимости — консультация хирурга, гинеколога или уролога.',
        'пупк.*бол' => 'Боль в области пупка может быть связана с заболеваниями желудочно-кишечного тракта (гастрит, язва, кишечная непроходимость), аппендицитом, заболеваниями печени, почек, гинекологическими проблемами. Для диагностики необходимы: консультация гастроэнтеролога или терапевта, УЗИ органов брюшной полости, анализы крови, при необходимости — консультация хирурга, гинеколога или уролога.'
    ];
    
    // Проверяем, содержит ли вопрос ключевые слова
    if (preg_match('/(симптом|признак|проявлен|как.*проявляется|что.*бывает|что.*это|что.*такое|характерн|профилактик|предотвращ|мрт|кт|узи|рентген|диагностик|увеличен|боль|болит|приступ|судорог|спазм|шейно.*воротников|воротников.*зон|орви|тремор|межпозвон.*грыж|грыж.*позвон|лечен|причин|профилактическ|осмотр|анализ.*кров|затруднен.*дыхан|кашель.*мокрот|бронхиальн.*астм|астм|отек.*квинке|тяжест.*ног|боль.*пупк)/ui', $question)) {
        // Ищем заболевание в вопросе
        foreach ($symptomTexts as $pattern => $text) {
            // Проверяем, является ли паттерн regex
            $isRegexPattern = (strpos($pattern, '.*') !== false || 
                              strpos($pattern, '.+') !== false ||
                              strpos($pattern, '|') !== false ||
                              preg_match('/[.*+?^${}()|[\]\\\]/', $pattern));
            
            if ($isRegexPattern) {
                // Это regex паттерн - используем preg_match
                try {
                    if (preg_match('/' . $pattern . '/ui', $questionLower)) {
                        return $text;
                    }
                } catch (Exception $e) {
                    // Если паттерн некорректный, пробуем как обычную строку
                    if (strpos($questionLower, $pattern) !== false) {
                        return $text;
                    }
                }
            } else {
                // Это обычная строка - используем strpos
                if (strpos($questionLower, $pattern) !== false) {
                    return $text;
                }
            }
        }
    }
    
    return '';
}

/**
 * Генерирует базовый fallback текст, если GPU и словарь не вернули текст
 */
function generateBasicFallbackText($question) {
    $questionLower = mb_strtolower($question);
    
    // Определяем тип запроса и генерируем базовый ответ
    if (preg_match('/(лечен|терапи|препарат|лекарств)/ui', $question)) {
        return 'Для получения информации о лечении рекомендуется обратиться к специалисту. Консультация врача поможет определить правильный подход к лечению с учетом индивидуальных особенностей.';
    } elseif (preg_match('/(симптом|признак|проявлен|как.*проявляется|что.*бывает)/ui', $question)) {
        return 'Симптомы могут варьироваться в зависимости от конкретного заболевания. Для точной диагностики рекомендуется обратиться к специалисту и пройти необходимое обследование.';
    } elseif (preg_match('/(диагностик|обследован|анализ|мрт|кт|узи|рентген)/ui', $question)) {
        return 'Для диагностики рекомендуется обратиться к специалисту, который назначит необходимые обследования. Выбор методов диагностики зависит от конкретной ситуации и симптомов.';
    } elseif (preg_match('/(причин|почему|отчего|из-за|из-за чего)/ui', $question)) {
        return 'Причины могут быть различными и зависят от конкретного заболевания или состояния. Для выяснения причин рекомендуется обратиться к специалисту и пройти диагностику.';
    } else {
        // Общий ответ
        return 'Для получения подробной информации рекомендуется обратиться к специалисту. Консультация врача поможет ответить на ваши вопросы и определить дальнейшие действия.';
    }
}

