<?php
require_once __DIR__ . '/../config.php';

header('Content-Type: application/json; charset=utf-8');

// Проверка метода
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Проверка авторизации через сессию (для админки)
session_start();
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['model']) || !isset($input['prompt'])) {
        throw new Exception('Missing required parameters: model and prompt');
    }

    $model = trim($input['model']);
    $prompt = trim($input['prompt']);
    $maxTokens = isset($input['max_tokens']) ? (int)$input['max_tokens'] : 2000;
    $temperature = isset($input['temperature']) ? (float)$input['temperature'] : 0.7;

    if (empty($model) || empty($prompt)) {
        throw new Exception('Model and prompt cannot be empty');
    }

    // Отправляем запрос к OpenRouter
    $data = [
        'model' => $model,
        'messages' => [
            [
                'role' => 'user',
                'content' => $prompt
            ]
        ],
        'max_tokens' => $maxTokens,
        'temperature' => $temperature
    ];

    $ch = curl_init(OPENROUTER_API_URL);
    configureOpenRouterProxy($ch); // Настройка прокси для OpenRouter
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: Bearer ' . OPENROUTER_API_KEY,
        'HTTP-Referer: ' . WIDGET_DOMAIN,
        'X-Title: AI Widget'
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError) {
        throw new Exception('CURL error: ' . $curlError);
    }

    if ($httpCode !== 200) {
        $errorData = json_decode($response, true);
        $errorMsg = $errorData['error']['message'] ?? $errorData['error'] ?? "HTTP $httpCode: $response";
        throw new Exception($errorMsg);
    }

    $result = json_decode($response, true);

    if (!isset($result['choices'][0]['message']['content'])) {
        throw new Exception('Invalid API response: missing content');
    }

    $answer = trim($result['choices'][0]['message']['content']);
    $tokens = [
        'input' => $result['usage']['prompt_tokens'] ?? 0,
        'output' => $result['usage']['completion_tokens'] ?? 0,
        'total' => $result['usage']['total_tokens'] ?? 0
    ];

    // Извлекаем стоимость из ответа OpenRouter (если есть)
    $cost = null;
    if (isset($result['usage']['prompt_tokens']) && isset($result['usage']['completion_tokens'])) {
        // OpenRouter может возвращать стоимость в разных форматах
        // Проверяем наличие поля cost или вычисляем по токенам
        if (isset($result['usage']['cost'])) {
            $cost = floatval($result['usage']['cost']);
        } elseif (isset($result['cost'])) {
            $cost = floatval($result['cost']);
        }
    }

    $response = [
        'success' => true,
        'response' => $answer,
        'tokens' => $tokens,
        'model' => $model
    ];

    if ($cost !== null) {
        $response['cost'] = $cost;
    }

    echo json_encode($response, JSON_UNESCAPED_UNICODE);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

