<?php
/**
 * Словарь медицинских терминов и аббревиатур
 * Используется для валидации и исправления медицинских текстов
 */

/**
 * Корректные медицинские аббревиатуры
 */
function getCorrectAbbreviations() {
    return [
        'ФГДС' => [
            'full' => 'Фиброгастродуоденоскопия',
            'description' => 'эндоскопическое исследование пищевода, желудка и двенадцатиперстной кишки',
            'wrong' => ['фистуло-дуодено-сфинктерная гистероскопия', 'фистуло дуодено сфинктерная']
        ],
        'ХОБЛ' => [
            'full' => 'Хроническая обструктивная болезнь легких',
            'description' => 'хроническое воспалительное заболевание дыхательных путей',
            'wrong' => ['ЧОБЛ', 'COPD', 'chronic obstructive pulmonary disease']
        ],
        'ЭКГ' => [
            'full' => 'Электрокардиограмма',
            'description' => 'метод исследования электрической активности сердца',
            'wrong' => ['ECG', 'EKG']
        ],
        'МРТ' => [
            'full' => 'Магнитно-резонансная томография',
            'description' => 'метод получения изображений внутренних органов',
            'wrong' => ['MRI']
        ],
        'КТ' => [
            'full' => 'Компьютерная томография',
            'description' => 'рентгенологический метод исследования',
            'wrong' => ['CT']
        ],
        'УЗИ' => [
            'full' => 'Ультразвуковое исследование',
            'description' => 'метод исследования с помощью ультразвуковых волн',
            'wrong' => ['ultrasound', 'US']
        ],
        'ОАК' => [
            'full' => 'Общий анализ крови',
            'description' => 'базовое лабораторное исследование',
            'wrong' => ['CBC', 'complete blood count']
        ],
        'ОРВИ' => [
            'full' => 'Острая респираторная вирусная инфекция',
            'description' => 'группа вирусных заболеваний верхних дыхательных путей',
            'wrong' => ['ARI', 'acute respiratory infection']
        ]
    ];
}

/**
 * Словарь специалистов (на русском языке)
 */
function getMedicalSpecialists() {
    return [
        'терапевт' => [
            'english' => ['therapist', 'general practitioner', 'GP'],
            'description' => 'врач общей практики'
        ],
        'педиатр' => [
            'english' => ['pediatrician', 'pediatru', 'pedиатр'],
            'description' => 'детский врач'
        ],
        'кардиолог' => [
            'english' => ['cardiologist', 'cardiologistу', 'cardio'],
            'description' => 'врач по заболеваниям сердца'
        ],
        'невролог' => [
            'english' => ['neurologist', 'neurologistу', 'neuro'],
            'description' => 'врач по заболеваниям нервной системы'
        ],
        'офтальмолог' => [
            'english' => ['ophthalmologist', 'ophthalmologistу', 'eye doctor'],
            'description' => 'глазной врач'
        ],
        'дерматолог' => [
            'english' => ['dermatologist', 'dermatologistу', 'dermato'],
            'description' => 'врач по заболеваниям кожи'
        ],
        'эндокринолог' => [
            'english' => ['endocrinologist', 'endocrinologistу', 'endocrino'],
            'description' => 'врач по заболеваниям эндокринной системы'
        ],
        'гастроэнтеролог' => [
            'english' => ['gastroenterologist', 'gastroenterologistу', 'gastro'],
            'description' => 'врач по заболеваниям ЖКТ'
        ],
        'пульмонолог' => [
            'english' => ['pulmonologist', 'respirologist', 'pulmo'],
            'description' => 'врач по заболеваниям легких'
        ],
        'уролог' => [
            'english' => ['urologist', 'urologistу', 'uro'],
            'description' => 'врач по заболеваниям мочеполовой системы'
        ],
        'гинеколог' => [
            'english' => ['gynecologist', 'gynecologistу', 'gyno'],
            'description' => 'врач по женским заболеваниям'
        ],
        'хирург' => [
            'english' => ['surgeon', 'surgeonу', 'surgery'],
            'description' => 'врач-хирург'
        ],
        'травматолог' => [
            'english' => ['traumatologist', 'orthopedic', 'orthopedist'],
            'description' => 'врач по травмам и заболеваниям опорно-двигательного аппарата'
        ],
        'ортопед' => [
            'english' => ['orthopedist', 'orthopedic', 'ortho'],
            'description' => 'врач по заболеваниям опорно-двигательного аппарата'
        ],
        'аллерголог' => [
            'english' => ['allergologist', 'allergist', 'allergy'],
            'description' => 'врач по аллергическим заболеваниям'
        ],
        'иммунолог' => [
            'english' => ['immunologist', 'immunologistу', 'immuno'],
            'description' => 'врач по заболеваниям иммунной системы'
        ],
        'ревматолог' => [
            'english' => ['rheumatologist', 'rheumatologistу', 'rheuma'],
            'description' => 'врач по ревматическим заболеваниям'
        ],
        'оториноларинголог' => [
            'english' => ['otorhinolaryngologist', 'ENT', 'ear nose throat'],
            'alias' => 'ЛОР',
            'description' => 'врач по заболеваниям уха, горла, носа'
        ],
        'ЛОР' => [
            'english' => ['ENT', 'ear nose throat'],
            'full' => 'оториноларинголог',
            'description' => 'врач по заболеваниям уха, горла, носа'
        ]
    ];
}

/**
 * Словарь распространенных медицинских ошибок
 */
function getCommonMedicalErrors() {
    return [
        // Неправильные специалисты для симптомов
        'кровотечение из носа' => [
            'wrong_specialist' => 'офтальмолог',
            'correct_specialist' => 'оториноларинголог (ЛОР) или терапевт',
            'reason' => 'Офтальмолог лечит глаза, а не нос'
        ],
        'носовое кровотечение' => [
            'wrong_specialist' => 'офтальмолог',
            'correct_specialist' => 'оториноларинголог (ЛОР) или терапевт',
            'reason' => 'Офтальмолог лечит глаза, а не нос'
        ],
        
        // Неправильные определения заболеваний
        'пиелонефрит' => [
            'wrong' => 'воспаление мочевыводящих путей и почек',
            'correct' => 'воспаление почечных лоханок и паренхимы почек',
            'reason' => 'Пиелонефрит - это воспаление именно почек, а не всех мочевыводящих путей'
        ],
        'колит' => [
            'wrong' => 'обезвоживание как причина',
            'correct' => 'воспаление толстой кишки (инфекция, аутоиммунные процессы)',
            'reason' => 'Обезвоживание - следствие колита, а не причина'
        ],
        
        // Неправильная терминология
        'дальнозоркость' => [
            'wrong' => 'путаница с близорукостью',
            'correct' => 'нарушение рефракции, при котором фокус изображения находится за сетчаткой',
            'reason' => 'Дальнозоркость и близорукость - разные состояния'
        ]
    ];
}

/**
 * Проверка текста на медицинские ошибки
 * Возвращает массив найденных ошибок
 */
function checkMedicalErrors($text, $question = '') {
    $errors = [];
    $textLower = mb_strtolower($text);
    $questionLower = mb_strtolower($question);
    
    // Проверяем аббревиатуры
    $abbreviations = getCorrectAbbreviations();
    foreach ($abbreviations as $abbr => $data) {
        if (isset($data['wrong'])) {
            foreach ($data['wrong'] as $wrongVersion) {
                if (stripos($text, $wrongVersion) !== false) {
                    $errors[] = [
                        'type' => 'wrong_abbreviation',
                        'found' => $wrongVersion,
                        'correct' => $abbr . ' (' . $data['full'] . ')',
                        'description' => 'Неверное написание или определение аббревиатуры'
                    ];
                }
            }
        }
    }
    
    // Проверяем специалистов на английском
    $specialists = getMedicalSpecialists();
    foreach ($specialists as $russian => $data) {
        if (isset($data['english'])) {
            foreach ($data['english'] as $englishVersion) {
                if (stripos($text, $englishVersion) !== false) {
                    $errors[] = [
                        'type' => 'english_specialist',
                        'found' => $englishVersion,
                        'correct' => $russian,
                        'description' => 'Английское название специалиста вместо русского'
                    ];
                }
            }
        }
    }
    
    // Проверяем распространенные ошибки
    $commonErrors = getCommonMedicalErrors();
    foreach ($commonErrors as $symptom => $errorData) {
        if (stripos($questionLower, $symptom) !== false || stripos($textLower, $symptom) !== false) {
            if (isset($errorData['wrong_specialist'])) {
                if (stripos($textLower, mb_strtolower($errorData['wrong_specialist'])) !== false) {
                    $errors[] = [
                        'type' => 'wrong_specialist',
                        'found' => $errorData['wrong_specialist'],
                        'correct' => $errorData['correct_specialist'],
                        'reason' => $errorData['reason'],
                        'description' => 'Неправильный специалист для данного симптома'
                    ];
                }
            }
            
            if (isset($errorData['wrong'])) {
                if (stripos($textLower, mb_strtolower($errorData['wrong'])) !== false) {
                    $errors[] = [
                        'type' => 'wrong_definition',
                        'found' => $errorData['wrong'],
                        'correct' => $errorData['correct'],
                        'reason' => $errorData['reason'],
                        'description' => 'Неверное медицинское определение'
                    ];
                }
            }
        }
    }
    
    return $errors;
}

/**
 * Исправление английских названий специалистов на русские
 */
function fixEnglishSpecialists($text) {
    $specialists = getMedicalSpecialists();
    
    foreach ($specialists as $russian => $data) {
        if (isset($data['english'])) {
            foreach ($data['english'] as $english) {
                // Регистронезависимая замена с сохранением падежей
                $text = preg_replace_callback(
                    '/\b' . preg_quote($english, '/') . '[а-яёуеаио]?\b/ui',
                    function($matches) use ($russian) {
                        return $russian;
                    },
                    $text
                );
            }
        }
    }
    
    return $text;
}

/**
 * Валидация корректности рекомендации специалиста для симптома
 */
function validateSpecialistRecommendation($question, $text) {
    $questionLower = mb_strtolower($question);
    $textLower = mb_strtolower($text);
    
    // Правила соответствия симптомов и специалистов
    $rules = [
        'кровотечение.*нос|носовое.*кровотечение' => [
            'correct' => ['лор', 'оториноларинголог', 'терапевт', 'скорая'],
            'wrong' => ['офтальмолог', 'окулист'],
            'reason' => 'При носовом кровотечении нужен ЛОР, а не офтальмолог'
        ],
        'глаз|зрение|катаракта|глаукома' => [
            'correct' => ['офтальмолог', 'окулист'],
            'wrong' => ['лор', 'терапевт'],
            'reason' => 'Для глазных проблем нужен офтальмолог'
        ],
        'боль.*грудь|груд.*боль|сердце' => [
            'correct' => ['кардиолог', 'терапевт', 'скорая'],
            'wrong' => ['невролог', 'ортопед'],
            'reason' => 'При болях в сердце нужен кардиолог или вызов скорой'
        ],
        'поясниц.*ног|защемлен.*нерв' => [
            'correct' => ['невролог', 'травматолог', 'ортопед'],
            'wrong' => ['терапевт'],
            'reason' => 'При защемлении нерва нужен невролог или ортопед'
        ]
    ];
    
    foreach ($rules as $symptomPattern => $rule) {
        if (preg_match('/' . $symptomPattern . '/ui', $questionLower)) {
            // Проверяем, нет ли неправильного специалиста в тексте
            if (isset($rule['wrong'])) {
                foreach ($rule['wrong'] as $wrongSpecialist) {
                    if (preg_match('/\b' . $wrongSpecialist . '\b/ui', $textLower)) {
                        return [
                            'valid' => false,
                            'error' => $rule['reason'],
                            'found' => $wrongSpecialist,
                            'correct' => implode(' или ', $rule['correct'])
                        ];
                    }
                }
            }
        }
    }
    
    return ['valid' => true];
}
