<?php
/**
 * Функции для работы с Embedding векторами
 * Используется для семантического поиска записей
 */

require_once __DIR__ . '/../config.php';

/**
 * Создать embedding для текста через GPU Embedding сервер
 * 
 * @param string $text Текст для создания embedding
 * @return array Массив из 1024 чисел (вектор) или null при ошибке
 */
function createEmbedding($text) {
    if (empty($text)) {
        return null;
    }
    
    $data = [
        'inputs' => $text
    ];
    
    $maxRetries = 3;
    $retryDelays = [1, 2, 4];
    
    for ($attempt = 0; $attempt < $maxRetries; $attempt++) {
        try {
            logParser("=== EMBEDDING REQUEST TO GPU SERVER ===");
            logParser("URL: " . EMBEDDING_API_URL);
            logParser("Model: " . EMBEDDING_MODEL);
            
            $ch = curl_init(EMBEDDING_API_URL);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json'
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            
            if (curl_errno($ch)) {
                curl_close($ch);
                throw new Exception('CURL error: ' . curl_error($ch));
            }
            
            curl_close($ch);
            
            if ($httpCode !== 200) {
                throw new Exception("GPU Embedding API returned HTTP $httpCode: " . substr($response, 0, 200));
            }
            
            $result = json_decode($response, true);
            
            if (!isset($result['embeddings'][0])) {
                throw new Exception('Invalid embedding response: no embedding data');
            }
            
            // GPU сервис не возвращает стоимость (это бесплатно!)
            logParser("Embedding created via GPU (cost: $0.00)");
            
            return $result['embeddings'][0];
            
        } catch (Exception $e) {
            if ($attempt < $maxRetries - 1) {
                $delay = $retryDelays[$attempt] ?? 1;
                sleep($delay);
                continue;
            }
            
            error_log("Failed to create embedding after $maxRetries attempts: " . $e->getMessage());
            return null;
        }
    }
    
    return null;
}

/**
 * Вызов кастомного API (совместимого с OpenAI форматом)
 * 
 * @param string $api_url URL кастомного API
 * @param string $api_key API ключ (опционально)
 * @param array $request_data Данные запроса (model, messages, max_tokens, temperature и т.д.)
 * @return array Ответ API
 * @throws Exception При ошибке запроса
 */
function callCustomAPI($api_url, $api_key, $request_data) {
    $headers = [
        'Content-Type: application/json'
    ];

    if (!empty($api_key)) {
        if (strpos($api_key, 'Bearer ') === 0) {
            $headers[] = 'Authorization: ' . $api_key;
        } else {
            $headers[] = 'Authorization: Bearer ' . $api_key;
        }
    }

    $ch = curl_init($api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($request_data));
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    
    // Для HTTP (не HTTPS) отключаем проверку SSL
    if (strpos($api_url, 'http://') === 0) {
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    } else {
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
    }

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError) {
        throw new Exception('cURL error: ' . $curlError);
    }

    if ($httpCode !== 200) {
        throw new Exception("API returned HTTP $httpCode: " . substr($response, 0, 500));
    }

    $responseData = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid JSON response: ' . json_last_error_msg());
    }

    return $responseData;
}

/**
 * Собрать текст из полей записи для создания embedding
 * Использует только поля с use_in_prompt = 1
 * 
 * @param PDO $db Подключение к БД
 * @param int $itemId ID записи
 * @return string Текст для создания embedding
 */
function buildItemTextForEmbedding($db, $itemId) {
    // Получаем все поля с use_in_prompt = 1 для этой записи
    $stmt = $db->prepare("
        SELECT pf.field_name, pf.field_value
        FROM parsed_items pi
        JOIN widget_sections ws ON ws.widget_id = pi.widget_id AND ws.section_name = pi.section_name
        JOIN section_fields sf ON sf.section_id = ws.id AND sf.field_name = pf.field_name
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.id = ? AND sf.use_in_prompt = 1
        
        UNION
        
        SELECT scf.child_field_name as field_name, pf.field_value
        FROM parsed_items pi
        JOIN widget_sections ws ON ws.widget_id = pi.widget_id AND ws.section_name = pi.section_name
        JOIN section_fields sf ON sf.section_id = ws.id
        JOIN section_child_fields scf ON sf.id = scf.field_id
        JOIN parsed_fields pf ON pi.id = pf.item_id AND pf.field_name = scf.child_field_name
        WHERE pi.id = ? AND scf.use_in_prompt = 1
        
        ORDER BY field_name
    ");
    $stmt->execute([$itemId, $itemId]);
    
    $textParts = [];
    while ($row = $stmt->fetch()) {
        $value = trim($row['field_value']);
        if (!empty($value)) {
            $textParts[] = $value;
        }
    }
    
    return implode(' ', $textParts);
}

/**
 * Сохранить embedding для записи в БД
 * 
 * @param PDO $db Подключение к БД
 * @param int $itemId ID записи
 * @param array $embedding Массив из 1024 чисел
 * @param string $textHash MD5 хеш текста
 * @return bool Успех операции
 */
function saveItemEmbedding($db, $itemId, $embedding, $textHash) {
    if (empty($embedding) || !is_array($embedding)) {
        return false;
    }
    
    try {
        // Проверяем, существует ли уже embedding для этой записи
        $stmt = $db->prepare("SELECT id FROM item_embeddings WHERE item_id = ?");
        $stmt->execute([$itemId]);
        $exists = $stmt->fetch();
        
        if ($exists) {
            // Обновляем существующий
            $stmt = $db->prepare("
                UPDATE item_embeddings 
                SET embedding_vector = ?, text_hash = ?, updated_at = NOW()
                WHERE item_id = ?
            ");
            $stmt->execute([json_encode($embedding, JSON_UNESCAPED_UNICODE), $textHash, $itemId]);
        } else {
            // Создаем новый
            $stmt = $db->prepare("
                INSERT INTO item_embeddings (item_id, embedding_vector, text_hash)
                VALUES (?, ?, ?)
            ");
            $stmt->execute([$itemId, json_encode($embedding, JSON_UNESCAPED_UNICODE), $textHash]);
        }
        
        return true;
    } catch (Exception $e) {
        error_log("Error saving embedding for item $itemId: " . $e->getMessage());
        return false;
    }
}

/**
 * Получить embedding для записи из БД
 * 
 * @param PDO $db Подключение к БД
 * @param int $itemId ID записи
 * @return array|null Массив из 1024 чисел или null
 */
function getItemEmbedding($db, $itemId) {
    try {
        $stmt = $db->prepare("SELECT embedding_vector FROM item_embeddings WHERE item_id = ?");
        $stmt->execute([$itemId]);
        $row = $stmt->fetch();
        
        if ($row && !empty($row['embedding_vector'])) {
            $embedding = json_decode($row['embedding_vector'], true);
            if (is_array($embedding)) {
                return $embedding;
            }
        }
        
        return null;
    } catch (Exception $e) {
        error_log("Error getting embedding for item $itemId: " . $e->getMessage());
        return null;
    }
}

/**
 * Создать embedding для вопроса пользователя
 * Можно добавить кеширование в будущем
 * 
 * @param string $question Вопрос пользователя
 * @return array|null Массив из 1024 чисел или null
 */
function createQuestionEmbedding($question) {
    if (empty($question)) {
        return null;
    }
    
    // TODO: Добавить кеширование для одинаковых вопросов
    return createEmbedding($question);
}

/**
 * Вычислить косинусное расстояние между двумя векторами
 * 
 * @param array $vec1 Первый вектор (массив чисел)
 * @param array $vec2 Второй вектор (массив чисел)
 * @return float Значение от -1 до 1 (1 = идентичны, 0 = ортогональны, -1 = противоположны)
 */
function cosineSimilarity($vec1, $vec2) {
    if (empty($vec1) || empty($vec2) || count($vec1) !== count($vec2)) {
        return 0.0;
    }
    
    $dotProduct = 0.0;
    $magnitude1 = 0.0;
    $magnitude2 = 0.0;
    
    for ($i = 0; $i < count($vec1); $i++) {
        $dotProduct += $vec1[$i] * $vec2[$i];
        $magnitude1 += $vec1[$i] * $vec1[$i];
        $magnitude2 += $vec2[$i] * $vec2[$i];
    }
    
    $magnitude1 = sqrt($magnitude1);
    $magnitude2 = sqrt($magnitude2);
    
    if ($magnitude1 == 0 || $magnitude2 == 0) {
        return 0.0;
    }
    
    return $dotProduct / ($magnitude1 * $magnitude2);
}

/**
 * Найти похожие записи по embedding среди кандидатов
 * 
 * @param PDO $db Подключение к БД
 * @param int $widgetId ID виджета
 * @param array $questionEmbedding Embedding вопроса пользователя (массив из 1024 чисел)
 * @param array $candidateIds Массив ID кандидатов по разделам: ['section_name' => [1, 2, 3], ...]
 * @param int $limit Максимальное количество результатов на раздел
 * @return array Результаты по разделам: ['section_name' => [['id' => 123, 'similarity' => 0.85], ...]]
 */
function findSimilarItemsByEmbedding($db, $widgetId, $questionEmbedding, $candidateIds, $limit = 20) {
    if (empty($questionEmbedding) || !is_array($questionEmbedding)) {
        return [];
    }
    
    $results = [];
    
    foreach ($candidateIds as $sectionName => $ids) {
        if (empty($ids) || !is_array($ids)) {
            continue;
        }
        
        // Получаем embedding для всех кандидатов этого раздела
        $placeholders = str_repeat('?,', count($ids) - 1) . '?';
        $stmt = $db->prepare("
            SELECT ie.item_id, ie.embedding_vector
            FROM item_embeddings ie
            JOIN parsed_items pi ON ie.item_id = pi.id
            WHERE pi.widget_id = ? 
              AND pi.section_name = ?
              AND pi.is_duplicate = 0
              AND ie.item_id IN ($placeholders)
        ");
        $params = array_merge([$widgetId, $sectionName], $ids);
        $stmt->execute($params);
        
        $similarities = [];
        while ($row = $stmt->fetch()) {
            $itemId = (int)$row['item_id'];
            $embedding = json_decode($row['embedding_vector'], true);
            
            if (is_array($embedding)) {
                $similarity = cosineSimilarity($questionEmbedding, $embedding);
                $similarities[] = [
                    'id' => $itemId,
                    'similarity' => $similarity
                ];
            }
        }
        
        // Сортируем по similarity (от большего к меньшему)
        usort($similarities, function($a, $b) {
            return $b['similarity'] <=> $a['similarity'];
        });
        
        // Берем топ-N результатов
        $results[$sectionName] = array_slice($similarities, 0, $limit);
    }
    
    return $results;
}

/**
 * Создать embedding для записи (используется при парсинге)
 * 
 * @param PDO $db Подключение к БД
 * @param int $itemId ID записи
 * @return bool Успех операции
 */
function createEmbeddingForItem($db, $itemId) {
    try {
        // Собираем текст из полей
        $itemText = buildItemTextForEmbedding($db, $itemId);
        
        if (empty($itemText)) {
            // Если нет текста, пропускаем
            return false;
        }
        
        // Проверяем, не изменился ли текст (по хешу)
        $textHash = md5($itemText);
        
        $stmt = $db->prepare("SELECT text_hash FROM item_embeddings WHERE item_id = ?");
        $stmt->execute([$itemId]);
        $existing = $stmt->fetch();
        
        // Если embedding уже существует и текст не изменился, пропускаем
        if ($existing && $existing['text_hash'] === $textHash) {
            return true;
        }
        
        // Создаем embedding
        $embedding = createEmbedding($itemText);
        
        if (empty($embedding)) {
            return false;
        }
        
        // Сохраняем embedding
        return saveItemEmbedding($db, $itemId, $embedding, $textHash);
        
    } catch (Exception $e) {
        error_log("Error creating embedding for item $itemId: " . $e->getMessage());
        return false;
    }
}

