<?php
/**
 * API endpoint для тестирования кастомного API с полным циклом запросов
 * POST /api/custom-api-query.php
 * 
 * Параметры:
 * - widget_key: ключ виджета
 * - question: вопрос пользователя
 * - api_url: URL кастомного API
 * - api_key: API ключ (опционально)
 * - _debug: режим отладки
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/embedding-functions.php';

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }

    $widget_key = $input['widget_key'] ?? '';
    $question = trim($input['question'] ?? '');
    $api_url = $input['api_url'] ?? '';
    $api_key = $input['api_key'] ?? '';
    $max_tokens_stage1 = $input['max_tokens_stage1'] ?? 200;
    $max_tokens_stage2 = $input['max_tokens_stage2'] ?? 1000;
    $max_tokens_stage3 = $input['max_tokens_stage3'] ?? 2048;
    $debugMode = $input['_debug'] ?? false;
    
    // Ограничиваем max_tokens до 2048 (ограничение API)
    $max_tokens_stage1 = min($max_tokens_stage1, 2048);
    $max_tokens_stage2 = min($max_tokens_stage2, 2048);
    $max_tokens_stage3 = min($max_tokens_stage3, 2048);

    if (empty($widget_key) || empty($question)) {
        throw new Exception('widget_key and question are required');
    }

    if (empty($api_url)) {
        throw new Exception('api_url is required');
    }

    $db = getDatabase();
    
    // Получаем виджет и настройки
    $stmt = $db->prepare("
        SELECT w.*, ws.*
        FROM widgets w
        LEFT JOIN widget_settings ws ON w.id = ws.widget_id
        WHERE w.widget_key = ? AND w.active = 1
    ");
    $stmt->execute([$widget_key]);
    $widget = $stmt->fetch();
    
    if (!$widget) {
        throw new Exception('Widget not found');
    }

    // Получаем модели для этапов
    $stage1_model = $widget['stage1_model'] ?? 'custom-model';
    $stage2_model = $widget['stage2_model'] ?? 'custom-model';
    $stage3_model = $widget['stage3_model'] ?? 'custom-model';
    $stage1_prompt = $widget['stage1_prompt'] ?? 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей кратко и по делу (2-3 предложения).';
    $stage2_prompt = $widget['stage2_prompt'] ?? null;
    $stage3_prompt = $widget['stage3_prompt'] ?? null;

    $debug = [];
    $debug['three_stage_search'] = [];

    // ========== ЭТАП 1: Быстрый текстовый ответ ==========
    $stage1StartTime = microtime(true);
    $stage1SystemPrompt = $stage1_prompt;
    $stage1UserPrompt = 'Вопрос пользователя: ' . $question;
    
    $stage1Request = [
        'model' => $stage1_model,
        'max_tokens' => $max_tokens_stage1,
        'messages' => [
            ['role' => 'user', 'content' => $stage1SystemPrompt . "\n\n" . $stage1UserPrompt]
        ],
        'temperature' => 0.7
    ];

    $debug['three_stage_search']['api_url'] = $api_url; // Добавляем URL в debug
    $stage1Response = callCustomAPI($api_url, $api_key, $stage1Request);
    $stage1Time = round((microtime(true) - $stage1StartTime) * 1000);

    $stage1Text = '';
    if (isset($stage1Response['choices'][0]['message']['content'])) {
        $stage1Text = trim($stage1Response['choices'][0]['message']['content']);
    }

    $debug['three_stage_search']['stage1'] = [
        'question' => $question,
        'time_ms' => $stage1Time,
        'response' => $stage1Text,
        'request' => $stage1Request,
        'api_response' => $stage1Response
    ];

    // ========== ЭТАП 2: Извлечение медицинских терминов ==========
    $stage2StartTime = microtime(true);
    
    // Формируем промпт для извлечения терминов
    if ($stage2_prompt) {
        $extractionPrompt = str_replace('{QUESTION}', $question, $stage2_prompt);
    } else {
        $extractionPrompt = 'ВАЖНО: Верни ТОЛЬКО валидный JSON, без дополнительного текста до или после JSON.

Твоя задача - максимально полно извлечь все релевантные медицинские термины из вопроса пользователя, включая:
- Прямо упомянутые симптомы, заболевания, состояния
- Связанные симптомы и состояния, которые могут быть связаны с упомянутыми
- Все возможные специализации и специалистов, которые могут помочь
- Типы услуг, диагностики и лечения, которые могут быть релевантны
- Ключевые слова для поиска, включая синонимы, медицинские термины и разговорные названия

Думай широко: если пользователь упоминает симптом, подумай о возможных причинах, связанных состояниях и специалистах, которые могут помочь.

Из вопроса извлеки и верни в формате JSON:
{
  "symptoms": [массив всех симптомов, включая связанные и возможные],
  "specializations": [массив всех релевантных специализаций и специалистов],
  "service_types": [массив всех типов услуг, диагностики и лечения],
  "keywords": [объединенный массив всех релевантных ключевых слов, включая синонимы и связанные термины]
}

Вопрос пользователя: ' . $question . '

Верни ТОЛЬКО JSON объект, без объяснений и дополнительного текста.';
    }

    $stage2Request = [
        'model' => $stage2_model,
        'max_tokens' => $max_tokens_stage2,
        'messages' => [
            ['role' => 'user', 'content' => $extractionPrompt]
        ],
        'temperature' => 0.3
    ];

    $stage2Response = callCustomAPI($api_url, $api_key, $stage2Request);
    $stage2Time = round((microtime(true) - $stage2StartTime) * 1000);

    // Парсим JSON из ответа
    $medicalTerms = [];
    $allExtractedTerms = [];
    
    if (isset($stage2Response['choices'][0]['message']['content'])) {
        $stage2Text = trim($stage2Response['choices'][0]['message']['content']);
        
        // Извлекаем JSON из ответа
        if (preg_match('/```json\s*(.*?)\s*```/s', $stage2Text, $matches)) {
            $stage2Text = $matches[1];
        } elseif (preg_match('/```\s*(.*?)\s*```/s', $stage2Text, $matches)) {
            $stage2Text = $matches[1];
        }
        
        $medicalTerms = json_decode($stage2Text, true);
        if (!$medicalTerms) {
            $medicalTerms = json_decode($stage2Text, true);
        }
        
        if ($medicalTerms) {
            $allExtractedTerms = array_merge(
                $medicalTerms['symptoms'] ?? [],
                $medicalTerms['specializations'] ?? [],
                $medicalTerms['service_types'] ?? [],
                $medicalTerms['keywords'] ?? []
            );
            $allExtractedTerms = array_values(array_unique($allExtractedTerms)); // array_values для сохранения массива, а не объекта
        }
    }

    $debug['three_stage_search']['stage2'] = [
        'question' => $question,
        'time_ms' => $stage2Time,
        'medical_terms' => $medicalTerms,
        'all_extracted_terms' => $allExtractedTerms,
        'request' => $stage2Request,
        'api_response' => $stage2Response
    ];

    // ========== ЭТАП 3: Основной запрос с данными ==========
    $stage3StartTime = microtime(true);
    
    // Получаем данные виджета (как в query.php)
    $stmt = $db->prepare("
        SELECT DISTINCT
            ws.section_name,
            sf.field_name,
            sf.use_in_prompt
        FROM widget_sections ws
        JOIN section_fields sf ON ws.id = sf.section_id
        WHERE ws.widget_id = ? AND ws.is_active = 1 AND sf.use_in_prompt = 1
        
        UNION
        
        SELECT DISTINCT
            ws.section_name,
            scf.child_field_name as field_name,
            scf.use_in_prompt
        FROM widget_sections ws
        JOIN section_fields sf ON ws.id = sf.section_id
        JOIN section_child_fields scf ON sf.id = scf.field_id
        WHERE ws.widget_id = ? AND ws.is_active = 1 AND scf.use_in_prompt = 1
        
        ORDER BY section_name
    ");
    $stmt->execute([$widget['id'], $widget['id']]);
    $fields_by_section = [];
    while ($row = $stmt->fetch()) {
        $fields_by_section[$row['section_name']][] = $row['field_name'];
    }

    // Фильтруем данные по ключевым словам (упрощенная версия)
    $prompt_data = [];
    foreach ($fields_by_section as $section_name => $fields) {
        $placeholders = str_repeat('?,', count($fields) - 1) . '?';
        $stmt = $db->prepare("
            SELECT pi.id, pf.field_name, pf.field_value
            FROM parsed_items pi
            JOIN parsed_fields pf ON pi.id = pf.item_id
            WHERE pi.widget_id = ? AND pi.section_name = ? 
            AND pf.field_name IN ($placeholders)
            AND pi.is_duplicate = 0
            ORDER BY pi.id
            LIMIT 100
        ");
        $params = array_merge([$widget['id'], $section_name], $fields);
        $stmt->execute($params);
        
        $items = [];
        while ($row = $stmt->fetch()) {
            if (!isset($items[$row['id']])) {
                $items[$row['id']] = ['id' => (int)$row['id']];
            }
            $items[$row['id']][$row['field_name']] = $row['field_value'];
        }
        
        $prompt_data[$section_name] = array_values($items);
    }

    $json_data = json_encode($prompt_data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    $full_prompt = ($stage3_prompt ?: $widget['claude_prompt']) . "\n\n" . $json_data;
    
    $promptSizeBefore = mb_strlen($full_prompt);
    $promptTokensBefore = estimateTokens($full_prompt);
    
    // Оптимизируем промпт (упрощенная версия - можно улучшить)
    $optimized_prompt = $full_prompt; // В реальной версии здесь фильтрация по ключевым словам
    $promptSizeAfter = mb_strlen($optimized_prompt);
    $promptTokensAfter = estimateTokens($optimized_prompt);
    $reductionPercent = $promptTokensBefore > 0 
        ? round((1 - $promptTokensAfter / $promptTokensBefore) * 100, 1) 
        : 0;

    $stage3SystemPrompt = $stage3_prompt ?: $widget['claude_prompt'];
    $stage3UserPrompt = "Вопрос пользователя: $question\n\nДанные:\n$json_data";
    
    $stage3Request = [
        'model' => $stage3_model,
        'max_tokens' => $max_tokens_stage3, // Ограничение API: максимум 2048 токенов
        'messages' => [
            ['role' => 'user', 'content' => $stage3SystemPrompt . "\n\n" . $stage3UserPrompt]
        ],
        'temperature' => 0.3
    ];

    $stage3Response = callCustomAPI($api_url, $api_key, $stage3Request);
    $stage3Time = round((microtime(true) - $stage3StartTime) * 1000);

    $stage3Text = '';
    if (isset($stage3Response['choices'][0]['message']['content'])) {
        $stage3Text = trim($stage3Response['choices'][0]['message']['content']);
        
        // Извлекаем JSON из markdown code block, если есть
        if (preg_match('/```json\s*(.*?)\s*```/s', $stage3Text, $matches)) {
            $stage3Text = trim($matches[1]);
        } elseif (preg_match('/```\s*(.*?)\s*```/s', $stage3Text, $matches)) {
            $stage3Text = trim($matches[1]);
        }
    }

    $debug['three_stage_search']['stage3'] = [
        'question' => $question,
        'time_ms' => $stage3Time,
        'prompt_before' => $full_prompt,
        'prompt_after' => $optimized_prompt,
        'prompt_size_before' => $promptSizeBefore,
        'prompt_size_after' => $promptSizeAfter,
        'prompt_tokens_before' => $promptTokensBefore,
        'prompt_tokens_after' => $promptTokensAfter,
        'reduction_percent' => $reductionPercent,
        'response' => $stage3Text,
        'request' => $stage3Request,
        'api_response' => $stage3Response
    ];

    // Парсим ответ этапа 3 (упрощенная версия)
    $parsedResponse = ['text' => $stage3Text, 'data_ids' => []];

    // Формируем итоговый ответ
    $result = [
        'success' => true,
        'text' => $stage3Text,
        'data' => $parsedResponse,
        'debug' => $debug,
        'response_time_ms' => $stage1Time + $stage2Time + $stage3Time
    ];

    echo json_encode($result, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

/**
 * Вызов кастомного API
 */
function callCustomAPI($api_url, $api_key, $request_data) {
    $headers = [
        'Content-Type: application/json'
    ];

    if (!empty($api_key)) {
        if (strpos($api_key, 'Bearer ') === 0) {
            $headers[] = 'Authorization: ' . $api_key;
        } else {
            $headers[] = 'Authorization: Bearer ' . $api_key;
        }
    }

    $ch = curl_init($api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($request_data));
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError) {
        throw new Exception('cURL error: ' . $curlError);
    }

    if ($httpCode !== 200) {
        throw new Exception("API returned HTTP $httpCode: " . substr($response, 0, 500));
    }

    $responseData = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid JSON response: ' . json_last_error_msg());
    }

    return $responseData;
}

