<?php
/**
 * API endpoint для очистки кеша виджета
 * POST /api/clear-cache.php
 * 
 * Очищает:
 * 1. Логи ответов в БД (widget_logs) - для очистки кешированных ответов
 * 2. Кеш rate limiting (файлы в папке /cache/)
 * 
 * НЕ удаляет:
 * - parsed_items и parsed_fields (спарсенные данные)
 * - sitemap_urls (кеш URLs)
 */

session_start();
require_once __DIR__ . '/../config.php';

header('Content-Type: application/json');

try {
    // Проверяем аутентификацию
    if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        exit;
    }
    
    // Получаем параметры
    $input = json_decode(file_get_contents('php://input'), true);
    $widget_id = $input['widget_id'] ?? 0;
    
    if (!$widget_id) {
        http_response_code(400);
        echo json_encode(['error' => 'widget_id is required']);
        exit;
    }
    
    // Получаем виджет
    $db = getDatabase();
    $stmt = $db->prepare("SELECT id, widget_key FROM widgets WHERE id = ?");
    $stmt->execute([$widget_id]);
    $widget = $stmt->fetch();
    
    if (!$widget) {
        http_response_code(404);
        echo json_encode(['error' => 'Widget not found']);
        exit;
    }
    
    // 1. Удаляем все логи ответов для этого виджета (очищаем кеш ответов)
    $stmtDelete = $db->prepare("DELETE FROM widget_logs WHERE widget_id = ?");
    $stmtDelete->execute([$widget_id]);
    $logsDeleted = $stmtDelete->rowCount();
    
    // 2. Очищаем файлы rate limiting кеша
    $cacheDir = WIDGET_ROOT . '/cache';
    $filesDeleted = 0;
    
    if (is_dir($cacheDir)) {
        $files = glob($cacheDir . '/rate_limit_*');
        foreach ($files as $file) {
            if (is_file($file)) {
                unlink($file);
                $filesDeleted++;
            }
        }
    }
    
    // 3. Очищаем кеш извлечения медицинских терминов (Этап 2) для этого виджета
    $termCacheDir = LOG_DIR . '/term_extraction_cache';
    $termCacheFilesDeleted = 0;
    
    if (is_dir($termCacheDir)) {
        // Удаляем все файлы кеша, которые могут принадлежать этому виджету
        // Формат ключа: md5($widgetId . '_' . $normalizedQuestion)
        // Проверяем по widget_id в содержимом файла
        // Также удаляем старые файлы без поля widget_id (они могут быть для этого виджета)
        $files = glob($termCacheDir . '/*.json');
        foreach ($files as $file) {
            if (is_file($file)) {
                $content = @file_get_contents($file);
                if ($content) {
                    $data = @json_decode($content, true);
                    // Удаляем файлы с widget_id этого виджета
                    // Или старые файлы без widget_id (они могут быть нерелевантными)
                    if ($data && (
                        (isset($data['widget_id']) && $data['widget_id'] == $widget_id) ||
                        !isset($data['widget_id']) // Старые файлы без widget_id
                    )) {
                        unlink($file);
                        $termCacheFilesDeleted++;
                    }
                }
            }
        }
    }
    
    // Возвращаем успешный ответ
    echo json_encode([
        'success' => true,
        'message' => 'Cache cleared successfully',
        'widget_id' => $widget_id,
        'widget_key' => $widget['widget_key'],
        'logs_cleared' => $logsDeleted,
        'rate_limit_files_cleared' => $filesDeleted,
        'term_extraction_cache_files_cleared' => $termCacheFilesDeleted
    ]);
    
} catch (Exception $e) {
    logError("Clear cache error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}
