<?php
/**
 * API endpoint для категоризации элементов данных пакетами
 * POST /api/categorize-items.php
 * Body: { "widget_id": 1, "section_name": "specialists", "model": "google/gemini-2.5-flash-lite", "offset": 0, "limit": 10 }
 * 
 * GET /api/categorize-items.php?widget_id=1&section_name=specialists
 * - Возвращает статистику категоризации
 */

ini_set('display_errors', 0);
error_reporting(E_ALL);

header('Content-Type: application/json');

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/optimization-functions.php';

// Обработка GET запроса - возвращает статистику
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $widgetId = (int)($_GET['widget_id'] ?? 0);
    $sectionName = trim($_GET['section_name'] ?? '');
    
    if (!$widgetId || !$sectionName) {
        http_response_code(400);
        echo json_encode(['error' => 'widget_id and section_name are required']);
        exit;
    }
    
    try {
        $db = getDatabase();
        
        // Получаем общее количество элементов
        $stmt = $db->prepare("
            SELECT COUNT(DISTINCT pi.id) as total
            FROM parsed_items pi
            WHERE pi.widget_id = ? AND pi.section_name = ? AND pi.is_duplicate = 0
        ");
        $stmt->execute([$widgetId, $sectionName]);
        $totalItems = (int)$stmt->fetch()['total'];
        
        // Получаем количество элементов с категориями
        $stmt = $db->prepare("
            SELECT COUNT(DISTINCT item_id) as categorized
            FROM item_categories
            WHERE widget_id = ? AND section_name = ?
        ");
        $stmt->execute([$widgetId, $sectionName]);
        $categorizedItems = (int)$stmt->fetch()['categorized'];
        
        // Получаем общее количество категорий
        $stmt = $db->prepare("
            SELECT COUNT(*) as total_categories
            FROM item_categories
            WHERE widget_id = ? AND section_name = ?
        ");
        $stmt->execute([$widgetId, $sectionName]);
        $totalCategories = (int)$stmt->fetch()['total_categories'];
        
        echo json_encode([
            'total_items' => $totalItems,
            'categorized_items' => $categorizedItems,
            'total_categories' => $totalCategories,
            'progress_percent' => $totalItems > 0 ? round(($categorizedItems / $totalItems) * 100, 2) : 0
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
    exit;
}

// Обработка POST запроса - выполняет категоризацию пакета элементов
$input = json_decode(file_get_contents('php://input'), true);
$widgetId = (int)($input['widget_id'] ?? 0);
$sectionName = trim($input['section_name'] ?? '');
$model = trim($input['model'] ?? 'google/gemini-2.5-flash-lite');
$offset = (int)($input['offset'] ?? 0);
$limit = (int)($input['limit'] ?? 10);

if (!$widgetId || !$sectionName) {
    http_response_code(400);
    echo json_encode(['error' => 'widget_id and section_name are required']);
    exit;
}

try {
    $db = getDatabase();
    
    // Проверяем существование виджета
    $stmt = $db->prepare("SELECT id FROM widgets WHERE id = ?");
    $stmt->execute([$widgetId]);
    if (!$stmt->fetch()) {
        http_response_code(404);
        echo json_encode(['error' => 'Widget not found']);
        exit;
    }
    
    // Загружаем элементы для обработки
    $items = loadSectionData($widgetId, $sectionName);
    
    if (empty($items)) {
        http_response_code(404);
        echo json_encode(['error' => 'No items found for this section']);
        exit;
    }
    
    // Получаем пакет элементов для обработки
    $batch = array_slice($items, $offset, $limit);
    
    if (empty($batch)) {
        echo json_encode([
            'processed' => 0,
            'total' => count($items),
            'offset' => $offset,
            'limit' => $limit,
            'categories_created' => 0,
            'completed' => true
        ]);
        exit;
    }
    
    // Выполняем категоризацию пакета
    $result = executePrefilterOptimization($widgetId, $sectionName, $batch, $model);
    
    // Получаем общее количество категорий после обработки
    $stmt = $db->prepare("
        SELECT COUNT(*) as total_categories
        FROM item_categories
        WHERE widget_id = ? AND section_name = ?
    ");
    $stmt->execute([$widgetId, $sectionName]);
    $totalCategories = (int)$stmt->fetch()['total_categories'];
    
    echo json_encode([
        'processed' => $result['processed'],
        'total' => count($items),
        'offset' => $offset,
        'limit' => $limit,
        'categories_created' => $result['categories_created'],
        'total_categories' => $totalCategories,
        'completed' => ($offset + $limit) >= count($items)
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

