<?php
$page_title = 'Управление URL';
require_once __DIR__ . '/includes/header.php';

$widget_id = $_GET['id'] ?? 0;

// Получаем виджет
$db = getDatabase();
$stmt = $db->prepare("
    SELECT w.*, ws.*
    FROM widgets w
    LEFT JOIN widget_settings ws ON w.id = ws.widget_id
    WHERE w.id = ?
");
$stmt->execute([$widget_id]);
$widget = (object) $stmt->fetch(PDO::FETCH_ASSOC);

if (!$widget) {
    header('Location: /admin/');
    exit;
}

// Обработка сохранения настроек краулинга
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'save_crawler_settings') {
    $crawler_enabled = isset($_POST['crawler_enabled']) ? 1 : 0;
    $crawler_max_depth = (int)($_POST['crawler_max_depth'] ?? 10);
    $crawler_max_urls = (int)($_POST['crawler_max_urls'] ?? 5000);
    $crawler_timeout = (int)($_POST['crawler_timeout'] ?? 300);
    
    // Парсим exclude patterns (по одному на строку)
    $exclude_patterns = [];
    if (!empty($_POST['crawler_exclude_patterns'])) {
        $lines = explode("\n", $_POST['crawler_exclude_patterns']);
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line) {
                $exclude_patterns[] = $line;
            }
        }
    }
    
    $stmt = $db->prepare("
        UPDATE widget_settings 
        SET crawler_enabled = ?,
            crawler_max_depth = ?,
            crawler_max_urls = ?,
            crawler_timeout = ?,
            crawler_exclude_patterns = ?
        WHERE widget_id = ?
    ");
    
    $stmt->execute([
        $crawler_enabled,
        $crawler_max_depth,
        $crawler_max_urls,
        $crawler_timeout,
        json_encode($exclude_patterns),
        $widget_id
    ]);
    
    $success = 'Настройки краулинга сохранены';
    
    // Перезагружаем данные
    $stmt = $db->prepare("SELECT * FROM widget_settings WHERE widget_id = ?");
    $stmt->execute([$widget_id]);
    $widget = (object) array_merge((array)$widget, $stmt->fetch());
}

// Обработка добавления URL вручную
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add_url') {
    $url = trim($_POST['url'] ?? '');
    $section = !empty($_POST['section']) ? $_POST['section'] : null;
    
    if (!empty($url)) {
        // Валидируем URL
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            $error = 'Некорректный URL: ' . htmlspecialchars($url);
        } else {
            try {
                // Добавляем URL в discovered_urls
                $stmt = $db->prepare("
                    INSERT INTO discovered_urls (widget_id, url, normalized_url, assigned_section, status, is_canonical, discovered_at)
                    VALUES (?, ?, ?, ?, 'new', 1, NOW())
                    ON DUPLICATE KEY UPDATE status = 'new', is_canonical = 1
                ");
                
                $normalized_url = parse_url($url, PHP_URL_PATH) . (parse_url($url, PHP_URL_QUERY) ? '?' . parse_url($url, PHP_URL_QUERY) : '');
                
                $stmt->execute([
                    $widget_id,
                    $url,
                    $normalized_url,
                    $section
                ]);
                
                $success = 'URL успешно добавлена: ' . htmlspecialchars($url);
            } catch (Exception $e) {
                $error = 'Ошибка добавления URL: ' . $e->getMessage();
            }
        }
    } else {
        $error = 'Пожалуйста, введите URL';
    }
}

// Получаем статистику по URL
$stmt = $db->prepare("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'new' THEN 1 ELSE 0 END) as new_count,
        SUM(CASE WHEN status = 'assigned' THEN 1 ELSE 0 END) as assigned_count,
        SUM(CASE WHEN status = 'parsed' THEN 1 ELSE 0 END) as parsed_count,
        SUM(CASE WHEN status = 'parse_error' THEN 1 ELSE 0 END) as error_count
    FROM discovered_urls
    WHERE widget_id = ? AND is_canonical = 1
");
$stmt->execute([$widget_id]);
$stats = $stmt->fetch();

// Получаем последнее задание краулинга
$stmt = $db->prepare("
    SELECT * FROM crawler_jobs 
    WHERE widget_id = ? 
    ORDER BY id DESC 
    LIMIT 1
");
$stmt->execute([$widget_id]);
$lastCrawlerJob = $stmt->fetch();

// Парсим exclude patterns
$excludePatterns = [];
if (!empty($widget->crawler_exclude_patterns)) {
    $excludePatterns = json_decode($widget->crawler_exclude_patterns, true) ?? [];
}
$excludePatternsText = implode("\n", $excludePatterns);

// Автоматические исключения
$autoExclude = [
    'Документы: .pdf, .doc, .docx, .xls, .xlsx, .ppt, .pptx',
    'Архивы: .zip, .rar, .7z, .tar, .gz',
    'Изображения: .jpg, .jpeg, .png, .gif, .bmp, .svg, .webp',
    'Медиа: .mp3, .mp4, .avi, .mov, .wmv, .flv',
    'Файлы кода: .css, .js, .json, .xml, .txt'
];

?>

<div class="container-fluid">
    <div class="row mb-3">
        <div class="col">
            <h2><?php echo htmlspecialchars($widget->name); ?> - Управление URL</h2>
            <nav aria-label="breadcrumb">
                <ol class="breadcrumb">
                    <li class="breadcrumb-item"><a href="/admin/">Виджеты</a></li>
                    <li class="breadcrumb-item"><a href="sections-list.php?id=<?php echo $widget_id; ?>">Разделы</a></li>
                    <li class="breadcrumb-item active">Управление URL</li>
                </ol>
            </nav>
        </div>
    </div>

    <?php if (isset($success)): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <?php echo htmlspecialchars($success); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <?php if (isset($error)): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <?php echo htmlspecialchars($error); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <!-- Секция 1: Настройки краулинга -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0">Настройки краулинга</h5>
        </div>
        <div class="card-body">
            <form method="POST">
                <input type="hidden" name="action" value="save_crawler_settings">
                
                <div class="form-check mb-3">
                    <input type="checkbox" class="form-check-input" id="crawler_enabled" name="crawler_enabled" value="1" <?php echo $widget->crawler_enabled ? 'checked' : ''; ?>>
                    <label class="form-check-label" for="crawler_enabled">
                        <strong>Использовать краулинг вместо sitemap.xml</strong>
                    </label>
                    <div class="form-text">Включите эту опцию если sitemap.xml на сайте отсутствует или сломан</div>
                </div>

                <div class="row mb-3">
                    <div class="col-md-4">
                        <label for="crawler_max_depth" class="form-label">Максимальная глубина</label>
                        <input type="number" class="form-control" id="crawler_max_depth" name="crawler_max_depth" 
                               value="<?php echo htmlspecialchars($widget->crawler_max_depth ?? 10); ?>" min="1" max="20">
                        <div class="form-text">Количество уровней вложенности от главной страницы</div>
                    </div>
                    <div class="col-md-4">
                        <label for="crawler_max_urls" class="form-label">Максимум URL</label>
                        <input type="number" class="form-control" id="crawler_max_urls" name="crawler_max_urls" 
                               value="<?php echo htmlspecialchars($widget->crawler_max_urls ?? 5000); ?>" min="100" max="50000">
                        <div class="form-text">Максимальное количество страниц для сбора</div>
                    </div>
                    <div class="col-md-4">
                        <label for="crawler_timeout" class="form-label">Таймаут (сек)</label>
                        <input type="number" class="form-control" id="crawler_timeout" name="crawler_timeout" 
                               value="<?php echo htmlspecialchars($widget->crawler_timeout ?? 300); ?>" min="60" max="3600">
                        <div class="form-text">Максимальное время работы краулера</div>
                    </div>
                </div>

                <div class="mb-3">
                    <label for="crawler_exclude_patterns" class="form-label">Шаблоны исключения (по одному на строку)</label>
                    <textarea class="form-control font-monospace" id="crawler_exclude_patterns" name="crawler_exclude_patterns" 
                              rows="5" placeholder="/admin/&#10;/login/&#10;?page=&#10;?sort="><?php echo htmlspecialchars($excludePatternsText); ?></textarea>
                    <div class="form-text">URL содержащие эти паттерны будут пропущены</div>
                </div>

                <div class="alert alert-info">
                    <strong>Автоматические исключения:</strong>
                    <ul class="mb-0">
                        <?php foreach ($autoExclude as $item): ?>
                            <li><?php echo htmlspecialchars($item); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>

                <button type="submit" class="btn btn-primary">
                    <i class="bi bi-save"></i> Сохранить настройки
                </button>
            </form>
        </div>
    </div>

    <!-- Секция: Добавить URL вручную -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0">Добавить страницу вручную</h5>
        </div>
        <div class="card-body">
            <form method="POST">
                <input type="hidden" name="action" value="add_url">
                
                <div class="row mb-3">
                    <div class="col-md-8">
                        <label for="manual_url" class="form-label">URL страницы</label>
                        <input type="url" class="form-control" id="manual_url" name="url" 
                               placeholder="https://example.com/page" required>
                        <div class="form-text">Введите полный URL: https://www.example.com/path</div>
                    </div>
                    <div class="col-md-4">
                        <label for="manual_section" class="form-label">Раздел (опционально)</label>
                        <select class="form-select" id="manual_section" name="section">
                            <option value="">Не назначать</option>
                            <option value="specialists">Специалисты</option>
                            <option value="services">Услуги</option>
                            <option value="articles">Статьи</option>
                            <option value="specializations">Специализации</option>
                        </select>
                    </div>
                </div>
                
                <button type="submit" class="btn btn-success">
                    <i class="bi bi-plus-circle"></i> Добавить URL
                </button>
            </form>
        </div>
    </div>

    <!-- Секция 2: Запуск краулинга -->
    <?php if ($widget->crawler_enabled): ?>
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0">Сбор URL</h5>
        </div>
        <div class="card-body">
            <div class="mb-3">
                <p><strong>Статус:</strong> 
                    <?php if ($stats['total'] > 0): ?>
                        Найдено <strong><?php echo $stats['total']; ?></strong> уникальных URL
                        (новых: <?php echo $stats['new_count']; ?>, 
                         назначено: <?php echo $stats['assigned_count']; ?>, 
                         спарсено: <?php echo $stats['parsed_count']; ?>, 
                         ошибок: <?php echo $stats['error_count']; ?>)
                    <?php else: ?>
                        URL еще не собраны
                    <?php endif; ?>
                </p>
                <?php if ($lastCrawlerJob): ?>
                    <p class="text-muted">
                        Последний запуск: <?php echo $lastCrawlerJob['started_at']; ?> 
                        (статус: <?php echo $lastCrawlerJob['status']; ?>)
                    </p>
                <?php endif; ?>
            </div>

            <div id="crawler-progress" style="display: none;">
                <div class="progress mb-3">
                    <div id="crawler-progress-bar" class="progress-bar progress-bar-striped progress-bar-animated" 
                         role="progressbar" style="width: 0%">0%</div>
                </div>
                <div id="crawler-status" class="alert alert-info">
                    Инициализация...
                </div>
            </div>

            <div class="btn-group">
                <button type="button" class="btn btn-success" id="start-crawler-btn" onclick="startCrawler(false)">
                    <i class="bi bi-play-circle"></i> Запустить сбор URL
                </button>
                <?php if ($stats['total'] > 0): ?>
                <button type="button" class="btn btn-warning" onclick="startCrawler(true)">
                    <i class="bi bi-arrow-clockwise"></i> Пересобрать (очистить и запустить заново)
                </button>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Секция 3: Дерево URL -->
    <?php if ($stats['total'] > 0): ?>
    <div class="card mb-4">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0">Дерево URL</h5>
            <div>
                <button type="button" class="btn btn-sm btn-success me-2" id="crawl-from-selected-btn" 
                        onclick="urlTreeManager.openCrawlModal()" disabled>
                    <i class="bi bi-arrow-down-circle"></i> Краулить с выбранных (<span class="selected-count">0</span>)
                </button>
                <button type="button" class="btn btn-sm btn-warning me-2" id="parse-selected-btn" 
                        onclick="urlTreeManager.parseSelected()" disabled>
                    <i class="bi bi-play-circle"></i> Спарсить выделенные (<span class="selected-parse-count">0</span>)
                </button>
                <button type="button" class="btn btn-sm btn-primary" onclick="urlTreeManager.loadURLs()">
                    <i class="bi bi-arrow-clockwise"></i> Обновить
                </button>
            </div>
        </div>
        <div class="card-body">
            <div id="url-tree-container">
                <div class="text-center">
                    <div class="spinner-border" role="status">
                        <span class="visually-hidden">Загрузка...</span>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Секция 4: Действия парсинга -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0">Парсинг назначенных URL</h5>
        </div>
        <div class="card-body">
            <div id="parsing-progress" style="display: none;">
                <div class="progress mb-3">
                    <div id="parsing-progress-bar" class="progress-bar progress-bar-striped progress-bar-animated" 
                         role="progressbar" style="width: 0%">0%</div>
                </div>
                <div id="parsing-status" class="alert alert-info">
                    Парсинг...
                </div>
            </div>

            <div class="row g-3">
                <div class="col-md-4">
                    <button type="button" class="btn btn-primary w-100" onclick="parseSection(null)">
                        <i class="bi bi-play-circle"></i> Парсить все назначенные
                    </button>
                </div>
                <div class="col-md-4">
                    <button type="button" class="btn btn-outline-primary w-100" onclick="parseSection('specialists')">
                        <i class="bi bi-person"></i> Парсить: Специалисты
                    </button>
                </div>
                <div class="col-md-4">
                    <button type="button" class="btn btn-outline-primary w-100" onclick="parseSection('services')">
                        <i class="bi bi-heart-pulse"></i> Парсить: Услуги
                    </button>
                </div>
                <div class="col-md-4">
                    <button type="button" class="btn btn-outline-primary w-100" onclick="parseSection('articles')">
                        <i class="bi bi-file-text"></i> Парсить: Статьи
                    </button>
                </div>
                <div class="col-md-4">
                    <button type="button" class="btn btn-outline-primary w-100" onclick="parseSection('specializations')">
                        <i class="bi bi-list"></i> Парсить: Специализации
                    </button>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <?php else: ?>
    <div class="alert alert-warning">
        <i class="bi bi-exclamation-triangle"></i>
        Краулинг отключен. Включите его в настройках выше для сбора URL без sitemap.xml
    </div>
    <?php endif; ?>
</div>

<script src="assets/url-tree.js"></script>
<script>
const widgetId = <?php echo $widget_id; ?>;
let urlTreeManager;
let currentCrawlerJobId = null;
let currentParsingJobId = null;

// Инициализация после загрузки DOM
document.addEventListener('DOMContentLoaded', function() {
    <?php if ($widget->crawler_enabled && $stats['total'] > 0): ?>
    // Инициализируем менеджер дерева
    urlTreeManager = new URLTreeManager(widgetId, 'url-tree-container');
    urlTreeManager.loadURLs();
    <?php endif; ?>
});

// Запуск краулинга
async function startCrawler(clearExisting) {
    if (clearExisting) {
        if (!confirm('Все существующие URL будут удалены. Продолжить?')) {
            return;
        }
    }
    
    const btn = document.getElementById('start-crawler-btn');
    btn.disabled = true;
    
    document.getElementById('crawler-progress').style.display = 'block';
    document.getElementById('crawler-status').textContent = 'Запуск краулинга...';
    
    try {
        const formData = new FormData();
        formData.append('widget_id', widgetId);
        formData.append('clear_existing', clearExisting ? 'true' : 'false');
        
        const response = await fetch('ajax/crawler-start.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.error) {
            throw new Error(data.error);
        }
        
        currentCrawlerJobId = data.job_id;
        pollCrawlerStatus();
        
    } catch (error) {
        alert('Ошибка запуска краулинга: ' + error.message);
        document.getElementById('crawler-progress').style.display = 'none';
        btn.disabled = false;
    }
}

// Polling статуса краулинга
async function pollCrawlerStatus() {
    try {
        const response = await fetch(`ajax/crawler-status.php?job_id=${currentCrawlerJobId}`);
        const data = await response.json();
        
        if (data.error) {
            throw new Error(data.error);
        }
        
        const percent = data.urls_processed > 0 
            ? Math.min(100, Math.round((data.urls_processed / Math.max(data.urls_found, data.urls_processed)) * 100))
            : 0;
        
        document.getElementById('crawler-progress-bar').style.width = percent + '%';
        document.getElementById('crawler-progress-bar').textContent = percent + '%';
        document.getElementById('crawler-status').textContent = 
            `Найдено: ${data.urls_found} | Обработано: ${data.urls_processed}`;
        
        if (data.status === 'running') {
            setTimeout(pollCrawlerStatus, 2000);
        } else if (data.status === 'completed') {
            document.getElementById('crawler-status').className = 'alert alert-success';
            document.getElementById('crawler-status').textContent = 
                `Краулинг завершен! Найдено ${data.urls_found} уникальных URL.`;
            setTimeout(() => {
                location.reload();
            }, 2000);
        } else if (data.status === 'failed') {
            document.getElementById('crawler-status').className = 'alert alert-danger';
            document.getElementById('crawler-status').textContent = 
                'Ошибка краулинга. Смотрите логи.';
            document.getElementById('start-crawler-btn').disabled = false;
        }
        
    } catch (error) {
        console.error('Error polling crawler status:', error);
        document.getElementById('crawler-status').className = 'alert alert-danger';
        document.getElementById('crawler-status').textContent = 'Ошибка получения статуса';
    }
}

// Парсинг раздела
async function parseSection(sectionName) {
    // Подсчитываем URL для парсинга
    const sections = sectionName ? [sectionName] : ['specialists', 'services', 'articles', 'specializations'];
    
    if (!confirm(`Начать парсинг назначенных URL${sectionName ? ' раздела ' + sectionName : ''}?`)) {
        return;
    }
    
    document.getElementById('parsing-progress').style.display = 'block';
    document.getElementById('parsing-status').textContent = 'Запуск парсинга...';
    
    try {
        // Если null - парсим все разделы последовательно
        if (!sectionName) {
            for (const section of sections) {
                await parseSectionInternal(section);
            }
        } else {
            await parseSectionInternal(sectionName);
        }
        
        document.getElementById('parsing-status').className = 'alert alert-success';
        document.getElementById('parsing-status').textContent = 'Парсинг завершен!';
        
        setTimeout(() => {
            if (urlTreeManager) {
                urlTreeManager.loadURLs();
            }
            document.getElementById('parsing-progress').style.display = 'none';
        }, 2000);
        
    } catch (error) {
        document.getElementById('parsing-status').className = 'alert alert-danger';
        document.getElementById('parsing-status').textContent = 'Ошибка парсинга: ' + error.message;
    }
}

// Внутренняя функция парсинга одного раздела
async function parseSectionInternal(sectionName) {
    const formData = new FormData();
    formData.append('widget_id', widgetId);
    formData.append('section_name', sectionName);
    
    const response = await fetch('ajax/url-parse-section.php', {
        method: 'POST',
        body: formData
    });
    
    const data = await response.json();
    
    if (data.error) {
        throw new Error(data.error);
    }
    
    if (data.total_pages === 0) {
        return; // Нет URL для парсинга
    }
    
    currentParsingJobId = data.job_id;
    
    // Ждем завершения
    await waitForParsingJob();
}

// Ожидание завершения задания парсинга
async function waitForParsingJob() {
    return new Promise((resolve, reject) => {
        const poll = async () => {
            try {
                const response = await fetch(`ajax/parsing-status.php?job_id=${currentParsingJobId}`);
                const data = await response.json();
                
                if (data.error) {
                    reject(new Error(data.error));
                    return;
                }
                
                const percent = data.total_pages > 0 
                    ? Math.round((data.processed_pages / data.total_pages) * 100)
                    : 0;
                
                document.getElementById('parsing-progress-bar').style.width = percent + '%';
                document.getElementById('parsing-progress-bar').textContent = percent + '%';
                document.getElementById('parsing-status').textContent = 
                    `Обработано: ${data.processed_pages} из ${data.total_pages}`;
                
                if (data.status === 'running') {
                    setTimeout(poll, 1000);
                } else if (data.status === 'completed') {
                    resolve();
                } else if (data.status === 'failed') {
                    reject(new Error('Parsing failed'));
                }
                
            } catch (error) {
                reject(error);
            }
        };
        
        poll();
    });
}
</script>

<!-- Модальное окно для краулинга с выбранных URL -->
<div class="modal fade" id="crawlFromUrlsModal" tabindex="-1" aria-labelledby="crawlFromUrlsModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="crawlFromUrlsModalLabel">
                    <i class="bi bi-arrow-down-circle"></i> Краулинг с выбранных URL
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p class="text-muted">
                    Краулинг начнется с выбранных URL и будет собирать новые страницы по ссылкам с них.
                </p>
                
                <div class="mb-3">
                    <label for="crawl_max_depth" class="form-label">Максимальная глубина</label>
                    <input type="number" class="form-control" id="crawl_max_depth" value="3" min="1" max="20">
                    <div class="form-text">На сколько уровней вглубь следовать по ссылкам (1-20)</div>
                </div>
                
                <div class="mb-3">
                    <label for="crawl_max_urls" class="form-label">Максимум новых URL</label>
                    <input type="number" class="form-control" id="crawl_max_urls" value="500" min="10" max="50000">
                    <div class="form-text">Сколько максимум новых страниц собрать (10-50000)</div>
                </div>
                
                <div class="mb-3">
                    <label for="crawl_timeout" class="form-label">Таймаут (сек)</label>
                    <input type="number" class="form-control" id="crawl_timeout" value="300" min="60" max="3600">
                    <div class="form-text">Максимальное время работы краулера (60-3600 сек)</div>
                </div>
                
                <div class="alert alert-info mb-0">
                    <strong>Примечание:</strong> Будут применены текущие шаблоны исключения из настроек краулинга.
                    Новые URL будут добавлены в дерево как "Не назначен".
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Отмена</button>
                <button type="button" class="btn btn-success" onclick="urlTreeManager.startCrawlFromSelected()">
                    <i class="bi bi-play-circle"></i> Запустить краулинг
                </button>
            </div>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>

