<?php
$page_title = 'Просмотр данных';
require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/../vendor/autoload.php';

use AIWidget\Parser\DuplicateDetector;

$section_id = $_GET['section_id'] ?? 0;

// Получаем раздел и виджет
$db = getDatabase();
$stmt = $db->prepare("
    SELECT ws.*, w.name as widget_name, w.id as widget_id
    FROM widget_sections ws
    JOIN widgets w ON ws.widget_id = w.id
    WHERE ws.id = ?
");
$stmt->execute([$section_id]);
$section = $stmt->fetch();

if (!$section) {
    header('Location: /admin/');
    exit;
}

$section_names = [
    'specialists' => 'Специалисты',
    'services' => 'Услуги',
    'specializations' => 'Специализации',
    'articles' => 'Статьи'
];

// Получаем данные
$page = max(1, (int)($_GET['page'] ?? 1));
$per_page = 50;
$offset = ($page - 1) * $per_page;

$stmt = $db->prepare("
    SELECT pi.*, 
        GROUP_CONCAT(CONCAT(pf.field_name, ':', SUBSTRING(pf.field_value, 1, 100)) SEPARATOR '||') as fields_preview
    FROM parsed_items pi
    LEFT JOIN parsed_fields pf ON pi.id = pf.item_id
    WHERE pi.widget_id = ? AND pi.section_name = ? AND pi.is_duplicate = 0
    GROUP BY pi.id
    ORDER BY pi.created_at DESC, pi.id DESC
");
$stmt->execute([$section['widget_id'], $section['section_name']]);
$items = $stmt->fetchAll();

// Анализ данных на корректность
$maxFieldLength = isset($_GET['max_length']) ? (int)$_GET['max_length'] : 200;
foreach ($items as &$item) {
    $hasEmpty = false;
    $hasInvalid = false;
    $invalidReasons = [];
    
    // Разбираем поля
    $fields = explode('||', $item['fields_preview']);
    foreach ($fields as $field) {
        if (!$field) continue;
        $parts = explode(':', $field, 2);
        if (count($parts) != 2) continue;
        
        $fieldName = trim($parts[0]);
        $fieldValue = trim($parts[1]);
        
        // Проверка на пустое значение
        if (empty($fieldValue)) {
            $hasEmpty = true;
            if (!in_array('empty', $invalidReasons)) {
                $invalidReasons[] = 'empty';
            }
        }
        
        // Проверка длины
        if (mb_strlen($fieldValue) > $maxFieldLength) {
            $hasInvalid = true;
            if (!in_array('too_long', $invalidReasons)) {
                $invalidReasons[] = 'too_long';
            }
        }
        
        // Проверка цены на некорректный текст
        if (stripos($fieldName, 'price') !== false || stripos($fieldName, 'цена') !== false) {
            if (preg_match('/(Цены|цены|Смотреть|смотреть|Прайс|прайс)/u', $fieldValue) || 
                (mb_strlen($fieldValue) > 100 && !preg_match('/[₽$€£¥]/u', $fieldValue))) {
                $hasInvalid = true;
                if (!in_array('price_text', $invalidReasons)) {
                    $invalidReasons[] = 'price_text';
                }
            }
        }
    }
    
    $item['has_empty'] = $hasEmpty;
    $item['has_invalid'] = $hasInvalid || $hasEmpty;
    $item['invalid_reasons'] = implode(',', $invalidReasons);
}
unset($item);

// Подсчет общего количества
$stmt = $db->prepare("
    SELECT COUNT(*) FROM parsed_items 
    WHERE widget_id = ? AND section_name = ? AND is_duplicate = 0
");
$stmt->execute([$section['widget_id'], $section['section_name']]);
$total = $stmt->fetchColumn();
$total_pages = ceil($total / $per_page);

// Поиск дублей
$duplicates = [];
$showDuplicates = isset($_GET['find_duplicates']);
if ($showDuplicates) {
    $detector = new DuplicateDetector($db);
    $duplicates = $detector->findDuplicates($section['widget_id'], $section['section_name'], 'name');
}
?>

<style>
/* Подсветка некорректных данных */
.has-invalid {
    background-color: #fff3f3 !important;
}

.has-invalid:hover {
    background-color: #ffe5e5 !important;
}

/* Подсветка выделенного элемента */
.table-warning {
    background-color: #fff3cd !important;
}

/* Инициализация Bootstrap tooltips */
</style>

<script>
// Инициализация tooltips после загрузки DOM
document.addEventListener('DOMContentLoaded', function() {
    const tooltipTriggerList = document.querySelectorAll('[data-bs-toggle="tooltip"]');
    const tooltipList = [...tooltipTriggerList].map(tooltipTriggerEl => new bootstrap.Tooltip(tooltipTriggerEl));
});
</script>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
    <h1 class="h2">
        <i class="bi bi-database"></i> Данные: <?php echo $section_names[$section['section_name']]; ?>
    </h1>
    <div class="btn-toolbar mb-2 mb-md-0">
        <a href="/admin/sections-list.php?widget_id=<?php echo $section['widget_id']; ?>" class="btn btn-sm btn-outline-secondary">
            <i class="bi bi-arrow-left"></i> К разделам
        </a>
    </div>
</div>

<div class="row">
    <div class="col-md-12">
        <div class="card mb-3">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Спарсенные данные (<?php echo $total; ?>)</h5>
                <div>
                    <a href="?section_id=<?php echo $section_id; ?>&find_duplicates=1" class="btn btn-sm btn-warning">
                        <i class="bi bi-search"></i> Найти дубли
                    </a>
                </div>
            </div>
            <div class="card-body">
                <?php if (empty($items)): ?>
                    <div class="text-center py-5">
                        <i class="bi bi-inbox" style="font-size: 4rem; opacity: 0.3;"></i>
                        <p class="mt-3 text-muted">Данные еще не спарсены</p>
                        <a href="/admin/section-parse.php?section_id=<?php echo $section_id; ?>" class="btn btn-primary">
                            Запустить парсинг
                        </a>
                    </div>
                <?php else: ?>
                    <!-- Фильтр поиска -->
                    <div class="mb-3">
                        <div class="row g-2">
                            <!-- Основной поиск -->
                            <div class="col-md-6">
                                <div class="input-group">
                                    <span class="input-group-text">
                                        <i class="bi bi-search"></i>
                                    </span>
                                    <input type="text" 
                                           class="form-control" 
                                           id="searchFilter" 
                                           placeholder="Поиск по всем полям (название, цена, URL и т.д.)..."
                                           autocomplete="off">
                                    <button class="btn btn-outline-secondary" type="button" id="clearSearch">
                                        <i class="bi bi-x-lg"></i> Очистить
                                    </button>
                                </div>
                            </div>
                            
                            <!-- Фильтр по ID -->
                            <div class="col-md-3">
                                <div class="input-group">
                                    <span class="input-group-text">
                                        <i class="bi bi-hash"></i>
                                    </span>
                                    <input type="text" 
                                           class="form-control" 
                                           id="idFilter" 
                                           placeholder="ID (100 или диапазон)"
                                           autocomplete="off">
                                    <button class="btn btn-primary" type="button" id="findById" title="Найти по точному ID (переход на нужную страницу)">
                                        <i class="bi bi-search"></i> Найти
                                    </button>
                                </div>
                                <small class="text-muted">Диапазон: 100-150, >100, <150. Или точный ID + "Найти"</small>
                            </div>
                            
                            <!-- Дополнительные фильтры -->
                            <div class="col-md-3">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" id="showEmptyOnly">
                                    <label class="form-check-label" for="showEmptyOnly">
                                        Только с пустыми полями
                                    </label>
                                </div>
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" id="showParentsOnly">
                                    <label class="form-check-label" for="showParentsOnly">
                                        Только родительские
                                    </label>
                                </div>
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" id="showInvalidOnly">
                                    <label class="form-check-label" for="showInvalidOnly">
                                        Только с некорректными данными
                                    </label>
                                </div>
                                <div class="input-group input-group-sm mt-1">
                                    <span class="input-group-text" style="font-size: 0.75rem;">Макс. длина</span>
                                    <input type="number" class="form-control" id="maxFieldLength" 
                                           value="200" min="50" max="5000" style="width: 80px; font-size: 0.85rem;">
                                </div>
                            </div>
                        </div>
                        
                        <small class="text-muted mt-2 d-block">
                            Найдено: <span id="foundCount"><?php echo count($items); ?></span> из <?php echo count($items); ?>
                        </small>
                    </div>
                    
                    <div class="table-responsive">
                        <table class="table table-hover" id="dataTable">
                            <thead>
                                <tr>
                                    <th width="5%">
                                        <input type="checkbox" id="select-all">
                                    </th>
                                    <th width="5%">ID</th>
                                    <th width="40%">Данные</th>
                                    <th width="30%">URL</th>
                                    <th width="15%">Дата</th>
                                    <th width="5%">Действия</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($items as $item): ?>
                                    <?php
                                    // Подготовка данных для tooltip
                                    $tooltipReasons = [];
                                    $reasonsArr = !empty($item['invalid_reasons']) ? explode(',', $item['invalid_reasons']) : [];
                                    foreach ($reasonsArr as $reason) {
                                        switch ($reason) {
                                            case 'empty':
                                                $tooltipReasons[] = 'Пустое поле';
                                                break;
                                            case 'too_long':
                                                $tooltipReasons[] = 'Слишком длинное значение';
                                                break;
                                            case 'price_text':
                                                $tooltipReasons[] = 'Цена содержит текст';
                                                break;
                                        }
                                    }
                                    $tooltipText = !empty($tooltipReasons) ? implode(', ', $tooltipReasons) : '';
                                    ?>
                                    <tr id="item-<?php echo $item['id']; ?>" 
                                        data-item-id="<?php echo $item['id']; ?>" 
                                        data-has-empty="<?php echo $item['has_empty'] ? '1' : '0'; ?>"
                                        data-has-invalid="<?php echo $item['has_invalid'] ? '1' : '0'; ?>"
                                        data-invalid-reasons="<?php echo htmlspecialchars($item['invalid_reasons']); ?>"
                                        data-parent-id="<?php echo $item['parent_item_id'] ?? ''; ?>"
                                        class="<?php echo (isset($_GET['highlight']) && $_GET['highlight'] == $item['id']) ? 'table-warning' : ''; ?> <?php echo $item['has_invalid'] ? 'has-invalid' : ''; ?>">
                                        <td>
                                            <input type="checkbox" class="item-checkbox" value="<?php echo $item['id']; ?>">
                                        </td>
                                        <td>
                                            <?php echo $item['id']; ?>
                                            <?php if ($item['has_invalid']): ?>
                                                <i class="bi bi-exclamation-triangle-fill text-warning ms-1" 
                                                   title="<?php echo htmlspecialchars($tooltipText); ?>" 
                                                   data-bs-toggle="tooltip"></i>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <small>
                                                <?php
                                                $fields = explode('||', $item['fields_preview']);
                                                foreach ($fields as $field) {
                                                    if (!$field) continue;
                                                    list($name, $value) = explode(':', $field, 2);
                                                    
                                                    // Проверяем, нужно ли пометить поле как проблемное
                                                    $isInvalid = false;
                                                    if (empty(trim($value))) {
                                                        $isInvalid = true;
                                                    } elseif (mb_strlen($value) > $maxFieldLength) {
                                                        $isInvalid = true;
                                                    } elseif ((stripos($name, 'price') !== false || stripos($name, 'цена') !== false) && 
                                                              (preg_match('/(Цены|цены|Смотреть|смотреть|Прайс|прайс)/u', $value) || 
                                                               (mb_strlen($value) > 100 && !preg_match('/[₽$€£¥]/u', $value)))) {
                                                        $isInvalid = true;
                                                    }
                                                    
                                                    $fieldClass = $isInvalid ? 'text-danger fw-bold' : '';
                                                    echo "<strong class='$fieldClass'>" . htmlspecialchars($name) . ":</strong> " . 
                                                         "<span class='$fieldClass'>" . htmlspecialchars($value) . "</span><br>";
                                                }
                                                ?>
                                            </small>
                                        </td>
                                        <td>
                                            <small>
                                                <a href="<?php echo htmlspecialchars($item['source_url']); ?>" target="_blank">
                                                    <?php echo htmlspecialchars(parse_url($item['source_url'], PHP_URL_PATH)); ?>
                                                </a>
                                            </small>
                                        </td>
                                        <td>
                                            <small><?php echo date('d.m.Y H:i', strtotime($item['created_at'])); ?></small>
                                        </td>
                                        <td>
                                            <button class="btn btn-sm btn-outline-danger" onclick="deleteItem(<?php echo $item['id']; ?>)">
                                                <i class="bi bi-trash"></i>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="d-flex justify-content-between align-items-center mt-3">
                        <div>
                            <button class="btn btn-danger" onclick="deleteSelected()">
                                <i class="bi bi-trash"></i> Удалить выбранные
                            </button>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Дубли -->
        <?php if ($showDuplicates && !empty($duplicates)): ?>
            <div class="card">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Найденные дубли (<?php echo count($duplicates); ?> групп)</h5>
                        <button class="btn btn-danger" onclick="deleteAllDuplicatesAllGroups()">
                            <i class="bi bi-trash"></i> Удалить все дубли во всех группах
                        </button>
                    </div>
                </div>
                <div class="card-body">
                    <?php foreach ($duplicates as $group): ?>
                        <div class="card mb-3">
                            <div class="card-header">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <strong><?php echo htmlspecialchars($group['value']); ?></strong>
                                        <span class="badge bg-warning"><?php echo $group['count']; ?> дублей</span>
                                        <span class="badge bg-secondary ms-2">
                                            <small>Поле: <code><?php echo htmlspecialchars($group['matched_field_name'] ?? 'name'); ?></code></small>
                                        </span>
                                    </div>
                                    <button class="btn btn-sm btn-danger" onclick="deleteAllDuplicates(<?php echo htmlspecialchars(json_encode(array_map(function($item) { return $item['id']; }, $group['items']))); ?>)">
                                        <i class="bi bi-trash"></i> Удалить дубли (оставить 1)
                                    </button>
                                </div>
                            </div>
                            <div class="card-body">
                                <div class="list-group">
                                    <?php foreach ($group['items'] as $item): ?>
                                        <div class="list-group-item">
                                            <div class="d-flex justify-content-between align-items-start">
                                                <div class="small">
                                                    <?php if (!empty($item['parent_item_id'])): ?>
                                                        <span class="badge bg-info me-2">
                                                            <i class="bi bi-diagram-3"></i> Дочерний элемент
                                                        </span>
                                                    <?php endif; ?>
                                                    <strong>ID:</strong> <?php echo $item['id']; ?><br>
                                                    <strong>URL:</strong> <a href="<?php echo htmlspecialchars($item['source_url']); ?>" target="_blank"><?php echo htmlspecialchars($item['source_url']); ?></a><br>
                                                    <?php if (!empty($item['parent_url'])): ?>
                                                        <strong>Родитель:</strong> <a href="<?php echo htmlspecialchars($item['parent_url']); ?>" target="_blank" class="text-muted"><?php echo htmlspecialchars($item['parent_url']); ?></a><br>
                                                    <?php endif; ?>
                                                    
                                                    <!-- Показываем совпадающее поле -->
                                                    <?php if (!empty($item['matched_field_name'])): ?>
                                                        <div class="mt-2 pt-2 border-top">
                                                            <small class="text-muted">
                                                                <strong>Совпадает:</strong>
                                                                <?php 
                                                                    $fieldName = $item['matched_field_name'];
                                                                    $fieldValue = $item['fields'][$fieldName] ?? '';
                                                                    if (mb_strlen($fieldValue) > 150) {
                                                                        $fieldValue = mb_substr($fieldValue, 0, 150) . '...';
                                                                    }
                                                                    echo htmlspecialchars($fieldValue);
                                                                ?>
                                                            </small>
                                                        </div>
                                                    <?php endif; ?>
                                                    
                                                    <strong class="d-block mt-2">Дата:</strong> <?php echo date('d.m.Y H:i', strtotime($item['created_at'])); ?>
                                                </div>
                                                <button class="btn btn-sm btn-danger" onclick="deleteItem(<?php echo $item['id']; ?>)">
                                                    <i class="bi bi-trash"></i> Удалить
                                                </button>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php elseif ($showDuplicates): ?>
            <div class="alert alert-success">
                <i class="bi bi-check-circle"></i> Дубли не найдены
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
// Выбрать все
document.getElementById('select-all')?.addEventListener('change', function(e) {
    // Выбираем только видимые элементы (не скрытые фильтром)
    document.querySelectorAll('.item-checkbox').forEach(cb => {
        const row = cb.closest('tr');
        if (row && row.style.display !== 'none') {
            cb.checked = e.target.checked;
        }
    });
});

// Удалить один элемент
function deleteItem(id) {
    if (!confirm('Удалить этот элемент?')) return;
    
    fetch('/admin/ajax/delete-items.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ item_ids: [id] })
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showAlert('Элемент удален', 'success');
            setTimeout(() => location.reload(), 1000);
        } else {
            showAlert('Ошибка: ' + data.error, 'danger');
        }
    });
}

// Удалить выбранные
function deleteSelected() {
    // Считаем только видимые выбранные элементы
    const selected = Array.from(document.querySelectorAll('.item-checkbox:checked')).filter(cb => {
        const row = cb.closest('tr');
        return row && row.style.display !== 'none';
    }).map(cb => parseInt(cb.value));
    
    if (selected.length === 0) {
        alert('Выберите элементы для удаления');
        return;
    }
    
    if (!confirm(`Удалить ${selected.length} видимых элементов?`)) return;
    
    fetch('/admin/ajax/delete-items.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ item_ids: selected })
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showAlert(`Удалено ${data.deleted} элементов`, 'success');
            setTimeout(() => location.reload(), 1000);
        } else {
            showAlert('Ошибка: ' + data.error, 'danger');
        }
    });
}

// Удалить все дубли (оставить только первый)
function deleteAllDuplicates(itemIdsToKeep) {
    if (!confirm('Вы уверены, что хотите удалить все дубли, кроме первого элемента в каждой группе?')) {
        return;
    }

    fetch('/admin/ajax/delete-all-duplicates.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ item_ids_to_keep: itemIdsToKeep })
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showAlert(`Удалено дублей: ${data.deleted_count}`, 'success');
            setTimeout(() => location.reload(), 1000);
        } else {
            showAlert('Ошибка: ' + data.error, 'danger');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('Ошибка удаления дублей: ' + error.message, 'danger');
    });
}

// Удалить все дубли во всех группах (оставить только первый)
function deleteAllDuplicatesAllGroups() {
    if (!confirm('Вы уверены, что хотите удалить все дубли во всех группах, кроме первого элемента в каждой группе?')) {
        return;
    }

    fetch('/admin/ajax/delete-all-duplicates-all-groups.php?section_id=<?php echo $section_id; ?>', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' }
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showAlert(`Удалено дублей во всех группах: ${data.deleted_count} (обработано групп: ${data.groups_processed})`, 'success');
            setTimeout(() => location.reload(), 1000);
        } else {
            showAlert('Ошибка: ' + data.error, 'danger');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('Ошибка удаления дублей во всех группах: ' + error.message, 'danger');
    });
}

// ===== ФИЛЬТРАЦИЯ ТАБЛИЦЫ =====
const searchInput = document.getElementById('searchFilter');
const idFilter = document.getElementById('idFilter');
const findByIdBtn = document.getElementById('findById');
const showEmptyOnly = document.getElementById('showEmptyOnly');
const showParentsOnly = document.getElementById('showParentsOnly');
const showInvalidOnly = document.getElementById('showInvalidOnly');
const maxFieldLength = document.getElementById('maxFieldLength');
const clearBtn = document.getElementById('clearSearch');
const foundCount = document.getElementById('foundCount');
const table = document.getElementById('dataTable');

// Функция для поиска по точному ID (с переходом на нужную страницу)
if (findByIdBtn && idFilter) {
    findByIdBtn.addEventListener('click', async () => {
        const searchId = idFilter.value.trim();
        
        if (!searchId || !/^\d+$/.test(searchId)) {
            alert('Введите точный ID (только цифры)');
            return;
        }
        
        const itemId = parseInt(searchId);
        const itemRow = document.getElementById(`item-${itemId}`);
        
        if (itemRow) {
            // Прокручиваем до элемента
            itemRow.scrollIntoView({ behavior: 'smooth', block: 'center' });
            
            // Подсвечиваем строку
            itemRow.classList.add('table-warning');
            setTimeout(() => {
                itemRow.classList.remove('table-warning');
            }, 3000);
            
            showAlert(`Найден элемент ID ${itemId}`, 'success');
        } else {
            showAlert(`Элемент с ID ${itemId} не найден на этой странице`, 'danger');
        }
    });
    
    // Enter в поле ID тоже запускает поиск
    idFilter.addEventListener('keypress', (e) => {
        if (e.key === 'Enter' && /^\d+$/.test(idFilter.value.trim())) {
            findByIdBtn.click();
        }
    });
}

if (searchInput && table) {
    const tbody = table.querySelector('tbody');
    const rows = tbody ? Array.from(tbody.querySelectorAll('tr')) : [];
    const totalCount = rows.length;
    
    /**
     * Проверить соответствует ли ID фильтру
     */
    function matchesIdFilter(itemId, filterValue) {
        if (!filterValue) return true;
        
        filterValue = filterValue.trim();
        
        // Точное совпадение: "100"
        if (/^\d+$/.test(filterValue)) {
            return itemId === parseInt(filterValue);
        }
        
        // Диапазон: "100-150"
        if (/^\d+-\d+$/.test(filterValue)) {
            const [min, max] = filterValue.split('-').map(v => parseInt(v));
            return itemId >= min && itemId <= max;
        }
        
        // Больше: ">100"
        if (/^>\d+$/.test(filterValue)) {
            const threshold = parseInt(filterValue.substring(1));
            return itemId > threshold;
        }
        
        // Меньше: "<150"
        if (/^<\d+$/.test(filterValue)) {
            const threshold = parseInt(filterValue.substring(1));
            return itemId < threshold;
        }
        
        return true;
    }
    
    /**
     * Функция комплексной фильтрации
     */
    function filterTable() {
        const searchText = searchInput.value.toLowerCase().trim();
        const idFilterValue = idFilter ? idFilter.value : '';
        const emptyFilterChecked = showEmptyOnly ? showEmptyOnly.checked : false;
        const parentsFilterChecked = showParentsOnly ? showParentsOnly.checked : false;
        const invalidFilterChecked = showInvalidOnly ? showInvalidOnly.checked : false;
        
        let visibleCount = 0;
        
        rows.forEach(row => {
            let shouldShow = true;
            
            // Фильтр по текстовому поиску
            if (searchText) {
                const rowText = row.textContent.toLowerCase();
                if (!rowText.includes(searchText)) {
                    shouldShow = false;
                }
            }
            
            // Фильтр по ID
            if (shouldShow && idFilterValue) {
                const itemId = parseInt(row.dataset.itemId);
                if (!matchesIdFilter(itemId, idFilterValue)) {
                    shouldShow = false;
                }
            }
            
            // Фильтр "только с пустыми полями"
            if (shouldShow && emptyFilterChecked) {
                const hasEmpty = row.dataset.hasEmpty === '1';
                if (!hasEmpty) {
                    shouldShow = false;
                }
            }
            
            // Фильтр "только родительские" (без parent_item_id)
            if (shouldShow && parentsFilterChecked) {
                const parentId = row.dataset.parentId;
                if (parentId && parentId !== '') {
                    shouldShow = false;
                }
            }
            
            // Фильтр "только с некорректными данными"
            if (shouldShow && invalidFilterChecked) {
                const hasInvalid = row.dataset.hasInvalid === '1';
                if (!hasInvalid) {
                    shouldShow = false;
                }
            }
            
            if (shouldShow) {
                row.style.display = '';
                visibleCount++;
            } else {
                row.style.display = 'none';
            }
        });
        
        // Обновляем счетчик
        if (foundCount) {
            foundCount.textContent = visibleCount;
        }
        
        // Показываем сообщение если ничего не найдено
        if (visibleCount === 0 && (searchText || idFilterValue || emptyFilterChecked || parentsFilterChecked || invalidFilterChecked)) {
            if (!document.getElementById('noResultsRow')) {
                const noResultsRow = document.createElement('tr');
                noResultsRow.id = 'noResultsRow';
                noResultsRow.innerHTML = `
                    <td colspan="6" class="text-center text-muted py-4">
                        <i class="bi bi-search" style="font-size: 2rem; opacity: 0.3;"></i>
                        <p class="mt-2 mb-0">Ничего не найдено по заданным фильтрам</p>
                    </td>
                `;
                tbody.appendChild(noResultsRow);
            }
        } else {
            const noResultsRow = document.getElementById('noResultsRow');
            if (noResultsRow) {
                noResultsRow.remove();
            }
        }
    }
    
    // Фильтрация при вводе текста
    searchInput.addEventListener('input', filterTable);
    
    // Фильтрация при вводе ID
    if (idFilter) {
        idFilter.addEventListener('input', filterTable);
    }
    
    // Фильтрация при изменении чекбоксов
    if (showEmptyOnly) {
        showEmptyOnly.addEventListener('change', filterTable);
    }
    
    if (showParentsOnly) {
        showParentsOnly.addEventListener('change', filterTable);
    }
    
    if (showInvalidOnly) {
        showInvalidOnly.addEventListener('change', filterTable);
    }
    
    // Фильтрация при изменении максимальной длины (с debounce)
    if (maxFieldLength) {
        let debounceTimer;
        maxFieldLength.addEventListener('input', () => {
            clearTimeout(debounceTimer);
            debounceTimer = setTimeout(() => {
                // Перезагружаем страницу с новым параметром для пересчета на сервере
                const urlParams = new URLSearchParams(window.location.search);
                urlParams.set('max_length', maxFieldLength.value);
                window.location.search = urlParams.toString();
            }, 1000);
        });
    }
    
    // Очистка всех фильтров
    if (clearBtn) {
        clearBtn.addEventListener('click', () => {
            searchInput.value = '';
            if (idFilter) idFilter.value = '';
            if (showEmptyOnly) showEmptyOnly.checked = false;
            if (showParentsOnly) showParentsOnly.checked = false;
            if (showInvalidOnly) showInvalidOnly.checked = false;
            filterTable();
            searchInput.focus();
        });
    }
    
    // Фокус на поле поиска при нажатии Ctrl+F или Cmd+F
    document.addEventListener('keydown', (e) => {
        if ((e.ctrlKey || e.metaKey) && e.key === 'f') {
            e.preventDefault();
            searchInput.focus();
            searchInput.select();
        }
    });
}

// Автоматическая прокрутка к выделенному элементу (если есть параметр highlight)
<?php if (isset($_GET['highlight'])): ?>
const highlightId = <?php echo (int)$_GET['highlight']; ?>;
const highlightEl = document.getElementById('item-' + highlightId);
if (highlightEl) {
    setTimeout(() => {
        highlightEl.scrollIntoView({ behavior: 'smooth', block: 'center' });
        // Мигание для привлечения внимания
        highlightEl.style.transition = 'background-color 0.5s';
        setTimeout(() => {
            highlightEl.style.backgroundColor = '';
        }, 2000);
    }, 300);
}
<?php endif; ?>

</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>

