<?php
$page_title = 'Генерация Embedding векторов';
require_once __DIR__ . '/../includes/header.php';

$widget_id = $_GET['id'] ?? 0;
$error = '';
$success = '';

// Получаем виджет
$db = getDatabase();
$stmt = $db->prepare("SELECT id, name FROM widgets WHERE id = ?");
$stmt->execute([$widget_id]);
$widget = $stmt->fetch();

if (!$widget) {
    header('Location: /admin/');
    exit;
}

// Получаем список разделов виджета
$stmt = $db->prepare("
    SELECT DISTINCT section_name 
    FROM widget_sections 
    WHERE widget_id = ? AND is_active = 1
    ORDER BY section_name
");
$stmt->execute([$widget_id]);
$sections = $stmt->fetchAll();

// Получаем статистику по embedding
$stats = [];
foreach ($sections as $sectionRow) {
    $sectionName = $sectionRow['section_name'];
    
    // Всего записей
    $stmt = $db->prepare("
        SELECT COUNT(*) as total
        FROM parsed_items
        WHERE widget_id = ? AND section_name = ? AND is_duplicate = 0
    ");
    $stmt->execute([$widget_id, $sectionName]);
    $total = $stmt->fetch()['total'];
    
    // Записей с embedding
    $stmt = $db->prepare("
        SELECT COUNT(*) as with_embedding
        FROM parsed_items pi
        JOIN item_embeddings ie ON pi.id = ie.item_id
        WHERE pi.widget_id = ? AND pi.section_name = ? AND pi.is_duplicate = 0
    ");
    $stmt->execute([$widget_id, $sectionName]);
    $withEmbedding = $stmt->fetch()['with_embedding'];
    
    $stats[$sectionName] = [
        'total' => $total,
        'with_embedding' => $withEmbedding,
        'without_embedding' => $total - $withEmbedding
    ];
}

$sectionNames = [
    'services' => 'Услуги',
    'specialists' => 'Специалисты',
    'articles' => 'Статьи',
    'specializations' => 'Специализации'
];
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
    <h1 class="h2">
        <i class="bi bi-vector-pen"></i> Генерация Embedding: <?php echo htmlspecialchars($widget['name']); ?>
    </h1>
    <div class="btn-toolbar mb-2 mb-md-0">
        <a href="/admin/widget-settings.php?id=<?php echo $widget_id; ?>" class="btn btn-sm btn-outline-secondary">
            <i class="bi bi-arrow-left"></i> Назад
        </a>
    </div>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
<?php endif; ?>

<?php if ($success): ?>
    <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
<?php endif; ?>

<div class="row">
    <div class="col-md-12">
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">Статистика по разделам</h5>
            </div>
            <div class="card-body">
                <table class="table table-sm">
                    <thead>
                        <tr>
                            <th>Раздел</th>
                            <th>Всего записей</th>
                            <th>С embedding</th>
                            <th>Без embedding</th>
                            <th>Процент</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($stats as $section => $stat): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($sectionNames[$section] ?? $section); ?></td>
                            <td><?php echo number_format($stat['total']); ?></td>
                            <td class="text-success"><?php echo number_format($stat['with_embedding']); ?></td>
                            <td class="text-danger"><?php echo number_format($stat['without_embedding']); ?></td>
                            <td>
                                <?php 
                                $percent = $stat['total'] > 0 ? round(($stat['with_embedding'] / $stat['total']) * 100, 1) : 0;
                                echo $percent . '%';
                                ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Генерация Embedding</h5>
            </div>
            <div class="card-body">
                <form id="generateForm">
                    <input type="hidden" name="widget_id" value="<?php echo $widget_id; ?>">
                    
                    <div class="mb-3">
                        <label for="section_name" class="form-label">Выберите раздел</label>
                        <select class="form-select" id="section_name" name="section_name" required>
                            <option value="">Все разделы</option>
                            <?php foreach ($sections as $sectionRow): ?>
                            <option value="<?php echo htmlspecialchars($sectionRow['section_name']); ?>">
                                <?php echo htmlspecialchars($sectionNames[$sectionRow['section_name']] ?? $sectionRow['section_name']); ?>
                                (<?php echo number_format($stats[$sectionRow['section_name']]['without_embedding']); ?> без embedding)
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="batch_size" class="form-label">Размер батча</label>
                        <input type="number" class="form-control" id="batch_size" name="batch_size" value="10" min="1" max="50" required>
                        <small class="form-text text-muted">Количество записей, обрабатываемых за один запрос (рекомендуется 10-20)</small>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" id="generateBtn">
                        <i class="bi bi-play-fill"></i> Начать генерацию
                    </button>
                    <button type="button" class="btn btn-secondary" id="stopBtn" style="display: none;">
                        <i class="bi bi-stop-fill"></i> Остановить
                    </button>
                </form>
                
                <div id="progressContainer" style="display: none;" class="mt-4">
                    <div class="progress mb-3">
                        <div class="progress-bar progress-bar-striped progress-bar-animated" role="progressbar" style="width: 0%"></div>
                    </div>
                    <div id="progressText" class="text-muted"></div>
                    <div id="logContainer" class="mt-3" style="max-height: 400px; overflow-y: auto; background: #f8f9fa; padding: 15px; border-radius: 8px; font-family: monospace; font-size: 0.9em;">
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
let isRunning = false;
let shouldStop = false;

document.getElementById('generateForm').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    if (isRunning) {
        return;
    }
    
    const widgetId = document.querySelector('input[name="widget_id"]').value;
    const sectionName = document.getElementById('section_name').value;
    const batchSize = parseInt(document.getElementById('batch_size').value);
    
    isRunning = true;
    shouldStop = false;
    document.getElementById('generateBtn').disabled = true;
    document.getElementById('stopBtn').style.display = 'inline-block';
    document.getElementById('progressContainer').style.display = 'block';
    document.getElementById('logContainer').innerHTML = '';
    
    let totalProcessed = 0;
    let totalErrors = 0;
    let offset = 0;
    
    try {
        while (!shouldStop) {
            const response = await fetch('/admin/ajax/generate-embeddings-batch.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    widget_id: parseInt(widgetId),
                    section_name: sectionName && sectionName !== '' ? sectionName : null,
                    offset: offset,
                    limit: batchSize
                })
            });
            
            const data = await response.json();
            
            if (!data.success) {
                throw new Error(data.error || 'Неизвестная ошибка');
            }
            
            totalProcessed += data.processed || 0;
            totalErrors += data.errors || 0;
            
            // Обновляем прогресс
            const progressPercent = data.total > 0 ? Math.round((data.processed_total / data.total) * 100) : 0;
            document.querySelector('.progress-bar').style.width = progressPercent + '%';
            document.getElementById('progressText').textContent = 
                `Обработано: ${data.processed_total} из ${data.total} (${progressPercent}%) | Ошибок: ${totalErrors}`;
            
            // Добавляем лог
            if (data.log) {
                const logDiv = document.createElement('div');
                logDiv.textContent = data.log;
                document.getElementById('logContainer').appendChild(logDiv);
                document.getElementById('logContainer').scrollTop = document.getElementById('logContainer').scrollHeight;
            }
            
            // Если обработано все или нет больше записей
            if (data.processed === 0 || data.processed_total >= data.total) {
                break;
            }
            
            offset += batchSize;
            
            // Небольшая задержка между батчами
            await new Promise(resolve => setTimeout(resolve, 500));
        }
        
        if (!shouldStop) {
            document.getElementById('progressText').textContent = 
                `Генерация завершена! Обработано: ${totalProcessed} записей | Ошибок: ${totalErrors}`;
            document.querySelector('.progress-bar').classList.remove('progress-bar-animated');
            document.querySelector('.progress-bar').classList.add('bg-success');
        } else {
            document.getElementById('progressText').textContent = 'Генерация остановлена пользователем';
        }
        
    } catch (error) {
        document.getElementById('progressText').textContent = 'Ошибка: ' + error.message;
        document.getElementById('logContainer').innerHTML += '<div class="text-danger">Ошибка: ' + error.message + '</div>';
    } finally {
        isRunning = false;
        shouldStop = false;
        document.getElementById('generateBtn').disabled = false;
        document.getElementById('stopBtn').style.display = 'none';
    }
});

document.getElementById('stopBtn').addEventListener('click', function() {
    shouldStop = true;
    document.getElementById('stopBtn').disabled = true;
});
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

