<?php
/**
 * Скрипт для генерации маппинга категорий и ключевых слов через AI
 * Запускается один раз для каждого виджета
 */

session_start();

// Проверка авторизации
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    http_response_code(401);
    die(json_encode(['error' => 'Unauthorized']));
}

require_once __DIR__ . '/../../config.php';

// Получаем widget_id и section из параметров
$widgetId = isset($_GET['widget_id']) ? (int)$_GET['widget_id'] : 0;
$sectionName = isset($_GET['section']) ? $_GET['section'] : 'services';
$batchSize = isset($_GET['batch_size']) ? (int)$_GET['batch_size'] : 10;
$reset = isset($_GET['reset']) && $_GET['reset'] === '1';

// Валидация section_name
$allowedSections = ['specialists', 'services', 'articles', 'specializations'];
if (!in_array($sectionName, $allowedSections)) {
    die(json_encode(['error' => 'Invalid section name']));
}

if (!$widgetId) {
    die(json_encode(['error' => 'widget_id is required']));
}

if ($batchSize < 1 || $batchSize > 100) {
    die(json_encode(['error' => 'batch_size must be between 1 and 100']));
}

set_time_limit(300); // 5 минут на выполнение
ini_set('memory_limit', '512M');

header('Content-Type: application/json');

try {
    $db = getDatabase();
    
    // Проверяем существование виджета
    $stmt = $db->prepare("SELECT id, name FROM widgets WHERE id = ?");
    $stmt->execute([$widgetId]);
    $widget = $stmt->fetch();
    
    if (!$widget) {
        throw new Exception('Widget not found');
    }
    
    // Получаем все элементы раздела с их полями (use_in_prompt = 1)
    $stmt = $db->prepare("
        SELECT DISTINCT sf.field_name, sf.field_order
        FROM section_fields sf
        JOIN widget_sections ws ON sf.section_id = ws.id
        WHERE ws.widget_id = ? 
          AND ws.section_name = ?
          AND ws.is_active = 1
          AND sf.use_in_prompt = 1
        ORDER BY sf.field_order
    ");
    $stmt->execute([$widgetId, $sectionName]);
    $fields = $stmt->fetchAll();
    
    if (empty($fields)) {
        throw new Exception("Не найдено полей для раздела: $sectionName. Убедитесь, что в разделе настроены поля с use_in_prompt = 1.");
    }
    
    $fieldNames = array_column($fields, 'field_name');
    $placeholders = str_repeat('?,', count($fieldNames) - 1) . '?';
    
    // Получаем все элементы раздела с их данными
    $stmt = $db->prepare("
        SELECT pi.id, pf.field_name, pf.field_value
        FROM parsed_items pi
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.widget_id = ? 
          AND pi.section_name = ?
          AND pi.is_duplicate = 0
          AND pf.field_name IN ($placeholders)
        ORDER BY pi.id, pf.field_name
    ");
    $stmt->execute(array_merge([$widgetId, $sectionName], $fieldNames));
    $itemsData = $stmt->fetchAll();
    
    // Группируем данные по элементам
    $items = [];
    foreach ($itemsData as $row) {
        if (!isset($items[$row['id']])) {
            $items[$row['id']] = ['id' => $row['id']];
        }
        $items[$row['id']][$row['field_name']] = $row['field_value'];
    }
    
    if (empty($items)) {
        throw new Exception("Не найдено элементов для раздела: $sectionName.");
    }
    
    // Если reset = 1, очищаем старый маппинг и category_id
    if ($reset) {
        $stmt = $db->prepare("DELETE FROM category_keywords WHERE widget_id = ? AND section_name = ?");
        $stmt->execute([$widgetId, $sectionName]);
        
        // Удаляем category_id из parsed_fields для элементов этого раздела
        $stmt = $db->prepare("
            DELETE pf FROM parsed_fields pf
            JOIN parsed_items pi ON pf.item_id = pi.id
            WHERE pi.widget_id = ? AND pi.section_name = ? AND pf.field_name = 'category_id'
        ");
        $stmt->execute([$widgetId, $sectionName]);
    }
    
    // Получаем уже обработанные элементы (имеющие category_id)
    $stmt = $db->prepare("
        SELECT DISTINCT pi.id
        FROM parsed_items pi
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.widget_id = ? 
          AND pi.section_name = ?
          AND pi.is_duplicate = 0
          AND pf.field_name = 'category_id'
    ");
    $stmt->execute([$widgetId, $sectionName]);
    $processedItemIds = array_column($stmt->fetchAll(), 'id');
    
    // Формируем список элементов с их ID для AI (только необработанные)
    $itemsArray = array_values($items);
    $itemsList = [];
    
    foreach ($itemsArray as $index => $item) {
        // Пропускаем уже обработанные элементы
        if (in_array($item['id'], $processedItemIds)) {
            continue;
        }
        
        $itemText = [];
        foreach ($fieldNames as $fieldName) {
            if (isset($item[$fieldName]) && !empty($item[$fieldName])) {
                $itemText[] = "$fieldName: " . mb_substr($item[$fieldName], 0, 200); // Ограничиваем длину
            }
        }
        $itemsList[] = [
            'id' => $item['id'],
            'text' => implode(" | ", $itemText)
        ];
    }
    
    $totalItems = count($itemsList);
    $totalProcessed = count($processedItemIds);
    
    if (empty($itemsList)) {
        echo json_encode([
            'success' => true,
            'completed' => true,
            'message' => 'Все элементы уже обработаны',
            'total_items' => $totalItems + $totalProcessed,
            'processed' => $totalProcessed,
            'remaining' => 0
        ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }
    
    // Берем только один батч
    $batch = array_slice($itemsList, 0, $batchSize);
    $batchNumber = floor($totalProcessed / $batchSize) + 1;
    $totalBatches = ceil(($totalItems + $totalProcessed) / $batchSize);
    
    // Формируем промпт для категоризации
    $contentType = [
        'services' => 'медицинских услуг',
        'specialists' => 'специалистов/врачей',
        'articles' => 'статей',
        'specializations' => 'специализаций'
    ][$sectionName] ?? 'элементов';
    
    $itemsTextForPrompt = implode("\n", array_map(function($i, $item) {
        return ($i + 1) . ". [ID: {$item['id']}] " . $item['text'];
    }, array_keys($batch), $batch));
    
    $prompt = "Проанализируй список $contentType и сгруппируй их в логические категории.

Элементы (батч $batchNumber из $totalBatches):
$itemsTextForPrompt

Верни JSON с категориями. Каждая категория должна содержать:
- category_name: название категории (краткое, на русском)
- keywords: массив ключевых слов для поиска этой категории (синонимы, связанные термины)
- item_ids: массив ID элементов (числа из [ID: ...]), которые относятся к этой категории

ВАЖНЫЕ ПРАВИЛА КАТЕГОРИЗАЦИИ:
- Травматологи и ортопеды должны быть в ОТДЕЛЬНОЙ категории \"Врачи-травматологи и ортопеды\" (даже если они также хирурги или детские врачи)
- Если врач имеет несколько специальностей, относи его к ПРИОРИТЕТНОЙ специальности (травматолог > хирург, травматолог > педиатр)
- Детские врачи (педиатры, детские травматологи) могут быть в отдельной категории только если они ТОЛЬКО детские
- Хирурги-травматологи должны быть в категории травматологов, а не хирургов

Важно:
- Каждый элемент должен быть отнесен хотя бы к одной категории
- Категории должны быть логически связаны
- Ключевые слова должны быть на русском языке
- ID элементов должны точно соответствовать ID из списка выше

Формат ответа (строго JSON, без markdown):
{
  \"categories\": [
    {
      \"category_name\": \"Название категории\",
      \"keywords\": [\"ключевое слово 1\", \"ключевое слово 2\"],
      \"item_ids\": [123, 456, 789]
    }
  ]
}";
    
    // Отправляем запрос в Gemini через OpenRouter
    $data = [
        'model' => GEMINI_MODEL,
        'max_tokens' => 4000,
        'messages' => [
            [
                'role' => 'user',
                'content' => $prompt
            ]
        ],
        'temperature' => 0.3
    ];
    
    $ch = curl_init(GEMINI_API_URL);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: Bearer ' . OPENROUTER_API_KEY,
        'HTTP-Referer: ' . WIDGET_DOMAIN,
        'X-Title: AI Widget'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_errno($ch) ? curl_error($ch) : null;
    curl_close($ch);
    
    if ($curlError) {
        throw new Exception("CURL error: $curlError");
    }
    
    if ($httpCode !== 200) {
        throw new Exception("HTTP $httpCode: " . substr($response, 0, 500));
    }
    
    $result = json_decode($response, true);
    
    if (!isset($result['choices'][0]['message']['content'])) {
        throw new Exception("Invalid API response");
    }
    
    $answer = trim($result['choices'][0]['message']['content']);
    $rawAnswer = $answer;
    
    // Извлекаем JSON из ответа
    if (preg_match('/```json\s*(.*?)\s*```/s', $answer, $matches)) {
        $answer = $matches[1];
    } elseif (preg_match('/```\s*(.*?)\s*```/s', $answer, $matches)) {
        $answer = $matches[1];
    }
    
    $categoriesData = json_decode($answer, true);
    
    if (!$categoriesData || !isset($categoriesData['categories']) || !is_array($categoriesData['categories'])) {
        throw new Exception("Failed to parse categories. Response: " . substr($rawAnswer, 0, 500));
    }
    
    // Валидируем и обрабатываем категории из этого батча
    $batchItemIds = array_column($batch, 'id');
    $batchCategories = [];
    
    foreach ($categoriesData['categories'] as $categoryData) {
        if (!isset($categoryData['category_name']) || !isset($categoryData['keywords']) || !isset($categoryData['item_ids'])) {
            continue;
        }
        
        $categoryName = trim($categoryData['category_name']);
        $keywords = is_array($categoryData['keywords']) ? array_filter(array_map('trim', $categoryData['keywords'])) : [];
        $itemIds = is_array($categoryData['item_ids']) ? array_map('intval', $categoryData['item_ids']) : [];
        
        // Фильтруем только существующие ID из этого батча
        $validItemIds = array_intersect($itemIds, $batchItemIds);
        
        if (empty($categoryName) || empty($keywords) || empty($validItemIds)) {
            continue;
        }
        
        // Создаем уникальный идентификатор категории
        $categoryId = md5($sectionName . '|' . $categoryName);
        $categoryUrl = 'category://' . $categoryId;
        
        $batchCategories[] = [
            'category_url' => $categoryUrl,
            'category_name' => $categoryName,
            'keywords' => array_values(array_unique($keywords)),
            'item_ids' => array_values($validItemIds)
        ];
    }
    
    if (empty($batchCategories)) {
        throw new Exception("Не удалось создать категории из батча. Проверьте формат ответа AI.");
    }
    
    // Сохраняем категории в БД
    $results = [];
    $processed = 0;
    $errors = [];
    
    foreach ($batchCategories as $category) {
        $categoryUrl = $category['category_url'];
        $categoryName = $category['category_name'];
        $keywords = $category['keywords'];
        $itemIds = array_unique($category['item_ids']);
        
        if (empty($keywords)) {
            $errors[] = "Категория '$categoryName' не имеет ключевых слов";
            continue;
        }
        
        // Проверяем, существует ли уже такая категория
        $stmt = $db->prepare("SELECT id, keywords FROM category_keywords WHERE widget_id = ? AND section_name = ? AND category_url = ?");
        $stmt->execute([$widgetId, $sectionName, $categoryUrl]);
        $existing = $stmt->fetch();
        
        if ($existing) {
            // Обновляем ключевые слова (объединяем)
            $existingKeywords = json_decode($existing['keywords'], true) ?: [];
            $mergedKeywords = array_values(array_unique(array_merge($existingKeywords, $keywords)));
            $stmt = $db->prepare("UPDATE category_keywords SET keywords = ? WHERE id = ?");
            $stmt->execute([json_encode($mergedKeywords, JSON_UNESCAPED_UNICODE), $existing['id']]);
        } else {
            // Сохраняем новую категорию в БД
            $stmt = $db->prepare("
                INSERT INTO category_keywords (widget_id, section_name, category_url, category_name, keywords)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $widgetId,
                $sectionName,
                $categoryUrl,
                $categoryName,
                json_encode($keywords, JSON_UNESCAPED_UNICODE)
            ]);
        }
        
        $processed++;
        $results[] = [
            'category' => $categoryName,
            'url' => $categoryUrl,
            'keywords_count' => count($keywords),
            'items_count' => count($itemIds),
            'keywords' => array_slice($keywords, 0, 10) // Первые 10 для примера
        ];
    }
    
    // Сохраняем связь элементов с категориями через специальное поле category_id
    foreach ($batchCategories as $category) {
        if (empty($category['item_ids'])) continue;
        
        $categoryUrl = $category['category_url'];
        $itemIds = array_unique($category['item_ids']);
        
        // Сохраняем category_id для каждого элемента через parsed_fields
        // Используем специальное поле 'category_id' для хранения связи
        foreach ($itemIds as $itemId) {
            // Проверяем, есть ли уже category_id для этого элемента
            $stmt = $db->prepare("SELECT id FROM parsed_fields WHERE item_id = ? AND field_name = 'category_id'");
            $stmt->execute([$itemId]);
            $existing = $stmt->fetch();
            
            if ($existing) {
                // Обновляем существующее значение (может быть несколько категорий через запятую)
                $stmt = $db->prepare("SELECT field_value FROM parsed_fields WHERE item_id = ? AND field_name = 'category_id'");
                $stmt->execute([$itemId]);
                $currentValue = $stmt->fetchColumn();
                $categoriesList = !empty($currentValue) ? explode(',', $currentValue) : [];
                if (!in_array($categoryUrl, $categoriesList)) {
                    $categoriesList[] = $categoryUrl;
                    $stmt = $db->prepare("UPDATE parsed_fields SET field_value = ? WHERE item_id = ? AND field_name = 'category_id'");
                    $stmt->execute([implode(',', $categoriesList), $itemId]);
                }
            } else {
                // Создаем новое поле
                $stmt = $db->prepare("INSERT INTO parsed_fields (item_id, field_name, field_value) VALUES (?, 'category_id', ?)");
                $stmt->execute([$itemId, $categoryUrl]);
            }
        }
    }
    
    $remaining = $totalItems - count($batch);
    $completed = $remaining <= 0;
    
    echo json_encode([
        'success' => true,
        'completed' => $completed,
        'widget_id' => $widgetId,
        'widget_name' => $widget['name'],
        'section_name' => $sectionName,
        'batch_number' => $batchNumber,
        'total_batches' => $totalBatches,
        'batch_size' => count($batch),
        'total_items' => $totalItems + $totalProcessed,
        'processed' => $totalProcessed + count($batch),
        'remaining' => $remaining,
        'categories_created' => $processed,
        'errors' => $errors,
        'results' => $results,
        'log' => [
            'prompt' => $prompt,
            'response' => $rawAnswer
        ]
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

