<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/includes/auth.php';
?>
<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Тестер моделей OpenRouter</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <style>
        .model-result {
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 15px;
            background: #f8f9fa;
        }
        .model-result.success {
            border-color: #28a745;
            background: #d4edda;
        }
        .model-result.error {
            border-color: #dc3545;
            background: #f8d7da;
        }
        .model-result.loading {
            border-color: #ffc107;
            background: #fff3cd;
        }
        .model-header {
            font-weight: bold;
            font-size: 1.1em;
            margin-bottom: 10px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .model-name {
            color: #495057;
        }
        .model-time {
            color: #6c757d;
            font-size: 0.9em;
        }
        .model-cost {
            color: #28a745;
            font-size: 0.85em;
            margin-left: 10px;
        }
        .model-response {
            margin-top: 10px;
            padding: 10px;
            background: white;
            border-radius: 4px;
            white-space: pre-wrap;
            word-wrap: break-word;
            max-height: 400px;
            overflow-y: auto;
        }
        .model-error {
            color: #dc3545;
            margin-top: 10px;
        }
        .loading-spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid #f3f3f3;
            border-top: 3px solid #3498db;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        #results-container {
            margin-top: 30px;
        }
        .stats {
            margin-bottom: 20px;
            padding: 15px;
            background: #e9ecef;
            border-radius: 8px;
        }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h1 class="mb-4">
            <i class="bi bi-cpu"></i> Тестер моделей OpenRouter
        </h1>

        <div class="card mb-4">
            <div class="card-body">
                <form id="testForm">
                    <div class="mb-3">
                        <label for="models" class="form-label">
                            <strong>Модели (по одной на строку)</strong>
                        </label>
                        <textarea 
                            class="form-control" 
                            id="models" 
                            rows="8" 
                            placeholder="google/gemini-2.5-flash-lite&#10;qwen/qwen2.5-14b-instruct&#10;qwen/qwen3-235b-a22b-2507"
                            required
                        ></textarea>
                        <small class="form-text text-muted">
                            Введите модели из OpenRouter, по одной на строку
                        </small>
                    </div>

                    <div class="mb-3">
                        <label for="prompt" class="form-label">
                            <strong>Промпт запроса</strong>
                        </label>
                        <textarea 
                            class="form-control" 
                            id="prompt" 
                            rows="5" 
                            placeholder="Введите промпт для тестирования..."
                            required
                        ></textarea>
                    </div>

                    <div class="mb-3">
                        <label for="maxTokens" class="form-label">
                            <strong>Максимум токенов</strong>
                        </label>
                        <input 
                            type="number" 
                            class="form-control" 
                            id="maxTokens" 
                            value="2000" 
                            min="100" 
                            max="32000"
                        >
                    </div>

                    <div class="mb-3">
                        <label for="temperature" class="form-label">
                            <strong>Temperature</strong>
                        </label>
                        <input 
                            type="number" 
                            class="form-control" 
                            id="temperature" 
                            value="0.7" 
                            min="0" 
                            max="2" 
                            step="0.1"
                        >
                    </div>

                    <button type="submit" class="btn btn-primary" id="testBtn">
                        <i class="bi bi-play-fill"></i> Запустить тест
                    </button>
                    <button type="button" class="btn btn-secondary" id="clearBtn">
                        <i class="bi bi-trash"></i> Очистить результаты
                    </button>
                </form>
            </div>
        </div>

        <div id="stats-container" style="display: none;">
            <div class="stats">
                <strong>Статистика:</strong>
                <span id="stats-text"></span>
            </div>
        </div>

        <div id="results-container"></div>
    </div>

    <script>
        const form = document.getElementById('testForm');
        const resultsContainer = document.getElementById('results-container');
        const statsContainer = document.getElementById('stats-container');
        const statsText = document.getElementById('stats-text');
        const testBtn = document.getElementById('testBtn');
        const clearBtn = document.getElementById('clearBtn');
        const modelsInput = document.getElementById('models');
        const promptInput = document.getElementById('prompt');
        const maxTokensInput = document.getElementById('maxTokens');
        const temperatureInput = document.getElementById('temperature');

        let isRunning = false;

        // Загрузка сохраненных данных при загрузке страницы
        function loadSavedData() {
            const savedModels = localStorage.getItem('modelTester_models');
            const savedPrompt = localStorage.getItem('modelTester_prompt');
            const savedMaxTokens = localStorage.getItem('modelTester_maxTokens');
            const savedTemperature = localStorage.getItem('modelTester_temperature');

            if (savedModels) modelsInput.value = savedModels;
            if (savedPrompt) promptInput.value = savedPrompt;
            if (savedMaxTokens) maxTokensInput.value = savedMaxTokens;
            if (savedTemperature) temperatureInput.value = savedTemperature;
        }

        // Сохранение данных при изменении
        function saveData() {
            localStorage.setItem('modelTester_models', modelsInput.value);
            localStorage.setItem('modelTester_prompt', promptInput.value);
            localStorage.setItem('modelTester_maxTokens', maxTokensInput.value);
            localStorage.setItem('modelTester_temperature', temperatureInput.value);
        }

        // Загружаем сохраненные данные при загрузке страницы
        loadSavedData();

        // Сохраняем данные при изменении полей
        modelsInput.addEventListener('input', saveData);
        promptInput.addEventListener('input', saveData);
        maxTokensInput.addEventListener('input', saveData);
        temperatureInput.addEventListener('input', saveData);

        clearBtn.addEventListener('click', () => {
            resultsContainer.innerHTML = '';
            statsContainer.style.display = 'none';
        });

        form.addEventListener('submit', async (e) => {
            e.preventDefault();

            if (isRunning) {
                alert('Тест уже выполняется. Подождите...');
                return;
            }

            const modelsText = document.getElementById('models').value.trim();
            const prompt = document.getElementById('prompt').value.trim();
            const maxTokens = parseInt(document.getElementById('maxTokens').value) || 2000;
            const temperature = parseFloat(document.getElementById('temperature').value) || 0.7;

            if (!modelsText || !prompt) {
                alert('Заполните все поля');
                return;
            }

            const models = modelsText.split('\n')
                .map(m => m.trim())
                .filter(m => m.length > 0);

            if (models.length === 0) {
                alert('Введите хотя бы одну модель');
                return;
            }

            isRunning = true;
            testBtn.disabled = true;
            testBtn.innerHTML = '<span class="loading-spinner"></span> Тестирование...';
            resultsContainer.innerHTML = '';
            statsContainer.style.display = 'none';

            const startTime = Date.now();
            const results = [];
            let completed = 0;
            let successCount = 0;
            let errorCount = 0;

            // Создаем контейнеры для каждой модели
            models.forEach((model, index) => {
                const resultDiv = document.createElement('div');
                resultDiv.className = 'model-result loading';
                resultDiv.id = `result-${index}`;
                resultDiv.innerHTML = `
                    <div class="model-header">
                        <span class="model-name">${model}</span>
                        <span class="model-time">
                            <span class="loading-spinner"></span> Ожидание...
                        </span>
                    </div>
                    <div class="model-response" style="display: none;"></div>
                `;
                resultsContainer.appendChild(resultDiv);
            });

            // Запускаем запросы последовательно (по очереди)
            for (let index = 0; index < models.length; index++) {
                const model = models[index];
                const modelStartTime = Date.now();
                const resultDiv = document.getElementById(`result-${index}`);
                const timeSpan = resultDiv.querySelector('.model-time');
                const responseDiv = resultDiv.querySelector('.model-response');

                // Обновляем статус - начинаем запрос
                timeSpan.innerHTML = `<span class="loading-spinner"></span> Выполняется...`;

                try {
                    const response = await fetch('/api/model-test.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({
                            model: model,
                            prompt: prompt,
                            max_tokens: maxTokens,
                            temperature: temperature
                        })
                    });

                    const data = await response.json();
                    const modelTime = Date.now() - modelStartTime;

                    if (data.success) {
                        resultDiv.className = 'model-result success';
                        
                        // Формируем строку времени и стоимости
                        let timeHtml = `<i class="bi bi-check-circle text-success"></i> ${modelTime} мс`;
                        if (data.cost !== undefined && data.cost !== null) {
                            const cost = parseFloat(data.cost);
                            if (!isNaN(cost) && cost > 0) {
                                timeHtml += `<span class="model-cost"><i class="bi bi-currency-dollar"></i> $${cost.toFixed(6)}</span>`;
                            }
                        }
                        timeSpan.innerHTML = timeHtml;
                        
                        responseDiv.textContent = data.response || 'Пустой ответ';
                        responseDiv.style.display = 'block';
                        
                        results.push({
                            model: model,
                            success: true,
                            time: modelTime,
                            response: data.response,
                            tokens: data.tokens || {},
                            cost: data.cost || 0
                        });
                        successCount++;
                    } else {
                        throw new Error(data.error || 'Неизвестная ошибка');
                    }
                } catch (error) {
                    const modelTime = Date.now() - modelStartTime;
                    resultDiv.className = 'model-result error';
                    timeSpan.innerHTML = `<i class="bi bi-x-circle text-danger"></i> ${modelTime} мс`;
                    responseDiv.className = 'model-error';
                    responseDiv.textContent = `Ошибка: ${error.message}`;
                    responseDiv.style.display = 'block';
                    
                    results.push({
                        model: model,
                        success: false,
                        time: modelTime,
                        error: error.message
                    });
                    errorCount++;
                }

                completed++;
            }

            // Показываем статистику после завершения всех запросов
            const totalTime = Date.now() - startTime;
            const avgTime = results.reduce((sum, r) => sum + r.time, 0) / results.length;
            const minTime = Math.min(...results.map(r => r.time));
            const maxTime = Math.max(...results.map(r => r.time));
            
            // Подсчитываем общую стоимость
            const totalCost = results.reduce((sum, r) => sum + (r.cost || 0), 0);
            const avgCost = successCount > 0 ? totalCost / successCount : 0;

            let statsHtml = `
                Всего моделей: ${models.length} | 
                Успешно: <span class="text-success">${successCount}</span> | 
                Ошибок: <span class="text-danger">${errorCount}</span> | 
                Общее время: ${totalTime} мс | 
                Среднее время: ${Math.round(avgTime)} мс | 
                Мин: ${minTime} мс | 
                Макс: ${maxTime} мс
            `;
            
            if (totalCost > 0) {
                statsHtml += ` | 
                Общая стоимость: <span class="text-success">$${totalCost.toFixed(6)}</span> | 
                Средняя стоимость: <span class="text-success">$${avgCost.toFixed(6)}</span>
                `;
            }
            
            statsText.innerHTML = statsHtml;
            statsContainer.style.display = 'block';
            isRunning = false;
            testBtn.disabled = false;
            testBtn.innerHTML = '<i class="bi bi-play-fill"></i> Запустить тест';
        });
    </script>
</body>
</html>

