<?php
$page_title = 'Тест GigaChat API';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/includes/auth.php';

// Функция для получения Access Token с кэшированием
function getGigaChatAccessToken($forceRefresh = false) {
    $cacheFile = sys_get_temp_dir() . '/gigachat_token_' . md5(GIGACHAT_CLIENT_ID) . '.json';
    
    // Проверяем кэш если не требуется принудительное обновление
    if (!$forceRefresh && file_exists($cacheFile)) {
        $cacheData = json_decode(file_get_contents($cacheFile), true);
        if ($cacheData && isset($cacheData['token']) && isset($cacheData['expires_at'])) {
            // Проверяем, не истек ли токен (оставляем запас 60 секунд)
            if ($cacheData['expires_at'] > (time() + 60)) {
                return [
                    'success' => true,
                    'token' => $cacheData['token'],
                    'from_cache' => true
                ];
            }
        }
    }
    
    // Генерируем RqUID в формате UUID
    $rqUID = sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
        mt_rand(0, 0xffff), mt_rand(0, 0xffff),
        mt_rand(0, 0xffff),
        mt_rand(0, 0x0fff) | 0x4000,
        mt_rand(0, 0x3fff) | 0x8000,
        mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
    );
    
    // Формируем запрос на получение токена
    $ch = curl_init(GIGACHAT_OAUTH_URL);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query(['scope' => GIGACHAT_SCOPE]));
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Отключаем проверку SSL для самоподписанных сертификатов
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/x-www-form-urlencoded',
        'Accept: application/json',
        'RqUID: ' . $rqUID,
        'Authorization: Basic ' . GIGACHAT_AUTH_KEY
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    if ($curlError) {
        return [
            'success' => false,
            'error' => 'CURL error: ' . $curlError
        ];
    }
    
    if ($httpCode !== 200) {
        return [
            'success' => false,
            'error' => 'HTTP ' . $httpCode . ': ' . $response
        ];
    }
    
    $result = json_decode($response, true);
    
    if (!isset($result['access_token'])) {
        return [
            'success' => false,
            'error' => 'Invalid response: ' . $response
        ];
    }
    
    $token = $result['access_token'];
    
    // Обрабатываем expires_at (timestamp) или expires_in (секунды)
    if (isset($result['expires_at'])) {
        $expiresAt = is_numeric($result['expires_at']) ? (int)$result['expires_at'] : strtotime($result['expires_at']);
    } elseif (isset($result['expires_in'])) {
        $expiresAt = time() + (int)$result['expires_in'];
    } else {
        $expiresAt = time() + GIGACHAT_TOKEN_CACHE_TIME;
    }
    
    // Сохраняем в кэш
    $cacheData = [
        'token' => $token,
        'expires_at' => $expiresAt,
        'cached_at' => time()
    ];
    file_put_contents($cacheFile, json_encode($cacheData));
    
    return [
        'success' => true,
        'token' => $token,
        'from_cache' => false,
        'expires_at' => $expiresAt
    ];
}

// Функция для отправки запроса к GigaChat API
function callGigaChatAPI($messages, $model = null, $temperature = 0.7, $maxTokens = 4096) {
    $tokenResult = getGigaChatAccessToken();
    
    if (!$tokenResult['success']) {
        return [
            'success' => false,
            'error' => 'Failed to get access token: ' . ($tokenResult['error'] ?? 'Unknown error')
        ];
    }
    
    $accessToken = $tokenResult['token'];
    $useModel = $model ?? GIGACHAT_MODEL;
    
    $data = [
        'model' => $useModel,
        'messages' => $messages,
        'temperature' => $temperature,
        'max_tokens' => $maxTokens
    ];
    
    $startTime = microtime(true);
    
    $ch = curl_init(GIGACHAT_API_URL);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Отключаем проверку SSL для самоподписанных сертификатов
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json',
        'Authorization: Bearer ' . $accessToken
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    $responseTime = round((microtime(true) - $startTime) * 1000); // в миллисекундах
    curl_close($ch);
    
    if ($curlError) {
        return [
            'success' => false,
            'error' => 'CURL error: ' . $curlError,
            'response_time_ms' => $responseTime
        ];
    }
    
    $result = json_decode($response, true);
    
    if ($httpCode !== 200) {
        return [
            'success' => false,
            'error' => 'HTTP ' . $httpCode . ': ' . $response,
            'http_code' => $httpCode,
            'response_time_ms' => $responseTime,
            'raw_response' => $response
        ];
    }
    
    if (!isset($result['choices'][0]['message']['content'])) {
        return [
            'success' => false,
            'error' => 'Invalid response format: ' . $response,
            'raw_response' => $response
        ];
    }
    
    return [
        'success' => true,
        'content' => $result['choices'][0]['message']['content'],
        'model' => $result['model'] ?? $useModel,
        'usage' => $result['usage'] ?? null,
        'response_time_ms' => $responseTime,
        'raw_response' => $response,
        'token_from_cache' => $tokenResult['from_cache'] ?? false
    ];
}

// Обработка AJAX запросов
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    if ($_POST['action'] === 'get_token') {
        $forceRefresh = isset($_POST['force_refresh']) && $_POST['force_refresh'] === 'true';
        $result = getGigaChatAccessToken($forceRefresh);
        echo json_encode($result);
        exit;
    }
    
    if ($_POST['action'] === 'test_request') {
        $question = $_POST['question'] ?? '';
        $systemPrompt = $_POST['system_prompt'] ?? 'Ты - помощник. Отвечай на вопросы пользователей кратко и по делу.';
        
        if (empty($question)) {
            echo json_encode([
                'success' => false,
                'error' => 'Вопрос не может быть пустым'
            ]);
            exit;
        }
        
        $messages = [
            [
                'role' => 'system',
                'content' => $systemPrompt
            ],
            [
                'role' => 'user',
                'content' => $question
            ]
        ];
        
        $result = callGigaChatAPI($messages);
        echo json_encode($result);
        exit;
    }
    
    echo json_encode(['success' => false, 'error' => 'Unknown action']);
    exit;
}
?>
<!DOCTYPE html>
<html lang="ru" data-bs-theme="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($page_title); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        pre {
            white-space: pre-wrap;
            word-wrap: break-word;
        }
        .code-block {
            background-color: #1e1e1e;
            color: #d4d4d4;
            padding: 15px;
            border-radius: 5px;
            border: 1px solid #3e3e3e;
            max-height: 400px;
            overflow-y: auto;
            font-size: 12px;
        }
    </style>
</head>
<body class="p-4">
    <div class="container-fluid">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="mb-0">Тест GigaChat API</h1>
            <div class="btn-group" role="group">
                <button class="btn btn-warning" id="refresh-token-btn">
                    <i class="bi bi-arrow-clockwise"></i> Обновить токен
                </button>
                <button class="btn btn-secondary" onclick="window.close()">Закрыть</button>
            </div>
        </div>
        
        <div class="alert alert-info">
            <div class="row">
                <div class="col-md-3">
                    <strong>Модель:</strong> <?php echo htmlspecialchars(GIGACHAT_MODEL); ?>
                </div>
                <div class="col-md-3">
                    <strong>API URL:</strong> <code><?php echo htmlspecialchars(GIGACHAT_API_URL); ?></code>
                </div>
                <div class="col-md-3">
                    <strong>OAuth URL:</strong> <code><?php echo htmlspecialchars(GIGACHAT_OAUTH_URL); ?></code>
                </div>
                <div class="col-md-3">
                    <strong>Client ID:</strong> <code><?php echo htmlspecialchars(substr(GIGACHAT_CLIENT_ID, 0, 20)) . '...'; ?></code>
                </div>
            </div>
        </div>
        
        <!-- Форма тестирования -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Тестовый запрос</h5>
            </div>
            <div class="card-body">
                <form id="test-form">
                    <div class="mb-3">
                        <label class="form-label">System Prompt:</label>
                        <textarea class="form-control" id="system-prompt" rows="2" 
                                  placeholder="Ты - помощник. Отвечай на вопросы пользователей кратко и по делу.">Ты - помощник. Отвечай на вопросы пользователей кратко и по делу.</textarea>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Вопрос пользователя:</label>
                        <textarea class="form-control" id="test-question" rows="3" 
                                  placeholder="Введите ваш вопрос...">Привет! Как дела?</textarea>
                    </div>
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-send"></i> Отправить запрос
                    </button>
                </form>
            </div>
        </div>
        
        <!-- Результаты -->
        <div id="test-results" style="display: none;" class="mt-4">
            <!-- Информация о токене -->
            <div class="card mb-3">
                <div class="card-header">
                    <h6 class="mb-0">Информация о токене</h6>
                </div>
                <div class="card-body">
                    <div id="token-info" class="text-muted">Загрузка...</div>
                </div>
            </div>
            
            <!-- Запрос -->
            <div class="card mb-3">
                <div class="card-header">
                    <h6 class="mb-0">Запрос к API</h6>
                </div>
                <div class="card-body">
                    <div class="mb-2">
                        <strong>URL:</strong> <code id="request-url"></code>
                    </div>
                    <div class="mb-2">
                        <strong>Headers:</strong>
                        <pre id="request-headers" class="code-block mt-2"></pre>
                    </div>
                    <div class="mb-2">
                        <strong>Body:</strong>
                        <pre id="request-body" class="code-block mt-2"></pre>
                    </div>
                </div>
            </div>
            
            <!-- Ответ -->
            <div class="card mb-3">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h6 class="mb-0">Ответ от API</h6>
                    <span id="response-status" class="badge"></span>
                </div>
                <div class="card-body">
                    <div class="mb-2">
                        <strong>HTTP Code:</strong> <span id="response-http-code" class="badge"></span>
                    </div>
                    <div class="mb-2">
                        <strong>Время выполнения:</strong> <span id="response-time" class="text-success"></span> мс
                    </div>
                    <div class="mb-2">
                        <strong>Использование токенов:</strong>
                        <div id="token-usage" class="mt-2"></div>
                    </div>
                    <div class="mb-2">
                        <strong>Ответ:</strong>
                        <div id="response-content" class="bg-success bg-opacity-25 p-3 mt-2 rounded" style="white-space: pre-wrap;"></div>
                    </div>
                    <div class="mb-2">
                        <strong>Сырой ответ (JSON):</strong>
                        <pre id="response-raw" class="code-block mt-2"></pre>
                    </div>
                </div>
            </div>
            
            <!-- Ошибки -->
            <div id="error-section" class="card mb-3" style="display: none;">
                <div class="card-header bg-danger text-white">
                    <h6 class="mb-0">Ошибка</h6>
                </div>
                <div class="card-body">
                    <div id="error-message" class="text-danger"></div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
    // Обновление токена
    document.getElementById('refresh-token-btn').addEventListener('click', async () => {
        const btn = document.getElementById('refresh-token-btn');
        const originalHTML = btn.innerHTML;
        btn.disabled = true;
        btn.innerHTML = '<i class="bi bi-hourglass-split"></i> Обновление...';
        
        try {
            const formData = new FormData();
            formData.append('action', 'get_token');
            formData.append('force_refresh', 'true');
            
            const response = await fetch(window.location.href, {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            
            if (data.success) {
                alert('Токен успешно обновлен!');
                updateTokenInfo(data);
            } else {
                alert('Ошибка обновления токена: ' + (data.error || 'Неизвестная ошибка'));
            }
        } catch (error) {
            alert('Ошибка: ' + error.message);
        } finally {
            btn.disabled = false;
            btn.innerHTML = originalHTML;
        }
    });
    
    // Обновление информации о токене
    function updateTokenInfo(tokenData) {
        const tokenInfo = document.getElementById('token-info');
        if (tokenData.from_cache) {
            tokenInfo.innerHTML = '<span class="text-success"><i class="bi bi-check-circle"></i> Токен загружен из кэша</span>';
        } else {
            tokenInfo.innerHTML = '<span class="text-info"><i class="bi bi-info-circle"></i> Токен получен заново</span>';
        }
        if (tokenData.expires_at) {
            const expiresDate = new Date(tokenData.expires_at * 1000);
            tokenInfo.innerHTML += '<br><small>Истекает: ' + expiresDate.toLocaleString('ru-RU') + '</small>';
        }
    }
    
    // Обработка формы
    document.getElementById('test-form').addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const question = document.getElementById('test-question').value;
        const systemPrompt = document.getElementById('system-prompt').value;
        const resultsDiv = document.getElementById('test-results');
        
        // Показываем секцию результатов
        resultsDiv.style.display = 'block';
        
        // Показываем загрузку
        document.getElementById('response-content').textContent = 'Загрузка...';
        document.getElementById('response-raw').textContent = 'Загрузка...';
        document.getElementById('error-section').style.display = 'none';
        
        // Формируем данные запроса для отображения
        const requestData = {
            model: '<?php echo htmlspecialchars(GIGACHAT_MODEL); ?>',
            messages: [
                { role: 'system', content: systemPrompt },
                { role: 'user', content: question }
            ],
            temperature: 0.7,
            max_tokens: 4096
        };
        
        // Отображаем запрос
        document.getElementById('request-url').textContent = '<?php echo htmlspecialchars(GIGACHAT_API_URL); ?>';
        document.getElementById('request-headers').textContent = JSON.stringify({
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'Authorization': 'Bearer <токен>'
        }, null, 2);
        document.getElementById('request-body').textContent = JSON.stringify(requestData, null, 2);
        
        try {
            // Сначала получаем токен
            const tokenFormData = new FormData();
            tokenFormData.append('action', 'get_token');
            
            const tokenResponse = await fetch(window.location.href, {
                method: 'POST',
                body: tokenFormData
            });
            
            const tokenData = await tokenResponse.json();
            
            if (!tokenData.success) {
                throw new Error('Не удалось получить токен: ' + (tokenData.error || 'Неизвестная ошибка'));
            }
            
            updateTokenInfo(tokenData);
            
            // Отправляем тестовый запрос
            const formData = new FormData();
            formData.append('action', 'test_request');
            formData.append('question', question);
            formData.append('system_prompt', systemPrompt);
            
            const startTime = Date.now();
            const response = await fetch(window.location.href, {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            
            if (!data.success) {
                // Показываем ошибку
                document.getElementById('error-section').style.display = 'block';
                document.getElementById('error-message').textContent = data.error || 'Неизвестная ошибка';
                
                if (data.http_code) {
                    document.getElementById('response-http-code').textContent = data.http_code;
                    document.getElementById('response-http-code').className = 'badge bg-danger';
                }
                
                if (data.raw_response) {
                    document.getElementById('response-raw').textContent = data.raw_response;
                }
                
                return;
            }
            
            // Успешный ответ
            document.getElementById('response-status').textContent = 'Успешно';
            document.getElementById('response-status').className = 'badge bg-success';
            document.getElementById('response-http-code').textContent = '200';
            document.getElementById('response-http-code').className = 'badge bg-success';
            document.getElementById('response-time').textContent = data.response_time_ms || 0;
            document.getElementById('response-content').textContent = data.content || 'Нет ответа';
            document.getElementById('response-raw').textContent = JSON.stringify(JSON.parse(data.raw_response), null, 2);
            
            // Показываем использование токенов
            if (data.usage) {
                const usageHtml = '<div>Входные токены: ' + (data.usage.prompt_tokens || 0) + '</div>' +
                                 '<div>Выходные токены: ' + (data.usage.completion_tokens || 0) + '</div>' +
                                 '<div>Всего токенов: ' + (data.usage.total_tokens || 0) + '</div>';
                document.getElementById('token-usage').innerHTML = usageHtml;
            } else {
                document.getElementById('token-usage').innerHTML = '<span class="text-muted">Информация недоступна</span>';
            }
            
        } catch (error) {
            document.getElementById('error-section').style.display = 'block';
            document.getElementById('error-message').textContent = 'Ошибка: ' + error.message;
            console.error('Error:', error);
        }
    });
    
    // Загружаем информацию о токене при загрузке страницы
    window.addEventListener('load', async () => {
        try {
            const formData = new FormData();
            formData.append('action', 'get_token');
            
            const response = await fetch(window.location.href, {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            if (data.success) {
                updateTokenInfo(data);
            }
        } catch (error) {
            console.error('Failed to load token info:', error);
        }
    });
    </script>
</body>
</html>

