<?php
$page_title = 'Тест кастомного API (полный цикл)';
require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/../config.php';

// Получаем список виджетов
$db = getDatabase();
$widgets = $db->query("
    SELECT id, name, widget_key, site_domain
    FROM widgets
    WHERE active = 1
    ORDER BY name ASC
")->fetchAll();

$widget_id = $_GET['widget_id'] ?? ($widgets[0]['id'] ?? 0);

// Получаем выбранный виджет
$widget = null;
if ($widget_id) {
    $stmt = $db->prepare("
        SELECT w.*, ws.*
        FROM widgets w
        LEFT JOIN widget_settings ws ON w.id = ws.widget_id
        WHERE w.id = ?
    ");
    $stmt->execute([$widget_id]);
    $widget = $stmt->fetch();
}

if ($widget) {
    $stage1_model = $widget['stage1_model'] ?? 'custom-model';
    $stage2_model = $widget['stage2_model'] ?? 'custom-model';
    $stage3_model = $widget['stage3_model'] ?? 'custom-model';
    $stage1_prompt = $widget['stage1_prompt'] ?? 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей кратко и по делу (2-3 предложения).';
    $stage2_prompt = $widget['stage2_prompt'] ?? null;
    $stage3_prompt = $widget['stage3_prompt'] ?? null;
} else {
    $stage1_model = 'custom-model';
    $stage2_model = 'custom-model';
    $stage3_model = 'custom-model';
    $stage1_prompt = 'Ты - помощник медицинской клиники. Отвечай на вопросы пользователей кратко и по делу (2-3 предложения).';
    $stage2_prompt = null;
    $stage3_prompt = null;
}
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
    <h1 class="h2"><i class="bi bi-cloud-upload"></i> Тест кастомного API (полный цикл)</h1>
</div>

<div class="row">
    <div class="col-12">
        <div class="alert alert-info">
            <i class="bi bi-info-circle"></i> 
            <strong>Инструкция:</strong> Эта страница позволяет протестировать ваш кастомный API с полным циклом запросов (как в виджете).
            Выберите виджет, укажите URL вашего API и отправьте тестовый запрос.
        </div>

        <div class="card bg-dark border-secondary mb-4">
            <div class="card-header bg-secondary">
                <h5 class="mb-0"><i class="bi bi-gear"></i> Настройки</h5>
            </div>
            <div class="card-body">
                <form id="apiConfigForm">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="widgetSelect" class="form-label">
                                <strong>Виджет</strong> <span class="text-danger">*</span>
                            </label>
                            <select class="form-select" id="widgetSelect" required>
                                <option value="">Выберите виджет...</option>
                                <?php foreach ($widgets as $w): ?>
                                    <option value="<?php echo $w['id']; ?>" <?php echo $w['id'] == $widget_id ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($w['name']); ?> (<?php echo htmlspecialchars($w['widget_key']); ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label for="apiUrl" class="form-label">
                                <strong>URL API</strong> <span class="text-danger">*</span>
                            </label>
                            <input 
                                type="url" 
                                class="form-control" 
                                id="apiUrl" 
                                placeholder="https://your-server.com/api/v1/chat/completions"
                                required
                            >
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="apiKey" class="form-label">
                                <strong>API ключ</strong>
                            </label>
                            <input 
                                type="text" 
                                class="form-control" 
                                id="apiKey" 
                                placeholder="Bearer YOUR_API_KEY или просто ключ"
                            >
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label for="testQuestion" class="form-label">
                                <strong>Тестовый вопрос</strong> <span class="text-danger">*</span>
                            </label>
                            <input 
                                type="text" 
                                class="form-control" 
                                id="testQuestion" 
                                placeholder="У меня болит спина"
                                value="У меня болит спина"
                                required
                            >
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label for="maxTokensStage1" class="form-label">
                                <strong>Max Tokens (Этап 1)</strong>
                            </label>
                            <input 
                                type="number" 
                                class="form-control" 
                                id="maxTokensStage1" 
                                value="200"
                                min="50"
                                max="2048"
                            >
                        </div>
                        <div class="col-md-4 mb-3">
                            <label for="maxTokensStage2" class="form-label">
                                <strong>Max Tokens (Этап 2)</strong>
                            </label>
                            <input 
                                type="number" 
                                class="form-control" 
                                id="maxTokensStage2" 
                                value="1000"
                                min="100"
                                max="2048"
                            >
                        </div>
                        <div class="col-md-4 mb-3">
                            <label for="maxTokensStage3" class="form-label">
                                <strong>Max Tokens (Этап 3)</strong>
                            </label>
                            <input 
                                type="number" 
                                class="form-control" 
                                id="maxTokensStage3" 
                                value="2048"
                                min="100"
                                max="2048"
                            >
                            <small class="form-text text-muted">
                                Максимум 2048 (ограничение API)
                            </small>
                        </div>
                    </div>
                    
                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary" id="testBtn">
                            <i class="bi bi-play-fill"></i> Запустить полный тест
                        </button>
                        <button type="button" class="btn btn-secondary" id="clearBtn">
                            <i class="bi bi-trash"></i> Очистить результаты
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <div id="test-results" style="display: none;">
            <!-- Результаты будут добавлены сюда динамически -->
        </div>
        
        <!-- Шаблоны для этапов (скрыты) -->
        <div id="templates" style="display: none;">
            <!-- Этап 1: Быстрый текстовый ответ -->
            <div id="template-stage1" class="mb-3">
                <h6><i class="bi bi-1-circle"></i> Этап 1: Быстрый текстовый ответ</h6>
                <div class="card bg-dark">
                    <div class="card-body">
                        <div class="mb-3">
                            <strong>Модель:</strong> <span class="text-info template-stage1-model"></span>
                        </div>
                        <div class="mb-3">
                            <strong>Запрос:</strong>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                        <strong>System Prompt:</strong>
                                        <pre class="template-stage1-system-prompt mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap;"></pre>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                        <strong>User Prompt:</strong>
                                        <pre class="template-stage1-user-prompt mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap;"></pre>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="mb-3">
                            <strong>Ответ:</strong>
                            <div class="template-stage1-text bg-success bg-opacity-25 p-2 mt-2 mb-0" style="max-height: 200px; overflow-y: auto; font-size: 12px; white-space: pre-wrap; border-left: 3px solid #198754;"></div>
                        </div>
                        <div class="mb-2">
                            <strong>Время выполнения:</strong> <span class="template-stage1-time text-success"></span> мс
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Этап 2: Извлечение медицинских терминов -->
            <div id="template-stage2" class="mb-3">
                <h6><i class="bi bi-2-circle"></i> Этап 2: Извлечение медицинских терминов</h6>
                <div class="card bg-dark">
                    <div class="card-body">
                        <div class="mb-3">
                            <strong>Модель:</strong> <span class="text-info template-stage2-model"></span>
                        </div>
                        <div class="mb-3">
                            <strong>Запрос:</strong>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                        <strong>System Prompt:</strong>
                                        <pre class="template-stage2-system-prompt mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 200px; overflow-y: auto;"></pre>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                        <strong>User Prompt:</strong>
                                        <pre class="template-stage2-user-prompt mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 200px; overflow-y: auto;"></pre>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="mb-3">
                            <strong>Ответ:</strong>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="bg-success bg-opacity-25 p-2 mt-2" style="font-size: 12px; border-left: 3px solid #198754;">
                                        <strong>Извлеченные ключевые слова:</strong>
                                        <pre class="template-stage2-keywords mb-0 mt-1" style="font-size: 11px; max-height: 150px; overflow-y: auto;"></pre>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="bg-secondary p-2 mt-2" style="font-size: 12px;">
                                        <strong>Полный JSON ответ:</strong>
                                        <pre class="template-stage2-full-response mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 200px; overflow-y: auto;"></pre>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="mb-2">
                            <strong>Время выполнения:</strong> <span class="template-stage2-time text-success"></span> мс
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Этап 3: Основной запрос с данными -->
            <div id="template-stage3" class="mb-3">
                <h6><i class="bi bi-3-circle"></i> Этап 3: Основной запрос с данными</h6>
                <div class="card bg-dark">
                    <div class="card-body">
                        <div class="mb-3">
                            <strong>Модель:</strong> <span class="text-info template-stage3-model"></span>
                        </div>
                        <div class="mb-3">
                            <strong>Оптимизация промпта:</strong>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="bg-warning bg-opacity-25 p-2 mt-2" style="font-size: 12px; border-left: 3px solid #ffc107;">
                                        <strong>До оптимизации:</strong>
                                        <div class="mt-1">Символов: <span class="template-stage3-size-before text-warning"></span></div>
                                        <div>Токенов: <span class="template-stage3-tokens-before text-warning"></span></div>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="bg-success bg-opacity-25 p-2 mt-2" style="font-size: 12px; border-left: 3px solid #198754;">
                                        <strong>После оптимизации:</strong>
                                        <div class="mt-1">Символов: <span class="template-stage3-size-after text-success"></span></div>
                                        <div>Токенов: <span class="template-stage3-tokens-after text-success"></span></div>
                                        <div class="mt-1"><strong>Сокращение:</strong> <span class="template-stage3-reduction text-success fw-bold"></span>%</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="mb-3">
                            <strong>Ответ:</strong>
                            <div class="bg-success bg-opacity-25 p-2 mt-2" style="font-size: 12px; border-left: 3px solid #198754;">
                                <pre class="template-stage3-response mb-0 mt-1" style="font-size: 11px; white-space: pre-wrap; max-height: 300px; overflow-y: auto;"></pre>
                            </div>
                        </div>
                        <div class="mb-2">
                            <strong>Время выполнения:</strong> <span class="template-stage3-time text-success"></span> мс
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
const form = document.getElementById('apiConfigForm');
const resultsDiv = document.getElementById('test-results');
const testBtn = document.getElementById('testBtn');
const clearBtn = document.getElementById('clearBtn');

let isRunning = false;

clearBtn.addEventListener('click', () => {
    resultsDiv.style.display = 'none';
    resultsDiv.innerHTML = '';
});

form.addEventListener('submit', async (e) => {
    e.preventDefault();

    if (isRunning) {
        alert('Тест уже выполняется. Подождите...');
        return;
    }

    const widgetId = document.getElementById('widgetSelect').value;
    const apiUrl = document.getElementById('apiUrl').value.trim();
    const apiKey = document.getElementById('apiKey').value.trim();
    const question = document.getElementById('testQuestion').value.trim();
    const maxTokensStage1 = parseInt(document.getElementById('maxTokensStage1').value) || 200;
    const maxTokensStage2 = parseInt(document.getElementById('maxTokensStage2').value) || 1000;
    const maxTokensStage3 = parseInt(document.getElementById('maxTokensStage3').value) || 2048;

    if (!widgetId || !apiUrl || !question) {
        alert('Заполните все обязательные поля');
        return;
    }

    // Проверяем ограничение max_tokens
    if (maxTokensStage1 > 2048 || maxTokensStage2 > 2048 || maxTokensStage3 > 2048) {
        alert('Max Tokens не может быть больше 2048 (ограничение API)');
        return;
    }

    // Получаем widget_key
    const widgetSelect = document.getElementById('widgetSelect');
    const selectedOption = widgetSelect.options[widgetSelect.selectedIndex];
    const widgetKey = selectedOption.text.match(/\(([^)]+)\)/)?.[1] || '';

    isRunning = true;
    testBtn.disabled = true;
    testBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Выполняется...';

    resultsDiv.style.display = 'block';
    resultsDiv.innerHTML = '<div class="alert alert-info">Выполняется тест...</div>';

    // Этапы будут созданы динамически

    try {
        const response = await fetch('/api/custom-api-query.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({
                widget_key: widgetKey,
                question: question,
                api_url: apiUrl,
                api_key: apiKey,
                max_tokens_stage1: maxTokensStage1,
                max_tokens_stage2: maxTokensStage2,
                max_tokens_stage3: maxTokensStage3,
                _debug: true
            })
        });

        const data = await response.json();

        if (!data.success) {
            throw new Error(data.error || 'Неизвестная ошибка');
        }

        // Показываем результаты
        resultsDiv.innerHTML = '';

        // Показываем URL API
        if (data.debug?.three_stage_search?.api_url) {
            const apiUrlDiv = document.createElement('div');
            apiUrlDiv.className = 'alert alert-info mb-3';
            apiUrlDiv.innerHTML = `<strong>API URL:</strong> <code>${data.debug.three_stage_search.api_url}</code>`;
            resultsDiv.appendChild(apiUrlDiv);
        }
        
        // Этап 1
        if (data.debug?.three_stage_search?.stage1) {
            const stage1 = data.debug.three_stage_search.stage1;
            const templateEl = document.getElementById('template-stage1');
            if (templateEl) {
            const template = templateEl.cloneNode(true);
            template.id = 'three-stage-stage1';
            template.style.display = 'block';
            const userMessage = stage1.request?.messages?.[0]?.content || '';
            const parts = userMessage.split('\n\n');
            
            const modelEl = template.querySelector('.template-stage1-model');
            const systemPromptEl = template.querySelector('.template-stage1-system-prompt');
            const userPromptEl = template.querySelector('.template-stage1-user-prompt');
            const textEl = template.querySelector('.template-stage1-text');
            const timeEl = template.querySelector('.template-stage1-time');
            
            if (modelEl) modelEl.textContent = '<?php echo htmlspecialchars($stage1_model); ?>';
            if (systemPromptEl) systemPromptEl.textContent = parts[0] || '<?php echo htmlspecialchars($stage1_prompt); ?>';
            if (userPromptEl) userPromptEl.textContent = parts[1] || (stage1.request?.messages?.[0]?.content || 'Вопрос пользователя: ' + question);
            if (textEl) textEl.textContent = stage1.response || 'Нет ответа';
            if (timeEl) timeEl.textContent = stage1.time_ms || 0;
            
            resultsDiv.appendChild(template);
            } else {
                console.error('Template stage1 not found');
            }
        }

        // Этап 2
        if (data.debug?.three_stage_search?.stage2) {
            const stage2 = data.debug.three_stage_search.stage2;
            const templateEl = document.getElementById('template-stage2');
            if (templateEl) {
            const template = templateEl.cloneNode(true);
            template.id = 'three-stage-stage2';
            template.style.display = 'block';
            
            const modelEl = template.querySelector('.template-stage2-model');
            const systemPromptEl = template.querySelector('.template-stage2-system-prompt');
            const userPromptEl = template.querySelector('.template-stage2-user-prompt');
            const keywordsEl = template.querySelector('.template-stage2-keywords');
            const fullResponseEl = template.querySelector('.template-stage2-full-response');
            const timeEl = template.querySelector('.template-stage2-time');
            
            if (modelEl) modelEl.textContent = '<?php echo htmlspecialchars($stage2_model); ?>';
            if (systemPromptEl) systemPromptEl.textContent = '';
            if (userPromptEl) userPromptEl.textContent = stage2.request?.messages?.[0]?.content || '';
            
            // Преобразуем all_extracted_terms в массив, если это объект
            let extractedTerms = stage2.all_extracted_terms || [];
            if (typeof extractedTerms === 'object' && !Array.isArray(extractedTerms)) {
                extractedTerms = Object.values(extractedTerms);
            }
            if (keywordsEl) keywordsEl.textContent = JSON.stringify(extractedTerms, null, 2);
            if (fullResponseEl) fullResponseEl.textContent = JSON.stringify(stage2.medical_terms || {}, null, 2);
            if (timeEl) timeEl.textContent = stage2.time_ms || 0;
            
            resultsDiv.appendChild(template);
            } else {
                console.error('Template stage2 not found');
            }
        }

        // Этап 3
        if (data.debug?.three_stage_search?.stage3) {
            const stage3 = data.debug.three_stage_search.stage3;
            const templateEl = document.getElementById('template-stage3');
            if (templateEl) {
            const template = templateEl.cloneNode(true);
            template.id = 'three-stage-stage3';
            template.style.display = 'block';
            
            const modelEl = template.querySelector('.template-stage3-model');
            const sizeBeforeEl = template.querySelector('.template-stage3-size-before');
            const tokensBeforeEl = template.querySelector('.template-stage3-tokens-before');
            const sizeAfterEl = template.querySelector('.template-stage3-size-after');
            const tokensAfterEl = template.querySelector('.template-stage3-tokens-after');
            const reductionEl = template.querySelector('.template-stage3-reduction');
            const responseEl = template.querySelector('.template-stage3-response');
            const timeEl = template.querySelector('.template-stage3-time');
            
            if (modelEl) modelEl.textContent = '<?php echo htmlspecialchars($stage3_model); ?>';
            if (sizeBeforeEl) sizeBeforeEl.textContent = (stage3.prompt_size_before || 0).toLocaleString();
            if (tokensBeforeEl) tokensBeforeEl.textContent = (stage3.prompt_tokens_before || 0).toLocaleString();
            if (sizeAfterEl) sizeAfterEl.textContent = (stage3.prompt_size_after || 0).toLocaleString();
            if (tokensAfterEl) tokensAfterEl.textContent = (stage3.prompt_tokens_after || 0).toLocaleString();
            if (reductionEl) reductionEl.textContent = (stage3.reduction_percent || 0);
            
            // Извлекаем текст из JSON, если ответ в формате markdown code block
            let stage3ResponseText = stage3.response || 'Нет ответа';
            if (stage3ResponseText.includes('```json')) {
                const jsonMatch = stage3ResponseText.match(/```json\s*([\s\S]*?)\s*```/);
                if (jsonMatch) {
                    try {
                        const jsonData = JSON.parse(jsonMatch[1]);
                        stage3ResponseText = jsonData.text || jsonMatch[1];
                    } catch (e) {
                        // Если не удалось распарсить, оставляем как есть
                    }
                }
            } else if (stage3ResponseText.includes('```')) {
                const jsonMatch = stage3ResponseText.match(/```\s*([\s\S]*?)\s*```/);
                if (jsonMatch) {
                    try {
                        const jsonData = JSON.parse(jsonMatch[1]);
                        stage3ResponseText = jsonData.text || jsonMatch[1];
                    } catch (e) {
                        // Если не удалось распарсить, оставляем как есть
                    }
                }
            }
            
            if (responseEl) responseEl.textContent = stage3ResponseText;
            if (timeEl) timeEl.textContent = stage3.time_ms || 0;
            
            resultsDiv.appendChild(template);
            } else {
                console.error('Template stage3 not found');
            }
        }

        // Сырой ответ
        const rawResponseDiv = document.createElement('div');
        rawResponseDiv.className = 'mb-3';
        rawResponseDiv.innerHTML = '<h6>Сырой ответ API:</h6><pre class="bg-dark text-light p-3 border border-secondary" style="max-height: 400px; overflow-y: auto; font-size: 12px;"></pre>';
        rawResponseDiv.querySelector('pre').textContent = JSON.stringify(data, null, 2);
        resultsDiv.appendChild(rawResponseDiv);

    } catch (error) {
        resultsDiv.innerHTML = `
            <div class="alert alert-danger">
                <strong>Ошибка:</strong> ${error.message}
            </div>
        `;
    } finally {
        isRunning = false;
        testBtn.disabled = false;
        testBtn.innerHTML = '<i class="bi bi-play-fill"></i> Запустить полный тест';
    }
});
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
