<?php
/**
 * AJAX endpoint для пакетной обработки связывания специалистов с услугами
 */

session_start();

// Проверка авторизации
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    http_response_code(401);
    die(json_encode(['error' => 'Unauthorized']));
}

require_once __DIR__ . '/../../config.php';

header('Content-Type: application/json');

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    $widgetId = isset($input['widget_id']) ? (int)$input['widget_id'] : 0;
    $batchSize = isset($input['batch_size']) ? (int)$input['batch_size'] : 10;
    $offset = isset($input['offset']) ? (int)$input['offset'] : 0;
    
    if (!$widgetId) {
        throw new Exception('widget_id is required');
    }
    
    if ($batchSize < 1 || $batchSize > 50) {
        throw new Exception('batch_size must be between 1 and 50');
    }
    
    $db = getDatabase();
    
    // Получаем специалистов для обработки
    $stmt = $db->prepare("
        SELECT pi.id, 
               MAX(CASE WHEN pf.field_name = 'name' THEN pf.field_value END) as name,
               MAX(CASE WHEN pf.field_name = 'description' THEN pf.field_value END) as description
        FROM parsed_items pi
        LEFT JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.widget_id = ? 
          AND pi.section_name = 'specialists'
          AND pi.is_duplicate = 0
        GROUP BY pi.id
        ORDER BY pi.id
        LIMIT ? OFFSET ?
    ");
    $stmt->execute([$widgetId, $batchSize, $offset]);
    $specialists = $stmt->fetchAll();
    
    if (empty($specialists)) {
        echo json_encode([
            'success' => true,
            'processed' => 0,
            'total' => 0,
            'log' => 'No specialists found'
        ]);
        exit;
    }
    
    // Получаем все услуги консультаций (кэшируем для всех специалистов)
    $stmt = $db->prepare("
        SELECT pi.id, pf.field_value as name
        FROM parsed_items pi
        JOIN parsed_fields pf ON pi.id = pf.item_id
        WHERE pi.widget_id = ? 
          AND pi.section_name = 'services'
          AND pi.is_duplicate = 0
          AND pf.field_name = 'name'
          AND (
            pf.field_value LIKE '%консультация%'
            OR pf.field_value LIKE '%прием%'
            OR pf.field_value LIKE '%осмотр%'
          )
        ORDER BY pi.id
    ");
    $stmt->execute([$widgetId]);
    $allServices = $stmt->fetchAll();
    
    if (empty($allServices)) {
        throw new Exception('No consultation services found');
    }
    
    $processed = 0;
    $errors = 0;
    $logMessages = [];
    
    // Обрабатываем каждого специалиста
    foreach ($specialists as $specialist) {
        try {
            $specialistId = $specialist['id'];
            $specialistName = $specialist['name'] ?? '';
            $specialistDesc = $specialist['description'] ?? '';
            
            if (empty($specialistName)) {
                $logMessages[] = "Skipping specialist ID $specialistId: no name";
                continue;
            }
            
            // Сначала делаем предварительный поиск по именам и специализациям
            $preFilteredServices = [];
            $nameParts = explode(' ', $specialistName);
            $lastName = !empty($nameParts) ? $nameParts[0] : '';
            
            // Извлекаем специализацию из описания
            $specialization = '';
            if (!empty($specialistDesc)) {
                $descLower = mb_strtolower($specialistDesc);
                if (stripos($descLower, 'невролог') !== false) {
                    $specialization = 'невролог';
                } elseif (stripos($descLower, 'травматолог') !== false || stripos($descLower, 'ортопед') !== false) {
                    $specialization = 'травматолог';
                } elseif (stripos($descLower, 'хирург') !== false) {
                    $specialization = 'хирург';
                } elseif (stripos($descLower, 'терапевт') !== false) {
                    $specialization = 'терапевт';
                } elseif (stripos($descLower, 'кардиолог') !== false) {
                    $specialization = 'кардиолог';
                } elseif (stripos($descLower, 'гинеколог') !== false) {
                    $specialization = 'гинеколог';
                } elseif (stripos($descLower, 'уролог') !== false) {
                    $specialization = 'уролог';
                } elseif (stripos($descLower, 'дерматолог') !== false) {
                    $specialization = 'дерматолог';
                } elseif (stripos($descLower, 'лор') !== false || stripos($descLower, 'отоларинголог') !== false) {
                    $specialization = 'лор';
                }
            }
            
            // Фильтруем услуги: сначала по имени, потом по специализации
            foreach ($allServices as $service) {
                $serviceName = mb_strtolower($service['name']);
                $matches = false;
                
                // Проверяем по имени специалиста
                if (!empty($lastName) && stripos($serviceName, $lastName) !== false) {
                    $matches = true;
                }
                
                // Проверяем по специализации
                if (!$matches && !empty($specialization)) {
                    if (stripos($serviceName, $specialization) !== false) {
                        $matches = true;
                    }
                }
                
                if ($matches) {
                    $preFilteredServices[] = $service;
                }
            }
            
            // Если нашли услуги по предварительному фильтру - используем их
            // Иначе используем ВСЕ услуги
            $servicesForPrompt = !empty($preFilteredServices) ? $preFilteredServices : $allServices;
            
            // Формируем промпт для AI
            $prompt = "Определи, какие услуги консультаций и приема относятся к этому специалисту.\n\n";
            $prompt .= "Специалист:\n";
            $prompt .= "- Имя: $specialistName\n";
            if (!empty($specialistDesc)) {
                $prompt .= "- Описание: $specialistDesc\n";
            }
            if (!empty($specialization)) {
                $prompt .= "- Специализация: $specialization\n";
            }
            $prompt .= "\nДоступные услуги консультаций:\n";
            
            foreach ($servicesForPrompt as $service) {
                $prompt .= "- ID {$service['id']}: {$service['name']}\n";
            }
            
            $prompt .= "\nВерни ТОЛЬКО валидный JSON без дополнительного текста:\n";
            $prompt .= "{\"service_ids\": [id1, id2, ...]}\n\n";
            $prompt .= "Включи ТОЛЬКО те услуги, которые явно относятся к этому специалисту:\n";
            $prompt .= "- Если услуга содержит имя специалиста (например, \"Консультация невролога Желомской Л.Г.\" или содержит фамилию \"$lastName\") - ОБЯЗАТЕЛЬНО включи её\n";
            if (!empty($specialization)) {
                $prompt .= "- Если услуга содержит специализацию \"$specialization\" (например, \"Консультация $specialization\" или \"Прием $specialization\") - включи её\n";
            }
            $prompt .= "- Если нет явной связи - НЕ включай услугу\n";
            $prompt .= "- НЕ включай услуги других специалистов или нерелевантные услуги\n";
            if (!empty($preFilteredServices)) {
                $prompt .= "\nВАЖНО: В списке выше есть услуги, которые содержат имя или специализацию этого специалиста - обязательно включи их!";
            }
            
            // Вызываем AI через OpenRouter
            $data = [
                'model' => GEMINI_MODEL,
                'messages' => [
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'max_tokens' => 500,
                'temperature' => 0.3
            ];
            
            $ch = curl_init(OPENROUTER_API_URL);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Authorization: Bearer ' . OPENROUTER_API_KEY,
                'HTTP-Referer: ' . WIDGET_DOMAIN,
                'X-Title: AI Widget'
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode !== 200) {
                throw new Exception("OpenRouter API error: HTTP $httpCode");
            }
            
            $result = json_decode($response, true);
            if (!isset($result['choices'][0]['message']['content'])) {
                throw new Exception("Invalid OpenRouter response");
            }
            
            $aiResponse = $result['choices'][0]['message']['content'];
            
            // Парсим JSON ответ
            $aiResponse = preg_replace('/^```json\s*|\s*```$/s', '', $aiResponse);
            $parsed = json_decode($aiResponse, true);
            
            if (!isset($parsed['service_ids']) || !is_array($parsed['service_ids'])) {
                throw new Exception("Invalid AI response format: " . substr($aiResponse, 0, 200));
            }
            
            $serviceIds = array_map('intval', $parsed['service_ids']);
            
            // Fallback: если AI не нашел услуги, но мы нашли их предварительным поиском - используем их
            if (empty($serviceIds) && !empty($preFilteredServices)) {
                $serviceIds = array_map(function($s) { return (int)$s['id']; }, $preFilteredServices);
                $logMessages[] = "⚠ Специалист ID $specialistId ($specialistName): AI не нашел услуги, использованы предварительно найденные (" . count($serviceIds) . " услуг)";
            }
            
            // Если предварительно нашли услуги по имени - добавляем их в любом случае
            if (!empty($lastName)) {
                foreach ($preFilteredServices as $preService) {
                    if (stripos(mb_strtolower($preService['name']), $lastName) !== false) {
                        $preServiceId = (int)$preService['id'];
                        if (!in_array($preServiceId, $serviceIds)) {
                            $serviceIds[] = $preServiceId;
                        }
                    }
                }
            }
            
            // Сохраняем связи в БД
            $db->beginTransaction();
            try {
                $savedCount = 0;
                foreach ($serviceIds as $serviceId) {
                    // Проверяем, что услуга существует
                    $checkStmt = $db->prepare("
                        SELECT id FROM parsed_items 
                        WHERE id = ? AND widget_id = ? AND section_name = 'services' AND is_duplicate = 0
                    ");
                    $checkStmt->execute([$serviceId, $widgetId]);
                    if (!$checkStmt->fetch()) {
                        continue; // Пропускаем несуществующие услуги
                    }
                    
                    // Сохраняем связь (игнорируем дубликаты)
                    $insertStmt = $db->prepare("
                        INSERT IGNORE INTO specialist_services (widget_id, specialist_id, service_id)
                        VALUES (?, ?, ?)
                    ");
                    $insertStmt->execute([$widgetId, $specialistId, $serviceId]);
                    $savedCount++;
                }
                $db->commit();
                
                $processed++;
                if ($savedCount > 0) {
                    $logMessages[] = "✓ Специалист ID $specialistId ($specialistName): связано $savedCount услуг" . (!empty($preFilteredServices) ? " (предварительно найдено: " . count($preFilteredServices) . ")" : "");
                } else {
                    // Проверяем, является ли специалист врачом или медсестрой
                    $isDoctor = stripos($specialistDesc, 'врач') !== false || stripos($specialistDesc, 'доктор') !== false;
                    $isNurse = stripos($specialistDesc, 'медсестра') !== false || stripos($specialistDesc, 'фельдшер') !== false;
                    $isAdmin = stripos($specialistDesc, 'советник') !== false || stripos($specialistDesc, 'заведующ') !== false;
                    
                    if ($isNurse || $isAdmin) {
                        $logMessages[] = "○ Специалист ID $specialistId ($specialistName): 0 услуг (медсестра/административный персонал - это нормально)";
                    } elseif ($isDoctor) {
                        $logMessages[] = "⚠ Специалист ID $specialistId ($specialistName): 0 услуг (ВРАЧ без услуг - возможно, нет услуг консультаций для этой специализации)";
                    } else {
                        $logMessages[] = "○ Специалист ID $specialistId ($specialistName): 0 услуг";
                    }
                }
                
            } catch (Exception $e) {
                $db->rollBack();
                throw $e;
            }
            
        } catch (Exception $e) {
            $errors++;
            $logMessages[] = "✗ Ошибка для специалиста ID {$specialist['id']}: " . $e->getMessage();
        }
    }
    
    // Подсчитываем общее количество специалистов
    $totalStmt = $db->prepare("
        SELECT COUNT(*) as total
        FROM parsed_items
        WHERE widget_id = ? 
          AND section_name = 'specialists'
          AND is_duplicate = 0
    ");
    $totalStmt->execute([$widgetId]);
    $total = $totalStmt->fetch()['total'];
    
    echo json_encode([
        'success' => true,
        'processed' => $processed,
        'errors' => $errors,
        'total' => $total,
        'processed_total' => $offset + $processed,
        'log' => implode("\n", $logMessages)
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

