<?php
require_once __DIR__ . '/../../config.php';

// Проверка авторизации
session_start();
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

header('Content-Type: application/json');

try {
    $input = json_decode(file_get_contents('php://input'), true);
    $widgetId = (int)($input['widget_id'] ?? 0);
    $sectionName = isset($input['section_name']) && $input['section_name'] !== '' ? $input['section_name'] : null;
    $offset = (int)($input['offset'] ?? 0);
    $limit = (int)($input['limit'] ?? 10);
    
    if ($widgetId <= 0) {
        throw new Exception('Invalid widget_id');
    }
    
    if ($limit < 1 || $limit > 50) {
        $limit = 10;
    }
    
    $db = getDatabase();
    require_once __DIR__ . '/../../api/embedding-functions.php';
    
    // Получаем записи без embedding
    $whereClause = "pi.widget_id = ? AND pi.is_duplicate = 0";
    $params = [$widgetId];
    
    if ($sectionName) {
        $whereClause .= " AND pi.section_name = ?";
        $params[] = $sectionName;
    }
    
    // Исключаем записи, у которых уже есть embedding
    $whereClause .= " AND NOT EXISTS (
        SELECT 1 FROM item_embeddings ie WHERE ie.item_id = pi.id
    )";
    
    // Получаем общее количество записей без embedding
    $stmt = $db->prepare("SELECT COUNT(*) as total FROM parsed_items pi WHERE $whereClause");
    $stmt->execute($params);
    $total = (int)$stmt->fetch()['total'];
    
    // Получаем батч записей для обработки
    $limitParam = $limit;
    $offsetParam = $offset;
    $stmt = $db->prepare("
        SELECT pi.id, pi.section_name
        FROM parsed_items pi
        WHERE $whereClause
        ORDER BY pi.id
        LIMIT ? OFFSET ?
    ");
    
    // Добавляем limit и offset в конец массива параметров
    $allParams = array_merge($params, [$limitParam, $offsetParam]);
    $stmt->execute($allParams);
    $items = $stmt->fetchAll();
    
    $processed = 0;
    $errors = 0;
    $logMessages = [];
    
    if (empty($items)) {
        $logMessages[] = "Нет записей для обработки. Все записи уже имеют embedding или записей нет.";
    }
    
    foreach ($items as $item) {
        try {
            $itemId = (int)$item['id'];
            
            // Собираем текст для embedding
            $itemText = buildItemTextForEmbedding($db, $itemId);
            
            if (empty($itemText)) {
                $logMessages[] = "Запись #$itemId: пропущена (нет текста для embedding)";
                continue;
            }
            
            // Создаем embedding
            $embedding = createEmbedding($itemText);
            
            if (empty($embedding)) {
                $errors++;
                $logMessages[] = "Запись #$itemId: ошибка создания embedding";
                continue;
            }
            
            // Сохраняем embedding
            $textHash = md5($itemText);
            $saved = saveItemEmbedding($db, $itemId, $embedding, $textHash);
            
            if ($saved) {
                $processed++;
                $logMessages[] = "Запись #$itemId: embedding создан и сохранен";
            } else {
                $errors++;
                $logMessages[] = "Запись #$itemId: ошибка сохранения embedding";
            }
            
        } catch (Exception $e) {
            $errors++;
            $logMessages[] = "Запись #{$item['id']}: ошибка - " . $e->getMessage();
            error_log("Error processing embedding for item {$item['id']}: " . $e->getMessage());
        }
    }
    
    $processedTotal = $offset + $processed;
    
    // Добавляем отладочную информацию
    $debugInfo = [
        'widget_id' => $widgetId,
        'section_name' => $sectionName,
        'offset' => $offset,
        'limit' => $limit,
        'items_found' => count($items)
    ];
    
    echo json_encode([
        'success' => true,
        'processed' => $processed,
        'errors' => $errors,
        'total' => $total,
        'processed_total' => $processedTotal,
        'log' => implode("\n", $logMessages),
        'debug' => $debugInfo
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

