<?php
/**
 * AJAX: Запуск краулинга с выбранных URL
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../includes/auth.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    $widgetId = $_POST['widget_id'] ?? null;
    $urlIds = $_POST['start_urls'] ?? [];
    $maxDepth = (int)($_POST['max_depth'] ?? 3);
    $maxUrls = (int)($_POST['max_urls'] ?? 500);
    $timeoutSeconds = (int)($_POST['timeout_seconds'] ?? 300);
    
    if (!$widgetId) {
        throw new Exception('Widget ID is required');
    }
    
    if (empty($urlIds) || !is_array($urlIds)) {
        throw new Exception('At least one URL must be selected');
    }
    
    // Валидация параметров
    if ($maxDepth < 1 || $maxDepth > 20) {
        throw new Exception('Max depth must be between 1 and 20');
    }
    
    if ($maxUrls < 10 || $maxUrls > 50000) {
        throw new Exception('Max URLs must be between 10 and 50000');
    }
    
    if ($timeoutSeconds < 60 || $timeoutSeconds > 3600) {
        throw new Exception('Timeout must be between 60 and 3600 seconds');
    }
    
    $db = getDatabase();
    
    // Получаем виджет
    $stmt = $db->prepare("SELECT * FROM widgets WHERE id = ?");
    $stmt->execute([$widgetId]);
    $widget = $stmt->fetch();
    
    if (!$widget) {
        throw new Exception('Widget not found');
    }
    
    // Получаем настройки
    $stmt = $db->prepare("SELECT * FROM widget_settings WHERE widget_id = ?");
    $stmt->execute([$widgetId]);
    $settings = $stmt->fetch();
    
    if (!$settings) {
        throw new Exception('Widget settings not found');
    }
    
    // Получаем выбранные URL из базы
    $placeholders = str_repeat('?,', count($urlIds) - 1) . '?';
    $stmt = $db->prepare("
        SELECT url FROM discovered_urls 
        WHERE id IN ($placeholders) 
        AND widget_id = ?
        AND is_canonical = 1
    ");
    $stmt->execute([...$urlIds, $widgetId]);
    $selectedUrls = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    if (empty($selectedUrls)) {
        throw new Exception('No valid URLs found for the selected IDs');
    }
    
    // Создаем задание краулинга с JSON массивом URL
    $startUrlsJson = json_encode($selectedUrls);
    
    $stmt = $db->prepare("
        INSERT INTO crawler_jobs 
        (widget_id, start_url, max_depth, max_urls, timeout_seconds, status)
        VALUES (?, ?, ?, ?, ?, 'running')
    ");
    
    $stmt->execute([
        $widgetId,
        $startUrlsJson,
        $maxDepth,
        $maxUrls,
        $timeoutSeconds
    ]);
    
    $jobId = $db->lastInsertId();
    
    // Запускаем краулер в фоне
    $phpBin = '/usr/bin/php';
    $scriptPath = __DIR__ . '/../../parser/run-crawler-multi.php';
    $logFile = __DIR__ . '/../../logs/crawler.log';
    
    $command = sprintf(
        '%s %s %d >> %s 2>&1 &',
        escapeshellcmd($phpBin),
        escapeshellarg($scriptPath),
        $jobId,
        escapeshellarg($logFile)
    );
    
    exec($command);
    
    echo json_encode([
        'success' => true,
        'job_id' => $jobId,
        'start_urls_count' => count($selectedUrls),
        'message' => 'Crawler started successfully from ' . count($selectedUrls) . ' URL(s)'
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'error' => $e->getMessage()
    ]);
}

