#!/bin/bash

# Главный скрипт ежедневного бэкапа
# Создает бэкап проекта и БД, загружает на FTP, выполняет ротацию

set -e

PROJECT_ROOT="/root/ai-widget"
BACKUP_DIR="${PROJECT_ROOT}/backups"
LOG_FILE="${PROJECT_ROOT}/logs/backup.log"

# Создаем необходимые директории
mkdir -p "${BACKUP_DIR}"
mkdir -p "$(dirname "${LOG_FILE}")"

# Функция логирования
log() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] $1" | tee -a "${LOG_FILE}"
}

log "=========================================="
log "Начало ежедневного бэкапа"

# 1. Создаем бэкап
log "Шаг 1: Создание бэкапа проекта и БД..."
"${PROJECT_ROOT}/backup.sh" >> "${LOG_FILE}" 2>&1

if [ $? -ne 0 ]; then
    log "ОШИБКА: Не удалось создать бэкап!"
    exit 1
fi

# Находим последний созданный бэкап
LATEST_BACKUP=$(ls -t "${BACKUP_DIR}"/ai-widget-backup-*-full.tar.gz 2>/dev/null | head -n 1)

if [ -z "${LATEST_BACKUP}" ]; then
    log "ОШИБКА: Не найден созданный бэкап!"
    exit 1
fi

log "Найден бэкап: $(basename "${LATEST_BACKUP}")"

# 2. Загружаем на FTP
log "Шаг 2: Загрузка на FTP..."
"${PROJECT_ROOT}/upload_to_ftp.sh" "${LATEST_BACKUP}" >> "${LOG_FILE}" 2>&1

if [ $? -ne 0 ]; then
    log "ОШИБКА: Не удалось загрузить бэкап на FTP!"
    exit 1
fi

# 3. Удаляем локальный бэкап (чтобы не занимать место)
log "Шаг 3: Удаление локального бэкапа..."
rm -f "${LATEST_BACKUP}"
log "Локальный бэкап удален"

# 4. Очистка старых локальных бэкапов (оставляем последние 3)
log "Шаг 4: Очистка старых локальных бэкапов..."
cd "${BACKUP_DIR}"
ls -t ai-widget-backup-*-full.tar.gz 2>/dev/null | tail -n +4 | xargs -r rm -f
log "Старые локальные бэкапы удалены (оставлено максимум 3)"

log "Ежедневный бэкап завершен успешно!"
log "=========================================="

exit 0

